#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "linux_fs.h"
#include "mount_guess_fstype.h"

extern int silently;

#define ETC_FILESYSTEMS		"/etc/filesystems"
#define PROC_FILESYSTEMS	"/proc/filesystems"

#define SIZE(a) (sizeof(a)/sizeof(a[0]))

// Most file system types can be recognized by a `magic' number
// in the superblock.  Note that the order of the tests is
// significant: by coincidence a filesystem can have the
// magic numbers for several file system types simultaneously.
// For example, the romfs magic lives in the 1st sector;
// xiafs does not touch the 1st sector and has its magic in
// the 2nd sector; ext2 does not touch the first two sectors. 

static int assemble4le(unsigned char *p)
 {
	return (p[0] | (p[1] << 8) | (p[2] << 16) | (p[3] << 24));
 }

static inline unsigned short swapped(unsigned short a) 
{
     return (a>>8) | (a<<8);
}

static int may_be_swap(const char *s) 
 {
	return (strncmp(s-10, "SWAP-SPACE", 10) == 0 ||	strncmp(s-10, "SWAPSPACE2", 10) == 0);
 }

#define NTFS_BOOT_EXTENDED_BOOT   0x80
#define NTFS_BOOT_MFT_1024_SIZE   0xF6
#define NTFS_BOOT_INDEX_4096_SIZE 0xF4
#define NTFS_BOOT_MAGIC           0xAA55

unsigned CheckNTFSBoot( t_NTFSBoot *Boot )
 {
  // Check signatures
  if ( *(unsigned long*)(Boot->cSystemID)   != 0x5346544e // 'SFTN'
    || *(unsigned long*)(Boot->cSystemID+4) != 0x20202020 // '    '
    || (Boot->wSectorSize        != 512            )
    || (Boot->bExtendedSignature != NTFS_BOOT_EXTENDED_BOOT)
    || (Boot->wMagic             != NTFS_BOOT_MAGIC        ) )
  {
    //printf("Incorrect Boot signatures. cSystemID: %s, wSectorSize: %d,bExtendedSignature: %x,wMagic: %x, sizeof: %d \n",Boot->cSystemID,Boot->wSectorSize,Boot->bExtendedSignature,Boot->wMagic,sizeof(*Boot));
    return -1;
  }

  // Check cluster size
  if ( (Boot->bClusterSize != 1 ) && (Boot->bClusterSize != 2  )
    && (Boot->bClusterSize != 4 ) && (Boot->bClusterSize != 8  )
    && (Boot->bClusterSize != 16) && (Boot->bClusterSize != 32 )
    && (Boot->bClusterSize != 64) && (Boot->bClusterSize != 128) )
  {
    //printf("Incorrect cluster size. bClusterSize=%d\n",Boot->bClusterSize);
    return -1;
  }

  if ( (Boot->u64MFTCluster     * (unsigned long)Boot->bClusterSize >= Boot->u64DiskSize)
    || (Boot->u64MFTCopyCluster * (unsigned long)Boot->bClusterSize >= Boot->u64DiskSize) )
  {
    //printf("Incorrect disk size\n");
    return -1;
  }

  // Check MFT record size
  if ( (Boot->bMFTRecordSize != NTFS_BOOT_MFT_1024_SIZE)
    && (Boot->bClusterSize != 1) && (Boot->bClusterSize != 2)
    && (Boot->bClusterSize != 4) && (Boot->bClusterSize != 8) )
  {
    //printf("Incorrect MFT record size\n");
    return -1;
  }

  // Check index record size
  if ( (Boot->dwIndexRecordSize != NTFS_BOOT_INDEX_4096_SIZE)
    && (Boot->dwIndexRecordSize != 1) && (Boot->dwIndexRecordSize != 2)
    && (Boot->dwIndexRecordSize != 4) && (Boot->dwIndexRecordSize != 8) )
  {
    //printf("Incorrect index record size\n");
    return -1;
  }

  // Everything is OK
  //printf("Found NTFS pasrtition\n");
  return 0;
 }
/*
unsigned CheckFATBoot( t_FATBoot *Boot )
 {
  if (Boot->ResSect==0 ||
      Boot->ClstSz==0 || //Boot->ClstSz>MaxClSz ||
      Boot->FatCnt==0 || Boot->FatSect==0 ||
      Boot->TotSect==0) return -1;

  DatOfs=Boot->ResSect+Boot->FatSect*Boot->FatCnt+Boot->RootEnts/16;

  Boot->TotClst=(Boot->TotSect-DatOfs)/Boot->ClstSz;

  // check FAT and cluster sizes
  if (Boot->FatTp==FFAT32_)
   {
    if (Boot->TotClst>0X0FFFFFF5L ||
        (Boot->TotClst+2)>Boot->FatSect*128 ||
        Boot->RootClst>=Boot->TotClst+2) return -1;
   }
  else
   {
    if (Boot->TotClst<=4085)
     {
      Boot->FatTp=FFAT12_;
      if ((Boot->TotClst+2)*3 > Boot->FatSect*1024) return -1;
     }
    else
     {
      if (Boot->TotClst>65525 || (Boot->TotClst+2)>Boot->FatSect*256) return -1;
     }
   }
  return 0; 
 }
*/
unsigned CheckFATBoot( t_FATBoot *boot )
 {
  unsigned long cFatTp; // 0 - FAT12, 1 - FAT16, 2- FAT32
  unsigned long dwTotFSct=0, dwDataStart=0, dwRootSector=0;

  if ( 0 == boot->bClusterSize )
   {
    //printf("bClusterSize: %d\n",boot->bClusterSize);
    return -1;
   }    
  
  if ( boot->wRootEntries == 0 )
   {
    cFatTp = 2;
    if ( boot->wFATSize )
     {
      //printf("wFATSize: %d\n",boot->wFATSize);
      return -1;
     } 
   }
  else if ( boot->FAT16.cSystemID[4] == '2' ) // "FAT12"
    cFatTp = 0;
  else
    cFatTp = 1;
    
  // Calculate the size of system area
  switch( cFatTp )
  {
    case 2:
      // Root dir sector
      dwDataStart  = boot->wReservedSectors + boot->FAT32.dwFatSize * boot->bFATNumber;
      dwRootSector = dwDataStart + boot->bClusterSize * boot->FAT32.dwRootCluster;
      break;
    case 1:
    case 0:
      dwRootSector  = boot->wReservedSectors + boot->wFATSize*boot->bFATNumber;
      // For FAT12/16 add root dir size
      dwDataStart   = dwRootSector + ( boot->wRootEntries*32 + 511 )/512;
      break;
  }

  dwTotFSct    = 0 == boot->wDiskSize? boot->dwDiskSize : boot->wDiskSize;  

  // Check root position
  if ( dwRootSector >= dwTotFSct )
   {
    //printf("dwRootSector(%d) >= dwTotFSct(%d)\n",dwRootSector,dwTotFSct);
    return -1;
   }    
  
  // Check partition size
  if ( dwTotFSct <= dwDataStart || 0 == ( dwTotFSct - dwDataStart) / boot->bClusterSize )
   {
    //printf("dwTotFSct(%d) <= dwDataStart(%d) || 0 == ( dwTotFSct(%d) - dwDataStart(%d)) / boot->bClusterSize(%d)\n",dwTotFSct,dwDataStart,dwTotFSct,dwDataStart,boot->bClusterSize);
    return -1;
   }    
  
  //printf("Found DOS pasrtition\n");
  return 0;
 }

static char *fstype(const char *device) 
 {
  int fd;
  char *type = NULL;
    
  union 
   {
    struct minix_super_block ms;
    struct ext_super_block es;
    struct ext2_super_block e2s;
   } sb;
	
  union 
   {
    struct xiafs_super_block xiasb;
    long bfs_magic;
    struct ntfs_super_block ntfssb;
    struct fat_super_block fatsb;
    struct xfs_super_block xfsb;
    struct cramfs_super_block cramfssb;
   } xsb;
	
  struct ufs_super_block ufssb;
    
  union 
   {
    struct iso_volume_descriptor iso;
    struct hs_volume_descriptor hs;
   } isosb;
	
  struct hfs_super_block hfssb;
  struct hpfs_super_block hpfssb;
  struct stat statbuf;

  if(silently!=0) printf ("Checking device %s\n", device );

  // opening and reading an arbitrary unknown path can have undesired
  // side effects - first check that `device' refers to a block device
  if ( stat (device, &statbuf) || !S_ISBLK(statbuf.st_mode) ) return 0;

  if ( (fd = open(device, O_RDONLY)) < 0) return 0;

  if ( lseek(fd, 1024, SEEK_SET) != 1024 || read(fd, (char *) &sb, sizeof(sb)) != sizeof(sb) )
	goto io_error;

  if (ext2magic(sb.e2s) == EXT2_SUPER_MAGIC
	|| ext2magic(sb.e2s) == EXT2_PRE_02B_MAGIC
	|| ext2magic(sb.e2s) == swapped(EXT2_SUPER_MAGIC))
   {	
     type = "ext2";
     // maybe even ext3? 
     if ((assemble4le(sb.e2s.s_feature_compat)
	  & EXT3_FEATURE_COMPAT_HAS_JOURNAL) && assemble4le(sb.e2s.s_journal_inum) != 0)
       type = "ext3";	// "ext3" 
   }	 
  else if (minixmagic(sb.ms) == MINIX_SUPER_MAGIC
	     || minixmagic(sb.ms) == MINIX_SUPER_MAGIC2
	     || minixmagic(sb.ms) == swapped(MINIX_SUPER_MAGIC2))
	 type = "minix";

  else if (extmagic(sb.es) == EXT_SUPER_MAGIC)
	 type = "ext";

  if (!type)
   {
    if (lseek(fd, 0, SEEK_SET) != 0 || read(fd, (char *) &xsb, sizeof(xsb)) != sizeof(xsb))
      goto io_error;

    if (xiafsmagic(xsb.xiasb) == _XIAFS_SUPER_MAGIC)
	      type = "xiafs";
    else
     if(!strncmp(xsb.xfsb.s_magic, XFS_SUPER_MAGIC, 4) ||
		 !strncmp(xsb.xfsb.s_magic, XFS_SUPER_MAGIC2, 4))
	      type = "xfs";
     else
      if(xsb.bfs_magic == 0x1badface)
	      type = "bfs";
     else
//     if(!strncmp(xsb.ntfssb.cSystemID, NTFS_SUPER_MAGIC,sizeof(NTFS_SUPER_MAGIC)))
      if(CheckNTFSBoot( &xsb.ntfssb)==0 )
	      type = "ntfs";
      else
       if(cramfsmagic(xsb.cramfssb) == CRAMFS_SUPER_MAGIC)
	      type = "cramfs";
      else
       if(CheckFATBoot(&xsb.fatsb )==0)
//     if ((!strncmp(xsb.fatsb.cSystemID, "MSDOS", 5) ||
//		   !strncmp(xsb.fatsb.cSystemID, "MSWIN", 5) ||
//		   !strncmp(xsb.fatsb.cSystemID, "MTOOL", 5) ||
//		   !strncmp(xsb.fatsb.cSystemID, "mkdosfs", 7) ||
//		   !strncmp(xsb.fatsb.cSystemID, "kmkdosfs", 8))
//		  && (!strncmp(xsb.fatsb.cSystemID, "FAT12   ", 8) ||
//		      !strncmp(xsb.fatsb.cSystemID, "FAT16   ", 8) ||
//		      !strncmp(xsb.fatsb.cSystemID, "FAT32   ", 8)))
	      type = "vfat";	/* only guessing - might as well be fat or umsdos */
    }

  if (!type)
   {
    if (lseek(fd, 8192, SEEK_SET) != 8192
	     || read(fd, (char *) &ufssb, sizeof(ufssb)) != sizeof(ufssb))
	      goto io_error;

	 if (ufsmagic(ufssb) == UFS_SUPER_MAGIC) /* also test swapped version? */
	      type = "ufs";
   }

  if (!type)
   {
    if (lseek(fd, 0x8000, SEEK_SET) != 0x8000 || read(fd, (char *) &isosb, sizeof(isosb)) != sizeof(isosb))
      goto io_error;

    if(strncmp(isosb.iso.id, ISO_STANDARD_ID, sizeof(isosb.iso.id)) == 0
	    || strncmp(isosb.hs.id, HS_STANDARD_ID, sizeof(isosb.hs.id)) == 0)
	      type = "iso9660";
   }

  if (!type)
   {
    if (lseek(fd, 0x400, SEEK_SET) != 0x400
            || read(fd, (char *) &hfssb, sizeof(hfssb)) != sizeof(hfssb))
             goto io_error;

        /* also check if block size is equal to 512 bytes,
           since the hfs driver currently only has support
           for block sizes of 512 bytes long, and to be
           more accurate (sb magic is only a short int) */
        if ((hfsmagic(hfssb) == HFS_SUPER_MAGIC &&
	     hfsblksize(hfssb) == 0x20000) ||
            (swapped(hfsmagic(hfssb)) == HFS_SUPER_MAGIC &&
             hfsblksize(hfssb) == 0x200))
             type = "hfs";
    }

    if (!type) {
        if (lseek(fd, 0x2000, SEEK_SET) != 0x2000
            || read(fd, (char *) &hpfssb, sizeof(hpfssb)) != sizeof(hpfssb))
             goto io_error;

        if (hpfsmagic(hpfssb) == HPFS_SUPER_MAGIC)
             type = "hpfs";
    }

    if (!type)
     {
	    /* perhaps the user tries to mount the swap space
	       on a new disk; warn her before she does mke2fs on it */
	    int pagesize = getpagesize();
	    int rd;
	    char buf[32768];

	    rd = pagesize;
	    if (rd < 8192)        rd = 8192;
	    if (rd > sizeof(buf)) rd = sizeof(buf);
	    if (lseek(fd, 0, SEEK_SET) != 0 || read(fd, buf, rd) != rd)
		    goto io_error;
	    if (may_be_swap(buf+pagesize) || may_be_swap(buf+4096) || may_be_swap(buf+8192))
		    type = "swap";
     }

    close (fd);
    return(type);

io_error:
    if(errno!=0) perror(device);
    close(fd);
    return 0;
 }

char *guess_fstype_from_superblock(const char *spec)
 {
  char *type=fstype(spec);

  if ( !type ) type = "unknown";
  return type;
 }
