/*				SUPER.CC
			A C++ implementation of
			   "super precision".
			       Version 2

                Copyright (c) 1997 William T. Wyatt, Jr.

	"Super precision" objects are implementations of a floating-point
data type having user-selectable precision and radix.  The intent is to
allow extremely long multiple-precision mantissas in the floating-point
representation.  Because all multiple-precision working is done here through
software emulation of arithmetic, execution times are orders of magnitude
longer (typically 1000x for roughly 50 decimal digits) than hardware execution
times for type "double".  Any radix from 2 to 16, and any precision from a few
digits to several billion digits may be selected by a program using the objects,
SUBJECT TO MEMORY LIMITATIONS.  At least 5 million decimal digits can be
accommodated in a 128-megabyte main memory without recourse to paging, at
least for simple programs.

[Historical note:
	In 1972 the author programmed "super precision" objects in Fortran-66,
but use of the objects in a target program required manually coding all
subroutine calls to the "super precision" software modules.  The validity of
the "super precision" software modules and algorithms was established
(Lozier).  A related effort to develop a language translator to convert
the use of an additional "super precision" data type into standard Fortran
was not completed, although the concept was proved (Orser).  Wyatt, Lozier,
and Orser jointly reported the undertaking in Vol. 2, No. 3 of the journal
"ACM Transactions on Mathematical Software", pp 209-231 (September 1976).
	The original Fortran software was distinguished by its portability and
by the completeness of the set of operations and functions supported by the
collection of over 120 subroutines, covering ordinary floating-point arithmetic,
unnormalized floating-point arithmetic, intrinsic and mathematical functions,
and input-output and base conversion operations.  The complete "Super Precision
Package" of Fortran subroutines by Wyatt was distributed by Lozier from the
National Bureau of Standards to a number of researchers around the world.
Subsequently, other exceptionally complete Fortran packages were developed by
Brent and, more recently, by Bailey for arbitrary precision floating-point
arithmetic.
	The advent of the C++ programming language allows a thorough
implementation of "super precision" objects as a class, with extensive function
and operator overloading, even more powerful than the original Fortran language
translator concept.  The material below is a C++ language expression of the
original "super precision" idea with many significant improvements.  Although
this work has few peers as a generic implementation of arbitrary precision
floating-point arithmetic in ANY programming language, it stands as the
preeminent implementation in C++ where OOP concepts combine simplicity of use
with great power and flexibility to create applications and further extension.]

Cautionary notes:
	This code yields different results when compiled at optimizations 0 and
2 by gcc.  This involves a last-digit difference in sqrt and slight differences
in mathematical functions that use sqrt (such as for range reduction).  Slight
differences in computational results are a common side-effect of high
optimization levels in many compilers.
	The "fast divide" algorithm is a recursive Newton-Raphson algorithm
(using FFT's to obtain intermediate product convolutions).  The default method
gives a "perfect divide" to the last digit and runs about 6 times longer than
the "fast multiply".  The perfect fast divide may be deselected, however, so
that an "imperfect divide" is done, possessing an uncertain last digit (or last
few digits in the case of a small radix) but running only about 5 times longer
than the "fast multiply".  The "standard" divide-and-correct division algorithm
always gives a "perfect" (certain) last digit but runs in n*n time.

Programming notes:
	The implementing base class is "super_precision" and defines a 32-bit
exponent-and-sign word, a pointer to a multiword mantissa appropriate for the
precision specified by the user, and a few additional data words to specify
radix, precision, and validity.  Together these make up a super_precision
object.
	Objects passed as arguments to a function are generally passed by
reference ("&" type objects).  Objects that are returned by functions are
returned by value because a super_precision object contains a pointer, except
when the "this" pointer points to a user-defined object (as in +=, -=, etc).
	All private functions are written to allow self-modification of the
arguments.  That is, when there are two or more super_precision arguments in a
function call, any of them may be the same object, including any modifiable
("output") object -- without harm to the algorithm.  The same is true of all
public and friend functions included in this class implementation.  However,
the use of the four super_precision "registers" is restricted (see next
paragraph), because they are reused cooperatively by many functions in the
library.
	To reduce the memory overhead of creating and destructing temporary
super_precision objects used in member functions, as well as to minimize
fragmentation of the Free Store, four static super_precision "register"
objects are defined and referred to through alias pointers in basic
arithmetic routines.  Also a friend class "super_precision_pool_ptr" defines
pointers to a static pool of super_precision objects that are dynamically
assigned and released as working (local) variables upon request.  The pool
starts with zero pointers and increases to keep up with demand, up to a limit
of 30 pointers.  The pointers are used indirectly by creating reference
variables to access the super_precision object pointed to.  The register-like
and pool super_precision objects have mantissas that are four words longer
than ordinary super_precision mantissas.  A "user pool" of super_precision
objects can also be defined and used/reused to minimize fragmentation of the
Free Store.  However, the user pool objects' mantissas do not carry the extra
four words precision that register-like and pool object mantissas carry.
	Additionally, a few static super_precision objects are defined (when
needed) and used to contain values of pi, log of "e", etc.  Also, an extra
static super_precision object is defined to serve as a returnable object in
some member and friend functions.

*/

#include "super.h"

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//                      COMPILE-TIME OPTIONS
// Choose which square root algorithm to use (division-based or not):
#define SQRT_ALGORITHM_DIVIDE
// Choose which pi algorithm to use -- quadratic or quartic convergence:
#define QUARTIC_PI_ALGORITHM
// Choose whether to sacrifice roundoff accuracy for speed, or not.
//#define SUPER_SPEED
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int32 debug = 0;			// Set to 1 if debug mode wanted.
int32 verbose = 0;			// Set to 1 if verbose mode wanted.
int super_precision_setup_called = 0;	// Signify if setup ever was called.

class super_precision					{ // CLASS DEFINITION
protected:
		// nonstatic data members:
	int32 sign_bias_exp;		// holds a sign and a biased exponent
	int32 *mantissa;		// points to integer array for mantissa
	int32 digits;			// no. significant digits in mantissa
						// = NDIGA
	short radix;			// number base of super_precision
						// = NBASE
	short validity;			// validity
	
		// default static values:
	static int32 def_digits;	// no. significant digits in mantissa
	static short def_radix;		// number base of super_precision object
	static short def_validity;	// validity (proper,infinite,NaN,etc.)

		// static object pointers:
	static super_precision *w;	// used in basic arithmetic (alias)
	static super_precision *zt;	// used in basic arithmetic (alias)
	static super_precision *zv;	// used in basic arithmetic (alias)
	static super_precision *zz;	// returnable obj for public fns (alias)
	static super_precision *pi;	// super_precision value for pi
	static super_precision *log_10;	// super_precision value for log10(e)
	static super_precision *log_radix;// super_precision log_def_radix(e)
	static int32 *zia;		// intermediate results
	static int32 *zib;		// intermediate results
	static int32 *t;		// unpacked intermediate results
	static int32 *v;		// unpacked intermediate results
	static double *dalt;		// fast_mult array pointer
	static char *output_result;	// result string in super_output
	static char *input_string;	// input string in operator>> overload

		// static variables
	static int32 rounding_method;	// rounding method flag = IROUND
	static int32 ndig;		// max no. of digits carried by mantissa
				// ndig is nf * nd and is >= digits.
	static int32 nd;		// no. of digits per mantissa word
	static int32 ind;		// equal to radix**nd
	static int32 bias;		// exponent bias = IBIAS
	static int32 nf;		// no. of 32-bit words in mantissa
	static int32 nfa;		// no. of words in repacked mult array
	static int32 nfah;		// no. of words in unpacked mult array
	static int32 nfahx;		// no. words used in unpacked mult array
	static int32 ndah;		// no. digits per unpacked mantissa word
	static int32 indah;		// equal to radix**ndah
	static int32 nda;		// nda = 2 * ndah
	static int32 inda;		// equal to indah**2
	static int32 naccum;		// equal to NDIG - 2*NDAH
	static int32 hdwr_base;		// hardware float-point arithmetic base
	static int32 nbit;		// number of bits in integer, less sign
	static int32 klip;		// if nonzero, is exact number of digits
	static int32 iklip;		// clipping factor to shorten mantissa
	static int32 indtrz;		// clipping factor for repacked array
	static int32 ndiv;		// no. digits in quotient estimator
	static int32 irbit;		// round bit, including trailing zeros
	static double baslog;		// log10 of def_radix
	static int32 orig_def_digits;	// save original value from setup
	static int32 orig_nf;		// save original value from setup
	static int32 max_output_string_size;	// max size of output_result
	static int32 max_input_string_size;	// max size of input_result
	static int32 block;		// length of blank-demarked digit string
	static int32 nblanks;		// max no. of blanks inserted in output
	static int32 output_base;	// output radix
	static int32 output_count;	// no. of digits in mantissa on output
	static int32 input_base;	// input radix
	static int32 fast_mult_digits;	// int32 = set_fast_multiply_threshold()
	static int32 dinda;		// scale factor in fast_mult array
	static int32 dnfa;		// no. words in combined fast_mult array
	static int32 dnda;		// no. digits/element in fast_mult array
	static int32 radix_powerof2;	// 0 for non-powerof2, else log2(radix)
	static int32 nfft;		// FFT size for fast_mult array
	static double double_fraction_max;	// largest integer in double
	static int32 double_fraction_digit_count;	// digits in double
	static int32 fast_div_digits;	// int32 = set_fast_divide_threshold ()
	static int32 log_Salamin_threshold;	// threshold for Salamin in log
	static int32 exp_Newton_threshold;	// threshold for Newton in exp
	static int32 atan_Salamin_threshold;	// threshold for Salamin in atan
	static int32 tan_Newton_threshold;	// threshold for Newton in tan
	static int pi_init;		// flag to signal need to initialize pi
	static int log_10_init;		// flag to initialize log_10
	static int log_radix_init;	// flag to initialize log_radix
	static int prev_radix;		// flag to record radix change
	static int32 radix_powers [32];	// powers of def_radix from 0 to nd
					
public:
	super_precision ( void);			// constructor
	~super_precision ( void);			// destructor
	super_precision ( const int32);			//conversion constructor
	super_precision ( const double);		//conversion constructor
	super_precision ( const char *);		//conversion constructor
	void *operator new ( size_t);			// overload new
	void operator delete ( void *);			// overload delete
#ifdef placement_new_array
	void *operator new [] ( size_t, int32);		// overload new []
	void operator delete [] ( void *);		// overload delete []
#endif
	super_precision ( const super_precision &);	// ovld copy-constructor
	static void setup ( const int32 digits, const short radix = 10);
							// initializer
	static void set_rounding_method ( const int32);	// change rounding mthd
	super_precision & operator= ( const super_precision &);	// overload a=b
	super_precision & operator= ( const int32); 		// overload a=i
	super_precision & operator= ( const double); 		// overload a=d
	super_precision & operator= ( const char *); 		// ovld a=string
	super_precision operator- ( void) const;		// overload -b
	super_precision operator+ ( void) const;		// overload +b
	super_precision operator+ ( const super_precision &) const; // ovld a+b
	super_precision & operator+=( const super_precision &);	// overload a+=b
	super_precision operator+ ( const int32) const; 	// overload a+i
	super_precision & operator+=( const int32); 		// overload a+=i
	super_precision operator- ( const super_precision &) const; // ovld a-b
	super_precision & operator-=( const super_precision &); // overload a-=b
	super_precision operator- ( const int32) const; 	// overload a-i
	super_precision & operator-=( const int32); 		// overload a-=i
	super_precision operator* ( const super_precision &) const; // ovld a*b
	super_precision & operator*=( const super_precision &); // overload a*=b
	super_precision operator* ( const int32) const; 	// overload a*i
	super_precision & operator*=( const int32); 		// overload a*=i
	super_precision operator/ ( const super_precision &) const; // ovld a/b
	super_precision & operator/=( const super_precision &);	// overload a/=b
	super_precision operator/ ( const int32) const;		// overload a/i
	super_precision & operator/=( const int32);		// overload a/=i
	super_precision operator% ( const super_precision &) const; // ovld a%b
	super_precision & operator%=( const super_precision &);	// overload a%=b
	super_precision operator% ( const int32) const;		// overload a%i
	super_precision & operator%=( const int32);		// overload a%=i
	super_precision & operator++( void);			// overload a++
	super_precision & operator++( int);			// overload ++a
	super_precision & operator--( void);			// overload a--
	super_precision & operator--( int);			// overload --a
	friend super_precision operator+ ( const int32,
				const super_precision &); 	// overload i+a
	friend super_precision operator- ( const int32,
				const super_precision &); 	// overload i-a
	friend super_precision operator* ( const int32,
				const super_precision &); 	// overload i*a
	friend super_precision operator/ ( const int32,
				const super_precision &); 	// overload i/a
	friend super_precision operator% ( const int32,
				const super_precision &); 	// overload i%a
	int operator< ( const super_precision &) const;		// overload a<b
	int operator> ( const super_precision &) const;		// overload a>b
	int operator==( const super_precision &) const;		// overload a==b
	int operator!=( const super_precision &) const;		// overload a!=b
	int operator<=( const super_precision &) const;		// overload a<=b
	int operator>=( const super_precision &) const;		// overload a>=b
	int operator! ( void) const;				// overload !a
	int operator&&( const super_precision &) const;		// overload a&&b
	int operator||( const super_precision &) const;		// overload a||b
	int operator< ( const int32) const;			// overload a<i
	int operator> ( const int32) const;			// overload a>i
	int operator==( const int32) const;			// overload a==i
	int operator!=( const int32) const;			// overload a!=i
	int operator<=( const int32) const;			// overload a<=i
	int operator>=( const int32) const;			// overload a>=i
	int operator&&( const int32) const;			// overload a&&i
	int operator||( const int32) const;			// overload a||i
	friend int operator< ( const int32,
				const super_precision &);	// overload i<a
	friend int operator> ( const int32,
				const super_precision &);	// overload i>a
	friend int operator==( const int32,
				const super_precision &);	// overload i==a
	friend int operator!=( const int32,
				const super_precision &);	// overload i!=a
	friend int operator<=( const int32,
				const super_precision &);	// overload i<=a
	friend int operator>=( const int32,
				const super_precision &);	// overload i>=a
	friend int operator&&( const int32,
				const super_precision &);	// overload i&&a
	friend int operator||( const int32,
				const super_precision &);	// overload i||a
	friend super_precision sqrt
				( const super_precision &);	// overload sqrt
	friend super_precision fabs
				( const super_precision &);	// overload fabs
	friend super_precision trunc
				( const super_precision &);	//overload trunc
	friend super_precision ceil
				( const super_precision &);	// overload ceil
	friend super_precision floor
				( const super_precision &);	//overload floor
	friend super_precision fmod ( const super_precision &,
				const super_precision &);	// overload fmod
	friend super_precision rint
				( const super_precision &);	// overload rint
	friend super_precision round_off ( const super_precision &,
				const int32);			// round off
	inline void super_neg ( void);				// *this=-*this
	super_precision & super_shift_public ( const int32);	// shift super
	super_precision & super_norm_public ( int32 &);		// norm super
	super_precision & super_klipr_public ( void);		// clip a
	void super_dump ( void) const;				// dump super
	void super_fill ( const int32, const int32);		// fill super
	friend void dump_tv ( const int32 tv []);		//dump t,v array
	friend int32 super_to_i ( const super_precision &);	// convert i=a
	friend double super_to_approx_double
				( const super_precision &);	// rough d = a
	friend double super_to_double
				( const super_precision &);	// exact d = a
	friend super_precision pow ( const super_precision &,
				const int32);			// overload pow
	friend super_precision pow ( const super_precision &,
				const super_precision &);	// overload pow
	friend super_precision pow ( const int32,
				const super_precision &);	// overload pow
	friend super_precision exp ( const super_precision &);	// overload exp
	friend super_precision log ( const super_precision &);	// overload log
	friend super_precision log10 ( const super_precision &);// overld log10
	friend super_precision log_Salamin
				( const super_precision &);	// Salamin algor
	friend void log_Salamin_complex
		( const super_precision &, const super_precision &,
		super_precision &, super_precision &);		// Salamin cplx
	void arithmetic_geometric_mean ( const super_precision &b);
								// get AGM
	void arithmetic_geometric_mean_cplx
			( super_precision &, super_precision &);// get cplx AGM
	friend super_precision pi ( void);			// returns pi
	friend super_precision log_10 ( void);			// rtn log10(e)
	friend super_precision log_radix ( void);	// rtn log_def_radix(e)
	friend super_precision sinh ( const super_precision &);	// overload sinh
	friend super_precision cosh ( const super_precision &);	// overload cosh
	friend super_precision tanh ( const super_precision &);	// overload tanh
	friend super_precision sin ( const super_precision &);	// overload sin
	friend super_precision cos ( const super_precision &);	// overload cos
	friend super_precision tan ( const super_precision &);	// overload tan
	friend super_precision tan_Salamin
				( const super_precision &);	// overload tan
	friend void sincos ( const super_precision &, super_precision &,
			super_precision &);			// sin(a),cos(a)
	friend super_precision asin ( const super_precision &);	// overload asin
	friend super_precision acos ( const super_precision &);	// overload acos
	friend super_precision atan ( const super_precision &);	// overload atan
	friend super_precision atan2 ( const super_precision &,
				const super_precision &);	//overload atan2
	friend super_precision asinh ( const super_precision &);//overload asinh
	friend super_precision acosh ( const super_precision &);//overload acosh
	friend super_precision atanh ( const super_precision &);//overload atanh
	friend super_precision hypot ( const super_precision &,
				const super_precision &);	//overload hypot
	friend super_precision max ( const super_precision &,
				const super_precision &);	// overload max
	friend super_precision min ( const super_precision &,
				const super_precision &);	// overload min
	friend super_precision nthrt ( const super_precision &,
				const int32);			// n'th root fn
	friend super_precision cbrt ( const super_precision &);	// overload cbrt
	friend super_precision frexp ( const super_precision &,
				int32 &);			//overload frexp
	friend super_precision ldexp ( const super_precision &,
				const int32);			//overload ldexp
	friend super_precision base_conv ( const super_precision &,
				const int32, const int32,
				const int32);			// convert base
	friend super_precision precision_conv ( const super_precision &);
								// conv. precis.
	friend char *super_output ( const super_precision &,
		const int32 count = 0,
		const int upper = 0, const int doplus = 0);	// make string
	friend super_precision super_convert_string ( const char *,
				int32 &);			//convert string
	friend ostream & operator<< ( ostream &,
				const super_precision &);	// overload <<
	friend istream & operator>> ( istream &,
				super_precision &);		// overload >>
	friend int32 super_set_output_block ( const int32 i = 0);// set blocking
	friend int32 super_set_output_base ( const int32 i = 0); // set o/p base
	friend int32 super_set_output_count ( const int32 i = 0);//set no. digit
	friend int32 super_set_input_base ( const int32 i = 0); //deflt i/p base
	static int conv_input_base;		// convert input to def_base?
	friend class super_precision_pool_ptr;			// friend class
	friend class super_precision_userpool;			// friend class
	static int std_ios_manip;			// follow ANSI usages
	static int show_output_base;	// show super_precision base on output
	static int32 set_fast_multiply_threshold ( const int32 i = -1);
							// set/show threshold
	static int32 set_fast_divide_threshold ( const int32 i = -1);
							// set/show threshold
	static int32 set_log_Salamin_threshold ( const int32 i = -1);
							// set/show threshold
	static int32 set_exp_Newton_threshold ( const int32 i = -1);
							// set/show threshold
	static int32 set_atan_Salamin_threshold ( const int32 i = -1);
							// set/show threshold
	static int32 set_tan_Newton_threshold ( const int32 i = -1);
							// set/show threshold
	static int perfect_fast_divide;				// switch to set
	super_precision increment_mantissa ( const int32 i = 1);// increase frac
	super_precision decrement_mantissa ( const int32 i = 1);// decrease frac
	int32 get_digit ( const int32) const;			// get digit(s)
	void put_digit ( const int32, const int32);		// put digit(s)
	int32 get_mantissa_word ( const int32) const;		// get frac word
	int disclose_radix ( void) const;			// show radix
	int32 disclose_digits ( void) const;			// show # digits
	int32 disclose_validity ( void) const;			// show validity
	int32 disclose_scale_factor ( void) const;		// show scale
	int32 disclose_mantissa_word_length ( void) const;	// show nf
	int32 *disclose_mantissa_pointer ( void) const;		// show mant ptr
	int32 disclose_long_div_threshold (void) const; 	// show indah
	int disclose_def_radix ( void) const;			//show def_radix
	int disclose_sign ( void) const;			// show sign
	friend int32 disclose_fftsize ( void);			// show nfft
	friend super_precision super_biggest ( void);		//retn biggest
	friend super_precision super_smallest ( void);		//retn smallest
	int32 leading_zero_count ( void) const;			//count left 0s
	int32 trailing_zero_count ( void) const;		//count right 0s
	int verify_scaling ( void) const;			//verify scaling
	static int permissive_error;	// if TRUE allows NaN,+Inf,-Inf,+-Inf
	friend super_precision super_NaN ( int i = 4);		// irreg forms
	friend super_precision super_Pinf ( void);		// sets +Inf
	friend super_precision super_Ninf ( void);		// sets -Inf
	friend super_precision super_PNinf ( void);		// sets +-Inf
	friend super_precision super_Unn ( void);		// sets Unnorm.
	int assert_invalid ( char *) const;			// test validity
	int assert_invalid ( const super_precision &, char *)
			const;					// test validity
	void assert_invalid_quit ( char *) const;		// test validity
	void assert_invalid_quit ( const super_precision &,
			char *) const;				// test validity
	static int32 debug_counter;
	static int32 debug_counter_divides;

protected:
	void super_move ( super_precision &a,
				const super_precision &b) const;  // set a=b
	void super_add ( super_precision &a, const super_precision &b,
			const super_precision &c) const;	// a = b + c
	void super_minus ( super_precision &a, const super_precision &b,
				const super_precision &c) const;// a = b - c
	void super_neg ( super_precision &a,
				const super_precision &b) const;  // a = -b
	void super_itoz ( super_precision &a, const int32 i) const;//convert a=i
	void super_mult ( super_precision &a, const super_precision &b,
				const super_precision &c) const;// a = b * c
	void super_multi ( super_precision &a, const super_precision &b,
				const int32 ia) const;		// a = b * i
	void super_div ( super_precision &a, const super_precision &b,
				const super_precision &c) const;// a = b / c
	void super_divi ( super_precision &a, const super_precision &b,
				const int32) const;		// a = b / i
	void super_shift ( super_precision &a, int32) const;	// shift a
	void super_norm ( super_precision &a, int32 &i) const;	// normalize a
	int super_test ( const super_precision &a,
				const super_precision &b) const;// compare a,b
	void super_klipr ( super_precision &a) const;		// clip a
	void super_zad ( super_precision &a, const super_precision &b,
				const super_precision &c) const;// >a = b + c
	void super_zmi ( super_precision &a, const super_precision &b,
				const super_precision &c) const;// >a = b - c
	void super_unpk ( const super_precision &a, int32 t[]) const;// >unpack
	void super_repk ( super_precision &a, int32 t[]) const;	// >repack
	void super_zmu ( super_precision &a, const super_precision &b,
				int32) const;			// >a = b * i
	void super_zqmnc ( int32 a[], int32, int32, int32) const;// inline-mnc
	void super_zqdnc ( int32 a[], int32, int32, int32, int32 &) const;
								// inline-dnc
	inline void super_zero ( super_precision &) const;	// a = 0
	void super_zqncc ( int32 a[], int32, int32, int32) const;// inline-ncc
	void super_zqanc ( int32 a[], int32 b[], int32 c[],
				int32, int32, int32 &) const;	// inline-anc
	void super_zqsnc ( int32 a[], int32 b[], int32 c[],
				int32, int32, int32 &) const;	// inline-snc
	void super_zqm1 ( int32 t[], int32 v[], int32,
				int32, int32 &, int32) const;	// inline-m1
	void super_zqm2 ( int32 t[], int32 v[], int32,
				int32 &) const;			// inline-m2
	void super_zqm2lsh ( int32 t[], int32 v[], int32,
				int32 &, int32) const;		// inline-m2lsh
	void super_zqnc	( int32 t[], int32, int32, int32 &,
				int32) const;			// inline-nc
	void super_zqupk ( int32 t [], int32 a [], int32, 
				int32) const;			// inline-unpk
	void super_zqrpk ( int32 a [], int32 t [], int32,
				int32 &, int32 &, int32 &,
				int32) const;			// inline-repk
	void super_zqdi ( int32 t [], int32, int32, int32) const;// inline-di
	void super_zqd1 ( int32 v [], int32 ia [], int32 ib [],
				int32 &, int32, int32, int32) const;// inline-d1
	void super_zqd2 ( int32 t [], int32 v [], int32 ia [],
				int32 &, int32, int32, int32) const;// inline-d2
	void super_zqncb ( int32 a [], int32, int32, int32) const;// inline-ncb
	static void change_mantissa_length
			( int32 nf = 0, int32 ndg = 0);	// change default length
	void super_chop_register ( void);		// chop reg precision
	super_precision convert_precision
			(const super_precision &) const;	//cnvt precision
	int32 super_ztoi ( const super_precision &) const;	// convert i=a
	double super_ztod_approx ( const super_precision &) const;// convert d=a
	super_precision super_dtoz_approx ( const double) const;  // convert a=d
	super_precision super_base_conv ( const super_precision &,
			const int32, const int32,
			const int32) const; 			// convert base
	double super_ztod_exact ( const super_precision &) const;
								// convert d=a
	super_precision super_dtoz_exact ( const double) const;	// convert a=d
	void super_dunpk ( const int32 a [], double t [],
			const int32, const int32, const int32) const;
								// fast_mult
	void super_drepk ( int32 t [], const double dalt [],
			const int32, const int32) const;	// fast_mult
	void super_fft_convol ( const super_precision &,
			const super_precision &, int32 t []) const;
							// FFT convolution
	void array_fft_convol ( int32 t [], int32 v [],
			int32 zia [], int32, int32, int32)
			const;				      // FFT convolution
	static char *validity_type_string ( int);		// return descrp
	static int super_irreg_fsa ( char *, int, int32 sgn = 0,
				int i = -2, int32 is = 0);	// irreg ops 

};			// END OF super_precision CLASS DEFINITION

class sp_int32						{ // CLASS DEFINITION
private:
	int32 element;					// only data element
	void * operator new ( size_t);			// overload new
	void operator delete ( void *);			// overload delete
#ifdef placement_new_array
	void * operator new [] ( size_t, int32);	// overload new []
	void operator delete [] ( void *);		// overload delete []
#endif
	friend class super_precision;		// permit use by friend class

};			// END OF sp_int32 CLASS DEFINITION

void *sp_int32 :: operator new				// OVERLOAD new
				( size_t size)
{
	void *ptr;
	ptr = malloc ( size);			// allocate memory block
//cout<<"new called with size="<<size<<endl;
	if ( !ptr)
	{
		cout << "Free Store exhausted in super_precision operator new";
		cout << " -- fatal error.\n";
		exit (1);
	}
	return ptr;
}

void sp_int32 :: operator delete			// OVERLOAD delete
				( void *ptr)
{
	free ( ptr);			// free memory associated with ptr
	ptr = 0;			// set ptr to NULL
}

// Optional code for newer compilers that recognize placement new[].
#ifdef placement_new_array
void *sp_int32 :: operator new []			// OVERLOAD new[]
				( size_t size, int32 count)
{
	void *ptr;
	ptr = malloc ( size * count);		// allocate memory block
//cout<<"new[] called with size="<<size<<endl;
	if ( !ptr)
	{
		cout << "Free Store exhausted in super_precision operator new";
		cout << "[] -- fatal error.\n";
		exit (1);
	}
	return ptr;
}

void sp_int32 :: operator delete []			// OVERLOAD delete[]
				( void *ptr)
{
	free ( ptr);			// free memory associated with ptr
	ptr = 0;			// set ptr to NULL
}
#endif

super_precision :: super_precision ( void)		// CLASS CONSTRUCTOR
{
//cout << "in constructor: nf=" << nf << " &nf=" << &nf << endl;
	sign_bias_exp = 0;
	digits = def_digits;
	radix = def_radix;
	validity = def_validity;
//	mantissa = new int32 [ nf];	// allocate int32 array for mantissa
				// allocate sp_int32 array for int32 mantissa
	mantissa = (int32*) new sp_int32 [ nf];
if(debug)cout << "constructor: mantissa =" <<  (int32) this << " " << (int32) mantissa << endl;
	if ( !mantissa)
	{
		cout << "Exhausted Free Store in super_precision constructor";
		cout << " -- fatal error.\n";
		exit (1);
	}
}

super_precision :: ~super_precision ( void)		// CLASS DESTRUCTOR
{
if(debug)cout << "Destructing " << (int32) this << " " << (int32) mantissa << endl;
	delete mantissa;		// free memory used for mantissa array
}

void *super_precision :: operator new			// OVERLOAD new
				( size_t size)
{
	void *ptr;
	ptr = malloc ( size);		// allocate memory block
	if ( !ptr)
	{
		cout << "Free Store exhausted in super_precision operator new";
		cout << " -- fatal error.\n";
		exit (1);
	}
	return ptr;
}


void super_precision :: operator delete			// OVERLOAD delete
				( void *ptr)
{
	free ( ptr);			// free memory associated with ptr
	ptr = 0;			// set ptr to NULL
}

// Optional code for newer compilers that recognize placement new[].
#ifdef placement_new_array
void *super_precision :: operator new []		// OVERLOAD new
				( size_t size, int32 count)
{
	void *ptr;
	ptr = malloc ( size * count);		// allocate memory block
	if ( !ptr)
	{
		cout << "Free Store exhausted in super_precision operator new";
		cout << "[] -- fatal error.\n";
		exit (1);
	}
	return ptr;
}

void super_precision :: operator delete []		// OVERLOAD delete[]
				( void *ptr)
{
	free ( ptr);			// free memory associated with ptr
	ptr = 0;			// set ptr to NULL
}
#endif

super_precision :: super_precision		// OVERLOAD copy-constructor
				( const super_precision &a)
{
	register int32 i;			// temporary

	sign_bias_exp = a.sign_bias_exp;
	digits = a.digits;
	radix = a.radix;
	validity = a.validity;
//	mantissa = new int32 [ nf];
	mantissa = (int32*) new sp_int32 [ nf];	// allocate sp_int32 array for
						// int32 mantissa
//if(debug)cout << "copy-constructor: mantissa =" << (int32) this << " " << (int32) mantissa << endl;
	if ( !mantissa)
	{
		cout << "Exhausted Free Store in super_precision ";
		cout << "copy-constructor -- fatal error.\n";
		exit (1);
	}
	for ( i = 0; i < nf; i++)  mantissa [i] = a.mantissa [i];
	
	return;
}

 
super_precision :: super_precision 			//CONVERSION CONSTRUCTOR
				( const int32 i)
{
	this->sign_bias_exp = 0;
	this->digits = def_digits;
	this->radix = def_radix;
	this->validity = def_validity;
//	this->mantissa = new int32 [ nf];	// allocate int32 array
	mantissa = (int32*) new sp_int32 [ nf];	// allocate sp_int32 array for
						// int32 mantissa
//if(debug)cout << "iconversion: mantissa =" << (int32) this << " " << (int32) mantissa << endl;
	if ( !this->mantissa)
	{
		cout << "Exhausted Free Store in super_precision ";
		cout << "int-conversion constructor -- fatal error.\n";
		exit (1);
	}

	super_itoz ( *this, i);		// Convert integer to super_precision.
}

super_precision :: super_precision 			//CONVERSION CONSTRUCTOR
				( const double d)
{
	this->sign_bias_exp = 0;
	this->digits = def_digits;
	this->radix = def_radix;
	this->validity = def_validity;
//	this->mantissa = new int32 [ nf];	// allocate int32 array
	mantissa = (int32*) new sp_int32 [ nf];	// allocate sp_int32 array for
						// int32 mantissa
//if(debug)cout << "dconversion: mantissa =" << (int32) this << " " << (int32) mantissa << endl;
	if ( !this->mantissa)
	{
		cout << "Exhausted Free Store in super_precision ";
		cout << "double-conversion constructor -- fatal error.\n";
		exit (1);
	}

	*this = super_dtoz_exact ( d);	// Convert double to super_precision.
}

super_precision :: super_precision 			//CONVERSION CONSTRUCTOR
				( const char *chstring)
{
	int string_base;

	this->sign_bias_exp = 0;
	this->digits = def_digits;
	this->radix = def_radix;
	this->validity = def_validity;
//	this->mantissa = new int32 [ nf];	// allocate int32 array
	mantissa = (int32*) new sp_int32 [ nf];	// allocate sp_int32 array for
						// int32 mantissa
//if(debug)cout << "dconversion: mantissa =" << (int32) this << " " << (int32) mantissa << endl;
	if ( !this->mantissa)
	{
		cout << "Exhausted Free Store in super_precision ";
		cout << "string-conversion constructor -- fatal error.\n";
		exit (1);
	}

        *this = super_convert_string ( chstring, string_base);
}

class super_precision_pool_ptr {			// CLASS DEFINITION
protected:
	super_precision *pointer;	// pointer to super_precision
	int pool_index;			// index to pool (0 to n_members-1)
	static int max_members;		// maximum pool size
	static int n_members;		// current pool size
	static int idle_top;		// index to top of idle stack
	static int idle [MAXPOOLSIZE];	// idle stack for pool pointers
	static super_precision *pool [MAXPOOLSIZE];	// pool pointer array
	static int32 pool_checksum;	// checksum for pool pointers
public:
	super_precision_pool_ptr ( void);			// constructor
	~super_precision_pool_ptr ( void);			// destructor
	static void pool_ptr_setup ( const int32 i = 0);	// initializer
	super_precision & operator* ( void);			// unary *
	static int32 checksum_pool_ptr ( void);		// check pool integrity
	static int32 disclose_pool_allocation_count ( void); //show ptrs alloc'd
};			// END OF super_precision_pool_ptr CLASS DEFINITION

super_precision_pool_ptr :: super_precision_pool_ptr ( void)	// CLASS CONSTRU
{
// This constructor pulls the top entry off the idle stack and uses that index
// into the pointer pool.  If the stack is empty, a new pool member is created,
// unless the maximum pool size would be exceeded.
// Although we might keep only a list of idle pool pointers (relying on the
// destructor to restore pool pointers to the idle stack), the separate list of
// all pool pointers (pool[]) allows a checksum to be done on the pool pointers
// to verify the pool integrity, by accessing static member function
// checksum_pool_ptr ().
// The mantissa size allocated is "register-size", meaning the original mantissa
// precision plus 4 additional mantissa words -- to allow small temporary
// increases by member and friend functions that need extra to maintain accuracy
// at the working precision level.

	if ( idle_top >= 0)
	{
				// Use index at top of idle stack.
		int i = idle [idle_top];
				// Found idle pool pointer.
		pointer = pool [i];
		pointer->radix = super_precision :: def_radix;
		pointer->digits = super_precision :: def_digits;
		pointer->validity = super_precision :: def_validity;
		pool_index = i;
				// Decrement idle stack size.
		idle_top--;
//cout << "found idle pool member: pool,i=" << (int32) pool[i]<<" "<<i<<endl;
//cout << "	idle_top=" << idle_top << endl;
		return;
	}
	else
	{
				// Check for maximum pool size.
		if ( n_members == max_members)
		{
			cout << "Pool allocation for super_precision_pool_ptr";
			cout << " class exceeds limit of " << max_members;
			cout << "objects,\nset by file SUPER.H --";
			cout << " fatal error.\n";
			exit (1);
		}

			// Add another super_precision to the pool.
		n_members++;
			// Boost mantissa to register-size.
		int i = super_precision :: nf;
		super_precision :: nf = super_precision :: orig_nf + 4;
			// Allocate new pool member.
		pool [n_members - 1] = new super_precision;
			// Restore mantissa length.
		super_precision :: nf = i;
		pointer = pool [n_members - 1];
		pool_index = n_members - 1;
		pool_checksum = checksum_pool_ptr ();
			// Leave idle_top equal to -1 (what it was already).
		idle_top = -1;
//cout << "Growing the pool to " << n_members << " members.\n";
//cout << "pointer=" << (int32) pointer<<endl;
		return;
	}
}

super_precision_pool_ptr :: ~super_precision_pool_ptr ( void)  // CLASS DESTRUCT
{
// This destructor adds the pool pointer index to the idle stack and verifies
// the pointer is uncorrupted by comparing to the list of pool pointers.

	idle [ ++idle_top] = pool_index;
//cout << "idled pool_index="<<pool_index<<" idle_top="<<idle_top<<endl;
//cout << "	pointer="<<(int32)pointer<<" pool="<<(int32)pool[pool_index]<< endl;
	if ( pointer != pool [pool_index])
	{
		// Corrupted pool pointer.
		cout << "A pointer for a super_precision_pool_ptr class object";
		cout << " has been corrupted -- fatal error";
		cout << "\n -- detected while destructing pool_index =";
		cout << pool_index << endl;
		// int32 super_precision_pool_ptr :: pool_checksum; STATIC
                int32 checksum = super_precision_pool_ptr :: pool_checksum;
		if ( checksum && ( checksum == checksum_pool_ptr ()))
		{
			cout << "Pool checksum still valid.\n";
		}
		else
		{
			cout << "Pool checksum invalid -- pool corrupted.\n";
		}
		exit (1);
	}
}

super_precision & super_precision_pool_ptr :: operator*		// OVERLOAD *
				( void)
{								// unary *a
// This function dereferences the pointer this.

	return  *pointer;
}

int32 super_precision_pool_ptr :: checksum_pool_ptr		// PUBLIC FN
				( void)
{							// check pool integrity
// This function performs a checksum on the super_precision pool pointers to
// allow a validity check.
        int32 sum = 0;
	if ( n_members <= 0)
	{
		cout << "Pool size invalid for super_precision_pool_ptr class";
		cout << ", n_members =" << n_members;
		cout << "\n -- fatal error.\n";
		exit (1);
	}
	else
	{
		int32 i;
		for ( i = 0; i < n_members; i++) sum += (int32) pool [i];
	}
	return sum;
}

int32 super_precision_pool_ptr :: disclose_pool_allocation_count   // PUBLIC FN
				( void)			// show max ptrs used
{	// PUBLIC STATIC FUNCTION
// This function (a static member of the super_precision_pool_ptr class) returns
// the number of pool pointers objects allocated at the time it is called.

	return super_precision_pool_ptr :: n_members;
}


class super_precision_userpool {			// CLASS DEFINITION
protected:
	super_precision *pointer;	// pointer to super_precision
	static int max_members;		// maximum pool size
	static int n_members;		// current pool size
	static int idle_top;		// index to top of idle stack
//	static super_precision *idle [MAXPOOLSIZE];//idle stack for pool ptrs
	static super_precision **idle;	//idle stack for pool ptrs
public:
	super_precision_userpool ( void);			// constructor
	~super_precision_userpool ( void);			// destructor
	static void setup ( const int32 i = 0);			// initializer
	super_precision & operator* ( void);			// unary *
	static int32 disclose_userpool_allocation_count ( void);
							//show ptrs alloc'd
};			// END OF super_precision_userpool CLASS DEFINITION

super_precision_userpool :: super_precision_userpool ( void)	// CLASS CONSTRU
{
// This constructor pulls the top entry off the idle stack.
// If the idle stack is empty, a new pool member is created,
// unless the maximum pool size would be exceeded.
// We keep only a list of idle pool pointers (relying on the destructor to
// restore valid pool pointers to the idle stack).

	if ( idle_top >= 0)
	{
				// Use index at top of idle stack.
		pointer = idle [idle_top];
				// Found idle pool pointer.
		pointer->radix = super_precision :: def_radix;
		pointer->digits = super_precision :: def_digits;
		pointer->validity = super_precision :: def_validity;
				// Decrement idle stack size.
		idle_top--;
//cout << "found idle pool member: idle_top=" << idle_top << endl;
		return;
	}
	else
	{
				// Check for maximum pool size.
		if ( n_members == max_members)
		{
			cout << "Pool allocation for super_precision_userpool";
			cout << " class exceeds limit of " << max_members;
			cout << " objects,\nset by super_precision_userpool::";
			cout << "setup -- fatal error.\n";
			exit (1);
		}

			// Add another super_precision to the pool.
		n_members++;
			// DO NOT boost mantissa to register-size for userpool.
			// Allocate new pool member.
		pointer = new super_precision;
			// Leave idle_top equal to -1 (what it was already).
		idle_top = -1;
//cout << "Growing the userpool to " << n_members << " members.\n";
//cout << "pointer=" << (int32) pointer<<endl;
		return;
	}
}

super_precision_userpool :: ~super_precision_userpool ( void)  // CLASS DESTRUCT
{
// This destructor adds the pool pointer index to the idle stack.

	idle [ ++idle_top] = pointer;
//cout << "Idled pointer="<<(int32)pointer<<" idle_top="<<idle_top<< endl;
}

super_precision & super_precision_userpool :: operator*		// OVERLOAD *
				( void)
{								// unary *a
// This function dereferences the pointer this.

	return  *pointer;
}

int32 super_precision_userpool :: disclose_userpool_allocation_count //PUBLIC FN
				( void)			// show max ptrs used
{	// PUBLIC STATIC FUNCTION
// This function (a static member of the super_precision_userpool class) returns
// the number of pool pointers objects allocated at the time it is called.

	return super_precision_userpool :: n_members;
}


//
// CLASSLESS PROTOTYPES.
//
// Here are prototypes for two classless functions used for fast multiplies.
int f4t ( double *w, int32 ntotal, int32 n, int32 nprev, int32 is);
int jamcor ( double *w, int32 n, int32 ii);


//
// FORCE ALLOCATION OF STATIC VARIABLES:
//
		// Some C++ compilers require class static variables to be
		// declared at file scope in order to get their memory
		// allocated.  The next file-scope declarations are intended
		// for use with such.  (Other C++ compilers such as GCC
		// do not require the file-scope declaration, but are happy
		// with declarations in a static function such as "setup".
	// STATIC ALLOCATION FOR super_precision CLASS:
int32 super_precision :: def_digits;
short super_precision :: def_radix;
short super_precision :: def_validity;
int32 *super_precision :: zia;
int32 *super_precision :: zib;
int32 *super_precision :: t;
int32 *super_precision :: v;
double *super_precision :: dalt;
int32 super_precision :: rounding_method;
int32 super_precision :: ndig;
int32 super_precision :: nd;
int32 super_precision :: ind;
int32 super_precision :: bias;
int32 super_precision :: nf;
int32 super_precision :: nfa;
int32 super_precision :: nfah;
int32 super_precision :: nfahx;
int32 super_precision :: ndah;
int32 super_precision :: indah;
int32 super_precision :: nda;
int32 super_precision :: inda;
int32 super_precision :: naccum;
int32 super_precision :: hdwr_base;
int32 super_precision :: nbit;
int32 super_precision :: klip;
int32 super_precision :: iklip;
int32 super_precision :: indtrz;
int32 super_precision :: ndiv;
int32 super_precision :: irbit;
double super_precision :: baslog;
int32 super_precision :: orig_def_digits;
int32 super_precision :: orig_nf;
int32 super_precision :: max_output_string_size;
int32 super_precision :: max_input_string_size;
int32 super_precision :: block;
int32 super_precision :: nblanks;
int32 super_precision :: output_base;
int32 super_precision :: output_count;
int32 super_precision :: input_base;
int super_precision :: conv_input_base;
int super_precision :: std_ios_manip;
int super_precision :: show_output_base;
int32 super_precision :: fast_mult_digits;
int32 super_precision :: dinda;
int32 super_precision :: dnfa;
int32 super_precision :: dnda;
int32 super_precision :: radix_powerof2;
int32 super_precision :: nfft;
double super_precision :: double_fraction_max;
int32 super_precision :: double_fraction_digit_count;
int32 super_precision :: fast_div_digits;
int32 super_precision :: log_Salamin_threshold;
int32 super_precision :: exp_Newton_threshold;
int32 super_precision :: atan_Salamin_threshold;
int32 super_precision :: tan_Newton_threshold;
int super_precision :: perfect_fast_divide;
int super_precision :: permissive_error;
int32 super_precision :: debug_counter = 0;
int32 super_precision :: debug_counter_divides = 0;
super_precision *super_precision :: w;
super_precision *super_precision :: zt;
super_precision *super_precision :: zv;
super_precision *super_precision :: zz;
super_precision *super_precision :: pi;
super_precision *super_precision :: log_10;
super_precision *super_precision :: log_radix;
char *super_precision :: output_result;
char *super_precision :: input_string;
int super_precision :: pi_init = 0;
int super_precision :: log_10_init = 0;
int super_precision :: log_radix_init = 0;
int super_precision :: prev_radix = 0;
int32 super_precision :: radix_powers [32];
	// STATIC ALLOCATION FOR super_precision_pool_ptr CLASS:
super_precision *super_precision_pool_ptr :: pool [MAXPOOLSIZE];
int super_precision_pool_ptr :: idle [MAXPOOLSIZE];
int super_precision_pool_ptr :: max_members;
int super_precision_pool_ptr :: n_members;
int super_precision_pool_ptr :: idle_top;
int32 super_precision_pool_ptr :: pool_checksum;
	// STATIC ALLOCATION FOR super_precision_userpool CLASS:
super_precision **super_precision_userpool :: idle;
int super_precision_userpool :: max_members;
int super_precision_userpool :: n_members;
int super_precision_userpool :: idle_top;
// END OF STATIC ALLOCATIONS AT FILE SCOPE.


void super_precision :: setup					// INITIALIZER
				( const int32 digits_, const short radix_)
{
// This function must define all static variables in the class.
	int32 i, j, k, l, m, n;		// 32-bit temporaries
	int32 nbitx;			// 32-bit temporaries
	double x, y, z;			// double temporaries
	int good_finish;		// boolean temporary

	char signature [] = "Super Precision Class Library, Version 2, 11/1/97";
	// Allocate super_precision class static members at file scope.
	// But initialize some of them here --------
	// int32 super_precision :: def_digits;
	// short super_precision :: def_radix;
	// short super_precision :: def_validity;
	// int32 *super_precision :: zia;
	// int32 *super_precision :: zib;
	// int32 *super_precision :: t;
	// int32 *super_precision :: v;
	// double *super_precision :: dalt;
	// int32 super_precision :: rounding_method;
	// int32 super_precision :: ndig;
	// int32 super_precision :: nd;
	// int32 super_precision :: ind;
	// int32 super_precision :: bias;
	// int32 super_precision :: nf;
	// int32 super_precision :: nfa;
	// int32 super_precision :: nfah;
	// int32 super_precision :: nfahx;
	// int32 super_precision :: ndah;
	// int32 super_precision :: indah;
	// int32 super_precision :: nda;
	// int32 super_precision :: inda;
	// int32 super_precision :: naccum;
	// int32 super_precision :: hdwr_base;
	// int32 super_precision :: nbit;
	// int32 super_precision :: klip;
	// int32 super_precision :: iklip;
	// int32 super_precision :: indtrz;
	// int32 super_precision :: ndiv;
	// int32 super_precision :: irbit;
	// double super_precision :: baslog;
	/* int32 */ super_precision :: orig_def_digits = 0;
	/* int32 */ super_precision :: orig_nf = 0;
	// int32 super_precision :: max_output_string_size;
	// int32 super_precision :: max_input_string_size;
	/* int32 */ super_precision :: block = 0;
	// int32 super_precision :: nblanks;
	/* int32 */ super_precision :: output_base = 0;
	/* int32 */ super_precision :: output_count = 0;
	/* int32 */ super_precision :: input_base = 0;
	/* int */ super_precision :: conv_input_base = 1; //convert input to
								// def_base=TRUE
	/* int */ super_precision :: std_ios_manip = 1;   // set public var TRUE
	/* int */ super_precision :: show_output_base = 0;// don't show base on
								// output
	// int32 super_precision :: fast_mult_digits;
	// int32 super_precision :: dinda;
	// int32 super_precision :: dnfa;
	// int32 super_precision :: dnda;
	// int32 super_precision :: radix_powerof2;  // =0 if non-powerof2 or
							// =log2(radix) elsewise
	// int32 super_precision :: nfft;
	// double super_precision :: double_fraction_max;
	// int32 super_precision :: double_fraction_digit_count;
	// int32 super_precision :: fast_div_digits;
	// int32 super_precision :: log_Salamin_threshold;
	// int32 super_precision :: exp_Newton_threshold;
	// int32 super_precision :: atan_Salamin_threshold;
	// int32 super_precision :: tan_Newton_threshold;
	/* int */ super_precision :: perfect_fast_divide = 1;	// default TRUE
	/* int */ super_precision :: permissive_error = 0;	// default FALSE
	/* int32 */ super_precision :: debug_counter = 0;
	// int32 super_precision :: debug_counter_divides = 0;
	// int32 super_precision :: radix_powers [32];
	// End of initialization of static members.

	// Validity check on parameters.
			// digits_ must be 2 or more.
	if ( digits_ < 2)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Illegal number of digits chosen, must be >= 2";
		cout << ", value chosen =" << digits_ << endl;
		exit (1);
	}
			// radix_ must be 2 to 16.
	if ( radix_ < 2 || radix_ > 16)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Illegal radix (number base) chosen, must be 2 to 16,";
		cout << " value chosen = " << radix_ << endl;
		exit (1);
	}

	def_digits = digits_;		// no. of digits specified = NDIGA
	def_radix = radix_;
	def_validity = 0;	// validity = 0 for properly formed.
				// NOTE: validity = 1 for +infinity
				//       validity = 2 for -infinity
				//       validity = 3 for +-infinity
				//       validity = 4 for NaN (not a number)
				//       validity = -1 for unnormalized

	rounding_method = TRUE;		// TRUE for rounded results
			// NOTE: We use "round-to-even" in addition and
			// subtraction when the digit being rounded is
			// ambiguous -- such as "5" in the decimal system.

	// Calculate the number of useful bits in the integer word, and compare
	// it with the number specified to be used:
	nbitx = 1;
	i = 1;
	j = 2;
	while ( ( i != 0) && ( (i*j)/j == i))
	{
		i *= 2;
		nbitx++;
	}
	if ( nbitx < 31)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Number of useful bits less sign = " << nbitx;
		cout << ", not 31 or more." << endl;
		exit (1);
	}
	// Save in static datum nbit:
	nbit = nbitx;

	x = (int) signature [0];

	// Detect the hardware floating-point arithmetic base:
	x = 1;
	do
	{
		x = x + x + 1;
		y = x - (double) (float) x;
	} while ( y == 0);
	hdwr_base = (int32) (y + 1);
	if ( hdwr_base)
	{
		// Truncated float-typecast:
		if ( verbose)
		{
			cout << "Hardware floating-point arithmetic base = ";
			cout << hdwr_base << " (truncated float-typecast).\n";
		}
	}
	else
	{
		// Rounded float-typecast:
		hdwr_base = 2;			// default if this doesn't work
		y = (float) x;
		for ( i = 1; i < 9; i++)
		{
			if ( y == (double) (float) ( x - i)) continue;
			hdwr_base = i + i;	// it worked; 2*i is hdwr_base
			if ( verbose)
			{
			cout << "Hardware floating-point arithmetic base = ";
			cout << hdwr_base << " (rounded float-typecast).\n";
			}
			break;
		}
	}

	// Verify that integer arithmetic can support the number range:
	j = 1;
	m = j;
	k = nbit - 2;
	while ( k--)
	{
		j = j + j;
		if ( j / 2 == m)
		{
			m = j;
			continue;
		}
		else
		{
		cout << "Fatal error detected in super_precision::setup --\n";
			cout << "Integer hardware test fails at k = ";
			cout << k << endl;
			exit (1);
		}
	}
	m = 3 * j + ( j - 1);
	if ( (( m - 3) / 4 + 1) != j)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Integer hardware test fails at m = " << m << endl;
		exit (1);
	}
	// Passed tests.  Set bias.
	bias = j + ( j - 1);
	if ( verbose)
	{
		cout << "bias = " << bias << endl;
	}
	
	// Calculate number of digits (to base def_radix) per mantissa word:
	k = bias + 1;
	i = 0;
	do
	{
		i++;
		k /= def_radix;
	} while ( k >= def_radix);
	// Save the digit count in static datum nd.
	nd = i;
//cout << "nd = " << nd << endl;
	if ( nd < 2)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "static variable nd < 2";
		exit (1);
	}

	// See if the radix is a power of 2.
	i = 1;
	j = 0;
	radix_powerof2 = 0;
	while ( i < 16)
	{
		i += i;
		j++;
		if ( i == def_radix) radix_powerof2 = j;
	}

	// Calculate and save def_radix**nd in static datum ind:
	ind = 1;
	i = nd;
	while ( i--) ind *= def_radix;
	// Calculate radix_powers:
	for ( i = 1, radix_powers [0] = 1; i <= nd; i++)
	{
		radix_powers [i] = radix_powers [i-1] * def_radix;
	}		// Now  pow(def_radix,n) == radix_powers[n]

	// Calculate static datum nf = number of elements in mantissa array:
	n = ( def_digits + nd - 1) / nd;
	nf = n;
	if ( verbose)
	{
		cout << "nf = " << nf << endl;
	}

	orig_nf = nf;			// Save it in a static variable.
	orig_def_digits = def_digits;	// Save this too, while we're here.

	// Calculate ndig = max. number of digits carried in mantissa array:
	ndig = nf * nd;

	// Define some conversion constants:
	baslog = log10 ( (double) def_radix);

	// This is a good place to compute fast_mult_digits and fast_div_digits,
	// using a figure of about 500 decimal digits as the threshold between
	// the standard (n*n) algorithms and the FFT-based (n*log n)
	// algorithms.  The threshold can be redefined by the public functions
	// set_fast_multiply_threshold() and set_fast_divide_threshold().
	// This default threshold is based on using a 80486, and is only a
	// guideline obtained using gcc-compiled executables.
	fast_div_digits = fast_mult_digits = (int32) ( 500.5 / baslog);

	// Let's go ahead and set the Salamin thresholds, too.  These values
	// are probably not optimum -- no effort has been made to optimize them.
	log_Salamin_threshold = exp_Newton_threshold = 20 * fast_div_digits;
        atan_Salamin_threshold = tan_Newton_threshold = 20 * fast_div_digits;

	// Calculate static datum naccum:
	l = bias + bias + 1;
	// Increase n by 4 to allow guard words for some routines.
	n += 4;
		// Save n and substitute the fast mult/div thresholds, if
		// they are smaller than n.
		int32 nsave = n;
		n = ( fast_mult_digits + nd - 1) / nd;
		n = ( n < nsave) ? n : nsave;
	naccum = nd % 2;
	good_finish = FALSE;
	do
	{
		i = nd - naccum;
		m = i / 2;
//		k = 1;
//		while ( m--) k *= def_radix;	// k = def_radix ** (i/2)
		k = radix_powers [m];		// k = def_radix ** (i/2)
		// In case intrz == 1 later on:
		k = ( def_radix <= k - 1) ? k * ( k - 1) : k * def_radix;
		m = l / k;
		j = ( n * nd + i - 1) / i;
		k = j + j;
		if ( i / 2 <= j * i - nd * n) k--;
		if ( ( nd * n) % ( i / 2) == 0) m += 1 - def_radix;
		if ( m >= k)
		{
			good_finish = TRUE;
			break;
		}
		naccum += 2;
	} while ( naccum + 2 <= nd);   // "+" fixes bug "*" in Fortran original.
	if ( !good_finish)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Too many digits requested for radix and word length.";
		cout << "\nTry a smaller radix. def_digits = " << def_digits;
		cout << "  def_radix = " << def_radix << endl;
		exit (1);
	}

	// Calculate static data nda and inda:
	nda = nd - naccum;
//	m = nda;
//	inda = 1;
//	while ( m--) inda *= def_radix;		// inda = def_radix ** nda
	inda = radix_powers [nda];		// inda = def_radix ** nda

	// Calculate and test ndah = number of digits in unpacked word:
	ndah = nda / 2;
	if ( nda % 2 != 0)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "nda is an odd number -- internal error. nda = " << nda;
		cout << endl;
		exit (1);
	}
			// If def_radix is 2, ndah must be 3 or more -- else the
			// "perfect fast divide" correction loop will fail.
	if ( ndah < 3 && def_radix == 2)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Too many binary digits requested, limit is about ";
		cout << "357,000,000 binary digits." << endl;
		cout << "Value requested =" << def_digits << endl;
		exit (1);
	}

	// Calculate indah = def_radix ** ndah:
//	i = ndah;
//	indah = 1;
//	while ( i--) indah *= def_radix;
	indah = radix_powers [ndah];		// indah = def_radix ** ndah
	
		// Okay, restore n that we saved.
		n = nsave;
	
	// Restore n to original value.
	n -= 4;

	// Calculate clipping factor to truncate/round mantissa to def_digits:
	klip = def_digits;			// default to clipping
	m = ndig - def_digits;
//	iklip = 1;
//	while ( m--) iklip *= def_radix;	// iklip = def_radix ** m;
	iklip = radix_powers [m];		// iklip = def_radix ** m

	// Calculate nfa = length of repacked mantissa:
	nfa = ( ndig + nda - 1) / nda;

	// Calculate nfah = length of unpacked mantissa:
	nfah = 2 * nfa;

	// Calculate nfahx = number of nonzero words in unpacked mantissa:
	i = nfa * nda - ndig;		// Make sure ndah*nfahx >= ndig.
	nfahx = nfah;
	if ( i >= ndah) nfahx--;
	// Calculate indtrz = clipping factor for repacked mantissa:
//	indtrz = 1;
//	while ( i--) indtrz *= def_radix;	// indtrz = def_radix ** i
	indtrz = radix_powers [i];		// indtrz = def_radix ** i
				// indtrz is used to define roundoff

	// Test for multiplication accuracy:
/*	i = indah * ( indah - 1);
	j = i * nfahx;
	if ( i != j / nfahx)
	{
		cout << "Fatal error -- product accumulation fails. Weird.\n";
		exit (1);
	}
*/

	// We will not use the NCOL variable, so rounding will be exact for even
	// radices.  This corresponds to the original case iround < 0 in ZMULT.

	// Test for accuracy of exact conversion routines:
	if ( hdwr_base > indah)
	{
		cout<< "Nonfatal error detected in super_precision::setup --\n";
		cout << "Warning -- hdwr_base > indah -- do not use exact ";
		cout << "conversion routines:\n";
		cout << "hdwr_base=" << hdwr_base << "  indah=" << indah <<endl;
		exit (1);
	}

	// Determine how many digits (of the base def_radix) can be handled
	// exactly in type-double working.
	double dtemp [60];			// See note below.
	y = 1;
	i = -1;
	do
	{
		i++;
		y *= def_radix;
			// If we use the following, compiler optimizations may
			// cause the RHS to be done entirely in the floating
			// point coprocessor at long double precision, without
			// memory references to limit it to double precision.
			// We must avoid this somehow.
//		z = y - ( y - 3);		// Force y-3 into memory.
		dtemp [i] = y - 3;		// Try to defeat optimization.
	} while ( i < 60);
	y = 1;
	i = -1;
	do
	{
		i++;
		y *= def_radix;
		z = y - dtemp [i];		// Ditto.
	} while ( z == 3);			// Okay. This works for gcc -O2.
	double_fraction_digit_count = i;

			// Now do it again for hardware base (usually binary).
	y = 1;
	i = -1;
	do
	{
		i++;
		y *= hdwr_base;
		dtemp [i] = y - 3;		// Try to defeat optimization.
	} while ( i < 60);
	y = 1;
	i = -1;
	do
	{
		i++;
		y *= hdwr_base;
		z = y - dtemp [i];		// Ditto.
	} while ( z == 3);			// Okay. This works for gcc -O2.
			// Have hardware digit count, now find maximum.
	j = i;
	y = 1;
	while ( j--) y *= hdwr_base;
	x = y / hdwr_base;
	if ( hdwr_base * x != y)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "Multiply-divide error in double_fraction_max";
		cout << "computation.  i=" << i << endl;
		exit (1);
	}
	double_fraction_max = y;

		// Reasonable limit is 53 bits for radix 2 for machines using
		// IEEE format for double.  If long double is used instead,
		// look for an upper limit of 64 bits on Intel's or 106 bits on
		// IBM RISC's.
	i = double_fraction_digit_count;
	ndiv = ( 53 > i / ndah) ? i / ndah : 53;	// In case of screwup.
	// ndiv = approx. number of t-array elements in quotient estimator:
	ndiv = i / ndah;
	j = ( 2 * bias + 1) / ( ( indah - 1) * ( indah - 1));// 16-bit int relic
	ndiv = ( ndiv > j) ? j : ndiv;
	if ( ndiv <= 1)
	{
		cout << "Fatal error detected in super_precision::setup --\n";
		cout << "ndiv < 2    double is only " << i;
		cout << " digits and ndah is " << ndah << endl;
		exit (1);
	}

	// If the radix has changed from a previous call to setup, clear stuff.
	if ( super_precision :: prev_radix &&
			( super_precision :: prev_radix != def_radix))
	{
		// Clear any previous calculation of pi, log_10, log_radix.
		super_precision :: pi_init = 0;
		super_precision :: log_10_init = 0;
		super_precision :: log_radix_init = 0;
	}

	// Allocate storage for internal super_precision registers:
	j = nf;			// save nf value
	nf = nf + 4;		// nf is used by the object constructor
	if ( super_precision_setup_called < nf - 4)
	{	// Need bigger mantissa arrays for this precision.
		if ( super_precision_setup_called)
		{	// Delete previous storage first.
			delete super_precision :: w;
			delete super_precision :: zt;
			delete super_precision :: zv;
			delete super_precision :: zz;
			// Discard old pool_ptr storage:
			super_precision_pool_ptr :: pool_ptr_setup ( -1);
			// Discard old userpool storage:
			super_precision_userpool :: setup ( -1);
			// Discard storage for pi, log_10, and log_radix.
			if ( super_precision :: pi)
					delete super_precision :: pi;
			if ( super_precision :: log_10)
					delete super_precision :: log_10;
			if ( super_precision :: log_radix)
					delete super_precision :: log_radix;
		}
		super_precision :: w = new super_precision;
		super_precision :: zt = new super_precision;
		super_precision :: zv = new super_precision;
		super_precision :: zz = new super_precision;
		super_precision_pool_ptr :: pool_ptr_setup ();
		// Wait until pi is needed to allocate storage for it.
		// Wait until log_10 is needed to allocate storage for it.
		// Wait until log_radix is needed to allocate storage for it.
		// But zero the mantissa pointers:
		super_precision :: pi = 0;
		super_precision :: log_10 = 0;
		super_precision :: log_radix = 0;
	}

	// Calculate size of output_result character array.
	max_output_string_size = 36 + nf * nd;
	i = super_set_output_block ( 10);		// Set blocking to 10.
	i = super_set_output_base ( 0);			// set o/p base=working
	i = super_set_output_count ( 0);		//set o/p digit=default
	i = super_set_input_base ( 0);			// set i/p base=working
	if ( super_precision_setup_called < nf - 4)
	{	// Need bigger mantissa arrays for this precision.
		if ( super_precision_setup_called)
		{			// Delete previous storage.
			delete super_precision :: output_result;
			delete super_precision :: input_string;
		}
					// Set pointers to NULL.
		/* char * */ super_precision :: output_result = 0;
		/* char * */ super_precision :: input_string = 0;
	}

		// Calculate length of unpacked registers.
	nf = 2 * ( ( nf * nd + nda - 1) / nda);
	nf++;		// Add one more element to accommodate fast_mult usage.
	if ( super_precision_setup_called < nf - 5)
	{
		if ( super_precision_setup_called) delete zia;
					// delete previous storage and get more.
					// zia and zib form double-length array.
		zia = new int32 [ 2 * nf];
		zib = zia + nf;
	}
	if ( super_precision_setup_called < nf - 5)
	{
		if ( super_precision_setup_called) delete t;
					// delete previous storage and get more.
					// t and v form double-length array.
		t = new int32 [ 2 * nf];
		v = t + nf;
	}
	nf = j;		// restore value of nf for mantissa array size

		// Calculate fast_mult parameters.
	dnda =  1 + double_fraction_digit_count / 2;
	do
	{
		dnda--;
		if ( !dnda)
		{
		cout << "Fatal error detected in super_precision::setup --\n";
			cout << "Failure of fast_mult parameter loop.";
			cout << "  double_fraction_digit_count=";
			cout << double_fraction_digit_count << endl;
			cout << "Requested maximum working digits ndig=";
			cout << ndig << endl;
			exit (1);
		}
			// Use nf+4 as maximum mantissa length.
		dnfa = 2 * ( ( ( nf + 4) * nd + dnda - 1) / dnda);
				// dnfa is doubled in size because we fill
				// alternate elements (fill reals or imaginaries
				// -- with real-imaginary alternation).
			// nfft will be FFT transform size.
		nfft = 1;
		while ( nfft < dnfa) { nfft *= 2;}
			// Last half of FFT input is zeros, since it yields a
			// circular convolution.
		x = double_fraction_max;
		x /= 1000;	// Safety margin for sincos accuracy at double
				// precision accuracy in f4t routine (good to
				// at least 750,000 decimal digits).
		x /= nfft / 2;	// FFT adds up nfft doubles but half are zero.
		y = 1;
		i = dnda;
		while ( i--) y *= def_radix;
	} while ( x < ( y - 1) * ( y - 1));	// Compare with largest addend.
			// Define for maximum digits in mantissa length nf.
	dnfa = 2 * ( ( ndig + dnda - 1) / dnda);
	dinda = (int32) y;			// Scale factor for fast_mult.
	if ( super_precision_setup_called < nf)
	{
		if ( super_precision_setup_called) delete dalt;
					// delete previous storage
					// Allocate real and imaginary.
		dalt = new double [ 2 * nfft];
	}


	// To allow certain memory allocation/deallocation do this LAST!
	// Reset global flag to signify setup has been called at least once, and
	// make saved value equal to largest mantissa size used.
	super_precision_setup_called = ( !super_precision_setup_called) ?
			nf : ( (super_precision_setup_called < nf) ? nf :
				super_precision_setup_called); 
	super_precision :: prev_radix = def_radix;

}		// End of setup function

void super_precision_pool_ptr :: pool_ptr_setup			// INITIALIZER
				( const int32 n)
{
// This function must define all static variables in the class.
// The argument n specifies the number of elements in the pool.
// If argument n is negative, delete all elements in the pool.
	int i;
	// Force allocation of class static variables.
	/* int32 */ super_precision_pool_ptr :: pool_checksum = 0;

	if ( n > MAXPOOLSIZE)
	{
		cout << "Fatal error in pool_ptr_setup -- requested ";
		cout << "count exceeds MAXPOOLSIZE." << endl;
		exit (1);
	}

	if ( n < 0)
	{
		// Delete the pool so it can be rebuilt at a longer precision.
		// Delete each member of the pool, but not the pointer array
		// itself.
		for ( i = 0; i < n_members; i++)
		{
			delete pool [i];
		}
		super_precision_pool_ptr :: n_members = 0;
		super_precision_pool_ptr :: idle_top = -1;
		return;
	}

				// declare pointer array at file scope
//	super_precision *super_precision_pool_ptr :: pool [MAXPOOLSIZE];
				// declare idle stack at file scope
//	int super_precision_pool_ptr :: idle [MAXPOOLSIZE];
				// define maximum pool size
	/* int */ super_precision_pool_ptr :: max_members = MAXPOOLSIZE;
				// define initial pool size
	/* int */ super_precision_pool_ptr :: n_members = n;
				// define index to top of idle stack
	/* int */ super_precision_pool_ptr :: idle_top = n_members - 1;
				// define pool_checksum for validity checks
//	int32 super_precision_pool_ptr :: pool_checksum;
	
	for ( i = 0; i < n_members; i++)
	{
		idle [i] = i;
		pool [i] = new super_precision;
	}
}

void super_precision_userpool :: setup				// INITIALIZER
				( const int32 n)
{
// This function must define all static variables in the class.
// The argument n specifies the number of elements in the pool.
// If argument n is negative, all idle members of userpool are delete'd.  It
// is important that all userpool members have passed out of scope when the
// idle stack is delete'd -- else strange things may begin to happen...
	int i;

	if ( n < 0)
	{
		// Delete userpool so it can be rebuilt (at longer precision?).
		// Delete each member of the userpool.
		for ( i = 0; i < n_members; i++)
		{
			delete idle [i];
		}
		delete [] idle;		// Then delete the idle stack itself.
		super_precision_userpool :: n_members = 0;
		super_precision_userpool :: idle_top = -1;
		return;
	}

				// declare idle stack of pointers
	super_precision_userpool :: idle = new super_precision * [n];
				// define maximum pool size
	/* int */ super_precision_userpool :: max_members = n;
				// define initial pool size
	/* int */ super_precision_userpool :: n_members = n;
				// define index to top of idle stack
	/* int */ super_precision_userpool :: idle_top = n_members - 1;
	
	for ( i = 0; i < n_members; i++)
	{
		idle [i] = new super_precision;
	}
}

void super_precision :: set_rounding_method			// PUBLIC FN
				( const int32 rounding_method_)
{
// This function redefines the static variable rounding_method.
// A value TRUE causes any results to be rounded off at the last digit carried,
// while a value FALSE causes truncation of any digits beyond the last digit
// carried.  For addition and subtraction, the round off is to the nearest even
// digit when the round digit is ambiguous -- i.e., halfway between 0 and radix
// (5 for decimal base).
	rounding_method = rounding_method_;
}

void super_precision :: change_mantissa_length			// PROTECTED FN
				( int32 new_nf, int32 new_def_digits)
{						//change default mantissa length
// After function setup is invoked, the working precision for mantissa arrays
// of new super_precision objects or of static super_precision "registers"
// (zz, zv, etc.) can be adjusted up to as much as 4 elements longer than the
// original default, or anything shorter except zero length.  This can be
// useful for running an intermediate calculation with more accuracy, so as
// to assure full accuracy at the default precision.
//
// For the change_mantissa_length function:
// Argument new_nf specifies a new mantissa array length, or, if zero, requests
// the original value declared by function setup to be restored.  The original
// value is available as the static variable orig_nf.  If new_nf is nonzero,
// then if new_def_digits is nonzero the latter is assigned to be the new
// def_digits.
// NOTE: When the mantissa is shortened or lengthened in this manner, the
// object's private datum "digits" is not altered, but remains as it was when
// the object was allocated.
//
// PROGRAMMING NOTE:	After increasing the default mantissa length with
// this function change_mantissa_length, care must be taken to zero-fill a
// lengthened mantissa before it is used.  This is done by initializing the
// new object to a small integer (a = 0;) or by using super_fill to load the
// added mantissa elements with the value zero.  Copying a "short" mantissa
// to a "long" mantissa can be done by adjusting the static class variable nf
// to the "short" mantissa length before invoking the super_move function, and
// restoring nf after using the super_move function.  The "long" mantissa
// must be zero-filled properly with super_fill.
// 	Likewise, after using change_mantissa_length to decrease the default
// mantissa length, care must be taken to "round" a longer mantissa down to 
// the new length before using it (copying it to another "normal-length"
// object, for example).  These round-down operations are done by accessing
// a private function super_chop_register for a static super_precision register
// object.
	int32 i, m;				// temporaries
	int32 nfmax;

	if ( new_nf < 0)
	{
		// Error -- negative argument prohibited.
		cout << "Fatal error in super_precision::change_";
		cout << "mantissa_length --" << endl;
		cout << " negative argument to change_mantissa_length ";
		cout << "not allowed.  arg =" << new_nf << endl;
		exit (1);
	}

	if ( !new_nf)
	{
		// Restore the original mantissa length if new_nf == 0.
		// If the saved value is zero, it was never really saved.
		if ( !orig_def_digits)
		{
			cout << "Fatal error in super_precision::change_";
			cout << "mantissa_length --" << endl;
			cout << "trying to change mantissa ";
			cout << "length before invoking setup function.";
			exit (1);
		}
		nf = orig_nf;
		def_digits = orig_def_digits;
		nfmax = orig_nf + 4;		// Local variable.
	}
	else
	{
		// Check new_nf to verify it is not too big.
		if ( new_nf > orig_nf + 4)
		{
			// Too big.
			cout << "Fatal error in super_precision::change_";
			cout << "mantissa_length --" << endl;
			cout << "trying to make new_nf too big.";
			cout << "  new_nf = " << new_nf;
			cout << " and orig_nf = " << orig_nf << endl;
			exit (1);
		}
		else
		{
			// Use the argument value as a mantissa length.
			nf = new_nf;
			nfmax = new_nf;		// Local variable.
			// If new_def_digits == 0, use all digits in mantissa.
			if ( !new_def_digits)
			{
				def_digits = nf * nd;
			}
			else
			{	// Else use the specified number if it is legal.
				if ( new_def_digits > ( nf - 1) * nd
						&& new_def_digits <= nf * nd)
				{		// Is legal.
					def_digits = new_def_digits;
				}
				else
				{
			cout << "Fatal error in super_precision::change_";
			cout << "mantissa_length --" << endl;
				   cout << "new_def_digits too ";
				   cout << " big or too small.  new_nf = ";
				   cout << new_nf << "  new_def_digits=";
				   cout << new_def_digits << endl;
				   exit (1);
				}
			}
		}
	}

	// Now that we have redefined nf and def_digits and defined nfmax, we
	// need to recalculate several derived parameters.

	// Calculate ndig = maximum number of digits carried in mantissa array:
	ndig = nf * nd;

	// Calculate clipping factor to truncate/round mantissa to def_digits:
	klip = def_digits;			// default to clipping
	m = ndig - def_digits;
//	iklip = 1;
//	while ( m--) iklip *= def_radix;	// iklip = def_radix ** m;
	iklip = radix_powers [m];		// iklip = def_radix ** m

	// Calculate nfa = length of repacked mantissa:
	nfa = ( ndig + nda - 1) / nda;

	// Calculate nfah = length of unpacked mantissa:
	nfah = 2 * nfa;

	// Calculate nfahx = number of nonzero words in unpacked mantissa:
	i = nfa * nda - ndig;
	nfahx = nfah;
	if ( i >= ndah) nfahx--;
	// Calculate indtrz = clipping factor for repacked mantissa:
//	indtrz = 1;
//	while ( i--) indtrz *= def_radix;	// indtrz = def_radix ** i
	indtrz = radix_powers [i];		// indtrz = def_radix ** i
				// indtrz is used to define roundoff

	// If def_ndig is below the fast_mult_digits threshold and the
	// fast_div_digits threshold, return and skip the succeeding
	// calculations.
	if ( ( !fast_mult_digits || def_digits < fast_mult_digits) &&
		( !fast_div_digits || def_digits < fast_div_digits)) return;

	// Calculate stuff for fast multiply option.
	double x, y;					// temporaries
	int32 dnda =  1 + double_fraction_digit_count / 2;
	do
	{
		dnda--;
		if ( !dnda)
		{
			cout << "Fatal error in super_precision::change_";
			cout << "mantissa_length --" << endl;
			cout << "Failure of fast_mult parameter loop.";
			cout << "  double_fraction_digit_count=";
			cout << double_fraction_digit_count << endl;
			cout << "Requested maximum working digits ndig=";
			cout << ndig << endl;
			exit (1);
		}
		dnfa = 2 * ( ( nfmax * nd + dnda - 1) / dnda);
				// dnfa is doubled in size because we fill
				// alternate elements (fill reals or imaginaries
				// -- with real-imaginary alternation).
			// nfft will be FFT transform size.
		nfft = 1;
		while ( nfft < dnfa) { nfft *= 2;}
			// Last half of FFT input is zeros, since it yields a
			// circular convolution.
		x = double_fraction_max;
		x /= 1000;	// Safety margin for sincos accuracy in FFT.
		x /= nfft / 2;	// FFT adds up nfft doubles but half are zero.
		y = 1;
		i = dnda;
		while ( i--) y *= def_radix;
	} while ( x < ( y - 1) * ( y - 1));	// Compare with largest addend.
	dnfa = 2 * ( ( ndig + dnda - 1) / dnda);
	dinda = (int32) y;			// Scale factor for fast_mult.

	return;
}

inline void super_precision :: super_move			// PROTECTED FN
			( super_precision &a, const super_precision &b) const
{								// set a = b
// Self-modification of the arguments is allowed -- they may be the same object.
	register int32 i;				// counter

// In this version of the implementation, super_move is a copy operation and
// copies everything from the argument b to the argument a, including mantissa,
// digits data member, radix data member, etc.  This requires the user to be
// careful not to overrun memory or otherwise mess up the target object.
//
// We do not need to delete the memory block a.mantissa already points to,
// unless by some chance the length of a.mantissa is different from b.mantissa
// -- which is not allowed in this version of the implementation.
//
// This function works for a default or non-default radix.
	a.sign_bias_exp = b.sign_bias_exp;
	a.digits = b.digits;
	a.radix = b.radix;
	a.validity = b.validity;
	for ( i = 0; i < nf; i++)
		a.mantissa [i] = b.mantissa [i];
}

super_precision & super_precision :: operator=			// OVERLOAD =
				( const super_precision &b)
{								// ASSIGN a=b
	super_move ( *this, b);
	return ( *this);
}

super_precision & super_precision :: operator=			// OVERLOAD =
				( const int32 i)
{								// ASSIGN a=i
	super_itoz ( *this, i);
	return ( *this);
}

super_precision & super_precision :: operator=			// OVERLOAD =
				( const double d)
{								// ASSIGN a=d
	*this = super_dtoz_exact ( d);
	return ( *this);
}

super_precision & super_precision :: operator=			// OVERLOAD =
				( const char *s)
{								// ASSIGN a=s
	int specified_base;

	*this = super_convert_string ( s, specified_base);
	// Return *this in the base specified in the string (if any).
	// Implicit base conversion to the working base is not done.
	return ( *this);
}

super_precision super_precision :: operator-			// OVERLOAD -
				( void) const
{								// unary -b
// Use static object *zz for intermediate results.

	super_neg ( *zz, *this);
	return ( *zz);
}

super_precision super_precision :: operator+			// OVERLOAD +
				( void) const
{								// unary +b
// Use static object *zz for intermediate results.

	super_move ( *zz, *this);
	return ( *zz);
}

super_precision super_precision :: operator+			// OVERLOAD +
				( const super_precision &b) const
{								// a + b
// Use static object *zz for intermediate results.

	super_add ( *zz, *this, b);
	return ( *zz);
}

super_precision & super_precision :: operator+=			// OVERLOAD +=
				( const super_precision &b)
{								// a += b
	super_add ( *this, *this, b);
	return ( *this);
}

super_precision super_precision :: operator-			// OVERLOAD -
				( const super_precision &b) const
{								// a - b
// Use static object *zz for intermediate results.

	super_minus ( *zz, *this, b);
	return ( *zz);
}

super_precision & super_precision :: operator-=			// OVERLOAD -=
				( const super_precision &b)
{								// a -= b
	super_minus ( *this, *this, b);
	return ( *this);
}

super_precision super_precision :: operator+			// OVERLOAD +
				( const int32 i) const
{								// a + i
// Use static object *zz for intermediate results.
// Overloading this mixed-mode operation with its own function serves
// to avoid a conversion-constructor instance, thereby helping reduce heap
// fragmentation.

	super_itoz ( *zz, i);
	super_add ( *zz, *this, *zz);
	return ( *zz);
}

super_precision & super_precision :: operator+=			// OVERLOAD +=
				( const int32 i)
{								// a + i
// Use static object *zz for intermediate results.
// Overloading this mixed-mode operation with its own function serves
// to avoid a conversion-constructor instance, thereby helping reduce heap
// fragmentation.

	super_itoz ( *zz, i);
	super_add ( *this, *this, *zz);
	return ( *this);
}

super_precision super_precision :: operator-			// OVERLOAD -
				( const int32 i) const
{								// a - i
// Use static object *zz for intermediate results.
// Overloading this mixed-mode operation with its own function serves
// to avoid a conversion-constructor instance, thereby helping reduce heap
// fragmentation.

	super_itoz ( *zz, i);
	zz->sign_bias_exp = -zz->sign_bias_exp;
	super_add ( *zz, *this, *zz);
	return ( *zz);
}

super_precision & super_precision :: operator-=			// OVERLOAD -=
				( const int32 i)
{								// a - i
// Use static object *zz for intermediate results.
// Overloading this mixed-mode operation with its own function serves
// to avoid a conversion-constructor instance, thereby helping reduce heap
// fragmentation.

	super_itoz ( *zz, i);
	zz->sign_bias_exp = -zz->sign_bias_exp;
	super_add ( *this, *this, *zz);
	return ( *this);
}

super_precision super_precision :: operator*			// OVERLOAD *
				( const super_precision &b) const
{								// a * b
// Use static object *zz for intermediate results.

	super_mult ( *zz, *this, b);
	return ( *zz);
}

super_precision & super_precision :: operator*=			// OVERLOAD *=
				( const super_precision &b)
{								// a *= b
	super_mult ( *this, *this, b);
	return ( *this);
}

super_precision super_precision :: operator*			// OVERLOAD *
				( const int32 i) const
{								// a * i
// Use static object *zz for intermediate results.

	super_multi ( *zz, *this, i);
	return ( *zz);
}


super_precision & super_precision :: operator*=			// OVERLOAD *=
				( const int32 i)
{								// a *= i
	super_multi ( *this, *this, i);
	return ( *this);
}

super_precision operator+					// OVERLOAD +
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION 					// i + a
// Use static object *zz for intermediate results.

	super_precision :: zz->super_itoz ( *super_precision :: zz, i);
	super_precision :: zz->super_add ( *super_precision :: zz,
					*super_precision :: zz, a);
	return ( *super_precision :: zz);
}

super_precision operator-					// OVERLOAD -
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION 					// i - a
// Use static object *zz for intermediate results.

	super_precision :: zz->super_itoz ( *super_precision :: zz, i);
	super_precision :: zz->super_minus ( *super_precision :: zz,
					*super_precision :: zz, a);
	return ( *super_precision :: zz);
}


super_precision operator*					// OVERLOAD *
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION 					// i * a
// Use static object *zz for intermediate results.

	super_precision :: zz->super_multi ( *super_precision :: zz, a, i);
	return ( *super_precision :: zz);
}


super_precision super_precision :: operator/			// OVERLOAD /
				( const super_precision &b) const
{								// a / b
// Use static object *zz for intermediate results.

	super_div ( *zz, *this, b);
	return ( *zz);
}


super_precision & super_precision :: operator/=			// OVERLOAD /=
				( const super_precision &b)
{								// a /= b
	super_div ( *this, *this, b);
	return ( *this);
}

super_precision super_precision :: operator/			// OVERLOAD /
				( const int32 i) const
{								// a / i
// Use static object *zz for intermediate results.

	super_divi ( *zz, *this, i);
	return ( *zz);
}

super_precision & super_precision :: operator/=			// OVERLOAD /=
				( const int32 i)
{								// a /= i
	super_divi ( *this, *this, i);
	return ( *this);
}

super_precision operator/					// OVERLOAD /
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION					// i / a
// Use static objects *zz, *w for intermediate results.

	if ( i == 1)
	{
			// Only a reciprocal is needed.  The super_div routine
			// always checks for a unit dividend, and if found
			// a simple reciprocal is done -- saving one long
			// multiply.
		super_precision :: zz->super_itoz ( *super_precision :: w, 1);
		super_precision :: zz->super_div ( *super_precision :: zz,
						*super_precision :: w, a);
	}
	else
	{
#ifndef SUPER_SPEED
	// Sacrifice speed for roundoff accuracy.  Do the long divide.
		super_precision :: zz->super_itoz ( *super_precision :: w, i);
		super_precision :: zz->super_div ( *super_precision :: zz,
						*super_precision :: w, a);
#else
	// Sacrifice roundoff accuracy for speed.  Use a trick.
		// The super_div routine always checks for a unit dividend,
		// and if found a simple reciprocal is done -- saving one
		// long multiply.  The multiply-by-integer is much faster
		// than a long multiply, so we set up a reciprocal and then
		// multiply by the integer.  Question: what does this do to
		// the roundoff?  Answer: It messes it up in some cases.
		super_precision :: zz->super_itoz ( *super_precision :: w, 1);
		super_precision :: zz->super_div ( *super_precision :: zz,
						*super_precision :: w, a);
		super_precision :: zz->super_multi ( *super_precision :: zz,
						*super_precision :: zz, i);
#endif
	}
	return ( *super_precision :: zz);
}


super_precision super_precision :: operator%			// OVERLOAD %
				( const super_precision &b) const
{								// a % b
	// If b == 0, return 0:
	if ( b.sign_bias_exp == 0) return ( super_precision ( 0));

	// Save rounding method and reset it to truncated:
	int i = rounding_method;
	rounding_method = FALSE;
	
        // Get pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	super_div ( a, *this, b);
	a = trunc ( a);
	a *= b;
	a = *this - a;

	// Restore rounding method:
	rounding_method = i;

	return ( a);
}


super_precision & super_precision :: operator%=			// OVERLOAD %=
				( const super_precision &b)
{								// a %= b
	// If b == 0, return 0:
	if ( b.sign_bias_exp == 0)
	{
		super_zero ( *this);
		return ( *this);
	}

	// Save rounding method and reset it to truncated:
	int i = rounding_method;
	rounding_method = FALSE;

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	super_div ( a, *this, b);
	a = trunc ( a);
	a *= b;
	a = *this - a;
	*this = a;

	// Restore rounding method:
	rounding_method = i;

	return ( *this);
}

super_precision super_precision :: operator%			// OVERLOAD %
				( const int32 i) const
{								// a % i
	// If i == 0, return 0:
	if ( i == 0) return ( super_precision ( 0));

	// Save rounding method and reset it to truncated:
	int j = rounding_method;
	rounding_method = FALSE;

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	super_divi ( a, *this, i);
	a = trunc ( a);
	a *= i;
	a = *this - a;

	// Restore rounding method:
	rounding_method = j;

	return ( a);
}

super_precision & super_precision :: operator%=			// OVERLOAD %=
				( const int32 i)
{								// a %= i
	// If i == 0, return 0:
	if ( i == 0)
	{
		super_zero ( *this);
		return ( *this);
	}

	// Save rounding method and reset it to truncated:
	int j = rounding_method;
	rounding_method = FALSE;

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	super_divi ( a, *this, i);
	a = trunc ( a);
	a *= i;
	a = *this - a;
	*this = a;

	// Restore rounding method:
	rounding_method = j;

	return ( *this);
}
super_precision operator%					// OVERLOAD %
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION					// i % a
	// If a == 0, return 0:
	if ( a.sign_bias_exp == 0) return ( super_precision ( 0));


	// Save rounding method and reset it to truncated:
	int j = super_precision :: rounding_method;
	super_precision :: rounding_method = FALSE;

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr b_, c_;
        super_precision & b = *b_;
        super_precision & c = *c_;

	super_precision :: zz->super_itoz ( b, i);
	super_precision :: zz->super_div ( c, b, a);
	c = trunc ( c);
	c *= a;
	c = b - c;

	// Restore rounding method:
	super_precision :: rounding_method = j;

	return ( c);
}


super_precision & super_precision :: operator++			// OVERLOAD ++a
				( void)
{								// ++a
// Increments the value of the super_precision object *this and returns the
// incremented value.

	super_itoz ( *zt, 1);
	super_add ( *this, *this, *zt);

	return ( *this);
}


super_precision & super_precision :: operator++			// OVERLOAD a++
				( int)
{								// a++
// Increments the value of the super_precision object *this and returns the
// original unincremented value.

	*zz = *this;
	super_itoz ( *zt, 1);
	super_add ( *this, *this, *zt);

	return ( *zz);
}


super_precision & super_precision :: operator--			// OVERLOAD --a
				( void)
{								// --a
// Decrements the value of the super_precision object *this and returns the
// decremented value.

	super_itoz ( *zt, -1);
	super_add ( *this, *this, *zt);

	return ( *this);
}


super_precision & super_precision :: operator--			// OVERLOAD a--
				( int)
{								// a--
// Decrements the value of the super_precision object *this and returns the
// original undecremented value.

	*zz = *this;
	super_itoz ( *zt, -1);
	super_add ( *this, *this, *zt);

	return ( *zz);
}


int super_precision :: operator<				// OVERLOAD a<b
				( const super_precision &b) const
{								// a < b
// Returns a TRUE or FALSE for the numerical relationship *this < b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) < 0);
}

int super_precision :: operator>				// OVERLOAD a>b
				( const super_precision &b) const
{								// a > b
// Returns a TRUE or FALSE for the numerical relationship *this > b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) > 0);
}

int super_precision :: operator==				// OVERLOAD a==b
				( const super_precision &b) const
{								// a == b
// Returns a TRUE or FALSE for the numerical relationship *this == b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) == 0);
}

int super_precision :: operator!=				// OVERLOAD a!=b
				( const super_precision &b) const
{								// a != b
// Returns a TRUE or FALSE for the numerical relationship *this != b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) != 0);
}

int super_precision :: operator<=				// OVERLOAD a<=b
				( const super_precision &b) const
{								// a <= b
// Returns a TRUE or FALSE for the numerical relationship *this <= b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) <= 0);
}

int super_precision :: operator>=				// OVERLOAD a>=b
				( const super_precision &b) const
{								// a >= b
// Returns a TRUE or FALSE for the numerical relationship *this >= b.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, b) >= 0);
}

int super_precision :: operator!				// OVERLOAD !a
				( void) const
{								// !a
// Returns a TRUE or FALSE for the numerical relationship a != 0 (or !(a==0)).
// Check to see if *this is properly formed and then see if sign_bias_exp == 0.
// If *this is not properly formed (is NaN, Inf, or UnN) then it is not true
// zero.

	return ( this->validity == 0 && this->sign_bias_exp == 0);
}

int super_precision :: operator&&				// OVERLOAD a&&b
				( const super_precision &b) const
{								// a && b
// Returns a TRUE or FALSE for the numerical relationship *this && b.
// Check to see if *this and b are not properly formed or if
// sign_bias_exp != 0 for *this and b.

	return ( ( this->validity != 0 || this->sign_bias_exp != 0)
		&&   ( b.validity != 0 ||     b.sign_bias_exp != 0));
}

int super_precision :: operator||				// OVERLOAD a||b
				( const super_precision &b) const
{								// a || b
// Returns a TRUE or FALSE for the numerical relationship *this || b.
// Check to see if *this and b are not properly formed or if
// sign_bias_exp != 0 for *this and b.

	return ( ( this->validity != 0 || this->sign_bias_exp != 0)
		||   ( b.validity != 0 ||     b.sign_bias_exp != 0));
}

int super_precision :: operator<				// OVERLOAD a<i
				( const int32 i) const
{								// a < i
// Returns a TRUE or FALSE for the numerical relationship *this < i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) < 0);
}

int super_precision :: operator>				// OVERLOAD a>i
				( const int32 i) const
{								// a > i
// Returns a TRUE or FALSE for the numerical relationship *this > i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) > 0);
}

int super_precision :: operator==				// OVERLOAD a==i
				( const int32 i) const
{								// a == i
// Returns a TRUE or FALSE for the numerical relationship *this == i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) == 0);
}

int super_precision :: operator!=				// OVERLOAD a!=i
				( const int32 i) const
{								// a != i
// Returns a TRUE or FALSE for the numerical relationship *this != i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) != 0);
}

int super_precision :: operator<=				// OVERLOAD a<=i
				( const int32 i) const
{								// a <= i
// Returns a TRUE or FALSE for the numerical relationship *this <= i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) <= 0);
}

int super_precision :: operator>=				// OVERLOAD a>=i
				( const int32 i) const
{								// a >= i
// Returns a TRUE or FALSE for the numerical relationship *this >= i.

	// super_test returns -1, 0, +1 as "*this" is <, ==, > b.
	return ( super_test ( *this, i) >= 0);
}

int super_precision :: operator&&				// OVERLOAD a&&i
				( const int32 i) const
{								// a && i
// Returns a TRUE or FALSE for the numerical relationship *this && i.
// Check to see if *this is not properly formed or if
// sign_bias_exp != 0 for *this.

	return ( ( this->validity != 0 || this->sign_bias_exp != 0)
		&&   ( i != 0));
}

int super_precision :: operator||				// OVERLOAD a||i
				( const int32 i) const
{								// a || i
// Returns a TRUE or FALSE for the numerical relationship *this || i.
// Check to see if *this is not properly formed or if
// sign_bias_exp != 0 for *this.

	return ( ( this->validity != 0 || this->sign_bias_exp != 0)
		||   ( i != 0));
}

int operator<							// OVERLOAD <
		( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i < a

// Returns a TRUE or FALSE for the numerical relationship i < a.

// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) < 0);
}

int operator>							// OVERLOAD >
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i > a

// Returns a TRUE or FALSE for the numerical relationship i > a.

// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) > 0);
}

int operator==							// OVERLOAD ==
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i == a

// Returns a TRUE or FALSE for the numerical relationship i == a.
//
// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) == 0);
}

int operator!=							// OVERLOAD !=
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i != a

// Returns a TRUE or FALSE for the numerical relationship i != a.
//
// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) != 0);
}

int operator<=							// OVERLOAD <=
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i <= a

// Returns a TRUE or FALSE for the numerical relationship i <= a.
//
// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) <= 0);
}

int operator>=							// OVERLOAD >=
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i >= a

// Returns a TRUE or FALSE for the numerical relationship i >= a.
//
// A pointer with file scope, *w, is also used as an object to access the
// member function super_test.  The scope resolution operator is required to
// access w from a friend function like this one.

	return ( super_precision :: w->super_test ( i, a) >= 0);
}

int operator&&							// OVERLOAD &&
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION					// i && a
// Returns a TRUE or FALSE for the numerical relationship *this && i.
// Check to see if a is not properly formed or if a.sign_bias_exp != 0.

	return ( ( a.validity != 0 || a.sign_bias_exp != 0)
		&&   ( i != 0));
}

int operator||							// OVERLOAD ||
				( const int32 i, const super_precision &a)
{	// FRIEND FUNCTION					// i || a
// Returns a TRUE or FALSE for the numerical relationship *this || i.
// Check to see if a is not properly formed or if a.sign_bias_exp != 0.

	return ( ( a.validity != 0 || a.sign_bias_exp != 0)
		||   ( i != 0));
}


inline void super_precision :: super_zqanc			// PROTECTED FN
				( int32 a[], int32 b[], int32 c[],
				  int32 j, int32 in, int32 &carry) const
{								// inline-anc
// super_zqanc Adds the mantissas of two operands, Normalizes the mantissa
// elements, with Carry propagation.  Hence Add-Normalize-Carry -> ANC.
	// j	   is the mantissa array length
	// in	   is the modulus of mantissa elements ( = radix ** nd or ndah)
	// carry   is the carry value into and out of the routine
	// a, b, c are the result and two input operands, respectively
// Self-modification of the arguments is allowed -- they may be the same object.

	register int32 m, n, ic;			// temporaries
	register int32 mask;				// alternate method

	m = j - 1;
	ic = carry;
	if ( !radix_powerof2)
	{		// For radix not-a-power-of-2:
		do
		{
// Standard method for any radix:		typical += time = 62.2 microsec
//			n = b [m] + c [m] + ic;
//			ic = n / in;
//			a [m] = n - in * ic;
// Possible faster method for any radix:	typical += time = 53.8 microsec
			n = b [m] + c [m] + ic - in;
			mask = n >> 31;
			a [m] = ( mask & in) + n;
			ic = ( ~mask) & 1;
		} while ( m--);
	}
	else
	{		// For radix equal to a power-of-2:
		int nshifts;
		if ( in == ind)		// Two possible cases.  Check for which.
		{
			nshifts = nd * radix_powerof2;
		}
		else
		{		// Only other possible case.
			nshifts = ndah * radix_powerof2;
		}
		do
		{
// Faster method for powers-of-two:
			n = b [m] + c [m] + ic;
			ic = n >> nshifts;
			a [m] = n - ( ic << nshifts);
		} while ( m--);
	}

	carry = ic;

	return;
}

inline void super_precision :: super_zqsnc			// PROTECTED FN
				( int32 a[], int32 b[], int32 c[],
				  int32 j, int32 in, int32 &carry) const
{								// inline-snc
// super_zqsnc Subtracts the mantissas of two operands, Normalizes the mantissa
// elements, with Carry propagation.  Hence Subtract-Normalize-Carry -> SNC.
	// j	   is the mantissa array length
	// in	   is the modulus of mantissa elements ( = radix ** nd)
	// carry   is the carry value into and out of the routine
	// a, b, c are the result and two input operands, respectively
// Self-modification of the arguments is allowed -- they may be the same object.

	register int32 m, ic;				// temporaries
//	register int32 mask;				// alternate method

	ic = carry;
	m = j - 1;
	do
	{
// Standard method:
		a [m] = b [m] - c [m] - ic;
		if ( a [m] >= 0)
		{
			ic = 0;
		}
		else
		{
			a [m] += in;
			ic = 1;
		}
// Alternate method (no faster on 80486/DX2-50):
//		a [m] = b [m] - c [m] - ic;
//		mask = a [m] >> 31;
//		ic = mask & 1;
//		a [m] += mask & in;
//
	} while ( m--);
	carry = ic;

	return;
}

inline void super_precision :: super_zqm1			// PROTECTED FN
				( int32 t[], int32 v[], int32 l,
				  int32 n, int32 &k, int32 in) const
{								// inline-m1
// super_zqm1 multiplies the mantissas of two operands, obtaining the low-order
// bit carried into the lowest-order digit.
	// t, v    are integer arrays into which mantissas have been unpacked
	// l	   is equal to 2 (since ncol is unused, for exactness)
	// n	   is the number of elements used in t and v (= nfahx)
	// k	   is the low-order bit calculated, to be carried upward.
	// in	   is the modulus of mantissa elements ( = radix ** ndah)

	int32 jj, kk;				// temporaries
	register int32 i, j;			// temporaries

	for ( jj = l - 1, k = 0; jj < n; jj++)	// clear k
	{
		kk = n - jj;
		for ( i = kk, j = n - 1; i < n; i++, j--)
		{
			k += t [i] * v [j];
		}
		k /= in;
	}

	return;
}

inline void super_precision :: super_zqm2			// PROTECTED FN
				( int32 t[], int32 v[],
				  int32 n, int32 &carry) const
{								// inline-m2
// super_zqm2 multiplies the mantissas of two operands, obtaining the high-order
// half of the product, but does not scale the products.  A carry k is used at
// the low-order end of the product.
	// t, v    are integer arrays into which mantissas have been unpacked
	// n	   is the number of elements used in t and v (= nfahx)
	// carry   is the carry added in at the low-order end

	int32 i, j;				// temporaries
	register int32 k, jj, kk;		// temporaries

	k = carry;
	for ( i = 0, j = n; i < n; i++)
	{
		for ( kk = 0, jj = j - 1; kk < j; kk++, jj--)
		{
			k += t [kk] * v [jj];
		}
		t [--j] = k;
		k = 0;
	}
	
	return;
}

inline void super_precision :: super_zqm2lsh			// PROTECTED FN
				( int32 t[], int32 v[],
				  int32 n, int32 &carry, int32 in) const
{								// inline-m2
// Especially for the case of v left-shifted 1 digit,
// super_zqm2lsh multiplies mantissas of two operands, obtaining the high-order
// half of the product, but does not scale the products.  A carry k is used at
// the low-order end of the product.
	// t, v    are integer arrays into which mantissas have been unpacked
	// n	   is the number of elements used in t and v (= nfahx)
	// carry   is the carry added in at the low-order end
	// in      is the scale factor for unpacked mantissas

	int32 i, j, insq;			// temporaries
	register int32 k, jj, kk, kcarry;	// temporaries

	insq = in * in;
	kcarry = carry;
	for ( i = 0, j = n; i < n; i++)
	{
		k = kcarry;
		for ( kk = 0, jj = j - 1; kk < j - 1; kk++, jj--)
		{
			k += t [kk] * v [jj];
		}
		kcarry = k / insq;
		k -= kcarry * insq;
		k += v [0] * t [j-1];
		t [--j] = k;
		kcarry *= in;
	}
	t [0] += kcarry * in;
	
	return;
}

inline void super_precision :: super_zqnc			// PROTECTED FN
				( int32 t[], int32 j,
				  int32 in, int32 &carry, int32 id) const
{								// inline-nc
// super_zqnc scales each element of the mantissa t, using and propagating a
// carry digit.
	// t       is an integer array into which mantissa has been unpacked
	// j	   is the number of elements used in t (= nfahx - 1)
	// in      is the scale (= radix ** ndah)
	// carry   is the carry added in at the low-order end
	// id      is a displacement of the scaled result toward the low-order

	register int32 l, m;			// temporaries

	if ( !radix_powerof2)
	{		// For any radix not a power-of-2.
		for ( l = j - 1; l >= 0; l--)
		{
			m = t [l] + carry;
			carry = m / in;
			t [ l + id] = m - carry * in;
		}
	}
	else
	{		// For a power-of-2 radix.
			// indah is the only possible value for in.
		int nshifts = ndah * radix_powerof2;
		for ( l = j - 1; l >= 0; l--)
		{
			m = t [l] + carry;
			carry = m >> nshifts;
			t [ l + id] = m - ( carry << nshifts);
		}
	}
	
	return;
}

inline void super_precision :: super_zqncc			// PROTECTED FN
			( int32 a[], int32 j, int32 in, int32 carry) const
{								// inline-ncc
// super_zqncc Normalizes/scales the mantissa elements, with Conditional Carry.
// Hence, -> NCC.  The mantissa is assumed to be scaled at first, then a carry
// is added to the least significant element.  The element is rescaled and any
// resulting carry is propagated to the next element.  The first element
// receives any carry from rescaling the second element, but is not rescaled
// itself.
	// j	   is the mantissa array length
	// in	   is the modulus of mantissa elements ( = radix ** nd)
	// carry   is the carry value into the routine
	// a []	   is the input and output operand.

	register int32 m;				// temporaries

	m = j - 1;
	a [m] += carry;
	if ( m >= 1) 
	{
		do
		{
			if ( a [m] < in) break;
			a [m] -= in;		// BREAKS WHEN a[m] >= 2*in
			m--;
			a [m]++;
			
		} while ( m);
	}

	return;
}


inline void super_precision :: super_zqdnc			// PROTECTED FN
		( int32 a[], int32 ia, int32 ja, int32 j, int32 &carry) const
{								// inline-dnc
// super_zqdnc Divides, Normalizes/scales the mantissa elements, with Carry.
// Hence, -> DNC.  The mantissa is assumed to be scaled at first.  Each element
// is divided by a divisor, scaled, and a remainder is propagated to the
// next **less** significant element (that is, the carries are propagated
// downward in significance, like a remainder).
	// ia	   is the divisor
	// ja	   is the initial remainder added into the first element
	// j	   is the mantissa array length
	// carry   is the final remainder carried out of the final element
	// a []	   is the input and output operand.

	register int32 i, ic, jy, k, l;			// temporaries

	jy = ja / ia;		// divide the initial remainder by the divisor
	ic = 0;			// clear the next remainder
	for ( i = 0; i < j; i++)
	{
		k = a [i] / ia;
		l = a [i] - k * ia;
		a [i] = k + ic * jy;
		ic = l;
	}
	carry = ic;

	return;
}

inline void super_precision :: super_zqmnc			// PROTECTED FN
			( int32 a[], int32 ia, int32 ja, int32 j) const
{								// inline-mnc
// super_zqmnc Multiplies, Normalizes/scales the mantissa elements, with Carry.
// Hence, -> MNC.  The mantissa is assumed to be scaled at first.  Each element
// is multiplied by a multiplier, scaled, and a remainder is propagated to the
// next **less** significant element (that is, the carries are propagated
// downward in significance, like a remainder).
	// ia	   is the multiplier
	// ja	   is the scale
	// j	   is the mantissa array length
	// a []	   is the input and output operand.

	register int32 i, jx, jy, l;			// temporaries

	a [0] *= ia;		// multiply the most significant element
	jx = j - 1;
	if ( jx > 0)
	{
		jy = ja / ia;
		for ( i = 0; i < jx; i++)
		{
			l = a [ i + 1] / jy;
			a [i] += l;
			a [i + 1] = ia * ( a [ i + 1] - l * jy);
		}
	}

	return;
}

inline void super_precision :: super_zqupk			// PROTECTED FN
				( int32 t [], int32 a [], int32 jj,
				int32 ll) const
{								// >unpack
	int32 i, k, ii, kk;			// temporaries

	i = 0;
	ii = 0;
	kk = a [0];
	while ( 1)
	{
		if ( jj >= ll)
		{
			jj /= ll;
			t [ii] += kk * jj;
			i++;
			if ( i == nf) break;
			kk = a [i];
			ll = ind;
			if ( jj > 1) continue;
			jj = inda;
			ii++;
			if ( ii == nfa) break;
			t [ii] = 0;
		}
		else
		{
			ll /= jj;
			k = kk / ll;
			t [ii] += k;
			kk -= k * ll;
			jj = inda;
			ii++;
			if ( ii == nfa) break;
			t [ii] = 0;
		}
	}

	return;
}

inline void super_precision :: super_zqrpk			// PROTECTED FN
				( int32 a[], int32 t[],
				int32 nfx, int32 &jj, int32 &ll,
				int32 &ii, int32 indx) const
{								// >repack
	int32 i, k;				// temporaries

	for ( i = 0; i < nfx; i++)
	{
		if ( jj >= ll)
		{
			jj /= ll;
			a [ii] += t [i] * jj;
		}
		else
		{
			ll /= jj;
			k = t [i] / ll;
			a [ii] += k;		// ii is -1
			ii++;
			if ( ii == nf) break;
			k = t [i] - k * ll;
			jj = ind / ll;
			a [ii] = k * jj;
		}

		ll = indx;
	}

	return;
}

inline void super_precision :: super_zero			// PROTECTED FN
				( super_precision &a) const
{								// set a = 0
	register int32 i;				// counter

	a.sign_bias_exp = 0;				// exponent word = 0
	for ( i = 0; i < nf; i++)  a.mantissa [i] = 0;	// zero entire mantissa

	return;
}

inline void super_precision :: super_unpk			// PROTECTED FN
			( const super_precision &a, int32 t []) const
{								// >unpack
	int32 jj, kk, ll;			// temporaries
	register int32 i, j, k;			// temporaries

	if ( naccum)
	{
		t [0] = 0;
		jj = inda;
		ll = ind;
		super_zqupk ( t, a.mantissa, jj, ll);
	}
	else
	{
		for ( i = 0; i < nf; i++)
		{
			t [i] = a.mantissa [i];
		}
	}
	
	j = nfah - 1;
	if ( !radix_powerof2)
	{	// In case radix is not a power of two:
		for ( i = 0; i < nfa; i++)
		{
			k = nfa - i - 1;
			kk = t [k] / indah;
			t [j--] = t [k] - kk * indah;
			t [j--] = kk;
		}
	}
	else
	{	// Only when radix is a power of two:
		int nshifts = ndah * radix_powerof2;
		for ( i = 0; i < nfa; i++)
		{
			k = nfa - i - 1;
			kk = t [k] >> nshifts;
			t [j--] = t [k] - ( kk << nshifts);
			t [j--] = kk;
		}
	}

	return;
}

inline void super_precision :: super_repk			// PROTECTED FN
			( super_precision &a, int32 t []) const
{								// >repack
	int32 ii, jj, kk, ll;			// temporaries
	register int32 i, j;			// temporaries

	if ( t [0] >= indah) super_zqdnc ( t, def_radix, indah, nfahx, kk);

	for ( i = 0, j = 0; i < nfa; i++)
	{
		t [i] = t [j] * indah + t [j + 1];
		j += 2;
	}

//	if ( t [0] < inda / def_radix)
	if ( t [0] < radix_powers [nda - 1])
	{
//		ll = inda / def_radix;
		ll = radix_powers [nda -1];
	}
	else
	{
		ll = inda;
	}

	if ( ( naccum == 0) && ( ll == inda))
	{
		for ( i = 0; i < nf; i++)
		{
			a.mantissa [i] = t [i];
		}

		return;
	}

	jj = ind;
	ii = 0;
	a.mantissa [0] = 0;
	super_zqrpk ( a.mantissa, t, nfa, jj, ll, ii, inda);

	return;
}

inline void super_precision :: super_zad			// PROTECTED FN
				( super_precision &a, const super_precision &b,
				const super_precision &c) const
{								// >a = b + c
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 ic;

	if ( rounding_method)
	{
		ic = irbit;
		if ( ic < 0)
		{	// This is a "semi-round" -- exactly 1/2 bit.
			// So round to nearest even number in last place.
			int32 j;
			j = ( b.mantissa [nf-1] + c.mantissa [nf-1]) / iklip;
			if ( j % 2 == 0) { ic = 0; }
			else { ic = -ic; }
			// Rounding to the nearest even digit is a "stable"
			// rounding method that avoids "drift" in repeated
			// adds and subtracts, in certain cases.
		}
	}
	else
	{
		ic = 0;
	}

	super_zqanc ( a.mantissa, b.mantissa, c.mantissa, nf, ind, ic);

	a.mantissa [0] += ic * ind;
	a.sign_bias_exp = b.sign_bias_exp;

	return;
}

inline void super_precision :: super_zmi			// PROTECTED FN
				( super_precision &a, const super_precision &b,
				const super_precision &c) const
{								// >a = b + c
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i, ic;

	if ( rounding_method)
	{
		ic = irbit;
		if ( ic < 0)
		{	// This is a "semi-round" -- exactly 1/2 bit.
			// So round to nearest even number in last place.
			int32 j;
			j = ( b.mantissa [nf-1] - c.mantissa [nf-1]) / iklip;
			if ( abs( j) % 2 == 0) { ic = 0; }
			else { ic = -ic; }
			// Rounding to the nearest even digit is a "stable"
			// rounding method that avoids "drift" in repeated
			// adds and subtracts, in certain cases.
		}
	}
	else
	{
		ic = 0;
	}

	for ( i = 0; i < nf; i++)
	{
		if ( b.mantissa [i] == c.mantissa [i]) continue;
		if ( b.mantissa [i] > c.mantissa [i])
		{
			// b mantissa is greater than c, so subtract c from b:
			super_zqsnc ( a.mantissa, b.mantissa, c.mantissa,
					nf, ind, ic);
			a.sign_bias_exp = b.sign_bias_exp;
			return;
		}
		else
		{
			// c mantissa is greater than b, so subtract b from c:
			super_zqsnc ( a.mantissa, c.mantissa, b.mantissa,
					nf, ind, ic);
			a.sign_bias_exp = c.sign_bias_exp;
			return;
		}

	}
	// b and c mantissas are equal, so result is zero:
	super_zero ( a);
	return;
}

inline void super_precision :: super_zmu			// PROTECTED FN
				( super_precision &a, const super_precision &b,
				int32 ii) const
{								// >a = b * i
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i, j, k, l, jk, intrz;		// temporaries

	// Check for zero product:
	if ( ( ii == 0) || ( b.sign_bias_exp == 0))
	{
		super_zero ( a);
		return;
	}

	super_unpk ( b, v);
	jk = abs ( ii);
	for ( i = 0; i < nfahx; i++)
	{
		v [i] *= jk;
	}
	k = 0;
	super_zqnc ( v, nfahx, indah, k, 0);
	l = 0;
	j = 1;
	while ( j <= k)
	{
		j *= def_radix;
		l++;
	}

	if ( rounding_method)
	{
		intrz = indtrz;
			// intrz is clipping factor for array of length nfahx.
		if ( nfahx != nfah) intrz = indtrz / indah;
		super_zqncc ( v, nfahx, indah, ( j * intrz) / 2);
		if ( v [0] >= indah)
		{
			k++;
			v [0] -= indah;
			if ( j <= k)
			{
				j *= def_radix;
				l++;
			}
		}
	}
	v [0] += indah * k;
	super_zqdnc ( v, j, indah, nfahx, k);
	super_repk ( a, v);
	i = abs ( b.sign_bias_exp);
	j = ( i - bias) + l;
	if ( ( j > 0) && ( j > bias + 1))
	{
		if ( permissive_error)
		{		// Set to +infinity or -infinity.
			super_zero ( a);
			a.validity = ( b.sign_bias_exp > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow in super_zmu -- fatal error.\n";
			exit (1);
		}
	}
	a.sign_bias_exp = ( b.sign_bias_exp > 0) ?  ( i + l)
						 :  - ( i + l);
	if ( jk != ii) a.sign_bias_exp = -a.sign_bias_exp;

	if ( klip > 0) super_klipr ( a);
	return;
}

inline void super_precision :: super_zqdi			// PROTECTED FN
				( int32 t [], int32 n, int32 ib,
				int32 indah) const
{								// >a = b / i
	int32 j, k, l;				// temporaries

	if ( n > 1)
	{
		for ( j = 0; j < n - 1; j++)
		{
			k = t [j] / ib;
			l = t [j] - k * ib;
			t [j] = k;
			t [ j + 1] += l * indah;
		}
	}

	return;
}

inline void super_precision :: super_zqd1			// PROTECTED FN
				( int32 v [], int32 ia [], int32 ib [],
				int32 &jc, int32 i4, int32 np1,
				int32 indah) const
{								// >a = b / c
	int32 i, j, k, n, ja, jj, kk, mm;		// temporaries

	n = np1 - 2;
	for ( i = 1; i <= i4; i++)
	{
		ja = jc;
		jj = i4 - i;
		for ( j = 0; j < i; j++)
		{
			k = jj + j;
			kk = n - j;
			ja += ia [k] * v [kk];
		}
		jc = ja / indah;
		ja -= jc * indah;
		if ( ja)
		{
			mm = ib [jj] - ja;
			if ( mm < 0)
			{
				mm += indah;
				jc++;
			}
			ib [jj] = mm;
		}
	}

	return;
}

inline void super_precision :: super_zqd2			// PROTECTED FN
				( int32 t [], int32 v [], int32 ia [],
				int32 &jc, int32 n, int32 i4,
				int32 indah) const
{								// >a = b / c
	int32 i, j, k, ii, np1, ja, kk, mm;		// temporaries

	if ( n <= 1) return;

	np1 = n + 1;
	for ( i = 2; i <= n; i++)
	{
		mm = np1 - i;
		ja = jc;
		ii = mm - 1;
		mm = ( mm > i4) ? i4 : mm;	// min ( mm, i4)
		for ( j = 0; j < mm; j++)
		{
			kk = ii - j;
			ja += ia [j] * v [kk];
		}
		jc = ja / indah;
		ja -= jc * indah;
		k = np1 - i;
		t [k] -= ja;
		if ( t [k] < 0)
		{
			t [k] += indah;
			jc++;
		}
	}
	
	return;
}

inline void super_precision :: super_zqncb			// PROTECTED FN
				( int32 a [], int32 j, int32 in,
				int32 ic) const
{								// >a = b / c
	int32 i, k, l;				// temporaries

	if ( j <= 0) return;

	a [j - 1] -= ic;
	k = j - 1;
	for ( i = 0; i < j - 1; i++)
	{
		l = k - i;
		if ( a [l] >= 0) return;
		a [l] += in;
		a [l - 1] -= 1;
	}

	return;
}


void super_precision :: super_klipr				// PROTECTED FN
				( super_precision &a) const
{								// clip a
// This function clips the super_precision object a to fewer than ndig digits
// by setting a certain number of the trailing digits in the last element of
// the mantissa to zero.  The resulting number is truncated or rounded depending
// on the value of rounding_method.
// The number of digits remaining after clipping is klip, and iklip is
// a.radix ** (ndig-klip).
// Proper functioning requires the trailing digits (to be zeroed) to be
// confined to the last element of the mantissa (i.e., ndig - klip < nd).
// This function requires special preparation to work for a non-default
// radix.  (For example, see its use in function roundoff().)

	int32 i;					// temporary

			// If argument is invalid, return with no change.
	if ( a.assert_invalid ( "super_klipr")) return;

	if ( !klip) return;		// quit if klip = 0

	i = a.mantissa [nf - 1] / iklip;
	if ( !rounding_method)
	{				// truncate last digit
		a.mantissa [nf - 1] = i * iklip;
		return;
	}

	// round last digit:
	i *= iklip;
	if ( ( a.mantissa [nf - 1] - i) < ( ( iklip + 1) / 2))
	{
		a.mantissa [nf - 1] = i;
		return;
	}

	a.mantissa [nf - 1] = i + iklip;
	super_zqncc ( a.mantissa, nf, ind, 0);
	if ( a.mantissa [0] < ind) return;
	a.mantissa [0] /= a.radix;
	if ( abs ( a.sign_bias_exp) != bias * 2 + 1)
	{
		if ( a.sign_bias_exp < 0)
		{		// increase magnitude for neg number
			a.sign_bias_exp--;
		}
		else
		{		// increase magnitude for pos number
			a.sign_bias_exp++;
		}
		return;
	}

	if ( permissive_error)
	{		// Set to +infinity or -infinity.
		i = a.sign_bias_exp;
		super_zero ( a);
		a.validity = ( i > 0) ? 1 : 2;
		return;
	}
	else
	{
		cout << "Overflow during a clip operation in super_klipr --";
		cout << " fatal error." << endl;
		exit (1);
	}
}					// end of super_klipr

void super_precision :: super_norm				// PROTECTED FN
			( super_precision &a, int32 &shift_cnt) const
{								// normalize a
// super_norm performs a shift of the mantissa with an accompanying
// adjustment of the exponent, so as to eliminate leading zeros in the 
// mantissa.  The shift count is returned in the second argument; a positive
// count indicates a left shift was done, and a negative count signals a right
// shift was done.  Rounding and/or clipping may be done for a right shift.
	int32 i, j, k, l, ntrz, jklip;			// temporaries
	int32 not_zeros;

			// If argument is invalid, return with no change.
	if ( a.assert_invalid ( "super_norm"))
	{
		shift_cnt = 0;
		return;
	}

	// Test for zero mantissa.
	shift_cnt = 0;			// clear shift count (of leading zeros)
	not_zeros = FALSE;
	for ( i = 0, not_zeros = FALSE; i < nf; i++)
	{
		shift_cnt += nd;		// bump shift count
		if ( a.mantissa [i] != 0)
		{
			not_zeros = TRUE;
			break;
		}
	}
	if ( !not_zeros)
	{		// mantissa is all zeros so return a true zero super
		a.sign_bias_exp = 0;		// clear sign-exponent word
		shift_cnt = 0;			// clear shift count
		return;
	}

	l = a.mantissa [i] / def_radix;
	if ( l != 0)
	{
		k = 1;
		do
		{
			k *= def_radix;
			shift_cnt--;
		} while ( l >= k);
	}
	shift_cnt--;
	
	if ( ( shift_cnt >= 0) || ( rounding_method == 0))
	{		// Do left shift or truncated right shift:
		super_shift ( a, shift_cnt);
		return;
	}

	// Do case of right shift with round:
	ntrz = 0;				// clear trailing zero count
	if ( klip > 0) ntrz = ndig - klip;
	
	// Test for overflow:
	if ( ntrz - shift_cnt > nd)
	{
		if ( permissive_error)
		{		// Set to +infinity or -infinity.
			i = a.sign_bias_exp;
			super_zero ( a);
			a.validity = ( i > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow during normalization in super_norm";
			cout << " -- fatal error.\n";
			exit (1);
		}
	}

	l = a.mantissa [nf - 1];
	super_shift ( a, shift_cnt);
	j = - shift_cnt;
//	k = 1;
//	while ( j--) k *= def_radix;		// get nbase ** (-shift_cnt)
	k = radix_powers [j];			// get nbase ** (-shift_cnt)
	jklip = iklip * k;
	if ( ( l % jklip) >= ( jklip + 1) / 2)
	{
		super_zqncc ( &a.mantissa [0], nf, ind, iklip);
	}
	if ( a.mantissa [0] < ind) return;

	a.mantissa [0] /= def_radix;
	if ( abs ( a.sign_bias_exp) == 2 * bias + 1)
	{
		if ( permissive_error)
		{		// Set to +infinity or -infinity.
			i = a.sign_bias_exp;
			super_zero ( a);
			a.validity = ( i > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow during normalization in super_norm";
			cout << " -- fatal error.\n";
			exit (1);
		}
	}

	if ( a.sign_bias_exp > 0)
		a.sign_bias_exp ++;
	else
		a.sign_bias_exp --;
	
	return;
}					// end of super_norm

void super_precision :: super_shift				// PROTECTED FN
			( super_precision &a, int32 shift_cnt) const
{								// shift a
// super_shift performs a shift of the mantissa with an accompanying
// adjustment of the exponent, so as to keep the same value.  If shift_cnt is
// positive a left shift is done; if shift_cnt is negative a right shift is
// done.  An exponent underflow is converted to the zero representation.
// An exponent overflow is a fatal error.
// An excessive left shift can result in a hardware arithmetic error in
// the called function super_zqmnc.
// The right shift truncates digits shifted beyond the number of digits
// (no. = def_digits) specified for mantissas, but a possible round bit
// "irbit" is set to iklip when a roundup is warranted.
// A left shift brings in any digits in the last mantissa element beyond
// def_digits count of digits, and does not perform a clip of the result to
// limit it to def_digits.  Thus, a left shift by super_shift should be followed
// by a call to super_klipr to correctly zero out the excess digits in the last
// mantissa element.
	int32 i, j, k, l, m;			// temporaries

			// If argument is invalid, return with no change.
	if ( a.assert_invalid ( "super_shift")) return;

	// If shift count is zero, just return.
	if ( shift_cnt == 0) return;

	irbit = 0;					// clear the round bit
	// Test to see if a right shift shifts all digits off the mantissa:
	if ( shift_cnt <= - def_digits)
	{
		i = 2 * a.mantissa [0];
		if ( i > ind) irbit = iklip;		// set a round bit
		else if ( i == ind) irbit = -iklip;	// set a semi-round bit
		super_zero ( a);
		return;
	}

	// Test for exponent underflow:
	if ( shift_cnt >= abs ( a.sign_bias_exp))
	{
		super_zero ( a);
		return;
	}

	// Test for exponent overflow:
	if ( ( shift_cnt < 0) &&
		( shift_cnt < ( abs ( a.sign_bias_exp) - ( 1 + bias + bias))))
	{
		if ( permissive_error)
		{		// Set to +infinity or -infinity.
			i = a.sign_bias_exp;
			super_zero ( a);
			a.validity = ( i > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "\nOverflow in super_shift -- fatal error.\n";
			cout << "shift count = " << shift_cnt;
			cout<<"  a.sign_bias_exp = " << a.sign_bias_exp << endl;
			exit (1);
		}
	}

	// Get adjusted exponent:
	i = abs ( shift_cnt);
	j = i / nd;
	k = i - j * nd;
	if ( a.sign_bias_exp > 0)
		a.sign_bias_exp -= shift_cnt;
	else
		a.sign_bias_exp += shift_cnt;
	
	// Shift mantissa in left or right direction.
	if ( shift_cnt > 0)
	{
		// Shift left:
		if ( j != 0)
		{			// shift whole words by j locations
			m = 0;
			for ( l = 0; l < nf; l++)
			{
				if ( l <= j - 1) continue;
				a.mantissa [m] = a.mantissa [l];
				m++;
			}
			for ( ; m < nf; m++)	// note: start with previous m
			{		// zero fill remaining words
				a.mantissa [m] = 0;
			}
		}
		if ( k != 0)
		{			// shift digits by k positions
//			m = 1;
//			while ( k--) m *= def_radix;	// m = nbase**k
			m = radix_powers [k];		// m = nbase**k
			super_zqmnc ( a.mantissa, m, ind, nf - j);
		}
	}
	else
	{
		// Shift right:
		if ( j != 0)
		{			// shift whole words by j locations
			i = a.mantissa [nf - j];	// last is (nf-1)'th
			for ( l = 0, m = nf; l < nf; l++)
			{
				if ( l <= j - 1) continue;
				m--;
				a.mantissa [m] = a.mantissa [nf - l - 1];
			}
			for ( l = 0; l < j; l++)
			{			// zero fill on left
				a.mantissa [l] = 0;
			}
		}
		if ( k != 0)
		{			// shift digits by k positions
//			m = 1;
//			while ( k--) m *= def_radix;	// m = nbase**k
			m = radix_powers [k];		// m = nbase**k
			super_zqdnc ( &a.mantissa [j], m, ind, nf - j, i);
	//		if ( klip)	// <-- is the original FORTRAN code
			if ( ( klip != 0) && ( iklip != 1))
			{
				i = a.mantissa [nf - 1] % iklip;
				m = iklip;
			}
		}

		j = ( m + 1) / 2;
		if ( i > j)  irbit = iklip;		// set a round bit
		else if ( i == j) irbit = -iklip;	// set a semi-round bit
	}
	
	// Clip result.
	j = rounding_method;
	rounding_method = 0;
	if ( ( shift_cnt < 0) && ( klip > 0)) super_klipr ( a);
	rounding_method = j;

	return;
}					// end of super_shift

void super_precision :: super_add				// PROTECTED FN
				( super_precision &a, const super_precision &b,
				const super_precision &c) const
{								// add a=b+c
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i, iexp, ib, ic, id;		// temporaries

	if ( b.assert_invalid ( c, "super_add"))
	{
		a = super_NaN ( super_irreg_fsa ( "add",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
		return;
	}

	// Check for zero operands -- operands are assumed to be well-formed.
	if ( b.sign_bias_exp == 0)
	{
		super_move ( a, c);		// b = 0, so copy c to a
		if ( klip)  super_klipr ( a);
		return;
	}
	if ( c.sign_bias_exp == 0)
	{
		super_move ( a, b);
		if ( klip)  super_klipr ( a);
		return;
	}

	// Both operands are nonzero.  Calculate shift count to equalize exp's:
	ib = abs ( b.sign_bias_exp);
	ic = abs ( c.sign_bias_exp);
	id = ib - ic;
	// If exponents are unequal and signs are unequal, test to see if a
	// guard digit should be carried.
	if ( id &&
		 (   ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp > 0))
		  || ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp < 0))))
	{
//		if ( ( ib > ic) && ( b.mantissa [0] < 2 * ( ind / def_radix)))
		if ( ( ib > ic) && ( b.mantissa [0] < 2 * radix_powers [nd -1]))
		{
			// Guard digit will be carried in b.mantissa:
			super_move ( *zv, c);
			if ( id != 1) super_shift ( *zv, 1 - id);
			super_move ( *w, b);
			super_zqmnc ( w->mantissa, def_radix, ind, nf);
			iexp = w->sign_bias_exp;
			w->sign_bias_exp = ( iexp > 0) ? abs ( ib - 1)
						      : - abs ( ib - 1);
			super_zmi ( *zv, *w, *zv);
			super_zqdnc ( w->mantissa, def_radix, ind, nf, i);
			super_move ( a, *zv);
			super_norm ( a, ib);
			if ( klip) super_klipr ( a);
			return;
		}
		else
		{	if ( ( ib < ic) &&
//				( c.mantissa [0] < 2 * ( ind / def_radix)))
				( c.mantissa [0] < 2 * radix_powers [nd - 1]))
			{
				// Guard digit will be carried in c.mantissa:
				super_move ( *zv, b);
				if ( id != 1) super_shift ( *zv, 1 + id);
				super_move ( *w, c);
				super_zqmnc ( w->mantissa, def_radix, ind, nf);
				iexp = w->sign_bias_exp;
				w->sign_bias_exp = ( iexp > 0) ? abs ( ic - 1)
							      : - abs ( ic - 1);
				super_zmi ( *zv, *w, *zv);
				super_zqdnc ( w->mantissa, def_radix, ind,nf,i);
				super_move ( a, *zv);
				super_norm ( a, ib);
				if ( klip) super_klipr ( a);
				return;
			}
		}
	}

	// No guard digit needed.  Do it the easy way.
	// Check for case of equal exponents:
	if ( !id)
	{
		irbit = 0;
		if ( ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0))
		  || ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)))
		{			// same signs, so add mantissas.
			super_zad ( a, b, c);		// uses irbit to round
		}
		else
		{			// opposite signs, so subtract mantissas
			super_zmi ( a, b, c);		// uses irbit to round
		}
		super_norm ( a, ib);
		if ( klip) super_klipr ( a);
		return;
	}

	// Unequal exponents.
	if ( id > 0)
	{
		// b is bigger than c.
		super_move ( *zv, c);
		super_shift ( *zv, -id);			// sets irbit
		if ( ( ( b.sign_bias_exp < 0) && ( zv->sign_bias_exp < 0))
		  || ( ( b.sign_bias_exp > 0) && ( zv->sign_bias_exp > 0)))
		{			// same signs, so add mantissas.
			super_zad ( a, b, *zv);		// uses irbit to round
		}
		else
		{			// opposite signs, so subtract mantissas
			super_zmi ( a, b, *zv);		// uses irbit to round
		}
	}
	else
	{
		// c is bigger than b.
		super_move ( *zv, b);
		super_shift ( *zv, id);				// sets irbit
		if ( ( ( c.sign_bias_exp < 0) && ( zv->sign_bias_exp < 0))
		  || ( ( c.sign_bias_exp > 0) && ( zv->sign_bias_exp > 0)))
		{			// same signs, so add mantissas.
			super_zad ( a, c, *zv);		// uses irbit to round
		}
		else
		{			// opposite signs, so subtract mantissas
			super_zmi ( a, c, *zv);		// uses irbit to round
		}
	}
	// Finish up with normalization and clipping:
	super_norm ( a, ib);
	if ( klip) super_klipr ( a);
	return;
}					// end of super_add

void super_precision :: super_minus				// PROTECTED FN
				( super_precision &a, const super_precision &b,
				const super_precision &c) const
{							// subtract a=b-c
// Self-modification of the arguments is allowed -- they may be the same object.

	if ( b.assert_invalid ( c, "super_minus"))
	{
		a = super_NaN ( super_irreg_fsa ( "minus",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
		return;
	}

	super_move ( *zt, c);
	zt->sign_bias_exp = - zt->sign_bias_exp;
	super_add ( a, b, *zt);
}

void super_precision :: super_neg				// PROTECTED FN
			( super_precision &a, const super_precision &b) const
{							// negative a=-b
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i;

	if ( b.assert_invalid ( "super_neg"))
	{
		a = super_NaN ( super_irreg_fsa ( "neg",
					b.validity, b.sign_bias_exp));
		return;
	}

	a.sign_bias_exp = - b.sign_bias_exp;
	a.digits = b.digits;
	a.radix = b.radix;
	a.validity = b.validity;
	for ( i = 0; i < nf; i++)
	{
		a.mantissa [i] = b.mantissa [i];
	}
}

void super_precision :: super_neg				// PROTECTED FN
			( void)
{
// This function changes the sign of the *this super_precision object,
// without checking for validity.

	this->sign_bias_exp = - this->sign_bias_exp;
}

void super_precision :: super_mult				// PROTECTED FN
			( super_precision &a, const super_precision &b,
			const super_precision &c) const
{								// a = b * c
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 intrz, kr, i, j, k, l, ij, kk, save;	// temporaries

//cout<<"debug_counter="<<++debug_counter<<endl;
++debug_counter;

	if ( b.assert_invalid ( c, "super_mult"))
	{
		a = super_NaN ( super_irreg_fsa ( "mult",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
		return;
	}

	if ( ( b.sign_bias_exp == 0) || ( c.sign_bias_exp == 0))
	{
		super_zero ( a);
		return;
	}

	if ( nfah == nfahx)
	{
			// intrz is clipping factor for array of length nfahx.
		intrz = indtrz;
	}
	else
	{
		intrz = indtrz / indah;
	}

// Fix final sign before getting too involved.
	int final_sign;
	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
	     ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0))  )
	{
		final_sign = 1;
	}
	else
	{
		final_sign= -1;
	}

// Choose between conventional and fast (FFT-based) multiplication algorithms.
    if ( !fast_mult_digits || ( b.digits < fast_mult_digits))
    {
	// Use conventional method.

	super_unpk ( b, t);
	super_unpk ( c, v);

			// Roundoff strategy:
			// If ndig > def_digits (and by construction ndah*nfahx
			// >= ndig), then no round digit add-in is needed
			// because extra digits allow klipr to make the correct
			// roundoff.
			// On the other hand, if ndig == def_digits then we need
			// a round digit add-in.  There are two cases:
			// (1) If ndah*nfahx == def_digits == ndig (in which
			// case intrz == 1, also), then shift the mantissa array
			// left one digit and add def_radix/2 to the low digit.
			// (2) Else (i.e., ndah*nfahx > ndig), add intrz/2 to
			// the low word in the mantissa array.

			// Test for need for 1-digit left shift.
	if ( intrz == 1) super_zqmnc ( v, def_radix, indah, nfahx);

			// Define round digit kr.
	if ( rounding_method && iklip == 1)
	{
		kr = ( intrz * indah) / 2;
		if ( intrz != 1) kr /= def_radix;	// not shifted 1 digit
	}
	else
	{			// Either truncation case, or ndig > def_digits.
		kr = 0;
	}

	ij = 0;			// Clear exponent adjustment for shifts.

	k = 0;
	l = 2;
	super_zqm1 ( t, v, l, nfahx, k, indah);
	k += kr;
	if ( intrz != 1)
	{
		super_zqm2 ( t, v, nfahx, k);
	}
	else
	{
		super_zqm2lsh ( t, v, nfahx, k, indah);
	}
	k = t [nfahx - 1] / indah;
	kk = t [nfahx - 1] - k * indah;
	super_zqnc ( t, nfahx - 1, indah, k, 1);
	t [0] = k;

			// If round digit added in and 1-digit left shift done,
			// AND leading t-element >= indah:
	if ( ( kr != 0) && ( ( intrz == 1 && t [0] >= indah)
			// Or if round digit added in and 1-digit left shift NOT
			// done (i.e., intrz != 1):
//		|| ( intrz != 1 && t [0] >= indah / def_radix)))
		|| ( intrz != 1 && t [0] >= radix_powers [ndah - 1])))
	{		// Then move round digit one digit to the left (multiply
			// original round digit by def_radix-1 and add in again.
		kr = ( def_radix - 1) * kr;
		k = ( kk + kr) / indah;
		save = t [0];
		super_zqncc ( t, nfahx, indah, k);
				// If leading digit was a zero and it became
				// nonzero, or if excess leading digit changes
				// from zero to nonzero, increment exponent.
//		if ( ( save < indah / def_radix && t [0] >= indah /
		if ( ( save < radix_powers [ndah - 1] && t [0] >= indah /
			def_radix) || ( save < indah && t [0] >= indah))
			ij++;
	}

//	i = indah / def_radix;
	i = radix_powers [ndah - 1];
	if ( i == 1) i = def_radix;
	if ( ( i == 1) && ( def_radix <= t [0])) ij = 1;
	if ( ( i <= t [0]) && ( intrz != 1)) ij = 1;
	if ( ( indah <= t [0]) && ( intrz == 1)) ij = 1;

			// Repack back into result mantissa.
	super_repk ( a, t);
	i = abs ( b.sign_bias_exp) - bias - 1;
	j = abs ( c.sign_bias_exp) - bias + ij;
		// Detect whether i+j would overflow 32-bit signed-integer.
		if ( i > 0 && j > 0)
		{			// bias + bias + 1 is greatest possible.
			if ( i > ( bias + bias + 1) - j)
			{
				if ( permissive_error)
				{		//Set to +infinity or -infinity.
					a.validity = ( final_sign > 0) ? 1 : 2;
					return;
				}
				else
				{
					cout << "Overflow in super_mult -- ";
					cout << "fatal error." << endl;
					exit (1);
				}
			}
		}
	j += i;
	// Detect whether j + bias would overflow 32-bit signed-integer.
	if ( j > bias + 1)
	{
		if ( permissive_error)
		{		//Set to +infinity or -infinity.
			a.validity = ( final_sign > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow in super_mult -- fatal error.\n";
			exit (1);
		}
	}
	j += bias;
	if ( j <= 0)
	{
		super_zero ( a);
		return;
	}
	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
	     ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0))  )
	{
		a.sign_bias_exp = j;
	}
	else
	{
		a.sign_bias_exp = -j;
	}

	if ( klip) super_klipr ( a);
	return;
    }

    else
    {
	// Use FFT-based fast multiplication algorithm to multiply the mantissas
	// of b and c, and return the result in array t.
	super_fft_convol ( b, c, t);


	ij = 0;			// Clear exponent adjustment for shifts.
				// Assume leading digit is zero.
	int32 kr, which;

		// Determine round digit kr, if needed.
	if ( rounding_method && iklip == 1)
	{
		kr = ( intrz * indah) / ( 2 * def_radix);
		if ( intrz / def_radix >= def_radix)
		{
			kr /= indah;
			which = nfahx;
		}
		else
		{
			which = nfahx + 1;
		}
			// Scale integer array.
		k = kr;
		super_zqnc ( t, which, indah, k, 0);
		t [0] += k * indah;
			// If leading digit is nonzero, increment exponent.
//		if ( t [0] >= indah / def_radix) ij++;
		if ( t [0] >= radix_powers [ndah - 1]) ij++;

			// Repack into result operand with correct rounding.
//		if ( indah / def_radix <= t [0])
		if ( radix_powers [ndah - 1] <= t [0])
		{
			kr = ( def_radix - 1) * kr;
			k = kr;
			save = t [0];
			super_zqncc ( t, which, indah, k);
				// If leading digit was a zero and it became
				// nonzero, or if excess leading digit changes
				// from zero to nonzero, increment exponent.
//			if ( ( save < indah / def_radix && t [0] >= indah /
//				def_radix) || ( save < indah && t [0] >= indah))
			if ( ( save < radix_powers [ndah - 1]
				&& t [0] >= radix_powers [ndah - 1])
				|| ( save < indah && t [0] >= indah))
				ij++;
		}
	}
	else
	{			// No round digit needed.
		kr = 0;
		which = nfahx;

			// Scale integer array.
		k = kr;
		super_zqnc ( t, which + 1, indah, k, 0);
		t [0] += k * indah;
			// If leading digit is nonzero, increment exponent.
//		if ( t [0] >= indah / def_radix) ij++;
		if ( t [0] >= radix_powers [ndah - 1]) ij++;

		// Shift left one digit if needed to normalize first element
		// so round digit can fit in.
//		if ( indah / def_radix > t [0])
		if ( radix_powers [ndah - 1] > t [0])
		{
			super_zqmnc ( t,  def_radix, indah, which + 1);
		}
	}

	// If a round digit is used and the leading mantissa digit is zero
	// (so that a left-shift will be needed to normalize, we need go ahead
	// and to shift left one digit with a carry-in from the right, in order
	// to avoid missing the last digit (which is a carry-in from the right).
//	if ( kr && t [0] < indah / def_radix)
	if ( kr && t [0] < radix_powers [ndah - 1])
	{
		super_zqmnc ( t, def_radix, indah, nfahx + 1);
	}

			// Repack back into result mantissa.
	super_repk ( a, t);
	i = abs ( b.sign_bias_exp) - bias - 1;
	j = abs ( c.sign_bias_exp) - bias + ij;
	j += i;
	if ( ( j > 0) && ( j > bias + 1))
	{
		if ( permissive_error)
		{		//Set to +infinity or -infinity.
			a.validity = ( final_sign > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow in super_mult -- fatal error.\n";
			exit (1);
		}
	}
	j += bias;
	if ( j <= 0)
	{
		super_zero ( a);
		return;
	}
	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
	     ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0))  )
	{
		a.sign_bias_exp = j;
	}
	else
	{
		a.sign_bias_exp = -j;
	}

	if ( klip) super_klipr ( a);
	return;
    }
}

void super_precision :: super_fft_convol			// PROTECTED FN
			( const super_precision & b, const super_precision & c,
			int32 t []) const
{							// FFT convolution
// This function obtains the numerical convolution of the mantissas of b and c,
// transforming them both into the frequency domain using an FFT algorithm,
// taking the product in the frequency domain, and inverse transforming back
// from the frequency domain.  Conversion between integer type and double type
// is also done before and after the transforms.  The resulting integer array
// is returned as argument t.

	int32 i, j, ireturn;

		// Unpack mantissas into double array, alternating elements,
		// padding with zeros throughout the rest of the array.
	super_dunpk ( b.mantissa, dalt, nf, ind, nfft);
	super_dunpk ( c.mantissa, dalt + 1, nf, ind, nfft);
	
		// Do forward FFT.
	ireturn = f4t ( dalt, nfft, nfft, 1, -1);
	
		// Obtain convolution product from packed transform.
	double tmp_real, tmp_imag;
	double scale = 1 / (double) nfft;
	dalt [0] = dalt [0] * dalt [1] * scale;
	dalt [1] = 0;
	for ( i = 1, j = nfft - i; i <= nfft / 2; i++, j--)
	{
		tmp_real = scale * 0.5 * ( dalt [2 * i] * dalt [2 * i + 1]
				+ dalt [2 * j] * dalt [2 * j + 1]);
		tmp_imag = scale * 0.25 *  ( ( dalt [2 * j] * dalt [2 * j]
				+ dalt [2 * i + 1] * dalt [2 * i + 1])
				- ( dalt [2 * i] * dalt [ 2 * i]
				+ dalt [2 * j + 1] * dalt [2 * j + 1]));
		dalt [2 * j] = 0;
		dalt [2 * j + 1] = 0;
		dalt [2 * i] = tmp_real;
		dalt [2 * i + 1] = tmp_imag;
	}

		// Setup for transform back to consecutive elements.
	ireturn = jamcor ( dalt, nfft, 1);

		// Do inverse FFT (halfsize because of jamcor's work).
	ireturn = f4t ( dalt, nfft/2, nfft/2, 1, 1);

		// Scale double array to size dinda. 
	double dscale = (double) dinda;
	double dcarry = 0;
	i = dnfa * 2;
	while ( --i >= 0)
	{
		dalt [i] += dcarry;
					// Add 0.5 to round off trunc.
//		dcarry = trunc ( ( 0.5 + dalt [i]) / dscale);
// "trunc" not in gcc 2.7 include library.  Use the following line:
		dcarry = (double) (int32) ( ( 0.5 + dalt [i]) / dscale);
		dalt [i] = dalt [i] - dcarry * dscale;
	}
	dalt [0] += dcarry * dscale;

		// Repack and convert to integer array of length nfahx+1.
	super_drepk ( t, dalt, nfahx + 1, 2 * dnfa);

	return;
}

void super_precision :: array_fft_convol			// PROTECTED FN
			( int32 b [], int32 c [], int32 t [],
			int32 array_length, int32 product_length,
			int32 result_length) const
{							// FFT convolution
// This function obtains the numerical convolution of the arrays b and c,
// transforming them both into the frequency domain using an FFT algorithm,
// taking the product in the frequency domain, and inverse transforming back
// from the frequency domain.  Conversion between integer type and double type
// is also done before and after the transforms.  The resulting integer array
// is returned as argument t.
// The working length of b is array_length, the working length of c is
// product_length, and the working length of t is result_length.
// It is required that array_length <= product_length,
// and product_length <= result_length.

	int32 i, j, ireturn, nfftx, half_nfftx;

		// Figure number of doubles to hold packed digits, and FFT size.
	int32 dnfax = ( ndah * product_length + dnda - 1) / dnda;
	half_nfftx = 1;
	while ( half_nfftx < dnfax) half_nfftx *= 2;
	nfftx = 2 * half_nfftx;
		// Note: Strictly speaking, the zero length of the FFT input
		// need only be >= the SUM of the nonzero LENGTHS dnfax 
		// computed for both array_length and product_length --
		// instead of >= TWICE the GREATER nonzero length dnfax so
		// computed.  However, an analysis shows that smaller FFT's
		// cannot be used to advantage in our situation and therefore
		// no advantage would accrue.  Hence we just pretend the arrays
		// are of equal length.

		// Unpack arrays into double array, alternating elements,
		// padding with zeros throughout the rest of the array.
	super_dunpk ( b, dalt, array_length, indah, nfftx);
	super_dunpk ( c, dalt + 1, product_length, indah, nfftx);
	
		// Do forward FFT.
	ireturn = f4t ( dalt, nfftx, nfftx, 1, -1);
	
		// Obtain convolution product from packed transform.
	double tmp_real, tmp_imag;
	double scale = 1 / (double) nfftx;
	dalt [0] = dalt [0] * dalt [1] * scale;
	dalt [1] = 0;
	for ( i = 1, j = nfftx - i; i <= nfftx / 2; i++, j--)
	{
		tmp_real = scale * 0.5 * ( dalt [2 * i] * dalt [2 * i + 1]
				+ dalt [2 * j] * dalt [2 * j + 1]);
		tmp_imag = scale * 0.25 *  ( ( dalt [2 * j] * dalt [2 * j]
				+ dalt [2 * i + 1] * dalt [2 * i + 1])
				- ( dalt [2 * i] * dalt [ 2 * i]
				+ dalt [2 * j + 1] * dalt [2 * j + 1]));
		dalt [2 * j] = 0;
		dalt [2 * j + 1] = 0;
		dalt [2 * i] = tmp_real;
		dalt [2 * i + 1] = tmp_imag;
	}

		// Setup for transform back to consecutive elements.
	ireturn = jamcor ( dalt, nfftx, 1);

		// Do inverse FFT (halfsize because of jamcor's work).
	ireturn = f4t ( dalt, half_nfftx, half_nfftx, 1, 1);

		// Scale double array to limit dinda. 
	double dscale = (double) dinda;
	double dcarry = 0;
//TEST FIX	
//	i = nfftx;
	i = array_length + product_length - 1;
	while ( --i >= 0)
	{
		dalt [i] += dcarry;
					// Add 0.5 to round off trunc.
//		dcarry = trunc ( ( 0.5 + dalt [i]) / dscale);
// "trunc" not in gcc 2.7 include library.  Use the following line:
		dcarry = (double) (int32) ( ( 0.5 + dalt [i]) / dscale);
		dalt [i] = dalt [i] - dcarry * dscale;
	}
	dalt [0] += dcarry * dscale;

		// Repack and convert to integer array of length result_length.
	super_drepk ( t, dalt, result_length, nfftx);

	return;
}



void super_precision :: super_multi				// PROTECTED FN
			( super_precision &a, const super_precision &b,
			const int32 ia) const
{								// a = b * i
// Self-modification of the arguments is allowed -- they may be the same object.

	if ( b.assert_invalid ( "super_multi"))
	{
		a = super_NaN ( super_irreg_fsa ( "multi",
					b.validity, b.sign_bias_exp,
					0, ia));
		return;
	}

	if ( ia == 1 || ia == -1)
	{		// Case: ia is + or - unity.  Just copy out b.
		super_move ( a, b);
		if ( ia < 0) a.sign_bias_exp = - a.sign_bias_exp;
		return;
	}

	if ( ia != 0 && b.sign_bias_exp != 0)
	{
		int32 tmp = ia, tmp1;
		tmp1 = tmp % indah;
		tmp /= indah;
		if ( tmp) { super_move ( *w, b);}
		super_zmu ( a, b, tmp1);
		while ( tmp)
		{
			w->sign_bias_exp += ( w->sign_bias_exp > 0) ? ndah
								    : -ndah;
			// It would be nice to check for an overflow!!
			tmp1 = tmp % indah;
			tmp /= indah;
			if ( tmp1)
			{
				super_zmu ( *zt, *w, tmp1);
				super_add ( a, a, *zt);
			}
		}
	}
	else
	{		// Case: a, ia are either or both zero.
		super_zero ( a);
	}

	return;
}


void super_precision :: super_itoz				// PROTECTED FN
			( super_precision &a, const int32 ia) const
{							// convert a=i
// Converts an integer ia to super_precision a.  Conversion is exact unless
// the super_precision mantissa is too short to hold ia, in which case ia is
// truncated or rounded as specified by rounding_method.
	int32 i, j, k, l, ib, ndg;		// temporaries

	// Test for case ia = 0:
	if ( ia == 0)
	{
		super_zero ( a);
		return;
	}

	ib = abs ( ia);
	// Test for single-digit integer:
	if ( ib < def_radix)
	{
		a.sign_bias_exp = ( ia > 0) ? ( bias + 1)
					    : - ( bias + 1);
//		a.mantissa [0] = ib * ( ind / def_radix);
		a.mantissa [0] = ib * ( radix_powers [nd - 1]);
		i = 0;
		while ( ++i < nf)  a.mantissa [i] = 0;
		return;
	}

	// Test for exact conversion:
	j = nf;
	while ( j)
	{
		k = ib / ind;
		a.mantissa [ --j] = ib - k * ind;
		ib = k;
		if ( ib == 0) break;
	}
	while ( j)
	{
		a.mantissa [ --j] = 0;
	}
	// If ib was reduced to zero, the conversion is exact.
	if ( !ib)
	{		// Figure sign and exponent, then normalize and clip.
		a.sign_bias_exp = ( ia > 0) ? ( bias + ndig)
					    : - ( bias + ndig);
		super_norm ( a, i);
		// This normalization will be a left shift, and therefore we
		// need to clip the result.
		if ( klip) super_klipr ( a);
		return;
	}

	// Conversion is not exact, so have to chop off some significance.
	ib = k * ind + a.mantissa [0];
	ndg = ndig;
	while ( ib >= ind)
	{
		l = a.mantissa [nf - 1];
		k = nf;
		if ( --k)
		{
			a.mantissa [k] = a.mantissa [k - 1];
		}			
		k = ib / ind;
		a.mantissa [0] = ib - k * ind;
		ib = k;
		ndg += nd;
		if ( rounding_method)
		{
			if ( l >= ( ind + 1) / 2)
				super_zqncc ( a.mantissa, nf, ind, 1);
			k = a.mantissa [0] / ind;
			a.mantissa [0] -= k * ind;
			ib += k;
		}
	}
	a.sign_bias_exp = ( ia > 0) ? ( bias + ndg)
				    : - ( bias + ndg);
	while ( ib > 0)
	{
		k = a.mantissa [nf - 1] / def_radix;
		l = a.mantissa [nf - 1] - def_radix * k;
		super_shift ( a, -1);
		k = ib / def_radix;
//		a.mantissa [0] += ( ib - k * def_radix) * ( ind / def_radix);
		a.mantissa [0] += ( ib - k * def_radix) * radix_powers [nd - 1];
		ib = k;
		if ( rounding_method)
		{
			if ( l >= ( def_radix + 1) / 2)
				super_zqncc ( a.mantissa, nf, ind, 1);
			k = a.mantissa [0] / ind;
			a.mantissa [0] -= k * ind;
			ib += k;
		}
	}
	if ( klip) super_klipr ( a);
	return;
}

void super_precision :: super_div				// PROTECTED FN
			( super_precision &a, const super_precision &b,
			const super_precision &c) const
{								// a = b / c
// This function performs division of two super_precision objects, using either
// the divide_and_correct "high school" algorithm or a "fast divide" algorithm
// based on using FFT's to obtain product convolutions.  The fast divide is
// actually faster only for long mantissas, more than approximately 500 decimal
// digits in length.
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i, j, k;
	int32 lb, np1, i3, i4, i5, isave, ish, shift_flag, intrz, ic,
		lll, jjj, iii, iloop, is, jc;
	double d, e;
//cout<<"debug_counter_divides="<<++debug_counter_divides<<endl;
++debug_counter_divides;

	if ( b.assert_invalid ( c, "super_div"))
	{
		a = super_NaN ( super_irreg_fsa ( "div",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
		return;
	}

	// Check for divide by zero:
	if ( !c.sign_bias_exp)
	{				// c is zero
		if ( b.sign_bias_exp)
		{			// b is nonzero
			if ( permissive_error)
			{		//Set to +-infinity.
				super_zero ( a);
				a.validity = 3;
				return;
			}
			else
			{
				cout << "Super_div divide by zero -- "
					<< "fatal error\n";
				exit (1);
			}
		}
		else
		{			// b is zero, too
			if ( permissive_error)
			{		//Set to Not-a-Number (NaN).
				super_zero ( a);
				a.validity = 4;
				return;
			}
			else
			{
				cout << "Super_div divide zero by zero --"
					<< " fatal error\n";
				exit (1);
			}
		}
	}
	
	// Check for zero dividend:
	if ( !b.sign_bias_exp)
	{				// b is zero, but c is nonzero
		super_zero ( a);
		return;
	}

// Fix final sign before getting too involved.
	int final_sign;
	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
	     ( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0))  )
	{
		final_sign = 1;
	}
	else
	{
		final_sign= -1;
	}

// Choose between conventional and fast (FFT-based) division algorithms.
    if ( !fast_div_digits || ( b.digits < fast_div_digits))
    {
	// Use conventional method.

	// Do a full divide.  Unpack the dividend and divisor.
	super_unpk ( b, t);
	super_unpk ( c, v);
//	lb = indah / def_radix;
	lb = radix_powers [ndah - 1];
	np1 = nfahx + 1;
	i4 = ( ndiv > nfahx) ? nfahx : ndiv;	// min ( ndiv, nfahx)
	i3 = i4 - 1;
	i5 = i4 + 1;
	isave = 0;
	ish = 0;
	shift_flag = TRUE;
	for ( i = 0; i < nf; i++)
	{
		if ( b.mantissa [i] != c.mantissa [i])
		{
			if ( b.mantissa [i] < c.mantissa [i]) shift_flag =FALSE;
			break;
		}
	}
	if ( shift_flag)
	{
		super_zqdnc ( t, def_radix, indah, nfahx, isave);
		ish = 1;
	}

	d = 0;
	for ( i = 0; i < i4; i++)
	{
		d *= indah;
		d += v [i];
	}
	d /= indah;

	if ( rounding_method)
	{
			// intrz is clipping factor for array of length nfahx.
		intrz = indtrz;
		if ( nfahx != nfah) intrz /= indah;
		if ( ( ish != 0) && ( intrz <= 1))
		{
			i = ( v [0] * indah + v [1]) / 2;
			j = i / indah;
			i -= j * indah;
			j += isave * lb;
			zib [0] = j;
			zib [1] = i;
			if ( zib [0] >= indah)
			{
				k = zib [0] / indah;
				zib [0] -= k * indah;
				super_zqncc ( t, nfahx, indah, k);
			}
		}
		else
		{
			i = ( v [0] * intrz + ( v [1] * intrz) / indah) / 2;
			j = i / indah;
			i -= j * indah;
			t [nfahx - 1] += j;
			zib [0] = i;
			zib [1] = 0;
		}
		for ( i = 2; i < i4; i++) zib [i] = 0;
	}
	else
	{
		zib [0] = isave * lb;
		zib [1] = 0;
		for ( i = 2; i < i4; i++) zib [i] = 0;
	}

	lll = indah;
	jjj = ind;
	iii = 0;
	a.mantissa [0] = 0;
	for ( iloop = 0; iloop < nfahx; iloop += i4)
	{
		is = 0;
		e = (double) t [0];
		for ( i = 1; i < i4; i++)
		{
			e *= (double) indah;
			e += (double) t [i];
		}
		e += (double) 1;
		e /= d;
		zia [0] = (int32) e;
		for ( i = 0; i < i3; i++)
		{
			e -= (double) zia [i];
			e *= (double) indah;
			zia [i + 1] = (int32) e;
		}

		jc = 0;
		super_zqd1 ( v, zia, zib, jc, i4, np1, indah);
		super_zqd2 ( t, v, zia, jc, nfahx, i4, indah);
		t [0] -= jc;

		// Correction loop:
		// First, force t [0] < 0.
		while ( t [0] >= 0)
		{
			ic = 0;
			i = np1 - i4;
			super_zqsnc ( zib, zib, v + i - 1, i4, indah, ic);
			if ( nfahx > ndiv)
			{
				super_zqsnc ( t + i5 - 1, t + i5 - 1, v,
					np1 - i5, indah, ic);
			}
			if ( ic) super_zqncb ( t, i4, indah, ic);
			super_zqncc ( zia, i4, indah, 1);
		}
		// Second, adjust till t [0] is just positive.
		while ( t [0] < 0)
		{			// add to remainder until t [0] > 0
			ic = 0;
			i = np1 - i4;
			super_zqanc ( zib, zib, v + i - 1, i4, indah, ic);
			if ( nfahx > ndiv)
			{
				super_zqanc ( t + i5 - 1, t + i5 - 1, v,
					np1 - i5, indah, ic);
			}
			super_zqncc ( t, i4, indah, ic);
			super_zqncb ( zia, i4, indah, 1);
			if ( zia [0] < 0) zia [0] += indah;
		}

		super_zqrpk ( a.mantissa, zia, i4, jjj, lll, iii, indah);
		if ( i5 <= nfahx)
		{
			for ( i = i5 - 1; i < nfahx; i++)
			{
				j = i - i4;
				t [j] = t [i];
			}
		}

		for ( i = 0; i < i4; i++)
		{
			j = nfahx - i4 + i;
			t [j] = zib [i];
		}
		for ( i = 0; i < i4; i++)
		{
			zib [i] = 0;
		}
	}

	i = abs ( b.sign_bias_exp) - bias;
	j = abs ( c.sign_bias_exp) - bias - ish;
	j = i - j;
	if ( ( j > 0) && ( j > bias + 1))
	{
		if ( permissive_error)
		{		//Set to +infinity or -infinity.
			a.validity = ( final_sign > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow in super_div -- fatal error.\n";
			exit (1);
		}
	}

	j += bias;
	if ( j <= 0)
	{
		// Convert exponent underflow to true zero:
		super_zero ( a);
		return;
	}

	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
		( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0)))
	{
		a.sign_bias_exp = j;
	}
	else
	{
		a.sign_bias_exp = -j;
	}

	if ( klip) super_klipr ( a);
	return;
    }

    else
    {
	// Use FFT-based fast division algorithm.

	int32 digit_count, nwords, product_nwords;
	int32 correction, ish = 0;
	int32 *zia_low = zia + nfahx;
	double dblcorr, dblscale;

		// Obtain approximate reciprocal of mantissa of divisor c,
		// then put into zia array.
		// Convert first two elements of mantissa to double type.
	if ( nf == 1)
	{
		d = (double) c.mantissa [0] / ind;
	}
	else
	{
		d = ( (double) c.mantissa [1] / ind + (double) c.mantissa [0])
						/ ind;
	}
	d = 1 / d;				// Take reciprocal.
	while ( d >= 1)
	{
		d /= def_radix;			// Try for range (1,1/def_radix]
		ish++;				// Keep track of exponent.
	}
		// Convert back to zia array.
	d *= indah;
	for ( i = 0; i < nfahx; i++)
	{
		zia [i] = (int32) d;
		d = indah * ( d - (double) zia [i]);
	}

		// Unpack divisor c into zib array.
	super_unpk ( c, zib);

		// Start loop to recursively build longer approximation to
		// reciprocal in zia.  Use Newton-Raphson iteration.
	digit_count = double_fraction_digit_count;	// Initial value
	nwords = ( digit_count + ndah - 1) / ndah;	// Initial value
	while ( nwords < nfahx)
	{
		// Start with array length nwords, and extend to a product
		// having length up to min(2*nwords,nfahx).
			// Compute product length.
		i = 1;
		while ( i <= nwords) i *= 2;
		product_nwords = ( i > nfahx) ? nfahx : i;

			// Multiply zia and zib arrays, result to t array.
			// zia has working length nwords, while zib and t have
			// working length product_nwords.
		array_fft_convol ( zia, zib, t, nwords, product_nwords,
				product_nwords);
			// Length of t array is product_nwords.

			// Shift left one digit.
		super_zqmnc ( t, def_radix, indah, product_nwords);
			// Maybe shift again to bring t[0] up to order indah.
//		if ( abs ( t[0] - indah/def_radix) < abs ( t[0] - indah))
		if ( abs ( t[0] - radix_powers [ndah-1]) < abs ( t[0] - indah))
		{
//cout<<"extra shift:t[0]="<<t[0]<<endl;
			super_zqmnc ( t, def_radix, indah, product_nwords);
		}

			// Subtract product t from 2, put result back in t.
			// Use "2's complement" algorithm -- take radix
			// complement and add 1 to low digit to get "negative".
		j = product_nwords;
		while ( j--)
		{
			t [j] = indah - t [j] - 1;	// Get radix complement.
		}
		super_zqncc ( t, product_nwords, indah, 1); // Add 1 at low end.
		t [0] += indah;
			// The quadratically-convergent Newton-Raphson recursion
			// actually converges at a rate a couple of bits short
			// of quadratic.  If not corrected, accuracy degrades by
			// a number of full digits when very long mantissas are
			// attained, and this can cause excessive incremental
			// correction loops in the "perfect divide" section
			// below.
			// Get the first few digits of the third-order term
			// (cubic convergence) if product_nwords = 2 * nwords.
		if ( product_nwords == 2 * nwords)
		{
			if ( t [0] >= indah)
			{		// t is greater than unity.
				dblscale = 1.0 / indah;
				i = nwords;
				dblcorr = t [i] * dblscale;
				i--;
				while ( t [i])
				{
					dblscale *= indah;
					dblcorr += t [i] * dblscale;
					i--;
					if ( i == 0) break;
				}
			}
			else
			{		// t is just less than unity.
				dblscale = 1.0 / indah;
				i = nwords;
				dblcorr = ( indah - t [i]) * dblscale;
				i--;
				while ( indah - 1 - t [i])
				{
					dblscale *= indah;
					dblcorr += ( indah - 1 - t [i]) *
								dblscale;
					i--;
					if ( i == 0) break;
				}
			}
				// Get the correction term and add it to t.
				// Note that the correction is derived from
				// the vicinity of t [nwords] and is added in
				// at t [product_nwords].
//TEST added 0.5 to next line:
			correction = (int32) (dblcorr * dblcorr + 0.5);
				// Now add the correction as a carry-in.
			super_zqncc ( t, product_nwords, indah, correction);
		}

			// Multiply zia and t arrays, result to zia array.
		array_fft_convol ( zia, t, zia, nwords, product_nwords,
				product_nwords);
			// Input length of zia array is nwords.
			// Output length of zia array is product_nwords.

			// Set next array length.
		nwords = product_nwords;
	}

		// We have 1/c in zia.
		// Do a quick test to see if dividend is unity.
	int unity = 0;
	if ( b.sign_bias_exp == bias + 1)
	{
//		if ( b.mantissa [0] == ind / def_radix)
		if ( b.mantissa [0] == radix_powers [nd - 1])
		{
			unity = 1;
			for ( i = 1; i < nf; i++)
			{
				if ( !b.mantissa [i])
					continue;
				unity = 0;	// mantissa word is nonzero.
				break;
			}
		}
	}

		// If dividend is unity, all we need is the reciprocal and we
		// can skip the next multiply.
	if ( !unity)
	{
		// Have 1/c in zia.  Discard contents of t, unpack b into t.
		super_unpk ( b, t);
		// Multiply t and zia, result to zia array.
		array_fft_convol ( t, zia, zia, nfahx, nfahx, nfahx);
			// All operands have working length nfahx.
	}
	else
	{			// Copy unity dividend (unpacked) into t.
//		t [0] = indah / def_radix;
		t [0] = radix_powers [ndah - 1];
		for ( i = 1; i < nfahx; i++) { t [i] = 0;}
		ish--;
	}

		// Normalize fraction if needed.
//	if ( zia [0] < indah / def_radix)
	if ( zia [0] < radix_powers [ndah - 1])
	{
		super_zqmnc ( zia, def_radix, indah, nfahx);
		ish--;
	}

		// Repack result back into a and get exponent.
	jjj = ind;
	iii = 0;
	lll = indah;
	a.mantissa [0] = 0;
	super_zqrpk ( a.mantissa, zia, nfahx, jjj, lll, iii, indah);

	correction = 0;
if ( perfect_fast_divide)
{
// Undertake a perfect divide by correcting the quotient.  Must get product.
// The quotient is in zia, the dividend is in t, the divisor is in zib.
// Copy the divisor to v.
	for ( i = 0; i < nfahx; i++) { v [i] = zib [i]; zib [i] = 0;}
// Put the double-length product of zia and v into zia (double-length form).
// This will overwrite zia and zib.  The low-order half of zia is zia_low.
	array_fft_convol ( zia, v, zia, nfahx, nfahx, 2 * nfahx);
// First make sure of alignment of t and zia.
	int32 hold_t = t [0], hold_zia = zia [0];
/*
	i = ( def_radix > 2) ? def_radix : 4;	// use 2 bits for base 2
	if ( ndah > 1)
	{
		iii = t [0];
		jjj = zia [0];
		j = indah / ( i * i);
		j = ( j == 0) ? 1 : j;		// FAILS for ndah < 3 bits
		iii /= j;				// first 2 digits of t
		jjj /= j;				// first 2 digits of zia
	}
	else
	{	// Need to use first two words of array since indah = def_radix.
		iii = indah * t [0] + t [1];
		jjj = indah * zia [0] + zia [1];
		iii /= ( indah * indah) / ( i * i);	// first 2 digits of t
		jjj /= ( indah * indah) / ( i * i);	// first 2 digits of zia
	}
	if ( ( jjj < i && ( iii != i || jjj != ( i - 1)))  ||
	     ( jjj = i && iii == i * i - 1 ))
*/
// The arrays t and zia will be almost exactly the same, except for a possible
// right shift needed for t.
	iii = indah * t [0] + t [1];
	jjj = indah * zia [0] + zia [1];
	if ( jjj + 1 < iii)
	{			// Right shift t.
		super_zqdnc ( t, def_radix, indah, nfahx, lll);
	}
// Subtract product zia from dividend t.  Logical extension of t is zeros.
// First, subtract low-order half of product from zero logical extension.
	for ( i = 0; i < nfahx; i++)
	{
		zia_low [i] = indah - 1 - zia_low [i];
	}
	super_zqncc ( zia_low, nfahx, indah, 1);
	zia_low [0] -= indah;
	if ( zia_low [0] >= 0)			// Get borrow lll.
	{
		lll = 0;
	}
	else
	{
		lll = 1;
		zia_low [0] += indah;
	}
// Second, subtract high-order half of product (zia) from dividend t, result
// back into zia.  Bring in borrow lll.
	super_zqsnc ( zia, t, zia, nfahx, indah, lll);
	zia [0] -= indah * lll;

// Look for correction.
	while ( zia [0] >= 0)			// Make remainder negative.
	{			// Nonnegative remainder.
		// Increment quotient and subtract divisor v from low-order
		// remainder zia_low until zia becomes negative.
		lll = 0;
		super_zqsnc ( zia_low, zia_low, v, nfahx, indah, lll);
		if ( lll) super_zqncb ( zia, nfahx, indah, lll);
		correction++;
		if (correction > 100)
		{
			cout << "Runaway correction (++) in super_div ";
			cout << "FFT-based divide -- fatal error." << endl;
			cout << "debug_counter_divides=" <<
						debug_counter_divides << endl;
			cout << "dividend [0] = " << hold_t << " divisor [0] = "
			     << hold_zia << endl;
			exit (1);
		}
	}
	while ( zia [0] < 0)			// Make remainder nonnegative.
	{			// Negative remainder.
		// Decrement quotient and add divisor v to low-order remainder
		// zia_low until zia becomes positive or zero.
		lll = 0;
		super_zqanc ( zia_low, zia_low, v, nfahx, indah, lll);
		if ( lll) super_zqncc ( zia, nfahx, indah, lll);
		correction--;
		if (correction < -100)
		{
			cout << "Runaway correction (--) in super_div ";
			cout << "FFT-based divide -- fatal error." << endl;
			cout << "debug_counter_divides=" <<
						debug_counter_divides << endl;
			cout << "dividend [0] = " << hold_t << " divisor [0] = "
			     << hold_zia << endl;
			exit (1);
		}
	}
// Now decide round digit.
	if ( rounding_method)
	{	// Do round-off, not truncation.
			// Do perfect round-off by doubling the remainder zia.
		super_zqmnc ( zia, 2, indah, 2 * nfahx);
			// Try incrementing the quotient.
		lll = 0;
		super_zqsnc ( zia_low, zia_low, v, nfahx, indah, lll);
		if ( lll) super_zqncb ( zia, nfahx, indah, lll);
			// If remainder nonnegative, round-up the quotient.
		if ( zia [0] >= 0) correction++;
	}
}

		// Get exponent.
	i = abs ( b.sign_bias_exp) - bias;
	j = abs ( c.sign_bias_exp) - bias;
		// Apply exponent adjustment to result.
	j = i - j + ish;
	if ( ( j > 0) && ( j > bias + 1))
	{
		if ( permissive_error)
		{		//Set to +infinity or -infinity.
			a.validity = ( final_sign > 0) ? 1 : 2;
			return;
		}
		else
		{
			cout << "Overflow in super_div -- fatal error.\n";
			exit (1);
		}
	}

	j += bias;
	if ( j <= 0)
	{
		// Convert exponent underflow to true zero:
		super_zero ( a);
		return;
	}

	if ( ( ( b.sign_bias_exp > 0) && ( c.sign_bias_exp > 0)) ||
		( ( b.sign_bias_exp < 0) && ( c.sign_bias_exp < 0)))
	{
		a.sign_bias_exp = j;
	}
	else
	{
		a.sign_bias_exp = -j;
	}

// Add perfect divide correction to packed quotient.
	if ( correction)
	{
		if ( correction > 0)
		{
			super_zqncc ( a.mantissa, nf, ind, correction);

		}
		else
		{
			super_zqncb ( a.mantissa, nf, ind, -correction);
		}
		super_norm ( a, i); 
	}

	if ( klip) super_klipr ( a);

	return;
    }
}

void super_precision :: super_divi				// PROTECTED FN
			( super_precision &a, const super_precision &b,
			const int32 ia) const
{
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i, k, m, ib, intrz;		// temporaries

	if ( b.assert_invalid ( "super_divi"))
	{
		a = super_NaN ( super_irreg_fsa ( "divi",
					b.validity, b.sign_bias_exp,
					0, ia));
		return;
	}

	// Test for divide by zero:
	if ( ia == 0)
	{
		if ( permissive_error)
		{
			if ( b.sign_bias_exp != 0)
			{	// Set to +-infinity.
				super_zero ( a);
				a.validity = 3;
			}
			else
			{	// Set to Not-a-Number (NaN).
				super_zero ( a);
				a.validity = 4;
			}
			return;
		}
		else
		{
			cout <<"Divide by zero in super_divi -- fatal error.\n";
			exit (1);
		}
	}

	// Test for zero dividend:
	if ( b.sign_bias_exp == 0)
	{
		super_zero ( a);
		return;
	}

	ib = abs ( ia);
	// Test for divisor greater than unpacked word maximum:
	if ( ib >= indah)
	{
		super_itoz ( *w, ia);
		super_div ( a, b, *w);
		return;
	}

	// Copy dividend to a and unpack:
	super_move ( a, b);
	super_unpk ( a, t);

	m = 1;
	k = def_radix;
	while ( k <= ib)
	{
		k *= def_radix;
		m++;
	}
	if ( ib * indah <= t [0] * k)
	{
		k /= def_radix;
		m--;
	}
	i = abs ( a.sign_bias_exp) - m;
	if ( i <= 0)
	{
		super_zero ( a);
		return;
	}
	if ( ( ( b.sign_bias_exp > 0) && ( ia > 0)) ||
	     ( ( b.sign_bias_exp < 0) && ( ia < 0)))
	{
		a.sign_bias_exp = i;
	}
	else
	{
		a.sign_bias_exp = -i;
	}
	super_zqmnc ( t, k, indah, nfahx);
	super_zqdi ( t, nfahx, ib, indah);

	i = t [ nfahx - 1];
	t [ nfahx - 1] /= ib;
			// intrz is clipping factor for array of length nfahx.
	intrz = indtrz;
	if ( nfahx != nfah) intrz /= indah;
	k = 2 * ( i / intrz - ib * ( t [nfahx - 1] / intrz));
	if ( rounding_method && ( iklip == 1) && ( ib <= k))
	{		// Do round off now instead of with clipping.
		super_zqncc ( t, nfahx, indah, intrz);
	}
		super_repk ( a, t);
	
	if ( klip) super_klipr ( a);
	return;
}

int super_precision :: super_test				// PROTECTED FN
				( const super_precision &a,
				const super_precision &b) const
{								// compare a,b
// This function tests two super_precision objects for relative numerical value.
// If the objects are numerically equal, a value of 0 is returned as the
// function value.
// If the a object is numerically greater than the b object, a value of +1 is
// returned as the function value.
// If the b object is numerically greater than the a object, a value of -1 is
// returned as the function value.
// Thus, the following equivalence is valid:
//                   a ## b  ----->  super_test ( a, b) ## 0
// where the operator ## may be any of  >, <, ==, >=, <=, !=.
// Self-modification of the arguments is allowed -- they may be the same object.
	int32 i;

	// Test for irregular forms.
	if ( a.validity || b.validity)
	{		// One or both of the operands is irregular.
			// Identify and treat resolvable cases.
		if ( a.validity && !b.validity)
		{	// a is irregular, b is not.
			if ( a.validity == 1) return +1;	// +Inf > any
			if ( a.validity == 2) return -1;	// -Inf < any
		}
		else
		{
			if ( !a.validity && b.validity)
			{	// b is irregular, a is not.
				if ( b.validity == 1) return -1;  // any < +Inf
				if ( b.validity == 2) return +1;  // any > -Inf
			}
			else
			{	// a and b are both irregular.
				if ( a.validity == 1 && ( b.validity == -1 ||
					b.validity == 2)) return +1;
				if ( a.validity == 2 && ( b.validity == -1 ||
					b.validity == 1)) return -1;
				if ( a.validity == -1 && b.validity == 1)
					return -1;
				if ( a.validity == -1 && b.validity == 2)
					return +1;
			}
		}
	}
	// NaN's are not equal or comparable to anything, even another NaN.
	// +-Inf's are not equal or comparable to anything, either.
	a.assert_invalid_quit ( b, "super_test");

	// Test for unnormalized operands.
//	i = ind / def_radix;
	i = radix_powers [nd - 1];
	if ( ( a.mantissa [0] >= i) && ( b.mantissa [0] >= i))
	{				// normalized operands

		if ( a.sign_bias_exp <= 0)
		{				// a is negative or zero
			if ( b.sign_bias_exp > 0)
			{			// b is positive
				return ( -1);	// a < b
			}
			else
			{			// b is negative or zero
				if ( a.sign_bias_exp > b.sign_bias_exp)
				{		// a is less negative than b
					return ( 1);
				}
				else
				{		// a is equal or more neg than b
					if ( a.sign_bias_exp < b.sign_bias_exp)
					{	// a is more negative than b
						return ( -1);
					}
					else
					{ // Exponents equal, compare mantissas.
						// ...below.
					}
				}
			}	// end of sign_bias_exp comparison section
			
			// Compare mantissas.
			for ( i = 0; i < nf; i++)
			{
				if ( a.mantissa [i] > b.mantissa [i])
				{	// a mantissa exceeds b mantissa
					return ( -1);
				}
				else
				{
					if ( a.mantissa [i] < b.mantissa [i])
					{	// b mantissa exceeds a mantissa
						return ( 1);
					}
					else
					{	// Try next mantissa word.
						continue;
					}
				}
			}	// end of mantissa comparison section
			// To get here the mantissas must be identical:
			return ( 0);
			
		}			// end of case for nonpositive a

		else
		{				// a is positive
			if ( b.sign_bias_exp <= 0)
			{			// b is nonpositive
				return ( 1);
			}
			else
			{		// b is positive also.
			}

			// Test exponents.
			if ( a.sign_bias_exp < b.sign_bias_exp)
			{	// a exponent < b exponent
				return ( -1);
			}
			else
			{
				if ( a.sign_bias_exp > b.sign_bias_exp)
				{	// a exponent > b exponent
					return ( 1);
				}
				else
				{	// Exponents are equal; test mantissas.
						// ...below.
				}
			}

			// Test mantissas.
			for ( i = 0; i < nf; i++)
			{
				if ( a.mantissa [i] < b.mantissa [i])
				{	// a mantissa < b mantissa
					return ( -1);
				}
				else
				{
					if ( a.mantissa [i] > b.mantissa [i])
					{ // a mantissa > b mantissa
						return ( 1);
					}
					else
					{	// Test next mantissa word.
						continue;
					}
				}
			}	// end of mantissa test section
			// To get here the mantissas must be identical:
			return ( 0);

		}			// end of case for positive a

	}				// end of normalized operand section

	else
	{				// unnormalized operands
	// Correctly formed super_precision zeros have zero sign_bias_exp words.
	// That is, a zero sign_bias_exp word identifies a super_precision zero.
		// Test for a.sign_bias_exp equal to zero.
		if ( a.sign_bias_exp == 0)
		{			// a is zero
			if ( b.sign_bias_exp > 0)
			{		// b is positive, so a < b
				return ( -1);
			}
			else
			{		// b is nonpositive
				if ( b.sign_bias_exp < 0)
				{	// b is negative, so a > b
					return ( 1);
				}
				else
				{	// b is zero, too; so a == b
					return ( 0);
				}
			}
		}	// End of section for zero a.
		
		else
		{	// a is nonzero.
		// Test for b.sign_bias_exp equal to zero.
			if ( b.sign_bias_exp == 0)
			{		// b is zero.
				if ( a.sign_bias_exp > 0)
				{	// a is positive, so a > b
					return ( 1);
				}
				else
				{
					if ( a.sign_bias_exp < 0)
					{	// a is neg, so a < b
						return ( -1);
					}
					else
					{	// a is zero, too.
						return ( 0);
					}
				}
			}		// End of case of b = 0.

			else
			{	// Operands are simply unnormalized.
				// Test signs of a and b.
			if ( ( ( a.sign_bias_exp > 0) && ( b.sign_bias_exp < 0))
			|| ( ( a.sign_bias_exp < 0) && ( b.sign_bias_exp > 0)))
			   {	// Opposite signs.
				if ( a.sign_bias_exp > 0)
				{	// a is positive, so b must be negative
					return ( 1);
				}
				else
				{	// a is negative, so b must be positive
					return ( -1);
				}
			   }
			   else
			   {	// Same signs.  Try to normalize them someday.
			   	// But for now, just fall through.
			   }
			}
		}		// End of section for nonzero a.
		cout << "Unnormalized operands in super_test -- fatal error.\n";
		exit (1);
	}			// End of section for unnormalized a and/or b.

	// Append an unnecessary return to quiet stupid compilers:
	return 0;

}

int32 super_precision :: super_ztoi				// PROTECTED FN
				( const super_precision &a) const
{							// convert a to i
// This function converts the super_precision object a to an integer.
	int32 i, j, ia;

	a.assert_invalid_quit ( "super_ztoi");

	// Check for zero case:
	if ( a.sign_bias_exp == 0)
	{
		return ( 0);		// return a zero.
	}

	*zt = ( bias * 2 + 1);		// use static to hold maxvalue
	*zv = a;			// copy argument to new object
	zv->sign_bias_exp = abs ( zv->sign_bias_exp);
	if ( *zv > *zt)
	{
		// Overflow would occur.
		cout << "Fatal error -- overflow in super_ztoi.\n";
		exit (1);
	}

	i = zv->sign_bias_exp - bias;
	if ( i <= 0)
	{			// Value truncated to zero.
		return ( 0);
	}

	j = ( i + nd - 1) / nd;
	zv->sign_bias_exp = a.sign_bias_exp;
	super_shift ( *zv, ( ( i - 1) % nd) - nd + 1);
	ia = 0;
	for ( i = 0; i < j; i++)
	{
		ia = zv->mantissa [i] + ia * ind;
	}
	if ( a.sign_bias_exp < 0) ia = -ia;

	return ( ia);

}

double super_precision :: super_ztod_approx			// PROTECTED FN
				( const super_precision &b) const
{							// rough double = super
// This function obtains a double approximating the super_precision object b.
// The algorithm is not careful to avoid overflow or underflow, so behavior
// near the machine limits may be undesirable.
	int32 i, j;			// temporaries
	double a, rd;			// temporaries

	b.assert_invalid_quit ( "super_ztod_approx");

	// Check for zero case:
	if ( b.sign_bias_exp == 0)
	{
		return ( 0.0);
	}

	rd = 1 / (double) ind;
	// Convert mantissa.
	a = 0;
	for ( i = 0, j = 0; i < 40; i += nd)
	{
		a += b.mantissa [j++] * rd;
		rd /= (double) ind;
	}

	// Transfer sign.
	if ( b.sign_bias_exp < 0) a = -a;

	// Ok, now scale according to exponent.
	j = abs ( b.sign_bias_exp) - bias;
	if ( j == 0)
	{			// exponent is zero, no scaling needed.
		return ( a);
	}
	if ( j > 0)
	{			// exponent > zero, scale up.
		rd = def_radix;
	}
	else
	{			// exponent < zero, scale down.
		rd = 1 / (double) def_radix;
		j = -j;
	}
	for ( i = 0; i < j; i++)
	{
		a *= rd;
	}

	return ( a);
}			// End of function super_ztod_approx

super_precision super_precision :: super_dtoz_approx		// PROTECTED FN
				( const double b) const
{							// rough super = double
// Use static super_precision register pointer zt to create returnable object.
	int32 i, j;				// temporaries
	double d, x, y, z, zd;			// temporaries

	// Check for zero case.
	if ( b == 0)
	{
		super_zero ( *zt);
		return ( *zt);
	}

	d = fabs ( b);
	z = log ( d) / log ( (double) def_radix);
	j = (int32) ( 100000.0 + z) - 99999;
	x = pow ( (double) def_radix, (double) j);
	zd = (double) def_radix;
	y = d / x;
	if ( y >= 1)
	{
		j++;
		y /= zd;
	}
	zd = j;
	i = (int32) ( ( z >= 0) ? zd : -zd);
	zt->sign_bias_exp = ( j >= 0) ? ( bias + i) : - ( bias + i);
	y *= ind;
	for ( i = 0; i < nf; i++)
	{
		zt->mantissa [i] = (int32) y;
		y = ind * ( y - (double) zt->mantissa [i]);
	}

	if ( klip) super_klipr ( *zt);
	return ( *zt);
}


super_precision sqrt						// overload sqrt
				( const super_precision &b)
{	// FRIEND function					// a = sqrt (b)

	if ( b.assert_invalid ( "sqrt"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "sqrt",
					b.validity, b.sign_bias_exp));
	}

	int32 i, j;				// temporaries
	double c;				// temporary

	super_precision_pool_ptr t_;		// temporary, returnable
	super_precision & t = *t_;		// reference variable
	super_precision_pool_ptr r_, w_, s_;	// temporaries
	super_precision & r = *r_;		// reference variable
	super_precision & w = *w_;		// reference variable
	super_precision & s = *s_;		// reference variable

	// Test for zero case.
	if ( b.sign_bias_exp == 0)
	{	// Return a zero super_precision object:
		t.super_zero ( t);
		return ( t);
	}

	// Test for negative argument error.
	if ( b.sign_bias_exp < 0)
	{
		if ( super_precision :: permissive_error)
		{		// Set to Not-a-Number (NaN).
			t.super_zero ( t);
			t.validity = 4;
			return ( t);
		}
		else
		{
			cout << "Fatal error -- super_precision arg < 0 in";
			cout << " sqrt.\n";
			exit (1);
		}
	}

	int32 save_nf = super_precision :: nf;
	int32 save_def_digits = super_precision :: def_digits;
	r = b;					// initialize an object
	j = ( r.sign_bias_exp - super_precision :: bias) / 2;
	r.sign_bias_exp -= 2 * j;
	c = r.super_ztod_approx ( r);
	r.sign_bias_exp = b.sign_bias_exp;
#ifdef SQRT_ALGORITHM_DIVIDE
	c = sqrt ( c);
#else
	c = 1 / sqrt ( c);
#endif
	w = w.super_dtoz_approx ( c);
	w.sign_bias_exp += j;

		// Shorten up the working mantissa length and increase it in
		// iteration loop.
	t.super_zero ( t);
	s.super_zero ( s);
	int32 working_nf = super_precision :: double_fraction_digit_count * 2;
	working_nf = ( working_nf + super_precision :: nd - 1)
						/ super_precision :: nd;
		// But can't exceed register length.
	working_nf = ( working_nf > super_precision::orig_nf +4)
				? super_precision :: orig_nf +4
				: working_nf;
	int32 target_digits = super_precision :: def_digits;
	super_precision :: change_mantissa_length ( working_nf);

	// Iteration loop doubles accuracy each pass.
	j = FALSE;
#ifdef SQRT_ALGORITHM_DIVIDE
	// Algorithm using division.
	for ( i = 0; i < 100; i++)
	{
		t = w;
		w += w;
		w = r / w;
		w += t / 2;
		if ( j)
		{
			w.digits = super_precision :: def_digits;
			return ( w);
		}

		if ( 2 * working_nf >= save_nf)
		{
				// Test convergence.
			s = t - w;
			if ( ( target_digits + 1) / 2 <
					( abs ( t.sign_bias_exp)
					- abs ( s.sign_bias_exp)))  j = TRUE;
				// Restore initial full precision.
			working_nf = save_nf;
			super_precision :: change_mantissa_length ( save_nf,
							save_def_digits);
		}
		else
		{		// Double working precision.
			working_nf += working_nf;
				// But can't exceed register length.
			working_nf = ( working_nf > super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
			super_precision :: change_mantissa_length ( working_nf);
		}
	}
#else
	// Algorithm using no division.  Slower than algorithm using division
	// for short to moderate precisions.  Same speed as algorithm using
	// division for very long precisions (roughly 1000 digits and up).
	for ( i = 0; i < 100; i++)
	{
		t = w;
		w *= w;
		w *= r;
		w -= 3;
		w /= -2;
		w *= t;
		if ( j)
		{
			w *= r;
			w.digits = super_precision :: def_digits;
			return ( w);
		}

		if ( 2 * working_nf >= save_nf)
		{
				// Test convergence.
			s = t - w;
			if ( ( target_digits + 1) / 2 <
					( abs ( t.sign_bias_exp)
					- abs ( s.sign_bias_exp)))  j = TRUE;
				// Restore initial full precision.
			working_nf = save_nf;
			super_precision :: change_mantissa_length ( save_nf,
							save_def_digits);
		}
		else
		{		// Double working precision.
			working_nf += working_nf;
				// But can't exceed register length.
			working_nf = ( working_nf > super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
			super_precision :: change_mantissa_length ( working_nf);
		}
	}
#endif

	cout << "Fatal error -- nonconvergence in super_precision sqrt.\n";
	exit (1);

	// Append an unnecessary return to quiet stupid compilers:
	return b;
}			// End of function sqrt.

super_precision fabs						// OVERLOAD fabs
				( const super_precision &b)
{	// FRIEND function					// fabs (a)
// Use static super_precision register pointer zt to create returnable object.

	if ( b.assert_invalid ( "fabs"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "fabs",
					b.validity, b.sign_bias_exp));
	}

	super_precision :: zt->super_move ( *super_precision :: zt, b);
	super_precision :: zt->sign_bias_exp = abs ( b.sign_bias_exp);
	return ( *super_precision :: zt);
}

super_precision trunc						//OVERLOAD trunc
				( const super_precision &a)
{	// FRIEND function					// trunc (a)

	if ( a.assert_invalid ( "trunc"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "trunc",
					a.validity, a.sign_bias_exp));
	}

	int32 i;				// temporary
        // Get pool pointer and define reference variable.
        super_precision_pool_ptr b_;
        super_precision & b = *b_;

	// Test for zero case, first:
	if ( a.sign_bias_exp == 0)
	{		// Argument is zero, return a zero.
		b.super_zero ( b);
	}

	else
	{
		i = abs ( a.sign_bias_exp) - super_precision :: bias;
		if ( i <= 0)
		{	// Argument magnitude is less than unity, return zero.
			b.super_zero ( b);
		}
		else
		{
			b.super_move ( b, a);
			if ( super_precision :: klip > 0)
			{
				i -= super_precision :: klip; 
			}
			else
			{
				i -= super_precision :: def_digits;
			};
			if ( i >= 0)
			{
				// No change needed.
			}
			else
			{
				b.super_shift ( b, i);
				b.super_shift ( b, -i);
			}
		}
	}
	return ( b);
}

super_precision ceil						// OVERLOAD ceil
				( const super_precision &a)
{	// FRIEND function					// ceil (a)
// This function returns the smallest whole number >= a.

	if ( a.assert_invalid ( "ceil"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "ceil", a.validity));
	}

	
	// Get rid of the zero case first.
	if ( a.sign_bias_exp == 0)
	{
		return ( a);
	}
	
	// Does it have a fractional part?  If not, return the argument value.
	int32 icnt;				// shift count
        // Get pool pointer and define reference variable.
        super_precision_pool_ptr b_;
        super_precision & b = *b_;

	b = a;
	icnt = abs ( a.sign_bias_exp) - super_precision :: bias;
	icnt -= super_precision :: def_digits;
	if ( icnt >= 0)
	{		// No possible fraction part, return argument
		return ( a);
	}
	else
	{	// Shift out the fraction part, see if value changed.
		if ( icnt + super_precision :: def_digits == 0)
		{		// If shift out all digits, b becomes zero.
			b = 0;
		}
		else
		{
			// First, save the exponent word to avoid overflow.
			int32 save_sign_word = b.sign_bias_exp;
			b.sign_bias_exp = super_precision :: bias;
			b.super_shift ( b, icnt);
			b.super_shift ( b, -icnt);
			b.sign_bias_exp = save_sign_word;	// restore exp
		}
		// b is now a whole number.
		if ( b == a)
		{		// No change, so a is whole number.
			return ( a);
			}
			else
			{		// Different, so a has fraction.
			if ( a.sign_bias_exp > 0)
			{	// Positive a, so return b + 1
				return ( b + 1);
			}
			else
			{	// Negative a, so return b
				return ( b);
			}
		}
	}
}

super_precision floor						//OVERLOAD floor
				( const super_precision &a)
{	// FRIEND function					// floor (a)
// This function returns the smallest whole number <= a.

	if ( a.assert_invalid ( "floor"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "floor", a.validity));
	}

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr b_;
        super_precision & b = *b_;

	b = a;

	return ( -ceil ( -b));
}

super_precision fmod						//OVERLOAD fmod
			( const super_precision &a, const super_precision &b)
{	// FRIEND function					// fmod (a)
// This function returns a % b.

	if ( a.assert_invalid ( b, "fmod"))
	{
		return super_NaN ( super_precision ::
			super_irreg_fsa ( "fmod", a.validity, a.sign_bias_exp,
					b.validity, b.sign_bias_exp));
	}

	return ( a % b);
}

super_precision rint						//OVERLOAD rint
			( const super_precision &a)
{	// FRIEND function					// rint (a)
// This function returns a copy of the argument "a" rounded to the nearest
// integer, except in the case of a tie, for which the nearest even integer is
// returned.

	if ( a.assert_invalid ( "rint"))
	{
		return super_NaN ( a.validity);
	}

	// Get rid of the zero case first.
	if ( a.sign_bias_exp == 0)
	{
		return ( a);
	}
	
	// Does it have a fractional part?  If not, return the argument value.
	int32 icnt;				// shift count
        // Get pool pointer and define reference variable.
        super_precision_pool_ptr b_;
        super_precision & b = *b_;

	b = a;
	icnt = abs ( a.sign_bias_exp) - super_precision :: bias;
	icnt -= super_precision :: def_digits;
	if ( icnt >= 0)
	{		// No possible fraction part, return argument
		return ( a);
	}
	else
	{	// Shift out the fraction part, see if value changed.
		if ( icnt + super_precision :: def_digits <= 0)
		{		// No possible integer part.  b becomes zero.
			b = 0;
		}
		else
		{
			// First, save the exponent word to avoid overflow.
			int32 save_sign_word = b.sign_bias_exp;
			b.sign_bias_exp = super_precision :: bias;
			b.super_shift ( b, icnt);
			b.super_shift ( b, -icnt);
			b.sign_bias_exp = save_sign_word;	// restore exp
		}
		// b is now a whole number.
		if ( b == a)
		{		// No change, so a is whole number.
			return ( a);
		}
		else
		{		// Different, so a has fraction.
        		// Get pool pointer and define reference variable.
        		super_precision_pool_ptr fraction_;
        		super_precision & fraction = *fraction_;

			fraction = fabs (a - b);
			fraction += fraction;
			if ( fraction > 1)
			{
				// Return next bigger integer.
				if ( b.sign_bias_exp > 0)
				{
					return ( b + 1);
				}
				else
				{
					return ( b - 1);
				}
			}
			if ( fraction < 1)
			{
				// Return next smaller integer.
				return ( b);
			}
			// 2*fraction is 1 exactly.  Return even number.
			if ( b % 2 == 0)
			{	// b is even, return it.
				return ( b);
			}
			else
			{	// b is odd, return next bigger integer.
				if ( b.sign_bias_exp > 0)
				{
					return ( b + 1);
				}
				else
				{
					return ( b - 1);
				}
			}
		}
	}
}

super_precision round_off					// PUBLIC FN
			( const super_precision &a, const int32 n)
{	// FRIEND function					// round off
// This function returns a copy of the argument a rounded off to n significant
// digits.
// The method resets nf, the length of the mantissa, and resets iklip and then
// uses super_klipr to round_off a copy of the argument.  Then nf and iklip
// are restored.
// This function works for a default or non-default radix.

	if ( a.assert_invalid ( "round_off"))
	{
		return super_NaN ( a.validity);
	}

	int32 i;

	// If requested significance is >= a.digits, just return copy of
	// argument.
	if ( n >= a.digits)
	{
		return ( a);
	}

	int32 new_nd, new_ind;
	if ( a.radix == super_precision :: def_radix)
	{			// Argument is in default base.
		new_nd = super_precision :: nd;
		new_ind = super_precision :: ind;
	}
	else
	{			// Argument is in non-default base.
		int32 jk = super_precision :: bias + 1;
		i = 0;
		new_ind = 1;
		while ( jk >= a.radix)
		{		// Calculate number of digits per mantissa word.
			i++;
			jk /= a.radix;
			new_ind *= a.radix;
		}
		new_nd = i;
	}

	int32 save_nf = super_precision :: nf;
	int32 save_iklip = super_precision :: iklip;
	int32 save_klip = super_precision :: klip;
	int32 save_ind = super_precision :: ind;
	int32 save_nd = super_precision :: nd;
	int32 new_nf = ( n + new_nd - 1) / new_nd;
	int32 new_iklip = 1;
	i = new_nf * new_nd - n;
	while ( i--)
	{
		new_iklip *= a.radix;
	}

        // Get pool pointer and define reference variable.
        super_precision_pool_ptr r_;
        super_precision & r = *r_;
		// Use r for result.
	r = a;
	super_precision :: nf = new_nf;
	super_precision :: iklip = new_iklip;
	super_precision :: klip = n;
	super_precision :: ind = new_ind;
	super_precision :: nd = new_nd;
	r.super_klipr ( r);
	super_precision :: nf = save_nf;
	super_precision :: iklip = save_iklip;
	super_precision :: klip = save_klip;
	super_precision :: ind = save_ind;
	super_precision :: nd = save_nd;
	for ( i = new_nf; i < super_precision :: nf; i++)
	{
		r.mantissa [i] = 0;
	}
	return ( r);
}

void sincos							// FRIEND FN
			( const super_precision &c,
			super_precision &xsin,
			super_precision &xcos)
{	// FRIEND FUNCTION					// sin(a),cos(a)
//  This function calculates sin(c) and cos(c) together and returns both by
//  reference as arguments xsin and xcos.

	if ( c.assert_invalid ( "sincos"))
	{
			xsin = super_NaN ();
			xcos = xsin;
			return;
	}


	// Test for zero argument.
	if ( c.sign_bias_exp == 0)
	{		// Return sin = 0 and cos = 1.
		xsin = 0;
		xcos = 1;
		return;
	}
	
	int32 i, j, negcos, negsin;
        // Get register pool pointer and define reference variable.
        super_precision_pool_ptr za_;
        super_precision & za = *za_;

	negsin = ( c.sign_bias_exp > 0) ?  FALSE : TRUE; // Remember sign of c

	za = fabs( c);
		// Boost mantissa length one word to allow for large argument
		// and loss of three digits to roundoff.
	super_precision :: change_mantissa_length ( super_precision :: nf + 1);
	za.mantissa [super_precision :: nf - 1] = 0;
	za.digits = super_precision :: def_digits;
		// Boost mantissa another word if argument is really big.
	if ( za > 1000000)
	{
		super_precision :: change_mantissa_length ( super_precision ::
								nf + 1);
		za.mantissa [super_precision :: nf - 1] = 0;
		za.digits = super_precision :: def_digits;
	}

//??? why not use ::pi() in next line?
	za = za % ( 2 * pi ());		// abs(arg) modulo 2*pi
	if ( za == 0)
	{		// modulus is zero
			// Restore mantissa length.
		super_precision :: change_mantissa_length ( );
		xsin = 0;
		xcos = 1;
		return;
	}

        // Get register pool pointers and define reference variables.
        super_precision_pool_ptr r_, s_, zb_;
        super_precision & r = *r_;
        super_precision & s = *s_;
        super_precision & zb = *zb_;

	r = za;
	s = ::pi () / 2;				// Calculate pi/2
	za -= ::pi ();
	negcos = ( za.sign_bias_exp > 0) ? TRUE : FALSE;
	if ( negcos) negsin = !negsin;
	za.sign_bias_exp = abs ( za.sign_bias_exp);
	if ( za < s)
	{
		r = za;
		if ( !negcos) negcos = !negcos;
	}
	else
	{
		if ( negcos) negcos = !negcos;
	}

       	// Check if Salamin algorithm is preferred.
       	if ( super_precision :: def_digits >=
       			super_precision :: tan_Newton_threshold)
       	{
       		s = tan_Salamin ( r);
       		r = s;
       		r *= r;
       		za = 1;
       		r += za;
       		zb = sqrt ( r);
       		zb = za / zb;		// cos = 1 / sqrt ( 1 + tan^2)
       		za = s * zb;		// sin = tan * cos
       	}

	else

	{		// Use power series with range reduction.
	// The following number i is a speed optimization parameter that works
	// for nf<1300, that is, up to over 11,000 decimal digits.  Below this
	// limit, the running time of the algorithm is roughly proportional to
	// the super_precision multiplication time, multiplied by the square
	// root of the precision.  Above this limit, the running time goes as
	// n^2 log n.
		i = (int32) ( (1.15 * super_precision :: nf) / 50);
		i = ( i > 30) ? 30 : i;
		j = 1;
		while ( i--) j *= 2;
		s = (super_precision) 8 / j;

		i = 0;
		while ( r > s)
		{
			r /= 2;
			i++;
		}
		s = -r * r;
		j = 3;
		za = r;
		zb = r;
		do
		{
			zb *= s;
			zb /= j * ( j - 1);
			za += zb;
			j += 2;
		} while ( ( ( abs ( zb.sign_bias_exp) +
					super_precision :: def_digits + 1)
			> abs ( za.sign_bias_exp))
			&& ( zb.sign_bias_exp != 0));
		zb = sqrt ( 1 - za * za);
		s = 1;
		for ( j = 0; j < i; j++)
		{
			r = za * zb;
			za = r + r;
			zb = sqrt ( s - za * za);
		}
	}

		// Restore mantissa length.
	super_precision :: change_mantissa_length ( );
		// Chop result length.
	za.super_chop_register ( );
	za.digits = super_precision :: def_digits;
	zb.super_chop_register ( );
	zb.digits = super_precision :: def_digits;

	if ( negsin)
	{
		xsin = -za;
	}
	else
	{
		xsin = za;
	}

	if ( negcos)
	{
		xcos = -zb;
	}
	else
	{
		xcos = zb;
	}

	return;
}

super_precision base_conv					// FRIEND FN
			( const super_precision &b, const int32 radix,
			const int32 from_def_radix, const int32 exact = FALSE)
{								// convert base
	return ( b.super_base_conv ( b, radix, from_def_radix, exact));
}

super_precision super_precision :: super_base_conv		// PROTECTED FN
			( const super_precision &b, const int32 altradix,
			const int32 from_def_radix, const int32 exact = FALSE)
			const
{								// convert base
// This function converts the base (radix) representation of the argument b
// and returns the new representation.  If argument "from_def_radix" is TRUE,
// then the base of argument b is def_radix and the base of the result is
// argument "altradix".  If argument "from_def_radix" is FALSE, then the base of
// argument b is argument "altradix" and the base of the result is def_radix.
// The argument "exact", when true, causes possible mantissa extension when
// altradix and def_radix are both powers of two, because the number of bits
// stored per mantissa word may differ for the two bases.
//
// Note:  Before base conversion is done as specified in the arguments passed,
// the member b.radix is checked to verify that it matches def_radix or argument
// "altradix".  If the match does not conform to the value of from_def_radix,
// a fatal error is issued.

	if ( b.assert_invalid ( "super_base_conv"))
				return super_NaN ( b.validity);

	// Check the input radix for validity.
	if ( from_def_radix)
	{			// Should be def_radix.
		if ( b.radix != def_radix)
		{
			cout << "In super_base_conv, source operand does not";
			cout << " match def_radix -- fatal error.\nradix = ";
			cout << b.radix << " def_radix = " << def_radix << endl;
			exit (1);
		}
	}
	else
	{
				// Should be altradix.
		if ( b.radix != altradix)
		{
			cout << "In super_base_conv, source operand does not";
			cout << " match altradix -- fatal error.\nradix = ";
			cout << b.radix << " altradix = " << altradix << endl;
			exit (1);
		}
	}

	// Check for a zero operand.
	if ( b.sign_bias_exp == 0)
	{
		if ( !from_def_radix)
		{		// Return a zero in the default_radix.
			return ( (super_precision) 0);
		}
		else
		{		// Return a zero with the alternate radix.
        		// Get register pool pointer, define reference variable.
        		super_precision_pool_ptr a_;
        		super_precision & a = *a_;
					// Use a for result.
			a = 0;		// Only difference is the radix itself.
			a.radix = altradix;
			return ( a);
		}
	}

	// Check for a zero conversion base.
	if ( altradix == 0)
	{		// Can't have a zero conversion base.
		cout << "In super_base_conv, conversion base equal to zero";
		cout << " -- fatal error.\n";
		exit (1);
	}

	// Check to see if conversion base is the same as def_radix.
	if ( altradix == def_radix)
	{		// No conversion needed.
		return ( b);
	}

        // Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;
					// Use a for result.

	int32 i, j, k, l, n, in, ii, indx, indb, ndx;		// temporaries
	int32 ndigx, ic, ix, knt, jnt, ja, jb, jc, na, mm, nn, iy, ll;
	int32 nf_to, nf_from;

	i = 1;
	j = 0;
	in = 0;
	ii = 0;
	while ( i < 16)
	{
		i += i;
		j++;
		if ( i == def_radix) in = j;	// in becomes log2 of def_radix
		if ( i == altradix) ii = j;	// ii becomes log2 of altradix
	}					// if they are a power of two.
	// See if both def_radix and altradix are powers of two.
	if ( in && ii)
	{		//  Yes.  This conversion can be done much faster.
		k = bias + 1;
		ndx = 0;
		indx = 1;
		while ( k >= altradix)
		{
			ndx++;			// ndx becomes number of digits
			k /= altradix;		//   per mantissa word.
			indx *= altradix;	// indx becomes altradix^ndx.
		}
		ndigx = ndx * nf;	// ndigx becomes no. digits in mantissa
		i = ndig * in;		// i = no. bits in def_radix operand
		j = ndigx * ii;		// j = no. bits in altradix operand
		nf_to = nf;		//no. mantissa words in altradix operand
		nf_from = nf;
		// If exact is TRUE, allow mantissa extension to carry any
		// significant bits that won't fit in the mantissa.
		if ( exact)
		{
			if ( from_def_radix)
			{
				nf_to = 0;
			}
			else
			{
				nf_from = 0;
			}
		}
		if ( from_def_radix)
		{		// Converting def_radix to altradix.
			k = ii;
			ii = in;
			in = k;		// Exchange ii and in.
			indb = indx;
			indx = ind;
		}
		else
		{		// Converting altradix to def_radix.
			indb = ind;
		}

		iy = abs ( b.sign_bias_exp) - bias;
		double dib = ii * iy;
		// dib is no. of bits left of radix point.

		n = indx;
		ic = -1;
		while ( n > abs ( b.mantissa [0]))
		{
			n /= 2;
			ic++;
		}
		// ic is no. of leading zero bits in first digit.

		double dix = dib - ic;
		// if dix > 0, then dix is the no. of non-leading-zero bits
		// left of the radix point.
		// if dix <= 0, then dix is the no. of leading zero bits right
		// of the radix point.

		ja = 0;
		if ( dix > 0) ja = in - 1;
		dix = ( dix + ja) / in;
		if ( dix > bias)
		{
			cout << "Exponent overflow in super_base_conv -- fatal";
			cout << " error." << endl;
			exit (1);
		}
		else
		{
			jb = (int32) dix;
		}
		// jb is the no. of digits (new base) left of the radix point.
		jc = (int32) ( (double) jb * in - dib);
		// jc is no. of fill (jc>0) or drop (jc<0) leading zero bits.

		ll = 1;
		nn = indx;
		if ( jc >= 0)
		{
			// Set ll to leading zero fill.
			while ( jc--) ll *= 2;
		}
		else
		{
			// Set nn to indx less dropped leading zeros.
			while ( jc++) nn /= 2;
		}
		mm = indb / ll;
		t [0] = 0;
		na = b.mantissa [0];
		l = 0;
		n = 0;
		while ( TRUE)
		{
			ix = ( nn > mm) ? mm : nn;
			nn /= ix;
			iy = na / nn;
			na -= iy * nn;
			t [l] = t [l] * ix + iy;
			mm /= ix;
			ll *= ix;
			if ( ll >= indb)
			{
				if ( l == nf_to) break;
				mm = indb;
				ll = 1;
				t [++l] = 0;
			}
			if ( nn <= 1)
			{
				if ( n == nf_from) break;
				nn = indx;
				na = b.mantissa [++n];
			}
		}
		t [l] *= mm;
		// Number of digits returned = l * ndx.
		for ( n = 0; n < l; n++)
		{
			a.mantissa [n] = t [n];
		}
		a.sign_bias_exp = ( b.sign_bias_exp > 0) ? ( jb + bias)
							 : -( jb + bias);
		if ( from_def_radix) a.radix = altradix;

		return ( a);

	}		// End of conversion between powers of two.

	// One or both of the conversion bases is not a power of two.
	int32 klip_save, iklip_save;			// temporaries
	k = bias + 1;
	ndx = 0;
	indx = 1;
	while ( k >= altradix)
	{
		ndx++;			// ndx becomes number of digits
		k /= altradix;		//   per mantissa word.
		indx *= altradix;	// indx becomes altradix^ndx.
	}
	klip_save = klip;
	iklip_save = iklip;
	if ( !from_def_radix)
	{	// Convert from altradix to def_radix.
		// Increase mantissa length by one.
		change_mantissa_length ( 1 + orig_nf);
		// Get register pool pointers and define reference variables.
	        super_precision_pool_ptr r_, s_, w_;
	        super_precision & r = *r_;
	        super_precision & s = *s_;
	        super_precision & w = *w_;

		r = indx;
		s = b.mantissa [0];
		for ( i = 1; i < nf; i++)
		{
			s *= r;
			s += b.mantissa [i];
		}
		knt = abs ( b.sign_bias_exp) - bias - nf * ndx;
		jnt = abs ( knt);
		if ( jnt != 0)
		{
			r = altradix;
			w = 1;
			while ( TRUE)
			{
				i = jnt / 2;
				l = jnt - i - i;
				jnt = i;
				if ( l) w *= r;
				if ( !jnt) break;
				r *= r;
			}
			if ( knt < 0)
			{
				s /= w;
			}
			else
			{
				s *= w;
			}
		}
		change_mantissa_length ();	// Restore original values.
		s.super_chop_register ();
		s.sign_bias_exp = ( b.sign_bias_exp > 0) ? abs (s.sign_bias_exp)
						     : -abs ( s.sign_bias_exp);
		a = s;
		klip = klip_save;
		iklip = iklip_save;
		a.digits = super_precision :: def_digits;
		super_klipr ( a);
		return ( a);
	}		// End of conversion from altradix to def_radix.

	else

	{	// Convert from def_radix to altradix.
		change_mantissa_length ( 1 + orig_nf);
		// Get register pool pointer and define reference variable.
	        super_precision_pool_ptr za_;
	        super_precision & za = *za_;

		nf--;
		za = b;
		nf++;
		za.mantissa [nf - 1] = 0;
		za.sign_bias_exp = abs( za.sign_bias_exp);
		if ( za.sign_bias_exp > bias)
		{
			j = 0;
		}
		else
		{
			j = 1;
			za = 1 / za;
		}
		i = 0;
		k = 1;
		// Get register pool pointer and define reference variable.
	        super_precision_pool_ptr r_, s_, w_, zc_;
	        super_precision & r = *r_;
	        super_precision & s = *s_;
	        super_precision & w = *w_;
	        super_precision & zc = *zc_;

		s = altradix;
		r = 1;
		w = s;
		ix = 1;
		while ( s <= za)
		{
			ix = k + i;
			i = k;
			k = ix;
			w = r * s;
			r = s;
			s = w;
		}
		if ( i)
		{
			while ( TRUE)
			{
				ii = i;
				i = k - i;
				k = ii;
				if ( !i) break;
				zc = r;
				r = s / r;
				s = zc;
				zc = w / r;
				if ( zc > za)
				{
					w = zc;
					ix -= i;
				}
			}
		}
		if ( !j)
		{
			r = za / w;
		}
		else
		{
			w /= altradix;
			ix = 1 - ix;
			r = w / za;
			if ( r.sign_bias_exp > bias)
			{
				ix++;
				r /= altradix;
			}
		}
		zc = indx;
		za = 0;
		for ( i = 0; i < nf - 1; i++)
		{
			r *= zc;
			w = trunc ( r);
			za.mantissa [i] = super_ztoi ( w);
			r -= w;
			if ( r.sign_bias_exp == 0) break;
		}
		za.sign_bias_exp = bias + ix;
		change_mantissa_length ();	// Restore original values.
		if ( za.mantissa [nf + 1] >= ( indx + 1) / 2)
		{
			super_zqncc ( za.mantissa, nf, indx, 1);
			if ( za.mantissa [0] >= indx)
			{
				super_zqdnc ( za.mantissa, altradix, indx,
					nf, i);
				za.sign_bias_exp ++;
			}
		}
		za.sign_bias_exp = ( b.sign_bias_exp > 0) ? za.sign_bias_exp
							  : -za.sign_bias_exp;
		a = za;
		a.radix = altradix;
		a.digits = ndx * nf;
		return ( a);
	}		// End of conversion from def_radix to altradix.
}

super_precision super_precision :: super_dtoz_exact		// PROTECTED FN
			( const double b) const
{								// convert a=d
// This function converts a double to a super_precision object exactly, to
// within the limits of accuracy of the number of digits carried in the
// super_precision result.
	int32 i, j, k, jj, indx, n;		// temporaries
	double c, fb;				// temporaries
	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	a = 0;

	// Check for a zero argument.
	if ( b == 0) return ( a);

	c = fabs ( b);
	i = 0;
	k = bias + 1;
	j = 0;
	indx = 1;
	while ( k >= hdwr_base)
	{
		j++;
		k /= hdwr_base;
		indx *= hdwr_base;
	}
	fb = (double) indx;
	while ( TRUE)
	{
		while ( c < 1)
		{
			c *= fb;
			i++;
		}
		while ( c >= 1)
		{
			c /= fb;
			i--;
		}
		jj = (int32) fb;
		if ( jj == hdwr_base) break;
		fb = (double) hdwr_base;
		i *= j;
	}
	jj = 0;
	do
	{
		fb = (double) indx;
		c *= fb;
		n = (int32) c;
		fb = (double) n;
		c -= fb;
		a.mantissa [jj] = n;
		jj++;
	} while ( ( jj < nf) && ( c > 0));
	a.radix = hdwr_base;				// Source in hdwr_base.
	a.sign_bias_exp = ( b > 0) ? ( bias - i) : -( bias - i);
	a = super_base_conv ( a, a.radix, FALSE);	// Result in def_radix.
	if ( klip) super_klipr ( a);
	return ( a);
}

double super_precision :: super_ztod_exact			// PROTECTED FN
			( const super_precision &b) const
{								// convert d=a
// This function converts a super_precision object to a double exactly, to
// within the limits of accuracy of the number of digits carried in the
// super_precision result and number of bits in the double.

	b.assert_invalid_quit ( "super_ztod_exact");

	double a, c, d, fb;			// temporaries
	int32 i, j, k, l, indx;

	// Test for zero argument.
	if ( b.sign_bias_exp == 0)
	{
		return ( a = 0);
	}

	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr r_;
        super_precision & r = *r_;

	r = super_base_conv ( b, hdwr_base, TRUE);	// Source in def_radix.
							// Result in hdwr_base.
	k = bias + 1;
	j = 0;
	indx = 1;
	while ( k >= hdwr_base)
	{
		j++;
		k /= hdwr_base;
		indx *= hdwr_base;
	}
	fb = (double) 1;
	d = (double) indx;
	i = abs ( r.sign_bias_exp) - bias;
	for ( l = 0; l < nf; l++)
	{
		c = (double) r.mantissa [l];
		a += c * fb;
		fb /= d;
		if ( fb == 0) break;
	}
	i -= j;
	a = ( b.sign_bias_exp > 0) ? a : -a;
	if ( i == 0)
	{
		return ( a);
	}
	if ( i < 0)
	{
		i = -i;
		k = i / j;
		if ( k)
		{
			for ( l = 1; l <= k; l++)
			{
				a /= d;
			}
			i -= k * j;
		}
		if ( i == 0)
		{
			return ( a);
		}
		c = hdwr_base;
		for ( l = 1; l <= i; l++)
		{
			a /= c;
		}
		return ( a);
	}
	else
	{
		k = i / j;
		if ( k)
		{
			for ( l = 1; l <= k; l++)
			{
				a *= d;
			}
		}
		i -= k * j;
		if ( i == 0)
		{
			return ( a);
		}
		c = hdwr_base;
		for ( l = 1; l <= i; l++)
		{
			a *= c;
		}
		return ( a);
	}

}		// End of function super_ztod_exact.

istream & operator>>						// OVERLOAD >>
			( istream &ist, super_precision &a)
{	// FRIEND FUNCTION					// ist >> a
// This function overloads the extraction operator for istream input.
// The istream returned is shortened by those characters deemed to represent
// one super_precision number, and whitespace may be deleted in certain cases.
// This function works for a default or non-default radix.
//
// Generally, if the public member variable std_ios_manip is TRUE, adhere to
// ANSI ios manipulators such as skipws.  Whitespace includes blanks, tabs, and
// end-of-lines.
//
// If std_ios_manip is TRUE and ios manipulator skipws is TRUE (default), then
// the input string is terminated by any whitespace -- and the string is then
// converted to a super_precision object.  Care must be given that the first
// character found in the input stream is part of the super_precision number
// (not a whitespace character), and that there is no imbedded whitespace in
// the super_precision number representation.
//
// If std_ios_manip is TRUE and ios manipulator skipws is FALSE (from executing
// cin >> resetiosflags (ios::skipws), for example), then whitespace in the
// input stream will block any further scan and a null string will be returned
// for conversion to a super_precision object.  Such whitespace can be bypassed
// by using the "ws" manipulator, as in the statement:
//               cin >> ws >> a >> ws >> b >> ws >> c;
// This will skip any leading whitespace (if any), read "a", skip whitespace,
// read "b", skip whitespace, and read "c".  Accordingly, this function executes
// a statement like "cin >> ws >> string" to skip any whitespace prefixed to the
// super_precision number representation "string", then read "string".  Thus,
// the external behavior of the overloaded >> operator is similar for both
// TRUE and FALSE settings of the ios manipulator skipws.
//
// If std_ios_manip is FALSE and ios manipulator skipws is TRUE, then successive
// strings are extracted from the input stream and concatenated here, until a
// "delimiter" is encountered -- which terminates the concatenation process and
// allows conversion of the string to a super_precision object.  A delimiter is
// any alphabetic character beyond "F" (or "f" in lowercase), or any
// non-alphanumeric character, but not a decimal ".", plus "+", minus "-", or
// angle brackets "<", ">".  This mode of input allows multiple lines, with
// possible interspersed whitespace, to be treated as one super_precision
// object.
//
// If std_ios_manip is FALSE and ios manipulator skipws is FALSE, then a full
// line is read and whitespace removed to produce a single string, which is
// then converted to a super_precision object.  This mode of input allows a
// super_precision string representation to be contain blanks or tabs but still
// be terminated by an end-of-line.
//
// If the input number base, in angle brackets, is not prefixed to the string
// being converted, its number base is assumed to conform to the static variable
// super_precision::input_base, which can be set by the public member function
// super_set_input_base ().
//
// If the public static variable super_precision::conv_input_base is TRUE (the
// default) the base of the input super_precision object is converted to the
// default base def_base.  If FALSE no base conversion is done.
	int32 i;
	char *whole, *fragment;
	char nondelimiters [] = { '0', '1', '2', '3', '4', '5', '6', '7', '8',
				  '9', 'A', 'B', 'C', 'D', 'E', 'F', 'a', 'b',
				  'c', 'd', 'e', 'f', '.', '+', '-', '<', '>',
				  '\0'};

	if ( !super_precision :: input_string)
	{			// Allocate input string if not already.
		int factor = (int) ( super_precision :: baslog / .301 + .999);
		// Allow for worst case -- use factor for binary base.
		super_precision :: max_input_string_size = 1 +
			factor * (super_precision :: max_output_string_size
			+ super_precision :: nblanks);
		super_precision :: input_string =
			new char [super_precision :: max_input_string_size];
	}
	// Make pointer whole an alias for super_precision::input_string.
	whole = super_precision :: input_string;

	int flags;
	flags = ist.flags ();
	int ios_skipws = flags & ios :: skipws;

	// Determine input base.  ANSI does not provide for nondecimal input
	// base for floating-point numbers, but we do -- using the static
	// variable input_base which is set by the function
	// super_set_input_base ().
	int base = super_precision :: input_base;

	if ( super_precision :: std_ios_manip)
	{			// Adhere to ANSI ios manipulators.
				// super_precision :: std_ios_manip = TRUE
		if ( ios_skipws)
		{		// ios_skipws = TRUE
				// Take a string from input stream.
				// The string will be terminated by whitespace!
			ist >> whole;
			a = super_convert_string ( whole, base);
			if ( a.radix != super_precision :: def_radix
				&& super_precision :: conv_input_base)
			{		// Convert to def_radix using a.radix.
				a = a.super_base_conv ( a, a.radix, FALSE);
			}
			return ist;
		}
		else
		{		// ios_skipws = FALSE
			// Skip any whitespace, then take a string from input.
			ist >> ws >> whole;
			a = super_convert_string ( whole, base);
				// Beware being blocked by EOF or suchlike:
			if ( base == 0)
			{
				cout << "In super_precision >> overload, ";
				cout << "null string on input stream," << endl;
				cout<< "possible end-of-file -- fatal error.\n";
				exit (1);
			}
			if ( a.radix != super_precision :: def_radix
				&& super_precision :: conv_input_base)
			{		// Convert to def_radix using a.radix.
				a = a.super_base_conv ( a, a.radix, FALSE);
			}
			return ist;
		}
	}
	else
	{			// Non-std ios manipulators
				// super_precision :: std_ios_manip = FALSE
		if ( ios_skipws)
		{		// ios_skipws = TRUE
				// Build (multiline) input string fragments
				// together until a valid delimiter is found.
			*whole = '\0';		// Start with empty string.
			do
			{
				// Take a string from input stream.
				fragment = whole + strlen ( whole);
				ist >> fragment;
				if ( !*fragment)
				{		// No more strings, maybe EOF.
					i = 0;
					break;
				}
				i = strspn ( fragment, nondelimiters);
			} while ( !( strlen ( fragment) - i));
			*(fragment + i) = '\0';
			a = super_convert_string ( whole, base);
			if ( a.radix != super_precision :: def_radix
				&& super_precision :: conv_input_base)
			{		// Convert to def_radix using a.radix.
				a = a.super_base_conv ( a, a.radix, FALSE);
			}
			return ist;
		}
		else
		{		// ios_skipws = FALSE
				// Take a line from input stream.
			do
			{		// Read a line until it is not empty.
				ist.getline ( whole,
					super_precision :: max_input_string_size);
					// Has possible imbedded whitespace.
					// Ignore imbedded whitespace.
			} while ( !strlen ( whole));
			a = super_convert_string ( whole, base);
			if ( a.radix != super_precision :: def_radix
				&& super_precision :: conv_input_base)
			{		// Convert to def_radix using a.radix.
				a = a.super_base_conv ( a, a.radix, FALSE);
			}
			return ist;
		}
	}
}

super_precision super_convert_string				// FRIEND FN
		( const char *str, int32 &input_base)
{	// FRIEND FUNCTION					//convert string
// This function converts a string pointer argument into a super_precision
// object and returns a copy of the super_precision object.  The string may
// have a leading number base specification, a sign, digits with or without an
// interposed decimal point, and an exponent with a sign.  The digits must be
// in the chosen number base (i.e., one of 0-9,A-F,a-f).  If the number base
// is 15 or 16, any exponent MUST HAVE a sign (+ or -) following the e or E
// exponent prefix.  Whitespace in any location will be ignored, except within
// the base specification itself (i.e., "1 0" is an error for base "10") or
// within an irregular form name (i.e., "NaN" is ok, "N aN" is not).
//
// The string converted is pointed to by argument str.  The number base of the
// conversion is returned as argument input_base.
// This function works for a default or non-default radix.

	int32 i, digit, negative_sign;
	char *temp, *tempexp;				// local pointers
	char whitespaces [] = { ' ', '\t', '\n', '\0'};

        // Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	// Initialize a to zero.
	a.super_zero ( a);
	
	// Work with a local pointer instead of str.
	temp = (char *) str;

	// Scan for non-whitespace character.  strspn returns the number of
	// leading characters matching a character in whitespaces.
	temp += strspn ( temp, whitespaces);
	if ( !*temp)
	{		// *temp is a null character.
		cout << "In super_convert_string, ";
		cout << "empty string for super_precision conversion" << endl;
		cout << " -- fatal error.\n";
		exit (1);
	}

	// Look for an irregular form name.  Acceptable names are "NaN", "+Inf",
	// "-Inf", "+-Inf", "Inf", "UnN".  "PrF" is ok, but returns a properly
	// formed true zero.  Only the leading characters of the string are
	// compared, so trailing garbage is ignored (e.g., "NaNabcdef" is
	// recognized as "NaN").  CASE IS SIGNIFICANT IN THE COMPARISONS.
	for ( i = -1; i <= 5; i++)
	{		// Compare with the names of the irregular forms.
	if ( !strncmp ( temp, super_precision :: validity_type_string ( i),
			strlen ( super_precision :: validity_type_string ( i))))
		{
				// "Inf" is converted to "+Inf".
			if ( i == 5) i = 1;
			a.validity = i;
			return a;
		}
	}

	// Look for a base specification <nn>.
	// Is first non-whitespace character a "<"?
	if ( *temp == '<')
	{			// Yes.  Read base specification.
		temp += strspn ( temp, whitespaces);	// Skip any whitespace.
		input_base = atol ( ++temp);		// Get base (in decimal)
		if ( input_base < 2 || input_base > 16)
		{		// Illegal input_base.
			cout << "In super_convert_string, ";
			cout << "illegal input base specified on input";
			cout << " -- fatal error. " << input_base << endl;
			exit (1);
		}
		temp = strchr ( temp, '>');		// skip to ">"
		if ( !temp)
		{
			cout << "In super_convert_string, ";
			cout << "badly formed base specification," << endl;
			cout << " no closing angle bracket -- fatal error.\n";
			exit (1);
		}
		temp++;					// go on to next
	}
	else
	{
		input_base = super_precision :: input_base;
		// Input base is static variable input_base.
	}

	char max_input_number, max_input_letter;
	max_input_number = ( input_base < 10)   ? input_base - 1
						: 9;
	max_input_letter = ( input_base > 10)   ? input_base - 10
						: 0;

	i = 1 + super_precision :: bias;
	int32 nd = 0;
	int32 ind = 1;
	while ( i >= input_base)
	{
		nd++;			// nd becomes number of digits
		i /= input_base;	//   per mantissa word.
		ind *= input_base;	// ind becomes input_base^ndx.
	}
	int32 ind_over_input_base = ind / input_base;

	// Scan for next non-whitespace character.
	temp += strspn ( temp, whitespaces);
	if ( !*temp)
	{		// *temp is a null character.
		cout << "In super_convert_string, ";
		cout << "empty string after base specification";
		cout << " -- fatal error.\n";
		exit (1);
	}

	// Is it a plus or minus sign?
	if ( *temp == '-') 
	{
		negative_sign = TRUE;
		temp++;
	}
	else
	{
		negative_sign = FALSE;
		if ( *temp == '+')
		{
			temp++;
		}
	}

	int32 decimal_scale = 0;
	int found_decimal = FALSE;
	int found_exponent = FALSE;
	int negative_exponent = FALSE;
	int not_leading = FALSE;
	int32 exponent = 0;				// build exponent here
	int32 lb = ind_over_input_base;
	int32 jw = 0;				// counter for mantissa word

	// Convert digit-by-digit.  Skip whitespace.
	while ( *(temp += strspn ( temp, whitespaces)))
	{		// Not a null end-of-string.

		// Check for exponent.
		if ( !found_exponent && ( *temp == 'e' || *temp == 'E'))
		{
			if ( input_base < 15)
			{	// Not hexadecimal or pentadecimal, e or E is
				// unambiguous prefix to exponent.
				found_exponent = TRUE;
				temp++;
				// Check for a possible exponent sign.
				temp += strspn ( temp, whitespaces);
				if ( *temp == '+')
				{		// Positive exponent.
					temp++;
				}
				else
				{
					if ( *temp == '-')
					{	// Negative exponent.
						negative_exponent = TRUE;
						temp++;
					}
					else
					{
						// No exponent sign.
					}
				}
				continue;		// Resume while loop.
			}
			else
			{	// Hexadecimal or pentadecimal base, e and E
				// are ambiguous -- resolve by REQUIRING A
				// POSITIVE OR NEGATIVE SIGN IN EXPONENT.
				tempexp = 1+ temp + strspn ( temp, whitespaces);
				if ( *tempexp == '+' || *tempexp == '-')
				{	// Found exponent.
					found_exponent = TRUE;
					if ( *tempexp == '-')
						negative_exponent = TRUE;
					temp += 2;
					continue;	// Resume while loop.
				}
			}
		}

		// Check for decimal.
		if ( !found_decimal && !found_exponent && *temp == '.')
		{
			found_decimal = TRUE;
			temp++;
			continue;
		}

		// Check for legal digit.
		if ( ( *temp >= '0' && *temp <= ( '0' + max_input_number)) ||
		     ( *temp >= 'A' && *temp < ( 'A' + max_input_letter)) ||
		     ( *temp >= 'a' && *temp < ( 'a' + max_input_letter)))
		{		// Is legal digit.
			digit = ( *temp < 'A')  ? *temp - '0'
						: *temp - 'A' + 10;
			digit &= ~32;
			temp++;
		}
		else
		{		// Illegal character in string.
			char onechar = *temp;
			cout << "In super_convert_string, ";
			cout << "illegal character in string -- fatal error.\n";
			cout << "Input_base = " << input_base << " character = "
				<< onechar << " or decimal "
				<< (unsigned int) onechar << endl;
			cout << "Valid digits are";
			for ( i = 0; i <= max_input_number ; i++)
			{	cout << " " << (char) ('0' + i);}
			for ( i = 0; i < max_input_letter ; i++)
			{	cout << " " << (char) ('A' + i);}
			for ( i = 0; i < max_input_letter ; i++)
			{	cout << " " << (char) ('a' + i);}
			cout << endl;
			exit (1);
		}

		if ( found_exponent)
		{		// Build exponent.
			exponent = exponent * input_base + digit;
		}
		else
		{		// Build mantissa unless digit is leading zero.
			if ( digit != 0 || not_leading)
			{		// Digit is not a leading zero.
				if ( !found_decimal) decimal_scale++;
				not_leading = TRUE;
				a.mantissa [jw] += digit * lb;
				lb /= input_base;
				if ( !lb)
				{
					if ( ++jw >= super_precision :: nf + 4)
							break;
					lb = ind_over_input_base;
					a.mantissa [jw] = 0;
				}
			}
			else
			{		// Digit is a leading zero.
				if ( found_decimal) decimal_scale--;
			}
		}
	}
	// Reached end of string.  If first word of mantissa is zero, then
	// no valid non-zero digit was ever read.  If so, just return with
	// the zero already stored in "a".
	if ( a.mantissa [0] == 0) return ( a);

	// Apply exponent.
	if ( negative_exponent)
	{
		a.sign_bias_exp = super_precision :: bias - exponent
					+ decimal_scale;
	}
	else
	{
		a.sign_bias_exp = super_precision :: bias + exponent
					+ decimal_scale;
	}

	// Apply sign.
	if ( negative_sign)
	{
		a.sign_bias_exp = - a.sign_bias_exp;
	}

	// Set radix of input super_precision object to input_base.
	a.radix = input_base;

	return ( a);
}

ostream & operator<<						// OVERLOAD <<
			( ostream &ost, const super_precision &a)
{	// FRIEND FUNCTION					// ost << a
// This function overloads the insertion operator for ostream output.
// If the public member variable std_ios_manip is TRUE, adhere to ANSI ios
// manipulators.  Otherwise, use some non-standard manipulators, mainly our
// own specifications for super_precision output only (i.e., base, scientific
// format, number of digits), and optional blocking/grouping of digit strings.

// Note: The static variable ios::showbase does not apply to floating-point
// numbers in ANSI usage, but to fill the need to exhibit the radix of a
// displayed number we set the static variable super_precision::show_output_base
// to TRUE, or to FALSE if the radix is not to be displayed.  The displayed
// radix precedes the sign and is enclosed in left and right angle brackets, as
// <8>-2.17253e23 or <16>a.5f21dE+d for example.  This can be done for output
// regardless of the value of super_precision::std_ios_manip.

// Note: The output number base is not affected by the ios manipulators dec,
// hex, and oct -- in accordance with ANSI usage for float and double types
// -- but can be specified by the function super_set_output_base (), which
// governs both ANSI and non-ANSI cases.

// Exponent prefix e or E can be confused with digits for bases 15 and 16,
// so we require an exponent sign for bases 15 and 16 when an exponent is
// explicit.  This helps distinguish the exponent prefix, but one must watch
// closely to avoid mistaking the exponent prefix for the mantissa end digit.
// To improve readability of the exponent prefix, we use the opposite case for
// this prefix -- "e" for uppercase digits, "E" for lowercase digits -- when
// the number base is 15 or 16 (only).
//
// In the gcc library, the precision limit set by iomanipulator setprecision ()
// is an unsigned two-byte integer, so the precision cannot be > 65535 for ANSI
// adherence (std_ios_manip == TRUE).  The field width is a 4-byte unsigned
// integer, however.

	// Before doing anything else, test for irregular form (NaN, etc.).
	if ( a.validity)
	{		// Yep, it's irregular.
		ost << super_precision :: validity_type_string ( a.validity);
		return ( ost);
	}
	
	int flags;
	flags = ost.flags ();
	char base_string [5];

	// Determine output base.  ANSI does not provide for nondecimal output
	// base for floating-point types float and double (and long double),
	// but we do -- using the static variable output_base which is set by
	// the function super_set_output_base ().
	int base = super_precision :: output_base;

        // Get register pool pointer and define reference variable.
        super_precision_pool_ptr b_;
        super_precision & b = *b_;

        // Convert to output base.
        b = a.super_base_conv ( a, base, TRUE);

	if ( super_precision :: show_output_base)
	{			// Always show base in decimal.
		base_string [0] = '<';
		if ( base > 9)
		{
			base_string [1] = '1';
			base_string [2] = '0' + base - 10;
			base_string [3] = '>';
			base_string [4] = '\0';
		}
		else
		{
			base_string [1] = '0' + base;
			base_string [2] = '>';
			base_string [3] = '\0';
		}
	}
	else
	{			// Don't show base.
		base_string [0] = '\0';
	}

	if ( super_precision :: std_ios_manip)
	{			// Adhere to ANSI ios manipulators.
				// super_precision :: std_ios_manip = TRUE
		int32 i, length, reloc, loc_decimal, save_block;
		char echar;
		char *ptr;		// super_precision converted to string

		// Determine scientific or fixed format, or neither.
		int scientific = 0;		// default
		if ( (ios :: scientific & flags) && !(ios :: fixed & flags))
							scientific = 1;
		int fixed = 0;
		if (ios :: fixed & flags) fixed = 1;
						// fixed overrides scientific


		// Determine number of digits to output.
		int32 dig_count = ost.precision ();	// default precision=6

		// Determine if plus can be prefixed to number.
		int doplus = ios :: showpos & flags;

		// Determine if alphabetics should be uppercase.
		int upper = ios :: uppercase & flags;

		// We really need to round it off to dig_count digits here,
		// because it might round up to the next power of radix, which
		// would affect the calculation of digits_left below.
		b = round_off ( b, dig_count);

		// How many digits to the left of the decimal?
		int32 digits_left = abs (b.sign_bias_exp)
						- super_precision :: bias;
		reloc = digits_left - 1;

		if ( ( digits_left > dig_count) || scientific ||
				( ( reloc < -4) && (b.sign_bias_exp != 0)))
		{		// Use scientific format.
			if ( scientific || fixed)
				dig_count++;	// add digit at left of decimal

				// Save blocking count before zeroing.
			save_block = super_set_output_block ( 0);

				// Build output string.
			ptr = super_output
				( b, dig_count, upper, doplus);

				// Restore blocking count.
			super_set_output_block ( save_block);

			// Maybe strip trailing zeros/decimal.
			if ( !scientific && !fixed && 
						!(ios :: showpoint & flags))
			{
				// Locate exponent.  Its opposite case helps.
				if ( base < 15)
				{
					echar = 'E';
					if ( !upper) echar = 'e';
				}
				else
				{    // Use opposite case for exponent prefix.
					echar = 'e';
					if ( !upper) echar = 'E';
				}
				length = strlen ( ptr);
				while ( *(ptr + --length) != echar)
				{
				     if ( length > 0) continue;
					cout<<"In super_precision << overload,";
					cout << " can't find 'e' in exponent ";
					cout << "-- fatal error.\n";
					exit (1);
				}
				// Found 'e'.
				int32 loc_exp = length;
				// Count trailing zeros/decimal.
				while ( *(ptr + --length) == '0')
				{
					if ( length > 0) continue;
					cout<<"In super_precision << overload,";
					cout << " string is all zeros ";
					cout << "-- fatal error.\n";
					exit (1);
				}
				// Found nonzero.  Drop it if it is a decimal.
				if ( *(ptr + length) == '.') length--;
				while ( *(ptr + loc_exp))
				{
					*(ptr + ++length) = *(ptr + loc_exp++);
				}
				*(ptr + length + 1) = '\0';
			}
		}
		else
		{		// Use fixed-type format.
				// Save blocking count before zeroing.
			save_block = super_set_output_block ( 0);

				// Build output string.
			ptr = super_output
				( b, dig_count, upper, doplus);

				// Restore blocking count.
			super_set_output_block ( save_block);

			// Strip exponent.
			if ( base < 15)
			{
				echar = 'E';
				if ( !upper) echar = 'e';
			}
			else
			{	// Use opposite case for exponent prefix.
				echar = 'e';
				if ( !upper) echar = 'E';
			}
			length = strlen ( ptr);
			while ( *(ptr + --length) != echar)
			{
				if ( length > 0) continue;
					cout<<"In super_precision << overload,";
					cout << " can't find 'e' in exponent ";
					cout << "-- fatal error.\n";
					exit (1);
			}
			// Found 'e'.  Replace it with null.
			*(ptr + length) = '\0';

			// Locate decimal point.
			if ( *(ptr + 1) == '.')
			{
				loc_decimal = 1;	// no leading sign
			}
			else
			{
				if ( *(ptr + 2) == '.')
				{
					loc_decimal = 2;	// leading sign
				}
				else
				{
					cout<<"In super_precision << overload,";
					cout <<" decimal not found";
					cout << " -- fatal error.\n";
					exit (1);
				}
			}

		    // Skip the following shuffle if it is true zero.
		    if ( b.sign_bias_exp != 0)
		    {

			// Now relocate decimal point.
			if ( reloc >= 0)
			{		// move decimal to right
				for ( i = 0; i < reloc; i++)
				{	// move some digits one space left
					*(ptr + loc_decimal + i) = *(ptr +
							loc_decimal + i + 1);
				}
				*(ptr + reloc + loc_decimal) = '.';
			}
			else
			{		// move decimal to left
				// Result must have a zero to the left of point
				// Allow up to three leading zeros after point
				// First swap first digit with decimal.
				*(ptr + loc_decimal) = *(ptr + loc_decimal - 1);
				*(ptr + loc_decimal - 1) = '.';
				// Then insert zeros.
				length = strlen ( ptr);
				for ( i = length; i >= loc_decimal ; i--)
				{
					*(ptr + i - reloc) = *(ptr + i);
				}
				for ( i = loc_decimal; i < loc_decimal - reloc;
							i++)
				{
					*(ptr + i) = '0';
				}
				// Finally swap decimal with first zero.
				*(ptr + loc_decimal - 1) = *(ptr + loc_decimal);
				*(ptr + loc_decimal) = '.';
			}
		    }

			// If showpoint FALSE, strip trailing zeros/decimal.
			if ( !(ios :: showpoint & flags))
			{
				length = strlen ( ptr);
				while ( *(ptr + --length) == '0')
				{
					if ( length > 0) continue;
					cout<<"In super_precision << overload,";
					cout << " string is all zeros ";
					cout << "-- fatal error.\n";
					exit (1);
				}
				// Found nonzero.  Drop it if it is a decimal.
				if ( *(ptr + length) == '.') length--;
				*(ptr + length + 1) = '\0';
			}
		}

		// Fill character is _fill.
		char fill = ost.fill ();

		// Field width is _width if big enough to contain number.
		int32 width = ost.width ();

		length = strlen ( ptr);
		if ( length >= width)
		{			// no room for fill character
			ost << base_string << ptr;
		}
		else
		{
			// First, create filler string.
			char *filler;
			filler = new char(width-length+1);
			for ( i = 0; i < width - length; i++)
			{
				filler [i] = fill;
			}
			filler [width - length] = '\0';
				// internal overrides left and right flags
			if ( ios :: right & flags && !( ios::internal & flags))
			{		// fill on left
				ost << base_string << filler << ptr;
			}
			else
			{
				if (ios::left & flags && !(ios::internal&flags))
				{	// fill on right
					ost << base_string << ptr << filler;
				}
				else
				{	// must be internal, put sign on left
					char sign = *ptr;
					if ( sign == '+' || sign == '-')
					{
						ost << base_string << sign
							<< filler << (ptr + 1);
					}
					else
					{	// no leading sign
						ost << base_string << filler
							<< ptr;
					}
				}
			}
			delete filler;
		}
		return ( ost);
	}
	else
	{			// Non-std ios manipulators
				// super_precision :: std_ios_manip = FALSE
		// ANSI usage:
		// Adhere to uppercase flag.

		// Non-ANSI usage:
		// Use own number base definition for super_precision output
				// with super_set_output_base ( int32).
		// Use scientific format always -- ignore fixed, scientific.
		// Always show trailing zeros -- ignore showpoint.
		// Ignore width and fill.
		// Ignore left, right, and internal.
		// Use own number-of-digits to output -- ignore precision --
				// with super_set_output_count ( int32).
		// Can use digit blocking/grouping with interspersed blanks
				// with super_set_output_block ( int32).

		ost << base_string
			<< super_output ( b, super_precision :: output_count,
					 ios :: uppercase & flags,
					 ios :: showpos & flags);
		return ( ost);
	}
}

char *super_output						// FRIEND FN
			( const super_precision &a,
			const int32 count,
			const int upper, const int doplus)
{	// FRIEND FN						// get string
// This function returns a string representation of the super_precision
// argument a, which may be in a non-default radix.  The number of digits of
// the mantissa to be returned is specified by argument "count".  If count is
// zero, the number of digits returned is def_digits.  If upper is TRUE, use
// uppercase for base 11 through 16; else use lowercase.
//
// Exception!  If the number base is 15 or 16, then for readability the
// exponent prefix is inserted with the opposite case -- "e" for upper==TRUE,
// "E" for upper==FALSE.

	int32 i, j, k, l, nbase, ncount, jk, indx, indexp, ndigx;
	int32 nsigndec, length_exponent, jdigit;
	int32 nblock;
	int displ;
	char ch, echar;

	if ( !super_precision :: output_result)
	{			// Allocate output string if not already.
		super_precision :: output_result =
			new char [super_precision :: max_output_string_size +
			super_precision :: nblanks];
	}

	// Before doing anything else, test for irregular form (NaN, etc.).
	if ( a.validity)
	{		// Yep, it's irregular.
		strcpy ( super_precision :: output_result,
			super_precision :: validity_type_string ( a.validity));
		return ( super_precision :: output_result);
	}

	nbase = a.radix;
	if ( nbase == super_precision :: def_radix)
	{			// Argument is in default base.
		ndigx = super_precision :: def_digits;
		indx = super_precision :: ind;
	}
	else
	{			// Argument is in non-default base.
		jk = super_precision :: bias + 1;
		j = 0;
		indx = 1;
		while ( jk >= nbase)
		{		// Calculate number of digits per mantissa word.
			j++;
			jk /= nbase;
			indx *= nbase;
		}
				// Now get number of digits in mantissa.
		ndigx = j * super_precision :: nf;
	}
				// Allow argument count to override ndigx.
	ncount = ( count <= 0) ? ndigx : count;
				// Except limit ncount to a maximum of ndigx.
	ncount = ( ncount <= ndigx) ? ncount : ndigx;
	// End of preliminaries.


	if ( a == 0)
	{
		// Argument is zero.  Zero is always written 0.0e0 or 0.0E0
		// regardless of the number of output digits specified.
		*super_precision :: output_result = '0';
		*(super_precision :: output_result + 1) = '.';
		*(super_precision :: output_result + 2) = '0';
		if ( nbase < 15)
		{
			echar = 'e';
			if ( upper) echar = 'E';
		}
		else
		{		// Use opposite case for exponent prefix.
			echar = 'E';
			if ( upper) echar = 'e';
		}
		*(super_precision :: output_result + 3) = echar;
		*(super_precision :: output_result + 4) = '0';
		*(super_precision :: output_result + 5) = '\0';
		return ( super_precision :: output_result);
	}

	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr s_;
        super_precision & s = *s_;
			// Use s for rounded copy.
	s = round_off ( a, ncount);

			// Count the "decimal" point and optional sign.
	nsigndec = 1;
	if ( s.sign_bias_exp < 0 || doplus) nsigndec++;

			// Count the number of characters in exponent.
	length_exponent = 1;		// count "e" character
	length_exponent++;		// count an exponent sign
			// Must be at least two digits in exponent.
	length_exponent += 2;
	jk = abs ( s.sign_bias_exp) - super_precision :: bias;
	jk = abs ( jk);
	indexp = nbase * nbase;
	while ( jk >= indexp)
	{
		indexp *= nbase;
		length_exponent++;
	}
	
	j = -1;			// j+1 counts the number of characters in string

			// Do leading sign and "decimal" point.
	if ( s.sign_bias_exp < 0 || doplus)
	{
		j = 0;
		*super_precision :: output_result = (s.sign_bias_exp < 0) ? '-'
									  : '+';
	}
	*(super_precision :: output_result + (++j)) = '.';

			// Do the mantissa digits.
	jdigit = ncount;	// jdigit counts the number of digits left to do
	nblock = 0;		// counts the number of digits in this block
	i = 0;			// i is the index of the mantissa element
	k = s.mantissa [i] % indx;
	l = indx / nbase;
	while ( jdigit--)
	{
		if ( nblock == super_precision :: block)
		{		// Insert a blank for readability.
			j++;
			*(super_precision :: output_result + j) = ' ';
			nblock = 0;		// Start next block.
		}
		nblock++;
		j++;
		jk = k / l;
		displ = jk;
		if ( jk > 9)
		{
			displ += 7;			// capital alphabetic
			if ( !upper) displ += 32;	// lowercase
		}
		*(super_precision :: output_result + j) = '0' + displ;
		k -= l * jk;
		l /= nbase;
		if ( !l)
		{
			i++;
			k = s.mantissa [i] % indx;
			l = indx / nbase;
		}
	}
			// Swap the "decimal point" and the first digit.
	ch = *(super_precision :: output_result + nsigndec - 1);
	*(super_precision :: output_result + nsigndec - 1) =
				*(super_precision :: output_result + nsigndec);
	*(super_precision :: output_result + nsigndec) = ch;

			// Do the exponent, decremented by 1.
	if ( nbase < 15)
	{
		echar = 'e';
		if ( upper) echar = 'E';
	}
	else
	{		// Use opposite case for exponent prefix.
		echar = 'E';
		if ( upper) echar = 'e';
	}
	*(super_precision :: output_result + ( ++j)) = echar;
	jk = abs ( s.sign_bias_exp)
					- super_precision :: bias - 1;
	if ( jk < 0)
	{
		*(super_precision :: output_result + (++j)) = '-';
	}
	else
	{
		*(super_precision :: output_result + (++j)) = '+';
	}
	k = abs ( jk);
	l = indexp / nbase;
	while ( l)
	{
		jk = k / l;
		displ = jk;
		if ( jk > 9)
		{
			displ += 7;			// capital alphabetic
			if ( !upper) displ += 32;	// lowercase
		}
		*(super_precision :: output_result + (++j)) = '0' + displ;
		k -= l * jk;
		l /= nbase;
	}

			// Add the zero byte to terminate string.
	*(super_precision :: output_result + (++j)) = '\0';

	return ( super_precision :: output_result);
}

int32 super_set_output_block					// FRIEND FN
				( const int32 i)
{	// FRIEND FN
// This function resets the number of digits grouped in a block, separated by
// a blank for readability, prepared for output by super_output ().
// The value returned is equal to the PREVIOUS value of block.
	int32 oldblock;
	
	oldblock = super_precision :: block;
	if ( i <= 0)
	{
		// If i is non-positive, cause no blanks to be inserted.
		super_precision :: block =
				super_precision :: max_output_string_size;
	}
	else
	{
		super_precision :: block = i;
	}
		// The following computation of the number of blanks expected
		// is a little too high, but that's ok.
	super_precision :: nblanks =
			( super_precision :: max_output_string_size - 4 - 1)
					/ super_precision :: block;
	return ( oldblock);
}

int32 super_set_output_base					// FRIEND FN
				( const int32 i)
{	// FRIEND FN
// This function resets the radix (base) of an output super_precision object.
// The value returned is equal to the PREVIOUS value of output_base.
	int32 oldbase;
	
	oldbase = super_precision :: output_base;
	if ( i <= 0)
	{
		// If i is non-positive, use the default radix.
		super_precision :: output_base =
				super_precision :: def_radix;
	}
	else
	{
		super_precision :: output_base = i;
	}

	return ( oldbase);
}

int32 super_set_output_count					// FRIEND FN
				( const int32 i)
{	// FRIEND FN
// This function resets the output digit count of an output super_precision
// object.
// The value returned is equal to the PREVIOUS value of output_count.
	int32 oldcount;
	
	oldcount = super_precision :: output_count;
	if ( i <= 0)
	{
		// If i is non-positive, use the default mantissa length.
		super_precision :: output_count =
				super_precision :: def_digits;
	}
	else
	{
		// Note: This will not output digits beyond the last mantissa
		// word, but can output trailing digits (clipped digits) in the
		// last mantissa word.  Use caution when setting i > def_digits.
		super_precision :: output_count =
				(i > super_precision :: ndig)
					? super_precision :: ndig
					: i;
	}

	return ( oldcount);
}

int32 super_set_input_base					// FRIEND FN
				( const int32 i)
{	// FRIEND FN
// This function resets the radix (base) of an input super_precision object.
// The value returned is equal to the PREVIOUS value of input_base.
	int32 oldbase;
	
	oldbase = super_precision :: input_base;
	if ( i <= 0)
	{
		// If i is non-positive, use the default radix.
		super_precision :: input_base =
				super_precision :: def_radix;
	}
	else
	{
		super_precision :: input_base = i;
	}

	return ( oldbase);
}

void super_precision :: super_dump				// PUBLIC FN
				( void) const
{							// (const) dump a
// Note: const keyword so this function can operate on const objects.							// dump a
	int32 i;

	if ( validity) cout << "validity=" << validity << "," << endl;
	cout << (( sign_bias_exp < 0) ? "-" : "+");
	cout << "(" << abs ( sign_bias_exp) - bias << ")";
	for ( i = 0; i < nf; i++) cout << " " << mantissa [i];
	cout << endl;

	return;
}

void dump_tv							// FRIEND FN
				( const int32 tv [])
{							// dump t,v arrays
	int32 i;

	for ( i = 0; i < super_precision :: nfah; i++)
		cout << " " << tv [i];
	cout << endl;
}

void super_precision :: super_fill				// PUBLIC FN
				( const int32 loc, const int32 val)
{								// fill a
	// If loc < 0, val is the sign of the object, as +1 or -1.
	// If loc = 0, val is the exponent in the appropriate base.
	// If loc > 0, val is loaded in loc-1'th element of the mantissa. 
	// The sign call can only be done if the exponent is already defined.

	if ( loc < 0)
	{
		sign_bias_exp = val * abs ( sign_bias_exp);
	}
	else
	{
		if ( loc == 0)
		{
			sign_bias_exp = ( bias + val);
		}
		else
		{
			if ( val >= 0 && val < ind)
			{
				if ( loc <= nf) mantissa [loc - 1] = val;
				else
				{
					cout << "Mantissa index too big in ";
					cout << "super_fill -- fatal error.";
					cout << " loc=" << loc << " nf=";
					cout << nf << endl;
					exit (1);
				}
			}
			else
			{
				cout << "Unscaled mantissa value in super_fill";
				cout << " -- fatal error.  loc=" << loc;
				cout << " val=" << val << endl;
				exit (1);
			}
		}
	}

	return;
}

super_precision & super_precision :: super_shift_public		// PUBLIC FN
						( const int32 i)
{
	super_shift ( *this, i);
	return ( *this);
}

super_precision & super_precision :: super_norm_public		// PUBLIC FN
						( int32 &i)
{
	super_norm ( *this, i);
	return ( *this);
}

super_precision & super_precision :: super_klipr_public		// PUBLIC FN
						( void)
{
	super_klipr ( *this);
	return ( *this);
}

int32 super_to_i 						// FRIEND FN
		( const super_precision &a)
{	// FRIEND FUNCTION (omit scope resolution operator ::)	// i = a

	// Use (but do not alter) static super_precision pointer zt to access
	// member function super_ztoi (which uses *zt as scratch).
	return ( super_precision :: zt->super_ztoi ( a));
}

double super_to_approx_double					// FRIEND FN
				( const super_precision &b)
{	//FRIEND FUNCTION					// rough double = super
// This function derives a double that approximates a super_precision object b.
// The algorithm is not careful to avoid overflow or underflow, so behavior
// near the machine limits may be undesirable.
// This is a friend function so we use the argument object to reference the
// static class variables.

	b.assert_invalid_quit ( "super_to_approx_double");

	int32 i, j;			// temporaries
	double a, rd;			// temporaries

	// Check for zero case:
	if ( b.sign_bias_exp == 0)
	{
		return ( 0.0);
	}

	rd = 1 / (double) b.ind;
	// Convert mantissa.
	a = 0;
	for ( i = 0, j = 0; i < 40; i += b.nd)
	{
		a += b.mantissa [j++] * rd;
		rd /= (double) b.ind;
	}

	// Transfer sign.
	if ( b.sign_bias_exp < 0) a = -a;

	// Ok, now scale according to exponent.
	j = abs ( b.sign_bias_exp) - b.bias;
	if ( j == 0)
	{			// exponent is zero, no scaling needed.
		return ( a);
	}
	if ( j > 0)
	{			// exponent > zero, scale up.
		rd = b.def_radix;
	}
	else
	{			// exponent < zero, scale down.
		rd = 1 / (double) b.def_radix;
		j = -j;
	}
	for ( i = 0; i < j; i++)
	{
		a *= rd;
	}

	return ( a);
}			// End of function super_to_approx_double

double super_to_double 						// FRIEND FN
			( const super_precision &a)
{	// FRIEND FUNCTION					// d = a

	return ( a.super_ztod_exact ( a));
}

super_precision pow						// FRIEND FN
			( const super_precision &b, const int32 ia)
{	// FRIEND FUNCTION					// a = pow(b,ia)
// This function calculates the quantity b raised to the ia power.

	if ( b.assert_invalid ( "pow"))
			return super_NaN ();

	int32 i, k, l;				// temporaries

	if ( b.sign_bias_exp == 0)
	{
		if ( ia > 0)
		{			// Result is zero.
			super_precision :: zt->super_zero
						( *super_precision :: zt);
			return ( *super_precision :: zt);
		}
		else
		{
			if ( super_precision :: permissive_error)
			{
				*super_precision :: zt = 0;
				if ( ia == 0)
				{		// zero to zeroth power = NaN
					super_precision :: zt->validity = 4;
				}
				else
				{		// zero to neg power = +-Inf
					super_precision :: zt->validity = 3;
				}
				return ( *super_precision :: zt);
			}
			else
			{
				cout << "Fatal error -- pow taking nonpositive";
				cout << " power of super_precision zero.";
				exit (1);
			}
		}
	}
	else
	{
		k = abs ( ia);
		if ( k == 0)
		{			// zeroth power yields unity
			*super_precision :: zt = 1;
			return ( *super_precision :: zt);
		}
		else
		{
			int first_store = 1;
			*super_precision :: zv = b;
			while ( k)
			{
				i = k / 2;
				l = k - i - i;
				if ( l)
				{
					if ( first_store)
					{
						first_store = 0;
						*super_precision :: zt =
							*super_precision :: zv;
					}
					else
					{
						*super_precision :: zt *=
							*super_precision :: zv;
					}
				}
				if ( i) *super_precision :: zv *=
						*super_precision :: zv;
				k = i;
			}
						// positive exponent
			if ( ia > 0) return ( *super_precision :: zt);
			else
						// negative exponent
			{
				*super_precision :: zv = 1;
				return ( *super_precision :: zv /
						*super_precision :: zt);
			}
		}
	}

	// Append an unnecessary return to quiet stupid compilers:
	return b;
}			// End of function pow (b,ia).

super_precision pow						// FRIEND FN
			( const super_precision &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// a = pow(b,c)
// This function calculates the quantity b raised to the c power.

	if ( b.assert_invalid ( c, "pow"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	a = log ( b);
	return ( exp ( a * c));
}

super_precision pow						// FRIEND FN
			( const int32 i,
			const super_precision &c)
{	// FRIEND FUNCTION					// a = pow(i,c)
// This function calculates the quantity i raised to the c power.

	if ( c.assert_invalid ( "pow"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_;
        super_precision & a = *a_;

	a = log ( (super_precision) i);
	return ( exp ( a * c));
}


super_precision exp						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = exp (b)
// This function calculates the quantity exp (b).

	if ( b.assert_invalid ( "exp"))
			return super_NaN ();

	int32 i, ii, ij, j, k, m, mm, neg_sign;		// temporaries
	double c, x;					// temporary

	// Test for zero argument.
	if ( b.sign_bias_exp == 0)
	{		// Return unity.
		return ( (super_precision) ( 1));
	}

	// If sign of argument is negative, make a note.
	neg_sign = ( b.sign_bias_exp < 0) ? TRUE : FALSE;

	// Get register pool pointers and define reference variables.
        super_precision_pool_ptr r_, s_, za_;
        super_precision & r = *r_;
        super_precision & s = *s_;
        super_precision & za = *za_;

		// Copy argument to working variable r.
	r = b;
		// Boost mantissa length one word to allow for large argument.
	int32 save_nf, save_def_digits;
	save_nf = super_precision :: nf;
	save_def_digits = super_precision :: def_digits;
	super_precision :: change_mantissa_length ( super_precision :: nf + 1);
	r.mantissa [save_nf] = 0;
	r.digits = super_precision :: def_digits;

		// Test for overflow or underflow.
	r = fabs ( r) * log_radix ();
	if ( r > ( 2 * super_precision :: bias + 1))
	{		// Out of range.  Is it over or under flow?
		if ( b.sign_bias_exp > 0)
		{	// Overflow.
			if ( super_precision :: permissive_error)
			{
				r.super_zero ( r);
				r.validity = 1;
				return ( r);
			}
			else
			{
				cout << "Argument out of range in";
				cout <<" super_precision exp -- fatal error.\n";
				exit (1);
			}
		}
		else
		{	// Underflow -- return a zero.
				// Restore mantissa length.
			super_precision :: change_mantissa_length ( );
			return ( (super_precision) ( 0));
		}
	}

	// Argument must be valid to get to here.

	if ( super_precision :: def_digits >
				super_precision :: exp_Newton_threshold)

	{			// Do Newton-Raphson calculation of exp.

		// Get register pool pointers and define reference variables.
        	super_precision_pool_ptr t_, u_, w_;
        	super_precision & t = *t_;
        	super_precision & u = *u_;
        	super_precision & w = *w_;

			// Get the arg's mantissa into r, arg's integer into ii.
		ii = super_to_i ( r);
		r -= ii;
		r /= log_radix ();

			// Get the best approximation we can.
		c = b.super_ztod_approx ( r);
		w = w.super_dtoz_approx ( exp (c));

		// Define the following at full precision.
		s.super_zero ( s);
		t.super_zero ( t);
		u.super_zero ( s);
		za = 1;
		w.super_zero ( s);

		// Shorten up the working mantissa length and increase it in
		// iteration loop.
		int32 working_nf =
			super_precision :: double_fraction_digit_count * 4;

		working_nf = ( working_nf + super_precision :: nd - 1)
						/ super_precision :: nd;
			// But can't exceed register length.
		working_nf = ( working_nf > super_precision::orig_nf + 4)
					? super_precision :: orig_nf + 4
					: working_nf;
		int32 target_digits = super_precision :: def_digits;
		super_precision :: change_mantissa_length ( working_nf);

		// We use a "quartically-convergent" Newton-Raphson algorithm --
		// the number of accurate digits QUADRUPLES with each iteration.
		j = FALSE;
		for ( i = 0; i < 100; i++)
		{
			t = w;
			s = log_Salamin ( w);
			s = r - s;
			u = za + s;		// za = 1
			w *= u;
			u = s * s;
			s *= u;
			u /= 2;
			s /= 3;
			u -= s;
			u += za;
			w *= u;
			if ( j) break;  

			if ( 4 * working_nf >= save_nf + 1)
			{
					// Test convergence.
				s = t - w;
				if ( ( target_digits + 3) / 4 <
					4 * ( abs ( t.sign_bias_exp)
					- abs ( s.sign_bias_exp) - 1)) j = TRUE;
					// Restore initial full precision.
				working_nf = save_nf + 1;
					// But can't exceed register length.
				working_nf = ( working_nf >
						super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
				super_precision :: change_mantissa_length (
						working_nf);
			}
			else
			{		// Quadruple working precision.
				working_nf *= 4;
					// But can't exceed register length.
				working_nf = ( working_nf >
						super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
				super_precision :: change_mantissa_length (
								working_nf);
			}
		}

		if ( !j)
		{
			cout << "Fatal error -- nonconvergence in ";
			cout << "super_precision exp.\n";
			exit (1);
		}
		else
		{
				// Restore mantissa length.
			super_precision :: change_mantissa_length ( );
				// Chop result length.
			w.super_chop_register ( );
			w.digits = super_precision :: def_digits;

			i = w.sign_bias_exp;
			w.sign_bias_exp += ii;
			// Doublecheck for overflow.
			if ( ( i != w.sign_bias_exp - ii) ||
				( w.sign_bias_exp < 0) ||
				( w.sign_bias_exp >
					2 * super_precision :: bias + 1))
			{		// Overflow.
				if ( super_precision :: permissive_error)
				{
					w.super_zero ( r);
					w.validity = 1;
					return ( w);
				}
				else
				{
					cout << "Argument out of range in ";
					cout << "super_precision exp -- fatal";
					cout << " error.\n";
					exit (1);
				}
			}

			// Take reciprocal if argument was negative:
			if ( neg_sign) w = za / w;
		}
		return ( w);
	}

	else

	{			// Do power series calculation of exp.

		// Now figure a speed/accuracy tradeoff parameter "i" used in
		// the argument reduction:
		i = 2 + (int32) sqrt ( 5 * (double) super_precision :: nf);
		i = ( i < 8) ? 8 : i;
		i = ( i > 76) ? 76 : i;
		// With this choice of i, the running time of exp is about the
		// same as n multiplications, where n = 46 + 3 (nf)^(2/3),
		// up to about nf=1130.

		ii = super_to_i ( r);
		r -= ii;
		r /= log_radix ();
		// Boost mantissa length again to allow for argument reduction.
		super_precision :: change_mantissa_length (
						super_precision :: nf + 2);
		r.mantissa [super_precision :: nf - 2] = 0;
		r.mantissa [super_precision :: nf - 1] = 0;
		r.digits = super_precision :: def_digits;

		x = super_precision :: baslog / 0.30103;
		m = (int32) ( x * super_precision :: nd);
		mm = i;
		if ( i >= m)
		{		// Following is to prevent 2^i from overflowing.
			j = 1; ij = m; while ( ij--) j *= 2;	// j = pow (2,m)
			while ( mm > m)
			{		// Reduce mm to something within range.
				r /= j;
				mm -= m;
			}
		}
		j = 1; ij = mm; while ( ij--) j *= 2;	// j = pow ( 2, mm)
		if ( j > 1) r /= j;

		s = r * r;
		za = r;
		k = 2;
		do
		{
			r *= s;
			r /= k * ( k + 1);
			za += r;
			k += 2;
		} while ( ( za.sign_bias_exp < ( r.sign_bias_exp + 1 +
						super_precision :: def_digits))
				&& ( r.sign_bias_exp != 0));
		r = za * za + 1;
		r = za + sqrt ( r);
//		r = za + sqrt ( za * za + 1);
		while ( i--) r *= r;	// Expand argument to original.
			// Restore mantissa length.
		super_precision :: change_mantissa_length ( );
			// Chop result length.
		r.super_chop_register ( );
		r.digits = super_precision :: def_digits;

		i = r.sign_bias_exp;
		r.sign_bias_exp += ii;
		// Doublecheck for overflow.
		if ( ( i != r.sign_bias_exp - ii) || ( r.sign_bias_exp < 0) ||
			( r.sign_bias_exp > 2 * super_precision :: bias + 1))
		{		// Overflow.
			if ( super_precision :: permissive_error)
			{
				r.super_zero ( r);
				r.validity = 1;
				return ( r);
			}
			else
			{
				cout << "Argument out of range in super_precis";
				cout << "ion exp -- fatal error.\n";
				exit (1);
			}
		}

		// Take reciprocal if argument was negative:
		if ( neg_sign) r = 1 / r;

		return ( r);
	}
}			// End of function exp (b).

super_precision log						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = log (b)
// This function calculates the quantity log (b) to the base e (natural log).

	if ( b.assert_invalid ( "log"))
			return super_NaN ();

	int32 i, j, jj;				// temporaries

	// First, check for negative or zero argument.
	if ( b.sign_bias_exp <= 0)
	{	// Yes, is error.
		if ( b.sign_bias_exp == 0 && super_precision::permissive_error)
		{		// log of 0 is -infinity
			*super_precision :: zt = 0;
			super_precision :: zt->validity = 2;
			return ( *super_precision :: zt);
		}
		else
		{
			if ( b.sign_bias_exp == 0)
			{
				cout << "Zero super_precision arg to log --";
				cout << " fatal error.\n";
			}
			else
			{
				cout <<"Negative super_precision arg to log --";
				cout << " fatal error.\n";
			}
			exit (1);
		}
	}

	// See if Salamin algorithm is preferred.
	if ( super_precision :: def_digits >=
				super_precision :: log_Salamin_threshold)
	{
		return log_Salamin ( b);
	}

	// Get register pool pointer and define reference variable.
        super_precision_pool_ptr a_, r_, s_, w_, za_, zb_;
        super_precision & r = *r_;
        super_precision & s = *s_;
        super_precision & w = *w_;
        super_precision & za = *za_;
        super_precision & zb = *zb_;

	zb = b;
	
	// Dig out log of exponent part.
	int32 exp_part = zb.sign_bias_exp - 1 - super_precision :: bias;
	if ( exp_part != -1) zb.sign_bias_exp -= exp_part;

	// Reduce the argument range.
	// The following number j approximately optimizes running speed up to
	// about nf=16,000 decimal digits.  Below this limit, running time is
	// roughly proportional to sqrt(nf) times the super_precision multiply
	// time; past this limit, proportional to nf times the super_precision
	// multiply time.
		j = ( super_precision :: nf + 4) / 18;
		i = 1 + (int32) sqrt ( (double) super_precision :: nf / 2);
		j = ( j < i) ? j : i;
		j = ( j > 30-1) ? 30-1 : j;  // but not more than 30-1
		i = 2;
		while ( j--) i *= 2;
	// Extend the working precision one word, except two words for nf>600.
	int32 nf_save = super_precision :: nf;
	if ( nf_save <= 600)
	{
		super_precision :: change_mantissa_length ( nf_save + 1);
		zb.mantissa [nf_save] = 0;
		zb.digits = super_precision :: def_digits;
	}
	else
	{
		super_precision :: change_mantissa_length (nf_save + 2);
		zb.mantissa [nf_save] = 0;
		zb.mantissa [nf_save + 1] = 0;
		zb.digits = super_precision :: def_digits;
	}
	za = i + 1;
	za /= i;
	j = 0;
	jj = 1;
	while ( ( zb > za) || ( zb < 2 - za))
	{
		j++;
		jj *= 2;
		zb = sqrt ( zb);
	}
	za = zb - 1;
	r = zb + 1;
	za /= r;
	r *= r;
	r.sign_bias_exp = -r.sign_bias_exp;
	r /= zb;
	s = za / (-2);
	zb = za;
	za *= za;
	i = 1;
	do
	{
		zb *= za;
		w = zb / ( i * ( i + 2));
		s += w;
		i += 2;
	} while ( ( w.sign_bias_exp != 0) && ( abs ( w.sign_bias_exp) + 1 +
			super_precision ::def_digits > abs ( s.sign_bias_exp)));
	
	// Expand argument range.
	s *= r;
	if ( j) s *= jj;
	if ( nf_save > 600)
	{
			// Reduce mantissa length back to first plus one word.
		super_precision :: change_mantissa_length ( nf_save + 1);
			// Do a proper roundoff.
		s.super_chop_register ( );
		s.digits = super_precision :: def_digits;
	}
		// Add log of exponent taken out earlier.
	if ( exp_part != -1 && exp_part != 0) s += exp_part / log_radix ();
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( );
		// Chop result length for log of mantissa.
	s.super_chop_register ( );
	s.digits = super_precision :: def_digits;

	return ( s);
	
}

super_precision log10						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = log10 (b)
// This function calculates the quantity log (b) to the base 10, that is, the
// common log of b.

	if ( b.assert_invalid ( "log10"))
			return super_NaN ();

	return ( log ( b) * log_10 ());
}

super_precision log_Salamin					// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = log (b)
// This function calculates the quantity log (b) to the base e (natural log)
// using the quadratically convergent Salamin algorithm.

	int32 new_exponent, old_exponent, save_nf, save_def_digits;

	// Get register pool pointer and define reference variable.
	super_precision_pool_ptr x_;
	super_precision & x = *x_;

	x = b;			// Copy argument to pool variable.

	// Boost mantissa length for new objects.
	save_nf = super_precision :: nf;
	save_def_digits = super_precision :: def_digits;
		// But can't exceed register length.
	int32 working_nf = save_nf + 1;;
	working_nf = ( working_nf > super_precision::orig_nf +4)
				? super_precision :: orig_nf +4
				: working_nf;
	super_precision :: change_mantissa_length ( working_nf);

	// Clear last mantissa word of x.
	x.mantissa [ save_nf] = 0;
	x.digits = super_precision :: def_digits;

	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr y_;
	super_precision & y = *y_;

	// Exponent must be at least the number of digits carried.
	new_exponent = super_precision :: def_digits + 1;
	if ( new_exponent > super_precision :: bias + 1)
	{
		cout << "Exponent too big in log_Salamin -- fatal error.\n";
		cout << " new_exponent = " << new_exponent << " bias = ";
		cout << super_precision :: bias << endl;
		exit (1);
	}
	old_exponent = x.sign_bias_exp - super_precision :: bias;
	x.sign_bias_exp = super_precision :: bias + new_exponent;

	// Apply Salamin algorithm.
	y = 1;
	y /= x;
	y *= 4;
	x = 1;
	x.arithmetic_geometric_mean ( y);
	x += x;
	y = pi ();
	y /= x;
	x = new_exponent - old_exponent;
	x /= log_radix ();
	y -= x;
	
		// Copy to register before restoring original mantissa length.
	super_precision :: zz->super_move ( *super_precision :: zz, y);
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( save_nf, save_def_digits);
		// Do proper roundoff of low-order digits remaining.
	*super_precision :: zz = super_precision :: zz->
				convert_precision ( *super_precision :: zz);

		// Done.
	return ( *super_precision :: zz);
}			// End of function log_Salamin ().

void super_precision :: arithmetic_geometric_mean		// PUBLIC FN
				( const super_precision &b)
{	// PUBLIC FUNCTION				// obtain AGM for this,b
// This function obtains the "arithmetic geometric mean" of the quantities
// *this and b.  The method is, of course, quadratically convergent.

	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr y_, z_;
	super_precision & y = *y_;
	super_precision & z = *z_;

	int j = FALSE;			// Convergence flag.
	y = b;
	while ( TRUE)
	{
		z = *this + y;
		z /= 2;
		*this *= y;
		*this = sqrt ( *this);
		if ( j) break;
		y = z;
		z -= *this;
					// If diff is small, break on next loop.
		if ( ( super_precision :: ndig + 1) / 2 <=
			super_precision :: bias - abs ( z.sign_bias_exp))
			j = TRUE;
	}

	return;
}

void super_precision :: arithmetic_geometric_mean_cplx		// PUBLIC FN
			( super_precision &a, super_precision &b)
{	// PUBLIC FUNCTION				// obtain AGM of this,ib
// This function obtains the "arithmetic geometric mean" of the quantities *this
// and a+i*b where i is the square root of -1.  The method is, of course,
// quadratically convergent.  The real part of the answer is returned in the
// argument a and the imaginary part is returned in the argument b. a and b must
// be separate objects (not the same memory block).

	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr yr_, yi_, zr_, zi_;
	super_precision & yr = *yr_;
	super_precision & yi = *yi_;
	super_precision & zr = *zr_;
	super_precision & zi = *zi_;
	super_precision_pool_ptr r_, s_, za_, zb_;
	super_precision &r = *r_;
	super_precision &s = *s_;
	super_precision &za = *za_;
	super_precision &zb = *zb_;

	int j = FALSE;			// Convergence flag.
	int32 k, kr, ki;
	yr = a;
	yi = b;
	a = *this;	// Hereafter a will be the real part of this.
	b = 0;		// And b will be the imaginary part of this.
	while ( TRUE)
	{		// Each iterate pair starts in (a,b) and (yr,yi).
		zr = a + yr;
		zi = b + yi;
		zr /= 2;
		zi /= 2;
			// Placed arithmetic mean in (zr,zi).

                r = a * yi;		// complex multiply (a,b)*(yr,yi)
                s = b * yr;
                za = r - s;
                r += s;
                s = a + b;
                zb = yr - yi;
                s *= zb;
                a = s + za;
                b = r;
                	// Placed product of iterate pair in (a,b).

		if ( a == 0 && b == 0)	// complex square root (a,b)
		{
			a = b = 0;
		}
		else
		{
			if ( a < 0)
			{
				r = hypot ( a, b);
				r -= a;
				r /= 2;
				r = sqrt ( r);
//				if ( b < 0) r = - r;
				if ( b < 0) r.super_neg ( );
				s = r;
				r += r;
				r = b / r;
			}
			else
			{
				r = hypot ( a, b);
				r += a;
				r /= 2;
				r = sqrt ( r);
				s = r;
				s += r;
				s = b / s;
			}
			a = r;
			b = s;
		}
			// Placed geometric mean in (a,b).

		if ( j) break;
		yr = zr;		// Copy arithmetic mean to (yr,yi).
		yi = zi;

		zr -= a;		// Get diff of arith & geom means.
		zi -= b;
					// If diff is small, break on next loop.
		kr = abs ( zr.sign_bias_exp);
		ki = abs ( zi.sign_bias_exp);
		k = ( kr > ki) ? kr : ki;
		if ( ( super_precision :: ndig + 2) / 2 <=
			super_precision :: bias - k) j = TRUE;
	}

	return;
}

void log_Salamin_complex					// FRIEND FN
		( const super_precision &a, const super_precision &b,
		  super_precision &real_ans, super_precision &imag_ans)
{	// FRIEND FUNCTION					// g = log(a+ib)
// This function calculates the quantity log (a+ib) to the base e (natural log)
// using the quadratically convergent Salamin algorithm, where a is the real
// part and ib is the imaginary part.  The complex result is returned in the
// arguments real_ans and imag_ans.

	int32 new_exponent, old_exponent, save_nf, save_def_digits;

	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr x_, y_;
	super_precision & x = *x_;
	super_precision & y = *y_;

	x = a;			// Copy real part to pool variable.
	y = b;			// Copy imaginary part to pool variable.

	// Boost mantissa length for new objects.  (Can do this for pool ptrs.)
	save_nf = super_precision :: nf;
	save_def_digits = super_precision :: def_digits;
		// But can't exceed register length.
	int32 working_nf = save_nf + 1;;
	working_nf = ( working_nf > super_precision::orig_nf +4)
				? super_precision :: orig_nf +4
				: working_nf;
	super_precision :: change_mantissa_length ( working_nf);

	// Clear last mantissa word of x and y.
	x.digits = super_precision :: def_digits;
	x.mantissa [ working_nf - 1] = 0;
	y.digits = super_precision :: def_digits;
	y.mantissa [ working_nf - 1] = 0;


	int32 diff;
	// Exponent must be at least the number of digits carried.
	new_exponent = super_precision :: def_digits + 1;
	if ( new_exponent > super_precision :: bias + 1)
	{
		cout << "Exponent too big in log_Salamin_complex -- fatal ";
		cout << "error.\nnew_exponent = " << new_exponent << " bias = ";
		cout << super_precision :: bias << endl;
		exit (1);
	}
	if ( abs ( x.sign_bias_exp) > abs ( y.sign_bias_exp))
	{
		old_exponent = abs ( x.sign_bias_exp) - super_precision :: bias;
		if ( x.sign_bias_exp > 0)
		{
			x.sign_bias_exp = super_precision:: bias + new_exponent;
		}
		else
		{
			x.sign_bias_exp = -super_precision:: bias -new_exponent;
		}
		if ( y.sign_bias_exp != 0)
		{
			diff = new_exponent - old_exponent;
			if ( diff >= abs ( y.sign_bias_exp))
			{
				cout << "Underflow in log_Salamin_complex.  ";
				cout << "This is a fatal error." << endl;
				exit (1);
			}
			if ( y.sign_bias_exp > 0)
			{
				y.sign_bias_exp += diff;
			}
			else
			{
				y.sign_bias_exp -= diff;
			}
		}
	}
	else
	{
		old_exponent = abs ( y.sign_bias_exp) - super_precision :: bias;
		if ( y.sign_bias_exp > 0)
		{
			y.sign_bias_exp = super_precision:: bias + new_exponent;
		}
		else
		{
			y.sign_bias_exp = -super_precision:: bias -new_exponent;
		}
		if ( x.sign_bias_exp != 0)
		{
			diff = new_exponent - old_exponent;
			if ( diff >= abs ( x.sign_bias_exp))
			{
				cout << "Underflow in log_Salamin_complex.  ";
				cout << "This is a fatal error." << endl;
				exit (1);
			}
			if ( x.sign_bias_exp > 0)
			{
				x.sign_bias_exp += diff;
			}
			else
			{
				x.sign_bias_exp -= diff;
			}
		}
	}

	// Apply Salamin algorithm.

	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr zr_, zi_;
	super_precision & zr = *zr_;
	super_precision & zi = *zi_;

	zr = x;		// Calculate 4/(x+iy).
	zr *= x;
	zi = y;
	zi *= y;
	zi += zr;
	zr = 1;
	zr /= zi;
	zr *= 4;
	zi = -y;
	y = zi * zr;	// Quotient to x+iy.
	x *= zr;

	zr = 1;
	zr.arithmetic_geometric_mean_cplx ( x, y);// Returns x=real part, y=imag
	x += x;
	y += y;

	zi = y;		// Calculate pi / cmplx(x,y)
	zi *= y;
	zr = x;
	zr *= x;
	zi += zr;
	zr = pi ();
	zr /= zi;
	zi = -y;
	zi *= zr;	// Quotient to zr + i*zi.
	zr *= x;

	x = new_exponent - old_exponent;
	x /= log_radix ();
	zr -= x;
	
		// Copy to register before restoring original mantissa length.
	super_precision :: zz->super_move ( *super_precision :: zz, zr);
	super_precision :: zv->super_move ( *super_precision :: zv, zi);
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( save_nf, save_def_digits);
		// Do proper roundoff of low-order digits remaining.
	*super_precision :: zz = super_precision :: zz->
				convert_precision ( *super_precision :: zz);
	*super_precision :: zv = super_precision :: zv->
				convert_precision ( *super_precision :: zv);

	real_ans = *super_precision :: zz;
	imag_ans = *super_precision :: zv;
		// Done.
	return;
}			// End of function log_Salamin_complex ().

super_precision pi						// FRIEND FN
			( void)
{	// FRIEND FUNCTION					// a = pi
// This function calculates the quantity pi and returns it.
	int32 save_nf, save_def_digits;

	if ( !super_precision :: pi_init)
	{		// Allocate memory block for pi using current defaults.
		save_nf = super_precision :: nf;	// save current nf
		super_precision :: nf += 4;		// boost by 4 words
		super_precision :: pi = new super_precision;	// allocate pi
		super_precision :: nf = save_nf;	// restore nf
		super_precision :: pi_init = 1;		// set initialized TRUE
		// Now fall thru to computational section below.
	}
	else
	{
		// Recalculate pi if def_digits != to static pi precision.
		if ( super_precision :: pi->digits ==
					super_precision :: def_digits)
		{				// same precision
			// Return the value already calculated.
			return ( *super_precision :: pi);
		}
		else
		{
			// Test for request for shorter precision.
			if ( super_precision :: pi->digits >
					super_precision :: def_digits)
			{			// shorter precision requested
				return ( super_precision:: pi->convert_precision
					( *super_precision :: pi));
				// *pi knows how long its own mantissa is, and 
				// convert_precision will return a mantissa with
				// only def_digits, the current working length.
			}
			else
			{			// longer precision requested
				// Delete the old static pi and allocate a new
				// larger pi.
				delete super_precision :: pi;
				super_precision :: pi = new super_precision;
				// Fall thru to computational section below.
			}
		}
	}

	// Getting here means we have to calculate pi or a longer pi.
	// First, boost mantissa length for new objects.
	save_nf = super_precision :: nf;
	save_def_digits = super_precision :: def_digits;
	super_precision :: change_mantissa_length ( 4 +
					super_precision :: orig_nf);
	// Get register pool pointers and define reference variables.
	super_precision_pool_ptr x_, y_, cpi_, num_, denom_, diff_, unity_;
	super_precision & x = *x_;
	super_precision & y = *y_;
	super_precision & cpi = *cpi_;
	super_precision & num = *num_;
	super_precision & denom = *denom_;
	super_precision & diff = *diff_;
	super_precision & unity = *unity_;

#ifndef QUARTIC_PI_ALGORITHM
	// Use quadratically-convergent AGM algorithm due to Borwein & Borwein.
	x = sqrt ( (super_precision) 2);
	cpi = x + 2;
	y = num = sqrt ( x);
	unity = 1;

	int j = FALSE;			// Convergence flag.
	while ( TRUE)
	{
		denom = unity / num;
		x = num + denom;
		x /= 2;
//		x = ( num + unity / num) / 2;
		num = x + unity;
		denom = y + unity;
		num /= denom;
		cpi *= num;
		if ( j) break;
		diff = x - y;
					// If diff is small, break on next loop.
		if ( ( super_precision :: ndig + 1) / 2 <=
			super_precision :: bias - abs ( diff.sign_bias_exp))
			j = TRUE;
		num = sqrt ( x);
		y *= x;
		y += unity;
		denom *= num;
		y /= denom;
//		y = ( y * x + unity) / ( num * denom);
	}
#else
	// Use quartically-convergent algorithm due to Borwein & Borwein.
	// It runs about the same speed as the quadratically-convergent
	// algorithm up to a few hundred (decimal) digits, but takes up to
	// 25 percent less cpu time at very high precisions.
	int32 k, kloop = 0;
	x = sqrt ( (super_precision) 2);
	y = 4 * x;
	cpi = 6;
	cpi -= y;			// cpi is approximation to 1/pi
	y = x;
	unity = 1;
	y -= unity;

	int j = FALSE;			// Convergence flag.
	while ( TRUE)
	{
		x = cpi;		// save last value for cpi
		diff = y;
		diff *= diff;
		diff *= diff;
		diff -= unity;		// get y^4 - 1
		diff.super_neg ( );	// get 1 - y^4
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  - - -
// Choose either a repeated sqrt or an nthrt call to get diff^(1/4):
//		diff = sqrt ( diff);
//		diff = sqrt ( diff);	// get (1-y^4)^(1/4)
		diff = nthrt ( diff, 4); // Sometimes a little faster.
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  - - -
		denom = diff + unity;
		num = unity - diff;
		y = num / denom;
		denom = y + unity;
		num = denom;
		denom *= denom;
		denom *= denom;		// get (1 + y)^4
		cpi *= denom;
		num *= y;
		num += unity;
		num *= y;		// get y * ( 1 + y + y^2)
		num *= 8;
		for ( k = 0; k < kloop; k++)
		{
			num *= 2;
		}
		kloop+=2;
		cpi -= num;
		if ( j) break;
		diff = cpi - x;
					// If diff is small, break on next loop.
		if ( ( super_precision :: ndig + 3) / 4 <=
			super_precision :: bias - abs ( diff.sign_bias_exp))
			j = TRUE;
					// But if diff is zero, break now.
                if ( diff.sign_bias_exp == 0) break;
	}
	cpi = unity / cpi;		// let cpi = pi now
#endif
	// Store pi now.
	*super_precision :: pi = cpi;
			// Specify accuracy of pi:
	super_precision :: pi->digits = super_precision :: def_digits -
				(int32) ( 2.5 / super_precision :: baslog);
		// Copy to register before restoring original mantissa length.
	super_precision :: zz->super_move ( *super_precision :: zz,
						*super_precision :: pi);
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( save_nf, save_def_digits);
		// Do proper roundoff of low-order digits remaining.
	*super_precision :: zz = super_precision :: zz->
				convert_precision ( *super_precision :: zz);

		// Done.
	return ( *super_precision :: zz);
}			// End of function pi ().

super_precision log_10						// FRIEND FN
			( void)
{	// FRIEND FUNCTION					// a = log(10)
// This function calculates the quantity log10(e), the common log of the natural
// log base, and returns it.
// The quantity e is the natural logarithm base 2.71828...

	// If def_radix is 10, we will use/reuse log_radix and not have to do a
	// separate computation of log_10.
	if ( super_precision :: def_radix == 10)  return ( log_radix ());
						// The easy way.


	int32 i, j, ii, save_nf, save_def_digits;	// temporaries

	if ( !super_precision :: log_10_init)
	{	// Allocate memory block for log_10 using register length.
		save_nf = super_precision :: nf;
		super_precision :: nf = super_precision :: orig_nf + 4;
		super_precision :: log_10 = new super_precision;
		super_precision :: nf = save_nf;
		super_precision :: log_10_init = 1;
		// Now fall thru to computational section below.
	}
	else
	{

		// Recalculate log_10 if def_digits != static log_10 precision.
		if ( super_precision :: log_10->digits ==
					super_precision :: def_digits)
		{				// same precision
			// Return the value already calculated.
			return ( *super_precision :: log_10);
		}
		else
		{
			// Test for request for shorter precision.
			if ( super_precision :: log_10->digits >
					super_precision :: def_digits)
			{			// shorter precision requested
				return ( super_precision ::
					log_10->convert_precision
					( *super_precision :: log_10));
				// *log_10 knows how long its mantissa is, and 
				// convert_precision will return a mantissa with
				// only def_digits, the current working length.
			}
			else
			{			// longer precision requested
				// Verify that the desired precision will fit in
				// the allocated mantissa.
				if ( super_precision :: nf >
						super_precision :: orig_nf + 4)
				{
					cout << "Requesting too much precision";
					cout << " for log_10 -- fatal error";
					cout << ".  nf=" <<
						super_precision :: nf
						<< "  orig_nf+4=" <<
						super_precision :: orig_nf+4
						<< endl;
					exit (1);
				}
				// Fall thru to computational section below.
			}
		}
	}

	// Getting here means we have to calculate log_10 or a longer log_10.

	if ( super_precision :: def_digits >=
				super_precision :: log_Salamin_threshold)

	{			// Do AGM calculation of log.
	
		int32 iexpon;
		double dexpon;

		// Get register pool pointers and define reference variables.
		super_precision_pool_ptr x_, y_;
		super_precision & x = *x_;
		super_precision & y = *y_;

		// Boost mantissa length for new objects.
		save_nf = super_precision :: nf;
		save_def_digits = super_precision :: def_digits;
		super_precision :: change_mantissa_length ( 3 +
						super_precision :: orig_nf);

		x = 10;			// Put 10 in x.

		// Exponent must be at least the number of digits carried.
		dexpon = 1.0 + super_precision :: def_digits * ( 1.000001 *
			log ( super_precision :: def_radix) / log((double) 10));
		iexpon = (int32) dexpon;
		x = pow ( x, iexpon);

		// Apply Salamin algorithm.
		y = 4;
		y /= x;
		x = 1;
		x.arithmetic_geometric_mean ( y);
		x *= 2 * iexpon;
		y = pi ();
		y /= x;
		// Calculate log_10 now.
		*super_precision :: log_10 = 1 / y;
	}
	else
	{			// Do power series calculation of log.

		// Get register pool pointers and define reference variables.
	        super_precision_pool_ptr r_, s_, za_, zb_, zc_;
	        super_precision & r = *r_;
	        super_precision & s = *s_;
	        super_precision & za = *za_;
	        super_precision & zb = *zb_;
	        super_precision & zc = *zc_;

		// First, boost mantissa length for new objects.
		save_nf = super_precision :: nf;
		save_def_digits = super_precision :: def_digits;
		super_precision :: change_mantissa_length ( 3 +
						super_precision :: orig_nf);

		// Reduce the argument range.
			j = ( super_precision :: nf + 4) / 18;
			i = 1 + (int32) sqrt ( (double) super_precision ::nf/2);
			j = ( j < i) ? j : i;
			j += 3;				// a little extra
			j = ( j > 30-1) ? 30-1 : j;  	// but not > 30-1
			i = 2;
		while ( j--) i *= 2;
		za = i + 1;
		za /= i;
		zb = 10;
		ii = 0;
		while ( ( zb > za) || ( zb < 2 - za))
		{
			ii++;
			zb = sqrt ( zb);
		}
		r = zb - za;
		za += zb;
		r /= za;
		za *= -za;
		za /= zb;
		s = -r / 2;
		zb = r;
		r *= r;

		i = 1;
		do
		{
			zb *= r;
			zc = zb / ( i * ( i + 2));
			s += zc;
			i += 2;
		}  while ( (zc.sign_bias_exp != 0)  &&
		   (abs ( zc.sign_bias_exp) + super_precision :: def_digits + 1
		   			> abs ( s.sign_bias_exp)));
		s *= za;
				// 13 is a speed optimization parameter
		for ( i = 0, j = 1; i < 13; i++) j *= 2;
		for ( i = 0; i < 1000; i++)
		{
			if ( ii < 13) break;
			ii -= 13;
			s *= j;
		}
		if ( ii)
		{
			for ( i = 0, j = 1; i < ii; i++) j *= 2;
			s *= j;
		}
		// Calculate log_10 now.
		*super_precision :: log_10 = 1 / s;
	}

			// Specify accuracy of log_10:
	super_precision :: log_10->digits = super_precision :: def_digits - 2;
		// Copy to register before restoring original mantissa length.
	super_precision :: zz->super_move ( *super_precision :: zz,
					*super_precision :: log_10);
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( save_nf, save_def_digits);
		// Do proper roundoff of low-order digits remaining.
	*super_precision :: zz = super_precision :: zz->
				convert_precision ( *super_precision :: zz);
		// Done.
	return ( *super_precision :: zz);
}			// End of function log_10 ().

super_precision log_radix					// FRIEND FN
			( void)
{	// FRIEND FUNCTION				// a = log(def_radix)
// This function calculates log(e) to the base "def_radix" and returns it.
// The quantity e is the natural logarithm base 2.71828...
	int32 i, j, ii, save_nf, save_def_digits;	// temporaries
	short def_radix = super_precision :: def_radix;	// nickname for static

	if ( !super_precision :: log_radix_init)
	{	// Allocate memory block for log_radix using original mantissa.
		save_nf = super_precision :: nf;
		super_precision :: nf = super_precision :: orig_nf + 4;
		super_precision :: log_radix = new super_precision;
		super_precision :: nf = save_nf;
		super_precision :: log_radix_init = 1;
		// Now fall thru to computational section below.
	}
	else
	{

		// Recalculate log_radix if def_digits != static log_radix
		// precision.
		if ( super_precision :: log_radix->digits ==
					super_precision :: def_digits)
		{				// same precision
			// Return the value already calculated.
			return ( *super_precision :: log_radix);
		}
		else
		{
			// Test for request for shorter precision.
			if ( super_precision :: log_radix->digits >
					super_precision :: def_digits)
			{			// shorter precision requested
				return ( super_precision ::
					log_radix->convert_precision
					( *super_precision :: log_radix));
				// *log_radix knows how long its mantissa is and 
				// convert_precision will return a mantissa with
				// only def_digits, the current working length.
			}
			else
			{			// longer precision requested
				// Verify that the desired precision will fit in
				// the allocated mantissa.
				if ( super_precision :: nf >
						super_precision :: orig_nf + 4)
				{
					cout << "Requesting too much precision";
					cout << " for log_radix -- fatal error";
					cout << ".  nf=" <<
						super_precision :: nf
						<< "  orig_nf+4=" <<
						super_precision :: orig_nf+4
						<< endl;
					exit (1);
				}
				// Fall thru to computational section below.
			}
		}
	}

	// Getting here means we have to calculate log_radix or a
	// longer log_radix.

	if ( super_precision :: def_digits >=
				super_precision :: log_Salamin_threshold)

	{			// Do AGM calculation of log.
	
		int32 new_exponent, old_exponent;

		// Get register pool pointers and define reference variables.
		super_precision_pool_ptr x_, y_;
		super_precision & x = *x_;
		super_precision & y = *y_;

		// Boost mantissa length for new objects.
		save_nf = super_precision :: nf;
		save_def_digits = super_precision :: def_digits;
		super_precision :: change_mantissa_length ( 3 +
						super_precision :: orig_nf);

		x = def_radix;			// Put def_radix in x.

		// Exponent must be at least the number of digits carried.
		new_exponent = super_precision :: def_digits + 1;
		if ( new_exponent > super_precision :: bias + 1)
		{
			cout<<"Exponent too big in log_radix -- fatal error.\n";
			cout << "new_exponent = " << new_exponent << " bias = ";
			cout << super_precision :: bias << endl;
			exit (1);
		}
		old_exponent = x.sign_bias_exp - super_precision :: bias;
		x.sign_bias_exp = super_precision :: bias + new_exponent;

		// Apply Salamin algorithm.
		y = 4;
		y /= x;
		x = 1;
		x.arithmetic_geometric_mean ( y);
		x *= 2 * ( new_exponent - old_exponent + 1);
		y = pi ();
		y /= x;
		// Calculate log_radix now.
		*super_precision :: log_radix = 1 / y;
	}
	else
	{			// Do power series calculation of log.

		// Get register pool pointer and define reference variable.
		super_precision_pool_ptr r_, s_, za_, zb_, zc_;
	        super_precision & r = *r_;
	        super_precision & s = *s_;
	        super_precision & za = *za_;
	        super_precision & zb = *zb_;
	        super_precision & zc = *zc_;

		// First, boost mantissa length for new objects.
		save_nf = super_precision :: nf;
		save_def_digits = super_precision :: def_digits;
		super_precision :: change_mantissa_length ( 3 +
						super_precision :: orig_nf);

		// Reduce the argument range.
			j = ( super_precision :: nf + 4) / 18;
			i = 1 + (int32) sqrt ( (double) super_precision::nf /2);
			j = ( j < i) ? j : i;
			j += 2;
			if ( def_radix >= 8) j++;     // a little extra for some
			j = ( j > 30-1) ? 30-1 : j;  	// but not > 30-1
			i = 2;
		while ( j--) i *= 2;
		za = i + 1;
		za /= i;
		zb = def_radix;
		ii = 0;
		while ( ( zb > za) || ( zb < 2 - za))
		{
			ii++;
			zb = sqrt ( zb);
		}

		za = 1;
		r = zb - za;
		za += zb;
		r /= za;
		za *= -za;
		za /= zb;
		s = -r / 2;
		zb = r;
		r *= r;

		i = 1;
		do
		{
			zb *= r;
			zc = zb / ( i * ( i + 2));
			s += zc;
			i += 2;
		}  while ( (zc.sign_bias_exp != 0)  &&
		   (abs ( zc.sign_bias_exp) + super_precision :: def_digits + 1
		   			> abs ( s.sign_bias_exp)));
		s *= za;
				// 13 is a speed optimization parameter
		for ( i = 0, j = 1; i < 13; i++) j *= 2;
		for ( i = 0; i < 1000; i++)
		{
			if ( ii < 13) break;
			ii -= 13;
			s *= j;
		}
		if ( ii)
		{
			for ( i = 0, j = 1; i < ii; i++) j *= 2;
			s *= j;
		}
		// Calculate log_radix now.
		*super_precision :: log_radix = 1 / s;
	}

			// Specify accuracy of log_radix:
	super_precision :: log_radix->digits = super_precision:: def_digits - 2;
		// Copy to register before restoring original mantissa length.
	super_precision :: zz->super_move ( *super_precision :: zz,
					*super_precision :: log_radix);
		// Restore mantissa length.
	super_precision :: change_mantissa_length ( save_nf, save_def_digits);
		// Do proper roundoff of low-order digits remaining.
	*super_precision :: zz = super_precision :: zz->
				convert_precision ( *super_precision :: zz);
		// Done.
	return ( *super_precision :: zz);
}			// End of function log_radix ().

super_precision sinh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = sinh(b)
// This function calculates sinh(b) and returns it.

	if ( b.assert_invalid ( "sinh"))
			return super_NaN ();


	// Test for zero case.
	if ( b.sign_bias_exp == 0)
	{		// Return a zero.
		return (super_precision) 0;
	}

	if ( fabs ( b) >= 1)
	{		// Use large-argument identity -- no precision loss.
		// Get register pool pointer and define reference variable.
        	super_precision_pool_ptr za_;
        	super_precision & za = *za_;

		za = exp ( b);
		return ( ( za - 1 / za) / 2);
	}
	else
	{		// Use small-argument expansion to avoid precision loss.
		// Get register pool pointers and define reference variables.
       		super_precision_pool_ptr r_, s_, w_;
       		super_precision & r = *r_;
       		super_precision & s = *s_;
       		super_precision & w = *w_;
       		int32 i;

		// Figure a speed/accuracy tradeoff parameter "i" used in the
		// argument reduction:
		i = 2 + (int32) sqrt ( 5 * (double) super_precision :: nf);
		i = ( i < 8) ? 8 : i;
		i = ( i > 76) ? 76 : i;
		// With this choice of i, the running time of sinh is about the
		// same as n multiplications, where n = 21 + 3 (nf)^(2/3), up to
		// about nf=1130.

		r = fabs ( b);				// abs(r) < 1

		// Boost mantissa length to allow for argument reduction.
		super_precision :: change_mantissa_length (
					super_precision :: nf + 2);
		r.mantissa [super_precision :: nf - 2] = 0;
		r.mantissa [super_precision :: nf - 1] = 0;
		r.digits = super_precision :: def_digits;

		double x = super_precision :: baslog / 0.301;
		int32 j, k, ij;
		int32 m = (int32) ( x * super_precision :: nd);
		int32 mm = i;
		if ( i >= m)
		{		// Following is to prevent 2^i from overflowing.
			j = 1; ij = m; while ( ij--) j *= 2;	// j=pow(2,m)
			while ( mm > m)
			{		// Reduce mm to something within range.
				r /= j;
				mm -= m;
			}
		}
		j = 1; ij = mm; while ( ij--) j *= 2;	// j = pow ( 2, mm)
		if ( j > 1) r /= j;

		s = r * r;
		w = r;
		k = 2;
		do
		{
			r *= s;
			r /= k * ( k + 1);
			w += r;
			k += 2;
		} while ( ( w.sign_bias_exp < ( r.sign_bias_exp + 1 +
						super_precision :: def_digits))
				&& ( r.sign_bias_exp != 0));

		// Have r = sinh of the reduced argument.  Use the recursion
		//   sinh(2x) = 2 sinh(x) (1 + 2 (sinh(x/2))^2)
		// to expand the argument to the original.
		s = w;			// sinh for reduced argument
		if ( i)			// If i > 0, double argument.
		{
			s *= w;
			r = 1;
			s += r;		// 1 + w * w
			s = sqrt ( s);	// cosh for w=sinh
			s = 2 * w * s;	// double-angle formula for sinh
			i--;		// count this doubled result in s
		}
		while ( i--) 		// Expand argument to original.
		{
			r = w;				// backstore sinh(x/2)
			w = s;				// backstore sinh(x)
			s = 2 * w * ( 1 + 2 * r * r);	// calculate sinh(2x)
		}
			// Restore mantissa length.
		super_precision :: change_mantissa_length ( );
			// Chop result length.
		s.super_chop_register ( );
		s.digits = super_precision :: def_digits;
			// Check sign.
		if ( b.sign_bias_exp < 0) s.sign_bias_exp = - s.sign_bias_exp;

		return ( s);
	}
}

super_precision cosh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = cosh(b)
// This function calculates cosh(e) and returns it.

	if ( b.assert_invalid ( "cosh"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr zd_;
       	super_precision & zd = *zd_;

	zd = exp ( b);
	return ( ( zd + 1 / zd) / 2);
}

super_precision tanh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = tanh(b)
// This function calculates tanh(e) and returns it.

	if ( b.assert_invalid ( "tanh"))
			return super_NaN ();


	// Test for zero case.
	if ( b.sign_bias_exp == 0)
	{		// Return zero.
		return (super_precision) 0;
	}

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr zc_, zd_;
       	super_precision & zc = *zc_;
       	super_precision & zd = *zd_;

	if ( fabs ( b) >= 1)
	{		// Use large-argument identity.
		zd = exp ( - ( b + b));
		zc = 1;
		return ( ( zc - zd) / ( zc + zd));
	}
	else
	{		// Use small-argument identity.
		return ( sinh ( b) / cosh ( b));
	}
}

super_precision sin						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = sin(b)
// This function calculates sin(b) and returns it.

	if ( b.assert_invalid ( "sin"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr xsin_, xcos_;
       	super_precision & xsin = *xsin_;
       	super_precision & xcos = *xcos_;

	sincos ( b, xsin, xcos);
	return ( xsin);
}

super_precision cos						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = cos(b)
// This function calculates cos(b) and returns it.

	if ( b.assert_invalid ( "cos"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr xsin_, xcos_;
       	super_precision & xsin = *xsin_;
       	super_precision & xcos = *xcos_;

	sincos ( b, xsin, xcos);
	return ( xcos);
}

super_precision tan						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = tan(b)
// This function calculates tan(b) and returns it.

	if ( b.assert_invalid ( "tan"))
			return super_NaN ();

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr xsin_, xcos_;
       	super_precision & xsin = *xsin_;
       	super_precision & xcos = *xcos_;

       	// Is argument closer to zero or to pole?  Use xsin and xcos as scratch.
       	xsin.super_move ( xsin, b); 
       	xsin.sign_bias_exp = abs ( xsin.sign_bias_exp);
       	xcos = pi ();
       	if ( xsin > xcos) xsin %= xcos;		// Do mod if more than pi.
       	xcos /= 2;		// Pole is at pi/2.  Zero is at 0 and pi.
       	xsin -= xcos;
       	xsin.sign_bias_exp = abs ( xsin.sign_bias_exp);
       	xcos /= 2;		// Get pi/4.
       	if ( xsin > xcos)
       	{			// Closer to zero.
       		// Check if Salamin algorithm is preferred.
       		if ( super_precision :: def_digits >=
       				super_precision :: tan_Newton_threshold)
       		{
       			*super_precision :: zt = tan_Salamin ( b);
       			return *super_precision :: zt;
       		}

		// Call sincos function with argument.
		sincos ( b, xsin, xcos);
		// Use private divide function directly, instead of divide
		// operator, to avoid creating temporary object. Use static
		// pointer zt for result.
		super_precision :: zt->super_div ( *super_precision :: zt, xsin,
									xcos);
	}
	else
	{			// Closer to pole.  Take complement.
		// Go to higher precision to make sure we get complement right.
		xsin = b;
		super_precision :: change_mantissa_length ( super_precision ::
								nf + 2);
		xsin.mantissa [super_precision :: nf - 1] = 0;
		xsin.mantissa [super_precision :: nf - 2] = 0;
		xsin.digits = super_precision :: def_digits;

		xcos = pi ();		// Get pi to extra precision.
		xcos /= 2;		// Get pi/2.
		// Going to extra precision lets us get this difference better.
		xsin -= xcos;		// Get arg - pi/2.
		xsin.sign_bias_exp = -xsin.sign_bias_exp;    // Get pi/2 - arg.

		// Restore precision and get properly rounded complement.
		super_precision :: change_mantissa_length ();
		xsin.super_chop_register ();
		xsin.digits = super_precision :: def_digits;

       		// Check if Salamin algorithm is preferred.
       		if ( super_precision :: def_digits >=
       				super_precision :: tan_Newton_threshold)
       		{
       			*super_precision :: zt = tan_Salamin ( b);
       			return *super_precision :: zt;
       		}

		// Call sincos function with argument complement (xsin).
		sincos ( xsin, xsin, xcos);

		// Use private divide function directly, instead of divide
		// operator, to avoid creating temporary object. Use static
		// pointer zt for result.
		super_precision :: zt->super_div ( *super_precision :: zt, xcos,
									xsin);
	}

	return ( *super_precision :: zt);
}

super_precision tan_Salamin					// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = tan(b)
// This function calculates tan(b) and returns it.  The method uses a
// Salamin algorithm to calculate the atan function and a recursive Newton-
// Raphson algorithm to get the tan function.  Suitable for very high
// precision.

	// Get register pool pointers and define reference variables.
       	super_precision_pool_ptr r_, s_, t_, u_, w_, za_;
       	super_precision & r = *r_;
       	super_precision & s = *s_;
       	super_precision & t = *t_;
       	super_precision & u = *u_;
       	super_precision & w = *w_;
       	super_precision & za = *za_;

		// Copy argument to working variable r.
	r = b;
		// Boost mantissa length one word to allow for large argument.
	int32 save_nf, save_def_digits;
	save_nf = super_precision :: nf;
	save_def_digits = super_precision :: def_digits;
	super_precision :: change_mantissa_length ( super_precision :: nf + 1);
	r.mantissa [save_nf] = 0;
	r.digits = super_precision :: def_digits;

		// Get the best approximation we can.
		// This method okay unless tan is out of range of double.?????
	double c = b.super_ztod_approx ( r);
	w = w.super_dtoz_approx ( tan (c));

	// Define the following at full precision.
	s.super_zero ( s);
	t.super_zero ( t);
	u.super_zero ( u);
	za.super_zero (za);

	// Shorten up the working mantissa length and increase it in
	// iteration loop.
	int32 working_nf =
		super_precision :: double_fraction_digit_count * 4;

	working_nf = ( working_nf + super_precision :: nd - 1)
					/ super_precision :: nd;
		// But can't exceed register length.
	working_nf = ( working_nf > super_precision::orig_nf + 4)
				? super_precision :: orig_nf + 4
				: working_nf;
	int32 target_digits = super_precision :: def_digits;
	super_precision :: change_mantissa_length ( working_nf);

	// We use a "quartically-convergent" Newton-Raphson algorithm --
	// the number of accurate digits QUADRUPLES with each iteration.
	int32 i, j;
	j = FALSE;
	for ( i = 0; i < 100; i++)
	{
		t = w;
		za = 1;
			// s = atan (w/za), u = log ( w/za).
		log_Salamin_complex ( za, w, u, s);
		s = r - s;
		w = t;
		w *= t;
		za = 1;
		za /= 3;
		za += w;
		za *= s;
		za += t;
		za *= s;
		za *= s;
		za += s;
		s = 1;
		w += s;
		za *= w;
		w = t + za;

		if ( j) break;  

		if ( 4 * working_nf >= save_nf + 1)
		{
				// Test convergence.
			if ( ( target_digits + 3) / 4 <
				4 * ( abs ( t.sign_bias_exp)
				- abs ( za.sign_bias_exp) - 1)) j = TRUE;
				// Restore initial full precision.
			working_nf = save_nf + 1;
				// But can't exceed register length.
			working_nf = ( working_nf > super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
			super_precision :: change_mantissa_length ( working_nf);
		}
		else
		{		// Quadruple working precision.
			working_nf *= 4;
				// But can't exceed register length.
			working_nf = ( working_nf > super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
			super_precision :: change_mantissa_length ( working_nf);
		}
	}

	if ( !j)
	{
		cout << "Fatal error -- nonconvergence in ";
		cout << "super_precision tan_Salamin.\n";
		exit (1);
	}
	else
	{
			// Restore mantissa length.
		super_precision :: change_mantissa_length ( );
			// Chop result length.
		w.super_chop_register ( );
		w.digits = super_precision :: def_digits;

	}
	return ( w);
}			// End of function tan_Salamin ().

super_precision asin						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = asin(b)
// This function calculates asin(b) and returns it.
// The result is on the interval [-pi/2,pi/2].

	if ( b.assert_invalid ( "asin"))
			return super_NaN ();


	// Test for zero argument.
	if ( b.sign_bias_exp == 0)
	{		// Return zero.
		return (super_precision) 0;
	}

	int32 i, j, k, jj;
	int32 negsign = ( b.sign_bias_exp < 0) ? TRUE : FALSE;
	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr zb_, zc_;
       	super_precision & zb = *zb_;
       	super_precision & zc = *zc_;
	zb = b;
	zb.sign_bias_exp = abs ( zb.sign_bias_exp);

		// Boost mantissa length one word to prevent loss of a couple
		// of digits to roundoff.
	super_precision :: change_mantissa_length ( super_precision :: nf + 1);
	zb.mantissa [super_precision :: nf - 1] = 0;
	zb.digits = super_precision :: def_digits;

	zc = 1;
	if ( zc == zb)
	{		// sin is unity.
		// Restore mantissa length.
		super_precision :: change_mantissa_length ();
		if ( negsign)
			return ( -pi () / 2);
		else
			return ( pi () / 2);
	}
	if ( zc < zb)
	{		// sin is greater than unity -- fatal error.
		cout << "Argument of super_precision asin is greater than one";
		cout << " -- fatal error.\n";
		exit (1);
	}

	// See if Salamin algorithm is preferred.
	if ( super_precision :: def_digits >=
				super_precision :: atan_Salamin_threshold)
	{
		// Get register pool pointers and define reference variables.
       		super_precision_pool_ptr a_, za_;
       		super_precision & a = *a_;
       		super_precision & za = *za_;
		a = zb;
		a *= zb;
		zc -= a;
		za = sqrt ( zc);
		log_Salamin_complex ( za, zb, a, zc);
		if ( negsign) zc.sign_bias_exp = -zc.sign_bias_exp;
			// Restore mantissa length and round result.
		super_precision :: change_mantissa_length ();
		zc.super_chop_register ();
		zc.digits = super_precision :: def_digits;

		return ( zc);
	}

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr s_, za_;
       	super_precision & s = *s_;
       	super_precision & za = *za_;

	i = (int32) (0.5 * sqrt ( 1.25 * super_precision :: nf));
	i = ( i > 30) ? 30 : i;
	i = ( i < 1) ? 1 : i;
	k = 1;
	while ( i--) k *= 2;
	
	za = zc / k;
	i = 0;
	while ( zb > za)
	{
		zb = sqrt ( ( zc - sqrt ( zc - zb * zb)) / 2);
		i++;
	}
	k = 2;
	zc = zb;
	s = zb * zb;
	do
	{
		zb *= s;
		zb *= k - 1;
		zb /= k;
		za = zb / ( k + 1);
		zc += za;
		k += 2;
	} while ( ( za.sign_bias_exp != 0) && ( abs ( za.sign_bias_exp) +
		super_precision :: def_digits + 1) > abs ( zc.sign_bias_exp));
	if ( i)
	{
		j = 1;
		for ( jj = 0; jj < i; jj++) { j *= 2;}
		zc *= j;
	}
	if ( negsign) zc.sign_bias_exp = -zc.sign_bias_exp;
		// Restore mantissa length and round result.
	super_precision :: change_mantissa_length ();
	zc.super_chop_register ();
	zc.digits = super_precision :: def_digits;

	return ( zc);
}

super_precision acos						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = acos(b)
// This function calculates acos(b) and returns it.

	if ( b.assert_invalid ( "acos"))
			return super_NaN ();

// The result is on the interval [0,pi].
	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr a_;
       	super_precision & a = *a_;

// The acos function likes to lose precision when the argument is near unity.
// Avoid this by testing to see if 2*fabs(arg) > 1.  If so, use the addition
// formula for inverse circular functions (see Abramowitz and Stegun, 4.4.35):
//   acos(1-x) = asin(sqrt(2*x-x*x))		(Let z2=1 and z1=sqrt(2*x-x*x).)
// where x = 1-arg and arg > 0, or
//   acos(-1+x) = -asin(sqrt(2*x-x*x))
// where x = 1+arg and arg < 0.
// If not so, just find the complement of the asin of arg:
//   acos(arg) = pi/2 - asin(arg)

	a = fabs ( b);
	a *= 2;
	if ( a <= 1)
	{		// Find the complement of the asin of b.
		a = pi () / 2 - asin ( b);
			// Function value is on interval (0,pi).
	}
	else
	{		// Use the addition formula.
		// Get register pool pointer and define reference variable.
       		super_precision_pool_ptr x_;
       		super_precision & x = *x_;
		x = 1 - fabs (b);		// This is x.
		a = 2 - x;
		a *= x;
		a = sqrt ( a);
		a = asin ( a);
		if ( b.sign_bias_exp < 0)
		{		// Adjust value to nearly pi if arg < 0.
			a = pi () - a;
		}
	}
	
	return ( a);
}

super_precision atan						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION					// a = atan(b)
// This function calculates atan(b) and returns it.

	if ( b.assert_invalid ( "atan"))
			return super_NaN ();

// The result is on the interval (-pi/2,pi/2).

	// Test for zero arg.
	if ( b.sign_bias_exp == 0)
	{
		return ( (super_precision) 0);
	}

	// See if Salamin algorithm is preferred.
	if ( super_precision :: def_digits >=
				super_precision :: atan_Salamin_threshold)
	{
		// Get register pool pointers and define reference variables.
       		super_precision_pool_ptr a_, za_, zb_;
       		super_precision & a = *a_;
       		super_precision & za = *za_;
       		super_precision & zb = *zb_;
       		a = 1;
		log_Salamin_complex ( a, b, za, zb);
		return zb;
	}

	// Get register pool pointers and define reference variables.
       	super_precision_pool_ptr a_, za_, zb_, zc_;
       	super_precision & a = *a_;
       	super_precision & za = *za_;
       	super_precision & zb = *zb_;
       	super_precision & zc = *zc_;
	int32 negsign, recip, i, k, jj;
	negsign = ( b.sign_bias_exp < 0) ? TRUE : FALSE;
	zb = b;
	zb.sign_bias_exp = abs ( zb.sign_bias_exp);

		// Boost mantissa length one word to prevent loss of a couple
		// of digits to roundoff.
	super_precision :: change_mantissa_length ( super_precision :: nf + 1);
	zb.mantissa [super_precision :: nf - 1] = 0;
	zb.digits = super_precision :: def_digits;

	zc = 1;
	if ( zb > zc)
	{
		recip = TRUE;
		zb = zc / zb;
	}
	else
	{
		recip = FALSE;
	}
	// The following number 8 is a speed/accuracy tradeoff parameter.
	i = (int32) ( sqrt ( 2.0 + super_precision :: nf) - 1);
	i = ( i < 1) ? 1 : i;
	i = ( i > 30-1) ? 30-1 : i;
	k = 2;
	while ( i--) k *= 2;
	zc /= k;
	i = 0;
	while ( zb > zc)
	{
		zb = zb / ( sqrt ( zb * zb + 1) + 1);
		i++;
	}
	k = 3;
	a = zb;
	zc = zb * zb;
	zc.sign_bias_exp = - zc.sign_bias_exp;
	do
	{
		zb *= zc;
		za = zb / k;
		a += za;
		k += 2;
	} while ( ( za.sign_bias_exp != 0) && ( ( abs ( za.sign_bias_exp) +
		super_precision :: def_digits + 1) > a.sign_bias_exp));
	if ( i)
	{
		k = 1;
		for ( jj = 0; jj < i; jj++) { k *= 2;}
		a *= k;
	}

		// Restore mantissa length and round result.
	super_precision :: change_mantissa_length ();
	a.super_chop_register ();
	a.digits = super_precision :: def_digits;

	if ( recip)
	{
		a = pi () / 2 - a;		// Take complement of a.
	}
	if ( negsign)
	{
		a.sign_bias_exp = - a.sign_bias_exp;
	}

	return ( a);
}

super_precision atan2						// FRIEND FN
			( const super_precision &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// atan2 (a,b)
// This function calculates atan2(b,c) (i.e., atan (b/c)) and returns it.
// Argument b is proportional to the sin of the angle, and argument c is
// proportional to the cos of the angle.  The result is on the interval
// (-pi,pi].

	if ( b.assert_invalid ( c, "atan2")) return super_NaN ();


	// Test for zero args.
	if ( c.sign_bias_exp == 0)
	{
		if ( b.sign_bias_exp == 0)
		{		// Both b and c are zero -- fatal error.
		      cout << "Both atan2 super_precision arguments are zero";
		      cout << " -- fatal error.\n";
		      exit (1);
		}
		else
		{		// c is zero, b is nonzero.
			if ( b.sign_bias_exp > 0)
			{
				return ( pi () / 2);
			}
			else
			{
				return ( - pi () / 2);
			}
		}
	}
	else
	{			// c is nonzero.
		if ( b.sign_bias_exp == 0)
		{		// b is zero.
			if ( c.sign_bias_exp > 0)
			{		// zero sin, positive cos --> 0
				return ( (super_precision) 0);
			}
			else
			{		// zero sin, negative cos --> pi
				return ( pi ());
			}
		}
		else
		{		// b is nonzero.
			int32 negc = FALSE;
			if ( c.sign_bias_exp < 0) negc = TRUE;
		// Get register pool pointer and define reference variable.
       			super_precision_pool_ptr a_;
       			super_precision & a = *a_;
			a = atan ( b / c);
			if ( !negc)
			{		// positive cos --> quadrant 1 or 4
				return ( a);
			}
			else
			{		// negative cos --> quadrant 2 or 3
				if ( a.sign_bias_exp > 0)
				{		// quadrant 3
					return ( a - pi ());
				}
				else
				{		// quadrant 2
					return ( a + pi ());
				}
			}
		}
	}

	// Append an unnecessary return to quiet stupid compilers:
	return b;
}			// End of function atan2.

super_precision asinh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION				// a = asinh (b)
// Use the logarithmic representation ln(x+sqrt(x*x+1)).

	if ( b.assert_invalid ( "asinh"))
			return super_NaN ();


	// Allocate a super_precision object from the pool.
	super_precision_pool_ptr a_;
	super_precision &a = *a_;

	if ( abs ( b.sign_bias_exp) <= super_precision :: bias)
	{
		a = b;
		a *= a;			// Watch out for overflow here.
		a++;
		a = sqrt ( a);
		a += b;
		a = log ( a);
	}
	else
	{		// This branch avoids overflow for arguments up to as
			// large as super_biggest()/2 - epsilon.
		a = 1;
		a /= b;
		a *= a;
		a++;
		a = sqrt ( a);
		a++;
		a *= b;
		a = log ( a);
	}

	return a;
}

super_precision acosh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION				// a = acosh (b)
// Use the logarithmic representation ln(x+sqrt(x*x-1)) for x >= 1.

	if ( b.assert_invalid ( "acosh"))
			return super_NaN ();


	// Allocate a super_precision object from the pool.
	super_precision_pool_ptr a_;
	super_precision &a = *a_;

	// Make sure the argument is not less than unity.
	if ( b < 1)
	{
		cout << "Argument to super_precision acosh is less than unity";
		cout << " -- fatal error.\n";
		exit (1);
	}

	if ( b.sign_bias_exp <= super_precision :: bias)	// b > 0 !
	{
		a = b;
		a *= a;		// Watch out for overflow here.
		a--;
		a = sqrt ( a);
		a += b;
		a = log ( a);
	}
	else
	{		// This branch avoids overflow for arguments up to as
			// large as super_biggest()/2 - epsilon.
		a = 1;
		a /= b;
		a *= a;
		a = 1 - a;
		a = sqrt ( a);
		a++;
		a *= b;
		a = log ( a);
	}

	return a;
}

super_precision atanh						// FRIEND FN
			( const super_precision &b)
{	// FRIEND FUNCTION				// a = atanh (b)
// Use the logarithmic representation (ln((1+x)/(1-x)))/2, valid for x*x < 1.

	if ( b.assert_invalid ( "atanh"))
			return super_NaN ();


	// Allocate super_precision objects from the pool.
	super_precision_pool_ptr a_, d_;
	super_precision &a = *a_, &d = *d_;

	// Make sure the argument is > -1 and < +1.
	if ( b <= -1 || b >= 1)
	{
		cout << "Argument to super_precision atanh is <= -1 or >= +1";
		cout << " -- fatal error.\n";
		exit (1);
	}

	d = 1;
	a = b;
	a += d;
	d -= b;
	a /= d;
	a = log ( a);
	a /= 2;

	return a;
}

super_precision hypot						// FRIEND FN
			( const super_precision &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// hypot (a,b)
// This function calculates hypot(b,c) (i.e., sqrt(b*b+c*c)) and returns it.

	if ( b.assert_invalid ( c, "hypot"))
	{
		return super_NaN ( super_precision ::
			super_irreg_fsa ( "hypot",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
	}


// The trick is to avoid intermediate overflow or underflow, without a divide.
		// Get register pool pointers and define reference variables.
       	super_precision_pool_ptr a_, d_;
       	super_precision & a = *a_;
       	super_precision & d = *d_;
	int32 bb, cc;
	bb = abs ( b.sign_bias_exp) - super_precision :: bias;
	cc = abs ( c.sign_bias_exp) - super_precision :: bias;
	if ( bb >= cc)
	{
		a = b;
		a.sign_bias_exp = 1 + super_precision :: bias;
		a *= a;
		d = c;
		d.sign_bias_exp = abs ( d.sign_bias_exp) - ( bb - 1);
		d *= d;
		a += d;
		a = sqrt ( a);
		a.sign_bias_exp += bb - 1;
		return ( a);
	}
	else
	{
		a = c;
		a.sign_bias_exp = 1 + super_precision :: bias;
		a *= a;
		d = b;
		d.sign_bias_exp = abs ( d.sign_bias_exp) - ( cc - 1);
		d *= d;
		a += d;
		a = sqrt ( a);
		a.sign_bias_exp += cc - 1;
		return ( a);
	}
}

super_precision max						// FRIEND FN
			( const super_precision &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// max (a,b)
// This function calculates the greater (more positive) of a,b and returns it.

	if ( b.assert_invalid ( c, "max"))
	{
		return super_NaN ( super_precision ::
			super_irreg_fsa ( "max",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
	}

	int32 i;

	i = b.super_test ( b, c);
	if ( i >= 0)
	{
		return ( b);
	}
	else
	{
		return ( c);
	}
}

super_precision min						// FRIEND FN
			( const super_precision &b,
			const super_precision &c)
{	// FRIEND FUNCTION					// min (a,b)
// This function calculates the lesser (more negative) of a,b and returns it.

	if ( b.assert_invalid ( c, "min"))
	{
		return super_NaN ( super_precision ::
			super_irreg_fsa ( "min",
					b.validity, b.sign_bias_exp,
					c.validity, c.sign_bias_exp));
	}

	int32 i;

	i = b.super_test ( b, c);
	if ( i <= 0)
	{
		return ( b);
	}
	else
	{
		return ( c);
	}
}

super_precision nthrt						// FRIEND FN
			( const super_precision &b, const int32 n)
{								// n'th root fn
// This function calculates and returns the n'th root of argument b.
// If argument n is negative, the reciprocal of b^(1/n) is returned.
// A value of zero for n is illegal.

	if ( b.assert_invalid ( "nthrt"))
	{
		return super_NaN ( super_precision ::
			super_irreg_fsa ( "nthrt",
					b.validity, b.sign_bias_exp,
					0, n));
	}

	int32 ia, inv;				// temporaries
	int32 i, j, k, ldig, negsign;
	double c, dia;

	// Test for illegal zero value of n.
	if ( !n)
	{
		cout << "Zero'th root attempted by super_precision function";
		cout << " nthrt -- fatal error.\n";
		exit (1);
	}

	if ( n > 0)
	{
		inv = FALSE;
		ia = n;
	}
	else
	{
		inv = TRUE;
		ia = -n;
		if ( b.sign_bias_exp == 0)
		{
			cout << "Trying to take a negative root of zero in ";
			cout <<"super_precision function nthrt -- fatal error.";
			cout << endl;
			exit (1);
		}
	}

	// If ia is unity, case is easy.
	if ( ia == 1)
	{
		if ( inv)
		{
			return ( 1 / b);
		}
		else
		{
			return ( b);
		}
	}

	// Check for illegal "even" root of negative number.
	if ( ( ia % 2 == 0) && ( b.sign_bias_exp < 0))
	{
		cout << "Trying to take an even root of a negative number in ";
		cout << "super_precision function nthrt -- fatal error.\n";
		exit (1);
	}

	// Start algorithm.
	// Get register pool pointers and define reference variables.
       	super_precision_pool_ptr a_, r_, s_, za_, zb_;
       	super_precision & a = *a_;
       	super_precision & r = *r_;
       	super_precision & s = *s_;
       	super_precision & za = *za_;
       	super_precision & zb = *zb_;
	k = ia / super_precision :: def_radix;
	ldig = 1 + ( super_precision :: klip + 1) / 2;
	while ( k)
	{
		ldig++;
		k /= super_precision :: def_radix;
	}
	ldig = ( ldig < super_precision :: orig_def_digits)
			? ldig : super_precision :: orig_def_digits - 1;
	dia = ia;
	zb = fabs ( b);
	if ( b.sign_bias_exp > 0)
	{
		negsign = FALSE;
	}
	else
	{
		negsign = TRUE;
	}
	j = ( zb.sign_bias_exp - super_precision :: bias) / ia;
	zb.sign_bias_exp -= ia * j;
	c = zb.super_ztod_approx ( zb);
	zb.sign_bias_exp = abs ( b.sign_bias_exp);
	c = pow ( c, 1 / dia);
	a = zb.super_dtoz_approx ( c);
	a.sign_bias_exp += j;
	zb /= ia;

		// Shorten up the working mantissa length and increase it in
		// iteration loop.
	int32 save_nf = super_precision :: nf;
	int32 save_def_digits = super_precision :: def_digits;
	int32 working_nf = super_precision :: double_fraction_digit_count * 2;
	working_nf = ( working_nf + super_precision :: nd - 1)
						/ super_precision :: nd;
		// But can't exceed register length.
	working_nf = ( working_nf > super_precision::orig_nf +4)
				? super_precision :: orig_nf +4
				: working_nf;
	super_precision :: change_mantissa_length ( working_nf);

	// Iteration loop doubles accuracy each pass.
	j = FALSE;
	for ( i = 0; i < 100; i++)
	{
		za = a;
		a = pow ( a, ia - 1);
		a = zb / a + ( za * ( ia - 1)) / ia;
		if ( j)
		{	// Finished with loop.  Get result ready to return.
			if ( negsign) a.sign_bias_exp = - a.sign_bias_exp;
			a.digits = super_precision :: def_digits;
			if ( inv)
			{
				return ( 1 / a);
			}
			else
			{
				return ( a);
			}
		}

		if ( 2 * working_nf >= save_nf)
		{
				// Test convergence.
			r = a - za;
			if ( ldig < abs ( a.sign_bias_exp) -
					abs ( r.sign_bias_exp)) j = TRUE;
				// Restore initial full precision.
			working_nf = save_nf;
			super_precision :: change_mantissa_length ( save_nf,
							save_def_digits);
		}
		else
		{		// Double working precision.
			working_nf += working_nf;
				// But can't exceed register length.
			working_nf = ( working_nf > super_precision::orig_nf +4)
						? super_precision :: orig_nf +4
						: working_nf;
			super_precision :: change_mantissa_length ( working_nf);
		}
	}

	cout << "Fatal error -- nonconvergence in super_precision nthrt.\n";
	exit (1);

	// Append an unnecessary return to quiet stupid compilers:
	return b;
}			// End of function nthrt.

super_precision cbrt						// overload cbrt
			( const super_precision &b)
{	// FRIEND function					// a = cbrt(b)
// This function calculates and returns the cube root of argument b.

	if ( b.assert_invalid ( "cbrt"))
	{
		return super_NaN ( super_precision ::
				super_irreg_fsa ( "cbrt",
						b.validity, b.sign_bias_exp));
	}

	return ( nthrt ( b, 3));
}

super_precision frexp						//overload frexp
			( const super_precision &b, int32 &e)
{	// FRIEND function					// f=frexp(b,e)
// This function returns the signed mantissa of b as the value, and returns the
// exponent of b as the argument e.  The returned value has an exponent of zero.

	b.assert_invalid_quit ( "frexp");

	e = abs ( b.sign_bias_exp) - super_precision :: bias;
	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr f_;
       	super_precision & f = *f_;
	f = b;
	f.sign_bias_exp = ( b.sign_bias_exp > 0) ? super_precision :: bias
						 : - super_precision :: bias;
	return ( f);
}

super_precision ldexp						//overload ldexp
			( const super_precision &f, const int32 e)
{	// FRIEND function					// a=ldexp(f,e)
// This function returns a super_precision value with b as the mantissa and the
// argument e as the exponent.  The returned value has the same sign as b.
// The exponent of b is ignored.

	f.assert_invalid_quit ( "ldexp");

	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr a_;
       	super_precision & a = *a_;
	a = f;
	a.sign_bias_exp = ( f.sign_bias_exp > 0) ? e + super_precision :: bias
						 : -(e + super_precision::bias);
	return ( a);
}

void super_precision :: super_chop_register			// PROTECTED FN
				( void)
{							// chop reg precision
// This function rounds a static super_precision "register" precision down to
// the default precision.  It only works with these "register" objects.

	if ( this->assert_invalid ( "super_chop_register")) return;

	if ( iklip > 1)
	{
	// If iklip > 1, then use the low-order digits in the last mantissa
	// element to decide the round bit.  This would be done by super_klipr.
		super_klipr ( *this);
	}
	else
	{
	// If iklip <= 1, then use the next lower-order mantissa element to
	// decide the round bit.  This would be an illegal memory fetch if
	// *this were not long enough (i.e., not a "register" object).
		if ( this->mantissa [nf] < ( ind + 1) / 2)
		{
			// Round bit is zero.  No adjustment is needed.
		}
		else
		{
			// Round bit is 1.  Add it in and rescale the mantissa.
			super_zqncc ( this->mantissa, nf, ind, 1);
			if ( this->mantissa [0] >= ind)
			{	// The only way this mantissa element can get
				// as big as ind is if all later elements are
				// zero identically.  So we only have to rescale
				// this one, by shifting one digit to the right.
				this->mantissa [0] /= def_radix;
				if ( abs ( this->sign_bias_exp) != 2 * bias + 1)
				{
					if ( this->sign_bias_exp < 0)
						this->sign_bias_exp--;
					else
						this->sign_bias_exp++;
				}
				else
				{
					if ( permissive_error)
					{		// Infinite result.
						if ( this->sign_bias_exp > 0)
							this->validity = 1;
						if ( this->sign_bias_exp < 0)
							this->validity = 2;
					}
					else
					{
				cout << "Overflow during a clip operation in";
				cout <<" super_chop_register -- fatal error.\n";
						exit (1);
					}
				}
			}
			else
			{
				// No further action needed.
			}
		}
	}
	return;
}			// End of function super_chop_register.


super_precision precision_conv					// FRIEND FN
				( const super_precision &a)
{								//cnvt precision
// This friend function converts a super_precision object having a possibly
// nondefault precision (not equal to def_digits) as argument "a", into a
// super_precision object having a the current default precision.  The
// conversion is carried out by a protected class function, which uses the
// value of a.digits as the precision of a.

	return a.convert_precision ( a);
}

super_precision super_precision :: convert_precision		// PROTECTED FN
				( const super_precision &a) const
{								//cnvt precision
// This function takes a super_precision object having a possibly-nondefault
// precision, as an argument, and returns a copy of it converted to a
// super_precision object having the current default precision.  The precision
// of the argument is taken from its "digits" data member.
// The current default precision is def_digits, and the current mantissa word
// length is nf.

	if ( a.assert_invalid ( "convert_precision")) return ( a);

	if ( a.digits == def_digits)
	{			// No change needed.  Just clip if needed.
		super_precision b = a;
		super_klipr ( b);
		return ( b);
	}
	else
	{			// Conversion of mantissa length needed.
		int32 i, save_nf, short_nf, long_nf;
		super_precision b;

		if ( a.digits < def_digits)
		{	// Argument is shorter than current default.
			// Fill result mantissa out with zeros.
				// Save mantissa length of current default.
			save_nf = nf;
				// Figure mantissa length of the short operand.
			short_nf = ( a.digits + nd - 1) / nd;
			nf = short_nf;	// Temporarily change default length.
			b = a;		// Copy the short operand.
			nf = save_nf;	// Restore the default mantissa length.
			b.digits = def_digits;
			for ( i = short_nf; i < nf; i++)
			{		// Fill extra mantissa words with zeros.
				b.mantissa [i] = 0;
			}
			return ( b);
		}	// End of converting a shorter to a longer mantissa.
		else
		{	// Truncate a longer operand to a shorter default.
			if ( !( def_radix % 2))
			{	// Radix is even.
				if ( iklip != 1)
				{	// Just copy default length and clip.
					b = a;
					b.digits = def_digits;
					super_klipr ( b); // Does rounding, too.
					return ( b);
				}
				else
				{	// Copy default length, get round bit
					// from next word in long mantissa.
					b = a;
					b.digits = def_digits;
					if ( a.mantissa [ nf] < ind / 2)
					{	// No roundup needed.
						return ( b);
					}
					else
					{	// Add round bit.
						super_zqncc ( b.mantissa, nf,
							ind, 1);
						if ( b.mantissa [0] < ind)
							return ( b);
						b.mantissa [0] /= def_radix;
						if ( abs ( b.sign_bias_exp) !=
							bias * 2 + 1)
						{
							if ( b.sign_bias_exp <0)
							{
							      b.sign_bias_exp--;
							}
							else
							{
							      b.sign_bias_exp++;
							}
							return ( b);
						}
						else
						{
					if ( permissive_error)
					{		// Infinite result.
						if ( b.sign_bias_exp > 0)
							b.validity = 1;
						if ( b.sign_bias_exp < 0)
							b.validity = 2;
					}
					else
					{
						cout << "\nOverflow in convert";
						cout << "_precision -- fatal";
						cout << " error.\n";
						exit (1);
					}
						}
					}	// End of roundbit add-in.
				}	// End of roundbit lookup for iklip=1.
			}	// End of even radix block.
			else
			{	// Radix is odd.
				b = a;	// Copy default mantissa length.
				b.digits = def_digits;
				if ( b.mantissa [nf - 1] % iklip == iklip / 2)
				{	// Marginal roundup -- maybe, maybe not.
					// Examine succeeding mantissa elements.
					// Figure long mantissa length.
					long_nf = ( a.digits + nd - 1) / nd;
					for ( i = nf; i < long_nf; i++)
					{
						if ( a.mantissa [i] <
							ind / 2)
							break;
						if ( a.mantissa [i] >
							ind / 2)
						{	// Adding 1 will cause
							// klipr to round-up:
						super_zqncc ( b.mantissa, nf,
							ind, 1);
							// Check proper scaling.
						if ( b.mantissa [0] < ind)
							break;
						b.mantissa [0] /= def_radix;
						if ( abs ( b.sign_bias_exp) !=
							bias * 2 + 1)
						{
							if ( b.sign_bias_exp <0)
							{
							      b.sign_bias_exp--;
							}
							else
							{
							      b.sign_bias_exp++;
							}
							break;
						}
						else
						{
					if ( permissive_error)
					{		// Infinite result.
						if ( b.sign_bias_exp > 0)
							b.validity = 1;
						if ( b.sign_bias_exp < 0)
							b.validity = 2;
					}
					else
					{
						cout << "\nOverflow in convert";
						cout << "_precision -- fatal";
						cout << " error.\n";
						exit (1);
					}
						}
						}	// End of round-up.
					}	// End of mantissa search loop.
				}	// End of marginal-roundup block.
				else
				{	// klipr will round up or down to
					// correct answer without further
					// adjustment.
				}
				super_klipr ( b);
				return ( b);
			}	// End of odd radix block.
		}	// End of converting a longer to a shorter mantissa.
	}	// End of conversion block.

	// Append an unnecessary return to quiet stupid compilers:
	return a;
}	// End of function convert_precision.

int32 super_precision :: set_fast_multiply_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of fast_mult_digits to be returned.
// If an argument of zero (FALSE) is supplied, the fast multiply is inhibited
// by setting fast_mult_digits = 0.
// If a positive argument is supplied, the fast multiply is enabled for working
// precisions at least as great as the argument (fast_mult_digits is set to i).
//
// Note: The fast multiply uses FFT's to obtain the convolution forming the
// product mantissa.  It is faster for mantissas longer than roughly 500
// decimal digits, and allows a multiply to be obtained in n*log(n) time instead
// of n*n running time, where n is the number of digits carried in the mantissa.

	if ( i < 0)
	{
		return fast_mult_digits;
	}
	else
	{
		return ( fast_mult_digits = i);
	}
}

int32 super_precision :: set_fast_divide_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of fast_div_digits to be returned.
// If an argument of zero (FALSE) is supplied, the fast divide is inhibited
// by setting fast_div_digits = 0.
// If a positive argument is supplied, the fast divide is enabled for working
// precisions at least as great as the argument (fast_div_digits is set to i).
//
// Note: The fast multiply uses FFT's to obtain the convolution forming the
// product mantissa.  It is faster for mantissas longer than roughly 500
// decimal digits, and allows a multiply to be obtained in n*log(n) time instead
// of n*n running time, where n is the number of digits carried in the mantissa.
// The divide is obtained from multiply operations by Newton-Raphson recursion.

	if ( i < 0)
	{
		return fast_div_digits;
	}
	else
	{
		return ( fast_div_digits = i);
	}
}

int32 super_precision :: set_log_Salamin_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of log_Salamin_threshold to be returned.
// If a nonnegative argument is supplied, the log_Salamin algorithm is enabled
// for working precisions at least as great as the argument (that is,
// log_Salamin_threshold is set to i).

	if ( i < 0)
	{
		return log_Salamin_threshold;
	}
	else
	{
		return ( log_Salamin_threshold = i);
	}
}

int32 super_precision :: set_exp_Newton_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of exp_Newton_threshold to be returned.
// If a nonnegative argument is supplied, the Salamin-Newton algorithm is
// enabled for evaluating the exp function for working precisions at
// least as great as the argument (that is, log_Salamin_threshold is set to i).

	if ( i < 0)
	{
		return exp_Newton_threshold;
	}
	else
	{
		return ( exp_Newton_threshold = i);
	}
}

int32 super_precision :: set_atan_Salamin_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of atan_Salamin_threshold to be returned.
// If a nonnegative argument is supplied, the atan-Salamin algorithm is enabled
// for working precisions at least as great as the argument (that is,
// atan_Salamin_threshold is set to i).

	if ( i < 0)
	{
		return atan_Salamin_threshold;
	}
	else
	{
		return ( atan_Salamin_threshold = i);
	}
}

int32 super_precision :: set_tan_Newton_threshold		// PUBLIC FN
					( const int32 i)
{	// PUBLIC FUNCTION				// set/show threshold
// If no argument is supplied, the default argument -1 causes the current
// value of tan_Newton_threshold to be returned.
// If a nonnegative argument is supplied, the tan_Salamin algorithm is enabled
// for working precisions at least as great as the argument (that is,
// tan_Newton_threshold is set to i).

	if ( i < 0)
	{
		return tan_Newton_threshold;
	}
	else
	{
		return ( tan_Newton_threshold = i);
	}
}

void super_precision :: super_dunpk				// PROTECTED FN
			( const int32 a [], double dalt [],
			const int32 nfx, const int32 indx,
			const int32 nfftx) const
{
	int32 ii, kk, ll;			// temporaries
	int32 i, k;
	int32 djj;

	dalt [0] = 0;
	djj = dinda;
	ll = indx;

	i = 0;
	ii = 0;
	kk = a [0];
	while ( 1)
	{
		if ( djj >= ll)
		{
			djj /= ll;
			dalt [ii] += kk * djj;
			i++;
			if ( i == nfx) break;
			kk = a [i];
			ll = indx;
			if ( djj > 1) continue;
			djj = dinda;
			ii += 2;
	// dnfa is twice as big because we fill every second element of dalt[].
//TEST FIX
//			if ( ii >= dnfa) break;
			if ( ii >= 2 * nfftx) break;
			dalt [ii] = 0;
		}
		else
		{
			ll /= djj;
			k = kk / ll;
			dalt [ii] += k;
			kk -= k * ll;
			djj = dinda;
			ii += 2;
//TEST FIX
//			if ( ii >= dnfa) break;
			if ( ii >= 2 * nfftx) break;
			dalt [ii] = 0;
		}
	}

	// Fill remainder of fast_mult array with zeros.
	for ( i = ii + 2; i < 2 * nfftx; i += 2)
	{
		dalt [i] = 0;
	}
	
	return;
}

void super_precision :: super_drepk				// PROTECTED FN
			( int32 t [], const double dalt [],
			const int32 result_length, const int32 nfftx) const
{
	int32 ii;
	double dkk, dll;
	int32 i, k;
	int32 jj;

		// Source is scaled to a range dinda, except for first element.
	double source_scale = (double) dinda;
		// Start with source_scale times source_scale.
	dll = source_scale * source_scale;
		// Target is scaled to a range of indah.
	int32 target_scale = indah;
		// Start with target_scale.
	jj = target_scale;

	i = 0;
	dkk = dalt [i];
	ii = 0;
	t [ii] = 0;
	while ( 1)
	{
		if ( jj >= dll)
		{
			jj /= (int32) dll;
			t [ii] += (int32) ( dkk + 0.5) * jj;
			i++;
//TEST FIX
//			if ( i == dnfa) break;
			if ( i == nfftx) break;
			dkk = dalt [i];
			dll = source_scale;
			if ( jj <= 1)
			{
				jj = target_scale;
				ii++;
				if ( ii >= result_length) break;
				t [ii] = 0;
			}
		}
		else
		{
			dll /= jj;
// TEST			k = (int32) ( ( dkk + 0.5) / dll);
			k = (int32) ( ( dkk + ( dkk >= 0 ? 0.5 : -0.5)) / dll);
			t [ii] += k;
			dkk -= k * dll;
			jj = target_scale;
			ii++;
			if ( ii >= result_length) break;
			t [ii] = 0;
		}
	}
	if ( i >= nfftx)
	{
		cout << "super_drepk: internal error, i >= nfftx";
		cout << "     i =" << i << " and nfftx =" << nfftx << endl;
		exit (1);
	}

	return;
}

super_precision super_precision :: increment_mantissa		// PUBLIC FN
				( const int32 increment)
{	// PUBLIC FUNCTION				// add arg to mantissa
// This function adds the argument "increment" to the super_precision object
// pointed to by "this", and normalizes the result if necessary.  The default
// value of increment is unity.

	if ( this->assert_invalid ( "increment_mantissa"))
					return super_NaN ( this->validity);


	if ( increment < 0)
	{
		cout << "Increment value is negative in increment_mantissa --";
		cout << " fatal error.  increment = " << increment << endl;
		exit (1);
	}

	int32 carry;
	if ( increment < ind / iklip)
	{
		carry = increment * iklip;
		if ( carry) super_zqncc ( this->mantissa, nf, ind, carry);
	}
	else
	{
		int32 j = ind / iklip;
		carry = ( increment % j) * iklip;
		if ( carry) super_zqncc ( this->mantissa, nf, ind, carry);
		carry = increment / j;
		if ( carry) super_zqncc ( this->mantissa, nf - 1, ind, carry);
	}
	
//	if ( this->mantissa [0] >= ind || this->mantissa [0] < ind / def_radix)
	if ( this->mantissa [0] >= ind || this->mantissa [0]
						< radix_powers [nd - 1])
			super_norm ( *this, carry);
	return *this;
}

super_precision super_precision :: decrement_mantissa		// PUBLIC FN
				( const int32 decrement)
{	// PUBLIC FUNCTION				// add arg to mantissa
// This function subtracts argument "decrement" from the super_precision object
// pointed to by "this", and normalizes the result if necessary.  The default
// value of decrement is unity.  Note that a unit decrement of a power of
// def_radix yields a string of digits def_radix-1, including the last digit --
// i.e., the decrement yields the next smallest representable number.

	if ( this->assert_invalid ( "decrement_mantissa"))
					return super_NaN ( this->validity);


			// Test for zero object.
	if ( this->sign_bias_exp == 0)
	{		// Return the object unchanged.
		return *this;
	}

	if ( decrement < 0)
	{
		cout << "Decrement value is negative in decrement_mantissa --";
		cout << " fatal error.  decrement = " << decrement << endl;
		exit (1);
	}

			// Is this->mantissa a power of def_radix?
	int32 j;
//	if ( this->mantissa [0] == ind / def_radix)
	if ( this->mantissa [0] == radix_powers [nd - 1])
	{
		for ( j = nf - 1; j > 0; j--)
		{
			if ( this->mantissa [j] != 0) break;
		}
			// Yes, it is.  Left shift before decrementing.
		super_shift ( *this, 1);
	}

	int32 carry;
	if ( decrement < ind / iklip)
	{
		carry = decrement * iklip;
		if ( carry) super_zqncb ( this->mantissa, nf, ind, carry);
	}
	else
	{
		j = ind / iklip;
		carry = ( decrement % j) * iklip;
		if ( carry) super_zqncb ( this->mantissa, nf, ind, carry);
		carry = decrement / j;
		if ( carry) super_zqncb ( this->mantissa, nf - 1, ind, carry);
	}

//	if ( this->mantissa [0] >= ind || this->mantissa [0] < ind / def_radix)
	if ( this->mantissa [0] >= ind || this->mantissa [0]
						< radix_powers [nd - 1])
			super_norm ( *this, carry);
	return *this;
}

int32 super_precision :: get_digit				// PUBLIC FN
				( const int32 index) const
{	// PUBLIC FUNCTION				// get digit at index
// This function returns the digit at "index" location in the mantissa -- the
// first digit is returned for index = 0, etc.  If index is negative, then
// the digit location is counted from the low-order end of the mantissa; i.e.,
// index = -1 is the last, index = -2 is the next-to-last, etc.

	this->assert_invalid_quit ( "get_digit");

			// Test to make sure *this has the default radix.
	if ( this->radix != def_radix)
	{
		cout << "Operand of get_digit does not have default radix --";
		cout << " fatal error.\n  this->radix=" << this->radix;
		cout << "  def_radix=" << def_radix << endl;
		exit (1);
	}

	int32 indexx = ( index >= 0) ? index : ( def_digits + index);
	int32 iword = indexx / nd;
	int32 idig = nd - 1 - ( indexx -iword * nd);
	if ( indexx >= def_digits || iword >= nf || idig >= nd)
	{
		cout << "Requested nonexistent digit in get_digit -- fatal";
		cout << " error.  index = " << index << " def_digits = ";
		cout << def_digits << endl;
		exit (1);
	}
//	int32 i = 1;
//	while ( idig--) i *= def_radix;
	int32 i = radix_powers [idig];		// i = def_radix ** idig
	i = this->mantissa [iword] / i;
	return  i % def_radix;
}

void super_precision :: put_digit				// PUBLIC FN
				( const int32 index, const int32 value)
{	// PUBLIC FUNCTION				// put digit at index
// This function puts digit "value" into the "index" location in the mantissa --
// the first digit is replaced for index = 0, etc.  If index is negative, then
// the digit location is counted from the low-order end of the mantissa; i.e.,
// index = -1 is the last, index = -2 is the next-to-last, etc.

	this->assert_invalid_quit ( "put_digit");

			// Test to make sure *this has the default radix.
	if ( this->radix != def_radix)
	{
		cout << "Operand of put_digit does not have default radix --";
		cout << " fatal error.\n  this->radix=" << this->radix;
		cout << "  def_radix=" << def_radix << endl;
		exit (1);
	}

	if ( value < 0 || value >= def_radix)
	{
		cout << "Invalid digit value supplied to put_digit () -- fatal";
		cout << " error.  value = " << value << "  def_radix = ";
		cout << def_radix << endl;
		exit (1);
	}
	int32 indexx = ( index >= 0) ? index : ( def_digits + index);
	int32 iword = indexx / nd;
	int32 idig = nd - 1 - ( indexx - iword * nd);
	if ( indexx >= def_digits || iword >= nf || idig >= nd)
	{
		cout << "Specified nonexistent digit in put_digit () -- fatal";
		cout << " error.  index = " << index << " def_digits = ";
		cout << def_digits << endl;
		exit (1);
	}
//	int32 i = 1;
//	while ( idig--) i *= def_radix;
	int32 i = radix_powers [idig];		// i = def_radix ** idig
	int32 j = this->mantissa [iword];
	int32 k = j / i;
	int32 l = j - i * k;
	this->mantissa [iword] = (( k / def_radix) * def_radix + value) * i + l;
	return;
}

int32 super_precision :: get_mantissa_word			// PUBLIC FN
				( const int32 index) const
{	// PUBLIC FUNCTION				// get mantissa word
// This function returns the mantissa word at "index" location in the mantissa.
// The first mantissa word is returned for index = 0, etc.  If index is
// negative, or greater than or equal to super_precision::nf, then a "fatal
// error" message is issued.

	if ( index < 0 || index >= nf)
	{
		cout << "Requested nonexistent mantissa word in get_digit ()";
		cout << " -- fatal error.  index = " << index << " nf = ";
		cout << nf << endl;
		exit (1);
	}
	return  this->mantissa [index];
}

int super_precision :: disclose_radix				// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show radix of this
	return ( this->radix);
}

int32 super_precision :: disclose_digits			// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show digits of this
	return ( this->digits);
}

int32 super_precision :: disclose_validity			// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show validity of this
	return ( this->validity);
}

int32 super_precision :: disclose_scale_factor			// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show static ind
	return ( ind);
}

int32 super_precision :: disclose_mantissa_word_length		// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show static nf
	return ( nf);
}

int32 * super_precision :: disclose_mantissa_pointer		// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show mantissa pointer
	return ( this->mantissa);
}

int32 super_precision :: disclose_long_div_threshold		// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION				// show static indah
	return ( indah);
}

int super_precision :: disclose_def_radix			// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION					//show def_radix
	return ( def_radix);
}

int super_precision :: disclose_sign				// PUBLIC FN
				( void) const
{	// PUBLIC FUNCTION					// show sign
	
	if ( this->validity == +1) return +1;
	if ( this->validity == +2) return -1;

	this->assert_invalid_quit ( "disclose_sign");

	int sign_val = 0;
	if ( this->sign_bias_exp > 0) sign_val = 1;
	if ( this->sign_bias_exp < 0) sign_val = -1;

	return ( sign_val);
}

int32 disclose_fftsize						// FRIEND FN
				( void)
{	// FRIEND FUNCTION
// This friend function discloses the current value of the super_precision
// Library's internal variable nfft, the size of the (complex) Fast Fourier
// Transform used to carry out long-mantissa multiplications.

	return ( super_precision :: nfft);
}

super_precision super_biggest					// FRIEND FN
				( void)
{	// FRIEND FUNCTION				// return biggest
// This function returns a super_precision object with the largest representable
// value possible for the current radix and precision.  The object is properly
// normalized.  A positive sign is assigned.
	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr f_;
       	super_precision & f = *f_;

       	f.sign_bias_exp = 2 * super_precision :: bias + 1;
       	int32 i;
       	for ( i = 0; i < super_precision :: nf; i++)
       	{
       		f.mantissa [i] = super_precision :: ind - 1;
       	}
       		// Clear digits beyond specified length.
       	if ( super_precision :: klip) f.mantissa [super_precision :: nf - 1] =
       		( f.mantissa [super_precision :: nf - 1]
       			/ super_precision :: iklip) * super_precision :: iklip;
       	return ( f);
}

super_precision super_smallest					// FRIEND FN
				( void)
{	// FRIEND FUNCTION				// return smallest
// This function returns a super_precision object with the smallest
// representable value possible for the current radix and precision.  The object
// is properly normalized.  A positive sign is assigned.
	// Get register pool pointer and define reference variable.
       	super_precision_pool_ptr f_;
       	super_precision & f = *f_;

       	f.sign_bias_exp = 1;
       	int32 i;
       	f.mantissa [0] = super_precision :: ind / super_precision :: def_radix;
       	for ( i = 1; i < super_precision :: nf; i++)
       	{
       		f.mantissa [i] = 0;
       	}
       	return ( f);
}

int32 super_precision :: leading_zero_count			// PUBLIC FN
				( void) const
{							// count leading zeros
// This function returns the number of leading zeros in the mantissa of this
// object.  It is assumed that each mantissa word is properly scaled!
	
	this->assert_invalid_quit ( "leading_zero_count");

	int32 i = 0, j, k, l;

	for ( j = 0; j < nf; j++)
	{
		k = mantissa [j];
		if ( k) break;
		i++;
	}

	if ( i == nf)
	{
		return ( digits);
	}
	else
	{
		i *= nd;
		l = ind;
		for ( j = 0; j < nd; j++)
		{
			l /= radix;
			if ( k >= l) break;
			i++;
			if ( i > digits) break;
		}
		return ( i);
	}
}

int32 super_precision :: trailing_zero_count			// PUBLIC FN
				( void) const
{							// count trailing zeros
// This function returns the number of "significant" trailing zeros in the
// mantissa of the this object.
	
	this->assert_invalid_quit ( "trailing_zero_count");

	int32 i, j, k, l;

	i = 0;
	for ( j = nf - 1; j >= 0; j--)
	{
		k = mantissa [j];
		if ( k) break;
		i++;
	}

	if ( i == nf)
	{
		return ( i * nd - ndig + digits);
	}
	else
	{
		i *= nd;
		for ( j = 0; j < nd; j++)
		{
			l = k / radix;
			if ( k - l * radix) break;
			k = l;
			i++;
		}
		return ( i - ndig + digits);
	}
}

int super_precision :: verify_scaling				// PUBLIC FN
				( void) const
{							// verify proper scaling
// This function verifies that the first mantissa word is less than ind and
// not less than ind/radix, and that subsequent mantissa words are less than
// ind but greater than or equal to 0.  A value 1 (TRUE) is returned if this
// is the case, or 0 (FALSE) is returned otherwise.
	
	this->assert_invalid_quit ( "verify_scaling");

	int32 i = ind / radix, k;

	if ( this->mantissa [0] >= ind || this->mantissa [0] < i)
		return 0;
	for ( k = 0; k < nf; k++)
	{
		if ( this->mantissa [k] >= ind || this->mantissa [k] < 0)
			return 0;
	}

	return 1;
}

int super_precision :: assert_invalid				// PROTECTED FN
			( char *s) const
{							//test validity of *this
// This function tests the validity of *this, and returns a FALSE value if
// if this->validity is zero.  If this->validity is not zero, then (1) if the
// class variable permissive_error is TRUE, a value of TRUE is returned, or
// (2) if permissive_error is FALSE, a fatal error message for the string *s is
// issued and execution is terminated.

	if ( !this->validity) return 0;
	if ( permissive_error)
	{
		return 1;
	}
	else
	{
		cout << "Fatal error -- the super_precision argument is NaN at";
		cout << " the top of\nsuper_precision function " << s << endl;
		exit (1);
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
	}

	// Append an unnecessary return to quiet stupid compilers:
	return 0;
}

int super_precision :: assert_invalid				// PROTECTED FN
			( const super_precision &b, char *s) const
{						// test validity of *this and b
// This function tests the validity of *this and b and returns a FALSE value if
// this->validity or b.validity is zero.  If either is nonzero, (1) if the class
// variable permissive_error is TRUE, a value of TRUE is returned, or (2) if
// permissive_error is FALSE, a fatal error message for the string *s is
// issued and execution is terminated.

	if ( !this->validity && !b.validity) return 0;
	if ( permissive_error)
	{
		return 1;
	}
	else
	{
		if ( this->validity && !b.validity)
		{
		cout << "Fatal error -- first super_precision argument is NaN";
		cout << " at the top of\nsuper_precision function "<< s << endl;
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
		exit (1);
		}
		if ( !this->validity && b.validity)
		{
		cout << "Fatal error -- second super_precision argument is NaN";
		cout << " at the top of\nsuper_precision function "<< s << endl;
	cout << "NaN type is " << validity_type_string ( b.validity) << endl;
		exit (1);
		}
		cout << "Fatal error -- both super_precision arguments are NaN";
		cout << " at the top of\nsuper_precision function "<< s << endl;
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
	cout << "NaN type is " << validity_type_string ( b.validity) << endl;
		exit (1);
	}

	// Append an unnecessary return to quiet stupid compilers:
	return 0;
}

void super_precision :: assert_invalid_quit			// PROTECTED FN
			( char *s) const
{							//test validity of *this
// This function tests the validity of *this, and returns if this->validity =0.
// If this->validity is not zero, then a fatal error message for the string *s
// is issued and execution is terminated.  The permissive_error flag does not
// affect this function.

	if ( !this->validity) return;
	cout << "Fatal error -- the super_precision argument is NaN at";
	cout << " the top of\nsuper_precision function " << s << endl;
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
	exit (1);
}

void super_precision :: assert_invalid_quit			// PROTECTED FN
			( const super_precision &b, char *s) const
{						// test validity of *this,b
// This function tests the validity of *this and b and returns if
// this->validity = 0 and b.validity = 0.  However, if either is nonzero then a
// fatal error message for the string *s is issued and execution is terminated.
// The permissive_error flag does not affect this function.

	if ( !this->validity && !b.validity) return;
	if ( this->validity && !b.validity)
	{
	cout << "Fatal error -- first super_precision argument is NaN";
	cout << " at the top of\nsuper_precision function " << s << endl;
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
	exit (1);
	}
	if ( !this->validity && b.validity)
	{
	cout << "Fatal error -- second super_precision argument is NaN";
	cout << " at the top of\nsuper_precision function " << s << endl;
	cout << "NaN type is " << validity_type_string ( b.validity) << endl;
	exit (1);
	}
	cout << "Fatal error -- both super_precision arguments are NaN";
	cout << " at the top of\nsuper_precision function " << s << endl;
	cout << "NaN type is " << validity_type_string ( this->validity) <<endl;
	cout << "NaN type is " << validity_type_string ( b.validity) << endl;
	exit (1);
}

char *super_precision :: validity_type_string			// PROTECTED FN
				( int i)
{							// yields descriptive
// This function returns a short string describing the validity classification
// corresponding to the argument i, which is assumed to be the validity member
// of a super_precision object.
	static char unnormalized []     = "UnN";
	static char proper_form []      = "PrF";
	static char pos_infinity []     = "+Inf";
	static char neg_infinity []     = "-Inf";
	static char pos_neg_infinity [] = "+-Inf";
	static char not_a_number []     = "NaN";
	static char infinity []         = "Inf";

	if ( i < -1 || i > 5)
	{
		cout << "Fatal error -- invalid validity member in super_";
		cout << "precision function validity_type_string =" << i <<endl;
		exit (1);
	}

	switch ( i)
	{
		case -1:
			return ( unnormalized);
		case 0:
			return ( proper_form);
		case 1:
			return ( pos_infinity);
		case 2:
			return ( neg_infinity);
		case 3:
			return ( pos_neg_infinity);
		case 4:
			return ( not_a_number);
		case 5:
			return ( infinity);
	}

	// Append an unnecessary return to quiet stupid compilers:
	return 0;
}

super_precision super_NaN					// FRIEND FN
				( int i)
{							// return a super NaN
// This function returns a super_precision NaN.  The default argument is 4,
// which is the validity member value for not-a-number.
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	// Check for valid range -1, or 1:4.
	if ( ( i != -1 && i < 1) || i > 4)
	{
		cout << "Illegal request for irregular form in super_NaN ";
		cout << "function -- fatal error. arg = " << i << endl;
		exit (1);
	}

	x.super_zero ( x);
	x.validity = i;
	return ( x);
}

super_precision super_Pinf					// FRIEND FN
				( void)
{
// This function returns a super_precision positive infinity.
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	x.super_zero ( x);
	x.validity = 1;
	return ( x);
}

super_precision super_Ninf					// FRIEND FN
				( void)
{
// This function returns a super_precision negative infinity.
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	x.super_zero ( x);
	x.validity = 2;
	return ( x);
}

super_precision super_PNinf					// FRIEND FN
				( void)
{
// This function returns a super_precision positive-or-negative infinity.
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	x.super_zero ( x);
	x.validity = 3;
	return ( x);
}

super_precision super_Unn					// FRIEND FN
				( void)
{
// This function returns a super_precision unnormalized number, having
// magnitude but uncertain sign.
	super_precision_pool_ptr x_;
	super_precision &x = *x_;

	x.super_zero ( x);
	x.validity = -1;
	return ( x);
}

int super_precision :: super_irreg_fsa				// PROTECTED FN
			( char *s, int a, int32 asign,
			int b, int32 bsign)
{							// irregular operands
// This function consults the appropriate finite-state-acceptor to determine
// the result of the operation "s" on the irregular super_precision objects
// whose "validity" member elements are equal to the arguments a and b.
// The signs of a and b are carried in the arguments asign and bsign.

	static char str1 [] = "add";
	static char str2 [] = "minus";
	static char str3 [] = "neg";
	static char str4 [] = "mult";
	static char str5 [] = "multi";
	static char str6 [] = "div";
	static char str7 [] = "divi";
	static char str8 [] = "sqrt";
	static char str9 [] = "fabs";
	static char str10 [] = "trunc";
	static char str11 [] = "ceil";
	static char str12 [] = "floor";
	static char str13 [] = "fmod";
	static char str14 [] = "hypot";
	static char str15 [] = "max";
	static char str16 [] = "min";
	static char str17 [] = "nthrt";
	static char str18 [] = "cbrt";
	static char* list [18] ={str1, str2, str3,  str4,  str5,  str6,  str7,
			         str8, str9, str10, str11, str12, str13, str14,
				 str15,str16,str17,str18 };
	static int number_operands [18] = { 2, 2, 1, 2, 1, 2, 1, 1, 1, 1,
				            1, 1, 2, 2, 2, 2, 1, 1};

		// unn = unnormalized, pro = proper form, pinf = +infinity,
		// ninf = -infinity, pninf = +or-infinity, nan = not a number
		// sinf = infinity with sign of b
		// rinf = infinity with opposite sign of b
	enum {unn = 0, pro, pinf, ninf, pninf, nan, sinf, rinf};

		// The following tables define the result of an operation on
		// each possible super_precision form.

	//    second operand b= unn   pro  pinf  ninf  pninf  nan    first op a
 	//			---------------------------------    ----------
	static int add_result [6] [6] =
			     {  unn,  unn, pinf, ninf, pninf, nan,  // a=unn
				unn,  pro, pinf, ninf, pninf, nan,  // a=pro
				pinf, pinf,pinf, nan,  nan,   nan,  // a=pinf
				ninf, ninf,nan,  ninf, nan,   nan,  // a=ninf
				pninf,pninf,nan, nan,  nan,   nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int minus_result [6] [6] =
			     {  unn,  unn, ninf, pinf, pninf, nan,  // a=unn
				unn,  pro, pinf, ninf, pninf, nan,  // a=pro
				pinf, pinf,nan,  pinf, nan,   nan,  // a=pinf
				ninf, ninf,ninf, nan,  nan,   nan,  // a=ninf
				pninf,pninf,nan, nan,  nan,   nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int mult_result [6] [6] =
			     {  unn,  unn, pninf,pninf,pninf, nan,  // a=unn
				unn,  pro, sinf, rinf, pninf, nan,  // a=pro
				pninf,sinf,pinf, ninf, pninf, nan,  // a=pinf
				pninf,rinf,ninf, pinf, pninf, nan,  // a=ninf
				pninf,pninf,pninf,pninf,nan,  nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int div_result [6] [6] =
			     {  unn,  unn, pninf,pninf,pninf, nan,  // a=unn
				unn,  pro, sinf, rinf, pninf, nan,  // a=pro
				pninf,sinf,pinf, ninf, pninf, nan,  // a=pinf
				pninf,rinf,ninf, pinf, pninf, nan,  // a=ninf
				pninf,pninf,pninf,pninf,nan,  nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int fmod_result [6] [6] =
			     {  unn,  unn, nan,  nan,  nan,   nan,  // a=unn
				unn,  pro, nan,  nan,  nan,   nan,  // a=pro
				nan,  nan, nan,  nan,  nan,   nan,  // a=pinf
				nan,  nan, nan,  nan,  nan,   nan,  // a=ninf
				nan,  nan, nan,  nan,  nan,   nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int hypot_result [6] [6] =
			     {  unn,  unn, pinf, pinf, pinf,  nan,  // a=unn
				unn,  pro, pinf, pinf, pinf,  nan,  // a=pro
				pinf, pinf,pinf, pinf, pinf,  nan,  // a=pinf
				pinf, pinf,pinf, pinf, pinf,  nan,  // a=ninf
				pinf, pinf,pinf, pinf, pinf,  nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int max_result [6] [6] =
			     {  unn,  unn, pinf, unn,  nan,   nan,  // a=unn
				unn,  pro, pinf, pro,  nan,   nan,  // a=pro
				pinf, pinf,pinf, pinf, pinf,  nan,  // a=pinf
				unn,  pro, pinf, ninf, pninf, nan,  // a=ninf
				nan,  nan, pinf, pninf,pninf, nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int min_result [6] [6] =
			     {  unn,  unn, unn,  ninf, nan,   nan,  // a=unn
				unn,  pro, pro,  ninf, nan,   nan,  // a=pro
				unn,  pro, pinf, ninf, pninf, nan,  // a=pinf
				ninf, ninf,ninf, ninf, ninf,  nan,  // a=ninf
				nan,  nan, pninf,ninf, pninf, nan,  // a=pninf
				nan,  nan, nan,  nan,  nan,   nan}; // a=nan
	static int neg_result [6] = {  unn,  pro, ninf, pinf, pninf, nan};
	static int multi_result [6] = {unn,  pro, sinf, rinf, pninf, nan};
	static int divi_result [6] = { unn,  pro, sinf, rinf, pninf, nan};
	static int sqrt_result [6] = { nan,  pro, pinf, nan,  nan,   nan};
	static int fabs_result [6] = { unn,  pro, pinf, pinf, pinf,  nan};
	static int trunc_result [6] = {unn,  pro, pinf, pinf, pninf, nan};
	static int ceil_result [6] = { unn,  pro, pinf, ninf, pninf, nan};
	static int floor_result [6] = {unn,  pro, pinf, pinf, pninf, nan};
	static int cbrt_result [6] = { unn,  pro, pinf, ninf, pninf, nan};
	static int nthrt_result [6] = {unn,  pro, pinf, nan,  nan,   nan};

	static int *op_result [18] =
		{ (int *) add_result, (int *) minus_result, (int *) neg_result,
		(int *) mult_result, (int *) multi_result, (int *) div_result,
		(int *) divi_result, (int *) sqrt_result, (int *) fabs_result,
		(int *) trunc_result, (int *) ceil_result, (int *) floor_result,
		(int *) fmod_result, (int *) hypot_result, (int *) max_result,
		(int *) min_result, (int *) nthrt_result, (int *) cbrt_result };

	//------------ Start here.

		// Match *s with one of the strings in list.
	int i, j = -1, fatal = 0;
	for ( i = 0; i < 18; i++)
	{			// If strncmp returns 0, s and list[i] match
				// thru strlen(s) characters.
		if ( !strncmp ( s, list [i], strlen ( s))) j = i;
	}
	if ( j < 0)
	{
		cout << "Fatal error in super_irreg_fsa -- no operation match.";
		cout << " op =" << s << endl;
		exit (1);
	}

		// Now check validity range of first argument.
	if ( a < -1 || a > 4)
	{
		cout << "Fatal error in super_irreg_fsa -- \n";
		cout << "Validity member of first operand is out of range:";
		cout << a << endl;
		fatal = 1;
	}

		// Now check validity range of second argument.
	if ( number_operands [j] == 2)
	{
		if ( b < -1 || b > 4)
		{
			cout << "Fatal error in super_irreg_fsa -- \n";
			cout << "Validity member of second operand is out of ";
			cout << "range: " << b << endl;
			fatal = 1;
			if ( b == -2) cout<< "Second argument omitted in call."\
				<< endl;
		}
	}
	if ( fatal) exit (1);

	if ( number_operands [j] == 1)
	{
		i = *(op_result [j] + a + 1);	// add one to a to put on 0 to 5
		if ( i == 5 || i == 6)
		{		// Use sign of second (integer) operand.
			j = ( bsign > 0) ? 0 : 1;
		}
	}
	else
	{
		i = *(op_result [j] + 6 * ( a + 1) + b + 1); // add 1 to a and b
		if ( i == 5 || i == 6)
		{		// Use sign of pro operand.
			if ( a == 0)
			{		// If a is pro, use its sign.
				j = ( asign > 0) ? 0 : 1;
			}
			else
			{		// If a isn't pro, b must be pro.
				j = ( bsign > 0) ? 0 : 1;
			}
		}
	}
	if ( i == 5)
	{		// Use same sign.
		i = ( j == 0) ? 1 : 2;
	}
	else
	{	// i == 6, so use opposite sign.
		i = ( j == 0) ? 2 : 1;
	}
	return i;
}


int32 f4t ( double *w, int32 ntotal, int32 n, int32 nprev, int32 is)
{
    /* System generated locals */
    int32 s__1, s__2;
    int32 i__1, i__2, i__3, i__4, i__5;

    /* Builtin functions */
//    double atan( double), cos( double), sin( double);

    /* Local variables */
    static int32 knpr, mnpr;
    static int32 knpr2, mnpr2;
    static double a, b;
    static int32 ntot2, i, k, m;
    static double x, y, d2;
    static int32 m2, n2;
    static double twopi;
    static int32 ia;
    static int32 ja, jb, jc, jd;
    static int32 nprev2, ib, ic, id;
    static double cs, sn, ocs, osn;
    static int32 sqrtn;

/*CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC*/
/*  F4T IS A MULTIDIMENSIONAL FAST FOURIER TRANSFORM SUBROUTINE.  THE */
/*  ARRAY W HOLDS THE ORIGINAL AND THE TRANSFORMED DATA (COMPLEX), */
/*  WITH ALTERNATING REAL AND IMAGINARY PARTS IN STORAGE. */
/*  NTOTAL IS THE TOTAL NUMBER OF COMPLEX DATA POINTS IN W TO BE */
/*  TRANSFORMED.  N IS THE SIZE OF THE TRANSFORM TO BE PERFORMED. */
/*  NPREV IS THE PRODUCT OF ALL DIMENSION SUBSCRIPT LIMITS TO THE LEFT */
/*  OF THE SUBSCRIPT BEING TRANSFORMED, OR IS 1 IF THERE ARE NONE TO */
/*  THE LEFT. IS -1 FOR A FORWARD TRANSFORM AND +1 FOR AN INVERSE */
/*  TRANSFORM.A THREE-DIMENSIONAL FORWARD TRANSFORM OF COMPLEX DATA */
/*  IN A(N1,N2,N3) IS DONE BY THE CALLS */
/*          CALL F4T(A,N1*N2*N3,N1,1,-1) */
/*          CALL F4T(A,N1*N2*N3,N2,N1,-1) */
/*          CALL F4T(A,N1*N2*N3,N3,N1*N2,-1) */
/*  IN ANY ORDER. */
/*  A ONE-DIMENSIONAL FORWARD TRANSFORM OF COMPLEX DATA IN A(N) IS */
/*  DONE BY : */
/*          CALL F4T(A,N,N,1,-1) */

/*  THIS VERSION OF THE FFT COMBINES BIT-REVERSAL SORTING WITH THE */
/*  SUCCESSIVE TWO-POINT TRANSFORMS SO THAT THERE IS NOT A SEPARATE */
/*  SORTING SECTION. */

/*  --- BY WILLIAM T. WYATT, JR., HARRY DIAMOND LABORATORIES, AUGUST 1971. */
/*  Ported from Fortran to C using f2c, 3 July 1995 -- William T. Wyatt, Jr. */

    /* Parameter adjustments */
    --w;

    /* Function Body */
    twopi = atan((double) (is)) * (float)8.;
    n2 = n + n;
    ntot2 = ntotal + ntotal;
    nprev2 = nprev + nprev;
    m2 = 2;
    mnpr2 = m2 * nprev;
/*  Calculate square root of n to nearest power of two: */
    sqrtn = 1;
    while ( sqrtn * sqrtn <= n) sqrtn += sqrtn;
    sqrtn /= 2;
L10:
    m = m2;
    m2 = m + m;
    mnpr = mnpr2;
    mnpr2 = mnpr + mnpr;
    k = n2 / m;
    knpr = k * nprev;
    knpr2 = knpr + knpr;
    x = twopi / (double) m;
    cs = (float)1.;
    sn = (float)0.;
    ocs = cos(x);
    osn = -(double)sin(x);
    d2 = -(double)(osn + osn);
    if (m > sqrtn) {
	goto L40;
    }

    i__1 = mnpr;
    i__2 = nprev2;
    for (i = 1; i__2 < 0 ? i >= i__1 : i <= i__1; i += i__2) {
	ia = i + nprev2 - 1;
	i__3 = ia;
	for (ib = i; ib <= i__3; ib += 2) {
	    i__4 = ntot2;
	    i__5 = knpr2;
	    for (ic = ib; i__5 < 0 ? ic >= i__4 : ic <= i__4; ic += i__5) {
		id = ic + knpr - 1;
		s__1 = id;
		s__2 = mnpr2;
		for (ja = ic; s__2 < 0 ? ja >= s__1 : ja <= s__1; ja += s__2) 
			{
/*  TWO TWO-POINT TRANSFORMS (WITH ROTATION FACTORS) WITH EXCHANGE: */
		    jb = ja + knpr;
		    jc = ja + mnpr;
		    jd = jc + knpr;
		    a = cs * w[jb] - sn * w[jb + 1];
		    b = cs * w[jb + 1] + sn * w[jb];
		    x = w[jc];
		    y = w[jc + 1];
		    w[jc] = w[ja] - a;
		    w[jc + 1] = w[ja + 1] - b;
		    w[ja] += a;
		    w[ja + 1] += b;
		    a = cs * w[jd] - sn * w[jd + 1];
		    b = cs * w[jd + 1] + sn * w[jd];
		    w[jd] = x - a;
		    w[jb] = x + a;
		    w[jd + 1] = y - b;
		    w[jb + 1] = y + b;
		}
	    }
	}
	a = ocs - d2 * sn;
	ocs = cs;
	cs = a;
	a = osn + d2 * ocs;
	osn = sn;
	sn = a;
    }
    goto L10;

L40:
    i__2 = mnpr;
    i__1 = nprev2;
    for (ib = 1; i__1 < 0 ? ib >= i__2 : ib <= i__2; ib += i__1) {
	ic = ib + nprev2 - 1;
	i__5 = ic;
	for (ia = ib; ia <= i__5; ia += 2) {
	    s__2 = ntot2;
	    s__1 = mnpr2;
	    for (ja = ia; s__1 < 0 ? ja >= s__2 : ja <= s__2; ja += s__1) {
/*  ONE TWO-POINT TRANSFORM (WITH ROTATION FACTOR) WITHOUT EXCHANGE. */
		jb = ja + mnpr;
		a = cs * w[jb] - sn * w[jb + 1];
		b = cs * w[jb + 1] + sn * w[jb];
		w[jb] = w[ja] - a;
		w[jb + 1] = w[ja + 1] - b;
		w[ja] += a;
		w[ja + 1] += b;
	    }
	}
	a = ocs - d2 * sn;
	ocs = cs;
	cs = a;
	a = osn + d2 * ocs;
	osn = sn;
	sn = a;
    }
    if (m2 <= n) {
	goto L10;
    }

    return 0;
} /* f4t */

int32 jamcor ( double *w, int32 n, int32 ii)
{
    /* System generated locals */
    int32 s__1;

    /* Builtin functions */
//    double atan( double), cos( double), sin( double);

    /* Local variables */
    static double temp, c;
    static int32 i, j;
    static double s, x, tempa, tempi, tempr, oc, os, ds2;
    static int32 np2;
    static int32 n2m1;

/*CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC*/
/*  JAMCOR IS A SUBROUTINE TO COMPLETE A ONE-DIMENSIONAL FAST FOURIER */
/* TRANSFORM OF A REAL DATA ARRAY PERFORMED IN SLIGHTLY OVER HALF THE CORE */
/* AND RUNNING TIME NEEDED FOR TRANSFORMING THE SAME NUMBER OF COMPLEX DATA*/
/*  POINTS. */
/*  N, THE NUMBER OF POINTS IN THE TRANSFORM, MUST BE EVEN. */
/*  W HAS LENGTH N+2 REAL WORDS. */
/*  II IS -1 FOR THE FORWARD TRANSFORM AND +1 FOR THE INVERSE TRANSFORM. */
/*  THE CALLING SEQUENCE FOR A FORWARD N-POINT TRANSFORM OF N REAL DATA */

/*  VALUES PACKED IN THE LOWEST N WORDS OF THE ARRAY W IS */
/*          CALL FOUR1(W,N/2,-1)          --- OR OTHER FFT SUBROUTINE */
/*          CALL JAMCOR(W,N,-1) */
/*  THE RESULTANT ECONOMIZED TRANSFORM IS IN W(1) THROUGH W(N+2) WITH THE */
/*  CONJUGATE SYMMETRIC (FOLDED) PART NOT GENERATED. */

/*  THE CALLING SEQUENCE FOR THE INVERSE TRANSFORM OF THE ECONOMIZED */
/*  TRANSFORM (FOLDED PART NOT USED) IS */
/*          CALL JAMCOR(W,N,1) */
/*          CALL FOUR1(W,N/2,1)           --- OR OTHER FFT SUBROUTINE */
/* THE RESULTANT N REAL POINTS ARE PACKED IN THE N LOWEST WORDS OF ARRAY W .*/

/*  --- BY WILLIAM T. WYATT, JR., MERDC, OCTOBER 1970. */
/*  Ported from Fortran to C using f2c, 3 July 1995 -- William T. Wyatt, Jr. */

    /* Parameter adjustments */
    --w;

    /* Function Body */
    np2 = n + 2;
    n2m1 = n / 2 - 1;
    x = atan((float)1.) * (float)8. / n;
    if (ii > 0) {
	x = -(double)x;
    }
    oc = (float)1.;
    os = (float)0.;
    c = cos(x);
    s = -(double)sin(x);
    ds2 = s + s;
    if (ii <= 0)
    {
/*              THE FORWARD TRANSFORM SECTION STARTS HERE. */

	    s__1 = n2m1;
	    for (i = 3; i <= s__1; i += 2) {
		j = np2 - i;
		temp = (w[i + 1] + w[j + 1]) * (float).5;
		tempa = (w[j] - w[i]) * (float).5;
		tempr = c * temp - s * tempa;
		tempa = c * tempa + s * temp;
		temp = tempr;
		tempr = (w[i] + w[j]) * (float).5;
		tempi = (w[i + 1] - w[j + 1]) * (float).5;
		w[j] = tempr - temp;
		w[j + 1] = tempa - tempi;
		w[i] = tempr + temp;
		w[i + 1] = tempi + tempa;
		temp = oc - ds2 * s;
		oc = c;
		c = temp;
		temp = os + ds2 * oc;
		os = s;
		s = temp;
	    }
	    w[n + 1] = w[1] - w[2];
	    w[n + 2] = (float)0.;
	    w[1] += w[2];
	    w[2] = (float)0.;
	    w[n2m1 + 3] = -(double)w[n2m1 + 3];
	    return 0;
    }
    else
    {

/*            THE INVERSE TRANSFORM SECTION STARTS HERE. */

	    s__1 = n2m1;
	    for (i = 3; i <= s__1; i += 2) {
		j = np2 - i;
		temp = w[i] - w[j];
		tempa = w[i + 1] + w[j + 1];
		tempr = temp * c - tempa * s;
		tempi = temp * s + tempa * c;
		temp = w[i] + w[j];
		tempa = w[i + 1] - w[j + 1];
		w[i] = temp - tempi;
		w[i + 1] = tempa + tempr;
		w[j] = temp + tempi;
		w[j + 1] = tempr - tempa;
		temp = oc - ds2 * s;
		oc = c;
		c = temp;
		temp = os + ds2 * oc;
		os = s;
		s = temp;
	    }
	    w[n2m1 + 2] *= (float)2.;
	    w[n2m1 + 3] *= (float)-2.;
	    temp = w[1];
	    w[1] = temp + w[n + 1];
	    w[2] = temp - w[n + 1];
	    return 0;
    }
} /* jamcor */

