;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;                                                                         ;
;       Forms Dictionary system, analyse form content                     ;
;                                                                         ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; --------------------------------------------------------------------------
; Procedure   : ScanOneForm()
; Description : Process a single form
;   Arguments : Target directory, table name, form number
;     Returns : Form title
; --------------------------------------------------------------------------

proc ScanOneForm(workdir, tabname, fnum)
   private pagenum, totpages, entries, afname, arow, astartcol, aendcol,
      alinktype, formlines, formname

array afname[100]
array arow[100]
array astartcol[100]
array aendcol[100]
array aftype[100]
array alinkform[100]
array formlines[23]
pagenum = 1

while True
   setdir workdir
   {Forms} {Change} select tabname select fnum
   formname = menuchoice()
   Enter
   window move getwindow() to -100, -100

   totpages = npages()
   while (pagenum > pageno())
      pgdn
   endwhile
   @ 4,1
   clear eol
   ?? "Table " + tabname + ", Form " + fnum + ", Page " + strval(pagenum)
   @ 6,1
   clear eol
   ?? "Scanning row "
   frame double from 0,0 to 8,37
   echo normal   echo off

   entries = ScanOnePage()
   {Cancel} {Yes}
   setdir homedir
   UpdateImageData()
   UpdateFieldData(entries)
   UpdateLinkData(entries)

   if pagenum = totpages then
      quitloop
   else
      pagenum = pagenum + 1
   endif
endwhile

return formname

endproc
WriteLib AppLib ScanOneForm
Release procs ScanOneForm

; --------------------------------------------------------------------------
; Procedure   : ScanOnePage()
; Description : Process one page of a form
; --------------------------------------------------------------------------

proc ScanOnePage()
   private i, j, fdata, oldfdata, intable, entrynum

intable = False
infield = False
entrynum = 0
home

for i from 1 to 23
   @ 6,14
   ?? strval(i)
   echo normal   echo off
   formlines[i] = cursorline()
   oldfdata = ""
   ctrlhome
   for j from 1 to 80
      fdata = fieldinfo()
      if fdata <> oldfdata then
         EntryEnd(i, j - 1, entrynum)
         if fdata <> "" then
            entrynum = entrynum + 1
            EntryStart(i, j, entrynum)
         endif
         oldfdata = fdata
      endif
      right
   endfor
   if fdata <> "" then
      EntryEnd(i, 80, entrynum)
   endif
   down
endfor

return entrynum

endproc
WriteLib AppLib ScanOnePage
Release procs ScanOnePage

; --------------------------------------------------------------------------
; Procedure   : EntryStart()
; Description : Save details of field or embedded table
;   Arguments : Start row and column, index into data capture arrays
; --------------------------------------------------------------------------

proc EntryStart(nrow, ncol, enum)
   private string1, string2,iscomma, infotype, nullval

infotype = substr(fdata, 1, 3)
iscomma = match(fdata, ".., ..", string1, string2)

switch
   case infotype = "Rec":
      afname[enum] = fdata
      aftype[enum] = ""
   case infotype = "Emb":
      intable = True
      if iscomma then
         aftype[enum] = "Linked"
      else
         aftype[enum] = "Unlinked"
         string1 = fdata
      endif
      nullval = match(string1, "Embedded .. table using form ..",
         afname[enum], alinkform[enum])
   otherwise:
      afname[enum] = string2
      aftype[enum] = string1
endswitch

arow[enum] = nrow
astartcol[enum] = ncol

endproc
WriteLib AppLib EntryStart
Release procs EntryStart

; --------------------------------------------------------------------------
; Procedure   : EntryEnd()
; Description : Save details of field or embedded table end location
;   Arguments : Start row and column, index into data capture arrays
; --------------------------------------------------------------------------

proc EntryEnd(nrow, ncol, enum)
   private x

if oldfdata = "" then
   return
endif
aendcol[enum] = ncol
if intable then
   intable = False
   x = astartcol[enum] - 1
   formlines[nrow] = substr(formlines[nrow], 1, x) + fill("+", ncol - x) +
      substr(formlines[nrow], ncol + 1, 80 - ncol)
endif

endproc
WriteLib AppLib EntryEnd
Release procs EntryEnd

; --------------------------------------------------------------------------
; Procedure   : UpdateImageData()
; Description : Add the screen image to table Frmimage
; --------------------------------------------------------------------------

proc UpdateImageData()
   private i

@ 6,1
clear eol
?? "Updating screen image data"
frame double from 0,0 to 8,37
echo normal   echo off

edit "Frmimage"
for i from 1 to 23
   if not isblank([Table]) then
      ins
   endif
   [Table] = tabname
   [Form] = iif(fnum = "F", fnum, "F" + fnum)
   [Page] = pagenum
   [Row] = i
   [Picture] = formlines[i]
endfor
Do_It!
clearimage

endproc
WriteLib AppLib UpdateImageData
Release procs UpdateImageData

; --------------------------------------------------------------------------
; Procedure   : UpdateFieldData()
; Description : Add the field data to table Frmfield
;    Argument : Number of array entries
; --------------------------------------------------------------------------

proc UpdateFieldData(entries)
   private i, string1, string2

@ 6,1
clear eol
?? "Updating field data"
frame double from 0,0 to 8,37
echo normal   echo off

view "Frmimage"
view "Frmfield"
editkey
for i from 1 to entries
   if aftype[i] = "Linked" or aftype[i] = "Unlinked" then
      loop
   endif
   if not isblank([Table]) then
      ins
   endif
   [Table] = tabname
   [Form] = iif(fnum = "F", fnum, "F" + fnum)
   [Page] = pagenum
   [Startrow] = arow[i]
   if match(afname[i], ".., wrap:..", string1, string2) then
      [Name/Formula] = string1
      [Endrow] = arow[i] + numval(string2) - 1
      ProcessWordWrap(i, numval(string2))
   else
      [Name/Formula] = afname[i]
      [Endrow] = arow[i]
   endif
   [Startcol] = astartcol[i]
   [Endcol] = aendcol[i]
   [Type] = iif(aftype[i] = "Formula", "Calculated", aftype[i])

endfor
Do_It!
clearall

endproc
WriteLib AppLib UpdateFieldData
Release procs UpdateFieldData

; --------------------------------------------------------------------------
; Procedure   : ProcessWordWrap()
; Description : Update screen image data to show word-wrapped fields
;    Argument : Array index, number of word-wrapped lines
; --------------------------------------------------------------------------

proc ProcessWordWrap(enum, nlines)
   private i, j, k, fcode

j = astartcol[enum] - 1
k = aendcol[enum]
moveto "Frmimage"
fcode = iif(fnum = "F", fnum, "F" + fnum)
locate tabname, fcode, pagenum, arow[enum]
for i from 2 to nlines
   down
   [Picture] = substr([Picture], 1, j) + fill("_", k - j) +
      substr([Picture], k + 1, 80 - k)
endfor
moveto "Frmfield"
[Wrap] = nlines

endproc
WriteLib AppLib ProcessWordWrap
Release procs ProcessWordWrap

; --------------------------------------------------------------------------
; Procedure   : UpdateLinkData()
; Description : Add the linked form data to table Frmlinks
;    Argument : Number of array entries
; --------------------------------------------------------------------------

proc UpdateLinkData(entries)
   private i, fcode

edit "Frmlinks"
for i from 1 to entries
   if not (aftype[i] = "Linked" or aftype[i] = "Unlinked") then
      loop
   endif
   fcode = iif(fnum = "F", fnum, "F" + fnum)
   locate tabname, fcode, afname[i]
   if retval = False then
      loop
   endif
   if [Linktype] = "None" then
      [Linktype] = aftype[i]
      [Page] = pagenum
      [Startrow] = arow[i]
      [Startcol] = astartcol[i]
      [Endrow] = arow[i]
      [Endcol] = aendcol[i]
      [Linkform] = alinkform[i]
   else
      [Endrow] = arow[i]
   endif
endfor
Do_It!
clearimage

endproc
WriteLib AppLib UpdateLinkData
Release procs UpdateLinkData

; --------------------------------------------------------------------------
; Procedure   : RecordFormLinks
; Description : Build the form links table with base data
; --------------------------------------------------------------------------

proc RecordFormLinks()
   private i, fnum, tabname

view "Frmlinks"
view "Formlist"
editkey
scan
   fnum = iif([Form] = "F", "F", substr([Form], 2, 2))
   tabname = workdir + [Table]
   if ismultiform(tabname, fnum) then
      moveto "Frmlinks"
      formtables tabname fnum linklist
      for i from 1 to arraysize(linklist)
         if not isblank([Table]) then
            ins
         endif
         [Table] = [Formlist->Table]
         [Form] = [Formlist->Form]
         [Linktable] = linklist[i]
         [Linktype] = "None"
      endfor
      moveto "Formlist"
   endif
endscan
Do_It!
clearall

endproc
WriteLib AppLib RecordFormLinks
Release procs RecordFormLinks

; --------------------------------------------------------------------------
; Procedure   : ScanForms()
; Description : Scan the forms in the selected list
; --------------------------------------------------------------------------

proc ScanForms()
   private i, workdir, tablenames, formnumbers, nrecs, msghandle

workdir = controls["Workdir"]
nrecs = nrecords("Formlist")
array tablenames[nrecs]
array formnumbers[nrecs]
array formtitles[nrecs]

window create @ 7,21 height 11 width 40 to msghandle
window setattributes msghandle from msgwindow
echo normal
echo off
clear
frame double from 0,0 to 8,37
@ 2,1
?? "Building list of linked forms"
echo normal   echo off
RecordFormLinks()

view "Formlist"
scan
   tablenames[recno()] = [Table]
   formnum = iif([Form] = "F", "F", substr([Form], 2, 2))
   formnumbers[recno()] = formnum
endscan
clearall

for i from 1 to nrecs
   clear
   frame double from 0,0 to 8,37
   @ 2,1
   ?? "Processing form " + strval(i) + " of " + strval(nrecs)
   echo normal   echo off
   formtitles[i] = ScanOneForm(workdir, tablenames[i], formnumbers[i])
endfor

edit "Formlist"
scan
    [Form name] = formtitles[recno()]
endscan
Do_It!
clearall
window close
echo normal   echo off
controls["Dictionary"] = "All"
controls["Menustate"] = "None"

endproc
WriteLib AppLib ScanForms
Release procs ScanForms
