/*
  Move To drive/directory.
  Copyright 1996-9 Jason Hood.

  Started:  14 April, 1996.
  Finished:  4 June.

  29 September to 1 November, 1996, v2.00:
    changed treatment of previous directories;
    removed "-> sorting -> indexing -> done" from the scan;
    added the extension option;
    changed the nonexistant directory message;
    changed the [wr]long functions to #define;
    fixed a bug with find function dealing with star not finding a directory;
    added open functions to open a file and exit the program if unable;
    added a hardware error handler to ignore drive not ready;
    modified reverse & finddirs function;
    added "+" & "-" options for adding and removing directories.

  9 to 11 February, 1997, v2.10:
    added "##" to the make the new extension permanent, default extension;
    added return codes;
    can accept a trailing (back)slash;
    can accept a partial and then exact path (eg. "mt w /not/scanned");
    added the directory structure file to the display;
    modified to work with MTMem 2.00;
    changed setdrive to distinguish between unavailable and invalid.

  2 March, 1997, v2.11:
    corrected a bug dealing with trailing (back)slash.

  27 March to 1 April, 1997, v2.20:
    more than two previous directories allowed;
    select a partial directory by number;
    list matches using number 0.

  17 July, 1997, v2.21:
    correctly identify the directory attribute ("&" not "="!);
    include hidden and system directories.

  9 November, 1997, v2.22:
    corrected bug with the indexer - non-alphabetical characters greater
     than "Z" skipped everything;
    corrected related bug with partdir().

  8 & 9 December, 1997, v2.23:
    stylistic changes;
    removed '*' from numbered list;
    proper correction of non-alphabetic directories;
    general maintenance;
    added version numbers after the modification dates.

  26 December, 1997, v2.30:
    added LFN support.

  18 to 20 May, 1998, v2.40:
    expand now does (back)slash removal first, followed by LFN testing,
     followed by slash replacement;
    made the previous directories local to main and used an array;
    ignored the current directory when creating the previous directories list;
    added the directory stack.

  27 May, 1998, v2.41:
    corrected LFN and remdir bugs I introduced in v2.40.

  30 June, 1998, v2.42:
    LFNgetname no longer resets the buffer;
    displayed most directories using lowercase (if not LFN, but drive always)
     and slashes instead of backslashes.

  4 November, 1998, v2.43:
    added e-mail address and directory stack commands to the help screen.

  10, 11, 15 to 17 December, 1999, v2.50:
    compiled in the small model, MaxDirs increased to 1800;
    allow display of all directories ("mt 0," or "mt 0,c:");
    display update path using the long name;
    corrected update (copying) problems when the file exceeded 32k;
    use LFNchdir since sometimes the normal one doesn't work (CD only?);
    set ES in isLFN and LFNgetname;
    assume a trailing quote should be a trailing backslash (command-line
     problem);
    test for a hardware error in setdrive to get around deleted current
     directory problems;
    wrote an assembly version of findmpx just for the fun of it;
    actually delete the current directory, moving back to the parent.


  Will change drive as well as directory.
  Allows use of slash ("/") as well as backslash ("\").
  Can use multiple dots (eg. treats "..." as "..\..").
  Previous directory can be selected by "mt;" (or "mt ;" if not so lazy).
  Use more semicolons ("mt;;;") or a number ("mt;3") to select others.
  Partial directory names, where searches always start from the root.
  Directories can be stacked, ie. save the current directory, move to another,
  and then retrieve the saved directory.
  "mt @drives" will construct a directory structure file for drives. "mt @"
  will update the directory structure file for drives already in the file.
  "mt +[+][path]" will create and add path to the file. If the other plus is
  present, it will move to the new path. "mt -[-]path" will delete path and
  remove it from the file (root is never deleted). The second minus will just
  remove the path from the file.
  Default file is "mtdirs.dat" (with path specified via mtmem). It can be
  changed using "mt #[#]ext ..." where "ext" is the new extension (ie.
  "mtdirs.ext") and "..." are the normal options. The second '#' will make
  the new extension permanent. If "ext" is omitted it will default to "dat".

  Path and previous directories are stored in memory, allocated by mtmem.

  Exit Status:
    0 - Successful operation (including help & status)
    1 - MTMem not loaded
    2 - Unable to create/load the structure file
    3 - Directory not found
    4 - Cannot create directory
    5 - Cannot delete directory
    6 - Directory stack empty
    7 - Directory stack full


  Acknowledgements: Tim Jones' WASTED.PAS for finding directories.
		    findmpx is a modified version of Ralf Brown's findtsrs.

  You are free to use this code, or a portion thereof, as long as an
  appropriate acknowledgement is made.

  Questions, suggestions and comments to jadoxa@hotmail.com.
*/

#include <dir.h>
#include <dos.h>
#include <string.h>
#include <fstream.h>
#include <iomanip.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <limits.h>

#define version "2.50"

#define EXIT_SUCCESS 0
#define EXIT_NOMTMEM 1
#define EXIT_NOMTDIR 2
#define EXIT_UNFOUND 3
#define EXIT_NOCREAT 4
#define EXIT_NOREMOV 5
#define EXIT_STKEMPT 6
#define EXIT_STKFULL 7


char mtdirs[MAXPATH];			// Directory structure file

char olddir[MAXDIR], newdir[260];	// The current and new directories

char** dirs;				// Directories for each drive
int    dirnum;				// Number of directories on each drive
const unsigned MaxDirs = 1800;		// Maximum number of directories/drive

int setdrive(char drive);		// Set the active drive

int popdir(unsigned segment, unsigned offset); // Remove a dir. from the stack

void expand(char* path);		// Expand the dots
int  find(char* path);			// Try and find path
int  partdir(int num, char* partial[], int which); // Find a partial dir.
void reverse(char drive, const char* path);	   // Reverse path

int  adddir(char* dir);			// Add a directory (tree)
int  remdir(char* dir);			// Remove a directory (tree)
void deldir();				// Delete directory
void update(int operation);		// Update the directory structure file
void copy(ifstream& is, ofstream& os, long size);	// Copy size bytes
void copy(char* path, int start, int num, ifstream& is, int alpha);
					// Copy directories

void dirfile(char* drives);		// Create directory structure for drives
void finddirs();			// Find directories
int  finddirec(struct ffblk& dir, const char* name = NULL);   // Find dir. name
long index(ofstream& os, char drive);	// Index and write drive's directories
int  sort(const void* a, const void* b);// How the directories are sorted
int  subs(const char* path);		// Number of directories in path

void help();				// Display help screen

int  isLFN(char drive); 		// Does drive support long names?
int  LFNchdir(const char* path);	// Change to a long-name directory
int  LFNmkdir(const char* path);	// Create a long-name directory
int  LFNgetname(int func, const char* name, char* buffer, int subst);
					// Convert from one to the other
#define TRUENAME  0			// Values for func
#define SHORTNAME 1
#define LONGNAME  2

char* longname(char* shortname);	// Return the long name


// Modify mtdirs' extension to ext (ie. "mtdirs.dat" becomes "mtdirs.ext")
#define modext(ext) strcpy(strrchr(mtdirs, '.')+1, ext)

const long zero = 0;			// Variable to simulate wlong(os, 0)

// Write a long variable to a file as a binary value (ie. four bytes)
#define wlong(os, num) os.write((char*)&num, sizeof(long))

// Read a long variable from a file as a binary value (ie. four bytes)
#define rlong(is, num) is.read((char*)&num, sizeof(long))


// Open the mtdirs file for reading. If it fails, exit the program with an
// error message.
void open(ifstream& is)
{
  is.open(mtdirs, ios::in | ios::binary);
  if (is) return;
  cout << "Unable to open \"" << longname(mtdirs) << "\".\n";
  exit(EXIT_NOMTDIR);
}

// Open the mtdirs file for writing. If it fails, exit the program with an
// error message.
void open(ofstream& os)
{
  os.open(mtdirs, ios::out | ios::binary);
  if (os) return;
  cout << "Unable to create \"" << longname(mtdirs) << "\".\n";
  exit(EXIT_NOMTDIR);
}


volatile int hardware_error = 0;

// Hardware error handler for drive not ready - just ignore it.
int handler(int, int, int, int)
{
  ++hardware_error;
  hardretn(_HARDERR_IGNORE);		// Could become fail
  return 0;
}


// Find the multiplex number associated with MTMem. Return -1 if not installed.
int findmpx()
{
#if 1
  static char* sig = "Adoxa   MTMem   ";

  asm {
	mov 	di,di		// SI and DI are automatically saved

	xor 	bx,bx		// BH = 0 (mpx #), BL = 0 (install check)
  }
  Loop:
  asm {
	mov	ax,bx
	int	0x2d
	cmp	al,0xff
	jne	Next
	mov	es,dx
	mov	si,sig
	mov	cx,8
	rep	cmpsw
	je	Found
  }
  Next:
  asm {
	inc	bh
	jnz	Loop
	mov	ax,-1
	jmp	short Exit
  }
  Found:
  asm {
	mov	al,ah
	mov	ah,0
  }
  Exit:
  return _AX;				// Too important a warning to turn off

#else

  char far* sig;
  union REGS regs;

  for (int mpx = 0; mpx <= 255; mpx++)
  {
    regs.h.ah = mpx;
    regs.h.al = 0;			// Installation check
    int86(0x2d, &regs, &regs);
    if (regs.h.al == 0xff)		// Installed?
    {
      sig = (char far*)MK_FP(regs.x.dx, regs.x.di);
      if (_fstrncmp("Adoxa   MTMem   ", sig, 16) == 0) return mpx;
    }
  }
  return -1;
#endif
}


int main(int argc, char* argv[])
{
  char     prevbuf[2*MAXDIR+1],		// Previous directories buffer
	  *prev[20];			// Assume no more than 20
  int      prevnum;			// Number of previous directories
  unsigned prevend;			// Memory limit
  int  	   j;

  int mtmem;				// Multiplex number
  if ((mtmem = findmpx()) == -1)
  {
    cout << "You must run \"MTMem\" first.\n";
    return EXIT_NOMTMEM;
  }
  union REGS regs;			// Get the address of the structure
  regs.h.ah = mtmem;			//  file and previous directories
  regs.h.al = 0x10;
  int86(0x2d, &regs, &regs);
  prevend = regs.x.cx - 1;
  movedata(regs.x.dx, regs.x.ax, _DS, (unsigned)mtdirs, MAXPATH);
  movedata(regs.x.dx, regs.x.bx, _DS, (unsigned)prevbuf, 2*MAXDIR+1);

  harderr(handler);			// Ignore an unready drive

  olddir[0] = getdisk() + 'A';		// Retrieve the current directory
  olddir[1] = ':';                      // getcwd & _getdcwd do the same thing
  olddir[2] = '\\';			//  as these four statements, but they
  getcurdir(0, olddir+3);		//  have more associated rigmarole

  prevnum = 0;
  char* prevcnt = prevbuf;
  while (*prevcnt)			// Find the previous dirs
  {
    if (strcmp(olddir, prevcnt)) prev[prevnum++] = prevcnt;
    prevcnt = strchr(prevcnt, 0) + 1;
  }

  if (argc == 1)			// No parameters
  {
    cout << "\n Current directory = " << longname(olddir);
    if (prevnum)
    {
      cout << "\nPrevious directory = " << longname(prev[0]);
      for (j = 1; j < prevnum; ++j)
	cout << endl << setw(18) << (j+1) << " = " << longname(prev[j]);
    }
    cout << "\n"
	    "\nDirectory structure file = " << longname(mtdirs)
	 << endl;
    return EXIT_SUCCESS;
  }

  // On the command line, BC 3.1 treats \" as ". Let's assume if a string
  // ends with a quote, it should actually end with a backslash.
  for (j = argc - 1; j >= 1; --j)
  {
    int last = strlen(argv[j]) - 1;
    if (argv[j][last] == '\"') argv[j][last] = '\\';
  }

  char*& dir = argv[1]; 		// Easier to type

  if (*dir == '?' || dir[1] == '?')     // First or second character help
  {                                     //  (switch char of your choice)
    help();
    return EXIT_SUCCESS;
  }

  int moved = 0,			// Flag to stop processing cmd line
      stack = 0,			// 1 for pushing, 2 for popping
      exit_code = EXIT_SUCCESS;

  while (!moved)
  {
    if (*dir == '#')                    // A different directory structure file
    {
      int permanent = 0;
      if (dir[1] == '#')                // A permament change
      {
	++dir;
	++permanent;
      }
      if (dir[1] == 0) modext("dat");
      else	       modext(dir+1);
      if (permanent)
	movedata(_DS, (unsigned)mtdirs, regs.x.dx, regs.x.ax, strlen(mtdirs)+1);
      if (--argc == 1) return EXIT_SUCCESS;
      argv++; dir = argv[1];
    }

    switch (*dir)
    {
      case '@':				// (Re)construct directory structure
	dirfile(dir+1);
	break;

      case '+': 			// Add a directory (tree) to the file
	moved = (dir[1] == '+');        //  and create it if it doesn't exist
	if (moved) dir++;		// Skip the second plus
	if (!adddir(dir+1))
	{
	  cout << "Unable to create \"" << newdir << "\".\n";
	  exit_code = EXIT_NOCREAT;
	}
	break;

      case '-':				// Remove a directory (tree) from the
	if (dir[1])			//  file and possibly delete it
	  if (!remdir(dir+1))
	  {
	    cout << "Unable to remove \"" << newdir << "\".\n";
	    exit_code = EXIT_NOREMOV;
	  }
	break;

      case ';':				// Use a previous directory
	if (isdigit(dir[1]))		// A number was specified
	{
	  if ((j = atoi(dir+1)) == 0) j = INT_MAX;
	}
	else for (j = 1; dir[j] == ';'; ++j);
	strcpy(newdir, (--j < prevnum) ? prev[j] : olddir);
	moved = 1;
	break;

      case '!':
	if (dir[1] == 0)                // Using the stack
	{
	  if (--argc == 1)		// No more arguments, so popping
	  {
	    if (!popdir(regs.x.dx, regs.x.cx)) return EXIT_STKEMPT;
	    stack = 2;
	    moved = 1;
	    break;
	  }
	  else
	  {
	    stack = 1;
	    argv++; dir = argv[1];	// Point to the directory
	  }
	}
	// Fall through to select the directory.

      default:
	moved = 1;
	char* endp;
	int which = (int)strtol(dir, &endp, 10);
	if (*endp == ',') argv[1] = endp+1;
	else
	{
	  which = 1;
	  if (argc == 2) if (find(dir))
	  {
	    moved = 2;			// Find the full path for prev. dirs.
	    break;
	  }
	}
	int exact = 0;
	if (*argv[argc-1] == '/' || *argv[argc-1] == '\\')
	{
	  argc--;
	  exact = 1;
	}
	int result = partdir(argc-1, argv+1, which);
	if (result)
	{
	  if (result == 1) cout << "No match found.\n";
	  else cout << char(result) << ": has not been scanned.\n";
	  return EXIT_UNFOUND;
	}
	if (which == 0) return EXIT_SUCCESS;
	if (exact)
	{
	  strcat(newdir, strupr(argv[argc]));
	  find(newdir);
	}
    }
    if (!moved) 			// No movement, so check next parameter
    {
      if (--argc == 1) return exit_code;// No more parameters, exit
      argv++; dir = argv[1];		// Point to the next
    }
  }

  int result = setdrive(*newdir);
  if (result)
  {
    cout << *newdir << ": is " << ((result == 1) ? "unavailable.\n" :
						   "an invalid drive.\n");
    return EXIT_UNFOUND;
  }
  result = (isLFN(*newdir)) ?
	   LFNchdir((LFNgetname(LONGNAME, newdir, newdir, 1), newdir)) :
	   chdir(newdir);
  if (result == -1)
  {
    cout << '\"' << longname(newdir) << "\" is an invalid path.\n";
    setdrive(*olddir);
    return EXIT_UNFOUND;
  }
  if (moved == 2 || isLFN(*newdir))	// Get the full path for below
  {
    newdir[2] = '\\';
    getcurdir(0, newdir+3);
  }

  if (stack)
  {
    if (stack == 1)
    {
      char far* pointer = (char far*)MK_FP(regs.x.dx, regs.x.cx);
      while (*pointer)
      {
	pointer = _fstrchr(pointer, 0) + 1;
      }
      int len = strlen(olddir)+1;
      if (len + FP_OFF(pointer) >= regs.x.cx + 2*MAXDIR+1) return EXIT_STKFULL;
      _fstrcpy(pointer, olddir);
      *(pointer + len) = 0;
    }
  }
  else if (strcmp(newdir, olddir))
  {
    int len = strlen(olddir)+1;
    movedata(_DS, (unsigned)olddir, regs.x.dx, regs.x.bx, len);
    regs.x.bx += len;
    for (j = 0; j < prevnum; ++j)
    {
      if (strcmp(newdir, prev[j]))
      {
	len = strlen(prev[j])+1;
	if ((regs.x.bx + len) > prevend) break;
	movedata(_DS, (unsigned)prev[j], regs.x.dx, regs.x.bx, len);
	regs.x.bx += len;
      }
    }
    // pokeb doesn't work right - doesn't seem to like regs.x.dx
    *((char far*)MK_FP(regs.x.dx, regs.x.bx)) = 0;
  }

  return EXIT_SUCCESS;
}


/* ------------------------------   setdrive   --------------------------------

  Set the active drive to "drive" (which is assumed to be capital).
  Return:
    0 for success;
    1 for unavailable drive (eg. no disk present);
    2 for invalid drive (eg. LASTDRIVE exceeded).

*/

int setdrive(char drive)
{
  drive -= 'A';
  setdisk(drive);
  if (getdisk() != drive) return 2;
  hardware_error = 0;
  chdir(".");                           // Simple test for disk present
  if (hardware_error)
  {
    setdisk(*olddir - 'A');		// Restore the old one if not
    return 1;
  }
  return 0;
}


/* -------------------------------   popdir   ---------------------------------

  Set newdir to the last entry in the stack and remove it from the stack.
  Return 1 for success, 0 if stack is empty.

*/

int popdir(unsigned segment, unsigned offset)
{
  char far* pointer = (char far*)MK_FP(segment, offset);
  char far* dir;        		// Points to the directory to be popped

  if (*pointer == 0) return 0;

  do
  {
    dir     = pointer;
    pointer = _fstrchr(pointer, 0) + 1;
  }
  while (*pointer);

  _fstrcpy(newdir, dir);		// Copy the directory
  *dir = 0;				//  and remove it from the stack

  return 1;
}


/* -------------------------------   expand   ---------------------------------

  Copy path to newdir, adding the current drive if no drive is specified,
  replacing '/' with '\', expanding the dots at the start and removing the
  trailing backslash. Convert a long name to short.

*/

void expand(char* path)
{
  int j;

  // Remove the ending (back)slash, but only if it's not the root
  j = strlen(path) - 1;
  if ((path[j] == '/' || path[j] == '\\') && j > 0 && path[j-1] != ':')
    path[j--] = 0;

  if (isLFN((path[1] == ':') ? *path : *olddir))
  {
    if (LFNgetname(SHORTNAME, path, newdir, 1) == 0)
      return;
  }

  // Replace all slashes with backslashes
  for (; j >= 0; --j) if (path[j] == '/') path[j] = '\\';

  if (path[1] == ':')                   // A drive has been specified
  {
    *newdir = toupper(*path);           // Make sure it's uppercase
    path += 2;				// Point past it
  }
  else *newdir = *olddir;		// Use current drive
  newdir[1] = ':';

  if (*path == 0)			// Only a drive has been specified
  {					//  so select the current directory
    newdir[2] = '.';                    //  on the new drive, as chdir("C:")
    newdir[3] = 0;			//  (for example) will not work
    return;
  }

  j = 2;
  for (int dots = 1; *path == '.'; ++dots, ++path)
  {
    if (dots > 2)			// Expand the dots by adding "\."
    {					//  at the appropriate positions
      newdir[j++] = '\\';               //  (Eg. for "..." the first two are
      newdir[j++] = '.';                //  copied, then "\." is added, then
    }					//  the third dot is copied, yielding
    newdir[j++] = '.';                  //  "..\..")
  }

  strcpy(newdir+j, path);		// Copy the rest of the path
}


/* --------------------------------   find   ----------------------------------

  See if path exists. It may end in a "*", in which case the first directory
  that matches will be selected. If the path does not exist, but it was
  expected to (parent shortcut, ending in star, or containing multiple paths)
  then there is no need for a search, so say it exists anyway.
  Returns 1 if path exists, 0 if not.

  Note: Extensions are not matched, so *.* would be required for directories
	that have extensions. Eg. if the directory is "director.y", using
	"d*" would fail; "d*.*" is needed.
*/

int find(char* path)
{
  struct ffblk dir;
  int star = (path[strlen(path)-1] == '*');	// End in star?

  expand(path);
  if (finddirec(dir, newdir))		// The path doesn't exist
    return (newdir[2] == '.' || star || strchr(newdir, '\\'));

  if (star)				// Expand the name found
  {
    for (int j = strlen(newdir)-2; newdir[j] != '\\' && newdir[j] != ':'; --j);
    strcpy(newdir+j+1, dir.ff_name);	// Replace with the actual name
  }
  return 1;
}


/* ------------------------------   partdir   ---------------------------------

  From the partial names try and find the which'th match. If the first name
  contains a drive specification then only that drive will be searched. If
  which is 0 then a numbered list of all matches will be displayed.
  Return 0 for success, 1 for no match, or the drive letter if that drive
  has not been scanned.

*/

int partdir(int num, char* partial[], int which)
{
  ifstream mt; open(mt);

  char drive, drv,			// Drive to search, current drive
       let,				// First letter to match
       path[MAXDIR],			// A possible path
      *dir;				// Subdirectory to match
  long index, next = 0;			// File positions
  int found,				// Pretty much self-explanatory
     *partlen = new int[num];		// Lengths of the partial names

  for (int j = 0; j < num; ++j)
    partlen[j] = strlen(strupr(partial[j]));

  if (partial[0][1] == ':')             // A drive has been specified
  {					//  so search only its directories
    drive = *partial[0];
    partial[0] += 2;			// Skip past it to the first name
    partlen[0] -= 2;
  }
  else drive = 0;			// No drive, so search all of them

  do					// For each drive required to search
  {
    mt.seekg(next);			// Point to the drive
    mt.get(drv);			// Get this drive letter
    rlong(mt, next);			// Pointer to next drive
    if (drive)				// Find the drive we want
    {
      while (next && drv != drive)
      {
	mt.seekg(next);
	mt.get(drv);
	rlong(mt, next);
      }
      if (drv != drive) return drive;	// Drive not in file
    }
    if (isalpha(let = *partial[num-1])) // First name start with a letter?
    {
      mt.seekg((let - 'A') * sizeof(long), ios::cur);
      rlong(mt, index);			// Then find the appropriate position
      if (index == 0) continue;		// No directories start with this letter
      mt.seekg(index);			// Starting position
      mt.getline(path, MAXDIR); 	// Read the first path, because of else
    }
    else
    {
      mt.seekg(26 * sizeof(long), ios::cur); // Start straight after indices
      // Search for the first path starting with let
      do
	mt.getline(path, MAXDIR);
      while (let > *path && !isalpha(*path));
    }

    // Skip all the smaller paths
    while (let == *path && num > subs(path))
      mt.getline(path, MAXDIR);

    while ((let == *path && let != 0) || (let == 0 && *path != 0))
    {
      found = 0;			// Number of matches found
      dir   = path;			// First subdirectory
      for (int j = num-1; j >= 0 &&
			  !strncmp(partial[j], dir, partlen[j]); --j)
      {
	found++;
	dir = strchr(dir, '/');		// Find the next subdirectory
	if (!dir) break;	 	// The path has reached the root
	++dir;				// Point past the slash
      }
      if (found == num && --which <= 0) // A successful match
      {
	reverse(drv, path);		// Put the full path into newdir
	if (which < 0) cout << setw(2) << -which << " = "
			    << longname(newdir) << endl;
	else
	{
	  if (strcmp(newdir, olddir))	// Only successful if not already here
	  {
	    next = 0;			// To terminate the do loop
	    break;			// To terminate the while loop
	  }
	  else ++which;
	}
      }
      mt.getline(path, MAXDIR);         // Try another match
    }
  }
  while (!drive && next);		// Drive not specified and more exist

  delete []partlen;

  return (which > 0);
}


/* ------------------------------   reverse   ---------------------------------

  Reverse path into newdir. If drive is null then reverse a DOS path (\) into
  mt's path (/), else add drive and reverse mt into DOS.

*/

void reverse(char drive, const char* path)
{
  char *beg,				// Where to place the current subdir.
       sep1, sep2;			// The separators
  int j, end = strlen(path)-1;		// Subdir's start and end

  if (drive)
  {
    newdir[0] = drive;
    newdir[1] = ':';
    newdir[2] = '\\';
    beg = newdir+3;
    sep1 = '/'; sep2 = '\\';
  }
  else
  {
    beg = newdir;
    sep1 = '\\'; sep2 = '/';
  }

  for (j = end-1; j > 0; --j)		// Search backwards for a separator
  {
    if (path[j] == sep1)		// Found one, so from here to the
    {					//  last one is the subdir
      memcpy(beg, path+j+1, end -= j);
      beg += end;
      *(beg++) = sep2;
      end = --j;			// Ready for the next
    }
  }
  memcpy(beg, path, ++end);		// Final subdirectory
  *(beg+end) = 0;
}


/* -------------------------------   adddir   ---------------------------------

  Add a directory (tree) to the file and create it if necessary. Unlike md
  this can create two directories in one hit. eg. "mt +this/that" is
  equivalent to "md this" "md this\that". If path is an empty string then add
  the current directory. Only new directories will be added to the file;
  existing directories are expected to already be in it.
  Returns 1 for success or 0 for unable to create.

*/

int adddir(char* path)
{
  char *sep = newdir+2,			// Current directory to create
       *start = NULL,			// Start of the tree
	orig[MAXDIR],			// Current path on another drive
       *ex;				// To remember newdir
  int success = 1;

  typedef int (*dirfunc)(const char* path);
  dirfunc cd, md;

  expand(path);

  if (isLFN(*newdir))
  {
    cd = LFNchdir;
    md = LFNmkdir;
  }
  else
  {
    cd = chdir;
    md = mkdir;
  }

  if (*newdir != *olddir)		// It's on a different drive
  {					//  so change to it and
    if (setdrive(*newdir)) return 0;	//  remember its current directory
    *orig = '\\';
    getcurdir(0, orig+1);
  }

  if (cd(newdir) == -1) 		// It doesn't already exist
  {
    do
      sep = strchr(sep+1, '\\');        // Skip past the dot directories
    while (*(sep-1) == '.');
    while (sep) 			// Branch down the tree
    {
      *sep = 0;
      if (md(newdir) == 0)
      {
	if (!start) start = strdup(newdir);
      }
      else if (errno == ENOENT) 	// Bad name
      {
	if (start)			// Some directories were made
	{
	  success = 0;
	  break;
	}
	else
	{
	  setdrive(*olddir);
	  return 0;
	}
      }
      *sep = '\\';
      sep = strchr(sep+1, '\\');
    }
    if (md(newdir) == -1)
    {
      if (!start)
      {
	setdrive(*olddir);
	return 0;
      }
      success = 0;
    }
    if (isLFN(*newdir)) LFNgetname(SHORTNAME, newdir, newdir, 1);
    if (start)
    {
      cd(start);
      free(start);
    }
    else cd(newdir);
  }
  ex = strdup(newdir);			// newdir corrupted by update
  update(0);				// Add the directory/ies to the file
  strcpy(newdir, ex);
  free(ex);
  if (*newdir != *olddir)
  {
    chdir(orig);
    setdrive(*olddir);
  }
  else chdir(olddir);
  return success;
}


/* -------------------------------   remdir   ---------------------------------

  Remove path from the file and delete it from disk if the first character
  is not "-". ie. "mt -path" will delete path and remove it from the file;
  "mt --path" will just remove path from the file. The root directory won't
  be deleted in any case.
  Returns 1 for success or 0 for doesn't exist / isn't a directory.

  971226 - Had to use a more elaborate test for deleting current directory,
	   since Win9x will actually allow the current to be removed.

*/

int remdir(char* path)
{
  char *ex, orig[MAXDIR];		// Expanded path, current dir.
  int del = (*path != '-'),             // Deleting as well as removing
      cur;				// Deleting current directory

  if (!del) ++path;			// Skip the leading minus

  expand(path);
  if (*newdir != *olddir) if (setdrive(*newdir)) return 0;
  *orig = '\\';
  getcurdir(0, orig+1);

  if (chdir(newdir) == -1)		// Change to the directory
  {					//  to be deleted
    setdrive(*olddir);
    return 0;
  }
  ex = strdup(newdir);			// update corrupts newdir
  update(1);				// Remove the directory from the file
  getcurdir(0, newdir);			// Test for root directory
  del = del && *newdir; 		// Only delete if it's not the root
  cur = !strcmp(newdir, orig+1);	// Is it the current directory?
  strcpy(newdir, ex);
  free(ex);
  if (del) deldir();			// Delete the directory from the disk

  chdir(orig);
  if (*newdir != *olddir) setdrive(*olddir);

  // Delete the directory. If it's the current, move to the parent.
  if (del)
  {
    if (cur) chdir("..");
    rmdir(newdir);
  }

  return 1;
}


/* -------------------------------   deldir   ---------------------------------

  Delete ALL files and subdirectories in the current drive/directory.

*/

void deldir()
{
  struct ffblk find;
  static int done;

  done = findfirst("*.*", &find,
		   FA_ARCH | FA_HIDDEN | FA_RDONLY | FA_SYSTEM | FA_DIREC);
  while (!done)
  {
    if (find.ff_attrib & FA_DIREC)
    {
      if (*find.ff_name != '.')         // Delete everything in a subdirectory
      {
	chdir(find.ff_name);
	deldir();
	chdir("..");
	rmdir(find.ff_name);
      }
    }
    else
    {
      if (find.ff_attrib & FA_RDONLY) _dos_setfileattr(find.ff_name, 0);
      unlink(find.ff_name);
    }
    done = findnext(&find);
  }
}


/* -------------------------------   update   ---------------------------------

  Update the directory structure file. Operation is 0 for adding directories,
  1 for removing.

  991211: Display long filename.

*/

void update(int operation)
{
  ifstream mtin; open(mtin);

  char drive = *newdir, drv;
  long size, next = 0, last = 1, remdrive;

  do					// See if the drive is in the file
  {
    remdrive = last;			// For removing a drive
    mtin.seekg(next);
    mtin.get(drv);
    last = mtin.tellg();		// For adding a drive
    rlong(mtin, next);
  }
  while (drv != drive && next);
  if (drv != drive && operation == 1)	// Drive not in file and I'm removing
  {					//  so there's nothing to do
    mtin.close();
    return;
  }

  char* mttmp = strdup(mtdirs); 	// Remember the filename
  modext("$~$");                        //  and create a temporary file
  ofstream mtout; open(mtout);

  size = (drv == drive ? mtin.tellg() - sizeof(long) - sizeof(char) :
			 (mtin.seekg(0, ios::end), mtin.tellg()));
  mtin.seekg(0);			// Copy all the drives before the one
  copy(mtin, mtout, size);		//  that's being changed
  mtin.seekg(sizeof(char) + 27*sizeof(long), ios::cur);

  char path[MAXDIR];
  int  root, num, count, j;
  char* lfntmp = longname(mttmp);

  getcurdir(0, path);			// Check for the root directory
  root = !(*path);			// True if root directory
  if (drv != drive)
    cout << "Adding drive " << drive << " to \"" << lfntmp << "\" : ";
  else if (root)
  {
    if (operation == 0)
      cout << "Updating drive " << drive << " of \"" << lfntmp << "\" : ";
    else
      cout << "Removing drive " << drive << " from \"" << lfntmp << "\".\n";
  }

  if (!root || operation == 0)		// No point in scanning if removing
  {					//  an entire drive
    dirs   = new char*[MaxDirs];
    dirnum = 0;
    finddirs();
    if (root || drv != drive) cout << dirnum << " directories.\n";
    qsort((void*)dirs, dirnum, sizeof(dirs[0]), sort);
  }
  else dirs = NULL;

  if (drv != drive)
  {
    long temp = mtout.tellp();		// Set the old last drive to point
    mtout.seekp(last);			//  to the new drive
    wlong(mtout, temp);
    mtout.seekp(temp);
    last = index(mtout, drive);
  }
  else if (root)
  {
    mtin.seekg(next);			// Skip what was in there
    last = (operation == 0) ? index(mtout, drive) : remdrive;
  }
  else
  {
    num   = dirnum;
    count = 0;
    while (count < num && !isalpha(*dirs[count])) ++count;
    mtin.getline(path, MAXDIR);
    copy(path, 0,     count, mtin, 0);
    copy(path, count, num,   mtin, 1);

    if (operation == 0) 		// Sort new and old
    {
      qsort((void*)dirs, dirnum, sizeof(dirs[0]), sort);
      last = index(mtout, drive);
    }
    else
    {
      dirs   += num;			// Skip the directories to remove
      dirnum -= num;
      last    = index(mtout, drive);	// Index the remainder
      dirs   -= num;
      dirnum += num;
      for (j = num-1; j >= 0; --j) free(dirs[j]);
    }
  }

  if (!next)				// This was the last drive
  {					//  so indicate such (assuming that
    mtout.seekp(last);			//  not all drives have been removed)
    wlong(mtout, zero);
  }
  else					// Copy all the other drives
  {
    // Adjustment for the new indices
    long offset = mtout.tellp() - mtin.tellg();
    for (;;)
    {
      mtin.get(drv);
      mtout.put(drv);
      rlong(mtin, next);
      if (next) next += offset;
      wlong(mtout, next);
      for (j = 0; j < 26; ++j)
      {
	rlong(mtin, last);
	if (last) last += offset;
	wlong(mtout, last);
      }
      if (next) copy(mtin, mtout, next-offset - mtin.tellg());
      else
      {
	last = mtin.tellg();		// The last drive is from here
	mtin.seekg(0, ios::end);	//  to the end of the file, since
	next = mtin.tellg();		//  next is zero.
	mtin.seekg(last);
	copy(mtin, mtout, next - last);
	break;
      }
    }
  }
  delete[] dirs;
  mtin.close();
  mtout.close();
  unlink(mttmp);			// Delete the original file
  rename(mtdirs, mttmp);		//  and replace it with the updated one
  strcpy(mtdirs, mttmp);		// Restore the name
  free(mttmp);
}


/* --------------------------------   copy   ----------------------------------

  Copy size bytes from file is to file os. The copy is done in blocks - the
  maximum memory can handle, up to 32k (INT_MAX).

  991211: use malloc() instead of new;
	  modified the loop to correct problems with files larger than 32k.

*/

void copy(ifstream& is, ofstream& os, long size)
{
  if (size <= 0) return;

  char* buffer;
  int	block = (size > INT_MAX ? INT_MAX : int(size));

  while (!(buffer = (char*)malloc(block))) // Determine the largest block size
    block >>= 1;			// Halve it each time

  do 					// Copy the full-size blocks
  {
    is.read( buffer, block);
    os.write(buffer, block);
    size -= block;
  }
  while (size > block);
  if (size)                             // Copy what's left over
  {
    is.read( buffer, int(size));
    os.write(buffer, int(size));
  }
  free(buffer);
}


/* --------------------------------   copy   ----------------------------------

  Copy directories from start to end-1, skipping duplicates. Stop copying when
  there are no more directories, or we've come across an alphabetical directory
  in the non-alphabetical copy.

*/

void copy(char* path, int start, int end, ifstream& is, int alpha)
{
  if (start < end)
  {
    // Copy all the smaller directories
    while (*path && (alpha || !isalpha(*path)) && *path < *dirs[start])
    {
      dirs[dirnum++] = strdup(path);
      is.getline(path, MAXDIR);
    }
    // Check these directories for duplicates
    while (*path && (alpha || !isalpha(*path)) && *path <= *dirs[end-1])
    {
      for (int j = start; j < end; ++j)
      {
	if (*path < *dirs[j]) j = end-1;	// It can't be in the list
	else if (!strcmp(path, dirs[j]))
	{
	  start = j+1;			// If it matches then
	  break;			//  skip all those before it
	}
      }
      if (j == end) dirs[dirnum++] = strdup(path);	// Wasn't found
      is.getline(path, MAXDIR);
    }
  }
  // Copy all the rest
  while (*path && (alpha || !isalpha(*path)))
  {
    dirs[dirnum++] = strdup(path);
    is.getline(path, MAXDIR);
  }
}


/* ------------------------------   dirfile   ---------------------------------

  Create the directory structure (see finddirs, index and sort for details).
  If drives is an empty string then scan the drives already in the file,
  otherwise scan those drives specified (invalid drives will be ignored).

*/

void dirfile(char* drives)
{
  char drv[26], orig[MAXDIR];		// Drives to scan, original dir.
  int  n = strlen(strupr(drives));	// Number of drives to scan
  long last = 0;			// Last drive position

  if (n)				// Specified drives
  {
    strcpy(drv, drives);
    cout << "Creating";
  }
  else					// Drives already there
  {
    ifstream mtin; open(mtin);
    do
    {
      mtin.seekg(last);
      mtin.get(drv[n++]);
      rlong(mtin, last);
    }
    while (last);
    mtin.close();
    cout << "Updating";
  }
  cout << " \"" << longname(mtdirs) << "\".\n";

  ofstream mt; open(mt);

  dirs = new char*[MaxDirs];		// Create the array of directories
  orig[0] = '\\';
  for (int j = 0; j < n; ++j)
  {
    cout << "Drive " << drv[j] << ' ';
    if (setdrive(drv[j]))
    {
      cout << "will be ignored.\n";
      continue;
    }
    getcurdir(0, orig+1);
    chdir("\\");
    dirnum = 0;
    finddirs();				// Get all the directories
    chdir(orig);
    cout << "has " << dirnum << " directories.\n";
    qsort((void*)dirs, dirnum, sizeof(dirs[0]), sort);
    last = index(mt, drv[j]);
  }
  mt.seekp(last);			// Write zero for last drive
  wlong(mt, zero);
  mt.close();
  delete[] dirs;

  setdrive(*olddir);			// Restore current drive
}


/* ------------------------------   finddirs   --------------------------------

  Starting from the current directory recursively find all directories for
  the current drive and store them as a reverse path separated by slashes.
  eg: "\LANGUAGE\BC" will be stored as "BC/LANGUAGE".
  Global variables dirs holds the directories; dirnum has the number found.

*/

void finddirs()
{
  struct ffblk dir;
  static int   done;
  static char  path[MAXDIR];

  getcurdir(0, path);
  if (*path)				// Don't store the root directory
  {					//  (it's a null string)
    reverse(0, path);
    dirs[dirnum++] = strdup(newdir);
  }
  for (done = finddirec(dir, "*.*"); !done; done = finddirec(dir))
  {
    chdir(dir.ff_name);
    finddirs();  			// Get any subdirectories
    chdir("..");			// Back to this one
  }
}


/* -----------------------------   finddirec   --------------------------------

  Find a directory that matches name, ignoring the . and .. directories.
  If name is not given (or is NULL), continue the find.
  Returns the same as findfirst/findnext.

*/

int finddirec(struct ffblk& dir, const char* name)
{
  int found = (name) ? findfirst(name, &dir, FA_DIREC | FA_HIDDEN | FA_SYSTEM):
		       findnext(&dir);

  while (found == 0 && (!(dir.ff_attrib & FA_DIREC) || *dir.ff_name == '.'))
    found = findnext(&dir);

  return found;
}


/* -------------------------------   index   ----------------------------------

  Write and index the directories for drive. First write the drive letter and
  an index to the next drive. Following this are 26 indices for each letter of
  the alphabet. An index of zero means no directories start with that letter.
  Then the directories, one per line, and a blank line to finish.
  It returns the position of the next drive index so zero can be written to
  indicate no more drives. However, os is left at the end of the file.

*/

long index(ofstream& os, char drive)
{
  char let;				// Current index letter
  long index, din,			// Disk indices
       lin[26] = { 0 }; 		// Letter indices

  os.put(drive);			// Write the drive letter
  din = os.tellp();			// The next drive index position
  os.seekp(27 * sizeof(long), ios::cur);// Skip it and the letters' indices

  // Write and free the non-alphabetical directories.
  for (int j = 0; j < dirnum && !isalpha(*dirs[j]); ++j)
  {
    os << dirs[j] << endl;
    free(dirs[j]);
  }
  // Now write and free the alphabetical directories, remembering the index
  // position of each letter.
  let = '@';
  for (; j < dirnum; ++j)
  {
    if (let != *dirs[j])
    {
      let = *dirs[j];
      lin[let - 'A'] = os.tellp();
    }
    os << dirs[j] << endl;
    free(dirs[j]);
  }
  os << endl;				// No more directories
  index = os.tellp();                   // Store the position for next drive
  os.seekp(din);
  wlong(os, index);
  for (j = 0; j < 26; ++j) wlong(os, lin[j]);
  os.seekp(0, ios::end);		// Ready for next drive
  return din;				// For writing 0 for no more drives
}


/* --------------------------------   sort   ----------------------------------

  This determines how the directories will be searched. Smaller paths are
  placed before larger paths, non-alphabetical entries are before alphabetical
  directories, and the directories are sorted alphabetically.

*/

int sort(const void* a, const void* b)
{
  char *dir1 = *(char**)a,              // qsort thinks I am sorting an array
       *dir2 = *(char**)b;		//  of pointers - this gets the string

  if (*dir1 == *dir2)			// Both start with the same character
  {					//  so sort by path length
    int len1 = subs(dir1),
	len2 = subs(dir2);
	 if (len1 < len2) return -1;	// Same size path length will be
    else if (len1 > len2) return  1;	//  sorted by name below
  }
  int let1 = (isalpha(*dir1) != 0),	// First directory is alphabetic
      let2 = (isalpha(*dir2) != 0);	// Second directory is alphabetic
  if (!(let1 ^ let2))			// Both are/are not letters
    return strcmp(dir1, dir2);		//  so sort by name

  return (let1) ? 1 : -1;		// Letters placed after non-letters
}

// Determine the number of directories in a path for the sort routine.
int subs(const char* path)
{
  int n = 0;                           	// Number of slashes
  for (; *path; ++path) if (*path == '/') ++n;
  return n+1;				// Number of paths
}


/* --------------------------------   help   ----------------------------------

  Display the help screen.

*/

void help()
{
  cout <<				// One really long string
"\n"
"mt - Move To drive/directory. mtmem - resident memory for mt.\n"
"Copyright 1996-9 Jason Hood <jadoxa@hotmail.com>. Freeware. Version "version"."
"\n\n"
"mt           Display the current status.\n"
"mt #[#]ext   Use file \"mtdirs.ext\" for the directory structure\n"
"              [and make it permanent].\n"
"mt @dc       Create directory structure for drives D: and C:.\n"
"mt @         Update directory structure (rescan drives).\n"
"mt ....      Equivalent to \"mt ..\\..\\..\".\n"
"mt;[;...|n]  Move to a previous directory (n = 0 is current directory).\n"
"mt wg*       Move to first subdirectory starting with \"wg\".\n"
"mt w/g       Move to subdirectory \"w\" subsubdirectory \"g\".\n"
"mt wg        Move to subdirectory \"wg\". If that fails search\n"
"              for a directory that starts with \"wg\".\n"
"mt w g       Search for a directory starting with \"w\" that\n"
"              has a subdirectory starting with \"g\".\n"
"mt w g /h    As above, then move to subsubdirectory \"h\".\n"
"mt n,partial Select the nth match (or n+1 if already in nth).\n"
"mt +         Add the current directory and its subdirs to the file.\n"
"mt +[+]path  Create path if necessary, add it to the file [and move to it].\n"
"mt -[-]path  Remove path from the file [but don't delete it from disk].\n"
"mt ! path    Move to path, preserving the current directory on the stack.\n"
"mt !         Move to the last preserved directory.\n"
"\n"
"Searches always begin from the root directory.\n";
}


// ---------------------------	 LFN Functions	 ------------------------------


/* -------------------------------   isLFN   ----------------------------------

  Return 1 if the drive supports long filenames, otherwise 0. Assumes drive
  is an uppercase letter.

*/

int isLFN(char drive)
{
  static char drv[] = "?:\\";           // Drive to test
  static int  drives[26] =		// Drives already tested.
  {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
  };

  union REGS regs;
  char buffer[33];
  int  letter = drive - 'A';

  if (drives[letter] == -1)
  {
    *drv      = drive;
    regs.x.ax = 0x71a0;
    regs.x.dx = (unsigned)drv;		// Assumes DS
    regs.x.di = (unsigned)buffer;	// Assumes ES
    regs.x.cx = sizeof(buffer);
    _ES = _DS;                          // ES gets corrupted
    regs.x.cflag = 1;
    intdos(&regs, &regs);
    drives[letter] = (regs.x.cflag == 0 && (regs.x.bx & 0x4000));
  }
  return drives[letter];
}


/* ------------------------------   LFNerror   --------------------------------

  Set errno after an LFN function. If LFN is not supported, use EINVFNC, else
  just copy _doserrno. Return 0 if the function was successful, -1 otherwise.

*/

int LFNerror(const union REGS& regs)
{
  if (regs.x.cflag == 0) return 0;

  errno = (regs.x.ax == 0x7100) ? EINVFNC : _doserrno;
  return -1;
}


/* ------------------------------   LFNchdir   --------------------------------

  Change directory, using a long name.

*/

int LFNchdir(const char* path)
{
  union REGS regs;

  regs.x.ax    = 0x713b;
  regs.x.dx    = (unsigned)path;	// Assumes DS
  regs.x.cflag = 1;	// Set carry, according to Ralf Brown's Int. List
  intdos(&regs, &regs);
  return LFNerror(regs);
}


/* ------------------------------   LFNmkdir   --------------------------------

  Make a directory, using a long name.

*/

int LFNmkdir(const char* path)
{
  union REGS regs;

  regs.x.ax    = 0x7139;
  regs.x.dx    = (unsigned)path;	// Assumes DS
  regs.x.cflag = 1;	// Set carry, according to Ralf Brown's Int. List
  intdos(&regs, &regs);
  return LFNerror(regs);
}


/* -----------------------------   LFNgetname	-------------------------------

  Get the truename, short name, or long name for a path. The buffer should
  be big enough to hold the name (260 bytes for a long name, 80 for short).
  If subst is 0, the full path will be used, otherwise the subst'ed drive.
  name and buffer can be the same.

*/

int LFNgetname(int func, const char* name, char* buffer, int subst)
{
  union REGS regs;

  regs.x.ax = 0x7160;
  regs.h.cl = func;
  regs.h.ch = (subst) ? 0 : 0x80;
  regs.x.si = (unsigned)name;		// Assumes DS
  regs.x.di = (unsigned)buffer;		// Assumes ES
  _ES = _DS;                            // ES gets corrupted
  regs.x.cflag = 1;	// Set carry, according to Ralf Brown's Int. List
  intdos(&regs, &regs);
  return LFNerror(regs);
}


/* ------------------------------   longname   --------------------------------

  If LFN is available, set newdir to the long name for the given name and
  return it, otherwise just return the given name. shortname is expected to
  start with an uppercase drive letter.

  980630: convert backslashes to slashes and use lowercase letters if not
	  LFN (the drive letter is always lowercased, however).

*/

char* longname(char* shortname)
{
  char* dir;
  int	j;

  dir = (isLFN(*shortname) && LFNgetname(LONGNAME, shortname, newdir, 1) == 0)
      ? newdir : strlwr(shortname);

  *dir = tolower(*dir);
  for (j = strlen(dir) - 1; j > 1; --j)
    if (dir[j] == '\\') dir[j] = '/';

  return dir;
}
