Unit SortAlgs;
(* SortDemo Version 2.4: 19 Mar 1995                                        *)

(* Copyright (C) 1995 Tapirsoft, Harald Selke                               *)
(* Based on a programme by K.L. Noell.                                      *)
(* Implementation of the sorting algorithms for the SortDemo package        *)
(* See SortDemo.Doc for full documentation.                                 *)
(*                                                                          *)
(* This programme is free software; you can redistribute it and/or modify   *)
(* it under the terms of the GNU General Public License (version 1) as      *)
(* published by the Free Software Foundation.                               *)
(*                                                                          *)
(* This programme is distributed in the hope that it will be useful,        *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of           *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *)
(* GNU General Public License for more details.                             *)
(*                                                                          *)
(* You should have received a copy of the GNU General Public License        *)
(* along with this programme; if not, write to the Free Software            *)
(* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                *)

{$A+}                        (* Word alignment on                   *)
{$B-}                        (* Boolean complete evaluation off     *)
{$D+}                        (* Debug information on                *)
{$E+}                        (* Emulate coprocessor if necessary    *)
{$F-}                        (* Don't force far calls               *)
{$G-}                        (* Don't generate 80286 code           *)
{$I+}                        (* I/O checking on                     *)
{$L+}                        (* Generate local symbol information   *)
{$N-}                        (* No numeric processing               *)
{$O-}                        (* No overlaying                       *)
{$R+}                        (* Range checking on                   *)
{$S+}                        (* Stack checking on                   *)
{$V-}                        (* No var-string checking              *)
{$X-}                        (* No extended syntax allowed          *)

Interface

Const ESC   = #27;                                    (* ASCII for <ESC>    *)
      CR    = #13;                                    (* ASCII for <Return> *)
      SPACE = #32;                                    (* ASCII for <Space>  *)

Type  Mode = (fast, slow, manual);

Var   DelayTime : Integer;
      speed : Mode;
      comps, swaps : LongInt;

(* If speed is slow, the programme waits for DelayTime milliseconds between *)
(* any two compare operations. If speed is manual, the sorting algorithms   *)
(* wait for a <Return> after each of the outer loops, and also before star- *)
(* ting. Otherwise the algorithms run until ready.                          *)
(* Comps and swaps count the number of comparisons and swaps needed.        *)
(* In the following procedures n is the number of elements to be sorted.    *)

Procedure BubbleSort (n : Integer);
(* Implementation by K.L. Noell, modified by Harald Selke.                  *)

Procedure ShakerSort (n : Integer);
(* Implementation by K.L. Noell, modified by Harald Selke.                  *)

Procedure SelectSort (n : Integer);
(* Implementation by Harald Selke                                           *)

Procedure InsertSort (n : Integer);
(* Implementation by K.L. Noell, modified by Harald Selke.                  *)

Procedure ShellSort (n : Integer);
(* Implementation by K.L. Noell, modified by Harald Selke.                  *)

Procedure QuickSort (n : Integer);
(* Implementation by Harald Selke.                                          *)

Procedure HeapSort (n : Integer);
(* Implementation by Harald Selke.                                          *)

Procedure MergeSort (n : Integer);
(* Implementation by Harald Selke.                                          *)


Implementation
Uses Crt, Graph, SortInps;

Function StopIt : Boolean;
Var c : Char;
Begin                                                             (* StopIt *)
  If KeyPressed Or (speed = manual) Then
  Begin
    c := ReadKey;
    Case c Of
      CR, SPACE: If speed <> manual Then c := ReadKey;
      '+': If DelayTime > 1 Then DelayTime := DelayTime Shr 1;
      '-': If DelayTime < MaxInt Shl 1 Then DelayTime := DelayTime Shl 1;
    End
  End;
  StopIt := (c = ESC)
End;   (* StopIt *)

Procedure Swap (i, j : Integer);
(* Swaps the contents of A [i] and A [j] and also the corresponding pixels. *)
Var temp : Integer;
Begin                                                               (* Swap *)
  EraseDot (i, A [i]);
  EraseDot (j, A [j]);
  temp := A [i]; A [i] := A [j]; A [j] := temp;
  DrawDot (i, A [i]);
  DrawDot (j, A [j]);
End;   (* Swap *)


Procedure BubbleSort;
Var i, iend : Integer;
    swapped : Boolean;
Begin                                                         (* BubbleSort *)
  comps := 0; swaps := 0;
  iend := n;
  Repeat
    If StopIt Then Exit;
    swapped := False;
    Dec (iend);
    For i := 1 To iend Do
    Begin
      If speed = slow Then Delay (DelayTime);
      Inc (comps);
      If A [i] > A [i+1] Then
      Begin
        Swap (i, i+1); Inc (swaps);
        swapped := True
      End
    End
  Until swapped = False
End;   (* BubbleSort *)

Procedure ShakerSort;
Var i, j, r, l : Integer;
Begin                                                         (* ShakerSort *)
  comps := 0; swaps := 0;
  l := 2;
  r := n;
  i := n-1;
  Repeat
    If StopIt Then Exit;
    For j := r DownTo l Do                  (* shake up *)
    Begin
      If speed = slow Then Delay (DelayTime);
      Inc (comps);
      If A [j-1] > A [j] Then
      Begin
        Swap (j, j-1); Inc (swaps);
        i := j
      End;
    End;
    l := i + 1;
    For j := l To r Do                      (* shake down *)
    Begin
      If speed = slow Then Delay (DelayTime);
      Inc (comps);
      If A [j-1] > A [j] Then
      Begin
        Swap (j, j-1); Inc (swaps);
        i := j
      End;
    End;
    r := i - 1;
  Until l > r;
End;   (* ShakerSort *)

Procedure SelectSort;
Var i, j, min : Integer;
Begin                                                         (* SelectSort *)
  comps := 0; swaps := 0;
  For i := 1 To n-1 Do
  Begin
    If StopIt Then Exit;
    min := i;
    For j := i+1 To n Do
    Begin
      If speed = slow Then Delay (DelayTime);
      Inc (comps);
      If A [j] < A [min] Then min := j
    End;
    Swap (i, min); Inc (swaps)
  End
End;   (* SelectSort *)

Procedure InsertSort;
Var i, j, h : Integer;
    finis : Boolean;
Begin                                                         (* InsertSort *)
  comps := 0; swaps := 0;
  For i := 2 To n Do
  Begin
    If StopIt Then Exit;
    finis := False;
    h := A [i];
    j := i;
    While Not finis Do
    Begin
      If speed = slow Then Delay (DelayTime);
      Inc (comps);
      If A [j-1] > h Then
      Begin
        Inc (swaps);
        EraseDot (j, A [j]);
        A [j] := A [j-1];
        DrawDot (j, A [j]);
        Dec (j)
      End
      Else finis := True;
    End;
    Inc (swaps);
    EraseDot (j, A [j]);
    A [j] := h;
    DrawDot (j, A [j]);
  End
End;   (* InsertSort *)

Procedure ShellSort;
Var i, j, incr, h : Integer;
    finis : Boolean;
Begin                                                          (* ShellSort *)
  comps := 0; swaps := 0;
  incr := 1;
  Repeat incr := 3 * incr + 1 Until incr >= n;
  Repeat
    If StopIt Then Exit;
    incr := incr Div 3;
    For i := incr + 1 To n Do
    Begin
      finis := False;
      h := A [i];
      j := i;
      While Not finis And (j > incr) Do
      Begin
        If speed = slow Then Delay (DelayTime);
        Inc (comps);
        If A [j-incr] > h Then
        Begin
          Inc (swaps);
          EraseDot (j, A [j]);
          A [j] := A [j-incr];
          DrawDot (j, A [j]);
          j := j - incr;
        End
        Else finis := True;
      End;
      Inc (swaps);
      EraseDot (j, A [j]);
      A [j] := h;
      DrawDot (j, A [j])
    End
  Until incr = 1
End;   (* ShellSort *)

Procedure QuickSort;
Var stopped : Boolean;

Procedure Sort (l, r : Integer);
(* Sorts elements A [l], ..., A [r] *)
Var pivot, pivotindex, i, j : Integer;
Begin                                                               (* Sort *)
  If l < r Then
  Begin
    pivotindex := l + (r-l) Div 2;
    pivot := A [pivotindex];
    Swap (pivotindex, r);
    Inc (Swaps);
    i := l-1;
    j := r;
    Repeat
      Repeat
        Inc (i); Inc (comps)
      Until A [i] >= pivot;
      Repeat
        Dec (j); Inc (comps)
      Until A [j] <= pivot;
      If speed = slow Then Delay (DelayTime);
      If i < j Then
      Begin
        Swap (i, j); Inc (swaps)
      End
    Until j <= i;
    Swap (i, r); Inc (swaps);
    stopped := StopIt;
    If stopped Then Exit;
    Sort (l, i-1);
    If stopped Then Exit;
    Sort (i+1, r);
  End
End;   (* Sort *)

Begin                                                          (* QuickSort *)
  comps := 0;
  swaps := 0;
  stopped := False;
  If StopIt Then Exit;
  Sort (1, n)
End;   (* QuickSort *)

Procedure HeapSort;
Var k : Integer;

Procedure DownHeap (r : Integer);
Var i, h : Integer;
    finis : Boolean;
Begin                                                           (* DownHeap *)
  h := A [r];
  EraseDot (r, A [r]);
  finis := False;
  While (r <= n Shr 1) And (Not finis) Do
  Begin
    If speed = slow Then Delay (DelayTime);
    i := r Shl 1;
    If i < n Then If A [i] < A [i+1] Then Inc (i);
    Inc (comps); Inc (comps);
    If h >= A [i] Then finis := True
    Else
    Begin
      A [r] := A [i];
      EraseDot (i, A [i]);
      DrawDot (r, A [r]);
      Inc (swaps);
      r := i
    End
  End;
  A [r] := h;
  DrawDot (r, A [r]);
End;   (* DownHeap *)

Begin                                                           (* HeapSort *)
  comps := 0; swaps := 0;
  If StopIt Then Exit;
  For k := n Shr 1 DownTo 1 Do DownHeap (k);
  Repeat
    If StopIt Then Exit;
    Swap (1, n); Inc (swaps);
    Dec (n);
    DownHeap (1)
  Until n <= 1
End;   (* HeapSort *)

Procedure MergeSort;
Var stopped : Boolean;
    B : Array [1..NMax] Of Integer;          (* MergeSort needs extra space *)

Procedure Sort (l, r : Integer);
(* Sorts elements A [l], ..., A [r] *)
Var i, j, k, m : Integer;
Begin                                                               (* Sort *)
  If l < r Then
  Begin
    m := (r + l) Shr 1;
    Sort (l, m);
    If stopped Then Exit;
    Sort (m+1, r);
    If stopped Then Exit;
    For i := l To r Do B [i] := A [i];
    i := l;
    j := m+1;
    For k := l To r Do
    Begin
      If B [i] < B [j] Then
      Begin
        EraseDot (i, B [i]);
        A [k] := B [i];
        Inc (i);
        DrawDot (k, A [k]);
      End
      Else
      Begin
        EraseDot (j, B [j]);
        A [k] := B [j];
        Inc (j);
        DrawDot (k,  A [k]);
      End;
      If i > m Then Begin i := m; B [i] := MaxInt End;
      If j > r Then Begin j := r; B [j] := MaxInt End;
      Inc (comps);
      Inc (swaps);
      If speed = slow Then Delay (DelayTime)
    End;
    stopped := StopIt
  End
End;   (* Sort *)

Begin                                                          (* MergeSort *)
  comps := 0;
  swaps := 0;
  stopped := False;
  If StopIt Then Exit;
  Sort (1, n)
End;   (* MergeSort *)

End.
