/* ------------------------------------------------------------------------ */
/* CGA2BMP.C (C) CopyLeft Bill Buckels 1999                                 */
/* All Rights Reversed.                                                     */
/*                                                                          */
/* Licence Agreement                                                        */
/* -----------------                                                        */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* If you don't agree, remove this source code from your computer now.      */
/*                                                                          */
/* Written by   : Bill Buckels                                              */
/*                589 Oxford Street                                         */
/*                Winnipeg, Manitoba, Canada R3M 3J2                        */
/*                                                                          */
/* Email: bbuckels@escape.ca                                                */
/* WebSite: http://www.escape.ca/~bbuckels                                  */
/*                                                                          */
/* Purpose      : This utility will allow you to convert CGA 4-color        */
/*                screen dump files of either a BASIC BSAVED IMAGE          */
/*                (.BAS) File or a ZSOFT CGA .PCX to a 320 x 200 x 16       */
/*                color Windows .BMP format suitable for use with           */
/*                Version 2.1 and greater of the WorkBook program,          */
/*                as well as many other programs that accept .BMP files.    */
/*                                                                          */
/*                CGA2BMP provides advanced color change capabilities       */
/*                using the keyboard keys [F1] through [F4], to             */
/*                independently change each of the 4 CGA colors in the      */
/*                source image to a color from a standard 16 color VGA      */
/*                palette in the target .BMP image.                         */
/*                                                                          */
/*                This utility is visual, and allows you to see             */
/*                your changes before either saving or abandoning.          */
/*                                                                          */
/*                Two Palettes are built in:                                */
/*                - PCX 16 Color Standard VGA Palette                       */
/*                - BMP 16 Color Standard VGA Windows Palette               */
/*                                                                          */
/*                A third palette is also available:                        */
/*                - CUSTOM 16 Color User Defined Palette.                   */
/*                                                                          */
/*                The user toggles between these by pressing the            */
/*                'B' (.BMP), 'P' (.PCX), and 'C' (CUSTOM) keys.            */
/*                                                                          */
/*                Support is provided for the following palette file        */
/*                formats to be used to define the CUSTOM palette:          */
/*                                                                          */
/*                - PaintShop Pro 16 Color Palette - Ascii Text             */
/*                - NeoPaint 16 Color Palette - Ascii Text                  */
/*                - Aldus Photostyler 16 Color Palette - Ascii Text         */
/*                - Aldus Photostyler CLR Color File - Ascii Text           */
/*                - GeoVu 16 Color Palette - Ascii Text                     */
/*                - Adobe ACT Palette File - Binary                         */
/*                - Alchemy GIF Contruction Set CMP File - Binary           */
/*                - Microsoft RIFF Palette File - Binary                    */
/*                                                                          */
/*  Some Notes on Color Palettes used in CGA2BMP                            */
/*  --------------------------------------------                            */
/*                                                                          */
/*  Colour palettes are files containing a series of colour definitions.    */
/*  More information on the formats supported by CGA2BMP is contained in    */
/*  the readme file, but a summary of this information is given below to    */
/*  make reading this program code easier.                                  */
/*                                                                          */
/*  Binary Formats Supported                                                */
/*  ------------------------                                                */
/*                                                                          */
/*  Adobe Colour Tables (.ACT files) are used by Photoshop in paletted      */
/*  images.                                                                 */
/*                                                                          */
/*  GIF Construction Set (.CMP) - Color Maps are used by by Alchemy         */
/*  Mindworks GIF Construction Set application.                             */
/*                                                                          */
/*  Microsoft palette (.PAL) - Windows palettes (.PAL files) are            */
/*  produced by a variety of programs. These are in RIFF format.             */
/*                                                                          */
/*  Text File Formats Supported                                             */
/*  ---------------------------                                             */
/*                                                                          */
/*  Paint Shop Pro Palettes (.PAL files) are created using 'Save Palette'   */
/*  in the Paint Shop Program. Not sure what versions, but probably 1-5     */
/*  at least.                                                               */
/*                                                                          */
/*  NeoPaint Palettes (.PAL files) are either created in the NeoPaint DOS   */
/*  Paint Program or the NeoPaint for Windows Paint Program.                */
/*                                                                          */
/*  The NeoPaint DOS palettes are different from Paint Shop Pro palettes    */
/*  by the fact that they use 6 bits of color information (values from      */
/*  0-63) rather than 8 bits of color (values from 0-255). (And also,       */
/*  they have different header text.)                                       */
/*                                                                          */
/*  The top half of the NeoPaint for Windows palettes are the same          */
/*  format as the NeoPaint for DOS palettes, and use 6 bits of color        */
/*  information, preserving backward compatibility. The bottom half of      */
/*  the file follows with a "duplicate" palette of 8 bit values, the        */
/*  same as that of Paint Shop Pro, allowing higher resolution              */
/*  standard 24 bit colors.                                                 */
/*                                                                          */
/*  PhotoStyler Version 2.0 Palettes (.PAL files) can be found in the       */
/*  Photostyler palettes subdirectory. This subdirectory contains           */
/*  sample files that you can load into the Color Palette. Choose Load      */
/*  Palette... from the tool-drawer menu. These are virtually the same      */
/*  format as Paint Shop Pro or Neo Paint, but are not interchangeable,     */
/*  because of minor differences. Photostyler also provide color files      */
/*  (.CLR files) that are stored under the Photostyler colors               */
/*  subdirectory. These include spreads and other interesting               */
/*  collections of assorted colors that are virtually the same format       */
/*  as the Photostyler palette files, but usually with less colors.         */
/*  Both file types are handled by cga2bmp.                                 */
/*                                                                          */
/*  GeoVu Palettes were originally used by a GeoPhysics application         */
/*  called GeoVu and have been adopted for use in CGA2BMP because they      */
/*  are easy for the user to type in. "Natural Color Names" are allowed     */
/*  so the  user can map the CGA colors more easily, and partial palette    */
/*  files are accepted.                                                     */
/*                                                                          */
/*  See the ReadMe for additional details and other drivel.                 */
/*                                                                          */
/* Revision     : 1.0 First Release                                         */
/* ------------------------------------------------------------------------ */
/* Written in Large Model MSC Version 6.00a                                 */
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <dos.h>
#include <bios.h>
#include <io.h>
#include <malloc.h>
#include <conio.h>

/* ------------------------------------------------------------------------ */
/* Declarations, Vars. etc.                                                 */
/* ------------------------------------------------------------------------ */

typedef unsigned char     uchar;
typedef unsigned int      uint;
typedef unsigned long     ulong;

uchar *szTitle[]= {
"                                  ",
" CGA2BMP(C)                       ",
" CopyLeft Bill Buckels 1999       ",
" All Rights Reversed.             ",
" Distributed as FreeWare.         ",
"                                  ",
" Email: bbuckels@escape.ca        ",
" http://www.escape.ca/~bbuckels   ",
" -------------------------------- ",
" Press [F1] To Change CGA BLACK   ",
" Press [F2] To Change CGA CYAN    ",
" Press [F3] To Change CGA MAGENTA ",
" Press [F4] To Change CGA WHITE   ",
"# Press 'C' For Custom Palette     ",
" Press 'B' For Bmp Palette Colors ",
" Press 'P' For Pcx Palette Colors ",
" -------------------------------- ",
" Press 'S' or [ENTER] to Save     ",
" Press 'Q' or [ESC] to Quit       ",
" Press 'H' For Help               ",
" Press Any Key To Continue...     ",
"                                  ",
NULL};

uchar *szTextTitle =
    "CGA2BMP(C) Version 1.0 CopyLeft Bill Buckels 1999\n"
    "All Rights Reversed.";

#define TRUE     1
#define FALSE    0
#define SUCCESS  0
#define VALID    SUCCESS
#define FAILURE  -1
#define INVALID  FAILURE

#define MCGA '\x13'
#define TEXT '\x03'

#define ASCIIZ     '\x00'
#define CRETURN    '\r'
#define LFEED      '\n'

#define ENTERKEY   '\x0d' /* character generated by the Enter Key          */
#define ESCKEY     '\x1b' /* character generated by the Esc key            */
#define FUNCKEY    '\x00' /* first character generated by function keys    */
#define UPARROW    'H'    /* second character generated by up-arrow key    */
#define DOWNARROW  'P'    /* second character generated by down-arrow key  */
#define LTARROW    'K'    /* second character generated by left-arrow key  */
#define RTARROW    'M'    /* second character generated by right-arrow key */
#define PGUP       'I'    /* second character generated by page up key     */
#define PGDOWN     'Q'    /* second character generated by page down key   */
#define HOMEKEY    'G'    /* second character generated by home key        */
#define ENDKEY     'O'    /* second character generated by end key         */

/* second character generated by numerical fkeys */
/* starting at character 59                      */
/* ending at character 68                        */
#define F1         ';'
#define F2         '<'
#define F3         '='
#define F4         '>'
#define F5         '?'
#define F6         '@'
#define F7         'A'
#define F8         'B'
#define F9         'C'
#define F10        'D'

#define FRAMESIZE ((unsigned)64000)
#define FRAMEADDR ((uchar *)0xa0000000l)

/* middle of screen */
#define XMIN 0
#define YMIN 0
#define XMOS 159
#define YMOS 99
#define XMAX 319
#define YMAX 199

#define SCREENWIDTH  (XMAX + 1)
#define RASTERWIDTH  SCREENWIDTH
#define SCREENHEIGHT (YMAX + 1)
#define CELL_SIZE    8

#define vload() memcpy(FRAMEADDR,(uchar *)&rawbuffer[0],FRAMESIZE)
#define vsave() memcpy((uchar *)&rawbuffer[0],FRAMEADDR,FRAMESIZE)

#define zap(x) memset(FRAMEADDR,x,FRAMESIZE)

// function prototypes

uchar GetMcgaPaletteIndex(uint),
      lsb(uint),
      msb(uint),
      SetCrtMode(uchar);

uint byteword(uchar, uchar),
     GetNextIndex(uint),
     GetSetNextIndex(uint);

int BSAVE_Read(uchar *),
    CheckForCGAPCX(uchar *),
    EatKeys(),
    LoadPalette(void),
    MakeBMP(uchar *, uchar *),
    PCX_Read(uchar *),
    ReadPaletteLine(uchar *, uchar *, uint);

void LineBox(int, int, int, int, uint),
     nocr(char *),
     PCMidFont(uchar *, int, int, int, int, int),
     PCRomFont(uchar *, int, int, int, int, int),
     PutPixel(int, int, uint, uchar *),
     SetInitialPalette(int),
     SetNextIndex(uchar, uchar),
     ShowTitle(void),
     SqueezeLine(char *),
     TogglePalette(uchar);

uchar outlinecolor;
uchar drawcolor;

uchar far *rawbuffer;

#define NUM_MCGA_COLORS    256
#define NUM_RGB_COLORS     3

uchar rgbinfo[NUM_MCGA_COLORS][NUM_RGB_COLORS];  /* the vga palette */

enum {  BLACK = 0,
        BLUE,
        GREEN,
        CYAN,
        RED,
        MAGENTA,
        BROWN,
        WHITE,
        GRAY,
        LBLUE,
        LGREEN,
        LCYAN,
        LRED,
        LMAGENTA,
        YELLOW,
        BWHITE,
        NUM_VGA_COLORS};

// Default Startup Palette
// standard 16 color PCX style VGA palette
uchar rgbArray[NUM_VGA_COLORS][NUM_RGB_COLORS]={
0x00, 0x00, 0x00,            /* BLACK    */
0x00, 0x00, 0xAA,            /* BLUE     */
0x00, 0xAA, 0x00,            /* GREEN    */
0x00, 0xAA, 0xAA,            /* CYAN     */
0xAA, 0x00, 0x00,            /* RED      */
0xAA, 0x00, 0xAA,            /* MAGENTA  */
0xAA, 0xAA, 0x00,            /* BROWN    */
0xAA, 0xAA, 0xAA,            /* WHITE    */
0x55, 0x55, 0x55,            /* GRAY     */
0x55, 0x55, 0xFF,            /* LBLUE    */
0x55, 0xFF, 0x55,            /* LGREEN   */
0x55, 0xFF, 0xFF,            /* LCYAN    */
0xFF, 0x55, 0x55,            /* LRED     */
0xFF, 0x55, 0xFF,            /* LMAGENTA */
0xFF, 0xFF, 0x55,            /* YELLOW   */
0xFF, 0xFF, 0xFF};           /* BWHITE   */

// standard 16 color PCX style VGA palette
uchar rgbPcxArray[NUM_VGA_COLORS][NUM_RGB_COLORS]={
0x00, 0x00, 0x00,            /* BLACK    */
0x00, 0x00, 0xAA,            /* BLUE     */
0x00, 0xAA, 0x00,            /* GREEN    */
0x00, 0xAA, 0xAA,            /* CYAN     */
0xAA, 0x00, 0x00,            /* RED      */
0xAA, 0x00, 0xAA,            /* MAGENTA  */
0xAA, 0xAA, 0x00,            /* BROWN    */
0xAA, 0xAA, 0xAA,            /* WHITE    */
0x55, 0x55, 0x55,            /* GRAY     */
0x55, 0x55, 0xFF,            /* LBLUE    */
0x55, 0xFF, 0x55,            /* LGREEN   */
0x55, 0xFF, 0xFF,            /* LCYAN    */
0xFF, 0x55, 0x55,            /* LRED     */
0xFF, 0x55, 0xFF,            /* LMAGENTA */
0xFF, 0xFF, 0x55,            /* YELLOW   */
0xFF, 0xFF, 0xFF};           /* BWHITE   */

/* Standard 16 Color BMP style VGA palette */
uchar rgbBmpArray[NUM_VGA_COLORS][NUM_RGB_COLORS]={
0x00, 0x00, 0x00,
0x00, 0x00, 0xBF,
0x00, 0xBF, 0x00,
0x00, 0xBF, 0xBF,
0xBF, 0x00, 0x00,
0xBF, 0x00, 0xBF,
0xBF, 0xBF, 0x00,
0xC0, 0xC0, 0xC0,
0x80, 0x80, 0x80,
0x00, 0x00, 0xFF,
0x00, 0xFF, 0x00,
0x00, 0xFF, 0xFF,
0xFF, 0x00, 0x00,
0xFF, 0x00, 0xFF,
0xFF, 0xFF, 0x00,
0xFF, 0xFF, 0xFF};

// standard 16 color PCX style VGA palette
// PlaceHolder for Custom Palette
uchar rgbCustomArray[NUM_VGA_COLORS][NUM_RGB_COLORS]={
0x00, 0x00, 0x00,            /* BLACK    */
0x00, 0x00, 0xAA,            /* BLUE     */
0x00, 0xAA, 0x00,            /* GREEN    */
0x00, 0xAA, 0xAA,            /* CYAN     */
0xAA, 0x00, 0x00,            /* RED      */
0xAA, 0x00, 0xAA,            /* MAGENTA  */
0xAA, 0xAA, 0x00,            /* BROWN    */
0xAA, 0xAA, 0xAA,            /* WHITE    */
0x55, 0x55, 0x55,            /* GRAY     */
0x55, 0x55, 0xFF,            /* LBLUE    */
0x55, 0xFF, 0x55,            /* LGREEN   */
0x55, 0xFF, 0xFF,            /* LCYAN    */
0xFF, 0x55, 0x55,            /* LRED     */
0xFF, 0x55, 0xFF,            /* LMAGENTA */
0xFF, 0xFF, 0x55,            /* YELLOW   */
0xFF, 0xFF, 0xFF};           /* BWHITE   */

int bCustomPalette = FALSE;

enum {  CGA_BLACK = 0,
        CGA_CYAN,
        CGA_MAGENTA,
        CGA_WHITE,
        NUM_CGA_COLORS};

uchar uiGlobalColorIndex[NUM_CGA_COLORS];

/* a microsoft compatible bsaved image format descriptor */
uchar BSAVED_header[7]={
    '\xfd','\x00','\xb8','\x00','\x00','\x00','\x40'};

// standard 16 color BitMap file header without colors
// we add colors later (RGB QUAD values are extracted from the
// current palette and used to save the image).

uchar BMP_header[]={
0x42, 0x4D, 0x76, 0x7D, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x28, 0x00, 
0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0xC8, 0x00, 
0x00, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x7D, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 
0x00, 0x00, 0x10, 0x00, 0x00, 0x00};

// headers for popular palette formats
// note: Adobe ACT and GeoVu palette files do not have headers.
//       We try to recognize these some other way, and after all,
//       the user must make a conscious decision to use a palette
//       file, so it is reasonable to expect that they will not
//       feed CGA2BMP a bogus file, since they would only be
//       "shooting themselves in the foot". If they haven't read the
//       docs, then they should.

// In the case of the .ACT file, this is a binary file, and a little
// hard for most users to create on their own. However, some programs
// use a 768 byte binary file and call it a pal file. Such a file
// could be anything. It could for example be a 6 - bit 256 color
// palette, or an 8 - bit, and it could have the red and blue positions
// in the opposite orientation from either the .ACT or the .CMP
// files that cga2bmp supports.

// In the case of the GeoVu file, the line format of the GeoVu palette is
// pretty specific, and we do test for 3 numeric values, so we should
// be ok on this as well.

// Gif Construction Set header
char *lpszGifCon = "ALCHCMPW";

// NeoPaint and PaintShop Pro headers
char *NeoPaint  = "NeoPaint Palette File";
char *PaintShop = "JASC-PAL";
char *PaintShopVersion = "0100";

// Aldus photostyler
char *AldusPal = "CWPAL";
char *AldusClr = "CWCLR";           // partial palettes
char *AldusVersion = "100";

#define FOURCC 4

// FOURCC (Four Character Codes) for RIFF Microsoft Paint Palette
// (RIFF - resource interchange file format)

char *lpszRIFF = "RIFF";
char *lpszPAL  = "PAL ";
char *lpszDATA = "data";

/* ------------------------------------------------------------------------ */
/* Low Level Video Routines, drawing routines, etc.                         */
/* ------------------------------------------------------------------------ */


uchar SetCrtMode(uchar vidmode)
{
  union REGS inregs, outregs;
  
  /* set mode */
  inregs.h.ah = 0;
  inregs.h.al = vidmode;
  int86(0x10, &inregs, &outregs);
  
  /* get mode */
  inregs.h.ah = 0xf;
  int86(0x10, &inregs, &outregs);
  
  /* return mode */
  return outregs.h.al;
  
}

int LoadPalette()
{
  union REGS regs;
  struct SREGS segregs;
  
  regs.h.ah = 0x10;                    /* fuction 10h */
  regs.h.al = 0x12;
  /* subfunction 12h - set block of color registers */
  regs.x.bx = 0;                       /* start with this reg */
  regs.x.cx = NUM_MCGA_COLORS;         /* do this many */
  regs.x.dx = (uint)rgbinfo;   /* offset to array */
  segregs.es = (uint)((long)rgbinfo >> 16);
  /* segment of array */
  int86x(0x10, &regs, &regs, &segregs);/* dump data to color registers */
  
  return SUCCESS;
  
}

/* ---------------------------------------------------------------------- */
/* Write a pixel at x,y using color                                       */
/* ---------------------------------------------------------------------- */
void PutPixel(int x, int y,uint pixelvalue, uchar *framebuffer)
{
    if (x<XMIN || x>XMAX || y<YMIN || y>YMAX)
      return;

    framebuffer[(y*RASTERWIDTH)+x]=(uchar)pixelvalue;
}

void LineBox(int x1, int y1, int x2, int y2, uint pixelvalue)
{
   int x, y;

   for (x = x1; x <= x2; x++)PutPixel(x, y1, pixelvalue, FRAMEADDR);
   for (y = (y1+1); y < y2; y++) {
     PutPixel(x1, y, pixelvalue, FRAMEADDR);
     PutPixel(x2, y, pixelvalue, FRAMEADDR);
   }
   for (x = x1; x <= x2; x++)PutPixel(x, y2, pixelvalue, FRAMEADDR);
   return;
}

/* ---------------------------------------------------------------------- */
/* PCRomFont                                                              */
/* Uses the rom font as a template for a bitmap font.                     */
/* This allows us to map a font using scaling techniques on any PC        */
/* without the need for an external font file or a bios supported font.   */
/* This was more prevalent in days gone by than it is today, of course.   */
/* ---------------------------------------------------------------------- */
void PCRomFont(uchar *str,
               int xorigin, int yorigin, int scale,
               int fontcolor, int outlinecolor)
{
    int scanline,
        yreg=yorigin,
        xreg=xorigin,
        byt,
        character,
        nibble,
        color;
    int x,y;

    /* flags etcetera */

    uchar *romfont=(uchar *) 0xffa6000el;
    /* a pointer to the 8 x 8 BITMAPPED font in the PC ROM */

    int target = strlen(str);  /* string length */

    for (scanline=0;scanline<CELL_SIZE;scanline++)/* finish the current scanline*/
    {                                     /*  before advancing to the next*/
      for (byt=0;byt<target;byt++)     /* run the scanline*/
      {
        /* get the bitmap  */
        character = romfont[(str[byt]&0x7f)*CELL_SIZE+scanline];
        for (nibble=0;nibble<CELL_SIZE;nibble++)
        {
          xreg+=scale;
          /* chew the byte to bits and lite the pixel if it's a swallow  */
          if (str[byt]!=CRETURN && str[byt]!=LFEED) {
            if (character & 0x80>>nibble)
              color = fontcolor;
            else
              color = outlinecolor;
            if (color > -1 ) {
              for (x=0; x!=scale; x++)
                for (y=0;y!=scale;y++)
                  PutPixel(xreg+x,yreg+y,color, FRAMEADDR);
            }
          }
        }
      }
      yreg+=scale;
      xreg=xorigin;
    }

}

/* ---------------------------------------------------------------------- */
/* PCMidFont                                                              */
/* Maps to PCRomfont, uses a centre-justified x-coordinate.               */
/* ---------------------------------------------------------------------- */
void PCMidFont(uchar *str, int xmiddle, int yorigin,
               int scale, int fontcolor, int outlinecolor)
{   /* centre justified string */
    PCRomFont(str,(xmiddle-(4*(strlen(str))*scale)),yorigin,scale,
              fontcolor, outlinecolor);
}

/* ---------------------------------------------------------------------- */
/* GetMcgaPalette is called at startup during the building of the         */
/* palette and loading of the input file.                                 */
/*                                                                        */
/* The palette for each color is stored in a separate 16 color bank.      */
/* We load the initial indices to map the CGA to the VGA palette.         */
/* This is changeable throughout the program so we only want to set       */
/* the global color arrays, etc. the first time...                        */
/* ---------------------------------------------------------------------- */
uchar GetMcgaPaletteIndex(uint cgacolor)
{

  uint uiIndex;

  switch(cgacolor) {
    case  CGA_WHITE:
          uiIndex = BWHITE; break;
    case  CGA_MAGENTA:
          uiIndex = LRED; break;
    case  CGA_CYAN:
          uiIndex = LBLUE; break;
    case  CGA_BLACK:
    default:
          cgacolor = uiIndex = BLACK; break;
  }
  uiIndex += (cgacolor * NUM_VGA_COLORS);    /* allow independent colors */
  return (uchar)uiIndex;
}

/* ---------------------------------------------------------------------- */
/* The following GetSet Index routines are called when the user toggles   */
/* through colors, changing them, and remapping them.                     */
/* ---------------------------------------------------------------------- */

uint GetNextIndex(uint current) {

  uint base   = current/NUM_VGA_COLORS;
  uint color  = current%NUM_VGA_COLORS;

  color++;
  current = (color%NUM_VGA_COLORS) + (base*NUM_VGA_COLORS);

  return (current);

}

void SetNextIndex(uchar oldcolor, uchar newcolor)
{
  uint idx;
  uchar *ptr = FRAMEADDR;

  for (idx = 0; idx < FRAMESIZE; idx++) {
    if (ptr[idx] == oldcolor)
      ptr[idx] = newcolor;
  }

}

uint GetSetNextIndex(uint idx)
{
   uint oldcolor = uiGlobalColorIndex[idx];
   uint newcolor = uiGlobalColorIndex[idx] = GetNextIndex(oldcolor);
   SetNextIndex((uchar)oldcolor, (uchar)newcolor);
   return (newcolor);
}

/* ---------------------------------------------------------------------- */
/* SetInitialPalette is called MAJOR at startup with bFirst set to TRUE.  */
/* Also called each time the user toggles between PCX, BMP, and CUSTOM    */
/* palette banks, to setup the Currently Used palette in memory.          */
/* ---------------------------------------------------------------------- */
void SetInitialPalette(int bFirst)
{
  uchar *src, *dest;
  uint idx, x, temp;


  /* at startup, we do some additional routines */
  if (bFirst == TRUE) {
    memset((uchar *)&rgbinfo[0][0], 0, (NUM_MCGA_COLORS*NUM_RGB_COLORS));

    /* Set the lightest and darkest color in the current palette.  */
    /* use the darkest color for the drawcolor                     */
    /* use the lightest color for the outline color                */

    drawcolor    = 254;
    outlinecolor = 255;

    // using 6 bit color model (VGA standard video uses 6 bits)
    // for the internal program. values are in the range 0-63
    // using 8 bit color model for file output
    // (24 bit standard RGB Model is used by most formats including
    // Windows BMP and HTML - values are in the range 0-255

    rgbinfo[255][0] = rgbinfo[255][1] =  rgbinfo[255][2] = 63;
  }

  for (idx = 0; idx < NUM_CGA_COLORS; idx++) {

      src = (uchar *)&rgbArray[0][0];
      dest = (uchar *)&rgbinfo[idx * NUM_VGA_COLORS][0];

      memcpy(dest, src, NUM_VGA_COLORS * NUM_RGB_COLORS);
  }

  // leave the drawcolor and outline colors alone
  for (idx = 0; (idx < NUM_MCGA_COLORS - 2) ; idx++) {
    for (x = 0; x < NUM_RGB_COLORS; x++) {
       temp = rgbinfo[idx][x];
       rgbinfo[idx][x] = temp >> 2; // downshift to 6 bits of color
    }
  }
  if (bFirst == TRUE) {
    // on startup
    // we load the initial indices to map the CGA to the VGA palette
    // this is changeable throughout the program so we only want to set
    // this global array the first time...
    // this is where the current colors are stored when the user presses the
    // FKEYS to toggle each color.

    for (idx = 0; idx < NUM_CGA_COLORS; idx++) {
      uiGlobalColorIndex[idx] = GetMcgaPaletteIndex(idx);
    }
  }
}

/* ---------------------------------------------------------------------- */
/* File Related Functions                                                 */
/* Image Loaders, Image Savers, Palette Loaders, etc.                     */
/* palette banks, to setup the Currently Used palette in memory.          */
/* ---------------------------------------------------------------------- */

/* type conversion functions */
uint byteword(uchar a, uchar b){
  return b << 8 | a;
}
uchar lsb(uint word){
  return word &0xff;
}
uchar msb(uint word){
  return word >> 8;
}

int CheckForCGAPCX(uchar *name)
{
  FILE *fp;
  /* reads a ZSOFT .PCX header but ignores the color map */
  int i;
  /* we only want CGA COLOR compatible full screens. */
  
  uchar pcxheader[128];
  uint zsoft,version,codetype,pixbits;
  uint xmin, ymin, xmax, ymax;
  uint x, y;
  uint no_planes, bytesperline;
  int status = VALID;
  
  /* read the file header */
  if((fp = fopen(name, "rb")) == NULL)return INVALID;
  for(i = 0;i < 128;i++)pcxheader[i] = fgetc(fp);
  fclose(fp);
  
  zsoft = pcxheader[0];
  version = pcxheader[1];
  codetype = pcxheader[2];
  pixbits = pcxheader[3];
  
  if(zsoft != 10)
    status = INVALID;
  if(codetype != 1)
    status = INVALID;
  if(pixbits != 2)        /* accept only CGA color images */
    status = INVALID;     /* monochrome images can't be mapped properly */
  
  xmin = byteword(pcxheader[4], pcxheader[5]);
  ymin = byteword(pcxheader[6], pcxheader[7]);
  xmax = byteword(pcxheader[8], pcxheader[9]);
  ymax = byteword(pcxheader[10], pcxheader[11]);

  no_planes = pcxheader[65];
  bytesperline = byteword(pcxheader[66], pcxheader[67]);

  x =  xmax - xmin;
  y =  ymax - ymin;

  if(x != XMAX)status = INVALID;
  if(y != YMAX)status = INVALID;
  if(no_planes != 1)status = INVALID;
  if(bytesperline != 80)status = INVALID;    /* full screens only */

  /* we can ignore the color map since we        */
  /* are limiting ourselves to CGA modes         */
  /* so we will not handle over 2-bits per pixel */

  return status;
  
}

int BSAVE_Read(uchar *name)
{
  int fh,fh2,y,i;
  uchar byte;
  uchar *crt;
  uchar namebuf[128];
  uchar headbuf[7];
  uchar linebuffer[80];
  long target = 16384l,target2,header = 7;

  /* open the file twice,eliminate the interleaf,read contiguous */
  sprintf(namebuf, "%s.BAS", name);
  
  if((fh = open(namebuf, O_RDONLY | O_BINARY)) == - 1)return INVALID;
  
  if((fh2 = open(namebuf, O_RDONLY | O_BINARY)) == - 1) {
    close(fh);
    return INVALID;
  }
  
  read(fh, headbuf, sizeof(BSAVED_header));
  
  /* only read the first 3 bytes, some of these are a little */
  /* different size, depending on how they were saved.       */

  for(i = 0;i < 3;i++)
    if(headbuf[i] != BSAVED_header[i]) {
      close(fh);
      close(fh2);
      return INVALID;
    }
  
  target2 = (target / 2) + header;
  lseek(fh, (long)(header), SEEK_SET)  ;
  lseek(fh2, (long)(target2), SEEK_SET);
  
  crt = (uchar *)&rawbuffer[0];
  
  /* translate from a 2 bit color pixel to an 8 bit color pixel */

  for(y = 0;y < SCREENHEIGHT;y += 2) {
    read(fh, linebuffer, 80);
    for(i = 0;i < 80;i++) {
      byte = linebuffer[i];
      *crt++ = GetMcgaPaletteIndex((byte >> 6));
      *crt++ = GetMcgaPaletteIndex((byte >> 4)&3);
      *crt++ = GetMcgaPaletteIndex((byte >> 2)&3);
      *crt++ = GetMcgaPaletteIndex((byte)&3);
    }
    read(fh2, linebuffer, 80);
    for(i = 0;i < 80;i++) {
      byte = linebuffer[i];
      *crt++ = GetMcgaPaletteIndex((byte >> 6));
      *crt++ = GetMcgaPaletteIndex((byte >> 4)&3);
      *crt++ = GetMcgaPaletteIndex((byte >> 2)&3);
      *crt++ = GetMcgaPaletteIndex((byte)&3);
    }
  }
  close(fh);
  close(fh2);
  return SUCCESS;
  
}

/* Translation for VGA to display CGA screen dumps with a better palette */
int PCX_Read(uchar *pcxfilename)
{
  uchar pcxheader[128];
  uchar *crt;
  uint packet;
  uchar bit[4];
  FILE *fp;
  uchar byte,bytecount;
  long wordcount,target;
  uchar name1[128], name2[128];
  
  sprintf(name1, "%s.PCX", pcxfilename);
  
  if(CheckForCGAPCX(name1) == - 1)return INVALID;

  if (NULL == (fp = fopen(name1, "rb")))return INVALID;
  
  target = filelength(fileno(fp));
  for(wordcount = 0;wordcount != 128;wordcount++)byte = fgetc(fp);
  
  crt = (uchar *)&rawbuffer[0];
  
  do {
    bytecount = 1;                     /* start with a seed count */
    byte = fgetc(fp);
    wordcount++;
    /* check to see if its raw */
    if(0xC0 == (0xC0 &byte)) {
      /* if its not, run encoded */
      bytecount = 0x3f &byte;
      byte = fgetc(fp);
      wordcount++;
    }
    /* translate from 2 bit pixel to 8 bit pixel */

    bit[0] = GetMcgaPaletteIndex((byte >> 6));
    bit[1] = GetMcgaPaletteIndex((byte >> 4)&3);
    bit[2] = GetMcgaPaletteIndex((byte >> 2)&3);
    bit[3] = GetMcgaPaletteIndex((byte)&3);
    packet = 0;
    while(packet++ < bytecount) {
      *crt++ = bit[0];
      *crt++ = bit[1];
      *crt++ = bit[2];
      *crt++ = bit[3];
    }
  }while(wordcount < target);
  fclose(fp);
  return(0);
}

/* ------------------------------------------------------------------------ */
/* MakeBMP()                                                                */
/* Writes a 16 Color Windows .BMP File from the Currently Loaded Image      */
/* Using the color mapping that has been selected by the user.              */
/* ------------------------------------------------------------------------ */
int MakeBMP(uchar *infile, uchar *bmpfile)
{
  uint temp;
  uchar *screenbuffer;
  int x, y, status = FAILURE;
  FILE *fp;
  
  puts(szTextTitle);
  printf("Infile  : %s\n", infile);
  printf("Outfile : %s\n", bmpfile);
  printf("    <= scanline");
  
  fp=fopen(bmpfile,"wb");

  if (NULL == fp) {
    puts("Unable to create bmp file.");
  }
  else {
    for (x=0; x < sizeof(BMP_header); x++) {
      temp=BMP_header[x];
      fputc(temp,fp);
    }

    for(y=0;y<NUM_VGA_COLORS;y++)
    {
      /* RGB Quad Structure b,g,r,0 */
      for(x=3;x>0;x--)
      {
        temp = rgbArray[y][x-1];
        fputc(temp,fp);
      }
      fputc(0, fp);
    }

    for (y = SCREENHEIGHT; y > 0; y--) {

      printf("\r%0003d",y);

      screenbuffer=(uchar *)&rawbuffer[((y-1)*SCREENWIDTH)];
      temp = 0;
      for (x = 0; x < SCREENWIDTH; x++) {
        temp <<=4;
        temp |=(screenbuffer[x]%NUM_VGA_COLORS);

        if (x % 2 == 1) {
          fputc(temp,fp);
          temp = 0;
        }
      }
    }

    fclose(fp);
    printf("\rDone !                    \n");
    status = SUCCESS;
  }
  return status;
}

/* ------------------------------------------------------------------------ */
/* palette reader and helper functions                                      */
/* ------------------------------------------------------------------------ */

// strip line feeds from ascii file lines...

void nocr(char *ptr) {
  int idx;
  for (idx = 0; ptr[idx] != 0; idx++)
    if (ptr[idx] == LFEED || ptr[idx] == CRETURN)
      ptr[idx] = 0;
}

// squeeze redundant whitespace from lines read-in from a palette file
// (leave only a single space character)
// this is important if the user has created their own palette file
// by hand... since they may accidentally type more than one whitespace
// between RGB values...

// Also, phototsyler version 2 palette file lines are fixed width,
// right justified so we need to massage these for our reader...

void SqueezeLine(char *ptr)
{
  int idx, jdx, len;
  char buf[128];

  idx = 0;
  while (ptr[idx] == ' ')idx++;  // remove leading whitespace
  strcpy(buf, &ptr[idx]);

  jdx = 0;
  ptr[jdx] = ASCIIZ;

  for (idx = 0; buf[idx] != ASCIIZ; idx++) {
    if (buf[idx] == 9) buf[idx] = ' ';         // no tabs please
    if (buf[idx] == ' ' && buf[idx +1] == ' ')
      continue;
    // truncate if any non-numeric characters
    if ((buf[idx] < '0' || buf[idx] > '9') && buf[idx] != ' ')
      buf[idx] = ASCIIZ;
    ptr[jdx] = buf[idx]; jdx++;
    ptr[jdx] = ASCIIZ;
  }

  // remove trailing whitespace...
  // this occurrs during parsing of photostyler
  len = strlen(ptr);
  while (len > 0) {
    len--;
    if (ptr[len] != ' ')
      break;
    ptr[len] = ASCIIZ;
  }
}

// split the RGB triple from a text line read-in from an
// ascii palette file.

int ReadPaletteLine(uchar *ptr, uchar *palptr, uint colordepth)
{
  int red, green, blue, idx, spaces = 0;

  red = atoi(ptr)&0xff;

  // there must be 2 spaces
  for (idx = 0; ptr[idx] != 0; idx++) {
    if (ptr[idx] == ' ' && ptr[idx+1] >= '0' && ptr[idx+1] <= '9') {
       spaces++;
       switch(spaces) {
         case 1:
           green = atoi(&ptr[idx+1])&0xff; break;
         case 2:
           blue = atoi(&ptr[idx+1])&0xff; break;
       }
    }
  }

  if (spaces!=2)
    return INVALID;

  if (colordepth == 6) {
     palptr[0] = red << 2;
     palptr[1] = green << 2;
     palptr[2] = blue << 2;
   }
   else {
     palptr[0] = red;
     palptr[1] = green;
     palptr[2] = blue;
   }
   return SUCCESS;

}

/* ----------------------------------------------------------------------- */
/* RiffRead - Read a Microsoft Paint Palette                               */
/* (RIFF - resource interchange file format)                               */
/* the RIFF palette is too different from the others...                    */
/* so it earned its own reader function which is called by ReadPaletteFile */
/* ----------------------------------------------------------------------- */
int RiffRead(FILE *fp)
{
   char buffer[128];
   unsigned long dwLen;
   unsigned palbufsize, numcolors, idx;
   unsigned char r, g, b;
   int temp;

   fread(buffer, FOURCC, 1, fp);    // RIFF chunk
   buffer[FOURCC] = ASCIIZ;
   if (strcmpi(buffer, lpszRIFF)) {
     rewind(fp);
     return FALSE;
   }
   fread(&dwLen, FOURCC, 1, fp);    // RIFF chunk size
   fread(buffer, FOURCC, 1, fp);
   buffer[FOURCC]  = ASCIIZ;
   if (strcmpi(buffer, lpszPAL))    // Form Type
     return INVALID;
   fread(buffer, FOURCC, 1, fp);
   buffer[FOURCC]  = ASCIIZ;
   if (strcmpi(buffer, lpszDATA))   // sub chunk ID
     return INVALID;
   fread(&dwLen, 4, 1, fp);         // sub chunk size
   fread(&palbufsize, 2, 1, fp);    // should be 768
   fread(&numcolors, 2, 1, fp);     // should be 16
   if (numcolors != NUM_VGA_COLORS)
     return INVALID;
   for (idx = 0; idx < numcolors; idx++) {
     b = fgetc(fp);
     g = fgetc(fp);
     r = fgetc(fp);
     temp = fgetc(fp);
     if (temp == EOF)
       return INVALID;
     rgbCustomArray[idx][0] = rgbArray[idx][0] = r;
     rgbCustomArray[idx][1] = rgbArray[idx][1] = g;
     rgbCustomArray[idx][2] = rgbArray[idx][2] = b;
   }
   // four bytes left (????) must be for some kind of internal use.
   // don't know what these are for but we don't need them so ignore them.

   return TRUE;
}

// ReadPaletteFile - called at startup if user has requested
// a custom palette... (i.e. entered on the command line)
// attempt to read the various palette files that we support

int ReadPaletteFile(char *lpszFile)
{
  FILE *fp;
  uchar buffer[128], *ptr;
  uint idx, jdx, colordepth;
  long target;
  int status, numcolors;

  // Try to read the  binary palette formats first.
  // If none of these then try to read the ascii formats.

  if (NULL != (fp = fopen(lpszFile, "rb"))) {

    target = filelength(fileno(fp));

    if (target == 782L) {
      fread(buffer, 14, 1, fp);
      if (!memcmp(buffer, lpszGifCon, strlen(lpszGifCon))) {
        if (buffer[12] != 4) {       // must be 4 bits
          fclose(fp);
          return FALSE;
        }
        // if we get to here, we have a valid .CMP palette file.
      }
      else {
        target = 0L;
      }
    }

    // assume that we have a valid .ACT or a .CMP file
    if (target == 768L || target == 782L) {
      for (idx = 0; idx < NUM_VGA_COLORS; idx++) {
        // read from red to blue
        rgbCustomArray[idx][2] = rgbArray[idx][2] = fgetc(fp);
        rgbCustomArray[idx][1] = rgbArray[idx][1] = fgetc(fp);
        rgbCustomArray[idx][0] = rgbArray[idx][0] = fgetc(fp);
      }
      fclose(fp);
      return TRUE;
    }
    else {
      status = RiffRead(fp);
      if (status) {
        fclose(fp);
        if (status != TRUE)
          status = FALSE;
        return status;
      }
    }
    fclose(fp);
  }

  if (NULL == (fp = fopen(lpszFile, "r")))
    return FALSE;

  if (NULL == fgets(buffer, 128, fp)) {
    fclose(fp);
    return FALSE;
  }
  nocr(buffer);

  // try for the Neopaint,
  // then Photostyler and the PaintShop Pro formats first
  // then last of all, test for the GeoVu, which we are also using.

  if (!strcmpi(NeoPaint, buffer)) {
    colordepth = 6;
  }
  else {
    colordepth = 8;
    if (strcmpi(PaintShop, buffer) && strcmpi(AldusPal, buffer) &&
        strcmpi(AldusClr, buffer)) {
      // try for the GeoVu format
      do {
        nocr(buffer);
        SqueezeLine(buffer);
        ptr = (char *)&buffer[0];
        for (jdx = 0; buffer[jdx] != ASCIIZ; jdx++) {
          if (buffer[jdx] == ' ') {
            buffer[jdx] = ASCIIZ;
            ptr = (char *)&buffer[jdx+1];
            break;
          }
        }
        idx = atoi(buffer)&0xff;

        // natural language indices in palette file
        if (SUCCESS == strcmpi("BLACK", buffer))  idx = BLACK;
        if (SUCCESS == strcmpi("CYAN", buffer))   idx = LBLUE;
        if (SUCCESS == strcmpi("MAGENTA", buffer))idx = LRED;
        if (SUCCESS == strcmpi("WHITE", buffer))  idx = BWHITE;

        if (idx > 15 || INVALID == ReadPaletteLine(ptr,
                      (char *)&rgbCustomArray[idx][0], colordepth)) {
           fclose(fp);
           return FALSE;
        }
      } while (NULL != fgets(buffer, 128, fp));

      fclose(fp);
      memcpy((char *)&rgbArray[0][0], (char *)&rgbCustomArray[0][0],
             NUM_VGA_COLORS * NUM_RGB_COLORS);
      return TRUE;
    }
  }

  /* check version if Paintshop palette since JASC may change someday */
  /* also check Aldus version although that product is old... */

  /* The Top Half of NeoPaint Windows Palettes are the same as their */
  /* DOS palettes so we use the 6 bit color values and handle both   */
  /* file types the same way... so no worry about neopaint versions. */

  if (NULL == fgets(buffer, 128, fp)) {
    fclose(fp);
    return FALSE;
  }
  nocr(buffer);
  SqueezeLine(buffer);
  if (colordepth == 8) {
    if (strcmpi(PaintShopVersion, buffer) && strcmpi(AldusVersion, buffer)) {
      fclose(fp);
      return FALSE;
    }
  }

  // only handling maximum 16 color palettes
  if (NULL == fgets(buffer, 128, fp) || (16 < (numcolors = atoi(buffer)) ||
      (1 > numcolors))) {
    fclose(fp);
    return FALSE;
  }

  for (idx = 0; idx < numcolors; idx++) {
    if (NULL == fgets(buffer, 128, fp)) {
      fclose(fp);
      return FALSE;
    }
    nocr(buffer);
    SqueezeLine(buffer);
    if (INVALID == ReadPaletteLine(buffer,
                      (char *)&rgbCustomArray[idx][0], colordepth)) {
      fclose(fp);
      return FALSE;
    }
  }
  fclose(fp);
  memcpy((char *)&rgbArray[0][0], (char *)&rgbCustomArray[0][0],
         NUM_VGA_COLORS * NUM_RGB_COLORS);
  return TRUE;

}

/* ------------------------------------------------------------------------ */
/* User Input and helper functions and Main Program                         */
/* ------------------------------------------------------------------------ */

// eat keystrokes... avoid mindlessly cycling through the program
// if the user leans on the keyboard and doesn't budge for a moment.
// A DOS program needs to do this.

int EatKeys()
{
  if (kbhit())
    while (kbhit())
      if (getch() == FUNCKEY)
         getch();

  return SUCCESS;
}

// show the title at startup and when 'H' (Help) is pressed.
void ShowTitle()
{
    int y, yorg, y1, x1;
    uchar *ptr;

    for (y = 0; szTitle[y]!= NULL; y++);
    if (bCustomPalette == FALSE)y--;
    yorg = y1 = ((SCREENHEIGHT - y*CELL_SIZE) / 2) - 1;
    for (y = 0; szTitle[y]!= NULL; y++) {
      ptr = (char *)&szTitle[y][0];

      // if a custom palette is in use, show it... otherwise skip it.
      if (szTitle[y][0] == '#') {
        if (bCustomPalette == FALSE)continue;
        ptr = (char *)&szTitle[y][1];
      }
      PCMidFont(ptr, XMOS, y1, 1, drawcolor, outlinecolor);
      y1+=CELL_SIZE;
    }
    x1 = strlen(ptr) * 4;
    LineBox(XMOS - x1 + 2, yorg+1, XMOS + x1 - 1, y1-2, drawcolor);

    EatKeys();
    while (!kbhit());
    EatKeys();

}

// change between palettes at the user's request.
// BMP, PCX, and CUSTOM are the three choices here.

void TogglePalette(uchar ptype)
{
  uchar *src;

  switch (ptype) {
    case 'C':
      src = (char *)&rgbCustomArray[0][0];break;
    case 'B':
      src = (char *)&rgbBmpArray[0][0];break;
    case 'P':
    default:
      src = (char *)&rgbPcxArray[0][0];break;
  }
  memcpy((char *)&rgbArray[0][0], src, NUM_VGA_COLORS * NUM_RGB_COLORS);
  SetInitialPalette(FALSE);
  LoadPalette();

}

void main(int argc, char **argv)
{
  int status = 0, idx, iMax;
  uchar c;
  uchar fname[128],sname[128],rgbname[128], bmpfile[128];
  uchar *wordptr;
  uchar scratchbuf[128];
  FILE *fp;

  bCustomPalette = FALSE;

  if(argc == 1) {
    puts(szTextTitle);
    puts("Command line Usage is \"CGA2BMP MyCGA.PCX\"");
    puts("                      \"CGA2BMP MyBSAVE.BAS\"");
    puts("                      \"CGA2BMP MyCGA.PCX PalettFile.Pal\"");
    puts("                      \"CGA2BMP MyBSAVE.BAS PalettFile.Pal\"");

    printf("Enter CGA Screen FileName (Blank to Exit): ");
    gets(fname);
    if (fname[0] == ASCIIZ)
      exit(1);
    printf("Enter Palette FileName (Blank for None) : ");
    gets(rgbname);
  }
  else {
    strcpy(fname, argv[1]);
    if (argc > 2)
      strcpy(rgbname, argv[2]);
    else
      rgbname[0] = ASCIIZ;
  }
  
  if((rawbuffer = malloc((unsigned)65000)) == NULL) {
    puts(szTextTitle);
    puts("Out of Memory...");
    exit(1);
  }
  
  /* replace the pukey CGA palette with more pleasing colors   */
  /* allow an optional substitution of a 16 color VGA palette  */
  /* using Standard Palette Formats (PaintShop Pro .PAL  etc.) */

  if (rgbname[0] != ASCIIZ) {
    // if we have failed during the palette read,
    // no harm done but no custom palette available either
    if (FALSE == (bCustomPalette = ReadPaletteFile(rgbname))) {
      puts(szTextTitle);
      printf("%s is an Invalid Palette File.\nPress Any Key...", rgbname);
      if (getch() == 0)
       getch();
      puts("");
    }
  }
  SetInitialPalette(TRUE);

  strcpy(sname, fname);
  wordptr = strtok(sname, ".");
  
  status = PCX_Read(sname);
  if(status)status = BSAVE_Read(sname);

  if (status) {
    puts(szTextTitle);
    printf("%s is an Unsupported Format.\n", fname);
    free(rawbuffer);
    exit(1);
  }
  
  if((SetCrtMode(MCGA)) == MCGA) {
    LoadPalette();
    // vload();
    ShowTitle();
    vload();

    do {
      c=toupper(getch());

      if (FUNCKEY == c) {
        switch(getch()) {
          case F1:
             GetSetNextIndex(CGA_BLACK); break;
          case F2:
             GetSetNextIndex(CGA_CYAN); break;
          case F3:
             GetSetNextIndex(CGA_MAGENTA); break;
          case F4:
             GetSetNextIndex(CGA_WHITE); break;
          default:
            break;
        }
      }
      else {

        if (c == 'S' || c == ENTERKEY)
          break;

        switch (c) {
          case 'B':
          case 'P':
          case 'C':
            if (bCustomPalette = FALSE && c == 'C')
              break;
            TogglePalette(c);
            break;
          case 'H':
            vsave();
            ShowTitle();
            vload();
            break;
          case 'Q':
            c = ESCKEY;
            break;
        }
      }
    } while (c!=ESCKEY);

    SetCrtMode(TEXT);

  }

  if(c!=ESCKEY) {
    sprintf(bmpfile, "%s.BMP", sname);
    MakeBMP(fname, bmpfile);
  }
  
  free(rawbuffer);
  puts("Have a Nice Dos!");
  exit(0);
  
}
