/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       DPAPC1.C
**     SYSTEM   NAME:       GOBBLER
**     ORIGINAL AUTHOR(S):  Tirza van Rijn
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1991/11/11
**
** DESCRIPTION: Packet capture application for the remote GOBBLER
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   DPAPC1.C  $
** LOGINFO:     $Log:   I:/TIRZA/CPROG/BEHOLDER/APS/DPAPC/VCS/DPAPC.C_V  $
**              
**                 Rev 1.0   11 Nov 1991 15:26:54   tirza
**              Initial revision.
*************************************************************************/
#if !defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/TIRZA/CPROG/BEHOLDER/APS/DPAPC/VCS/DPAPC.C_V   1.0   17 May 1991 15:26:54   tirza  $";
#endif


#include <stdio.h>                          
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <time.h>
#include <snmp.h>
#include <mib2.h>
#include <dp.h>
#include <bufm.h>
#include <error.h>
#include <power3.h>


#define DEFAULT_DUMPFILE_SIZE       10240   /* limit = 10K = 10240 bytes */ 
#define DEFAULT_NR_OF_PACKETS         100   /* limit = 100 packets */
#define DEFAULT_RUNTIME               100   /* limit = 100 s */
#define DEFAULT_DUMPFILE_NAME       "pktcapt.dmp" 
#define REC_HEADER          0xfe            /* Dump file record types */
#define REC_VERSION         0xfd
#define REC_HDRINFO         0xfc
#define REC_TIMESTAMP       0xfb
#define REC_FRAME           0xf8            /* Marks start of a frame */
#define FR_DUMP_HDRSIZE     (1+sizeof(unsigned int)) /* ID + len */
#define OjiNumber(x) (sizeof((x))/sizeof(unsigned))


/* Function prototypes */
static int  ProcessEvent(DPEVENT Event);
static void ReceivePkt(void);
static void EverySecond(void);
static int  Init(void);
static void End(void);
static int  Start(void);
static void Stop(void);
static void Reset(void);
static void Show(void);
static void Hide(void);
static void FreeTime(void);
static void KeyPressed(void);
static unsigned DPAPCGetMDumpfileSize(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetMNrOfPackets(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetMRuntime(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetDumpfileName(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetDumpfileSize(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetnPkts(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetSeconds(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetTotalPackets(pkt_obj *Obj, void *Arg);
static unsigned DPAPCGetMissedPackets(pkt_obj *Obj, void *Arg);
static unsigned DPAPCSetMDumpfileSize(pkt_obj *Obj, void *Arg);
static unsigned DPAPCSetMNrOfPackets(pkt_obj *Obj, void *Arg);
static unsigned DPAPCSetMRuntime(pkt_obj *Obj, void *Arg);
static unsigned DPAPCSetDumpfileName(pkt_obj *Obj, void *Arg);


DPAPPS DpaPktCapt = {
    "PktCapt",                              /* application name */
    ProcessEvent,                           /* event processing function */
    0,                                      /* status bits, must be 0 */
/* accept event mask, tells the beholder which events this */
/* application wishes to handle                              */
/*   DPE_DEFAULT = DPE_INIT | DPE_END | DPE_START | DPE_STOP */
    DPE_DEFAULT | DPE_RESET | DPE_FREETIME | DPE_EVERYSECOND | DPE_RECEIVEPKT | DPE_KEYPRESSED,
/* start event mask, tells the beholder which events to          */
/* generate for this application during the 'DpStart()' function  */
    DPE_RESET | DPE_START | DPE_SHOW,           
    0,                                           /* timer event mask */
    0,                                           /* timer value */
    0,                                           /* timer start */
    0,                                           /* timer stop */
    0,                                           /* timer timer */
    1,                                           /* # of active filters */
    0,                                           /* active filter # */
    {0}                                          /* list of active filters */
};
struct rec_hdr {
    unsigned char type;
    int len;
};
static struct rec_hdr hdr;
static char rec_header[] = "FRAME dumpfile";
static char rec_version[] = "v1.00";
static char rec_timestamp[18];
static long max_DumpfileSize;
static long DumpfileSize;
static unsigned long max_NrOfPackets;
static unsigned long nPkts;
static long max_Runtime;
static long Seconds;
static FILE *dumpfile;
static char DumpfileName[80];
static int DumpfileNameLength;
static DPSTAT *pStat;
static long InitTotalPackets;
static long InitMissedPackets;


static int ProcessEvent(DPEVENT Event)
{
    int ret = 0;

    switch(Event)
    {
        case DPE_RECEIVEPKT:    ReceivePkt();   break;
        case DPE_EVERYSECOND:   EverySecond();  break;
        case DPE_INIT:    ret = Init();         break;
        case DPE_END:           End();          break;
        case DPE_START:   ret = Start();        break;
        case DPE_STOP:          Stop();         break;
        case DPE_SHOW:          Show();         break;
        case DPE_HIDE:          Hide();         break;
        case DPE_RESET:         Reset();        break;
        case DPE_FREETIME:      FreeTime();     break;
        case DPE_KEYPRESSED:    KeyPressed();   break;
    }
    return ret;
}

static void ReceivePkt(void)
{
    DPBUF *frame = DpReceivePkt();

    if ((dumpfile != NULL) &&
        ((DumpfileSize + 3 + sizeof(DPBUF) + frame->Size) <= max_DumpfileSize) &&
        (nPkts <= max_NrOfPackets) &&
        (Seconds <= max_Runtime))
    {
        hdr.type = REC_FRAME;
        hdr.len = sizeof(DPBUF) + frame->Size + FR_DUMP_HDRSIZE;
        fwrite(&hdr.type, sizeof(hdr.type), 1, dumpfile);
        fwrite(&hdr.len, sizeof(hdr.len), 1, dumpfile);
        fwrite(frame, sizeof(DPBUF), 1, dumpfile);
        fwrite(frame->pBuf, sizeof(BYTE), (frame->Size), dumpfile);
        DumpfileSize = DumpfileSize + 3 + sizeof(DPBUF) + frame->Size;

        nPkts++;
    }
    else
        DpAppsEvent(DpAppsGetApps("PktCapt"), DPE_STOP);
}

static void EverySecond(void)
{
    Seconds++;                         
}

static int Init(void)
{
    mib_prf *Prf;
    unsigned GetMDumpfileSize[] = {1,3,6,1,4,1,99,99,98,1};
    unsigned GetMNrOfPackets[]  = {1,3,6,1,4,1,99,99,98,2};
    unsigned GetMRuntime[]      = {1,3,6,1,4,1,99,99,98,3};
    unsigned GetDumpfileName[]  = {1,3,6,1,4,1,99,99,98,4};
    unsigned GetDumpfileSize[]  = {1,3,6,1,4,1,99,99,98,5};
    unsigned GetnPkts[]         = {1,3,6,1,4,1,99,99,98,6};
    unsigned GetSeconds[]       = {1,3,6,1,4,1,99,99,98,7};
    unsigned GetTotalPackets[]  = {1,3,6,1,4,1,99,99,98,8};
    unsigned GetMissedPackets[] = {1,3,6,1,4,1,99,99,98,9};


    max_DumpfileSize = DEFAULT_DUMPFILE_SIZE;
    max_NrOfPackets  = DEFAULT_NR_OF_PACKETS;
    max_Runtime      = DEFAULT_RUNTIME;
    strcpy (DumpfileName, DEFAULT_DUMPFILE_NAME);
    DumpfileSize = 0L;
    nPkts = 0L;
    Seconds = 0L;

    ERR_DEB(ERR_PCA, 1, "DPAPC=Init: starting"); 
    if ((Prf = MibAddPrf ("test", strlen("test"))) == NULL)
        return -1;
    if ((MibAddObj (Prf, GetMDumpfileSize, OjiNumber(GetMDumpfileSize), DPAPCGetMDumpfileSize, NULL, DPAPCSetMDumpfileSize, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetMNrOfPackets, OjiNumber(GetMNrOfPackets), DPAPCGetMNrOfPackets, NULL, DPAPCSetMNrOfPackets, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetMRuntime, OjiNumber(GetMRuntime), DPAPCGetMRuntime, NULL, DPAPCSetMRuntime, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetDumpfileName, OjiNumber(GetDumpfileName), DPAPCGetDumpfileName, NULL, DPAPCSetDumpfileName, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetDumpfileSize, OjiNumber(GetDumpfileSize), DPAPCGetDumpfileSize, NULL, NULL, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetnPkts, OjiNumber(GetnPkts), DPAPCGetnPkts, NULL, NULL, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetSeconds, OjiNumber(GetSeconds), DPAPCGetSeconds, NULL, NULL, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetTotalPackets, OjiNumber(GetTotalPackets), DPAPCGetTotalPackets, NULL, NULL, NULL)) < 0)
        return -1;
    if ((MibAddObj (Prf, GetMissedPackets, OjiNumber(GetMissedPackets), DPAPCGetMissedPackets, NULL, NULL, NULL)) < 0)
        return -1;
    ERR_DEB(ERR_PCA, 1, "DPAPC=Init: successfully added SNMP-variables"); 
    return 0;
}

static void End(void)
{
}

static int Start(void)
{
    if (((pStat = DpDevGetStatistics(0)) != NULL) || ((pStat = DpDevGetStatistics(1)) != NULL))
    {
        InitTotalPackets = pStat->TotalPackets;
        InitMissedPackets = pStat->MissedPackets;
    }
    
    dumpfile = fopen(DumpfileName, "wb");
    if (dumpfile != NULL)
    {
        hdr.type = REC_HEADER;
        hdr.len = sizeof(rec_header) + FR_DUMP_HDRSIZE;
        fwrite(&hdr.type, sizeof(hdr.type), 1, dumpfile);
        fwrite(&hdr.len, sizeof(hdr.len), 1, dumpfile);
        fwrite(rec_header, sizeof(rec_header), 1, dumpfile);

        hdr.type = REC_VERSION;
        hdr.len = sizeof(rec_version) + FR_DUMP_HDRSIZE;
        fwrite(&hdr.type, sizeof(hdr.type), 1, dumpfile);
        fwrite(&hdr.len, sizeof(hdr.len), 1, dumpfile);
        fwrite(rec_version, sizeof(rec_version), 1, dumpfile);

        hdr.type = REC_HDRINFO;
        hdr.len = sizeof(DPHDRINFO) + FR_DUMP_HDRSIZE;
        fwrite(&hdr.type, sizeof(hdr.type), 1, dumpfile);
        fwrite(&hdr.len, sizeof(hdr.len), 1, dumpfile);
        fwrite(hdrinfo, sizeof(DPHDRINFO), 1, dumpfile);

        hdr.type = REC_TIMESTAMP;
        hdr.len = sizeof(rec_timestamp) + FR_DUMP_HDRSIZE;
        _strdate(rec_timestamp);
        rec_timestamp[8] = ' ';
        _strtime(rec_timestamp+9);
        fwrite(&hdr.type, sizeof(hdr.type), 1, dumpfile);
        fwrite(&hdr.len, sizeof(hdr.len), 1, dumpfile);
        fwrite(rec_timestamp, sizeof(rec_timestamp), 1, dumpfile);

        fflush(dumpfile);
        DumpfileSize = filelength(fileno(dumpfile));

        return 0;              
    }
    else
        return -1;
}

static void Stop(void)
{
    if (dumpfile != NULL)
        fclose(dumpfile);
    nPkts = 0L;
    Seconds = 0L;
}

static void Reset(void)
{
    max_DumpfileSize = DEFAULT_DUMPFILE_SIZE;
    max_NrOfPackets  = DEFAULT_NR_OF_PACKETS;
    max_Runtime      = DEFAULT_RUNTIME;
    strcpy (DumpfileName, DEFAULT_DUMPFILE_NAME);
    nPkts = 0L;
    Seconds = 0L;
    DumpfileSize = 0L;
}

static void Show(void)
{
}

static void Hide(void)
{
}

static void FreeTime(void)
{
    if (Seconds > max_Runtime)
        DpAppsEvent(DpAppsGetApps("PktCapt"), DPE_STOP);
}

static void KeyPressed(void)
{
}

static unsigned DPAPCGetMDumpfileSize(pkt_obj *Obj, void *Arg)
{                                                                         
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMDumpfileSize: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMDumpfileSize: Maximum dumpfile size was %ld", max_DumpfileSize); 
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = max_DumpfileSize;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMDumpfileSize: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetMNrOfPackets(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMNrOfPackets: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMNrOfPackets: Maximum number of packets was %ld", max_NrOfPackets); 
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = max_NrOfPackets;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMNrOfPackets: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetMRuntime(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMRuntime: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMRuntime: Maximum runtime was %ld", max_Runtime); 
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = max_Runtime;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMRuntime: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetDumpfileName(pkt_obj *Obj, void *Arg)
{                                                                         
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileName: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileName: Dumpfile name was %s", DumpfileName); 
    Obj->ObjTag = PKT_OCTETSTRING;
    memcpy (Obj->ObjSyn.SynBufChr, DumpfileName, DumpfileNameLength);
    Obj->ObjSynLen = DumpfileNameLength;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileName: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetDumpfileSize(pkt_obj *Obj, void *Arg)
{                                                                         
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileSize: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileSize: Dumpfile size was %ld", DumpfileSize); 
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = DumpfileSize;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetDumpfileSize: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetnPkts(pkt_obj *Obj, void *Arg)
{                                                                         
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetnPkts: starting...");
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetnPkts: Number of captured packets was %ld", nPkts);
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = nPkts;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetnPkts: ended.");
    return (PKT_NOERROR);
}

static unsigned DPAPCGetSeconds(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetSeconds: starting..."); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetSeconds: Runtime was %ld", Seconds); 
    Obj->ObjTag = PKT_INTEGER;
    Obj->ObjSyn.SynLngInt = Seconds;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetRuntime: ended."); 
    return (PKT_NOERROR);
}

static unsigned DPAPCGetTotalPackets(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetTotalPackets: starting...");
    if (((pStat = DpDevGetStatistics(0)) != NULL) || ((pStat = DpDevGetStatistics(1)) != NULL))
    {
        ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetTotalPackets: Total packets was %ld", pStat->TotalPackets - InitTotalPackets); 
        Obj->ObjTag = PKT_INTEGER;
        Obj->ObjSyn.SynLngInt = pStat->TotalPackets - InitTotalPackets;
        ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetTotalPackets: ended.");
        return (PKT_NOERROR);
    }
}

static unsigned DPAPCGetMissedPackets(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMissedPackets: starting...");
    if (((pStat = DpDevGetStatistics(0)) != NULL) || ((pStat = DpDevGetStatistics(1)) != NULL))
    {
        ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMissedPackets:  Missed packets was %ld", pStat->MissedPackets - InitMissedPackets); 
        Obj->ObjTag = PKT_INTEGER;
        Obj->ObjSyn.SynLngInt = pStat->MissedPackets - InitMissedPackets;
        ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCGetMissedPackets: ended.");
        return (PKT_NOERROR);
    }
}

static unsigned DPAPCSetMDumpfileSize(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMDumpfileSize: starting..."); 
    if (Obj->ObjTag != PKT_INTEGER)
        return (PKT_BADVALUE);
    max_DumpfileSize = Obj->ObjSyn.SynLngInt;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMDumpfileSize: Maximum dumpfile size is %ld", max_DumpfileSize); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMDumpfileSize: ended."); 
    return(PKT_NOERROR);
}

static unsigned DPAPCSetMNrOfPackets(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMNrOfPackets: starting..."); 
    if (Obj->ObjTag != PKT_INTEGER)
        return (PKT_BADVALUE);
    max_NrOfPackets = Obj->ObjSyn.SynLngInt;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMNrOfPackets: Maximum number of packets is %ld", max_NrOfPackets); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMNrOfPackets: ended."); 
    return(PKT_NOERROR);
}

static unsigned DPAPCSetMRuntime(pkt_obj *Obj, void *Arg)
{
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMRuntime: starting..."); 
    if (Obj->ObjTag != PKT_INTEGER)
        return (PKT_BADVALUE);
    max_Runtime = Obj->ObjSyn.SynLngInt;
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMRuntime:  Maximum runtime is %ld", max_Runtime); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetMRuntime: ended."); 
    return(PKT_NOERROR);
}

static unsigned DPAPCSetDumpfileName(pkt_obj *Obj, void *Arg)
{                                                                         
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetDumpfileName: starting..."); 
    if (Obj->ObjTag != PKT_OCTETSTRING)
        return (PKT_BADVALUE);
    DumpfileNameLength = Obj->ObjSynLen;
    memcpy (DumpfileName, Obj->ObjSyn.SynBufChr, DumpfileNameLength);
    DumpfileName[DumpfileNameLength] = '\0';
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetDumpfileName: Dumpfile name is %s", DumpfileName); 
    ERR_DEB(ERR_PCA, 1, "DPAPC=DPAPCSetDumpfileName: ended."); 
    return (PKT_NOERROR);
}
