#include "pwinc.h"

PWWIN *wintable[MAX_WINS];
int    _pwb_nlevels = 0;

static void   pwb_add(PWWIN *win, int level);
static void   pwb_delete(PWWIN *win);
static void   pwb_shadeblock(int level, int srow, int scol, int nrow, int ncol);
static void   pwb_refreshblock_recur(int level, int srow, int scol, int erow, int ecol);
static void   movehide(PWWIN *win, int row, int col);
static void   pwb_addlevel(int level);
static void   pwb_deletelevel(int level);

static int    _pwb_activelevel;
static int    _pwb_active = 1;
static int    _pwb_offset;                     /* buffer */
static int    _pwb_nrow, _pwb_ncol;
static PWCELL far *_pwb_buffer;
static int    _pwb_level;
static PWWIN  BGwin;

/* [DOC] Function *********************************************
** Name:        pw_win                                    [API]
** SYNOPSIS:    PWWIN *pw_win(id)
**              int id              Window ID
** Descr:       Returns a pointer to window 'id'.
**                'id's are defined in POWER3.H and can be
**                one of the following:
**                  PWID_TOP        Return top window handle
**                  PWID_BACKGROUND Return backgrnd handle
**                  PWID_ACTIVE     Return active win handle
**              For now, the Active window has no meaning.  It
**                is supported for later versions of POWER.
**              Note that the background window has the
**                following coordinates:
**                  srow=1, scol=1, nrow=23, ncol=78
**                and all border characters are: background.
** RETURNS:     Pointer to requested window handle.
**************************************************************/
PWWIN *pw_win(int id)
{
    switch (id)
    {
    case PWID_TOP:
        return wintable[_pwb_nlevels-1];
    case PWID_BACKGROUND:
        return wintable[0];
    case PWID_ACTIVE:
        return wintable[_pwb_active];
    default:
        return NULL;
    }
}

void pwb_init1(PWCELL far *oldbuf)
{
    int i;
    PWWIN *win = &BGwin;

    _pwb_nlevels = 0;
    for (i=_pwb_nlevels; i<MAX_WINS; i++)
        wintable[i] = NULL;                 /* no other windows openend yet */
    win->buffer = oldbuf;
    win->srow   = win->scol = 1;
    win->nrow   = _pwv_scrnrow-2;
    win->ncol   = _pwv_scrncol-2;
    win->wtype  = PWW_INIT;
    win->ttype  = PWT_PLAIN;
    win->header = "Background";
    win->status = 0;
    pwg_initwin(win);
    pwb_add(win, 0);
}

int pwb_init2(void)
{
    int i;
    PWCELL cell = (pwv_attrib(PWW_INIT, PWT_NORMAL) << 8) + _pwcf_backgr[BGCHAR];
    PWWIN *win = wintable[0];
    PWCELL far *fptr;

    fptr = win->buffer = pwg_malloc(_pwv_scrnrow * _pwv_scrncol * sizeof(PWCELL));
    if (fptr == NULL)
        return - 9;
    for (i=(_pwv_scrnrow-1)*_pwv_scrncol; i; i--)
        *(fptr++) = cell;
    cell = (pwv_attrib(PWW_KEYINFO, PWT_NORMAL) << 8) + ' ';
    for (i = _pwv_scrncol; i; i--)
        *(fptr++) = cell;
    pwb_initwin(win);
    pwb_show(win);
    return 0;
}

void pwb_exit(void)
{
    int level;

    for (level=0; level<_pwb_nlevels; level++)
    {
        pwg_free(wintable[level]->buffer);              /* don't update screen */
        free(wintable[level]);
    }
}

int pwb_open(PWWIN *win, int level)
{
    int i, row;
    PWCELL cell;
    PWCELL cell0;
    PWCELL cellv;
    unsigned char c;
    PWCELL far *buf;

    if (_pwb_nlevels >= MAX_WINS)
        return -1;                          /* too many windows open */
    if ((level == 0) || (level > _pwb_nlevels))
        level = _pwb_nlevels;
    if (level < 0)
        level += _pwb_nlevels;
    if (level < 0)
        level = 1;
    pwb_add(win, level);
    pwb_initwin(win);
    buf = win->buffer;
    cell0 = pwv_attrib(win->wtype, PWT_ACTIVEBORDER) << 8;
    cellv = (pwv_attrib(win->wtype, PWT_ACTIVEBORDER) << 8) + _pwcf_border[PWCF_V];

    i = 0;
    *(buf++) = (cell0 + _pwcf_border[PWCF_TL]);    /* topleft */
    if ((win->status & PWM_NOHEADER) == 0)
    {
        while ((i < _pwb_ncol-2) && ((c = win->header[i]) != '\0'))
        {
            *(buf++) = (cell0 + c);                /* header */
            i++;
        }
    }
    cell = cell0 + _pwcf_border[PWCF_H];
    for (; i < _pwb_ncol-2; i++)
        *(buf++) = (cell);                         /* top */
    *(buf++) = (cell0 + _pwcf_border[PWCF_TR]);    /* topright */

    /* cls inside and side borders */
    cell = (pwv_attrib(win->wtype, PWT_NORMAL) << 8) + ' ';
    for (row=0; row<_pwb_nrow-2; row++)
    {
        *(buf++) = (cellv);
        for (i=0; i<_pwb_ncol-2; i++)
            *(buf++) = (cell);
        *(buf++) = (cellv);
    }

    *(buf++) = (cell0 + _pwcf_border[PWCF_BL]);     /* bottomleft */
    cell = cell0 + _pwcf_border[PWCF_H];
    for (i=0; i < _pwb_ncol-2; i++)
        *(buf++) = (cell);                          /* bottom */
    *(buf++) = (cell0 + _pwcf_border[PWCF_BR]);     /* bottomright */
    if ((win->status & PWM_HIDE) == 0)
    {
        pwb_show(win);
    }
    return 0;                                       /* successful call */
}

static void pwb_addlevel(int level)
{
    int i;

    for (i=0; i<(_pwv_scrnrow-1)*_pwv_scrncol; i++) /* update _pwv_levbuf[] */
        if (_pwv_levbuf[i] >= (unsigned char)level)
            _pwv_levbuf[i]++;
}

static void pwb_deletelevel(int level)
{
    int i;

    for (i=0; i<(_pwv_scrnrow-1)*_pwv_scrncol; i++) /* update _pwv_levbuf[] */
        if (_pwv_levbuf[i] > (unsigned char)level)
            _pwv_levbuf[i]--;
}

void pwb_shade(PWWIN *win)
{
    pwb_shadeblock(win->level, win->tsrow+1, win->tscol+win->tncol,
                               win->tnrow-1, 2);
    pwb_shadeblock(win->level, win->tsrow+win->tnrow, win->tscol+2,
                               1, win->tncol);
}

void pwb_shaderestore(PWWIN *win)
{
    int i;
    if ((i = (win->tscol+win->tncol+2 > _pwv_scrncol) ? _pwv_scrncol-(win->tscol+win->tncol) : 2) != 0)
        pwb_refreshblock(win->level, win->tsrow+1, win->tscol+win->tncol,
                                     win->tnrow-1, i);
    i = (win->tscol+2+win->tncol > _pwv_scrncol) ? _pwv_scrncol-(win->tscol+2) : win->tncol;
        pwb_refreshblock(win->level, win->tsrow+win->tnrow, win->tscol+2,
                                     1, i);
}

void pwb_close(PWWIN *win)
{
    pwb_hide(win);
    pwb_delete(win);
}

void pwb_pop(PWWIN *win)
{
    if ((win->status & STAT_SHADE) && !(win->status & STAT_HIDE))
    {
        win->status = win->status ^ STAT_HIDE; /* hide window */
        pwb_shaderestore(win);
        win->status = win->status ^ STAT_HIDE; /* show window */
    }
    pwb_delete(win);
    pwb_add(win, _pwb_nlevels);
    pwb_show(win);
}

void pwb_hide(PWWIN *win)
{
    if (!(win->status & STAT_HIDE))         /* visible ? */
    {
        win->status = win->status ^ STAT_HIDE; /* hide window */
        if (win->status & STAT_SHADE)
            pwb_shaderestore(win);
        pwb_refreshblock(win->level, win->tsrow, win->tscol, win->tnrow, win->tncol);
    }
}

void pwb_show(PWWIN *win)
{
    if (win->status & STAT_HIDE)            /* hided ? */
    {
        win->status = win->status ^ STAT_HIDE; /* visible window */
        pwb_refreshblock(win->level, win->tsrow, win->tscol, win->tnrow, win->tncol);
        if (win->status & STAT_SHADE)
            pwb_shade(win);
    }
}

void pwb_move(PWWIN *win)
{
    PWKEY key;
    int oldrow = win->srow;
    int oldcol = win->scol;
    PWKEY *PrevMouDef = pwl_GetMouDef();
    unsigned oldstatus = win->status;
    int r, c;

    if (oldstatus & STAT_SHADE)
    {
        win->status ^= STAT_SHADE;
        pwb_shaderestore(win);
    }
    pwg_keyinfo("Move", "Cursor=Move  Enter=Place");
    pwl_SetMouDef(_pw_MouDefDefault);      /* switch to de mou settings */
    pws_GetCursor(&r, &c);
    pwg_CursorOff();
    while (((key = pw_getkey()) != PWK_ESC) && (key != PWK_ENTER))
    {
        switch (key)
        {
        case PWK_LEFT:
            if (win->tscol > 0)
                movehide(win, 0, -1);
            break;
        case PWK_RIGHT:
            if (win->tscol+win->tncol < _pwv_scrncol)
                movehide(win, 0, 1);
            break;
        case PWK_UP:
            if (win->tsrow > 0)
                movehide(win, -1, 0);
            break;
        case PWK_DOWN:
            if (win->tsrow+win->tnrow < _pwv_scrnrow-1)
                movehide(win, 1, 0);
        }
    }
    if (key != PWK_ENTER)
    {
        pwb_hide(win);
        win->srow = oldrow;
        win->scol = oldcol;
        win->tsrow = win->srow;
        win->tscol = win->scol;
        if ((win->status & PWM_NOBORDER) == 0)
        {
            win->tscol--;
            win->tsrow--;
        }
        pwb_show(win);
    }
    win->status = oldstatus;
    if (oldstatus & STAT_SHADE)
        pwb_shade(win);
    if (r < _pwv_scrnrow)
        pws_cursor(win->srow+r-oldrow, win->scol+c-oldcol);
    pwl_SetMouDef(PrevMouDef);      /* switch to defined settings */
    pwg_keyinfo(win->keyleft, win->keyright);
}

int pwb_top(PWWIN *win)
{
    return (win->level == _pwb_nlevels-1);
}

PWWIN *pwb_activewin(void)
{
    if (_pwb_nlevels > 1)
        return wintable[_pwb_nlevels-1];
    else
        return NULL;
}

void pwb_initwin(PWWIN *win)
{
    _pwb_level  = win->level;
    _pwb_nrow   = win->nrow+2;
    _pwb_ncol   = win->ncol+2;
    _pwb_buffer = win->buffer;
}

void pwb_initoffset(int crow, int ccol)
{
    _pwb_offset = (crow)*_pwb_ncol + ccol;
}

static void pwb_add(PWWIN *win, int level)
{
    int i;

    if (_pwb_nlevels)
        pwb_addlevel(level);
/* ************************************************************
                  BEFORE      AFTER               BEFORE  AFTER
       level   window id  window id          id    level  level
     --------  ---------  ---------  -----------  ------  -----
       win[0] =  INIT  0    INIT  0  win_pwv_levbuf[0] =     0      0
           1        A  5       A  5           1        2      3
     level=2        B  1     NEW  2           2        0      2
           3        C  4       B  1           3        0      0
           4                   C  4           4        3      4
          ..      .. ..       .. ..           5        1      1
   _pwb_nlevels              _pwb_nlevels          ..
** ***********************************************************/
    for (i=_pwb_nlevels++; i>level; i--)
    {
        wintable[i] = wintable[i-1];
        wintable[i]->level++;
    }
    wintable[level] = win;
    win->level = level;
    win->status = win->status | STAT_HIDE;   /* not yet displayed */
}

static void pwb_delete(PWWIN *win)
{
    int i;
    int level = win->level;

    pwb_deletelevel(level);
/* ************************************************************
                   AFTER     BEFORE                AFTER BEFORE
       level   window id  window id          id    level  level
     --------  ---------  ---------  -----------  ------  -----
       win[0] =  INIT  0    INIT  0  win_pwv_levbuf[0] =     0      0
           1        A  5       A  5           1        2      3
     level=2        B  1     NEW  2           2        0      2
           3        C  4       B  1           3        0      0
           4       .. ..       C  4           4        3      4
          ..       .. ..      .. ..           5        1      1
   _pwb_nlevels                  .. ..
                           wintables          ..
** ***********************************************************/
    --_pwb_nlevels;
    for (i=level; i<_pwb_nlevels; i++)
    {
        wintable[i] = wintable[i+1];
        wintable[i]->level--;               /* visible window */
    }
    wintable[_pwb_nlevels] = NULL;
}

static void pwb_shadeblock(int level, int srow, int scol, int nrow, int ncol)
{
    int row, col;
    unsigned offset = srow*_pwv_scrncol + scol;
    
    if (scol+ncol > _pwv_scrncol)
        ncol = _pwv_scrncol-scol;
    if (srow+nrow > _pwv_scrnrow-1)
        nrow = _pwv_scrnrow-1-srow;
    for (row=nrow; row; row--)
    {
        for (col=ncol; col; col--)
        {
            if (_pwv_levbuf[offset] < (unsigned char)level)
                pwv_rawputcell(offset, pwv_shadecell(pwv_rawgetcell(offset)));
            offset++;
        }
        offset += (_pwv_scrncol - ncol);
    }
}

void pwb_refreshblock(int activelevel, int srow, int scol, int nrow, int ncol)
{
    _pwb_activelevel = activelevel;
    pwb_refreshblock_recur(_pwb_nlevels-1, srow, scol, srow+nrow-1, scol+ncol-1);
}

static void pwb_refreshblock_recur(int level, int srow, int scol, int erow, int ecol)
{
    PWWIN *win = wintable[level];
    int wsrow, wscol, werow, wecol;
    int nsrow, nscol, nerow, necol;

    if (!(win->status & STAT_HIDE))         /* visible ? */
    {
        wsrow = win->tsrow;
        wscol = win->tscol;
        werow = wsrow+win->tnrow-1;
        wecol = wscol+win->tncol-1;
        nsrow = (srow > wsrow) ? srow : wsrow;
        nscol = (scol > wscol) ? scol : wscol;
        nerow = (erow < werow) ? erow : werow;
        necol = (ecol < wecol) ? ecol : wecol;
        if ((nerow >= nsrow) && (necol >= nscol))
        {
            if (nsrow > srow)
                pwb_refreshblock_recur(level-1, srow,    scol,    nsrow-1, ecol);
            if (nscol > scol)
                pwb_refreshblock_recur(level-1, nsrow,   scol,    nerow,   nscol-1);
            if (necol < ecol)
                pwb_refreshblock_recur(level-1, nsrow,   necol+1, nerow,   ecol);
            if (nerow < erow)
                pwb_refreshblock_recur(level-1, nerow+1, scol,    erow,    ecol);
        
            if ((level==-_pwb_activelevel) || (level<=_pwb_activelevel))
            {
                pwv_levelblock(nsrow, nscol, nerow-nsrow+1, necol-nscol+1, _pwv_levbuf, level);
                if ((win->status & PWM_NOBORDER) == 0)
                    pwv_display(nsrow, nscol, nerow-nsrow+1, necol-nscol+1, 
                            win->buffer+(nsrow-wsrow  )*(win->ncol+2)+nscol-wscol  , 
                            (win->ncol+2));
                else
                    pwv_display(nsrow, nscol, nerow-nsrow+1, necol-nscol+1, 
                            win->buffer+(nsrow-wsrow+1)*(win->ncol+2)+nscol-wscol+1, 
                            (win->ncol+2));
            }
        }
        else
            pwb_refreshblock_recur(level-1, srow, scol, erow, ecol);
        if (win->status & STAT_SHADE)
        {
            wsrow = win->tsrow+1;
            wscol = win->tscol+win->tncol;
            werow = wsrow+win->tnrow-2;
            wecol = wscol+1;
            nsrow = (srow > wsrow) ? srow : wsrow;
            nscol = (scol > wscol) ? scol : wscol;
            nerow = (erow < werow) ? erow : werow;
            necol = (ecol < wecol) ? ecol : wecol;
            if ((nerow >= nsrow) && (necol >= nscol))
                pwb_shadeblock(win->level, nsrow, nscol, nerow-nsrow+1, necol-nscol+1);
            wsrow = win->tsrow+win->tnrow;
            wscol = win->tscol+2;
            werow = wsrow;
            wecol = wscol+win->tncol-1;
            nsrow = (srow > wsrow) ? srow : wsrow;
            nscol = (scol > wscol) ? scol : wscol;
            nerow = (erow < werow) ? erow : werow;
            necol = (ecol < wecol) ? ecol : wecol;
            if ((nerow >= nsrow) && (necol >= nscol) && (nsrow <_pwv_scrnrow-1))
                pwb_shadeblock(win->level, nsrow, nscol, nerow-nsrow+1, necol-nscol+1);
        }
    }
    else
        pwb_refreshblock_recur(level-1, srow, scol, erow, ecol);
}

void pwb_refreshwin(PWWIN *win)
{
    if (!(win->status & STAT_HIDE))         /* visible ? */
    {
        _pwb_activelevel = -win->level;
        pwb_refreshblock_recur(_pwb_nlevels-1, win->tsrow, win->tscol, win->tsrow+win->tnrow-1, win->tscol+win->tncol-1);
    }
}

static void movehide(PWWIN *win, int row, int col)
{
    int level = win->level;

    win->status = win->status ^ STAT_HIDE;
    if (row > 0)
        pwb_refreshblock(level, win->tsrow, win->tscol, 1, win->tncol);
    else
    if (row < 0)
        pwb_refreshblock(level, win->tsrow+win->tnrow-1, win->tscol, 1, win->tncol);
    else
    if (col > 0)
        pwb_refreshblock(level, win->tsrow, win->tscol, win->tnrow, 1);
    else
    if (col < 0)
        pwb_refreshblock(level, win->tsrow, win->tscol+win->tncol-1, win->tnrow, 1);
    win->srow += row;
    win->scol += col;
    win->tsrow += row;
    win->tscol += col;
    pwb_show(win);
}

void pwb_scroll_up(PWWIN *win, int srow, int scol, int nrow, int ncol, int scrolls)
{
    PWCELL cell = (pwv_attrib(win->wtype, PWT_NORMAL) << 8) + ' ';
    int i;
    PWCELL far *fptr1;

    pwb_initwin(win);
	if (scrolls < nrow)
	{
	    pwb_initoffset(srow+1, scol+1);
    	fptr1 = _pwb_buffer+_pwb_offset;
	    pwv_moveleft(fptr1, _pwb_ncol, fptr1+(_pwb_ncol*scrolls), _pwb_ncol, nrow-scrolls, ncol);
	}
    pwb_initoffset(srow+nrow-scrolls+1, scol+1);
    fptr1 = _pwb_buffer+_pwb_offset;
    for (; scrolls; scrolls--)
    {
        for (i=ncol; i; i--)
            *(fptr1++) = cell;
        fptr1 += (_pwb_ncol-ncol);
    }
    pwb_refreshwin(win);
}

void pwb_scroll_down(PWWIN *win, int srow, int scol, int nrow, int ncol, int scrolls)
{
    PWCELL cell = (pwv_attrib(win->wtype, PWT_NORMAL) << 8) + ' ';
    int i;
    PWCELL far *fptr1;

    pwb_initwin(win);
	if (scrolls < nrow)
	{
    	pwb_initoffset(srow+nrow, scol+1);
    	fptr1 = _pwb_buffer+_pwb_offset-1;
    	pwv_moveright(fptr1, _pwb_ncol, fptr1-(_pwb_ncol*scrolls), _pwb_ncol, nrow-scrolls, ncol);
	}
    pwb_initoffset(srow+1, scol+1);
    fptr1 = _pwb_buffer+_pwb_offset;
    for (; scrolls; scrolls--)
    {
        for (i=ncol; i; i--)
            *(fptr1++) = cell;
        fptr1 += (_pwb_ncol-ncol);
    }
    pwb_refreshwin(win);
}

void pwb_scroll_left(PWWIN *win, int srow, int scol, int nrow, int ncol, int scrolls)
{
    PWCELL cell = (pwv_attrib(win->wtype, PWT_NORMAL) << 8) + ' ';
    int i, j;
    PWCELL far *fptr1;

    pwb_initwin(win);
	if (scrolls < ncol)
	{
    	pwb_initoffset(srow+1, scol+1);
    	fptr1 = _pwb_buffer+_pwb_offset;
    	pwv_moveleft(fptr1, _pwb_ncol, fptr1+scrolls, _pwb_ncol, nrow, ncol-scrolls);
	}
    pwb_initoffset(srow+1, scol+ncol-scrolls+1);
    fptr1 = _pwb_buffer+_pwb_offset;
    for (j=0; j<scrolls; j++)
        for (i=0; i<nrow; i++)
            fptr1[i*_pwb_ncol+j] = cell;
    pwb_refreshwin(win);
}

void pwb_scroll_right(PWWIN *win, int srow, int scol, int nrow, int ncol, int scrolls)
{
    PWCELL cell = (pwv_attrib(win->wtype, PWT_NORMAL) << 8) + ' ';
    int i, j;
    PWCELL far *fptr1;

    pwb_initwin(win);
	if (scrolls < ncol)
	{
    	pwb_initoffset(srow+nrow, scol+scrolls);
    	fptr1 = _pwb_buffer+_pwb_offset;
    	pwv_moveright(fptr1, _pwb_ncol, fptr1-scrolls, _pwb_ncol, nrow, ncol-scrolls);
	}
    pwb_initoffset(srow+1, scol+1);
    fptr1 = _pwb_buffer+_pwb_offset;
    for (j=0; j<scrolls; j++)
        for (i=0; i<nrow; i++)
            fptr1[i*_pwb_ncol+j] = cell;
    pwb_refreshwin(win);
}

void pwb_shellinit(void)
{
	int i;

	for (i=0; i<_pwb_nlevels; i++)
		wintable[i]->status |= STAT_DOSSHELL;
}

void pwb_shellend(void)
{
	int i;

	for (i=0; i<_pwb_nlevels; i++)
		wintable[i]->status &= (~STAT_DOSSHELL);
}

