/****************************************************************************
*
*						  Protected Mode Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: video.c $
* Version:		$Revision: 1.1 $
*
* Language:		ANSI C
* Environment:	any
*
* Description:  Test program to check the ability to generate real mode
*               interrupts and to be able to obtain direct access to the
*				video memory from protected mode. Compile and link with
*				the appropriate command line for your DOS extender.
*
*				Functions tested:	PMODE_getBIOSPointer()
*									PMODE_mapLinearPointer()
*									PMODE_freeLinearPointer()
*									PMODE_int86()
*
* $Id: video.c 1.1 1994/03/10 09:05:43 kjb release $
*
****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "pmode.h"

/* Pointer to video memory */

void *bios = NULL;
unsigned char *video_memory = NULL;

/* Routine to return the current video mode number */

int getVideoMode(void)
{
	return PMODE_getByte(bios, 0x49);
}

/* Routine to set a specified video mode */

void setVideoMode(int mode)
{
	RMREGS r;

	r.x.ax = mode;
	PMODE_int86(0x10, &r, &r);
}

/* Routine to return a pointer to the video memory area */

unsigned char *getVideoMemPointer(void)
{
	if (getVideoMode() == 0x7)
		return PMODE_mapLinearPointer(0xB0000, 0xFFFF);
	else return PMODE_mapLinearPointer(0xB8000, 0xFFFF);
}

/* Routine to clear a rectangular region on the display by calling the
 * video BIOS.
 */

void clear(int startx, int starty, int endx, int endy, unsigned char attr)
{
	RMREGS r;

	r.x.ax = 0x0600;
	r.h.bh = attr;
	r.h.cl = startx;
	r.h.ch = starty;
	r.h.dl = endx;
	r.h.dh = endy;
	PMODE_int86(0x10, &r, &r);
}

/* Routine to fill a rectangular region on the display using direct
 * video writes.
 */

#define SCREEN(x,y)	(video_memory + ((y) * 160) + ((x) << 1))

void fill(int startx, int starty, int endx, int endy, unsigned char c,
	unsigned char attr)
{
	unsigned char *v;
	int	x,y;

	for (y = starty; y <= endy; y++) {
		v = SCREEN(startx,y);
		for (x = startx; x <= endx; x++) {
			*v++ = c;
			*v++ = attr;
			}
		}
}

/* Routine to display a single character using direct video writes */

void writeChar(int x, int y, unsigned char c, unsigned char attr)
{
	unsigned char *v = SCREEN(x,y);
	*v++ = c;
	*v = attr;
}

/* Routine to draw a border around a rectangular area using direct video
 * writes.
 */

static unsigned char border_chars[] = {
	186, 205, 201, 187, 200, 188        /* double box chars */
	};

void border(int startx, int starty, int endx, int endy, unsigned char attr)
{
	unsigned char 	*v,*b;
	int				i;

	b = border_chars;
    
	for (i = starty+1; i < endy; i++) {
		writeChar(startx, i, *b, attr);
		writeChar(endx, i, *b, attr);
		}
    b++;
	for (i = startx+1, v = SCREEN(startx+1, starty); i < endx; i++) {
		*v++ = *b;
		*v++ = attr;
		}
	for (i = startx+1, v = SCREEN(startx+1, endy); i < endx; i++) {
		*v++ = *b;
		*v++ = attr;
		}
    b++;
	writeChar(startx, starty, *b++, attr);
	writeChar(endx, starty, *b++, attr);
	writeChar(startx, endy, *b++, attr);
	writeChar(endx, endy, *b++, attr);
}

void main(void)
{
	printf("Program running in ");
	switch (_PMODE_modeType) {
		case PMODE_realMode:
			printf("real mode.\n\n");
			break;
		case PMODE_286:
			printf("16 bit protected mode.\n\n");
			break;
		case PMODE_386:
			printf("32 bit protected mode.\n\n");
			break;
        }

	printf("Hit any key to start 80x25 text mode and perform some direct video output.\n");
	getch();

	setVideoMode(0x3);

	if ((bios = PMODE_getBIOSPointer()) == NULL) {
		printf("Could not obtain pointer to BIOS data area!!\n");
		exit(1);
		}

	if ((video_memory = getVideoMemPointer()) == NULL) {
		printf("Could not obtain pointer to video memory!!\n");
		exit(1);
		}

	fill(0, 0, 79, 24, 176, 0x1E);
	border(0, 0, 79, 24, 0x1F);
	getch();
	clear(0, 0, 79, 24, 0x7);

	PMODE_freeLinearPointer(video_memory);
}
