#!/bin/sh

#	TX -- A driver for the T3X compiler
#	Copyright (C) 1998,1999,2000 Nils M Holm
#	For details, see `txtrn.t'.
#	For conditions of use, see the file LICENSE.

RELEASE="T3X Release 5.12"

TX=/usr/local/t3x/r5
TXB=$TX/bin
TXL=$TX/lib
LIBS=/usr/lib/libtermcap.a
if [ -f /usr/lib/libcompat.a ]; then
	LIBS="$LIBS /usr/lib/libcompat.a"
fi
if [ -f /usr/X11R6/lib/libX11.a ]; then
	LIBS="$LIBS /usr/X11R6/lib/libX11.a"
fi
LIBS="$LIBS /usr/lib/libc.a"
S86=s86b

USAGE=\
"Usage: tx [-h] [-L] [-cknqCOPSV] [-m type] [-o file] [-l {g|v}] program"
HELP="       tx -h explains the options"
LICENSE="       tx -L prints the conditions of use"

longusage() {
cat<<EOF

-c	compile only, generate object file
-k	keep intermediate files
-l g|v	link output against the Vio or Graphics library
-m type	generate code for the given machine type
-n	generate native code (default=generate Tcode)
-o file	write output to the given file
-q	use a quicker 8086 assembler (experimental)
-C	generate ugly and inefficient C code
-O	run the Tcode optimizer
-P	run the preprocessor
-S	generate assembly language code
-V	verbose operation

Each option must be prefixed with an extra '-'.
-c, -l, -m, and -S imply -n.
Use "txx file" to run a Tcode program.

EOF

echo -n Installed native code generators:
for name in $TXB/txcg*; do
	echo -n $name | sed 's/.*txcg\(.*\)/ \1/'
done
echo .
echo
}

license() {
cat <<EOF

T3X -- A Compiler for the Procedural Language T, version 3X
Copyright (C) 1996-2000 Nils M Holm.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS \`\`AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

EOF
}

usage() {
	echo "$RELEASE"
	echo "$USAGE"
	echo "$HELP"
	echo "$LICENSE"
}


abort() {
	if [ x$OUTFILE != x ]; then
		rm -f $OUTFILE.tcode $OUTFILE.unopt $OUTFILE.s $OUTFILE.o \
			$OUTFILE.txpp
	fi
	rm -f TXPP.ERR TXTRN.ERR TXOPT.ERR TXCG.ERR
	exit 1
}

OUTFILE=""
NATIVE=n
DONTASM=n
DONTLINK=n
LIB=libtx.a
VERBOSE=+x
OPTIMIZE=n
PREP=n
RM=rm
GEN_C=n
TXOPT=txopt
if uname -r | grep -q ^3\. ; then
	MACHINE=386e
else
	MACHINE=386
fi

while true; do
	case $1 in
	-c)
		DONTLINK=y
		NATIVE=y
		shift ;;
	-k)
		RM='echo keeping: '
		shift ;;
	-l)
		if [ $# -lt 2 ]; then echo $USAGE; exit 1; fi
		LIB=libtx$2.a
		NATIVE=y
		shift; shift ;;
	-lg)
		LIB=libtxg.a
		NATIVE=y
		shift ;;
	-lv)
		LIB=libtxv.a
		NATIVE=y
		shift ;;
	-o)
		if [ $# -lt 2 ]; then echo $USAGE; exit 1; fi
		OUTFILE=$2
		shift; shift ;;
	-n)
		NATIVE=y
		shift ;;
	-q)
		S86=s86q
		shift ;;
	-m)
		if [ $# -lt 2 ]; then echo $USAGE; exit 1; fi
		MACHINE=$2
		NATIVE=y
		shift; shift ;;
	-L)
		license
		exit 0 ;;
	-C)
		GEN_C=y
		shift ;;
	-O)
		OPTIMIZE=y
		shift ;;
	-P)
		PREP=y
		shift ;;
	-S)
		DONTASM=y
		NATIVE=y
		shift ;;
	-V)
		VERBOSE=-x
		shift ;;
	-h)
		echo
		echo $RELEASE
		echo
		echo $USAGE
		longusage
		exit 0 ;;
	-*)
		echo $USAGE
		exit 1 ;;
	*)
		break ;;
	esac
done

if [ $GEN_C = y ]; then
	case $LIB in
	libtx.a)	LIB=libtxc.a ;;
	libtxv.a)	LIB=libtxvc.a ;;
	libtxg.a)	LIB=libtxgc.a ;;
	esac
fi

if [ $NATIVE = y -a ! -f $TXB/txcg$MACHINE ]; then
	echo "TX: $MACHINE: unsupported architecture"
	exit 1
fi

if [ x$1 = x ]; then
	usage
	exit 1
fi

if [ ! -f $1 ]; then
        echo "TX: $1: no such file"
        exit 1
fi

if [ x$OUTFILE = x ]; then
	OUTFILE=`basename $1 .t`
	if [ $OUTFILE = $1 ]; then
		echo "TX: input program has no .t suffix"
		exit 1
	fi
fi

rm -f TXPP.ERR TXTRN.ERR TXOPT.ERR TXCG.ERR

if [ $NATIVE = n ]; then
	set $VERBOSE
	if [ $PREP = y ]; then
		$TXB/txpp <$1 >$OUTFILE.txpp
		if [ -f TXPP.ERR ]; then abort; fi
		$TXB/txtrn <$OUTFILE.txpp >$OUTFILE
		$RM $OUTFILE.txpp
	else
		$TXB/txtrn <$1 >$OUTFILE
	fi
	if [ -f TXTRN.ERR ]; then abort; fi
	if [ $OPTIMIZE = y ]; then
		if [ `ls -l $OUTFILE | awk '{ print $5 }'` -gt 32759 ]; then
			echo 'program too big to optimize'
			abort
		fi
		mv $OUTFILE $OUTFILE.unopt
		$TXB/$TXOPT <$OUTFILE.unopt >$OUTFILE
		$RM $OUTFILE.unopt
		if [ -f TXOPT.ERR ]; then abort; fi
	fi
	exit 0
fi

set $VERBOSE

if [ $PREP = y ]; then
	$TXB/txpp <$1 >$OUTFILE.txpp
	if [ -f TXPP.ERR ]; then abort; fi
	$TXB/txtrn <$OUTFILE.txpp >$OUTFILE.tcode
	$RM $OUTFILE.txpp
	if [ -f TXTRN.ERR ]; then abort; fi
else
	$TXB/txtrn <$1 >$OUTFILE.tcode
	if [ -f TXTRN.ERR ]; then abort; fi
fi
if [ $OPTIMIZE = y ]; then
	if [ `ls -l $OUTFILE.tcode | awk '{ print $5 }'` -gt 32759 ]; then
		echo 'program too big to optimize'
		abort
	fi
	mv $OUTFILE.tcode $OUTFILE.unopt
	$TXB/$TXOPT <$OUTFILE.unopt >$OUTFILE.tcode
	$RM $OUTFILE.unopt
	if [ -f TXOPT.ERR ]; then abort; fi
fi

if [ $GEN_C = y ]; then
	$TXB/txcgc <$OUTFILE.tcode >$OUTFILE._c
	if [ -f TXCG.ERR ]; then
		rm $OUTFILE._c
		abort
	fi
	sed -n -e 's/^DECL \(.*\)/\1/p' <$OUTFILE._c >$OUTFILE.c
	cat $TXL/txcgc_wrap >>$OUTFILE.c
	grep -v ^DECL <$OUTFILE._c >$OUTFILE.c
	rm $OUTFILE._c
	if [ $DONTASM = n ]; then
		cc -o $OUTFILE $OUTFILE.c $LIBS
		$RM $OUTFILE.c
		strip $OUTFILE
	fi
	exit 0
fi

$TXB/txcg$MACHINE <$OUTFILE.tcode >$OUTFILE.s
$RM $OUTFILE.tcode
if [ -f TXCG.ERR ]; then abort; fi
if [ $DONTASM = y ]; then exit 0; fi

case $MACHINE in
	386|386e)
		as -o $OUTFILE.o $OUTFILE.s
		;;
	86a)	$S86 <$OUTFILE.s >$OUTFILE.o
		;;
	*)	echo "TX: $MACHINE: no assembler defined for this architecture"
		echo "TX: generator output has been placed in $OUTFILE.s"
		exit 1
esac
$RM $OUTFILE.s
if [ $? != 0 ]; then abort; fi
if [ $DONTLINK = y ]; then exit 0; fi

case $MACHINE in
	386|386e)
		ld -o $OUTFILE $OUTFILE.o $TXL/$LIB $LIBS
		;;
	86a)	if [ "$LIB" = "libtxg.a" ]; then
			echo "TX: -lg is not available on the 86A platform"
			abort
		fi
		echo -t exe >__ldcontrol
		echo -o $OUTFILE.exe >>__ldcontrol
		echo $OUTFILE.o >>__ldcontrol
		echo $TXL/libtx86.o >>__ldcontrol
		# The below line will be uncommented by the
		# S86KIT install script.
		#NOVIOLIB echo $TXL/libvio.o >>__ldcontrol
		sld <__ldcontrol
		rm __ldcontrol
		;;
	*)	echo "TX: $MACHINE: no loader defined for this architecture"
		echo "TX: assembler output has been placed in $OUTFILE.o"
		exit 1
esac
$RM $OUTFILE.o
if [ $? != 0 ]; then exit 1; fi

case $MACHINE in
	386|386e)
		strip $OUTFILE
		;;
esac

