/* ------------------------------------------------------------------------ */
/* System       : Library Routines                                          */
/* Program      : sound.c                                                   */
/* Description  : Sound Functions                                           */
/*                (C) CopyLeft Bill Buckels 1997.                           */
/*                All Rights Reversed.                                      */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/*                                                                          */
/*  a sound function for Microsoft C                                        */
/*                                                                          */
/* ------------------------------------------------------------------------ */
#include <dos.h>

#define TIMEOUT 0x2c00

/* freq of sound in Hertz                  */
/* duration of sound 18.2 ticks per second */

void sound(int freq, int tlen)
{
  union REGS inregs, outregs;
  unsigned frdiv;
  long lDiv;
  int seed, hiseed, hold=0, setting;

  if (tlen < 1)
    return;

  if (freq != 32767) {
    lDiv = 1331000L/freq;        /* timer divisor */
    frdiv = (unsigned)lDiv;
    outp(0x43, 0xB6) ;           /* write timer mode register */
    outp(0x42, frdiv & 0x00FF);  /* write divisor LSB */
    outp(0x42, frdiv >> 8);      /* write divisor MSB */
    setting = inp(0x61);         /* get current port B setting */
    outp(0x61,setting | 0x03);   /* turn speaker on */
  }

  if(tlen > 0) {
    tlen = ((tlen*1000) /182); /* convert from 18.2 ticks to 100ths */
    inregs.x.ax = TIMEOUT;
    int86(0x21, &inregs, &outregs);
    seed = (outregs.x.dx)&0xff;

    while(hold<tlen) {
      inregs.x.ax = TIMEOUT;
      int86(0x21,&inregs,&outregs);
      if(seed!=(outregs.x.dx&0xff)) {
        hiseed= (outregs.x.dx)&0xff;
        if (hiseed < seed)
          hiseed+=100;
        hold+=(hiseed-seed);
        seed =(outregs.x.dx)&0xff;
      }
    }
  }

  /* restore port B setting */
  if (freq != 32767)
    outp(0x61,setting);
  return;
}

void beep()
{
  sound(500,1);
  sound(1000,1);
  sound(500, 1);

}

void bronx()
{
  sound(60, 5);
  sound(40, 10);
}

#ifdef _BUILDEXE_

/* build the application if _BUILDEXE_ is defined */
/* use the make file SOUND.MAK to do so */
/* test of library function sound(frequency, duration) */

/* -------------------------------------------------------------------- */
/* SOUND.EXE                                                            */
/*                                                                      */
/* commandline utility which can be called from a batchfile to provide  */
/* a beep... or series of beeps...                                      */
/* in different frequencies and of different durations...               */
/*                                                                      */
/* binary .snd files of frequency words followed by a duration bytes    */
/* can also be used as program input.                                   */
/* the .snd file is named as a commandline argument.                    */
/*                                                                      */
/* Usage: sound [sound.snd]                                             */
/*                                                                      */
/* *or* alternately                                                     */
/*                                                                      */
/* Command Usage can be of the following forms                          */
/*                                                                      */
/* Usage 1: sound                                                       */
/*          a short beep                                                */
/* Usage 2: sound [frequency]                                           */
/*          a short beep in a "certain" frequenecy                      */
/* Usage 3: sound [frequency] [duration]                                */
/*          a beep in a "certain" frequency for a "certain" duration    */
/* Usage 4: sound [freq1] [dur1] [freq2] [dur2]...                      */
/*          beeps of "certain" frequencies for "certain" durations      */
/*          (a series of frequency duration pairs)                      */
/*                                                                      */
/* -------------------------------------------------------------------- */

#include <stdio.h>
#include <io.h>
#include <string.h>

#define FUNCKEY 0
#define ESCKEY  27

void main(int argc, char **argv)
{

  FILE *fp;
  char buffer[128];
  int frequency = 523; /* Frequency in Hertz (cycles per second) */
  int duration  = 9;   /* Duration in Clock Ticks - 18.2 ticks per second */
  int idx, ch;

  /* if a sound file is used as a command line argument */
  if (argc == 2 && (NULL != (fp = fopen(argv[1], "rb")))) {
    /* I used the number -1 to encode the logical EOF in my sound files */
    while((frequency = getw(fp)) != - 1)
    {
      if (EOF == (duration = fgetc(fp)))
        break;
      if (kbhit()) {
        if (FUNCKEY == (ch = getch()))
          getch();
        if (ESCKEY == ch)
          break;
      }
      if (duration != 0)
        sound(frequency, duration);
    }
    fclose(fp);
  }
  else if (argc > 3) {                     /* a series of beeps */
    for (idx = 2; idx < argc; idx++) {
      frequency = atoi(argv[idx - 1]);
      duration  = atoi(argv[idx]);
      idx++;
      sound(frequency, duration);
    }
  }
  else {                                   /* a single beep */
    if (argc > 2) {
      duration = atoi(argv[2]);
    }
    if (argc > 1) {
      frequency = atoi(argv[1]);
    }
    sound(frequency, duration);
  }
  exit(0);

}

#endif
