/***************************************************************************

	SETUP.C program to read the current configuration of PCREMOT2 and allow
	the user to modify the configuration.

	Compiled with Microsoft C version 5.1 and 6.0

 ***************************************************************************/

#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <bios.h>

#define TRUE  1
#define FALSE 0

struct config {
   char cManned_flag;      /* 0-unattended, 1-manned mode */
   char cComm_flag;        /* 0-comm1, 1-comm2, 2-comm3, 3-comm4 */
   int  iComm_port3;       /* port address for comm port 3 */
   int  iComm_port4;       /* port address for comm port 4 */
   char cComm3_int;        /* interrupt number for comm port 3 */
   char cComm4_int;        /* interrupt number for comm port 4 */
   char cSpeed_flag;       /* 0-1200, 1-2400, 2-4800, 3-9600 */
   char cDesnow_flag;      /* 0-NO desnow, 1-desnow */
   char cNull_modem;       /* 0-modem, 1-null modem cable */
   char cSmile_flag;       /* 0-NO smile, 1-smile, you're on candid camera */
   char cZero1;
	unsigned char cClear_code;	/* alt key combination to clear the screen */
   char cZero2;
   unsigned char cExit_code;  /* alt key combination to exit pcremote */
   char cZero3;
   unsigned char cShell_code; /* alt key combination to shell to DOS */
   char cZero4;
   unsigned char cTransfer_code;/* alt key combination to start a file transfer */
   int  iPassword_size;    /* length of the password */
   char crgPassword[20];   /* secret password */
   char szModem_setup5[30];   /* custom modem setup string */
	char szDial[4];			/* dial string */
} stOldConfig, stNewConfig;

unsigned char Ascii_code[] = "QWERTYUIOPASDFGHJKLZXCVBNMQ    ";
unsigned char Scan_code[] = { 16,17,18,19,20,21,22,23,24,25,30,
										31,32,33,34,35,36,37,38,44,45,46,47,
										48,49,50,16 };

char szMannedPrompt[]   = "1  - Host/Remote mode.       ";
char szCommPrompt[]     = "2  - Communications port.    ";
char szSpeedPrompt[]    = "3  - Communications speed.   ";
char szDesnowPrompt[]   = "4  - Enable/disable desnow.  ";
char szNullPrompt[]     = "5  - Modem/Null modem.       ";
char szSmilePrompt[]    = "6  - Enable/disable smile.   ";
char szClearPrompt[]    = "7  - Redisplay screen key.   ";
char szExitPrompt[]     = "8  - Exit PCREMOT2 key.      ";
char szShellPrompt[]    = "9  - Shell to DOS key.       ";
char szTransferPrompt[] = "10 - File transfer key.      ";
char szPasswordPrompt[] = "11 - Password.               ";
char szSetupPrompt[]    = "12 - Modem setup string.     ";
char szDialPrompt[]		= "13 - Tone/Pulse dial.        ";
char szSaveExit[]       = "14 - Save new configuration. ";
char szNosaveExit[]     = "15 - Exit, do not save.      ";

char szPcremoteFile[]  = "pcremot2.com";    /* name of the pcremote file */
char szZcopyFile[]     = "zcopy.com";       /* name of the zcopy file */

#define PCREMOTEOFFSET 3   /* offset from start of file to config offset */
#define ZCOPYOFFSET 6      /* offset from start of file to config offset */

int iPcremoteOffset;     /* offset from start of file to config */
int iZcopyOffset;        /* offset from start of file to config */

char szSelection[40];	/* buffer to get the user selection */
int  iSelection;			/* user selections converted to int */

/* function declarations */
int ReadPcremoteConfig(void);
int WritePcremoteConfig(void);
int WriteZcopyConfig(void);
int Old2New(void);
int DisplayConfig(void);
int GetAltKey(char *);
int GetNewCommPort(void);
int GetNewBaudRate(void);
int GetNewPassword(void);
int GetNewModemSetup(void);

main()
{
   /* open the pcremote.com file and read the offset to the configuration */
   if(ReadPcremoteConfig()) return(1);
   Old2New();
   while(TRUE)
   {
      DisplayConfig();
		gets(szSelection);	/* get response from the user */
		iSelection=atoi(szSelection);	/* convert to a number */
		switch(iSelection)
		{
			case 1:
				stNewConfig.cManned_flag = stNewConfig.cManned_flag ? (char)0 : (char)1 ;
				break;
			case 2:
				GetNewCommPort();
				break;
			case 3:
				GetNewBaudRate();
				break;
			case 4:
				stNewConfig.cDesnow_flag = stNewConfig.cDesnow_flag ? (char)0 : (char)1 ;
				break;
			case 5:
				stNewConfig.cNull_modem = stNewConfig.cNull_modem ? (char)0 : (char)1 ;
				break;
			case 6:
				stNewConfig.cSmile_flag = stNewConfig.cSmile_flag ? (char)0 : (char)1 ;
				break;
			case 7:
				GetAltKey(&stNewConfig.cClear_code);
				break;
			case 8:
				GetAltKey(&stNewConfig.cExit_code);
				break;
			case 9:
				GetAltKey(&stNewConfig.cShell_code);
				break;
			case 10:
				GetAltKey(&stNewConfig.cTransfer_code);
				break;
			case 11:
				GetNewPassword();
				break;
			case 12:
				GetNewModemSetup();
				break;
			case 13:
				if(stNewConfig.szDial[3] == 'T')
					stNewConfig.szDial[3] = 'D';
				else
					stNewConfig.szDial[3] = 'T';
				break;
			case 14:
				WritePcremoteConfig();
				return(0);
			case 15:
				return(0);
		}
   }
}

int ReadPcremoteConfig()
{
   int hFile;  /* handle to the open pcremote.com file */
   if((hFile=open(szPcremoteFile, O_RDONLY|O_BINARY))==-1)  /* open the pcremote.com file */
   {
      printf("\nUnable to open PCREMOT2.COM\n");
      return(1);
   }
   if(lseek(hFile,(long) PCREMOTEOFFSET, SEEK_SET)==-1)  /* locate the config offset */
   {
      printf("\nUnable to read PCREMOT2 configuration.\n");
      close(hFile);
      return(1);
   }
   if(read(hFile,(char *) &iPcremoteOffset, 2)!=2)   /* read the config offset */
   {
      printf("\nUnable to read PCREMOT2 configuration.\n");
      close(hFile);
      return(1);
   }
   iPcremoteOffset+=PCREMOTEOFFSET;  /* adjust for the start of the file */
   if(lseek(hFile,(long) iPcremoteOffset, SEEK_SET)==-1)  /* locate the config */
   {
      printf("\nUnable to read PCREMOT2 configuration.\n");
      close(hFile);
      return(1);
   }
   /* read the old configuration */
   if(read(hFile,(char *) &stOldConfig, sizeof(struct config))!=sizeof(struct config))
   {
      printf("\nUnable to read PCREMOT2 configuration.\n");
      close(hFile);
      return(1);
   }
   close(hFile);  /* read the configuration so close the file */
   return(0);
}

int WritePcremoteConfig()
{
   int hFile;  /* handle to the open pcremote.com file */
   if((hFile=open(szPcremoteFile, O_WRONLY|O_BINARY))==-1)  /* open the pcremote.com file */
   {
      printf("\nUnable to open PCREMOT2.COM.\n");
      return(1);
   }
   if(lseek(hFile,(long) iPcremoteOffset, SEEK_SET)==-1)  /* locate the config */
   {
      printf("\nUnable to open PCREMOT2.COM.\n");
      close(hFile);
      return(1);
   }
   /* write the new configuration */
   if(write(hFile,(char *) &stNewConfig, sizeof(struct config))!=sizeof(struct config))
   {
      printf("\nUnable to write PCREMOT2 configuration.\n");
      close(hFile);
      return(1);
   }
   close(hFile);  /* wrote the configuration so close the file */
	/* if comm port 3 or 4 is selected then copy the port configuration
		to zcopy.com */
	if(stNewConfig.cComm_flag==2 || stNewConfig.cComm_flag==3)
		WriteZcopyConfig();
   return(0);
}

int WriteZcopyConfig()
{
   int hFile;  /* handle to the open zcopy.com file */
   if((hFile=open(szZcopyFile, O_RDWR|O_BINARY))==-1)  /* open the zcopy.com file */
   {
      printf("\nUnable to open ZCOPY.COM\n");
      return(1);
   }
   if(lseek(hFile,(long) ZCOPYOFFSET, SEEK_SET)==-1)  /* locate the config offset */
   {
      printf("\nUnable to read ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   if(read(hFile,(char *) &iZcopyOffset, 2)!=2)   /* read the config offset */
   {
      printf("\nUnable to read ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   iZcopyOffset+=(ZCOPYOFFSET+6);  /* adjust for the start of the file */
   if(lseek(hFile,(long) iZcopyOffset, SEEK_SET)==-1)  /* locate the config */
   {
      printf("\nUnable to read ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   /* write the new configuration */
   if(write(hFile,(char *) &stNewConfig.iComm_port3, 2)!=2)
   {
      printf("\nUnable to write ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   if(write(hFile,(char *) &stNewConfig.cComm3_int, 1)!=1)
   {
      printf("\nUnable to write ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   if(write(hFile,(char *) &stNewConfig.iComm_port4, 2)!=2)
   {
      printf("\nUnable to write ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
   if(write(hFile,(char *) &stNewConfig.cComm4_int, 1)!=1)
   {
      printf("\nUnable to write ZCOPY configuration.\n");
      close(hFile);
      return(1);
   }
	close(hFile);  /* read the configuration so close the file */
   return(0);
}

int Old2New()
/* copy the current configuration to the new configuration parameters */
{
   stNewConfig.cManned_flag = stOldConfig.cManned_flag ;
   stNewConfig.cComm_flag = stOldConfig.cComm_flag ;
   stNewConfig.iComm_port3 = stOldConfig.iComm_port3 ;
   stNewConfig.iComm_port4 = stOldConfig.iComm_port4 ;
   stNewConfig.cComm3_int = stOldConfig.cComm3_int ;
   stNewConfig.cComm4_int = stOldConfig.cComm4_int ;
   stNewConfig.cSpeed_flag = stOldConfig.cSpeed_flag ;
   stNewConfig.cDesnow_flag = stOldConfig.cDesnow_flag ;
   stNewConfig.cNull_modem = stOldConfig.cNull_modem ;
   stNewConfig.cSmile_flag = stOldConfig.cSmile_flag ;
   stNewConfig.cClear_code = stOldConfig.cClear_code ;
   stNewConfig.cExit_code = stOldConfig.cExit_code ;
   stNewConfig.cShell_code = stOldConfig.cShell_code ;
   stNewConfig.cTransfer_code = stOldConfig.cTransfer_code ;
   stNewConfig.iPassword_size = stOldConfig.iPassword_size ;
	stNewConfig.cZero1=stOldConfig.cZero1;
	stNewConfig.cZero2=stOldConfig.cZero2;
	stNewConfig.cZero3=stOldConfig.cZero3;
	stNewConfig.cZero4=stOldConfig.cZero4;
   strncpy(stNewConfig.crgPassword, stOldConfig.crgPassword, sizeof(stOldConfig.crgPassword));
   strncpy(stNewConfig.szModem_setup5, stOldConfig.szModem_setup5, sizeof(stOldConfig.szModem_setup5));
	strncpy(stNewConfig.szDial, stOldConfig.szDial, sizeof(stOldConfig.szDial));
   return(0);
}

int DisplayConfig()
/* display the new configuration parameters */
{
	int iLoop;
	char *psTemp;
   char szBuffer[80];   /* temporary buffer to assemble a line */
   printf("\n\n\n\n\n\n\n");   /* blank the screen the easy way */
	printf("PCREMOT2 configuration program.\nModifies the default configuration.\n\n");
   /* display the manned flag */
   strcat(strcpy(szBuffer, szMannedPrompt), stNewConfig.cManned_flag ? "manned" : "host");
   printf("%s\n", szBuffer);
   /* display the current communications port */
   strcpy(szBuffer, szCommPrompt);
   sprintf(strchr(szBuffer,0), "%d", stNewConfig.cComm_flag+1);
   printf("%s\n", szBuffer);
   /* display the current speed flag */
   strcpy(szBuffer, szSpeedPrompt);
   switch(stNewConfig.cSpeed_flag)
   {
      case 0:
         strcat(szBuffer, "1200 baud");
         break;
      case 1:
         strcat(szBuffer, "2400 baud");
         break;
      case 2:
         strcat(szBuffer, "4800 baud");
         break;
      case 3:
         strcat(szBuffer, "9600 baud");
         break;
      case 4:
         strcat(szBuffer, "19200 baud");
         break;
      case 5:
         strcat(szBuffer, "38400 baud");
         break;
   }
   printf("%s\n", szBuffer);
   /* display the current desnow flag */
   strcat(strcpy(szBuffer, szDesnowPrompt), stNewConfig.cDesnow_flag ? "enabled" : "disabled");
   printf("%s\n", szBuffer);
   /* display the current null modem flag */
   strcat(strcpy(szBuffer, szNullPrompt), stNewConfig.cNull_modem ? "null modem" : "modem");
   printf("%s\n", szBuffer);
   /* display the current smile flag */
   strcat(strcpy(szBuffer, szSmilePrompt), stNewConfig.cSmile_flag ? "enabled" : "disabled");
   printf("%s\n", szBuffer);
	/* display the clear alt key code */
	for(iLoop=0;iLoop<sizeof(Scan_code)&& Scan_code[iLoop]!=stNewConfig.cClear_code; iLoop++);
	strcat(strcpy(szBuffer, szClearPrompt),"Alt-");
	*(strchr(szBuffer,0)+1)=0;
	*strchr(szBuffer,0)=Ascii_code[iLoop];
   printf("%s\n", szBuffer);
	/* display the exit pcremote alt key */
	for(iLoop=0;iLoop<sizeof(Scan_code)&& Scan_code[iLoop]!=stNewConfig.cExit_code; iLoop++);
	strcat(strcpy(szBuffer, szExitPrompt),"Alt-");
	*(strchr(szBuffer,0)+1)=0;
	*strchr(szBuffer,0)=Ascii_code[iLoop];
   printf("%s\n", szBuffer);
	/* display the shell to DOS alt key */
	for(iLoop=0;iLoop<sizeof(Scan_code)&& Scan_code[iLoop]!=stNewConfig.cShell_code; iLoop++);
	strcat(strcpy(szBuffer, szShellPrompt),"Alt-");
	*(strchr(szBuffer,0)+1)=0;
	*strchr(szBuffer,0)=Ascii_code[iLoop];
   printf("%s\n", szBuffer);
	/* display the file transfer alt key */
	for(iLoop=0;iLoop<sizeof(Scan_code)&& Scan_code[iLoop]!=stNewConfig.cTransfer_code; iLoop++);
	strcat(strcpy(szBuffer, szTransferPrompt),"Alt-");
	*(strchr(szBuffer,0)+1)=0;
	*strchr(szBuffer,0)=Ascii_code[iLoop];
   printf("%s\n", szBuffer);
	/* display the password, can't really display it though */
	strcpy(szBuffer, szPasswordPrompt);
	psTemp=strchr(szBuffer,0);	/* find the end of the string */
	for(iLoop=0;iLoop<stNewConfig.iPassword_size; iLoop++, psTemp++)
		*psTemp='*';
	*psTemp=0;	/* terminate the string */
   printf("%s\n", szBuffer);
	/* display the custom modem setup5 sting */
	strcat(strcpy(szBuffer, szSetupPrompt), stNewConfig.szModem_setup5);
   printf("%s\n", szBuffer);
	/* display the tone or dial pulse prompt */
	if(stNewConfig.szDial[3]=='T')
	   strcat(strcpy(szBuffer, szDialPrompt), "Tone");
	else 
	   strcat(strcpy(szBuffer, szDialPrompt), "Pulse");
   printf("%s\n", szBuffer);
	/* display the operator prompts */
	strcat(strcpy(szBuffer,"\n"), szSaveExit);
   printf("%s\n", szBuffer);
	strcat(strcpy(szBuffer,""), szNosaveExit);
   printf("%s\n", szBuffer);
   printf("\nSelection: ");
   return(0);
}

int GetAltKey(cpNewKey)
char *cpNewKey;	/* place to store new alt key value */
{
	char cTemp;	/* temporary character key */
	unsigned iTemp;
	int iLoop;
	while(TRUE)
	{
		printf("\n\nEnter the new alt-key combination: ");
		iTemp = _bios_keybrd(_KEYBRD_READ);
		if((iTemp&0xff)==0)
		{
			cTemp=(char) (iTemp>>8)&0xff;
			/* make sure the key is a valid key in the scan code array */
			for(iLoop=0;iLoop<sizeof(Scan_code)&&Scan_code[iLoop]!=cTemp;iLoop++);
			if(iLoop!=sizeof(Scan_code))	/* then it's a valid alt key */
			{
				*cpNewKey=cTemp;
				return(0);
			}
			else printf("\nYou must use alpha keys only.");
		}
		if(cTemp==13) return(0);	/* check for carriage return key */
		printf("\nYou must hold down the ALT key when you\nmake your selection.");
	}
	return(0);
}

int GetNewCommPort()
{
	char cTemp;	/* temporary comm port number */
	while(TRUE)
	{
		printf("\nEnter the new port number: ");
		gets(szSelection);
		if(strlen(szSelection)==9) return(0);	/* changed his mind */
		cTemp=(char) atoi(szSelection);
		if(cTemp>=1 && cTemp<=4)	/* make sure it's within range */
		{
			stNewConfig.cComm_flag=cTemp-1;
			if(cTemp==3)	/* check for comm 3 */
			{
				printf("\nCurrent Port address is %4X hex.", stNewConfig.iComm_port3);
				printf("\nEnter new port address or press Enter for no change: ");
				gets(szSelection);
				if(strlen(szSelection)) sscanf(szSelection, "%X", &stNewConfig.iComm_port3);
				printf("\nCurrent interrupt is %d.", stNewConfig.cComm3_int);
				printf("\nEnter the new interrupt or press Enter for no change: ");
				gets(szSelection);
				if(strlen(szSelection)) stNewConfig.cComm3_int=(char) atoi(szSelection);
			}
			if(cTemp==4)	/* check for comm 4 */
			{
				printf("\nCurrent Port address is %4X hex.", stNewConfig.iComm_port4);
				printf("\nEnter new port address or press Enter for no change: ");
				gets(szSelection);
				if(strlen(szSelection)) sscanf(szSelection, "%X", &stNewConfig.iComm_port4);
				printf("\nCurrent interrupt is %d.", stNewConfig.cComm4_int);
				printf("\nEnter the new interrupt or press Enter for no change: ");
				gets(szSelection);
				if(strlen(szSelection)) stNewConfig.cComm4_int=(char) atoi(szSelection);
			}
			return(0);
		}
		printf("\nPort number must be 1, 2, 3, or 4.");
	}
	return(0);
}

int GetNewBaudRate()
{
	char cTemp;	/* temporary baud rate number */
	while(TRUE)
	{
		printf("\n1 for 1200 baud\n2 for 2400 baud\n3 for 4800 baud\n4 for 9600 baud");
		printf("\n5 for 19200 baud\n6 for 38400 baud");
		printf("\n\nEnter the new baud rate number: ");
		gets(szSelection);
		if(strlen(szSelection)==0) return(0);	/* changed his mind */
		cTemp=(char) atoi(szSelection);
		if(cTemp>=1 && cTemp<=6)	/* make sure it's within range */
		{
			stNewConfig.cSpeed_flag=cTemp-1;
			return(0);
		}
		printf("\nBaud selection must be 1, 2, 3, 4, 5, or 6.");
	}
	return(0);
}

int GetNewPassword()
{
	printf("\nEnter new password, maximum 20 characters: ");
	gets(szSelection);
	strupr(szSelection);	/* the password must be uppercase */
	if(strlen(szSelection)==0) return(0);	/* changed his mind */
	stNewConfig.iPassword_size=strlen(szSelection);
	strncpy(stNewConfig.crgPassword, szSelection, sizeof(stNewConfig.crgPassword));
	return(0);
}

int GetNewModemSetup()
{
	printf("\nEnter modem setup string, maximum 30 characters: ");
	gets(szSelection);
	if(strlen(szSelection)==0) return(0);	/* changed his mind */
	if(strlen(szSelection)<29)
		strcat(strcpy(stNewConfig.szModem_setup5, szSelection), "\r");
	return(0);
}

/* end of file SETUP.C */

