//  ____________________________________________________
// |                                                    |
// |  Project:     POWER VIEW INTERFACE                 |
// |  File:        PVINI.CPP                            |
// |  Compiler:    WPP386 (10.6)                        |
// |                                                    |
// |  Subject:     INI files processor implementation   |
// |                                                    |
// |  Author:      Emil Dotchevski                      |
// |____________________________________________________|
//
// E-mail: zajo@geocities.com
// URL:    http://www.geocities.com/SiliconValley/Bay/3577

#ifndef NOINI

#define uses_conio
#define uses_ctype
#define uses_errno
#define uses_io
#define uses_fcntl
#define uses_stdarg
#define uses_stdio
#define uses_string
#define uses_ini
#define uses_system

#include "PVuses.h"

#define EC_BAD_INI 223

class Tini
{
  public:
    boolean valid;
    boolean report;
    char *filename;
    char buffer[256];
    uint cur_line;
    uint cur_section_ofs;
    uint cur_section_line;
    char *cur_section;
    Tini( char *_filename );
    virtual ~Tini( void );
    virtual void seek( uint line, uint offset )=0;
    virtual uint get_seed( void )=0;
    virtual char *next_line( void )=0;
    boolean seek_section( boolean fatal, char *section );
    char *seek_var( boolean fatal, char *var );
};

class Tini_ro : public Tini
{
  public:
    FILE *ini_file;
    Tini_ro( char *_filename );
    virtual ~Tini_ro( void );
    virtual void seek( uint line, uint offset );
    virtual uint get_seed( void );
    virtual char *next_line( void );
};

class Tini_rw : public Tini, public Ttext_editor
{
  public:
    uint cur_line_ptr;
    Tini_rw( char *_filename );
    virtual void seek( uint line, uint offset );
    virtual uint get_seed( void );
    virtual char *next_line( void );
};

/*
Tini
*/
  Tini::Tini( char *_filename )
  {
    report = 1;
    filename = STRDUP(min_path(fexpand(strcpy(buffer,_filename))));
  }

  Tini::~Tini( void )
  {
    FREE( filename );
  }

  boolean Tini::seek_section( boolean fatal, char *section )
  {
    uint i;
    cur_section = NULL;
    for( i=0; i<2; i++ )
    {
      while( next_line()!=NULL )
        if( stricmp( buffer, section )==0 )
        {
          cur_section = section;
          cur_section_ofs = get_seed();
          cur_section_line = cur_line;
          return 1;
        }
      seek( 0, 0 );
      cur_section_line = cur_line;
    }
#ifdef CYR
    ini_err( fatal, " %s   ", section );
#else
    ini_err( fatal, "section %s not found", section );
#endif
    return 0;
  }

  char *Tini::seek_var( boolean fatal, char *var )
  {
    uint i;
    for( i=0; i<2; i++ )
    {
      while( next_line()!=NULL && *buffer!='[' )
      {
        char *n;
        char *p = strchr( buffer, '=' );
#ifdef CYR
        ini_lnerr( p==NULL, " " );
#else
        ini_lnerr( p==NULL, "syntax error" );
#endif
        *p = 0; n = p;
        while( *buffer && *--n==' ' ) *n = 0;
        if( stricmp( var, buffer )==0 )
        {
          strcpy( buffer, p+1 );
          return buffer;
        }
      }
      seek( cur_section_line, cur_section_ofs );
    }
#ifdef CYR
    ini_err( fatal, " '%s'   ,  %s", var, cur_section );
#else
    ini_err( fatal, "variable '%s' not found, section %s", var, cur_section );
#endif
    return NULL;
  }


/*
Tini_ro
*/
  Tini_ro::Tini_ro( char *_filename ):
    Tini( _filename )
  {
    ini_file = fopen( filename, "rt" );
    valid = ini_file!=NULL;
    if( valid ) seek( 0, 0 );
  }

  Tini_ro::~Tini_ro( void )
  {
    fclose( ini_file );
  }

  void Tini_ro::seek( uint line, uint offset )
  {
#ifdef CYR
    ini_err( fseek( ini_file, offset, SEEK_SET )!=0, " " );
#else
    ini_err( fseek( ini_file, offset, SEEK_SET )!=0, "seek failed" );
#endif
    cur_line=line;
  }

  uint Tini_ro::get_seed( void )
  {
    return ftell( ini_file );
  }

  char *Tini_ro::next_line( void )
  {
    loop:
      if( errno=0, fgets( buffer, sizeof(buffer), ini_file )!=NULL )
      {
        char *p = strchr( buffer, ';' );
        cur_line++;
        if( p != NULL ) *p = 0;                  //cut comments
        p = strchr( buffer, '\n' );
        if( p != NULL ) *p = 0;                  //cut eol
        p = strchr( buffer, 0 );
        while( p>buffer && *(--p)==' ' ) *p = 0; //cut eol spaces
        if( *buffer==0 ) goto loop;
        return strupr( buffer );
      }
#ifdef CYR
    ini_err( errno!=EZERO, "    ini " );
#else
    ini_err( errno!=EZERO, "can't read ini file" );
#endif
    return NULL;
  }


/*
Tini_rw
*/
  Tini_rw::Tini_rw( char *_filename ):
    Tini( _filename ),
    Ttext_editor( 4096, NULL )
  {
    set_name( _filename );
    valid = load();
    if( valid ) seek( 0, 0 );
  }

  #pragma off( unreferenced )
  void Tini_rw::seek( uint line, uint offset )
  {
    set_cur_ptr( offset, 0 );
    cur_line = cur_pos_y;
  }
  #pragma on( unreferenced )

  uint Tini_rw::get_seed( void )
  {
    return cur_ptr;
  }

  char *Tini_rw::next_line( void )
  {
    loop:
      if( cur_ptr>=buf_len ) return NULL;
      get_line_str( buffer, sizeof(buffer) );
      cur_line_ptr = cur_ptr;
      set_cur_xy( 0, ++cur_line, 0 );
      char *p = strchr( buffer, ';' );
      if( p != NULL ) *p = 0;                  //cut comments
      p = strchr( buffer, '\n' );
      if( p != NULL ) *p = 0;                  //cut eol
      p = strchr( buffer, 0 );
      while( p>buffer && *(--p)==' ' ) *p = 0; //cut eol spaces
      if( *buffer==0 ) goto loop;
      return strupr( buffer );
  }


static Tini *cur_ini=NULL;
#define INI_RO  ((Tini_ro *)cur_ini)
#define INI_RW  ((Tini_rw *)cur_ini)

FILE *read_ini( char *filename )
{
  close_ini();
  cur_ini = NEW( Tini_ro( filename ) );
  if( !cur_ini->valid )
  {
    close_ini();
    return NULL;
  }
  return INI_RO->ini_file;
}

Ttext_editor *open_ini( char *filename )
{
  close_ini();
  cur_ini = NEW( Tini_rw( filename ) );
  if( !cur_ini->valid )
  {
    close_ini();
    return NULL;
  }
  return (Ttext_editor *)INI_RW;
}

void close_ini( void )
{
  if( cur_ini!=NULL ) DELETE( cur_ini );
  cur_ini = NULL;
}

boolean seek_section( boolean fatal, char *section )
{
  if( cur_ini==NULL ) return 0;
  return cur_ini->seek_section( fatal, section );
}

char *seek_var( boolean fatal, char *var )
{
  if( cur_ini==NULL ) return NULL;
  return cur_ini->seek_var( fatal, var );
}

boolean ini( char *var, boolean &x, boolean _default )
{
  long l;
  x=_default;
  if( !ini( var, l, (long)_default ) ) return 0;
#ifdef CYR
  ini_lnerr( l<0 || l>1, "  0  1" );
#else
  ini_lnerr( l<0 || l>1, "0 or 1 expected" );
#endif
  x=l;
  return 1;
}

boolean ini( char *var, long &x, long _default )
{
  char *v;
  x=_default;
  if( cur_ini==NULL || cur_ini->cur_section==NULL || (v=seek_var(0,var))==NULL ) return 0;
#ifdef CYR
  ini_lnerr( sscanf(v,"%d",&x)!=1, "   " );
#else
  ini_lnerr( sscanf(v,"%d",&x)!=1, "Integer expected" );
#endif
  return 1;
}

#ifndef NOFLOAT
boolean ini( char *var, double &x, double _default )
{
  char *v;
  x=_default;
  if( cur_ini==NULL || cur_ini->cur_section==NULL || (v=seek_var(0,var))==NULL ) return 0;
#ifdef CYR
  ini_lnerr( sscanf(v,"%lf",&x)!=1, "   " );
#else
  ini_lnerr( sscanf(v,"%lf",&x)!=1, "Floating point expected" );
#endif
  return 1;
}
#endif

boolean set_ini( char *var, char *value, ... )
{
  va_list argptr;
  char buf[256];
  if( seek_var( 0, var )==NULL ) return 0;
  va_start( argptr, value );
  vsprintf( buf, value, argptr );
  va_end( argptr );
  INI_RW->set_select( INI_RW->line_start(INI_RW->cur_line_ptr), INI_RW->line_end(INI_RW->cur_line_ptr), 0 );
  INI_RW->insert_string( var, 0 );
  INI_RW->insert_string( "=", 0 );
  INI_RW->insert_string( buf, 0 );
  return 1;
}

boolean save_ini( void )
{
  return INI_RW->save();
}

void ini_err( boolean flag, char *msg, ... )
{
  va_list argptr;
  char buf[256];

  if( !cur_ini->report ) return;
  va_start( argptr, msg );
  vsprintf( buf, msg, argptr );
  va_end( argptr );
  if( flag )
  {
#ifdef CYR
    cprintf( "  %s: %s\r\n", cur_ini->filename, buf );
#else
    cprintf( "Error in %s: %s\r\n", cur_ini->filename, buf );
#endif
    exit( EC_BAD_INI );
  }
}

void ini_lnerr( boolean flag, char *msg, ... )
{
  va_list argptr;
  char buf[256];

  if( !cur_ini->report ) return;
  va_start( argptr, msg );
  vsprintf( buf, msg, argptr );
  va_end( argptr );
  if( flag )
  {
#ifdef CYR
    cprintf( "  %s: %s,  %d\r\n", cur_ini->filename, buf, cur_ini->cur_line );
#else
    cprintf( "Error in %s: %s, line %d\r\n", cur_ini->filename, buf, cur_ini->cur_line );
#endif
    exit( EC_BAD_INI );
  }
}

void ini_errs( boolean report )
{
  cur_ini->report=report;
}

char *ini_section( void )
{
  return cur_ini->cur_section;
}

uint ini_line( void )
{
  return cur_ini->cur_line;
}

#endif //NOINI
