/*
 *  Routines for sampling from mu-law encoded audio devices (/dev/audio)
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "freq.h"

#ifdef SC_MULAW

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#ifdef LINUX_MIXER
 #include <sys/soundcard.h>
#endif
#include "extern.h"

/* Function prototypes */
void reset_mulaw_sc(void);
void halt_mulaw_sc(void);
void cleanup_mulaw_sc(void);
void recordblock_mulaw_sc(void *buffer);
void set_mixer_mulaw_sc(int mix,int level);

int _fp_audio=0;
int _fp_output=-1;

short mulaw[256];

int ml_devmask,ml_stereodev,ml_devcaps,ml_exclusive_input=1;
int ml_pcmlevel=0;

char *rec_buffer;

void init_mulaw_sc(void)
{
   #ifdef LINUX_MIXER
      int parm,mix;
   #endif
   int i;

   DOUT("Setting up mu-law soundcard pointers");
   reset_soundcard=reset_mulaw_sc;
   halt_soundcard=halt_mulaw_sc;
   cleanup_soundcard=cleanup_mulaw_sc;
   recordblock=recordblock_mulaw_sc;
   set_mixer=NULL;
   mixers=0;
   SampleRate=8000;
   sample_size=16;

   /* If the audio device is undefined, define the default */
   if(audio_device==NULL)
      audio_device="/dev/audio";

   /* If the audio device begins with /dev/dsp, flag it */
   if(strncmp(audio_device,"/dev/dsp",8)==0)
   {
      printf("You selected %s for MULAW input, changing this to /dev/audio\n",audio_device);
      audio_device="/dev/audio";
   }

   /* Open and initialize the AUDIO device */
   DOUT("Opening the AUDIO device");
   _fp_audio=open(audio_device,O_RDONLY);
   if(_fp_audio<=0)
   {
      printf("Unable to open the audio device: %s\n",audio_device);
      exit(1);
   }

   /* Open and initialize the OUTPUT device */
   if(output_device!=NULL)
   {
      DOUT("Opening the OUTPUT device");
      _fp_output=open(output_device,O_WRONLY | O_CREAT);
      if(_fp_output<0)
      {
	 printf("Error opening output device %s\n",output_device);
	 exit(1);
      }
   }

   /* 
    *  Set up mu-law conversion table
    */
   for(i=0;i<256;i++)
   {
      short mu,e,f,y;
      short etab[]={0,132,396,924,1980,4092,8316,16764};

      mu=255-i;
      e=(mu&0x70)/16;
      f=mu&0x0f;
      y=f*(1<<(e+3));
      y=etab[e]+y;
      if(mu&0x80)
	 y=-y;
      mulaw[i]=y;
   }
   
   /* Set up a recording buffer */
   rec_buffer=(char *)malloc(MAX_LEN);
   if(rec_buffer==NULL)
   {
      puts("Error allocating memory in init_mulaw_sc()");
      exit(1);
   }

   #ifdef LINUX_MIXER
      /* If the mixer device is undefined, set the default */
      if(mixer_device==NULL)
	 mixer_device="/dev/mixer";

      DOUT("Checking out the mixer");
      mix=open(mixer_device,O_RDWR);
      if(mix >= 0)
      {
	 DOUT("A mixer device does exist");
	 ioctl(mix,SOUND_MIXER_READ_DEVMASK,&ml_devmask);
	 ioctl(mix,SOUND_MIXER_READ_STEREODEVS,&ml_stereodev);
	 ioctl(mix,SOUND_MIXER_READ_CAPS,&ml_devcaps);
	 
	 /* Get and save the PCM level, then set it to 0 */
	 /* This prevents the clicking noises when buffers start and stop */
	 ioctl(mix,SOUND_MIXER_READ_PCM,&ml_pcmlevel);
	 parm=0;
	 ioctl(mix,SOUND_MIXER_WRITE_PCM,&parm);
	 
	 if(ml_devcaps&SOUND_CAP_EXCL_INPUT)
	 {
	    ml_exclusive_input=1;
	    DOUT("Mixer requires exclusive inputs");
	 }
	 else
	 {
	    int src=SOUND_MASK_LINE|SOUND_MASK_CD|SOUND_MASK_MIC;
	    ml_exclusive_input=0;
	    DOUT("Mixer allows multiple simultaneous inputs");
	    ioctl(mix,SOUND_MIXER_WRITE_RECSRC,&src);
	 }
	 if(ml_devmask&SOUND_MASK_LINE)
	 {
	    int level,left,right=0;
	    DOUT("Found a Line mixer");
	    ioctl(mix,SOUND_MIXER_READ_LINE,&level);
	    left=level&0x7f;
	    if(ml_stereodev&SOUND_MASK_LINE)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    ext_level=left; 
	    mixers=1;
	 }
	 if(ml_devmask&SOUND_MASK_MIC)
	 {
	    int level,left,right=0;
	    DOUT("Found a Mic mixer");
	    ioctl(mix,SOUND_MIXER_READ_MIC,&level);
	    left=level&0x7f;
	    if(ml_stereodev&SOUND_MASK_MIC)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    mic_level=left; 
	    mixers=1;
	 }
	 if(ml_devmask&SOUND_MASK_CD)
	 {
	    int level,left,right=0;
	    DOUT("Found a CD mixer");
	    ioctl(mix,SOUND_MIXER_READ_CD,&level);
	    left=level&0x7f;
	    if(ml_stereodev&SOUND_MASK_CD)
	    {
	       right=(level>>8)&0x7f;
	       DOUT("   in stereo");
	    }
	    if(right>left) left=right; /* Use the larger */
	    int_level=left; 
	    mixers=1;
	 }
	 DOUT("Closing the mixer device");
	 close(mix);
      }
      if(mixers)
         set_mixer=set_mixer_mulaw_sc;
   #endif /* LINUX_MIXER */
}

void reset_mulaw_sc(void)
{
   int i;

   SampleRate=8000;    /* audio device only has one sample rate */
   sample_size=16;

   DOUT("Initializing the buffers");
   /* Reset the buffer pointers */
   queue_buffer=0;     /* Pointer to next buffer to be queued */
   record_buffer=0;    /* Pointer to next buffer to be filled */
   process_buffer=0;   /* Pointer to next buffer to be FFTed  */

   for(i=0;i<BUFFERS;i++)
      flag[i]=0;
   /*
    *  This function starts the DMA process.
    */
   recordblock_mulaw_sc(buffer[queue_buffer]);
   if(++queue_buffer>=BUFFERS)
      queue_buffer=0;
}


void halt_mulaw_sc(void)
{
}


void cleanup_mulaw_sc(void)
{
   #ifdef LINUX_MIXER
      if(mixers)
      {
	 /* Restore the PCM mixer level */
	 int mixer=open(mixer_device,O_RDWR);
	 if(mixer>=0) 
	 {
	    ioctl(mixer,SOUND_MIXER_WRITE_PCM,&ml_pcmlevel);
	    close(mixer);
	 }
      }
   #endif /* LINUX_MIXER */

   close(_fp_audio);
   if(_fp_output>=0)
      close(_fp_output);
      
   free(rec_buffer);
}

void recordblock_mulaw_sc(void *buffer)
{
   int i;
   short *ptr=(short *)buffer;
   unsigned char *ptr2=rec_buffer;
  
   read(_fp_audio,rec_buffer,fftlen);

   /* Copy to the output device, if requested */
   if(_fp_output>=0)
      write(_fp_output,rec_buffer,fftlen);

   /* Convert to 16-bit linear */
   for(i=0;i<fftlen;i++)
      *(ptr++)=mulaw[(int)*(ptr2++)];

   /* Toggle the buffer-filled flag */
   flag[record_buffer]=1;
   if(++record_buffer>=BUFFERS)
      record_buffer=0;   
}

#ifdef LINUX_MIXER

void set_mixer_mulaw_sc(int mix,int level)
{
   /* Set the mixer level for the Linux mixer device */
   int mixer=open(mixer_device,O_RDWR);
   int src=SOUND_MASK_LINE|SOUND_MASK_CD|SOUND_MASK_MIC;

   DOUT("Opening the mixer device");
   if(mixer>=0)
   {
      if((mix==MIXER_EXT) && (ml_devmask&SOUND_MASK_LINE))
      {
	 DOUT("Setting the external line level");
	 if(ml_stereodev&SOUND_MASK_LINE)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_LINE,&level);
	 if(ml_exclusive_input)
	    src=SOUND_MASK_LINE;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      else if((mix==MIXER_INT) && (ml_devmask&SOUND_MASK_CD))
      {
	 DOUT("Setting the internal line (CD) level");
	 if(ml_stereodev&SOUND_MASK_CD)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_CD,&level);
	 if(ml_exclusive_input)
	    src=SOUND_MASK_CD;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      else if((mix==MIXER_MIC) && (ml_devmask&SOUND_MASK_MIC))
      {
	 DOUT("Setting the mic level");
	 if(ml_stereodev&SOUND_MASK_MIC)
	    level+=level*256;
	 /* Set the mixer level, select this as the active recording source */
	 ioctl(mixer,SOUND_MIXER_WRITE_MIC,&level);
	 if(ml_exclusive_input)
	    src=SOUND_MASK_MIC;
	 ioctl(mixer,SOUND_MIXER_WRITE_RECSRC,&src);
      }
      DOUT("Closing the mixer device");
      close(mixer);
   }
}
#endif /* LINUX_MIXER */
#endif /* SC_MULAW */



