/*
 *  Routines for sampling from VESA audio AI drivers
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "specgram.h"

#ifdef SC_VESA

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <alloc.h>
#include <mem.h>

#include "display.h"
#include "extern.h"
#include "vesa_aud.h"

/* Function prototypes */
void reset_vesa(void);
void halt_vesa(void);
void cleanup_vesa(void);
void recordblock_vesa(void *);

int vesa_record_buffer;
void far *vesa_buffer[2];

unsigned long bufferlen=0;
unsigned long bufferdiv=0;
VESAHANDLE hWave=0;
GeneralDeviceClass gdc;  /* Receives a copy of the VESA driver info block */
fpWAVServ wfn=NULL;      /* Pointer to wave functions */
char far *memptr=NULL;   /* Pointer to memory buffer used by the VESA driver */
//BLOCKHANDLE hBlock[2] = {0,0};
/* tables for reporting attached volume info and services */
fpVolServ vsptr;

/* Same as in SB.C, but duplicated here to keep the code independant */
void far *vesa_aligned_malloc(long len)
{
   void far *ptr;
   unsigned int far *orig_ptr;
   unsigned seg,orig_seg,orig_off;
   long lin_addr;

   ptr=farmalloc(len*2+32);
   if(ptr!=NULL)
   {
      /* Compute a new pointer to the buffer such that the next "len" bytes */
      /* do not cross a page boundary, and the offset is zero */
      /* (as required by DMA transfers) */
      orig_off=FP_OFF(ptr);
      orig_seg=FP_SEG(ptr);
      /* reserve 4 bytes for the original pointer */
      lin_addr=(orig_seg*16L)+orig_off+4L;
      if((lin_addr&0xF0000L) != ((lin_addr+len-1)&0xF0000L))
         lin_addr=(lin_addr+len-1)&0xF0000L;
      else
         lin_addr=(lin_addr+15)&0xFFFF0L;

      seg=(unsigned int)(lin_addr/16);
      orig_ptr=(unsigned far *)MK_FP(seg-1,0x000C);
      orig_ptr[0]=orig_off;
      orig_ptr[1]=orig_seg;
      ptr=MK_FP(seg,0);
/*
      printf("Original: %04x:%04x, New: %04x:%04x, Linear: %05lx\n",
      orig_seg,orig_off,FP_SEG(ptr),FP_OFF(ptr),lin_addr);
      getch();
*/
   }
   return ptr;
}

void vesa_aligned_free(void far *ptr)
{
   if(ptr!=NULL)
   {
      unsigned far *old_ptr=(unsigned far *)MK_FP(FP_SEG(ptr)-1,0x000c);
      /*
      printf("Freeing: %04x:%04x, Ptr: %04x:%04x\n",
      FP_SEG(old_ptr),FP_OFF(old_ptr),FP_SEG(ptr),FP_OFF(ptr));
      */
      farfree(MK_FP(old_ptr[1],old_ptr[0]));
   }
}

/* Interrupt handler for DMA complete message from VESA driver */
void far pascal Callback(VESAHANDLE han,void far *fptr,long len,long filler)
{
   /* Set up the Data Segment register */
   _asm {
      push  ds
      mov   ax,seg buffer_full
      mov   ds,ax
   }

   buffer_full=1;

   /* If the driver requests, let it prepare the buffer */
   if(gdc.u.gdwi.wifeatures & WAVEPREPARE)
      (wfn->wsWavePrepare)(0,sample_size,0,fptr,len);

   _asm {
      pop   ds
   }
}

/* Null callback function */
void far pascal NullFunc(VESAHANDLE han,void far *fptr,long len,long filler)
{
}

void init_vesa(void)
{
   int volsize=0;

   hWave = VESAFindADevice(WAVDEVICE);
   if(hWave==0)
   {
      draw_cleanup_graphics();
      puts("The VESA audio AI is not installed.");
      exit(1);
   }
   if(VESAQueryDevice(hWave,VESAQUERY2,&gdc) == 0)
   {
      VESACloseDevice(hWave);
      draw_cleanup_graphics();
      puts("Cannot query the installed AI device.");
      exit(1);
   }
   if(gdc.gdclassid != WAVDEVICE)
   {
      VESACloseDevice(hWave);
      draw_cleanup_graphics();
      puts("The VESA audio AI returned a non-DAC device.");
      exit(1);
   }
   /* Print some GDC information */
   /* PrintGeneralDeviceInfo(&gdc,hWave); */

   if((memptr = (char far *)AllocateBuffer(gdc.u.gdwi.wimemreq))==NULL)
   {
      VESACloseDevice(hWave);
      draw_cleanup_graphics();
      puts("There is not enough memory for the VESA audio device.");
      exit(1);
   }
   if((wfn = (fpWAVServ)VESAOpenADevice(hWave,0,memptr)) == NULL)
   {
      VESACloseDevice(hWave);
      draw_cleanup_graphics();
      puts("Unable to open the installed audio device.");
      exit(1);
   }

   /* Check for supported modes */
   if(gdc.u.gdwi.wiSampleSize&WAVE16BITREC)
      sample_size=16;
   else
      sample_size=8;
   /* Check if stereo mode is supported */
   if(!(gdc.u.gdwi.wifeatures&(WAVESR8K|WAVESR11K|WAVESR22K|WAVESR44K|WAVEVARIRSTER)))
   {
      stereo=0;   /* Force mono mode */
      stereo_diff=0;
   }

   /* Allocate buffers for the sampling */
   vesa_buffer[0]=vesa_aligned_malloc((long)MAX_LEN*(stereo+1)*(sample_size/8));
   vesa_buffer[1]=vesa_aligned_malloc((long)MAX_LEN*(stereo+1)*(sample_size/8));
   if((vesa_buffer[0]==NULL) || (vesa_buffer[1]==NULL))
   {
      draw_cleanup_graphics();
      puts("Error allocating memory in init_vesa()");
      exit(1);
   }

   /* Set up the callback routines */
   wfn->wsApplRSyncCB = &Callback;
   wfn->wsApplPSyncCB = &NullFunc;

   /* Set up soundcard function pointers */
   reset_soundcard=reset_vesa;
   halt_soundcard=halt_vesa;
   cleanup_soundcard=cleanup_vesa;
   recordblock=recordblock_vesa;
   set_mixer=NULL;
   mixers=0;
}

void reset_vesa(void)
{
   /* Initialize VESA audio AI stuff */
   int i;
   long l;

   /* Verify the SampleRate variable */
   l=(wfn->wsDeviceCheck)(WAVESAMPLERATE,SampleRate);
   if(!l)
      SampleRate=5000;
   else
      SampleRate=l;

   /* Select the stereo mode, sampling rate, and bits per sample */
   (wfn->wsPCMInfo)(stereo+1,SampleRate,0,0,sample_size);
   bufferlen=(long)fftlen*(sample_size/8)*(stereo+1);
   bufferdiv=bufferlen/4;

   /* Reset the buffer pointers */
   vesa_record_buffer=0;
   buffer_full=0;

//   if(hBlock[0]) (wfn->wsWaveRegister)(0,hBlock[0]);
//   hBlock[0] = (wfn->wsWaveRegister)(vesa_buffer[0],fftlen*(sample_size/8)*(stereo+1));
//   if(hBlock[1]) (wfn->wsWaveRegister)(0,hBlock[1]);
//   hBlock[1] = (wfn->wsWaveRegister)(vesa_buffer[1],fftlen*(sample_size/8)*(stereo+1));

   /* This function starts the DMA process. */
//   (wfn->wsRecordBlock)(hBlock[vesa_record_buffer],0);
   (wfn->wsRecordCont) (vesa_buffer[vesa_record_buffer],bufferlen,bufferdiv);
}

void halt_vesa(void)
{
   (wfn->wsStopIO)(0);
}

void cleanup_vesa(void)
{
   /* Clean up the VESA audio AI */
   int i,stat;

   if(wfn)
   {
      (wfn->wsStopIO)(0);
//      if(hBlock[0]) (wfn->wsWaveRegister)(0,hBlock[0]);
//      if(hBlock[1]) (wfn->wsWaveRegister)(0,hBlock[1]);
      if(vesa_buffer[0]) vesa_aligned_free(vesa_buffer[0]);
      if(vesa_buffer[1]) vesa_aligned_free(vesa_buffer[1]);
      VESACloseDevice(hWave);
   }
   if(memptr)
   {
      FreeBuffer(memptr);
      memptr=NULL;
   }
   if(vsptr)
   {
      FreeBuffer(vsptr);
      vsptr=NULL;
   }
}

void recordblock_vesa(void *buffer)
{
   /* Start recording a new buffer. */
//   (wfn->wsRecordBlock)(hBlock[vesa_record_buffer],0);
   /* Advance recorded count if buffer is already full */
   while(!buffer_full && !done);
   buffer_full=0;

   vesa_record_buffer=1-vesa_record_buffer;
   (wfn->wsRecordCont) (vesa_buffer[vesa_record_buffer],bufferlen,bufferdiv);
   _fmemcpy(buffer,vesa_buffer[1-vesa_record_buffer],bufferlen);
}

void set_mixer_vesa(int mix,int level)
{
}

#endif

