/*****************************************************************************
 DTIFF.C - DTIFF, Display Tag Image File Format.
 Written by Tom Cervenka for Late Night Software, (c) 1988
 All Rights Reserved.
 *****************************************************************************/

/* #define DEBUG */
/* #define NOGRAPH */
/* #define EDEBUG */

#ifdef NOGRAPH
#define TESTXRES 639
#define TESTYRES 479
#endif

#define ON 0
#define OFF 1

/* SetOff() sets bit p to 1 in byte buffer */
#define SetOff(buffer,p) (*(buffer+(p/8))|=(128>>(p%8)))
/* IsOn() = 1 if the bit p in buffer buf is reset to 0 */
#define IsOn(buf,p)   ( !(*(buf+(p/8)) & (128>>(p%8))) )
/* IsPixOn() = 1 if the bit p in buffer buf is set to 1 */
#define IsPixOn(buf,p)   ( (*(buf+(p/8)) & (128>>(p%8))) )

#include <stdio.h>
#include <stdlib.h>
#include <tiff.h>   /* Header file for TIFF routines form DEST TACS */
#include <apptag.h> /* Another */
#include <fcntl.h>
#include <io.h>
#include <alloc.h>
#include <graphics.h>
#include <conio.h>
#include <ctype.h>
#include <string.h>

#define FULRES_TYPE 1    /* More TIFF STUFF */
#define SINGLE_PLANE 1   /* ""              */

#define TRUE 1
#define FALSE 0

#define ONE 1L

/* Function Prototypes */

void main(int argc,char *argv[]);
void close_tiff(int fh);
void AsIs(void);
void FitAll(void);
void Syntax(void);
void Help(void);
int Lastline(int y,int yres);
int FitIt(int fullres, int maxres, int *xs);
int FindOn(char *buf, int i, int res);
void MaskIt(char *buf, int res, int max, int *offs);

const char doc[]={"List 'DTIFF.DOC' for details."};

int fh,gdriver=DETECT,gmode,ErrorCode,MaxX,MaxY;
int rows=0,cols=0,strips=0;
long rowsize;

int rot=0,bestfit=TRUE,bkcolor=WHITE,pixcolor=BLUE;

int vgax=FALSE,vgaxx=720,vgaxy=512; /* Hi-res option for VGA */

LONGPTR imagebuf;

struct palettetype pal;

void main(int argc,char *argv[])
{

 int i,lomode,himode;
 char *cmd;
 char tifname[128];
 tifname[0]=0;

/* Evaluate Arguments */

 if(argc<=1)
	{
	Help();
	exit(0);
	}

 if(argc>6)
 	{
	printf("Too many parameters.");
	Syntax();
	exit(1);
	}

/* 1st arg is file name */

 strcat(tifname,argv[1]);
 if(!strchr(tifname,'.')) strcat(tifname,".tif");

 fh=open(tifname,O_TEXT|O_BINARY);
 if(fh==-1)
	{
	perror("Can't open the TIF file.");
     Syntax();
	exit(1);
	}

/* the rest of the args are options */

 for(i=2;i<argc;i++)
	{
	cmd=argv[i];
	if(*cmd==*"/")
		{
		cmd++;
		switch (tolower(*cmd)) {
                                         /* Rotate */
			case 'r'	:	cmd++;
						rot=atoi(cmd);
						if(rot!=0 && rot!=90 && rot!=180 && rot!=270)
							{
							printf("Bad rotation.");
							Syntax();
							close(fh);
							exit(1);
							}
                              break;
                                                 /* AS IS */
			case	'a'	:    if(strlen(cmd)>1)
							{
							printf("Bad option.");
							Syntax();
							close(fh);
							exit(1);
							}
						bestfit=0;
						break;
                                               /* Use Graphics Driver d */
			case 'd'  :	cmd++;
						if(!stricmp(cmd,"ATT")) gdriver=ATT400;
						if(!stricmp(cmd,"ATT400")) gdriver=ATT400;
/*						if(!stricmp(cmd,"PC3270")) gdriver=PC3270; */
						if(!stricmp(cmd,"VGA")) gdriver=VGA;
						if(!stricmp(cmd,"VGAX")) {gdriver=VGA;vgax=TRUE;}
						if(!stricmp(cmd,"HERC")) gdriver=HERCMONO;
						if(!stricmp(cmd,"IBM8514")) gdriver=IBM8514;
						if(!stricmp(cmd,"EGAMONO")) gdriver=EGAMONO;
						if(!stricmp(cmd,"EGA64")) gdriver=EGA64;
						if(!stricmp(cmd,"EGA")) gdriver=EGA;
						if(!stricmp(cmd,"MCGA")) gdriver=MCGA;
						if(!stricmp(cmd,"CGA")) gdriver=CGA;
						if(gdriver==DETECT)
							{
							printf("Bad graphics driver selected.");
							Syntax();
							close(fh);
							exit(1);
							}
						break;
									  /* hi-res vga horizontal resolution */
			case 'h'	:	cmd++;
						vgaxx=abs(atoi(cmd));
                              break;
									  /* hi-res vga vertical resolution */
			case 'v'	:	cmd++;
						vgaxy=abs(atoi(cmd));
                              break;

			default	:	printf("Bad option.");
						Syntax();
						close(fh);
						exit(1);
			}
		}
	else
		{
		printf("Bad parameter.");
		Syntax();
		close(fh);
		exit(1);
		}
     }

#ifdef DEBUG
 printf("rotation = %d\n%s\n",rot,bestfit?"bestfit":"as is");
 getch();
#endif

/* Read IMAGE WIDTH (Columns) TAG */

 if(read_tag(fh,1,T_IMG_WIDTH,(long)&cols,(short)sizeof(cols))==0)
	{
 	printf("No tag for columns. %s\n",doc);
     close_tiff(fh);
	close(fh);
	exit(1);
     }

/* Read IMAGE LENGTH (Rows) TAG */

 if(read_tag(fh,1,T_IMG_LENGTH,(long)&rows,(short)sizeof(rows))==0)
	{
 	printf("No tag for rows. %s\n",doc);
     close_tiff(fh);
 	close(fh);
	exit(1);
     }

 rowsize=((long)cols+7)/8;   /* Convert columns (bits) to bytes */

 imagebuf=farmalloc(rowsize);  /* Alloc row buffer */
 if(imagebuf==NULL)
 	{
	printf("Not enough RAM for image. %s\n",doc);
	close_tiff(fh);
	close(fh);
	exit(1);
	}

#ifdef DEBUG

 printf("Number of cols (horz) = %u\n",cols);
 printf("Number of rows (vert) = %u\n",rows);
 printf("Width of 1 row in bytes = %u\n",rowsize);
 getch();

#endif

#ifndef NOGRAPH

/* detect graphics hardware is adapter was not specified */

 if(gdriver==DETECT)
	 detectgraph(&gdriver,&gmode);

#ifdef DEBUG
 printf("coreleft=%u\n",coreleft());
#endif

 ErrorCode=-1;

/* Register the Driver */

 switch (gdriver)
	{

	case CGA		:
	case MCGA		:   	ErrorCode=registerbgidriver(CGA_driver);
					break;
	case EGA		:
	case EGA64	:
	case EGAMONO	:
	case VGA		:	ErrorCode=registerbgidriver(EGAVGA_driver);
					break;
	case HERCMONO	:    ErrorCode=registerbgidriver(Herc_driver);
					break;
	case ATT400	:	ErrorCode=registerbgidriver(ATT_driver);
					break;
/*	case PC3270	:	ErrorCode=registerbgidriver(PC3270_driver);
					break; */
	case IBM8514	:    ErrorCode=registerbgidriver(IBM8514_driver);
					break;

	}

#ifdef DEBUG
 printf("coreleft=%u\n",coreleft());
#endif


 if(ErrorCode<0)
     {
 	printf("Graphics System Error: Can't register driver. %s\n",doc);
     farfree(imagebuf);
     close_tiff(fh);
	close(fh);
    	exit( 1 );
	}

/* Set the graphics mode for the driver, we want highest spatial resolution */

 switch (gdriver)
	{

	case CGA		:    gmode=CGAHI; break;
	case MCGA		:   	gmode=MCGAHI; break;
	case EGA		:    gmode=EGAHI; break;
	case EGA64	:    gmode=EGA64HI; break;
	case EGAMONO	:    gmode=EGAMONOHI; break;
	case VGA		:	gmode=VGAHI; break;
	case HERCMONO	:    gmode=HERCMONOHI; break;
	case ATT400	:	gmode=ATT400HI; break;
	case IBM8514	:    gmode=IBM8514HI; break;
	default		:;
	}


 /* Startup BGI */

 initgraph(&gdriver,&gmode,"");
 ErrorCode = graphresult();		/* Read result of initialization*/
 if( ErrorCode != grOk )		/* Error occured during init	*/
     {
 	printf("Graphics System Error: %s\n %s\n",grapherrormsg(ErrorCode),doc);
     farfree(imagebuf);
     close_tiff(fh);
	close(fh);
    	exit(1);
	}

 if(vgax)
	{
	char *vgaxmsg;
	if((vgaxmsg=vgaxset(vgaxx,vgaxy,0)))
		{
		closegraph();
		printf("Graphics System Error: %s\n %s\n",vgaxmsg,doc);
		farfree(imagebuf);
		close_tiff(fh);
		close(fh);
		exit(1);
		}
	}

 MaxX=getmaxx();
 MaxY=getmaxy();

 if(vgax)
	{
	MaxX=vgaxx-1;
	MaxY=vgaxy-1;
	}

#else
 MaxX=TESTXRES;
 MaxY=TESTYRES;
#endif

#ifdef DEBUG
 if(gdriver==HERCMONO)
	{
#ifndef NOGRAPH
	restorecrtmode();
#endif
	}

 printf("screen cols=%d screen rows=%d\n",MaxX,MaxY);
 getch();

 if(gdriver==HERCMONO)
	{
#ifndef NOGRAPH
	setgraphmode(getgraphmode());  /* Set the mode */
#endif
	}
#endif

#ifndef NOGRAPH
 if(gdriver==EGA64 || gdriver==EGA || gdriver==VGA)
	{
     getpalette(&pal);             /* Change palette to get background light */
     setpalette(0,EGA_WHITE);      /* and image dark for EGA/VGA systems */
     setpalette(15,EGA_BLACK);
	setpalette(1,EGA_BLACK);
	pixcolor=15; if(gdriver==EGA64) pixcolor=1;
	}
 else
     if(gdriver!=HERCMONO && gdriver!=CGA && gdriver!=MCGA && gdriver!=EGAMONO)
		{
		setbkcolor(bkcolor);  /* Change background color for others */
		}
 if(gdriver==HERCMONO || gdriver==CGA || gdriver==ATT400 || gdriver==MCGA || gdriver==EGAMONO)
	{
	setcolor(1); if(gdriver==EGAMONO) setcolor(3);
	rectangle(0,0,getmaxx(),getmaxy());
	setfillstyle(SOLID_FILL,1); if(gdriver==EGAMONO) setfillstyle(SOLID_FILL,3);
	floodfill(1,1,1); if(gdriver==EGAMONO) floodfill(1,1,3);
	pixcolor=0;
	setcolor(0);
	}
 else
	{
	/* if(!vgax) */ cleardevice();
	}
#endif

 if(bestfit)
	FitAll();
 else
	AsIs();
 getch();
#ifndef NOGRAPH
 if(gdriver==EGA64 || gdriver==EGA || gdriver==VGA)
	 setallpalette(&pal);
 closegraph();
#endif
 farfree(imagebuf);
 close_tiff(fh);
 close(fh);
}

/*****************************************************************************
 AsIs() displays image starting at col 0 to col (MaxCols of adapter) and from
 row 0 to row (MaxRows of adapter). If image is larger than adapter, image
 will not be completely displayed
 ****************************************************************************/

void AsIs(void)
{
 int linecount,x,y,i,xres,yres,xstart,xmove,ymove;
 long startpos;
 unsigned char k,mask,pix;

/* set coordinates depending on rotation
   xres=column resolution
   yres=row resolution
   xstart=starting column
   y=starting row
   xmove=value to change xstart after pixel is written
   ymove=value to change ystart after pixel is written */

 switch (rot)
	{
 	case 0	:	(cols>MaxX) ? (xres=MaxX) : (xres=cols);
				(rows>MaxY) ? (yres=MaxY) : (yres=rows);
				xstart=0;
				y=0;
				xmove=1;
				ymove=1;
				break;

	case 90	:	(cols>MaxY) ? (xres=MaxY) : (xres=cols);
				(rows>MaxX) ? (yres=MaxX) : (yres=rows);
				xstart=xres-1;
				y=0;
				xmove=-1;
				ymove=1;
				break;

	case 180	:	(cols>MaxX) ? (xres=MaxX) : (xres=cols);
				(rows>MaxY) ? (yres=MaxY) : (yres=rows);
				xres=MaxX;
				yres=MaxY;
				xstart=xres-1;
				y=yres-1;
				xmove=-1;
				ymove=-1;
				break;

	case 270	:	(cols>MaxY) ? (xres=MaxY) : (xres=cols);
				(rows>MaxX) ? (yres=MaxX) : (yres=rows);
				xstart=0;
				y=yres-1;
				xmove=1;
				ymove=-1;
				break;
	}

#ifdef DEBUG
 printf("xres=%d yres=%d y=%d xstart=%d xmove=%d ymove=%d",
	    xres,yres,y,xstart,xmove,ymove);
 getch();
#endif

 linecount=rows; /* counts down to zero */
 startpos=0;     /* row to read starts at 0 */

/* display image row by row until 1) image is done or 2) no more rows left on
   the display or 3) a key is hit */

 while(linecount>0 && Lastline(y,yres) && !kbhit())
 	{
	if(read_image(fh,FULRES_TYPE,SINGLE_PLANE, /* read image 1 row at a time */
		      startpos,ONE,
	              imagebuf,rowsize)==NULL)
        	{
		farfree(imagebuf);
          getch();
          closegraph();
		close_tiff(fh);
          close(fh);
		printf("Error reading image line %u\n%s\n",rows-linecount+1,doc);
		exit(1);
		}
	x=xstart;  /* reset starting column before column loop */

/* Column loop */

	for(i=0;i<=xres/8;i++) /* byte loop */
		for(k=0,mask=128;k<8;k++,mask=mask>>1)  /* bit loop */
			{
			pix=*(imagebuf+i);
			pix=pix & mask;
			if(pix)
#ifndef NOGRAPH
				{
				if(gdriver==ATT400 || gdriver==IBM8514 || gdriver==MCGA || gdriver==EGAMONO)
					(rot==0 || rot==180) ? (putpixel(x,y,pixcolor)) :
								   (putpixel(y,x,pixcolor)) ;
				else
					{
					if(gdriver==VGA || gdriver==EGA64 || gdriver==EGA)
						{
						(rot==0 || rot==180) ? (vgaxputpixel(x,y,pixcolor)) :
										   (vgaxputpixel(y,x,pixcolor)) ;
						}
					else
						{
						if(gdriver==HERCMONO)
							{
							(rot==0 || rot==180) ? (hercputpixel(x,y,pixcolor)) :
											   (hercputpixel(y,x,pixcolor)) ;
							}
						else
							{
							(rot==0 || rot==180) ? (cgaputpixel(x,y,pixcolor)) :
											   (cgaputpixel(y,x,pixcolor)) ;
							}
						}
					}
				}
#endif
#ifdef DEBUG
#ifdef NOGRAPH
 printf("%s\n",(rot==0 || rot==180) ? "(putpixel(x,y,BLUE)" :
							   "(putpixel(y,x,BLUE)") ;
 if(getch()==*"s") exit(0);
#endif
#endif
			x+=xmove; /* increment column position */
			}
	y+=ymove;    /* increment row position */
	linecount--;  /* next line */
	startpos++;   /*    ""     */
	}
}

/*****************************************************************************
 Returns FALSE if value row position y is out of bounds for adapter.
 This condition depends on the rotation of the image
 ****************************************************************************/

int Lastline(int y,int yres)
{
 if(rot==90 || rot==180)
     {
	if (y>=0) return(TRUE);
	else return(FALSE);
     }
 else
	{
	if(y<=yres) return(TRUE);
	else return(FALSE);
	}
}

/* Fitall() displays the entire image on the display by masking out columns and
   rows until it fits in the bounds of the adapter. Fitall() doesn't mask if the
   image fits in the display. */

void Fitall(void)
{

 char *cbuf,*rbuf;
 int linecount,x,y,i,cres,rres,xstart,xmove,ymove,rcnt,coffs,roffs,cmax,rmax;
 long startpos;


 cbuf=calloc(cols+1,sizeof(char));  /* alloc column mask buffer */
 if(cbuf==NULL)
     {
	farfree(imagebuf);
     closegraph();
	close_tiff(fh);
     close(fh);
	printf("Can't allocate space for the column mask.\n%s\n",doc);
	exit(1);
	}
 coffs=0;

 rbuf=calloc(rows+1,sizeof(char)); /* alloc row mask buffer */
 if(rbuf==NULL)
     {
	farfree(imagebuf);
     closegraph();
	close_tiff(fh);
     close(fh);
	printf("Can't allocate space for the row mask.\n%s\n",doc);
     free(cbuf);
	exit(1);
	}
 roffs=0;

/* set coordinates depending on rotation
   cmax=column resolution
   rmax=row resolution
   xstart=starting column
   y=starting row
   xmove=value to change xstart after pixel is written
   ymove=value to change ystart after pixel is written */

 switch (rot)
	{
 	case 0	:	cmax=MaxX+1;
				rmax=MaxY+1;
				xstart=0;
				y=0;
				xmove=1;
				ymove=1;
				break;

	case 90	:	cmax=MaxY+1;
				rmax=MaxX+1;
				xstart=cmax;
				y=0;
				xmove=-1;
				ymove=1;
				break;

	case 180	:	cmax=MaxX+1;
				rmax=MaxY+1;
				xstart=cmax;
				y=rmax;
				xmove=-1;
				ymove=-1;
				break;

	case 270	:	cmax=MaxY+1;
				rmax=MaxX+1;
				xstart=0;
				y=rmax;
				xmove=1;
				ymove=-1;
				break;
	}

 MaskIt(cbuf,cols,cmax,&coffs); /* build column mask */
 MaskIt(rbuf,rows,rmax,&roffs); /* build row mask */

 linecount=rows;
 startpos=0;
 rcnt=0;     /* row mask buffer position */

/* display image row by unmaked row until done or key is hit */

 while(linecount>0 && !kbhit())
     {
	if(IsOn(rbuf,rcnt))  /* display row if not masked */
		{
		if(read_image(fh,FULRES_TYPE,SINGLE_PLANE,  /* read one row at a time */
				    startpos,ONE,
	     	         imagebuf,rowsize)==NULL)
	        	{
			farfree(imagebuf);
			getch();
	          closegraph();
			close_tiff(fh);
	          close(fh);
			printf("Error reading image line %u\n%s\n",rows-linecount+1,doc);
			exit(1);
			}
          x=xstart;            /* reset column position */
		for(i=0;i<=cols;i++) /* column loop */
			{
			if(IsOn(cbuf,i)) /* check pixel if not masked */
				{
				if(IsPixOn(imagebuf,i)) /* display pixel if on */
#ifndef NOGRAPH
					{
					if(gdriver==ATT400 || gdriver==IBM8514 || gdriver==MCGA || gdriver==EGAMONO )
						(rot==0 || rot==180) ? (putpixel(x,y,pixcolor)) :
										   (putpixel(y,x,pixcolor)) ;
					else
						{
						if(gdriver==VGA || gdriver==EGA64 || gdriver==EGA)
							{
							(rot==0 || rot==180) ? (vgaxputpixel(x,y,pixcolor)) :
											   (vgaxputpixel(y,x,pixcolor)) ;
							}
						else
							{
							if(gdriver==HERCMONO)
								(rot==0 || rot==180) ? (hercputpixel(x,y,pixcolor)) :
												   (hercputpixel(y,x,pixcolor)) ;
							else
								{
								(rot==0 || rot==180) ? (cgaputpixel(x,y,pixcolor)) :
												   (cgaputpixel(y,x,pixcolor)) ;
								}
							}
						}
					}
#endif
				x+=xmove; /* next column */
				}
			}
		y+=ymove; /* next row */
		}
	rcnt++; /* next row mask buffer position */
	linecount--; /* next image row */
	startpos++; /*  next image row */
	}

 free(rbuf);  /* dealloc row mask buffer */
 free(cbuf);  /* dealloc col mask buffer */

#ifdef DEBUG
 getch();
 closegraph();
 printf("\n%d columns, Skipped %d , DisplayedONs=%d, picture=%d\n\n",
	    cmax,coffs,cols-coffs,cols);
 printf("\n%d rows, Skipped %d , DisplayedONs=%d, picture=%d\n\n",
	    rmax,roffs,rows-roffs,rows);
 getch();
#endif

}

int FitIt(int fullres, int maxres, int *xs)
{
/*****************************************************************************
 Provides information necessary to squeeze a large picture onto a small screen
 Input:
		fullres - resolution of the picture in dots or pixels
		maxres  - maximum resolution (in one direction) of the display
 Output:
		*xs - number of picture dots in a row to skip when displaying.
			 this is repeated for entire picture. For example, if *xs=2,
			 you should skip 2 dots then display one for the entire picture.
			 if *xs=0 then no dots should be skipped this way.

 Return:
		the number of dots to display before skipping 1 dot. again this
		pattern is reapeted for the length of the picture.

		**NOTE** if the return value==0, then no dots should be skipped this
		way.
*****************************************************************************/

 int b,quo,rem,ll,c1,c2;

 *xs=b=0; /*Init to no skips*/

 quo=fullres/maxres; /* quo is the fraction of the picture that'll fit on the */
 rem=fullres%maxres; /* screen. rem>0 if there are dots left over */

 if(quo>1)
	*xs=quo-1; /* changes into the number of repeating skips - if quo=3 we */
 else           /* want to skip 2, display 1 */
	if(quo==0) quo++; /* quo==1 means that the whole picture fits on the */
				   /* screen with the possible exception of rem dots. */

 if(rem) /* check for left over dots */
	{
     ll=fullres/quo;
	rem=ll-maxres;              /* set rem to the number of not skipped by
                                  *xs and subtract maxres to find out how
						    many more dots must be skipped */

	if(rem>0) b=ll/rem;                /* then divide this into the remaining
								lines to evenly distribute the
								leftovers if there are any */
     c1=ll-(ll/(b+1));
	if(c1>maxres)
		{
		c2=ll-(ll/b);
		if(abs(maxres-c2)<abs(maxres-c1)) b--;
		}

	}

 return(b);
}


int FindOn(char *buf, int i, int res)
{
 char mask;

 if(i<0) return(i); /* invalid dot */

 do	{

     /* each dot is stored in a byte in the buffer, this is how to find the
	   value of the bit that corresponds to the dot (i), if mask==0 then the
	   dot will be diplayed (ON) else if mask==1 the dot will be skipped
	   because if is (OFF);                                                */

     mask=*(buf+(i/8)) & (128>>(i%8));

	i++; /* next dot */
	}
 while(mask!=ON && i<=res+1); /* search all dots */

 if(mask==ON && i<=res+1)    /* valid ON dot */
	{
	i--;                    /* fixup dot position */
	return(i);
	}
 else
	return(-9);	/* no ON dots were found so return some negative number */

} 												/* end of FindOn */

void MaskIt(char *buf, int res, int max, int *offs)
{
 int blk,cnt,skp;
 register i;

 blk=FitIt(res,max,&skp); /* get the info about dots to skip */

 if(blk==0 && skp==0)
	return;			/* nothing to skip, assume all dots are ON */

 if(skp)	/* repeating skips */
	{

	i=0; /* first dot */
	while(i<=res && i>=0)	/* go through all dots */
		{

		for(cnt=0;cnt<skp && (i>=0 && i<=res);cnt++)
			{
			i=FindOn(buf,i,res); /* find the next ON dot */

			if(i>=0)             /* check for a valid dot */
				{
				SetOff(buf,i);   /* set the dot OFF */

				(*offs)++;      /* increment OFF count */
				i++;            /* i = next dot */
				}
               }
		i=FindOn(buf,i,res); /* find the next ON dot */
		i++;				 /* and skip it */

		}
	}                                             /* end of repeating skips */


 if(blk) /* number of dots to display, then turn one OFF */
	{
	i=0; /* first dot */
	while(i<=res && i>=0)	/* go through all dots */
     	{
		for(cnt=0;cnt<blk && (i>=0 && i<=res);cnt++)
			{
			i=FindOn(buf,i,res); /* find next ON dot */

			if(i>=0)
				i++;     /* if i is valid, skip to the next dot */
			}

		i=FindOn(buf,i,res); /* this is the dot to skip */

          if(i>=0)
			{
			SetOff(buf,i);
			(*offs)++;     /* increment OFF count */

			i++;
			}
		}
	}                                      /* end of display dots, skip one */

}								/* end of MaskIt */




void close_tiff(int fh)
{
#ifdef DEBUG
 printf("In close_tiff\n");
 getch();
#endif
 if(!close_read(fh))
	printf("error closing read routines");
}

void Help(void)
{
 printf("\n\n"
"DTIFF - Display Tag Image File Format, Version 2.0 (c) 1988 Late Night Software\n"
"\n"
"DTIFF is a utility that displays TIFF file images on systems that have a CGA,\n"
"EGA, VGA, MCGA, Hercules, IBM 8514/A, or AT&T display adapter cards.\n"
"The command to use is:\n"
"\n"
"                     dtiff <tiff file> [ options ]\n"
"Where:\n"
"<tiff file> is the DOS name of the Tag Image File Format to display.\n"
"[options] are separated by a space and begin with a '/'. They are:\n"
"\n"
"        /rn   : Rotate the the picture 'n' degrees, n=0,90,180 or 270\n"
"\n"
"        /a    : Display the picture 'as is'. If the picture is larger than\n"
"                your display, you'll will see only a portion of it.\n"
"\n"
"        /dADP : Use the adapter 'ADP', ADP=CGA,MCGA,EGA,EGA64,EGAMONO,VGA,\n"
"                IBM8514,HERC,or ATT. If not specified, DTIFF will try to \n"
"                detect the type of adapter that is installed in your system.\n"
"\n"
"List the file 'DTIFF.DOC' for more detailed information.\n"
"Late Night Software, PO BOX 81471, Chicago, IL  60681-0471\n\n");
}

void Syntax(void)
{
 printf("\n\nDo you want to see the instructions? If so, press 'Y' -> ");
 if(tolower(getch())==*"y")
	Help();
}
