{$F+} { Compiler Directive: Generate far procedure calls: On } { DO NOT CHANGE! }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Editor
    version 3.3

    The editor unit contains several procedures useful for changing
    information in several types of data structures.  The editor
    procedures uses the popular WordStar (R) key-stroke combinations to
    assist in performing this task.

    Purpose:
      This unit supplies special procedures that make it easy to allow
      values to be changed and altered with a full editing facility.

    How it works:
      The value is passed to the procedure and allowed to be altered.

    Features:
      Escape is allowed to override the changed value.
      This unit will automatically link with the windows unit if it's used.
        Window movement keys are fully supported.
      This unit will automatically links with the lock unit if it's used.

    Versions
      3.3 - Revamped to work better with the pointer unit.

    Copyright 1989, All rights reserved.
      Paul R. Renaud

    Compilers:
      Turbo Pascal versions 4.0 to 6.0
      Speed Pascal/2 version 1.5

    Systems:
      MS-DOS, MDOS, OS/2

    Common commands..

      Left - ^S, <-                 Right - ^D, ->
      Up - ^E, up                   Down - ^X, down
      Word Left - ^A, ^<-           Word Right - ^F, ^->
      Start of line - ^QS, home     End of line - ^QD, end
      Tab left - shift tab          Tab right - ^I, tab
      Toggle Insert - ^V, ins       Delete character - ^G, del
      Delete line - ^Y              Backspace - <-backspace, ^H
      Delete to end - ^QY           Delete word right - ^T
      Restore line - ^JL

      Escape from editor - [esc]

*****************************************************************************)

Unit Editor;

  Interface

    Uses
      CRT,
      Core,
      KeyBoard,
     {$IFNDEF OS2}
      String_Utilities;
     {$ELSE}
      String_U;
     {$ENDIF}

(***********************************************************

  Tab amount.
    This value determines the amount of spaces the tab key
    will move the cursor.

***********************************************************)

    Const
      Tab_Amount: Byte = 8;

(***********************************************************

  Insert Mode.
    This flag determines which mode the editor procedure is
    working in.  The user changes it by pressing the Insert
    key, but the value can be set by the program if desired.

***********************************************************)

      Insert_Mode: Boolean = False;

(***********************************************************

  Insert tab
    This flag determines if the tab key is allowed to add
    character when the system is in insert mode.  Otherwise,
    pressing the tab key merely moves the cursor.

***********************************************************)

      Insert_Tab: Boolean = True;

(***********************************************************

  Insert at end of line.
    This flag allows the current string to increase it's
    size when extra characters are added.  Otherwise the
    size of the line must remain constant.

***********************************************************)

      Insert_At_EoLn: Boolean = True;

(***********************************************************

  Exit on tab.
    This flag allows the tab key to act just like the return
    key.

***********************************************************)

      Exit_On_Tab: Boolean = False;

(***********************************************************

  Possible values that cause edits to terminate.

***********************************************************)

      Up_Key = Keyboard.Press_Up_Arrow;
      Down_Key = Keyboard.Press_Down_Arrow;
      Enter_Key = Keyboard.Press_Enter;
      Escape_Key = Keyboard.Press_Escape;

(***********************************************************

  Move Windows.
    Allows the user to move the top window, only if the
    windows unit is used by the program.

***********************************************************)

      Move_The_Windows: Boolean = True;

(***********************************************************

  Lock System.
    Allows the user to lock the system, only if the  Lock
    unit is used by the program.

***********************************************************)

      Lock_The_System: Boolean = True;

(***********************************************************

  Command.
    Holds the last command received by the editor routine
    before it terminated.  This is supplied to be useful in
    checking the last command entered.

***********************************************************)

    Var
      Command: Byte;

(***********************************************************

  Character.
    Holds the last character received by the editor routine
    before it terminated.  This is supplied to be useful in
    checking the last character entered.

***********************************************************)

      Character: Char;

(***********************************************************

  Help Screen.
    This variable procedure is called internally by the
    editor when the help key is pressed.  It is available
    for replacement by the main program.

***********************************************************)

   {$IFNDEF VER40}
    Var
      Help_Screen: Procedure;
   {$ENDIF}

(***********************************************************

  Function: Convert data to String.

    This function will convert the source data into a string
    and return it.   Source points to the original data
    source, which is most commonly an array of bytes or
    characters.  Length hold the size of the data source in
    bytes.

***********************************************************)

    Function Convert_To_String( Var Source; Length: Byte ): String;

(***********************************************************

  The editing procedures.

    These functions allows the editing of the values passed
    to them.  Row and column determines where on the screen
    the editing will take place and is dependent on the CRT
    screen management unit.  The working space consists of
    the given coordinates and the remainder of the current
    line as determined by CRT. If the data is longer then
    the working space, single line scrolling occurs to
    increase the working field; no wrap around is performed.
    Field holds the value which will be edited.  All values
    are edited like data strings then converted back to
    their original forms.  The original values are those
    passed to it by the calling program.  The size of the
    values are determined by the value passed to the
    procedure with the exception of strings: A string may
    change it's size up to the maximum allowable size,
    depending on the value of Insert_At_EoLn.
    The standard function keys are fully supported; Pressing
    F1 calls up the help screen, Left, Right, End and Home
    work too.  The procedure end when Up, Down, Enter or
    Escape are encountered. ( As of version 2.0, additional
    function keys, Delete, Insert, Move cursor, Restore and
    Clear line were added )

***********************************************************)

    Procedure Edit_Byte( Row, Column: Byte; Var Field: Byte );
    Procedure Edit_Word( Row, Column: Byte; Var Field: Word );
    Procedure Edit_Short_Integer( Row, Column: Byte; Var Field: ShortInt );
    Procedure Edit_Integer( Row, Column: Byte; Var Field: Integer );
    Procedure Edit_Long_Integer( Row, Column: Byte; Var Field: LongInt );
    Procedure Edit_Real( Row, Column: Byte; Var Field: Real );
    Procedure Edit_String( Row, Column: Byte; Var Field: String );

(***********************************************************

  The toggling procedures.

    These procedure allow the values given to be changed by
    toggling.  Pressing the Up or Right Arrows will increase
    the given value, pressing the Down or left Arrows will
    decrease the value.  Pressing the F1 key will invoke the
    help screen.   Clear and Restore Line are also supported.
    If Escape is pressed, the value will not change.  Holding
    down the Arrow key will invoke rapid changes in the value
    so that the longer the key is held, the faster the value
    changes.  Row and column determine where the value is
    displayed.  These values are wrapped if they exceed the
    end of the current line.

***********************************************************)

    Procedure Toggle_Byte( Row, Column: Byte; Var Field: Byte );
    Procedure Toggle_Word( Row, Column: Byte; Var Field: Word );
    Procedure Toggle_Short_Integer( Row, Column: Byte; Var Field: ShortInt );
    Procedure Toggle_Integer( Row, Column: Byte; Var Field: Integer );
    Procedure Toggle_Long_Integer( Row, Column: Byte; Var Field: LongInt );

(***********************************************************

  The setting procedures.

    These procedure allow the values to be changed in a
    special way.  ( A cross between toggling and editing )
    Pressing the Up Arrow will only increase the current
    digit's value, while pressing the Down Arrow will
    decrease it.  Pressing the F1 key invokes the help
    screen.  The digit keys and the Restore key stroke are
    also supported.  If Escape is pressed, the value will
    not change.  Row and Column determine where the value is
    displayed.  These values are wrapped if they exceed the
    end of the current line.

***********************************************************)

    Procedure Set_Byte( Row, Column: Byte; Var Field: Byte );
    Procedure Set_Word( Row, Column: Byte; Var Field: Word );
    Procedure Set_Short_Integer( Row, Column: Byte; Var Field: ShortInt );
    Procedure Set_Integer( Row, Column: Byte; Var Field: Integer );
    Procedure Set_Long_Integer( Row, Column: Byte; Var Field: LongInt );
    Procedure Set_Real( Row, Column: Byte; Var Field: Real );

(***********************************************************

  The getting procedures.

    These procedures allows the user to enter a value in a
    form much like an electronic calculator; The cursor
    stays at the end of the line and the line moves over to
    the left whenever a character is added.  The value is
    wrapped if it exceeds the end of the current line.  Row
    and Column determines where on the screen the editing
    will take place and is dependent on the CRT screen
    management unit.  Field holds the value which will be
    returned.  The size of the value is determined by the
    value passed to the procedure except for strings: A
    string will always keep the same length.  Pressing F1
    will invoke the help screen.  The procedure will end
    when Up, Down, Enter or Escape are pressed.  Pressing
    Backspace deletes the previous character, Delete clears
    the line.

***********************************************************)

    Procedure Get_Byte( Row, Column: Byte; Var Field: Byte );
    Procedure Get_Word( Row, Column: Byte; Var Field: Word );
    Procedure Get_Short_Integer( Row, Column: Byte; Var Field: ShortInt );
    Procedure Get_Integer( Row, Column: Byte; Var Field: Integer );
    Procedure Get_Long_Integer( Row, Column: Byte; Var Field: LongInt );
    Procedure Get_Real( Row, Column: Byte; Var Field: Real );
    Procedure Get_String( Row, Column: Byte; Var Field: String );

(***********************************************************

  Procedure: Change Boolean.

    This procedure allows the given boolean value to be
    easily altered.  Pressing the Up or Down Arrows will
    return control to the program while pressing the F1 key
    will invoke the help screen.  The F and T keys will set
    the value and the space acts as a toggle.  If Escape is
    pressed, the value will not change.  Row and Column
    determine where the value is displayed.  The value will
    be wrapped if it exceeds the end of the current line.

***********************************************************)

    Procedure Change_Boolean( Row, Column: Byte; Var Field: Boolean );

{----------------------------------------------------------------------------}

  Implementation

    Const
     { These are the legitimate values for word separators. }
      Word_Separators: Set of char = [ ' ', '.', ':', ',', '(', ')', '{', '}', '[', ']',
                                       '<', '>', '?', '/', ';', '!', '@', '^', '&', '*',
                                       '-', '+', '\', '~', '`', '=', '|', '''' ];
     { These values will cause the editor to stop. }
      Leave_Values: Set of Byte = [ Press_Enter, Press_Down_Arrow, Press_Up_Arrow, Press_Escape, Pointer_Down, Pointer_Up ];

    Type
     { This structure holds information necessary for displaying and editing the string. }
      Cluster_Type = Record
                       Row,
                       Start,
                       Where,
                       Column,
                       Finish,
                       String_Length: Byte;
                       Character: Char;
                       Line_Altered: Boolean;
                     End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Window left.
    This procedure moves the current window left
    if the Move_The_Windows variable allows it.

*************************************************)

    Procedure Window_Left;
      Var
        Okay: Boolean;
      Begin
       {$IFNDEF VER40}
        Okay := ( Move_The_Windows and Core.Left_Routine );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Window right.
    This procedure moves the current window right
    if the Move_The_Windows variable allows it.

*************************************************)

    Procedure Window_Right;
      Var
        Okay: Boolean;
      Begin
       {$IFNDEF VER40}
        Okay := ( Move_The_Windows and Core.Right_Routine );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Window up.
    This procedure moves the current window up if
    the Move_The_Windows variable allows it.

*************************************************)

    Procedure Window_Up;
      Var
        Okay: Boolean;
      Begin
       {$IFNDEF VER40}
        Okay := ( Move_The_Windows and Core.Up_Routine );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Window down.
    This procedure moves the current window down
    if the Move_The_Windows variable allows it.

*************************************************)

    Procedure Window_Down;
      Var
        Okay: Boolean;
      Begin
       {$IFNDEF VER40}
        Okay := ( Move_The_Windows and Core.Down_Routine );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: System lock.
    This procedure locks the computer system if
    the Lock the system variable allows it.

*************************************************)

    Procedure System_Lock;
      Begin
       {$IFNDEF VER40}
        If Lock_The_System
          then
            Core.Lock_Routine;
        {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write truncated.
    This procedure writes the section of the data
    string defined by start and finished on the
    screen.

*************************************************)

    Procedure Write_Truncated( Data: String; Var Info: Cluster_Type );
      Var
        Save_Row,
        Save_Column: Byte;
      Begin
        Save_Row := WhereY;
        Save_Column := WhereX;
        While ( ( Info.Where > Info.Finish ) and ( Info.Where <= Length( Data ) ) ) do
          Begin
            Inc( Info.Start );
            Inc( Info.Finish );
          End;
        While ( Info.Where < Info.Start ) do
          Begin
            Dec( Info.Start );
            Dec( Info.Finish );
          End;
        Write( Screen, Copy( Data, Info.Start, Succ( Info.Finish - Info.Start ) ) );
        Cursor_Column_Start := ( Save_Column + Info.Where - Info.Start );
        Cursor_Column_Finish := Cursor_Column_Start;
        GotoXY( Cursor_Column_Start, Save_Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Convert to string.
    As previously defined.

*************************************************)

    Function Convert_To_String( Var Source; Length: Byte ): String;
      Var
        Destination: String;
      Begin
        Move( Source, Destination[ 1 ], Length );
        While ( ( Length > 0 ) and ( Destination[ Length ] = ' ' ) ) do
          Dec( Length );
        Destination[ 0 ] := Chr( Length );
        Convert_To_String := Destination;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display.
    This procedure merely displays the given long
    integer with the given Length at the specified
    location.

*************************************************)

    Procedure Display( Row, Column, Length: Byte; Field: LongInt );
      Begin
        GotoXY( Column, Row );
        Cursor_Column_Start := WhereX;
        Write( Screen, Field: Length );
        Cursor_Column_Finish := WhereX;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Show.
    This procedure merely displays the given
    string and appropriate sign on the screen at
    the given location.

*************************************************)

    Procedure Show( Row, Column: Byte; Data: String; Sign: Char );
      Begin
        GotoXY( Column, Row );
        Write( Screen, Data, Sign );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write it.
    This procedure displays the given boolean
    variable at the given location.

*************************************************)

    Procedure Write_It( Row, Column: Byte; Field: Boolean );
      Begin
        GotoXY( Column, Row );
        Cursor_Column_Start := WhereX;
        Write( Screen, Field );
        Cursor_Column_Finish := WhereX;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Print it.
    This procedure displays the given data string
    at the given location.

*************************************************)

    Procedure Print_It( Row, Column: Byte; Data: String );
      Begin
        GotoXY( Column, Row );
        Cursor_Column_Start := WhereX;
        Write( Screen, Data );
        Cursor_Column_Finish := WhereX;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Fix the line.
    This procedure fixes the line to represent the
    given field.

*************************************************)

    Procedure Fix_Line( Length: Byte; Field: LongInt; Var Data: String; Var Sign: Char );
      Var
        Count: Byte;
      Begin
        Str( Field: Length, Data );
        For Count := 1 to Length do
          If Not ( Data[ Count ] in [ '0' .. '9' ] )
            then
              Data[ Count ] := '0';
        If ( Field < 0 )
          then
            Sign := '-'
          else
            Sign := '+';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Fix the line real.
    This procedure fixes the line to represent the
    given real field.

*************************************************)

    Procedure Fix_Line_Real( Length: Byte; Field: Real; Var Data: String; Var Sign: Char );
      Var
        Count: Byte;
      Begin
        Str( Field: Length, Data );
        For Count := 1 to Length do
          If Not ( Data[ Count ] in [ '0' .. '9', 'E', 'e', '.' ] )
            then
              Data[ Count ] := '0';
        If ( Field < 0 )
          then
            Sign := '-'
          else
            Sign := '+';
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Process display.
    This procedure is called when the data line
    was altered and should be updated on the
    screen.

*************************************************)

    Procedure Process_Display( Var Data: Cluster_Type; Var Line: String );
      Begin
        Data.Line_Altered := True;
        GotoXY( Data.Column, Data.Row );
        Write_Truncated( Line, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move right.
    This procedure moves the editing row buffer
    cursor right and updates the screen.

*************************************************)

    Procedure Move_Right( Var Line: String; Var Data: Cluster_Type; Amount: Byte );
      Var
        Count: Byte;
        Change: Boolean;
      Begin
        Change := False;
        For Count := 1 to Amount do
          If ( Data.Where < Data.String_Length )
            then
              Begin
                Inc( Data.Where );
                Change := True;
              End;
        If Change
          then
            Begin
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move left.
    This procedure moves the editing row buffer
    cursor left and updates the screen.

*************************************************)

    Procedure Move_Left( Var Line: String; Var Data: Cluster_Type; Amount: Byte );
      Var
        Count: Byte;
        Change: Boolean;
      Begin
        Change := False;
        For Count := 1 to Amount do
          If ( Data.Where > 1 )
            then
              Begin
                Dec( Data.Where );
                Change := True;
              End;
        If Change
          then
            Begin
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move where.
    This procedure moves the editing row buffer
    cursor to the given position and updates the
    screen.

*************************************************)

    Procedure Move_Where( Var Line: String; Var Data: Cluster_Type; Where: Byte );
      Begin
        Data.Where := Where;
        GotoXY( Data.Column, Data.Row );
        Write_Truncated( Line, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move word right.
    This procedure moves the editing row buffer
    cursor to the right-most word and updates the
    screen.

*************************************************)

    Procedure Move_Word_Right( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where < Data.String_Length )
          then
            Begin
              While ( ( Data.Where < Data.String_Length ) and ( not ( Line[ Data.Where ] in Word_Separators ) ) ) do
                Inc( Data.Where );
              While ( ( Data.Where < Data.String_Length ) and ( Line[ Data.Where ] in Word_Separators ) ) do
                Inc( Data.Where );
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move word left.
    This procedure moves the editing row buffer
    cursor to the left-most word and updates the
    screen.

*************************************************)

    Procedure Move_Word_Left( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where > 1 )
          then
            Begin
              While ( ( Data.Where > 1 ) and ( not ( Line[ Data.Where ] in Word_Separators ) ) ) do
                Dec( Data.Where );
              While ( ( Data.Where > 1 ) and ( Line[ Data.Where ] in Word_Separators ) ) do
                Dec( Data.Where );
              While ( ( Data.Where > 1 ) and ( not ( Line[ Data.Where ] in Word_Separators ) ) ) do
                Dec( Data.Where );
              If ( ( Data.Where < Data.String_Length ) and ( not ( Line[ Succ( Data.Where ) ] in Word_Separators ) ) )
                then
                  Inc( Data.Where );
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add character.
    This procedure adds the given character to the
    editing row buffer and updates the screen.

*************************************************)

    Procedure Add_Character( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where < Data.String_Length )
          then
            Begin
              If Insert_Mode
                then
                  Begin
                    Insert( Data.Character, Line, Data.Where );
                    Data.String_Length := Length( Line );
                  End
                else
                  Line[ Data.Where ] := Data.Character;
              Data.Line_Altered := True;
            End
          else
            If Insert_At_EoLn
              then
                Begin
                  Line := Line + ' ';
                  Data.String_Length := Length( Line );
                  Line[ Data.Where ] := Data.Character;
                  Data.Line_Altered := True;
                End;
        If ( Data.Where < Data.String_Length )
          then
            Inc( Data.Where );
        GotoXY( Data.Column, Data.Row );
        Write_Truncated( Line, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add character reverse
    This procedure adds the given character to
    the string at the appropriate location.

*************************************************)

    Procedure Add_Character_Reverse( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where > 0 )
          then
            Begin
              If ( Insert_Mode and ( Data.Where > 1 ) )
                then
                  Begin
                    Move( Line[ 2 ], Line[ 1 ], Pred( Data.Where ) );
                    Line[ Data.Where ] := Data.Character;
                  End
                else
                  Begin
                    Line[ Data.Where ] := Data.Character;
                    If ( Data.Where < Data.String_Length )
                      then
                        Inc( Data.Where );
                  End;
              Process_Display( Data, Line );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete character.
    This procedure removes the character under the
    editing row buffer cursor and updates the
    screen.

*************************************************)

    Procedure Delete_Character( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where < Data.String_Length )
          then
            Begin
              Delete( Line, Data.Where, 1 );
              Data.String_Length := Length( Line );
            End
          else
            Line[ Data.Where ] := ' ';
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete character reverse.
    This procedure deletes the character under the
    cursor and moves the remaining characters
    right.

*************************************************)

    Procedure Delete_Character_Reverse( Var Line: String; Var Data: Cluster_Type );
      Begin
        Move( Line[ 1 ], Line[ 2 ], Pred( Data.Where ) );
        Line[ 1 ] := ' ';
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Arrow delete.
    This procedure removes the character to the
    left of the editing row buffer cursor and
    updates the screen.

*************************************************)

    Procedure Arrow_Delete( Var Line: String; Var Data: Cluster_Type );
      Begin
        If ( Data.Where > 1 )
          then
            Begin
              Dec( Data.Where );
              Delete( Line, Data.Where, 1 );
              Data.String_Length := Length( Line );
              Process_Display( Data, Line );
            End
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move delete line.
    This procedure erases the entire line, moves
    the editing row buffer cursor to the left-most
    position and updates the screen.

*************************************************)

    Procedure Move_Delete_Line( Var Line: String; Var Data: Cluster_Type );
      Begin
        FillChar( Line[ 1 ], Data.String_Length, ' ' );
        Data.Where := 1;
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move delete line reverse.
    This procedure clears the entire line.

*************************************************)

    Procedure Move_Delete_Line_Reverse( Var Line: String; Var Data: Cluster_Type );
      Begin
        FillChar( Line[ 1 ], Data.String_Length, ' ' );
        Data.Where := Data.String_Length;
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete end.
    This procedure erases the remainder of the line
    from the editing row buffer cursor and updates
    the screen.

*************************************************)

    Procedure Delete_End( Var Line: String; Var Data: Cluster_Type );
      Begin
        FillChar( Line[ Data.Where ], Succ( Data.String_Length - Data.Where ), ' ' );
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete end reverse.
    This procedure clears the remaining line and
    updates the screen.

*************************************************)

    Procedure Delete_End_Reverse( Var Line: String; Var Data: Cluster_Type );
      Begin
        FillChar( Line[ 1 ], Data.Where, ' ' );
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete word right.
    This procedure erases the word to the right of
    the editing row buffer cursor and updates the
    screen.

*************************************************)

    Procedure Delete_Word_Right( Var Line: String; Var Data: Cluster_Type );
      Begin
        While ( ( Data.Where < Data.String_Length ) and ( Line[ Data.Where ] in Word_Separators ) ) do
          Begin
            Delete( Line, Data.Where, 1 );
            Dec( Data.String_Length );
          End;
        While ( ( Data.Where < Data.String_Length ) and ( not ( Line[ Data.Where ] in Word_Separators ) ) ) do
          Begin
            Delete( Line, Data.Where, 1 );
            Dec( Data.String_Length );
          End;
        Process_Display( Data, Line );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Tab.
    This procedure moves the editing row buffer
    cursor to the right one tab space and updates
    the screen.

*************************************************)

    Procedure Tab( Var Line: String; Var Data: Cluster_Type );
      Var
        Count,
        Tab_To: Word;
      Begin
        If Exit_On_Tab
          then
            Command := Press_Enter
          else
            Begin
              If ( Insert_Tab and Insert_Mode )
                then
                  Begin
                    Tab_To := Succ( Data.Where div Tab_Amount ) * Tab_Amount;
                    For Count := Succ( Data.Where ) to Tab_To do
                      Begin
                        Insert( ' ', Line, Data.Where );
                        Inc( Data.Where );
                        Inc( Data.String_Length );
                      End;
                    Data.Line_Altered := True;
                  End
                else
                  Begin
                    Data.Where := Succ( Data.Where div Tab_Amount ) * Tab_Amount;
                    If ( Data.Where > Data.String_Length )
                      then
                        Data.Where := Data.String_Length;
                  End;
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Inverse tab.
    This procedure moves the editing row buffer
    cursor to the left one tab space and updates
    the screen.

*************************************************)

    Procedure Inverse_Tab( Var Line: String; Var Data: Cluster_Type );
      Var
        Hold: LongInt;
      Begin
        If Exit_On_Tab
          then
            Command := Press_Enter
          else
            Begin
              Hold := ( Pred( Data.Where ) Div Tab_Amount ) * Tab_Amount;
              If ( Hold > 0 )
                then
                  Data.Where := Hold
                else
                  Data.Where := 1;
              GotoXY( Data.Column, Data.Row );
              Write_Truncated( Line, Data );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore line.
    This procedure resets the line to the
    original value.

*************************************************)

    Procedure Restore_Line( Var Data: Cluster_Type; Var Field, Original_Field: String );
      Begin
        Data.Where := 1;
        Field := Original_Field;
        GotoXY( Data.Column, Data.Row );
        Write_Truncated( Field, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore line.
    This procedure resets the line to the
    original value.

*************************************************)

    Procedure Restore_Line_Reverse( Var Data: Cluster_Type; Var Field, Original_Field: String );
      Begin
        Field := Original_Field;
        Data.Where := Data.String_Length;
        GotoXY( Data.Column, Data.Row );
        Write_Truncated( Field, Data );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize data.
    This procedure initializes the data cluster
    for the editing procedures.

*************************************************)

    Procedure Initialize_Data( Var Data: Cluster_Type; Row, Column: Byte; Field: String );
      Begin
        Cursor_Row := Row;
        Data.Row := Row;
        Data.Column := Column;
        Data.String_Length := Length( Field );
        Data.Line_Altered := False;
        GotoXY( Data.Column, Data.Row );
        Data.Start := 1;
        Data.Finish := Succ( Succ( ( Right_Of_Window^ - Left_Of_Window^ ) - Data.Column ) );
        If ( Data.Finish > Succ( Length( Field ) ) )
          then
            Data.Finish := Succ( Length( Field ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Extended capabilities.
    This procedure handles the code for the
    extended commands of the editor routines.

*************************************************)

    Procedure Extended_Capabilities( Command: Byte );
      Begin
        Case Command of
          Press_Insert:
            Insert_Mode := Not( Insert_Mode );
          Press_F1:
           {$IFNDEF VER40}
            Help_Screen;
           {$ELSE}
            Default_Help_Screen;
           {$ENDIF}
          Pressed_Move_Window_Left:
            Window_Left;
          Pressed_Move_Window_Right:
            Window_Right;
          Pressed_Move_Window_Up:
            Window_Up;
          Pressed_Move_Window_Down:
            Window_Down;
          Pressed_Lock:
            System_Lock;
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Edit the string.
    This procedure allows the string to be edited
    at the provided screen location.

*************************************************)

    Procedure Edit_The_String( Row, Column: Byte; Var Field: String );
      Var
        Data: Cluster_Type;
        Original_Field: String;
      Begin
        Field := Field + ' ';
        Original_Field := Field;
        Initialize_Data( Data, Row, Column, Field );
        Data.Where := 1;
        Write_Truncated( Field, Data );
        GotoXY( Data.Column, Data.Row );
        Repeat
          Get_Command( Character, Command );
          Data.Character := Character;
          Case Command of
            Press_Numbers,
            Press_Capital_Letters,
            Press_Lower_Letters,
            Press_Extra_Characters:
              Add_Character( Field, Data );
            Pointer_Right:
              Move_Right( Field, Data, Adjust_Amount );
            Press_Right_Arrow:
              Move_Right( Field, Data, 1 );
            Pointer_Left:
              Move_Left( Field, Data, Adjust_Amount );
            Press_Left_Arrow:
              Move_Left( Field, Data, 1 );
            Press_Home:
              Move_Where( Field, Data, 1 );
            Press_End:
              Move_Where( Field, Data, Data.String_Length );
            Press_Delete:
              Delete_Character( Field, Data );
            Press_Delete_Arrow:
              Arrow_Delete( Field, Data );
            Press_Word_Right,
            Press_Control_Right_Arrow:
              Move_Word_Right( Field, Data );
            Press_Word_Left,
            Press_Control_Left_Arrow:
              Move_Word_Left( Field, Data );
            Press_Delete_Line:
              Move_Delete_Line( Field, Data );
            Press_Delete_End:
              Delete_End( Field, Data );
            Press_Restore_Line:
              Restore_Line( Data, Field, Original_Field );
            Press_Control_T:
              Delete_Word_Right( Field, Data );
            Press_Tab:
              Tab( Field, Data );
            Press_Shift_Tab:
              Inverse_Tab( Field, Data );
            else
              Extended_Capabilities( Command );
          End; { Case }
        Until ( Command in Leave_Values );
        If ( Command = Press_Escape )
          then
            Field := Original_Field;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reverse edit the string.
    This procedure allows the string to be reverse
    edited at the provided screen location.

*************************************************)

    Procedure Reverse_Edit_The_String( Row, Column: Byte; Var Field: String );
      Var
        Data: Cluster_Type;
        Original_Field: String;
      Begin
        Original_Field := Field;
        Initialize_Data( Data, Row, Column, Field );
        Data.Where := Data.String_Length;
        Write_Truncated( Field, Data );
        Repeat
          Get_Command( Character, Command );
          Data.Character := Character;
          Case Command of
            Press_Numbers,
            Press_Capital_Letters,
            Press_Lower_Letters,
            Press_Extra_Characters:
              Add_Character_Reverse( Field, Data );
            Pointer_Right:
              Move_Right( Field, Data, Adjust_Amount );
            Press_Right_Arrow:
              Move_Right( Field, Data, 1 );
            Pointer_Left:
              Move_Left( Field, Data, Adjust_Amount );
            Press_Left_Arrow:
              Move_Left( Field, Data, 1 );
            Press_Home:
              Move_Where( Field, Data, 1 );
            Press_End:
              Move_Where( Field, Data, Data.String_Length );
            Press_Delete:
              Delete_Character_Reverse( Field, Data );
            Press_Delete_Arrow:
              Arrow_Delete( Field, Data );
            Press_Word_Right,
            Press_Control_Right_Arrow:
              Move_Word_Right( Field, Data );
            Press_Word_Left,
            Press_Control_Left_Arrow:
              Move_Word_Left( Field, Data );
            Press_Delete_Line:
              Move_Delete_Line_Reverse( Field, Data );
            Press_Delete_End:
              Delete_End_Reverse( Field, Data );
            Press_Restore_Line:
              Restore_Line_Reverse( Data, Field, Original_Field );
            Press_Control_T:
              Delete_Word_Right( Field, Data );
            Press_Tab:
              Inverse_Tab( Field, Data );
            Press_Shift_Tab:
              Tab( Field, Data );
            else
              Extended_Capabilities( Command );
          End; { Case }
        Until ( Command in Leave_Values );
        If ( Command = Press_Escape )
          then
            Field := Original_Field;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Toggle number.
    This procedure allows the given value to be
    toggled on the screen at the given location.

*************************************************)

    Procedure Toggle_Number( Row, Column: Byte; Var Field: LongInt; Length, Rate: Byte; Low, High: LongInt );
      Var
        Stop: Boolean;
        Count: Word;
        Character: Char;
        Value,
        Change,
        Original_Field: LongInt;
      Begin
        Cursor_Row := Row;
        Original_Field := Field;
        Display( Row, Column, Length, Field );
        Stop := False;
        Change := 1;
        Count := 0;
        Repeat
          If Data_Ready
            then
              Begin
                Get_Command( Character, Command );
                Case Command of
                  Press_Down_Arrow,
                  Press_Left_Arrow:
                   { Lower the value. }
                    Begin
                      Value := ( Field - Change );
                      If ( ( Value >= Low ) and ( Value < Field ) )
                        then
                          Field := Value
                        else
                          Field := Low;
                      Display( Row, Column, Length, Field );
                    End;
                  Press_Up_Arrow,
                  Press_Right_Arrow:
                   { Raise the value. }
                    Begin
                      Value := ( Field + Change );
                      If ( ( Value <= High ) and ( Value > Field ) )
                        then
                          Field := Value
                        else
                          Field := High;
                      Display( Row, Column, Length, Field );
                    End;
                  Press_Enter,
                  Press_Escape:
                    Stop := True;
                  Press_Delete_Line:
                    Begin
                      Field := 0;
                      Display( Row, Column, Length, Field );
                    End;
                  Press_Restore_Line:
                    Begin
                      Field := Original_Field;
                      Display( Row, Column, Length, Field );
                    End;
                  else
                    Extended_Capabilities( Command );
                End; { Case }
                Inc( Change, Rate );
                Count := 0;
              End
            else
              If ( Count > 800 )
                then
                  Change := 1;
          Inc( Count );
        Until Stop;
        If ( Command = Press_Escape )
          then
            Field := Original_Field;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set number.
    This procedure allow the number to be changed
    by one digit at a time.

*************************************************)

    Procedure Set_Number( Row, Column: Byte; Var Field: LongInt; Length: Byte; Low, High: LongInt );
      Var
        Stop,
        Escape: Boolean;
        Sign,
        Character: Char;
        Data: String;
        Count: Word;
        Result: LongInt;
      Begin
        Cursor_Row := Row;
        Repeat
          Stop := False;
          Fix_Line( Length, Field, Data, Sign );
          Count := 1;
          Show( Row, Column, Data, Sign );
          Repeat
            Cursor_Column_Start := Pred( Column + Count );
            Cursor_Column_Finish := Cursor_Column_Start;
            GotoXY( Cursor_Column_Start, Row );
            If Data_Ready
              then
                Begin
                  Get_Command( Character, Command );
                  Case Command of
                    Press_Numbers:
                     { The number is in between zero and nine. }
                      Begin
                        Data[ Count ] := Character;
                        If ( Count < Length )
                          then
                            Inc( Count );
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Extra_Characters:
                      Begin
                        Case Character of
                          '-': Sign := '-';
                          '+': Sign := '+';
                        End; { Case }
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Down_Arrow:
                     { Decrement the digit. }
                      Begin
                        If ( Data[ Count ] > '0' )
                          then
                            Dec( Byte( Data[ Count ] ) )
                          else
                            Data[ Count ] := '9';
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Up_Arrow:
                     { Increment the digit. }
                      Begin
                        If ( ( Data[ Count ] < '9' ) and ( Count > 1 ) )
                          then
                            Inc( Byte( Data[ Count ] ) )
                          else
                            Data[ Count ] := '0';
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Left_Arrow:
                      If ( Count > 1 )
                        then
                          Dec( Count );
                    Press_Right_Arrow:
                      If ( Count < Length )
                        then
                          Inc( Count );
                    Press_Enter,
                    Press_Escape:
                      Stop := True;
                    Press_Restore_Line:
                      Begin
                        Fix_Line( Length, Field, Data, Sign );
                        Count := 1;
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Home:
                      Count := 1;
                    Press_End:
                      Count := Length;
                    else
                      Extended_Capabilities( Command );
                  End; { Case }
                End
          Until Stop;
          Escape := ( Command = Press_Escape );
          If ( Not Escape )
            then
              Val( Data, Result, Count );
        Until ( ( ( Count = 0 ) and ( Result >= Low ) and ( Result <= High ) ) or Escape );
        If ( Not Escape )
          then
            Field := Result;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set real.
    This procedure allows the setting of a real
    number at the specified location.

*************************************************)

    Procedure Set_Real( Row, Column: Byte; Var Field: Real );
      Const
       {$IFNDEF VER40}
        Low = -1.7e38;
        High = 1.7e38;
       {$ELSE}
        Low = -1.7e36;
        High = 1.7e36;
       {$ENDIF}
        Length = 27;
      Var
        Stop,
        Escape: Boolean;
        Sign,
        Character: Char;
        Data: String;
        Count: Word;
        Result: Real;
      Begin
        Cursor_Row := Row;
        Repeat
          Stop := False;
          Fix_Line_Real( Length, Field, Data, Sign );
          Count := 1;
          Show( Row, Column, Data, Sign );
          Repeat
            Cursor_Column_Start := Pred( Column + Count );
            Cursor_Column_Finish := Cursor_Column_Start;
            GotoXY( Cursor_Column_Start, Row );
            If Data_Ready
              then
                Begin
                  Get_Command( Character, Command );
                  Case Command of
                    Pointer_Left:
                      Repeat
                        If ( Count > 1 )
                          then
                            Dec( Count );
                        Dec( Adjust_Amount );
                      Until ( Adjust_Amount = 0 );
                    Pointer_Right:
                      Repeat
                        If ( Count < Length )
                          then
                            Inc( Count );
                        Dec( Adjust_Amount );
                      Until ( Adjust_Amount = 0 );
                    Press_Numbers:
                      { Character is between zero and nine. }
                      If ( Data[ Count ] in [ '0' .. '9' ] )
                        then
                          Begin
                            Data[ Count ] := Character;
                            If ( Count < Length )
                              then
                                Inc( Count );
                            Show( Row, Column, Data, Sign );
                          End;
                    Press_Extra_Characters:
                      Begin
                        Case Character of
                          '-': Sign := '-';
                          '+': Sign := '+';
                        End; {Case}
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Down_Arrow:
                      If ( Data[ Count ] in [ '0' .. '9' ] )
                        then
                          Begin
                            If ( Data[ Count ] > '0' )
                              then
                                Dec( Byte( Data[ Count ] ) )
                              else
                                Data[ Count ] := '9';
                            Show( Row, Column, Data, Sign );
                          End;
                    Press_Up_Arrow:
                      If ( Data[ Count ] in [ '0' .. '9' ] )
                        then
                          Begin
                            If ( ( Data[ Count ] < '9' ) and ( Count > 1 ) )
                              then
                                Inc( Byte( Data[ Count ] ) )
                              else
                                Data[ Count ] := '0';
                            Show( Row, Column, Data, Sign );
                          End;
                    Press_Left_Arrow:
                      If ( Count > 1 )
                        then
                          Dec( Count );
                    Press_Right_Arrow:
                      If ( Count < Length )
                        then
                          Inc( Count );
                    Press_Enter:
                      Stop := True;
                    Press_Escape:
                      Stop := True;
                    Press_Restore_Line:
                      Begin
                        Fix_Line_Real( Length, Field, Data, Sign );
                        Count := 1;
                        Show( Row, Column, Data, Sign );
                      End;
                    Press_Home:
                      Count := 1;
                    Press_End:
                      Count := Length;
                    else
                      Extended_Capabilities( Command );
                  End; { Case }
                End
          Until Stop;
          Escape := ( Command = Press_Escape );
          If ( Not Escape )
            then
              Val( Data, Result, Count );
        Until ( ( ( Count = 0 ) and ( Result >= Low ) and ( Result <= High ) ) or Escape );
        If ( Not Escape )
          then
            Field := Result;
        GotoXY( Column, Row );
      End;

{----------------------------------------------------------------------------}

  {$I Editor2.Pas}

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Initialize help screen.

*************************************************)

   {$IFNDEF VER40}
    Begin
      Help_Screen := Procedure_Default;
   {$ENDIF}
    End.


