(*****************************************************************************

  TextFile
    version 2.0

  Purpose:
    This unit is designed to incorporate all the necessary routines to handle
    text file structures ( such as ASCII text files ).
    This unit is not intended as a stand alone used.

  Features:
    Can handles very large files ( file space permitting ) of up to 2 billion
      lines of up to 60 thousand characters each. ( Line size is determined
      by TextLine )
    Extensive error checking.
    Stores data is a dynamic file structure to conserve resources.
    file storage provides enormous storage capacity.
    Blank lines are stored as nul pointers which take up no extra space.

  Limitations:
    This unit should not be overlaid.
    The speed of operations, especially block operations are severely limited
      by file access time..

  CopyRight 1994, All rights reserved.
    By Paul Renaud

  Compiler:
    Turbo Pascal versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextFile;

  Interface

    Uses
      Core,
      FileMem,
      TextLine;

    Const
     { Defines a segment of the file for buffer use only. }
      File_Length = 1000;  { Optimal at 2000 }

    Type
     { Defines the line data pointer. }
      Text_Info_Type = Pointer_Type;
     { Defines the file segment array structure. }
      Text_Array_Type = Packed array [ 1 .. File_Length ] of Text_Info_Type;
     { Defines a pointer to the data storage.
      Text_Buffer_Pointer_Type = ^Text_Buffer_Type; }
     { Defines the file segment storage record. }
      Text_Buffer_Type = Record
                           Line_Start: LongInt;
                           Data: Text_Array_Type;
                           Next,
                           Previous: Pointer_Type;
                         End;
     { Defines text data storage structure. }
      Text_Type = Record
                    Where: Pointer_Type;
                    Buffer: Text_Buffer_Type;
                    FileSize: LongInt;
                  End;

    Var
     { This data structure provides working space for some of the routines. }
      Work_Space: Line_Pointer_Type;
     { Warning!!!!! }
     { These procedure must be initialize, or the system will crash. }
      Read_Status: Procedure( Where: LongInt );
      Write_Status: Procedure( Where, Limit: LongInt );

(***********************************************************

  Function: Allocate text.

    This function allocates the text data structure to hold
    the text.  If it fails, it returns false.

***********************************************************)

    Function Allocate_Text( Var Text: Text_Type ): Boolean;

(***********************************************************

  Function: Check allocation.

    This function returns true if the text data structure
    has been properly allocated.

***********************************************************)

    Function Check_Allocation( Var Text: Text_Type ): Boolean;

(***********************************************************

  Procedure: Get text line.

    This procedure returns the text line of Text, specified
    by Row in Line.

***********************************************************)

    Procedure Get_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );

(***********************************************************

  Function: Put text line.

    This procedure puts the given text line in Text, at the
    specified row.  It returns false if there isn't any
    more space to do so.

***********************************************************)

    Function Put_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type; Reduce: Boolean ): Boolean;

(***********************************************************

  Function: Read text.

    This function reads the text out of a given file and
    stores it in the Text data structure.  It returns false
    if an input error occurs.

***********************************************************)

    Function Read_Text( Var InFile: Text; Var Text: Text_Type ): Boolean;

(***********************************************************

  Function: Write text.

    This function attempts to store the text structure in
    the given text file.  It returns false if an output
    error occurs.

***********************************************************)

    Function Write_Text( Var OutFile: File; Var Text: Text_Type ): Boolean;

(***********************************************************

  Procedure: Dispose text.

    This procedure disposes of the text within the Text
    storage structure.  The structure is still left
    allocated so that data can be stored in it.

***********************************************************)

    Procedure Dispose_Text( Var Text: Text_Type );

(***********************************************************

  Function: Delete text line.

    This function attempts to delete the given row from the
    text storage structure.  All succeeding rows are moved
    forward after the row is deallocated.

***********************************************************)

    Function Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;

(***********************************************************

  Function: Split text.

    This function attempts to splits the text structure at
    the given location.  This means that the text line is
    split in half and the remaining text on the line is
    converted into a new line.  If the function fails, it
    returns false.  Data loss is highly unlikely.

***********************************************************)

    Function Split_Text( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type; Reduce: Boolean ): Boolean;

(***********************************************************

  Function: Remove return.

    This function attempts combine the line specified by
    Data with the succeeding line.  It undoes what split
    text does.  If the function fails, it returns false.
    Data loss is highly unlikely.

***********************************************************)

    Function Remove_Return( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type ): Boolean;

(***********************************************************

  Function: Get text size.

    This function returns the amount of lines in the text
    data structure.

***********************************************************)

    Function Get_Text_Size( Var Text: Text_Type ): LongInt;

(***********************************************************

  Function: Swap lines.

    This function attempts to swap the data of the two given
    text rows.  It fails only under very unusual conditions.

***********************************************************)

    Function Swap_Lines( Var Text: Text_Type; Row1, Row2: LongInt ): Boolean;

(***********************************************************

  Procedure: Initialize Text.

    This procedure initializes the text variable for storing
    the data.  The text should always be initialized only
    once in the program, before it is used.

***********************************************************)

    Procedure Initialize_Text( Var Text: Text_Type );

(***********************************************************

  Procedure: Erase text.

    This procedure disposes of the text variable entirely
    and should only be after Dispose_Text.

***********************************************************)

    Procedure Erase_Text( Var Text: Text_Type );

(***********************************************************

  Procedure: Peek text line.

    This procedure returns the text line of Text, specified
    by Row in Line.

***********************************************************)

    Procedure Peek_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );

{----------------------------------------------------------------------------}

  Implementation

    Const
     { This constant defines an uninitialized pointer. }
      Nul = 0;

    {$DEFINE Quick} { Select alternate code for faster processing speed. }

{----------------------------------------------------------------------------}

(*************************************************

  Function: Valid data.
    This function returns true if the line is
    allocated.

*************************************************)

    Function Valid_Data( Info: Text_Info_Type ): Boolean;
      Begin
        Valid_Data := ( Info <> Nul );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Deallocate data pointer.
    This function releases the memory used by
    the line back to the free list and sets the
    line to a blank line.

*************************************************)

    Function Deallocate_Data_Pointer( Var Info: Text_Info_Type ): Boolean;
      Begin
        If ( Info = Nul )
          then
            Deallocate_Data_Pointer := True
          else
            Begin
              Deallocate_Data_Pointer := Dispose_Pointer( Info );
              Info := Nul;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize node.
    This procedure initialized a new part of the
    text storage structure to blank lines.

*************************************************)

    Procedure Internal_Initialize( Var Buffer: Text_Buffer_Type );
      Var
        Count: Word;
      Begin
        For Count := 1 to File_Length do
          Buffer.Data[ Count ] := Nul;
        Buffer.Line_Start := 1;
        Buffer.Previous := Nul;
        Buffer.Next := Nul;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal dispose.
    This function deallocates all the lines in
    a portion of the text structure.

*************************************************)

    Function Internal_Dispose( Var Buffer: Text_Buffer_Type ): Boolean;
      Var
        Count: Word;
        Okay: Boolean;
      Begin
        Okay := True;
        For Count := 1 to File_Length do
          If Okay and Valid_Data( Buffer.Data[ Count ] )
            then
              Okay := Deallocate_Data_Pointer( Buffer.Data[ Count ] );
        Internal_Dispose := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Save buffer.
    This procedure saves the text block data
    structure buffer.

*************************************************)

    Procedure Save_Buffer( Var Text: Text_Type );
      Begin
        Put_Data( Text.Where, Text.Buffer, SizeOf( Text_Buffer_Type ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Go forward.
    This procedure moves the text block buffer
    to the next block.

*************************************************)

    Procedure Go_Forward( Var Text: Text_Type );
      Begin
        If Valid_Data( Text.Buffer.Next )
          then
            Begin
              Text.Where := Text.Buffer.Next;
              Get_Data( Text.Where, Text.Buffer, SizeOf( Text_Buffer_Type ) );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Switch forward.
    This procedure saves the current buffer block
    then switches the buffer block to the next
    one.

*************************************************)

    Procedure Switch_Forward( Var Text: Text_Type );
      Begin
        If Valid_Data( Text.Where )
          then
            Begin
              Save_Buffer( Text );
              Go_Forward( Text );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Go reverse.
    This procedure moves the text block buffer
    to the previous block.

*************************************************)

    Procedure Go_Reverse( Var Text: Text_Type );
      Begin
        If Valid_Data( Text.Buffer.Previous )
          then
            Begin
              Text.Where := Text.Buffer.Previous;
              Get_Data( Text.Where, Text.Buffer, SizeOf( Text_Buffer_Type ) );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Switch Reverse.
    This procedure saves the current buffer block
    then switches the buffer block to the previous
    one.

*************************************************)

    Procedure Switch_Reverse( Var Text: Text_Type );
      Begin
        If Valid_Data( Text.Where )
          then
            Begin
              Save_Buffer( Text );
              Go_Reverse( Text );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal dispose text.
    This function disposes of the entire text
    structure, except the allocated block.

*************************************************)

    Function Internal_Dispose_Text( Var Text: Text_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := True;
        If Valid_Data( Text.Where )
          then
            Begin
              While Valid_Data( Text.Buffer.Next ) do
                Switch_Forward( Text );
              While Okay and Valid_Data( Text.Buffer.Previous ) do
                Begin
                  Okay := Internal_Dispose( Text.Buffer );
                  If Okay
                    then
                      Begin
                        Switch_Reverse( Text );
                        Okay := Dispose_Pointer( Text.Buffer.Next );
                        Text.Buffer.Next := Nul;
                      End;
                End;
              If Okay
                then
                  Okay := Internal_Dispose( Text.Buffer );
            End;
        Internal_Dispose_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Expand text.
    This function expands the text structure so
    that more lines can be added to it when
    necessary.

*************************************************)

    Function Expand_Text( Var Text: Text_Type ): Boolean;
      Var
        Hold: LongInt;
        Old_Pointer: Pointer_Type;
      Begin
        While Valid_Data( Text.Buffer.Next ) do
          Switch_Forward( Text );
        If ( Text.Buffer.Next = Nul )
          then
            Begin
              Old_Pointer := Text.Where;
              Text.Buffer.Next := New_Pointer( SizeOf( Text_Buffer_Type ) );
              Hold := Text.Buffer.Line_Start;
              Switch_Forward( Text );
              Internal_Initialize( Text.Buffer );
              Text.Buffer.Previous := Old_Pointer;
              Text.Buffer.Line_Start := ( Hold + File_Length );
              Switch_Reverse( Text );
              Expand_Text := True;
            End
          else
            Write_Error( 201, 'Expand_Text: Extention already initialized' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Page row.
    This function tries to shift the text
    structure block to the appropriate one
    containing the given line.  Then Row is
    altered to reflect the correct line of the
    current block.

*************************************************)

    Function Page_Row( Var Text: Text_Type; Var Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := True;
        If ( Row < Text.Buffer.Line_Start )
          then
            If ( Text.Buffer.Line_Start > 1 )
              then
                While ( Text.Buffer.Line_Start > Row ) do
                  If Valid_Data( Text.Buffer.Previous )
                    then
                      Switch_Reverse( Text )
                    else
                      Write_Error( 204, 'Page_Row: Pointer invalid' )
              else
                Write_Error( 204, 'Page_Row: Line_Start invalid' )
          else
            While Okay and ( Row >= ( Text.Buffer.Line_Start + File_Length ) ) do
              If ( Text.Buffer.Next = Nul )
                then
                  Begin
                    Okay := Expand_Text( Text );
                    If Okay
                      then
                        Switch_Forward( Text );
                  End
                else
                  Switch_Forward( Text );
        Row := Succ( Row - Text.Buffer.Line_Start );
        Page_Row := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get text size.
    As previously defined.

*************************************************)

    Function Get_Text_Size( Var Text: Text_Type ): LongInt;
      Begin
        Get_Text_Size := Text.FileSize;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get text line.
    As previously defined.

*************************************************)

    Procedure Get_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );
      Begin
        If ( Row > 0 ) and ( Page_Row( Text, Row ) and Valid_Data( Text.Buffer.Data[ Row ] ) )
          then
            Get_Buffer_Data( Text.Buffer.Data[ Row ], Line, SizeOf( Line_Type ) )
          else
            Line.Size := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Peek text line.
    As previously defined.

*************************************************)

    Procedure Peek_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );
      Begin
        If ( Row > 0 ) and ( Page_Row( Text, Row ) and Valid_Data( Text.Buffer.Data[ Row ] ) )
          then
            Peek_Buffer_Data( Text.Buffer.Data[ Row ], Line, SizeOf( Line_Type ) )
          else
            Line.Size := 0;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Swap lines.
    As previously defined.

*************************************************)

    Function Swap_Lines( Var Text: Text_Type; Row1, Row2: LongInt ): Boolean;
      Var
        Temporary: LongInt;
        Hold1,
        Hold2: Text_Info_Type;
      Begin
        Swap_Lines := True;
        Temporary := Row1;
        If Page_Row( Text, Temporary )
          then
            Begin
              Hold1 := Text.Buffer.Data[ Temporary ];
              If Page_Row( Text, Row2 )
                then
                  Begin
                    Hold2 := Text.Buffer.Data[ Row2 ];
                    Text.Buffer.Data[ Row2 ] := Hold1;
                    If Page_Row( Text, Row1 )
                      then
                        Text.Buffer.Data[ Row1 ] := Hold2
                      else
                        Swap_Lines := False;
                  End
                else
                  Swap_Lines := False;
            End
          else
            Swap_Lines := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Remove.
    This Function removes the given line from
    the structure.

*************************************************)

    Function Remove( Var Where: Text_Info_Type ): Boolean;
      Begin
        If Valid_Data( Where )
          then
            Remove := Deallocate_Data_Pointer( Where )
          else
            Remove := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Put text line.
    As previously defined.

*************************************************)

    Function Put_Text_Line( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type; Reduce: Boolean ): Boolean;
      Var
        Okay: Boolean;
        Size,
        Hold_Length: Word;
        Hold: LongInt;
        Temporary_Pointer: Pointer_Type;
      Begin
        Hold := Row;
        Okay := Page_Row( Text, Row );
        Hold_Length := Line.Size;
        If Reduce
          then
            Reduce_Line( Line );
        If ( ( Line.Size > 0 ) and Okay )
          then
            Begin
              If Valid_Data( Text.Buffer.Data[ Row ] )
                then
                  Okay := Deallocate_Data_Pointer( Text.Buffer.Data[ Row ] );
              If Okay
                then
                  Begin
                    Size := Calculate_Size( Line );
                    Temporary_Pointer := Put_New_Pointer( Size, Line );
                    If Valid_Data( Temporary_Pointer )
                      then
                        Text.Buffer.Data[ Row ] := Temporary_Pointer
                      else
                        Okay := False;
                  End;
            End
          else
            Okay := Remove( Text.Buffer.Data[ Row ] );
        If ( ( Hold > Text.FileSize ) and Okay )
          then
            Text.FileSize := Hold;
        Line.Size := Hold_Length;
        Put_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read text.
    As previously defined.

*************************************************)

    Function Read_Text( Var InFile: Text; Var Text: Text_Type ): Boolean;
      Var
        Okay: Boolean;
        Counter: LongInt;
      Begin
        Okay := True;
        If ( Text.Where = Nul )
          then
            Okay := Allocate_Text( Text );
        Counter := 0;
        While ( Okay and ( not EOF( InFile ) ) ) do
          Begin
            Inc( Counter );
            Okay := Read_Line( InFile, Work_Space^ );
            If Okay
              then
                Begin
                  Okay := Put_Text_Line( Text, Counter, Work_Space^, True );
                  If ( Okay and Odd( Counter ) )
                    then
                      Read_Status( Counter );
                End
          End;
        Read_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write text fast.
    As previously defined.

*************************************************)

    Function Write_Text( Var OutFile: File; Var Text: Text_Type ): Boolean;
      Var
        Okay: Boolean;
        Counter: LongInt;
      Begin
        Okay := True;
        For Counter := 1 to Text.FileSize do
          If Okay
            then
              Begin
                Peek_Text_Line( Text, Counter, Work_Space^ );
                Okay := Write_Line_Fast( OutFile, Work_Space^ );
                If ( Okay and Odd( Counter ) )
                  then
                    Write_Status( Counter, Text.FileSize );
              End;
        Write_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move back.
    This procedure moves the lines back in a text
    block so that a new one can be added.

*************************************************)

    Procedure Move_Back( Var Buffer: Text_Array_Type; Start, Finish: Word );
     {$IFDEF Quick}
      Begin
        If ( Start < Finish )
          then
            Move( Buffer[ Start ], Buffer[ Succ( Start ) ], ( Finish - Start ) * SizeOf( Text_Info_Type ) );
      End;
     {$ELSE}
      Var
        Count: Word;
      Begin
        For Count := Finish downto Succ( Start ) do
          Buffer[ Count ] := Buffer[ Pred( Count ) ];
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Move forward.
    This procedure moves the lines forward in a
    text block so that one can be deleted.

*************************************************)

    Procedure Move_Forward( Var Buffer: Text_Array_Type; Start, Finish: Word );
     {$IFDEF Quick}
      Begin
        If ( Start < Finish )
          then
            Move( Buffer[ Succ( Start ) ], Buffer[ Start ], ( Finish - Start ) * SizeOf( Text_Info_Type ) );
      End;
     {$ELSE}
      Var
        Count: Word;
      Begin
        For Count := Start to Pred( Finish ) do
          Buffer[ Count ] := Buffer[ Succ( Count ) ];
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Link back.
    This procedure moves the last line from the
    previous block to the next.

*************************************************)

    Procedure Link_Back( Var Text: Text_Type; Var Row: LongInt );
      Var
        Hold: Text_Info_Type;
      Begin
        While ( Row < Text.Buffer.Line_Start ) do
          Begin
            Move_Back( Text.Buffer.Data, 1, File_Length );
            Switch_Reverse( Text );
            Hold := Text.Buffer.Data[ File_Length ];
            Switch_Forward( Text );
            Text.Buffer.Data[ 1 ] := Hold;
            Switch_Reverse( Text );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Link forward.
    This procedure moves the first line from the
    next block to the previous.

*************************************************)

    Procedure Link_Forward( Var Text: Text_Type; Var Row: LongInt );
      Var
        Hold: Text_Info_Type;
      Begin
        While Valid_Data( Text.Buffer.Next ) do
          Begin
            Switch_Forward( Text );
            Hold := Text.Buffer.Data[ 1 ];
            Switch_Reverse( Text );
            Text.Buffer.Data[ File_Length ] := Hold;
            Switch_Forward( Text );
            Move_Forward( Text.Buffer.Data, 1, File_Length );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Insert text line.
    This function inserts a text line into the
    text structure at the

*************************************************)

    Function Insert_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
        Point: LongInt;
      Begin
        Point := Succ( Text.FileSize );
        Okay := Page_Row( Text, Point );
        If Okay
          then
            Begin
              Link_Back( Text, Row );
              Okay := Page_Row( Text, Row );
              If Okay
                then
                  Begin
                    Move_Back( Text.Buffer.Data, Row, File_Length );
                    Text.Buffer.Data[ Row ] := Nul;
                    Inc( Text.FileSize );
                  End;
            End;
        Insert_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal delete text line.
    This function deletes the given text line from
    the text structure.

*************************************************)

    Function Internal_Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := Page_Row( Text, Row );
        If Okay
          then
            Begin
              Move_Forward( Text.Buffer.Data, Row, File_Length );
              Link_Forward( Text, Row );
              Text.Buffer.Data[ File_Length ] := Nul;
              Dec( Text.FileSize );
              If ( Text.FileSize < Text.Buffer.Line_Start )
                then
                  If Valid_Data( Text.Buffer.Previous )
                    then
                      Begin
                        Okay := Internal_Dispose( Text.Buffer );
                        If Okay
                          then
                            Begin
                              Switch_Reverse( Text );
                              Okay := Dispose_Pointer( Text.Buffer.Next );
                              Text.Buffer.Next := Nul;
                            End;
                      End;
            End;
        Internal_Delete_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Delete text line.
    As previously defined.

*************************************************)

    Function Delete_Text_Line( Var Text: Text_Type; Row: LongInt ): Boolean;
      Var
        Okay: Boolean;
        New_Row: LongInt;
      Begin
        New_Row := Row;
        Okay := Page_Row( Text, New_Row );
        If Okay
          then
            Begin
              Okay := Remove( Text.Buffer.Data[ New_Row ] );
              If Okay
                then
                  Okay := Internal_Delete_Text_Line( Text, Row );
            End;
        Delete_Text_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Split text.
    As previously defined.

*************************************************)

    Function Split_Text( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type; Reduce: Boolean ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := Insert_Text_Line( Text, Succ( Data.Row ) );
        If Okay
          then
            Begin
              Append_Line( Buffer, ' ' );
              Delete_Line( Buffer, 1, Pred( Data.Column ) );
              Okay := Put_Text_Line( Text, Succ( Data.Row ), Buffer, True );
              If Okay
                then
                  Begin
                    Get_Text_Line( Text, Data.Row, Buffer );
                    Buffer.Size := Pred( Data.Column );
                    Okay := Put_Text_Line( Text, Data.Row, Buffer, Reduce );
                  End;
            End;
        Split_Text := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Remove return.
    As previously defined.

*************************************************)

    Function Remove_Return( Var Text: Text_Type; Var Buffer: Line_Type; Var Data: Point_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Get_Text_Line( Text, Succ( Data.Row ), Work_Space^ );
        Combine_Lines( Buffer, Work_Space^ );
        Okay := ( Data.Row > 0 ) and Put_Text_Line( Text, Data.Row, Buffer, True );
        If Okay
          then
            Okay := Delete_Text_Line( Text, Succ( Data.Row ) );
        Remove_Return := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate text.
    As previously defined.

*************************************************)

    Function Allocate_Text( Var Text: Text_Type ): Boolean;
      Begin
        Text.Where := New_Pointer( SizeOf( Text_Buffer_Type ) );
        Internal_Initialize( Text.Buffer );
        Text.FileSize := 0;
        Allocate_Text := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose text.
    As previously defined.

*************************************************)

    Procedure Dispose_Text( Var Text: Text_Type );
      Begin
        If not Internal_Dispose_Text( Text )
          then
            Write_Error( 201, 'Dispose_Text: Dispose failure' );
        Text.FileSize := 0
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Initialize text.
    As previously defined.

*************************************************)

    Procedure Initialize_Text( Var Text: Text_Type );
      Begin
        Text.Where := Nul;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Erase text.
    As previously defined.

*************************************************)

    Procedure Erase_Text( Var Text: Text_Type );
      Var
        Temporary_Pointer: Pointer_Type;
      Begin
        If Valid_Data( Text.Where )
          then
            Begin
              While Valid_Data( Text.Buffer.Previous ) do
                Switch_Reverse( Text );
              While Valid_Data( Text.Where ) do
                Begin
                  Temporary_Pointer := Text.Where;
                  Text.Where := Text.Buffer.Next;
                  If not Dispose_Pointer( Temporary_Pointer )
                    then
                      Write_Error( 201, 'Erase_Text: Dispose error' );
                End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Check allocation.
    As previously defined.

*************************************************)

    Function Check_Allocation( Var Text: Text_Type ): Boolean;
      Begin
        If ( Text.Where = Nul )
          then
            Check_Allocation := Allocate_Text( Text )
          else
            Check_Allocation := True;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Allocate the work space.

*************************************************)

  Begin
    New( Work_Space );
    If ( Work_Space = Nil )
      then
        RunError( 203 );
  End.

