{ *******************************************************************
  *			    This file is part of the WMFH package				*
  ******************************************************************* }


{

	 	Fs_unit.pas	   A Turbo Pascal unit for file name selection


			 Copyright (c) 1997   Gianfranco Boggio-Togna                              *

							C.P. 14021                                   
					  I-20140 Milano (Italy)                             

					 e-mail: gbt@computer.org                                                     *


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

}


{
						ACKNOWLEDGEMENTS

  The procedures for directory handling are based (with modifications)
  on the following module from the SWAG library:    

  > DIRDEMO.PAS
  > Author: Trevor Carlsen. Released into the public domain 1989
  > 						Last modification 1992.


  The function returning the drive list also comes from the SWAG library.

  > * Function ...... Drives
  > ...
  > * Copyright ..... None.  This routine is public domain.
  > * Author ........ Martin Richardson
  > ...

}


{$R+}
{$S+}
UNIT  FS_unit ;

{ ************************************************************************
  *																		 *
  *				        U N I T    I N T E R F A C E  					 *
  *																		 *
  ************************************************************************ }

INTERFACE

USES Dos, Crt, WND_Unit, KB_Unit, MS_Unit ;

VAR

	FS_text_BG:             Integer ;	{ the main window colors } 
	FS_text_FG:             Integer ;	

	FS_file_list_BG:        Integer ;	{ the file list window }
	FS_file_list_FG:        Integer ;

	FS_sel_file_BG:         Integer ;	{ the file selection bar }
	FS_sel_file_FG:         Integer ;

	FS_button_BG:           Integer ;	{ the button colors }
	FS_button_FG:           Integer ;

	FS_button_high_FG:      Integer ;	{ the highlighted letter }	
	FS_button_special_BG:   Integer ;	{ rename / delete / help / cancel }
	FS_button_special_FG:   Integer ;	{ drive list only }

	FS_first_line:			Integer ;	
	FS_last_line:			Integer ;	
	FS_first_column: 		Integer ;	
	FS_last_column:			Integer ;	

PROCEDURE FS_open ( directory:  String ;      { initial directory }
					mask:  		String ;      { file select mask (wildcard) }
					hide:		String ;	  { file exclude mask (wildcard) }	
					title:  	String ;      { 1st line title }
					sorting: 	   Char ;	  { directory sorting method }
					VAR filename:  String) ;  { set on exit }
	
{ Only files whose name matches 'mask' and does NOT match 'hide' are listed }
{ Note that 'mask' may be changed by the user, but 'hide' cannot		    }


{ ************************************************************************
  *																		 *
  *					 U N I T	I M P L E M E N T A T I O N				 *
  *																		 *
  ************************************************************************ }

IMPLEMENTATION
	
	
CONST
		Copyright:	String =
					'WMFH 1.0 - Copyright 1997 Gianfranco Boggio-Togna' ;

CONST

	{ These are the mininum values that can still produce a readable
	  screen.  If you want to reduced them, you will probably have to
	  alter the layout and make extensive changes to "set_coordinates"	}

	min_height = 18 ;	  { lines  }
	min_width  = 74 ;	  { colums }
	

	list_area = 0 ;
	name_area = 1 ;
	path_area = 2 ;
	mask_area = 3 ;
	
	
	hex_digit: String = '0123456789abcdef' ;
	
	max_dir_entries   = 1000 ;     { maximum number of directory entries }


TYPE
	
	Str3    = String [3] ;
	Str8    = String [8] ;
	Str18   = String [18] ;
	SType   = ( _name,      _ext,      _date,      _size,
				_name_desc, _ext_desc, _date_desc, _size_desc,
				_dir ) ;
	
	Dir_rec  = RECORD
				   name :  NameStr ;
				   ext  :  ExtStr ;
				   size :  Str8 ;
				   date :  Str18 ;
				   Lsize,
				   Ldate:  LongInt ;
				   dir  :  Boolean ;
			   END ;
	
	Dir_rec_ptr = ^ Dir_rec ;
	
	Dir_ptrs    =  ARRAY [1..max_dir_entries] OF Dir_rec_ptr ;
	
	Process_Status = (PS_continue, PS_cancelled, PS_selected,
					  PS_area_changed, PS_mouse_clicked ) ;
	
	Mouse_zone = RECORD
					 min_y:     Byte ;
					 max_y:     Byte ;
					 min_x:     Byte ;
					 max_x:     Byte ;
					 mode:      Byte ;
					 area:      Byte ;
					 key:       Word ;
				 END ;
	
CONST

	months:   ARRAY [1..12] OF Str3 =
				( 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
				  'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' ) ;
	
	
	max_zones = 100 ;

VAR
	
	{ area coordinates }
	
	help_x,
	help_y,
	help_w,

	drives_y,
	drives_x,

	sort_options_y,
	sort_options_x,
	sort_options_text_y,

	action_buttons_y,
	action_buttons_x,

	file_list_lines,     { actually MINUS 1 }
	
	file_list_x,
	file_list_y,
	file_list_w,
	
	
	path_x, 
	path_y, 
	path_w,
	
	mask_x, 
	mask_y, 
	mask_w,
	
	name_x,
	
	name_y, 
	name_w,

	sort_text:	Integer ;


	max_file_list_lines,
	max_drive:  Integer ;       { max number of drive letters on screen }


	dir_entries:        Dir_ptrs ;
	mouse_zones:        ARRAY [1..max_zones] OF Mouse_zone ;
	
	first_entry:        Integer ;
	number_of_entries:  Integer ;
	path:               DirStr ;
	
	file_number:        Integer ;
	sel_number:         Integer ;

	last_filename:      String ;
	last_directory:     String ;
	restore:            Boolean ;
	
	current_name:       String ;
	current_path:       String ;
	current_mask:       String ;
	hide_mask:			String ;

	drives_list:        String ;
	last_drive_index:   Integer ;
	max_drive_area:     Integer ;

	area:               Integer ;
	n_areas:            Integer ;
	n_zones:            Integer ;
	no_save:            Integer ;
	i:                  Integer ;
	
	scroll_bar:          Integer ;

	last_sort_type:      SType ;
	requested_sort_type: SType ;
	

{ -----------------------------------------------------------------------
					Service procedures and functions
  ----------------------------------------------------------------------- }


FUNCTION rtrim (s:  String):   String ;
VAR
	ws:  String ;
BEGIN
	ws := s ;
	WHILE (Length (ws) > 0) AND (ws [Length (ws) ] = ' ')  DO
	  Dec (ws [0]) ;
	rtrim := ws ;
END ;

PROCEDURE up_case_str (VAR s:  String) ;
VAR
	i:    Integer ;
BEGIN
	FOR  i := 1  TO  Length (s)  DO
	  s [i] := UpCase (s [i]) ;
END ;


PROCEDURE  add_mouse_zone (y_min, x_min, y_max, x_max, m, a:  Byte; k:  Word) ;
BEGIN
	Inc (n_zones) ;
	WITH  mouse_zones [n_zones]  DO
	  BEGIN
		min_y := y_min ;
		max_y := y_max ;
		min_x := x_min ;
		max_x := x_max ;
		mode  := m ;
		area  := a ;
		key   := k ;
	  END ;
END ;

FUNCTION  alt_letter (letter: Char):  Word ;
CONST
	alt_letters: ARRAY [0..25]  OF  Word =
			  ( KB_AltA, KB_AltB, KB_AltC, KB_AltD, KB_AltE, KB_AltF,
				KB_AltG, KB_AltH, KB_AltI, KB_AltJ, KB_AltK, KB_AltL,
				KB_AltM, KB_AltN, KB_AltO, KB_AltP, KB_AltQ, KB_AltR,
				KB_AltS, KB_AltT, KB_AltU, KB_AltV, KB_AltW, KB_AltX,
				KB_AltY, KB_AltZ ) ;
BEGIN
	alt_letter := alt_letters [Ord (letter) - Ord ('A') ]  ;
END ;

{ -----------------------------------------------------------------------
					Sort the directory entries
  ----------------------------------------------------------------------- }

PROCEDURE quicksort (VAR data:  Dir_ptrs ;
				 	 left, right:  Word ;
				  	 SortType:  SType) ;
VAR
	pivotBool:  			Boolean ;
	pivotStr, tempStr:  	String ;
	pivotLong, tempLong:  	LongInt ;
	lower, upper, middle:  	Word ;
	t:  					Dir_rec_ptr ;
	
BEGIN
	IF  right < left {+ 2}  THEN
		Exit ;
	lower := left ;
	upper := right ;
	middle := (left + right) DIV 2 ;
	CASE SortType OF
	  _dir              :  pivotBool  := data [middle]^.dir ;
	  _name, _name_desc :  pivotStr   := data [middle]^.name ;
	  _ext,  _ext_desc  :  pivotStr   := data [middle]^.ext ;
	  _size, _size_desc :  pivotLong  := data [middle]^.Lsize ;
	  _date, _date_desc :  pivotLong  := data [middle]^.Ldate ;
	END ; { CASE SortType }
	
	REPEAT
	  CASE SortType OF
		_dir: 
		BEGIN
		  WHILE data [lower]^.dir > pivotBool DO
			Inc (lower) ;
		  WHILE pivotBool > data [upper]^.dir DO
			Dec (upper) ;
		END ;
		_name: 
		BEGIN
		  WHILE data [lower]^.name < pivotStr DO
			Inc (lower) ;
		  WHILE pivotStr < data [upper]^.name DO
			Dec (upper) ;
		END ;
		_name_desc: 
		BEGIN
		  WHILE data [lower]^.name > pivotStr DO
			Inc (lower) ;
		  WHILE pivotStr > data [upper]^.name DO
			Dec (upper) ;
		END ;
		_ext:  
		BEGIN
		  WHILE data [lower]^.ext < pivotStr DO
			Inc (lower) ;
		  WHILE pivotStr < data [upper]^.ext DO
			Dec (upper) ;
		END ;
		_ext_desc:  
		BEGIN
		  WHILE data [lower]^.ext > pivotStr DO
			Inc (lower) ;
		  WHILE pivotStr > data [upper]^.ext DO
			Dec (upper) ;
		END ;
		_size: 
		BEGIN
		  WHILE data [lower]^.Lsize < pivotLong DO
			Inc (lower) ;
		  WHILE pivotLong < data [upper]^.Lsize DO
			Dec (upper) ;
		END ;
		_size_desc: 
		BEGIN
		  WHILE data [lower]^.Lsize > pivotLong DO
			Inc (lower) ;
		  WHILE pivotLong > data [upper]^.Lsize DO
			Dec (upper) ;
		END ;
		_date:  
		BEGIN
		  WHILE data [lower]^.Ldate < pivotLong DO
			Inc (lower) ;
		  WHILE pivotLong < data [upper]^.Ldate DO
			Dec (upper) ;
		END ;
		_date_desc:  
		BEGIN
		  WHILE data [lower]^.Ldate > pivotLong DO
			Inc (lower) ;
		  WHILE pivotLong > data [upper]^.Ldate DO
			Dec (upper) ;
		END ;
	  END ; { CASE SortType }
	  
	  IF lower <= upper THEN
		BEGIN
		  t := data [lower] ;
		  data [lower] := data [upper] ;
		  data [upper] := t ;
		  Inc (lower) ;
		  Dec (upper) ;
		END ;
	  
	UNTIL lower > upper ;
	
	IF left < upper THEN
	  quicksort (data, left, upper, SortType) ;
	IF lower < right THEN
	  quicksort (data, lower, right, SortType) ;
	
END ; { quicksort }


{ -----------------------------------------------------------------------
				 		Read the current directory 
  ----------------------------------------------------------------------- }

PROCEDURE read_directory ;

VAR
	dirinfo:      SearchRec ;
	i:           Integer ;
	
FUNCTION  wild_match ( pattern, name: String) : Boolean ;

VAR
	p, n, lp, ln:	Integer ;
	c:				Char ;
	dot_found:		Boolean ;

BEGIN

	wild_match := False ;
	lp := Length (pattern) ;
	ln := Length (name) ;
	p := 1 ;
	n := 1 ;

	IF  lp = 0  THEN
		Exit ;

	IF  pattern = '*.*'  THEN		{ to emulate DOS conventions }
	  BEGIN
		wild_match := True ;
		Exit ;
	  END ;
	WHILE  p <= lp  DO
	  BEGIN
	  	c := pattern[p] ;
		Inc(p) ;

		CASE  c  OF

		'?':	BEGIN
					IF  n > ln  THEN
						Exit ;
 					IF  name[n] = '.'  THEN
						Exit ;
					Inc (n) ;
				END ;

		'*':	BEGIN
					IF  p <= lp  THEN
						c := pattern[p] ;
					WHILE  (p <= lp)  AND  (c = '*')  DO
					  BEGIN
  						Inc (p) ;
						c := pattern[p] ;
					  END ;
					IF  p > lp  THEN
					  BEGIN
						IF  Pos ('.', Copy(name, n, ln-n+1)) = 0  THEN
							wild_match := True ;
				 		Exit ;
					  END ;
					IF  c = '.'	THEN
					  BEGIN
						WHILE  (n <= ln)  AND  (name[n] <> '.')  DO
							Inc (n) ;
						IF  n > ln  THEN
							Exit ;
					  END
					ELSE
					  BEGIN
						dot_found := False ;
					  	WHILE  (n <= ln)  AND NOT dot_found  DO
						  BEGIN
						  	IF  wild_match (Copy (pattern, p, lp-p+1),
											Copy (name, n, ln-n+1) ) THEN
							  BEGIN
								wild_match := True ;
								Exit ;
							  END ;
							IF  name[n] = '.' THEN
								dot_found := True
							ELSE
								Inc (n) ;
						  END ;
						Exit ;
					  END ;
				END ;
		ELSE
				BEGIN
					IF  c <> name[n]  THEN
						Exit ;
					Inc (n) ;
				END ;
		END ;

	  END ;

	IF  n > ln  THEN 
		wild_match := True ;

END ;


FUNCTION create_record:  Dir_rec_ptr ;
VAR
	Dt:   DateTime ;
	st:   Str8 ;
	p:    Dir_rec_ptr ;
BEGIN
	p := New (Dir_rec_ptr) ;
	WITH p^ DO
	  BEGIN
		FillChar (p^, SizeOf (Dir_rec), 32) ;          {initialize}
		FSplit (dirinfo.name, path, name, ext) ;       { Split File name up }
		IF ext [1] = '.' THEN                          { get rid of dot }
		  ext := Copy (ext, 2, 3) ;
		IF ext [1] = '.' THEN
		  BEGIN
			IF ext [2] = '.'  THEN
			  name := '..      ' 
			ELSE
			  name := '.       ' ;
			ext := '   ' ;
		  END ;
		{name[0] := #8 ;  ext[0] := #3}
		; { Force to a set length For Formatting }
		Lsize := dirinfo.size ;
		Ldate := dirinfo.time ;
		Str (dirinfo.size:  8, size) ;
		UnpackTime (dirinfo.time, Dt) ;
		date := '' ;
		Str (Dt.day:  2, st) ;
		date := st + ' ' + months [Dt.month] + ' ' ;
		Str ( Dt.year:  4, st) ;
		date := date + st + #255#255 ;
		Str (Dt.hour:  2, st) ;
		date := date + st + ':' ;
		Str (Dt.Min:  2, st) ;
		IF  st[1] = ' ' THEN
			st[1] := '0' ;
		date := date + st ;
		dir := dirinfo.attr AND directory = directory ;
	  END ; { WITH }
	create_record := p ;
END ; { create_record }

BEGIN { read_directory }
	FOR  i := 1  TO  number_of_entries  DO
	  Dispose (dir_entries [i]) ;
	number_of_entries := 0 ;  { For keeping a Record of the number of entries read }
	FindFirst ('*.*', AnyFile-VolumeID, dirinfo) ;
	WHILE (DosError = 0) AND (number_of_entries < max_dir_entries) DO
	  BEGIN
		IF  (dirinfo.attr = directory) OR
		  ( wild_match ( current_mask, dirinfo.name)  AND
		    NOT wild_match ( hide_mask, dirinfo.name) )  THEN
		  BEGIN
			Inc (number_of_entries) ;
			dir_entries [number_of_entries] := create_record ;
		  END ;
		FindNext (dirinfo) ;
	  END ; { WHILE }
	max_file_list_lines := file_list_lines ;
	IF  number_of_entries > 0  THEN
	  BEGIN
		IF number_of_entries < file_list_lines + 1 THEN
		  max_file_list_lines := number_of_entries - 1 ;
		quicksort (dir_entries, 1, number_of_entries, _dir) ;
		first_entry := 1 ;
		WHILE (first_entry <= number_of_entries) AND 
			  (dir_entries [first_entry]^.dir)  DO
		  Inc (first_entry) ;
		IF  first_entry > 1  THEN
		  quicksort (dir_entries, 1, first_entry - 1, _name) ;
		IF  NOT restore  THEN
		  last_sort_type := requested_sort_type ;
		quicksort (dir_entries, first_entry, number_of_entries, last_sort_type) ;
		IF  NOT restore  THEN
		  BEGIN
			file_number := 1 ;
			sel_number := 1 ;
		  END 
		ELSE
		  restore := False ;
	  END
	ELSE
	  BEGIN
		max_file_list_lines := -1 ;
		file_number := 0 ;
		sel_number := 0 ;
	  END ;
END ; { read_directory }


{ -----------------------------------------------------------------------
				 		Display the directory 
  ----------------------------------------------------------------------- }

PROCEDURE display_directory ;
VAR
	x, y:  Integer ;
	nm:  String [10] ;
	ex:  String ;
BEGIN
	MS_Hide;
	TextColor (FS_file_list_FG) ; 
	y := file_list_y ;
	IF  file_number <> 0  THEN
	  FOR x := file_number TO file_number + max_file_list_lines DO
		WITH dir_entries [x]^ DO
		BEGIN
		  GotoXY (file_list_x + 1, y) ;
		  Inc (y) ;
		  nm := name ;
		  ex := ext ;
		  WHILE (Length (nm) > 0)  AND (nm [Length (nm) ] = ' ')  DO
			Dec (nm [0]) ;
		  IF dir THEN
			nm := '[' + nm + ']' ;
		  WHILE Length (nm) < 10 DO
			BEGIN
			  Inc (nm [0]) ;
			  nm [Length (nm) ] := ' ' ;
			END ;
		  WHILE Length (ex) < 3 DO
			BEGIN
			  Inc (ex [0]) ;
			  ex [Length (ex) ] := ' ' ;
			END ;
		  IF  x <> sel_number  THEN
			BEGIN
			  TextColor (FS_file_list_FG) ; 
			  TextBackground (FS_file_list_BG) ; 
			END 
		  ELSE
			BEGIN
			  TextColor (FS_sel_file_FG) ;
			  TextBackground (FS_sel_file_BG) ;
			END ;
		  Write (' ') ;
		  Write (nm, ' ') ;
		  Write (ex, ' ') ;
		  Write (size:  8, date:  20) ;
		  Write (' ') ;
		  IF  x = sel_number  THEN
			BEGIN
			  GotoXY (name_x + 1, name_y) ;
			  TextBackground (FS_file_list_BG) ;
			  Write ('            ') ;
			  IF  dir  THEN
				current_name := '' 
			  ELSE
				BEGIN
				  current_name := rtrim (nm) ;
				  IF  ex <> '   '  THEN
					current_name := current_name + '.' + rtrim (ex) ;
				  GotoXY (name_x + 1, name_y) ;
				  TextColor (FS_file_list_FG) ; 
				  Write (current_name);
				END ;
			END ;
		END ; { WITH }

	TextBackground (FS_file_list_BG) ;
	
	FOR x := max_file_list_lines TO file_list_lines - 1 DO
	  BEGIN
		GotoXY (file_list_x + 1, y) ;
		Inc (y) ;
		Write (' ':  file_list_w) ;
	  END ;
	WND_show_scroll_bar (scroll_bar, file_list_y, file_list_x + file_list_w + 1,
						 number_of_entries, file_number) ;
	MS_Show;
END ; { display_directory }



{ -----------------------------------------------------------------------
				 		Display drive list
  ----------------------------------------------------------------------- }

PROCEDURE  show_drives ;
VAR
	i, n:  Integer ;
	y, x:  Integer ;
	
{ Function 'Drives' is from the SWAG Library	}

FUNCTION Drives:  String ;

 {

 *****************************************************************************
 * Function ...... Drives
 * Purpose ....... To return a string containing the valid drives for the
 *                 current system.
 * Parameters .... None
 * Returns ....... A string of the valid drives.
 * Notes ......... None
 * Copyright ..... None.  This routine is public domain.
 * Author ........ Martin Richardson
 * Date .......... March 3, 1993
 *                 August 6, 1993 (fix)
 *****************************************************************************


 *****************************************************************************
 * FUNCTION ...... Drives
 * Purpose ....... To return a string containing the valid drives for the
 *                 current system.
 * Parameters .... None
 * Returns ....... A string of the valid drive letters.
 * Notes ......... Rather than changing to each drive to see if it exists, we
 *                 can instead call DOS FUNCTION 26h - Parse a file name.
 *                 If the file name is invalid (eg, F:), then DOS will say
 *                 so.  So, by testing each drive letter as a file name,
 *                 DOS will tell us which are good and which are bad!
 * Author ........ Martin Richardson
 * Date .......... August 6, 1993
 * Update ........ 02-01-94: Corrected problem where local VAR variables were
 *                           not being used, but a random memory location was
 *                           instead!
 *                        :  Added comments for clarity.
 *****************************************************************************

 }

ASSEMBLER ;
VAR
	driveinfo:    ARRAY [1..2]  OF Char ;
	buffer:       ARRAY [1..40] OF Char ;
	drivestring:  ARRAY [1..25] OF Char ;
	ASM
	PUSH   SI                     { Save Important Registers }
	PUSH   DI
	PUSH   ES
	PUSH   DS
	
	MOV    SI, SS                 { The Stack Segment (SS) points to the }
	MOV    DS, SI                 { VAR's above.  Point DS to it... }
	PUSH   DS
	POP    ES                     { ...and ES as well. }
	
	LEA    SI, driveinfo          { DS:SI - Where we test each drive letter }
	LEA    DI, buffer             { ES:DI - FCB Buffer }
	LEA    BX, drivestring        { DS:BX - Our resultant string }
	
	MOV    Byte Ptr [SI], '@'     { The character before 'A' }
	Xor    cx, cx                 { Zero out CX }
	
	@Scan: 
	Inc    Byte Ptr [SI]          { Next Drive Letter }
	MOV    Byte Ptr [SI + 1], ':'
	MOV    ax, $2906              { DOS FUNCTION 29h - Parse Filename }
	Int    21h                    {   DS:SI - String to be parsed }
	{   ES:DI - FCB }
	LEA    SI, driveinfo          { DS:SI }
	CMP    al, $FF                { AL = FFh if FUNCTION fails (invalid }
	JE     @NotValid              {     drive letter) }
	
	Inc    cx                     { Add one more to our string length... }
	PUSH   cx                     { ...and save it. }
	MOV    CL, Byte Ptr DS:  [SI]   { Grab the valid drive letter... }
	MOV    [BX], CL               { ...and stuff it into our result }
	Inc    BX                     { Next position in result string }
	POP    cx                     { Get our length counter back }
	
	@NotValid: 
	CMP    Byte Ptr [SI], 'Z'     { Did we go through all letters? }
	JNE    @Scan                  { Nope, so next letter }
	
	LEA    SI, drivestring        { Store DriveString to #Result }
	LES    DI, @Result
	Inc    DI
	REP    MOVSB
	
	XCHG   ax, DI                 { This is the only way to store the }
	MOV    DI, Word Ptr @Result   {   length that I can get to work. }
	SUB    ax, DI
	Dec    ax
	STOSB
	
	POP    DS                     { Restore Important Registers }
	POP    ES
	POP    DI
	POP    SI
END ;

BEGIN
	
	y := drives_y ;
	x := drives_x ;

	drives_list := Drives ;
	n := Length (drives_list) ;
	IF  n > max_drive THEN
	  n := max_drive ;
	
	last_drive_index := n ;

	FOR i := 1 TO n DO
	  BEGIN
		WND_open_window (y, x, y, x + 3,
						 FS_button_FG, FS_button_BG,
						 WND_narrow_shadow, WND_no_frame,
						 drives_list [i] + ':',
						 no_save) ;
		
		add_mouse_zone (y, x, y, x + 3, 0, list_area,
						alt_letter (drives_list [i])  ) ;
		Inc (x, 6) ;
	  END ;

	max_drive_area := n_zones ;

	IF  Length (drives_list) > max_drive THEN   
	  BEGIN
		WND_open_window (y, x - 2, y, x + 2,
						 FS_button_special_FG, FS_button_BG,
						 WND_narrow_shadow, WND_no_frame,
						 '\' +
						 hex_digit [FS_button_high_FG + 1] +
						 '\x\' +
						 hex_digit [FS_button_high_FG + 1] +
						 '\x',
						 no_save) ;
		add_mouse_zone (y, x - 2, y, x + 2, 2, list_area, 0) ;
	  END ;
END ;

{ -----------------------------------------------------------------------
				 	Display sort options buttons
  ----------------------------------------------------------------------- }

PROCEDURE  show_sort_options  ;
CONST
	labels:  ARRAY [1..2, 1..4] OF String [4] = ( 
				   ('name', 'ext ', 'size', 'time'),
				   ('Name', 'Ext ', 'Size', 'Time') ) ;
	select:   ARRAY [1..2, 1..4] OF Word = ( (KB_NN, KB_EE, KB_SS, KB_TT),
											 (KB_N , KB_E , KB_S , KB_T ) ) ;
	arrows:   ARRAY [1..2] OF String [1] = ('', '') ;
	shifts:   ARRAY [1..2] OF String [1] = ('', ' ') ;
	displ:    ARRAY [1..2] OF Integer = (0, 12) ;
VAR
	i, j, yy:  Integer ;
	y, x:  	   Integer ;
BEGIN
	
	y := sort_options_y ;
	x := sort_options_x ;

	IF  (sort_options_text_y > 0)  AND (sort_text <> 0)  THEN
	 BEGIN
		GotoXY (x, sort_options_text_y) ;
		TextColor (FS_text_FG) ;
		TextBackground (FS_text_BG) ;
		Write ('   Sort files by: ') ;
	 END ;
	FOR  i := 1  TO  2  DO
	  BEGIN
		yy := y ;
		FOR  j := 1  TO  4  DO
		  BEGIN
			WND_open_window (yy, x + displ [i], yy, x + displ [i] + 7,
							 FS_button_FG, FS_button_BG,
							 WND_narrow_shadow, WND_no_frame,
							 arrows [i] + ' \' +
							 		hex_digit [FS_button_high_FG + 1] +
									Copy (labels [i, j], 1, 1) +
									'\x' + Copy (labels [i, j], 2, 4),
							 no_save) ;
			
			add_mouse_zone (yy, x + displ [i], yy, x + displ [i] + 7,
							0, list_area, select [i, j] ) ;
			Inc (yy, 2) ;
		  END ;
	  END ;
END ;


{ -----------------------------------------------------------------------
				 		Display action buttons
  ----------------------------------------------------------------------- }

PROCEDURE  show_action_buttons  ;
CONST
	displ:    ARRAY [1..2] OF Integer = (0, 12) ;
VAR
	i, j, yy:  Integer ;
	y, x:  	   Integer ;
BEGIN
	y := action_buttons_y ;
	x := action_buttons_x ;

	WND_open_window (y, x + displ [1], y, x + displ [1] + 7,
					 FS_button_FG, FS_button_special_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [FS_button_high_FG + 1] + 'R\xename',
					 no_save) ;
	add_mouse_zone (y, x + displ [1], y, x + displ [1] + 7,
					0, list_area, KB_RR) ;
	
	WND_open_window (y, x + displ [2], y, x + displ [2] + 7,
					 FS_button_FG, FS_button_special_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [FS_button_high_FG + 1] + 'D\xelete',
					 no_save) ;
	add_mouse_zone (y, x + displ [2], y, x + displ [2] + 7,
					0, list_area, KB_DD) ;
	
	WND_open_window (y + 2, x + displ [1], y + 2, x + displ [1] + 7,
					 FS_button_FG, FS_button_special_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [FS_button_high_FG + 1] + 'H\xelp',
					 no_save) ;
	add_mouse_zone (y + 2, x + displ [1], y + 2, x + displ [1] + 7,
					0, list_area, KB_HH) ;
	
	WND_open_window (y + 2, x + displ [2], y + 2, x + displ [2] + 7,
					 FS_button_FG, FS_button_special_BG,
					 WND_narrow_shadow, WND_no_frame,
					 '\' + hex_digit [FS_button_high_FG + 1] + 'C\xancel',
					 no_save) ;
	add_mouse_zone (y + 2, x + displ [2], y + 2, x + displ [2] + 7,
					0, list_area, KB_CC) ;
END ;


{ -----------------------------------------------------------------------
				 		Display the file entries
  ----------------------------------------------------------------------- }

PROCEDURE  show_file_list  ;
VAR
	i:  Integer ;
BEGIN
	WND_open_window (file_list_y, file_list_x,
					 file_list_y + file_list_lines, file_list_x + file_list_w,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '',
					 no_save) ;
	add_mouse_zone (file_list_y, file_list_x,
					file_list_y + file_list_lines, file_list_x + file_list_w,
					1, list_area, 0) ;

	GotoXY (name_x - 5, name_y) ;
	TextColor (FS_text_FG) ;
	TextBackground (FS_text_BG) ;
	Write ('File:') ;
	WND_open_window (name_y, name_x,
					 name_y, name_x + name_w,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '',  no_save) ;
	add_mouse_zone ( name_y, name_x,
					 name_y, name_x + name_w,
					 0, name_area, 0) ;
END ;


{ -----------------------------------------------------------------------
				 		Display path and mask 
  ----------------------------------------------------------------------- }

PROCEDURE show_path_and_mask ;

VAR
	i:   Integer ;
	
BEGIN
	GotoXY (path_x - 5, path_y) ;
	TextColor (FS_text_FG) ;
	TextBackground (FS_text_BG) ;
	Write ('Path:') ;
	WND_open_window (path_y, path_x, path_y, path_x + path_w,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '',
					 no_save) ;
	add_mouse_zone (path_y, path_x, path_y, path_x + path_w,
					0, path_area, 0) ;
	
	GotoXY (path_x, path_y) ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	FOR  i := 1 TO  path_w  DO
	  Write (' ') ;
	
	GotoXY (mask_x - 5, path_y) ;
	TextColor (FS_text_FG) ;
	TextBackground (FS_text_BG) ;
	Write ('Mask:') ;
	WND_open_window (mask_y, mask_x, mask_y, mask_x + mask_w,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '',
					 no_save) ;
	add_mouse_zone (mask_y, mask_x, mask_y, mask_x + mask_w,
					0, mask_area, 0) ;
	GotoXY (mask_x, path_y) ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	FOR  i := 1 TO  mask_w  DO
	  Write (' ') ;
END ;


{ -----------------------------------------------------------------------
				 	Update all screen areas
  ----------------------------------------------------------------------- }

PROCEDURE  update_all_areas ;
VAR
	i:   Integer ;
	
BEGIN
	GotoXY (path_x, path_y) ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	FOR  i := 1 TO  path_w  DO
	  Write (' ') ;
	GotoXY (path_x + 1, path_y) ;
	current_path := FExpand ('.') ;
	GotoXY (path_x + 1, path_y) ;
	Write (current_path) ;
	
	GotoXY (mask_x - 5, mask_y) ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	GotoXY (mask_x + 1, mask_y) ;
	FOR  i := 1 TO  mask_w  DO
	  Write (' ') ;
	GotoXY (mask_x + 1, mask_y) ;
	Write (current_mask) ;
	
	read_directory ;
	display_directory ;
END ;


{ -----------------------------------------------------------------------
				 	 Switch to another drive
  ----------------------------------------------------------------------- }

PROCEDURE  change_drive (key:  Word) ;

VAR
	ws:  String ;
	n:   Integer ;
	current_dir: String ;
	
BEGIN
	CASE key OF
	  KB_AltA:  n :=  1 ; KB_AltB:  n :=  2 ; KB_AltC:  n :=  3 ;
	  KB_AltD:  n :=  4 ; KB_AltE:  n :=  5 ; KB_AltF:  n :=  6 ;
	  KB_AltG:  n :=  7 ; KB_AltH:  n :=  8 ; KB_AltI:  n :=  9 ;
	  KB_AltJ:  n := 10 ; KB_AltK:  n := 11 ; KB_AltL:  n := 12 ;
	  KB_AltM:  n := 13 ; KB_AltN:  n := 14 ; KB_AltO:  n := 15 ;
	  KB_AltP:  n := 16 ; KB_AltQ:  n := 17 ; KB_AltR:  n := 18 ;
	  KB_AltS:  n := 19 ; KB_AltT:  n := 20 ; KB_AltU:  n := 21 ;
	  KB_AltV:  n := 22 ; KB_AltW:  n := 23 ; KB_AltX:  n := 24 ;
	  KB_AltY:  n := 25 ; KB_AltZ:  n := 26 ;
	END ;
	
	IF  n <= Length (drives_list)  THEN
	  BEGIN
		GetDir (0, current_dir) ;
		GetDir (n, ws) ;
		{$i-}
		Chdir (ws) ;
		IF  IOResult <> 0  THEN
		  Chdir (current_dir) ;
		{$i+}
		update_all_areas ;
	  END ;
	
END ;


{ -----------------------------------------------------------------------
						 	 Rename a file
  ----------------------------------------------------------------------- }

PROCEDURE  rename_file (filename:  String) ;
VAR
	key:            Word ;
	mouse:          Integer ;   
	new_name:       String ;
	window_ptr:     Integer ;
	f:  		    File ;
	
LABEL
	1 ;

BEGIN
	WND_save_cursor (True) ;
	window_ptr := 1 ;
	WND_open_window (name_y, name_x + 16,
					 name_y, name_x + name_w + 26,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '', window_ptr) ;
	
	GotoXY (name_x + 15  , name_y) ;
	TextColor (FS_text_FG) ;
	TextBackground (FS_text_BG) ;
	Write (' New name:') ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	
1: 
	mouse := 1 ;
	new_name := rtrim ( WND_input ( name_y, name_x + 26,
									filename,
									' ',
									'',
									12,
									0,
									True,
									key,
									mouse
								  )
					  ) ;
	CASE  key  OF

	  KB_Esc:  
	  ;


	  KB_Enter:  
	  	BEGIN
			{$i-}
			Assign (f, filename) ;
			IF  IOResult = 0  THEN
		  	  BEGIN
				Rename (f, new_name) ;
				IF  IOResult <> 0  THEN
			  	    ;
		  	  END ;
			update_all_areas ;
			{$i+}
	  	END ;
	ELSE
	  GOTO 1 ;
	END;

	WND_close_window (window_ptr) ;
	WND_restore_cursor ;
END ;


{ -----------------------------------------------------------------------
						 	 Delete a file
  ----------------------------------------------------------------------- }

PROCEDURE  delete_file (filename:  String) ;
VAR
	key:         Word ;
	mouse:       Integer ;   
	option:      String ;
	window_ptr:  Integer ;
	f:      	 File ;
LABEL
	1 ;

BEGIN
	WND_save_cursor (True) ;
	window_ptr := 1 ;

	WND_open_window (name_y, name_x + 16,
					 name_y, name_x + name_w + 22,
					 FS_file_list_FG, FS_file_list_BG,
					 WND_no_shadow, WND_no_frame,
					 '',  window_ptr) ;
	
	GotoXY (name_x + 15  , name_y) ;
	TextColor (FS_text_FG) ;
	TextBackground (FS_text_BG) ;
	Write (' Delete file? ') ;
	TextColor (FS_file_list_FG) ;
	TextBackground (FS_file_list_BG) ;
	
1: 
	mouse := 1 ;
	option := rtrim  ( WND_input ( name_y, name_x + 31,
								   'yes',
								   ' ',
								   '',
								   3,
								   0,
								   True,
								   key,
								   mouse
								 )
					 ) ;

	CASE  key  OF
	  
	  KB_Esc:   ;


	  KB_Enter:  

	  BEGIN
		{$i-}
		IF (option [1] = 'y')  OR (option [1] = 'Y')  THEN
		  BEGIN
			Assign (f, filename) ;
			IF  IOResult = 0  THEN
			  BEGIN
				Erase (f) ;
				IF  IOResult <> 0  THEN
				  ;
			  END ;
		  END ;
		update_all_areas ;
		{$i+}
	  END ;

	ELSE
	  GOTO 1 ;
	END;

	WND_close_window (window_ptr) ;
	WND_restore_cursor ;

END ;

{ -----------------------------------------------------------------------
						  Show help screen	 
  ----------------------------------------------------------------------- }

PROCEDURE  show_help ;

VAR
	key:         	   Word ;
	window_ptr:  	   Integer ;
	dummy_window_ptr:  Integer ;

BEGIN

	WND_save_cursor (False) ;
	window_ptr := 1 ;

	WND_open_window (help_y, help_x,
					 help_y + min_height - 2, help_x + help_w - 1,
					 FS_button_high_FG, FS_button_BG,
					 WND_no_shadow, WND_no_frame,
					 '',  window_ptr) ;
	
	dummy_window_ptr := 1 ;
	WND_open_window (help_y, help_x + 1,
					 help_y + min_height - 2 , help_x + help_w - 2,
					 FS_text_BG, FS_button_BG,
					 WND_no_shadow, WND_single_frame,
					 'Help',  dummy_window_ptr) ;
	
	GotoXY (help_x + 6, help_y + 2) ;
	TextColor (FS_text_BG) ;
	Write ('To switch to another drive, press ') ;
	TextColor (FS_button_high_FG) ;
	Write ('Alt') ;
	TextColor (FS_text_BG) ;
	Write (' + ') ;
	TextColor (FS_button_high_FG) ;
	Write ('drive letter') ;
	TextColor (FS_text_BG) ;
	Write ('.') ;
	GotoXY (help_x + 6, help_y + 4) ;
	Write ('To move from one input field to the next press ') ;
	TextColor (FS_button_high_FG) ;
	Write ('Tab') ;
	TextColor (FS_text_BG) ;
	Write ('.') ;
	GotoXY (help_x + 6, help_y + 5) ;
	Write ('To move to the previous field press ') ;
	TextColor (FS_button_high_FG) ;
	Write ('Shift Tab') ;
	TextColor (FS_text_BG) ;
	Write ('.') ;
	GotoXY (help_x + 6, help_y + 7) ;
	Write ('To activate a button, type the highlighted letter.') ;
	GotoXY (help_x + 6, help_y + 9) ;
	Write ('To move up/down the directory tree use the ') ;
	TextColor (FS_button_high_FG) ;
	Write ('left');
	TextColor (FS_text_BG) ;
	Write ('/');
	TextColor (FS_button_high_FG) ;
	Write ('right');
	TextColor (FS_text_BG) ;
	Write (' arrow.');
	GotoXY (help_x + 6, help_y + 10) ;
	Write ('To select a file, move the selection bar over the file name') ;
	GotoXY (help_x + 6, help_y + 11) ;
	Write ('and press ') ;
	TextColor (FS_button_high_FG) ;
	Write ('Enter') ;
	TextColor (FS_text_BG) ;
	Write ('.  With the mouse, click on the file name to') ;
	GotoXY (help_x + 6, help_y + 12) ;
	Write ('place the selection bar over it. Click on the bar to select') ;
	GotoXY (help_x + 6, help_y + 13) ;
	Write ('the file.  To exit without selecting a file, use the ') ; 
	TextColor (FS_button_high_FG) ;
	Write ('Cancel') ;
	TextColor (FS_text_BG) ;
	GotoXY (help_x + 6, help_y + 14) ;
	Write ('button, or press ') ;
	TextColor (FS_button_high_FG) ;
	Write ('Escape') ;
	TextColor (FS_text_BG) ;
	Write ('.') ;
	REPEAT
	UNTIL  KeyPressed OR MS_LeftPressed OR MS_RightPressed ;
	IF  KeyPressed  THEN
		key := KB_read ;
	IF MS_LeftPressed THEN
		REPEAT UNTIL NOT MS_LeftDown ;
	IF MS_RightPressed THEN
		REPEAT UNTIL NOT MS_RightDown ;

	WND_close_window (window_ptr) ;
	WND_restore_cursor ;
END ;	


{ -----------------------------------------------------------------------
				Process user action within the file list
  ----------------------------------------------------------------------- }

PROCEDURE  process_file_list (key:  Word ;
							  VAR filename:  String ;
							  VAR status:  Process_Status)   ;
VAR
	i:      Integer ;
	ws:     String ;

BEGIN
	status := PS_continue ;
	
	CASE key OF
	  

	  KB_NN, KB_F5: 
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _name) ;
			last_sort_type := _name ;
	  	END ;
	  

	  KB_N,	KB_ShiftF5:  
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _name_desc) ;
			last_sort_type := _name_desc ;
	  	END ;
	  

	  KB_EE, KB_F6:        
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _ext) ;
			last_sort_type := _ext ;
	  	END ;
	  

	  KB_E,	KB_ShiftF6:  
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _ext_desc) ;
			last_sort_type := _ext_desc ;
	  	END ;

	  
	  KB_SS, KB_F7:        
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _size) ;
			last_sort_type := _size ;
	  	END ;

	  
	  KB_S,	KB_ShiftF7:  
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _size_desc) ;
			last_sort_type := _size_desc ;
	  	END ;

	  
	  KB_TT, KB_F8:       
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _date) ;
			last_sort_type := _date ;
	  	END ;

	  
	  KB_T,	KB_ShiftF8:  
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
			quicksort (dir_entries, first_entry, number_of_entries, _date_desc) ;
			last_sort_type := _date_desc ;
	  	END ;

	  
	  KB_Home:     
	  	BEGIN
			file_number := 1 ;
			sel_number := 1 ;
	  	END ;

	  
	  KB_End:      
	  	BEGIN
			file_number := number_of_entries - max_file_list_lines ;
			sel_number := number_of_entries ;
	  	END ;

	  
	  KB_Up:       
	  	BEGIN
			IF sel_number > 1 THEN
			  	Dec (sel_number) ;
			IF sel_number < file_number THEN
			  	Dec (file_number) ;
	  	END ;

	  
	  KB_Down:     
	  	BEGIN
			IF sel_number < number_of_entries  THEN
			  	Inc (sel_number) ;
			IF sel_number > (file_number + max_file_list_lines) THEN
			  	Inc (file_number) ;
	  	END ;

	  
	  KB_PgDown:   
	  	BEGIN
			i := sel_number - file_number ;
			IF (file_number + (file_list_lines + 1)) > 
		  	   (number_of_entries - (max_file_list_lines + 1)) THEN
			  	file_number := number_of_entries - max_file_list_lines 
			ELSE
			  	Inc (file_number, file_list_lines + 1) ;
			sel_number := file_number + i {max_file_list_lines} ;
	  	END ;

	  
	  KB_PgUp:     
	  	BEGIN
			i := sel_number - file_number ;
			IF (file_number - (file_list_lines + 1)) > 0 THEN
			  	Dec (file_number, file_list_lines + 1)
			ELSE
			  	file_number := 1 ;
			sel_number := file_number + i {max_file_list_lines} ;
	  	END ;
	  

	  KB_Enter:  
	  	IF dir_entries [sel_number]^.dir  THEN
		  BEGIN
		  	ws := rtrim (dir_entries [sel_number]^.name) ;
		  	Chdir (ws) ;
		  	update_all_areas ;
		  END 
	  	ELSE
	   	  BEGIN
		 	filename := current_name ;
		 	filename := FExpand ('.\' + filename) ;
		 	status := PS_selected ;
	   	  END ;
	  

	  KB_RR,  KB_R:
	  	rename_file (current_name) ;

	  
	  KB_DD, KB_D:
	  	delete_file (current_name) ;

	  KB_HH, KB_H, KB_F1:
	  	show_help ;

	  KB_Left:     
	  	BEGIN
			i := 1 ;
			WHILE  (i <= number_of_entries) AND
				   (dir_entries [i]^.name <> '..      ')  DO
			  	Inc (i) ;
			IF i <= number_of_entries  THEN
			  BEGIN
			  	Chdir ('..') ;
			  	update_all_areas ;
		  	  END ;
		END ;

	  
	  KB_Right:    
	  	IF dir_entries [sel_number]^.dir  THEN
		  BEGIN
		  	ws := rtrim (dir_entries [sel_number]^.name) ;
		  	Chdir (ws) ;
		  	update_all_areas ;
		  END ;
	  

	  KB_AltA, KB_AltB, KB_AltC, KB_AltD, KB_AltE,
	  KB_AltF, KB_AltG, KB_AltH, KB_AltI, KB_AltJ,
	  KB_AltK, KB_AltL, KB_AltM, KB_AltN, KB_AltO,
	  KB_AltP, KB_AltQ, KB_AltR, KB_AltS, KB_AltT,
	  KB_AltU, KB_AltV, KB_AltW, KB_AltX, KB_AltY,
	  KB_AltZ:  

		  change_drive (key) ;

	  
	  KB_CC, KB_C, KB_Esc:       
		  status := PS_cancelled ;

	  
	  KB_Tab:       
	  	BEGIN
			area := (area + 1) MOD n_areas ;
			status := PS_area_changed ;
	  	END ;

	  
	  KB_ShiftTab: 
	  	BEGIN
			area := (area + n_areas - 1) MOD n_areas ;
			status := PS_area_changed ;
	  	END ;
	  
	END {CASE}
	;
	
	display_directory ;
	
END ;


{ -----------------------------------------------------------------------
			   	Process user action within the 'path' field
  ----------------------------------------------------------------------- }

PROCEDURE  process_path (VAR status:  Process_Status) ;
VAR
	key:           Word ;
	mouse:         Integer ;    
	save_path:     String ;
LABEL
	1 ;

BEGIN
	WND_save_cursor (True) ;
	save_path := current_path ;
1: 
	GotoXY (path_x + 1, path_y) ;
	mouse := 1 ;
	current_path := rtrim ( WND_input ( path_y, path_x + 1,
										current_path, ' ', '',
										path_w, 0, True, key, mouse
						  )       	  ) ;

	CASE  key  OF

	  KB_Esc: 
		BEGIN
		  current_path := save_path ;
		  update_all_areas ;
		END ;

	  KB_Tab:   
	  	BEGIN
			current_path := save_path ;
			area := (area + 1) MOD n_areas ;
			status := PS_area_changed ;
	  	END ;


	  KB_ShiftTab:      
	  	BEGIN
			current_path := save_path ;
			area := (area + n_areas - 1) MOD n_areas ;
			status := PS_area_changed ;
	  	END ;

	  
	  KB_Enter:         
	  	BEGIN
			up_case_str (current_path);
			area := 0 ;
			{$i-}
			Chdir (current_path) ;
			IF  IOResult <> 0  THEN
			  BEGIN
				Chdir (save_path) ;
				current_path := save_path ;
		  	  END ;
			{$i+}
			update_all_areas ;
			status := PS_area_changed ;
	  	END ;


	ELSE
		GOTO 1 ;
END;

	WND_restore_cursor ;
	IF  mouse = 1 THEN
		status := PS_mouse_clicked ;
END ;


{ -----------------------------------------------------------------------
			   	Process user action within the 'name' field
  ----------------------------------------------------------------------- }

PROCEDURE  process_name (VAR filename:  String; VAR  status:  Process_Status) ;
VAR
	key:     Word ;
	mouse:   Integer ;
LABEL
	1 ;

BEGIN
	WND_save_cursor (True) ;
1: 
	GotoXY (name_x + 1, name_y) ;
	mouse := 1 ;
	filename := rtrim ( WND_input ( name_y, name_x + 1,
									current_name,
									' ',
									'',
									12,
									0,
									True,
									key,
									mouse
					  )  	      ) ;

	CASE  key  OF
	  
	  KB_Esc:           filename := '' ;
	  

	  KB_Tab:           
	  	BEGIN
		  filename := '' ;
		  area := (area + 1) MOD n_areas ;
		  status := PS_area_changed ;
	  	END ;

	  
	  KB_ShiftTab:      
	  	BEGIN
		  filename := '' ;
		  area := (area + n_areas - 1) MOD n_areas ;
		  status := PS_area_changed ;
	  	END ;

	  
	  KB_Enter:       
	  	BEGIN
		  status := PS_selected ;
		  up_case_str (filename);
		  filename := FExpand ('.\' + filename) ;
	  	END ;

	ELSE
	    GOTO 1 ;
	END;

	WND_restore_cursor ;
	IF  mouse = 1 THEN
		status := PS_mouse_clicked ;

END ;


{ -----------------------------------------------------------------------
			   	Process user action within the 'mask' field
  ----------------------------------------------------------------------- }

PROCEDURE  process_mask (VAR status:  Process_Status) ;
VAR
	key:           Word ;
	mouse:         Integer ;
	save_mask:     String ;
LABEL
	1 ;

BEGIN
	WND_save_cursor (True) ;
	save_mask := current_mask ;
1: 
	GotoXY (mask_x + 1, mask_y) ;
	mouse := 1 ;
	current_mask := rtrim ( WND_input ( mask_y, mask_x + 1,
										save_mask ,
										' ',
										'',
										12,
										0,
										True,
										key,
										mouse
				  		  )   	      ) ;

	CASE  key  OF
	  
	  KB_Esc: 
		BEGIN
		  current_mask := save_mask ;
		  update_all_areas ;
		END ;

	  KB_Tab:           
	  	BEGIN
		  current_mask := save_mask ;
		  area := (area + 1) MOD n_areas ;
		  status := PS_area_changed ;
	  	END ;
	  

	  KB_ShiftTab: 
	  	BEGIN
		  current_mask := save_mask ;
		  area := (area + n_areas - 1) MOD n_areas ;
		  status := PS_area_changed ;
	  	END ;
	  

	  KB_Enter:         
	  	BEGIN
		  up_case_str (current_mask);
		  update_all_areas ;
		  area := 0 ;
	  	END ;
	  
	ELSE
		GOTO 1 ;
	END;

	WND_restore_cursor ;
	IF  mouse = 1 THEN
		status := PS_mouse_clicked ;
END ;



{ -----------------------------------------------------------------------
		   	Translate mouse action to keyboard equivalent
  ----------------------------------------------------------------------- }

FUNCTION  translate_mouse:  Word ;
VAR
	i, n:  Integer ;
	x, y:  Byte ;
BEGIN

	x := MS_WhereX ;
	y := MS_WhereY ;

	IF  (x < FS_first_column)  OR  (x > FS_last_column)  OR
	    (y < FS_first_line) OR (y > FS_last_line)  THEN
	 BEGIN
		translate_mouse := KB_Esc ;
		area := list_area ;
		Exit ;
	 END ;

	i := 1 ;
	WHILE  (i <= n_zones) AND ( (x < mouse_zones[i].min_x)  OR
		   (x > mouse_zones[i].max_x)  OR
		   (y < mouse_zones[i].min_y)  OR
		   (y > mouse_zones[i].max_y)  )  DO
	  Inc (i) ;

	IF  i <= n_zones  THEN
	  BEGIN
		CASE  mouse_zones[i].mode  OF

		0:  BEGIN
			  area := mouse_zones[i].area ;
			  translate_mouse := mouse_zones[i].key ;
			END ;

		1:  BEGIN
			  area := list_area ;
			  n := y - file_list_y  ;
			  IF (n <= max_file_list_lines) AND
			     (x < file_list_x+file_list_w-1) THEN
				BEGIN
				  IF  (sel_number = file_number + n)  OR
					  dir_entries [file_number + n]^.dir THEN
					BEGIN
					  translate_mouse := KB_Enter ;
					  sel_number := file_number + n ;
					END 
				  ELSE
					BEGIN
					  sel_number := file_number + n ;
					  translate_mouse := KB_Space ;
					  display_directory ;
					END 
			   END ;
			END ;

		2:  BEGIN
			  area := list_area ;
			  IF  x = mouse_zones[i].min_x + 1  THEN
				BEGIN
				  IF  last_drive_index < Length (drives_list)  THEN
					BEGIN
					  Inc(last_drive_index) ;
					  gotoXY (x-4,y) ;
					  TextColor(FS_button_FG) ;
					  TextBackground(FS_button_BG) ;
					  write (drives_list[last_drive_index]) ;
					  mouse_zones[max_drive_area].key :=
						alt_letter (drives_list [last_drive_index]) ;
					END ;
				END 
			  ELSE
				IF  x = mouse_zones[i].min_x + 3  THEN
					IF  last_drive_index > max_drive  THEN
					  BEGIN
						Dec(last_drive_index) ;
						gotoXY (x-6,y) ;
						TextColor(FS_button_FG) ;
						TextBackground(FS_button_BG) ;
						write (drives_list[last_drive_index]) ;
						mouse_zones[max_drive_area].key :=
						alt_letter (drives_list [last_drive_index]) ;
					  END ;
			END ;
		END ;

	  END
	ELSE
	  translate_mouse := 0 ;
	
END ;


{ -----------------------------------------------------------------------
		   				Set screen coordinates
  ----------------------------------------------------------------------- }

FUNCTION  set_coordinates: Boolean ;
VAR
	height,
	h,
	width:	Integer ;

	h_slack,
	v_slack_drives,
	v_slack_buttons,
	v_slack_sort_text,
	v_slack_name:		Integer ;

BEGIN

	height := FS_last_line - FS_first_line + 1 ;
	width  := FS_last_column - FS_first_column + 1 ;

	IF  (height < min_height)  OR  (width < min_width) THEN
	  BEGIN
	  	set_coordinates := False ;
		Exit ;
	  END ;

	help_x := FS_first_column + (width - min_width) DIV 2 + 1 ;
	help_w := min_width - 2 ;
	help_y := FS_first_line + (height - min_height) DIV 2 + 1 ;
	IF  (height - min_height) MOD 2	 = 1  THEN
		Inc (help_y) ;


	sort_text	 		  := 1 ;
	h_slack				  := 6 ;	
	v_slack_drives		  := 1 ;	
	v_slack_sort_text 	  := 1 ;
	v_slack_buttons		  := 1 ;
	v_slack_name		  := 2 ;

	file_list_lines := 14 ;     { actually MINUS 1 }


	h := 25 ;
	IF  height < h  THEN
	  BEGIN
		Dec (v_slack_name) ;
		Dec (h) ;
	  END ;
	IF  height < h  THEN
	  BEGIN
		Dec (file_list_lines) ;
		Dec (h) ;
	  END ;
	IF  height < h  THEN
	  BEGIN
		Dec (file_list_lines) ;
		Dec (h) ;
	  END ;
	IF  height < h  THEN
	  BEGIN
		Dec (v_slack_drives) ;
		Dec (h) ;
	  END ;
	WHILE  height < h  DO
	  BEGIN
		Dec (file_list_lines) ;
		Dec (h) ;
	  END ;

	IF  file_list_lines < 14  THEN
		Dec (v_slack_sort_text) ; 
	IF  file_list_lines < 13  THEN
		Dec (sort_text) ; 
	IF  file_list_lines < 13  THEN
		Dec (v_slack_buttons) ; 
		
	IF  width <= 76  THEN  
		h_slack := 0 
	ELSE
		IF  width <= 78  THEN  
			h_slack := 2 
		ELSE
			IF  width < 80  THEN  
				h_slack := 4 ;

	file_list_w := 46 ;

	drives_y := FS_first_line + 1 + v_slack_drives ;
	drives_x := FS_first_column + 2 ;

	path_x := FS_first_column + 2 + 5 ;
	path_y := drives_y + 2 ;
	path_w := file_list_w - 5 ;

	file_list_x := FS_first_column + 2 ;
	file_list_y := path_y + 2 ;

	sort_options_text_y	:= file_list_y ;
	sort_options_y :=  sort_options_text_y + sort_text + v_slack_sort_text ;
	sort_options_x := file_list_x + file_list_w  + h_slack + 2 ;

	action_buttons_y :=  sort_options_y + 8 + v_slack_buttons ;
	action_buttons_x := sort_options_x ;

	mask_x := sort_options_x + 5 ;
	mask_y := path_y ;
	mask_w := 14 ;
	
	
	name_x := file_list_x + 5 ;
	
	name_y := file_list_y + file_list_lines + 1 + v_slack_name ;
	name_w := 13 ;

	max_file_list_lines := file_list_lines ;
	max_drive := (FS_last_column - FS_first_column	- 3) DIV 6 ;
END ;


{ -----------------------------------------------------------------------
  ----------------------------------------------------------------------- 
  ----------------------------------------------------------------------- 
  ----------------------------------------------------------------------- }


PROCEDURE FS_open ( directory:     String ;
					mask:          String ;
					hide:		   String ;	  
					title:         String ;
					sorting: 	   Char ;	
					VAR filename:  String) ;
VAR
	key:                Word ;
	starting_dir:       String ;
	save:               Integer ;
	status:             Process_Status ;
	move:               Integer ;           

BEGIN

	IF  NOT set_coordinates  THEN
		Exit ;

	CASE sorting OF
	  'N': requested_sort_type :=_name_desc ;
	  'e': requested_sort_type :=_ext ;
	  'E': requested_sort_type :=_ext_desc ;
	  's': requested_sort_type :=_size ;
	  'S': requested_sort_type :=_size_desc ;
	  't': requested_sort_type :=_date ;
	  'T': requested_sort_type :=_date_desc ;
	ELSE
	   requested_sort_type := _name	;
	END ; 

	IF  scroll_bar = 0  THEN
		scroll_bar := WND_new_scroll_bar  ( FS_sel_file_BG,
										    FS_button_BG,
										    FS_button_high_FG,
										    file_list_lines+1) ;
	area := 0 ;
	n_areas := 4 ;
	n_zones := 0 ;
	number_of_entries := 0 ;
	current_name := '' ;
	restore := False ;
	up_case_str (hide);
	hide_mask := hide ;
	up_case_str (mask);
	directory := FExpand (directory) ;

	IF  (directory = last_directory) AND
		(last_filename <> '')        AND
		(mask = current_mask)        THEN
	  restore := True ;

	WND_save_cursor (False);
	current_mask := mask;

	GetDir (0, starting_dir) ;
	
	{$i-}
	Chdir (directory) ;
	IF  IOResult <> 0  THEN
	  BEGIN
		Chdir (starting_dir) ;
		Exit ;
	  END ;
	{$i+}
	save := 1 ;
	WND_open_window (FS_first_line,
					 FS_first_column,
					 FS_last_line, 
					 FS_last_column,
					 FS_text_FG, FS_text_BG,
					 WND_no_shadow, WND_single_frame,
					 title,
					 save) ;
	
	{MS_Init ; }
	MS_Show ;
	show_drives ;
	show_sort_options ;
	show_action_buttons ;
	show_path_and_mask ;
	
	show_file_list ;
	update_all_areas ;
	
	
	status := PS_continue ;
	REPEAT
	  IF  status = PS_mouse_clicked  THEN
		key := translate_mouse 
	  ELSE
		BEGIN
		  key := 0 ;
		  WHILE  (area = list_area) AND (key = 0)  DO
			BEGIN
			  REPEAT
			  UNTIL  KeyPressed OR MS_LeftPressed OR MS_RightPressed ;
			  IF  KeyPressed  THEN
				key := KB_read 
			  ELSE
				BEGIN
				  WHILE  MS_LeftDown  AND
				  		 WND_check_scroll_bar (scroll_bar, move)  DO
					BEGIN
					  IF  move = -2  THEN        { meaning +1 }
						BEGIN
						  process_file_list (KB_Down, filename, status) ;
						  MS_delay ;
						END
					  ELSE
						IF  move = -1  THEN
						  BEGIN
							process_file_list (KB_Up, filename, status) ;
							MS_Delay ;
						  END
						ELSE
						  BEGIN
							sel_number := move + (sel_number - file_number) ;
							file_number := move ;
							display_directory ;
							MS_Delay ;
						  END ;
					END ;
				  IF MS_LeftPressed THEN
					 REPEAT UNTIL NOT MS_LeftDown ;
				  IF MS_RightPressed THEN
					 REPEAT UNTIL NOT MS_RightDown ;
				  key := translate_mouse ;
				END ;
			END ;
		END ;

	  CASE  area  OF
		
		list_area:   process_file_list (key, filename, status) ;
		path_area:   process_path (status) ;
		mask_area:   process_mask (status) ;
		name_area:   process_name (filename, status);
		
	  END ;
	  
	UNTIL (status = PS_selected) OR  (status = PS_cancelled) ;
	
	IF  status = PS_cancelled  THEN
	  filename := '' ;

	last_filename := filename ;
	last_directory := directory ;
	FOR  i := 1  TO  number_of_entries  DO
	  Dispose (dir_entries [i]) ;
	MS_Hide ;
	WND_close_window (save) ;
	WND_restore_cursor ;
	Chdir (starting_dir) ;

END;


{ ----------------------------------------------------------------------- 
						Unit initialization
  ----------------------------------------------------------------------- }

BEGIN

	last_filename := '' ;
	last_directory := '' ;
	no_save := 0 ;
	scroll_bar := 0 ;

	FS_text_BG      	  := LightGray ;
	FS_text_FG  	      := Black ;

	FS_file_list_BG       := Black ;
	FS_file_list_FG       := Green ;

	FS_sel_file_FG        := White ;
	FS_sel_file_BG        := Brown ;

	FS_button_BG          := Green ;
	FS_button_special_BG  := Red ;
	FS_button_FG          := LightGray ;
	FS_button_high_FG     := White ;
	FS_button_special_FG  := LightGreen ;
	
	FS_first_line		  := 1 ;	
	FS_last_line		  := 25 ;
	FS_first_column		  := 1 ;
	FS_last_column		  := 80 ;

END.

