// GALBlast
#include <windows.h>
#include <stdio.h>
#include <ver.h>
#include <stdlib.h>
#include <stdarg.h>
#include <mmsystem.h>
#include <dos.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <commdlg.h>

#define LATTICE 0xA1
#define NATIONAL 0x8F
#define SGSTHOMSON 0x20

static char *caption[]={"Read GAL","Verify GAL","Read PES","Test SCLK","Write GAL","Erase GAL","Erase ALL","Burn Security","Write PES","VPP Test"};
#define READGAL 0
#define VERIFYGAL 1
#define READPES 2
#define SCLKTEST 3
#define WRITEGAL 4
#define ERASEGAL 5
#define ERASEALL 6
#define BURNSECURITY 7
#define WRITEPES 8
#define VPPTEST 9

typedef enum { UNKNOWN,GAL16V8,GAL18V10,GAL20V8,GAL20RA10,GAL20XV10,GAL22V10,GAL26CV12,GAL6001,GAL6002} GALTYPE;

static int cfg16V8[]=
{
    2128,2129,2130,2131,2132,2133,2134,2135,2136,2137,2138,2139,2140,2141,2142,2143,2144,2145,2146,2147,2148,2149,2150,2151,2152,2153,2154,2155,2156,2157,2158,2159,
    2048,2049,2050,2051,
    2193,
    2120,2121,2122,2123,2124,2125,2126,2127,
    2192,
    2052,2053,2054,2055,
    2160,2161,2162,2163,2164,2165,2166,2167,2168,2169,2170,2171,2172,2173,2174,2175,2176,2177,2178,2179,2180,2181,2182,2183,2184,2185,2186,2187,2188,2189,2190,2191
};
static int cfg16V8AB[]=
{
    2048,2049,2050,2051,
    2193,
    2120,2121,2122,2123,
    2128,2129,2130,2131,2132,2133,2134,2135,2136,2137,2138,2139,2140,2141,2142,2143,2144,2145,2146,2147,2148,2149,2150,2151,2152,2153,2154,2155,2156,2157,2158,2159,2160,2161,2162,2163,2164,2165,2166,2167,2168,2169,2170,2171,2172,2173,2174,2175,2176,2177,2178,2179,2180,2181,2182,2183,2184,2185,2186,2187,2188,2189,2190,2191,
    2124,2125,2126,2127,
    2192,
    2052,2053,2054,2055
};
static int cfg18V10[]=
{
    3457,3456,
    3459,3458,
    3461,3460,
    3463,3462,
    3465,2464,
    3467,3466,
    3469,3468,
    3471,3470,
    3473,3472,
    3475,3474
};
static int cfg20V8[]=
{
    2640,2641,2642,2643,2644,2645,2646,2647,2648,2649,2650,2651,2652,2653,2654,2655,2656,2657,2658,2659,2660,2661,2662,2663,2664,2665,2666,2667,2668,2669,2670,2671,
    2560,2561,2562,2563,
    2705,
    2632,2633,2634,2635,2636,2637,2638,2639,
    2704,
    2564,2565,2566,2567,
    2672,2673,2674,2675,2676,2677,2678,2679,2680,2681,2682,2683,2684,2685,2686,2687,2688,2689,2690,2691,2692,2693,2694,2695,2696,2697,2698,2699,2700,2701,2702,2703
};
static int cfg20V8AB[]=
{
    2567,2566,2565,2564,
    2705,
    2632,2633,2634,2635,
    2640,2641,2642,2643,2644,2645,2646,2647,2648,2649,2650,2651,2652,2653,2654,2655,2656,2657,2658,2659,2660,2661,2662,2663,2664,2665,2666,2667,2668,2669,2670,2671,2672,2673,2674,2675,2676,2677,2678,2679,2680,2681,2682,2683,2684,2685,2686,2687,2688,2689,2690,2691,2692,2693,2694,2695,2696,2697,2698,2699,2700,2701,2702,2703,
    2636,2637,2638,2639,
    2704,
    2563,2562,2561,2560
};
static int cfg20RA10[]=
{
    3200,3201,3202,3203,3204,3205,3206,3207,3208,3209
};
static int cfg20XV10[]=
{
    1630,
    1628,1629,
    1620,1621,1622,
    1610,1611,1612,1613,1614,
    1600,1601,1602,1603,1604,
    1627,1626,
    1623,1624,1625,
    1619,1618,1617,1616,1615,
    1609,1608,1607,1606,1605
};
static int cfg22V10[]=
{
    5809,5808,
    5811,5810,
    5813,5812,
    5815,5814,
    5817,5816,
    5819,5818,
    5821,5820,
    5823,5822,
    5825,5824,
    5827,5826
};
static int cfg26CV12[]=
{
    6345,6344,
    6347,6346,
    6349,6348,
    6351,6350,
    6353,6352,
    6355,6354,
    6357,6356,
    6359,6358,
    6361,6360,
    6363,6362,
    6365,6364,
    6367,6366
};
static int cfg6001[]=
{
    8221,8220,
    8179,8183,8187,8191,8195,8199,8203,8207,8211,8215,
    8214,8210,8206,8202,8198,8194,8190,8186,8182,8178,
    8216,8217,8212,8213,8208,8209,8204,8205,8200,8201,
      8196,8197,8192,8193,8188,8189,8184,8185,8180,8181,
    8156,8159,8162,8165,8168,8171,8174,8177,
    8154,8157,8160,8163,8166,8169,8172,8175,
    8176,8173,8170,8167,8164,8161,8158,8155,
    8218,8219
};
static int cfg6002[]=
{
    8257,8256,

    8179,8183,8187,8191,8195,8199,8203,8207,8211,8215,
    8214,8210,8206,8202,8198,8194,8190,8186,8182,8178,
    8216,8217,8212,8213,8208,8209,8204,8205,8200,8201,
      8196,8197,8192,8193,8188,8189,8184,8185,8180,8181,
    8255,8254,8253,8252,8251,8250,8249,8248,8247,
      8246,8245,8244,8243,8242,8241,8240,8239,8238,
    8220,8221,8222,8223,8224,8225,8226,8227,8228,
      8229,8230,8231,8232,8233,8234,8235,8236,8237,
    8156,8159,8162,8165,8168,8171,8174,8177,
    8154,8157,8160,8163,8166,8169,8172,8175,
    8176,8173,8170,8167,8164,8161,8158,8155,
    8218,8219
};
static struct
{
    GALTYPE type;
    unsigned char id0,id1;
    char *name;
    int fuses;
    int pins;
    int rows;
    int bits;
    int uesrow;
    int uesfuse;
    int uesbytes;
    int eraserow;
    int eraseallrow;
    int pesrow;
    int pesbytes;
    int cfgrow;
    int *cfg;
    int cfgbits;
}
galinfo[]=
{
    {UNKNOWN,  0x00,0x00,"unknown",     0, 0, 0,  0, 0,   0,0, 0, 0, 0, 8, 0,NULL,0},
    {GAL16V8,  0x00,0x1A,"GAL16V8",  2194,20,32, 64,32,2056,8,63,54,58, 8,60,cfg16V8AB,sizeof(cfg16V8AB)/sizeof(int)},
    {GAL18V10, 0x50,0x51,"GAL18V10", 3540,20,36, 96,36,3476,8,61,60,58,10,16,cfg18V10,sizeof(cfg18V10)/sizeof(int)},
    {GAL20V8,  0x20,0x3A,"GAL20V8",  2706,24,40, 64,40,2568,8,63,59,58, 8,60,cfg20V8AB,sizeof(cfg20V8AB)/sizeof(int)},
    {GAL20RA10,0x60,0x61,"GAL20RA10",3274,24,40, 80,40,3210,8,61,60,58,10,16,cfg20RA10,sizeof(cfg20RA10)/sizeof(int)},
    {GAL20XV10,0x65,0x66,"GAL20XV10",1671,24,40, 40,44,1631,5,61,60,58, 5,16 /* 4,5,6,29,36,37 macht was,38,44,46,52 CLEARALL,53 macht was,60 CLEARALL */,cfg20XV10,sizeof(cfg20XV10)/sizeof(int)},
    {GAL22V10, 0x48,0x49,"GAL22V10", 5892,24,44,132,44,5828,8,61,60,58,10,16,cfg22V10,sizeof(cfg22V10)/sizeof(int)},
    {GAL26CV12,0x58,0x59,"GAL26CV12",6432,28,52,122,52,6368,8,61,60,58,12,16,cfg26CV12,sizeof(cfg26CV12)/sizeof(int)},
    {GAL6001,  0x40,0x41,"GAL6001",  8294,24,78, 75,97,8222,9,63,62,96, 8, 8,cfg6001,sizeof(cfg6001)/sizeof(int)},
    {GAL6002,  0x44,0x44,"GAL6002",  8330,24,78, 75,97,8258,9,63,62,96, 8, 8,cfg6002,sizeof(cfg6002)/sizeof(int)},
};

static HANDLE hInstance;
static HICON appicon;
static int vppmul=128;
static char progname[]="GALBlast";
static int far *lptbase=(int far *)0x00400008L;
static int duration[16]={1,2,5,10,20,30,40,50,60,70,80,90,100,200,0,0};
static int lpt=0,security=0,erase=0,pulse=0,vpp=0;
static GALTYPE gal;
static char fusemap[10000];
static char backup[10000];
static unsigned short checksum;
static int pesread=0;
static unsigned char pes[12],pesbackup[12];
static char buffer[16348];
static unsigned char port;

static void Debug(char format[],...)
{
    static char buffer[1024];
    va_list arg;

    va_start(arg,format);
    wvsprintf(buffer,format,arg);
    va_end(arg);
    OutputDebugString(buffer);
}

static int Message(HWND wnd,LPSTR format,LPSTR caption,int flags,...)
{
    static char buffer[1024];
    va_list arg;

    va_start(arg,flags);
    wvsprintf(buffer,format,arg);
    va_end(arg);
    if(!caption) caption=progname;
    MessageBeep(flags&MB_ICONMASK);
    return MessageBox(wnd,buffer,caption,flags);
}

static void WaitCursor(BOOL on)
{
    POINT p;
    HWND w;
    HCURSOR h;

    if(on)
    {
        SetCursor(LoadCursor(0,IDC_WAIT));
    }
    else
    {
        GetCursorPos(&p);
	w=WindowFromPoint(p);
        h=w?GetClassWord(w,GCW_HCURSOR):0;
        SetCursor(h?h:LoadCursor(0,IDC_ARROW));
    }
}

static unsigned short CheckSum(int n)
{
    unsigned short c,e;
    long a;
    int i;

    c=e=0;
    a=0;
    for(i=0;i<n;i++)
    {
        e++;
        if(e==9)
	{
            e=1;
	    a+=c;
            c=0;
	}
        c>>=1;
	if(fusemap[i]) c+=0x80;
    }
    return (unsigned short)((c>>(8-e))+a);
}

static int ParseFuseMap(HWND wnd,char *ptr)
{
    int i,n,type,checksumpos,address,pins,lastfuse,state; // 0=outside JEDEC, 1=skipping comment or unknown, 2=read command

    state=2;
    security=0;
    checksum=0;
    checksumpos=0;
    pins=0;
    lastfuse=0;
    for(n=0;ptr[n];n++)
    {
	if(ptr[n]=='*') state=2; // else ignored
	else switch(state)
	{
	case 2:
	    if(!isspace(ptr[n])) switch(ptr[n])
	    {
	    case 'L':
		address=0;
		state=3;
		break;
	    case 'F':
		state=5;
		break;
	    case 'G':
                state=13;
		break;
	    case 'Q':
		state=7;
		break;
	    case 'C':
		checksumpos=n;
		state=14;
                break;
	    default:
		state=1;
	    }
	    break;
	case 3:
	    if(!isdigit(ptr[n])) return n;
	    address=ptr[n]-'0';
	    state=4;
	    break;
	case 4:
	    if(isspace(ptr[n]))
	    {
		state=6;
	    }
	    else if(isdigit(ptr[n]))
	    {
		address=10*address+(ptr[n]-'0');
	    }
	    else
	    {
		return n;
	    }
	    break;
	case 5:
	    if(isspace(ptr[n])) break; // ignored
	    if(ptr[n]=='0'||ptr[n]=='1')
	    {
		memset(fusemap,ptr[n]-'0',sizeof(fusemap));
	    }
	    else
	    {
                return n;
	    }
	    state=1;
	    break;
	case 6:
	    if(isspace(ptr[n])) break; // ignored
	    if(ptr[n]=='0'||ptr[n]=='1')
	    {
		fusemap[address++]=ptr[n]-'0';
	    }
	    else
	    {
		return n;
	    }
	    break;
	case 7:
	    if(isspace(ptr[n])) break; // ignored
            if(ptr[n]=='P')
	    {
		pins=0;
                state=8;
            }
            else if(ptr[n]=='F')
	    {
		lastfuse=0;
		state=9;
	    }
	    else state=2;
            break;
	case 8:
	    if(isspace(ptr[n])) break; // ignored
	    if(!isdigit(ptr[n])) return n;
	    pins=ptr[n]-'0';
            state=10;
	    break;
	case 9:
	    if(isspace(ptr[n])) break; // ignored
            if(!isdigit(ptr[n])) return n;
	    lastfuse=ptr[n]-'0';
	    state=11;
            break;
        case 10:
            if(isdigit(ptr[n]))
	    {
                pins=10*pins+(ptr[n]-'0');
	    }
	    else if(isspace(ptr[n]))
	    {
               state=12;
	    }
	    else return n;
            break;
	case 11:
            if(isdigit(ptr[n]))
	    {
		lastfuse=10*lastfuse+(ptr[n]-'0');
	    }
            else if(isspace(ptr[n]))
	    {
	       state=12;
            }
            else return n;
            break;
	case 12:
            if(!isspace(ptr[n])) return n;
	    break;
	case 13:
	    if(isspace(ptr[n])) break; // ignored
	    if(ptr[n]=='0'||ptr[n]=='1')
	    {
		security=ptr[n]-'0';
	    }
	    else
	    {
		return n;
	    }
	    state=1;
	    break;
	case 14:
	    if(isspace(ptr[n])) break; // ignored
            if(isdigit(ptr[n]))
            {
		checksum=ptr[n]-'0';
	    }
            else if(toupper(ptr[n])>='A'&&toupper(ptr[n])<='F')
	    {
		checksum=toupper(ptr[n])-'A'+10;
	    }
            else return n;
	    state=15;
	    break;
	case 15:
	    if(isdigit(ptr[n]))
            {
		checksum=16*checksum+ptr[n]-'0';
	    }
	    else if(toupper(ptr[n])>='A'&&toupper(ptr[n])<='F')
            {
		checksum=16*checksum+toupper(ptr[n])-'A'+10;
	    }
            else if(isspace(ptr[n]))
            {
                state=2;
	    }
	    else return n;
	    break;
	}
    }
    if(lastfuse||pins)
    {
	if(checksum&&checksum!=CheckSum(lastfuse))
	{
	    if(Message(wnd,"Checksum given %04X calculated %04X",NULL,MB_OKCANCEL,checksum,CheckSum(lastfuse))==IDCANCEL)
	    {
		return checksumpos;
	    }
	}
	for(type=0,i=1;i<sizeof(galinfo)/sizeof(galinfo[0]);i++)
	{
	    if((lastfuse==0||galinfo[i].fuses==lastfuse||galinfo[i].uesfuse==lastfuse&&galinfo[i].uesfuse+8*galinfo[i].uesbytes==galinfo[i].fuses)
	    && (pins==0||galinfo[i].pins==pins))
	    {
		if(gal==0)
		{
		    type=i;
		    break;
		}
		else if(!type)
                {
		    type=i;
		}
	    }
	}
	if(gal!=type)
	{
	    gal=type;
	    SendDlgItemMessage(wnd,101,CB_SETCURSEL,gal-1,0L);
	    SendMessage(wnd,WM_COMMAND,101,MAKELONG(0,CBN_SELCHANGE));
	}
    }
    return n;
}

static BOOL CheckJEDEC(HWND wnd)
{
    int position;

    position=ParseFuseMap(wnd,buffer);
    if(position==strlen(buffer)) return TRUE;
    Message(wnd,"Error in JEDEC",NULL,MB_OK|MB_ICONEXCLAMATION);
    SendDlgItemMessage(wnd,102,EM_SETSEL,0,MAKELONG(position,position));
    SendMessage(wnd,WM_NEXTDLGCTL,GetDlgItem(wnd,102),TRUE);
    return FALSE;
}

static void Delay(int msec)
{
    long start=timeGetTime();
    while(timeGetTime()<start+msec) ;
}

/* set/reset individual pins of LPT port */
static void SetSTROBE(BOOL on)
{
    outportb(lptbase[lpt]+2,on?inportb(lptbase[lpt]+2)&~0x01:inportb(lptbase[lpt]+2)|0x01);
}

static void SetFEED(BOOL on)
{
    outportb(lptbase[lpt]+2,on?inportb(lptbase[lpt]+2)&~0x02:inportb(lptbase[lpt]+2)|0x02);
}

static void SetINIT(BOOL on)
{
    outportb(lptbase[lpt]+2,on?inportb(lptbase[lpt]+2)|0x04:inportb(lptbase[lpt]+2)&~0x04);
}

static void SetSEL(BOOL on)
{
    outportb(lptbase[lpt]+2,on?inportb(lptbase[lpt]+2)&~0x08:inportb(lptbase[lpt]+2)|0x08);
}

static void SetRow(int row)
{
    port=(port&0x81)|(row<<1);
    outportb(lptbase[lpt],port);
}

static BOOL GetACK(void)
{
    return (inportb(lptbase[lpt]+1)&0x40)!=0;
}

/* set/reset individual pins of GAL */
static void SetVCC(BOOL on)
{
    SetINIT(!on);
}

static void SetVPP(int volt)
{
    // 48 -> 0x80 -> 12V, volt in
    outportb(lptbase[lpt],(unsigned char)MulDiv(volt,vppmul,48));
    SetFEED(0);
    SetFEED(1);
    outportb(lptbase[lpt],port);
}

static void SetSDIN(BOOL on)
{
    if(on)
    {
        port|=0x01;
    }
    else
    {
        port&=0xFE;
    }
    outportb(lptbase[lpt],port);
}

static void SetSTB(BOOL on)
{
    SetSTROBE(on);
}

static void SetPV(BOOL on)
{
    SetSEL(on);
}

static void SetSCLK(BOOL on)
{
    if(on)
    {
        port|=0x80;
    }
    else
    {
        port&=0x7F;
    }
    outportb(lptbase[lpt],port);
}

static BOOL GetSDOUT(void)
{
    return GetACK();
}

static void TurnOff(void)
{
    Delay(100);
    SetRow(0x3F);
    SetSDIN(1);
    // SetSCLK(1);
    // SetPV(1);
    SetVPP(0); // turn off VPP
    Delay(2);
    SetVCC(0); // turn off VCC and drivers
    WaitCursor(FALSE);
}

BOOL WINAPI GrayDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    RECT r;
    PAINTSTRUCT ps;

    switch(msg)
    {
    case WM_INITDIALOG:
	return TRUE;
    case WM_COMMAND:
	switch(LOWORD(wParam))
	{
	case IDCANCEL:
	    EndDialog(wnd,FALSE);
	    return TRUE;
	}
	break;
    case WM_CTLCOLOR:
	if(HIWORD(lParam)==CTLCOLOR_STATIC)
	{
	    SetBkMode((HDC)wParam,TRANSPARENT);
	    return GetStockObject(LTGRAY_BRUSH);
	}
	return FALSE;
    case WM_ERASEBKGND:
	if(IsIconic(wnd)) return TRUE;
	GetClientRect(wnd,&r);
	FillRect((HDC)wParam,&r,GetStockObject(LTGRAY_BRUSH));
	SetWindowLong(wnd,DWL_MSGRESULT,1);
	return TRUE;
    case WM_PAINT:
	if(IsIconic(wnd))
	{
	    BeginPaint(wnd,&ps);
	    DefWindowProc(wnd,WM_ICONERASEBKGND,(WORD)ps.hdc,0L);
	    DrawIcon(ps.hdc,0,0,appicon);
	    EndPaint(wnd,&ps);
	    return TRUE;
	}
	return FALSE;
    case WM_QUERYDRAGICON:
	return appicon;
    }
    return FALSE;
}

BOOL WINAPI _export AssureDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    int i,j,n;
    char buffer[128];
    static int useerase;

    switch(msg)
    {
    case WM_INITDIALOG:
        SetWindowText(wnd,caption[lParam]);
        SetDlgItemText(wnd,IDOK,caption[lParam]);
        switch(pes[3])
        {
        case LATTICE:
            strcpy(buffer,"Lattice ");
            break;
        case NATIONAL:
            strcpy(buffer,"National ");
            break;
        case SGSTHOMSON:
            strcpy(buffer,"ST ");
	    break;
        default:
            strcpy(buffer,"Unknown ");
	}
        strcat(buffer,galinfo[gal].name);
        SetDlgItemText(wnd,100,buffer);
	j=wsprintf(buffer,"PES:");
        for(i=0;i<galinfo[gal].pesbytes;i++)
        {
	    j+=wsprintf(buffer+j," %02X",pes[i]);
        }
        SetDlgItemText(wnd,103,buffer);
	wsprintf(buffer,"VPP=%d.%02dV Prog-Pulse=%dms Erase-Pulse=%dms",vpp/4,(vpp%4)*25,pulse,erase);
        SetDlgItemText(wnd,104,buffer);
        n=-1;
        for(i=48;i<83;i++)
        {
            wsprintf(buffer,"%d.%02d",i/4,(i%4)*25);
	    j=(int)SendDlgItemMessage(wnd,101,CB_ADDSTRING,0,(LPARAM)(LPSTR)buffer);
	    SendDlgItemMessage(wnd,101,CB_SETITEMDATA,j,i);
	    if(i==vpp) n=j;
        }
	SendDlgItemMessage(wnd,101,CB_SETCURSEL,n,0l);
	n=-1;
        if(lParam==ERASEGAL||lParam==ERASEALL)
        {
            useerase=TRUE;
            SetDlgItemText(wnd,99,"Erase-Pulse");
            for(i=25;i<=800;i<<=1)
            {
                wsprintf(buffer,"%d",i/2);
                j=(int)SendDlgItemMessage(wnd,102,CB_ADDSTRING,0,(LPARAM)(LPSTR)buffer);
                SendDlgItemMessage(wnd,102,CB_SETITEMDATA,j,i/2);
                if(i/2==erase) n=j;
            }
        }
        else
        {
            useerase=FALSE;
            SetDlgItemText(wnd,99,"Prog-Pulse");
            for(i=0;i<14;i++)
            {
                wsprintf(buffer,"%d",duration[i]);
		j=(int)SendDlgItemMessage(wnd,102,CB_ADDSTRING,0,(LPARAM)(LPSTR)buffer);
                SendDlgItemMessage(wnd,102,CB_SETITEMDATA,j,duration[i]);
                if(duration[i]==pulse) n=j;
            }
        }
        SendDlgItemMessage(wnd,102,CB_SETCURSEL,n,0l);
        return TRUE;
    case WM_COMMAND:
        switch(LOWORD(wParam))
        {
        case IDOK:
	    i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
	    if(i==-1)
            {
                Message(wnd,"Select VPP first",progname,MB_ICONEXCLAMATION|MB_OK);
                SetFocus(GetDlgItem(wnd,101));
                return TRUE;
            }
            vpp=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
            i=(int)SendDlgItemMessage(wnd,102,CB_GETCURSEL,0,0L);
	    if(i==-1)
            {
                GetDlgItemText(wnd,99,buffer,sizeof(buffer));
                Message(wnd,"Select %s first",progname,MB_ICONEXCLAMATION|MB_OK,(LPSTR)buffer);
                SetFocus(GetDlgItem(wnd,102));
                return TRUE;
            }
            if(useerase)
            {
                erase=(int)SendDlgItemMessage(wnd,102,CB_GETITEMDATA,i,0L);
            }
            else
            {
		pulse=(int)SendDlgItemMessage(wnd,102,CB_GETITEMDATA,i,0L);
            }
            EndDialog(wnd,1);
            return TRUE;
        case IDCANCEL:
            EndDialog(wnd,0);
	    return TRUE;
        }
    }
    return GrayDlgProc(wnd,msg,wParam,lParam);
}

static BOOL TurnOn(HWND wnd,int mode)
{
    BOOL writeorerase;

    if(mode==WRITEGAL||mode==ERASEGAL||mode==ERASEALL||mode==BURNSECURITY||mode==WRITEPES||mode==VPPTEST)
    {
        if(DialogBoxParam(hInstance,MAKEINTRESOURCE(4),wnd,AssureDlgProc,mode)!=1) return FALSE;
        writeorerase=TRUE;
    }
    else
    {
        writeorerase=FALSE;
    }
    WaitCursor(TRUE);
    SetVPP(0); // assure VPP is 0
    SetSTB(1); // set all (disabled) outputs to H (high impedance)
    SetSCLK(1);
    SetSDIN(1);
    SetPV(1);
    SetRow(0x3F);
    SetVCC(1); // turn on VCC and drivers
    Delay(100);
    SetVPP(writeorerase?vpp:48); // turn on VPP (48=12V)
    SetSCLK(0); // now that all pins are in programming mode
    SetPV(0); // bring critical pins down to L
    Delay(10);
    if(writeorerase)
    {
	SetPV(1);
	Delay(10);
    }
    return TRUE;
}

static BOOL ReceiveBit(void)
{
    BOOL bit;

    bit=GetSDOUT();
    SetSCLK(1);
    SetSCLK(0);
    return bit;
}

static void DiscardBits(int n)
{
    while(n-->0) ReceiveBit();
}

static void SendBit(int bit)
{
    SetSDIN(bit);
    SetSCLK(1);
    SetSCLK(0);
}

static void SendBits(int n,int bit)
{
    while(n-->0) SendBit(bit);
}

static void SendAddress(int n,int row)
{
    while(n-->0)
    {
	SendBit(row&1);
	row>>=1;
    }
}

static void Strobe(int msec)
{
    Delay(2);
    SetSTB(0);
    Delay(msec);
    SetSTB(1);
    Delay(2);
}

static void StrobeRow(int row)
{
    switch(gal)
    {
    case GAL16V8:
    case GAL20V8:
	SetRow(row);
	break;
    case GAL6001:
    case GAL6002:
	SetRow(0);
	SendBits(95,0);
	SendBit(1);
	SendAddress(7,row);
	SendBits(16,0);
	break;
    default:
	SetRow(0);
	SendBits(galinfo[gal].bits,0);
	SendAddress(6,row);
    }
    Strobe(2);
}

static BOOL GetSetup(HWND wnd)
{
    int i;

    i=(int)SendDlgItemMessage(wnd,100,CB_GETCURSEL,0,0L);
    if(i==-1)
    {
	Message(wnd,"Select LPT port first",NULL,MB_OK|MB_ICONEXCLAMATION);
	return FALSE;
    }
    lpt=(int)SendDlgItemMessage(wnd,100,CB_GETITEMDATA,i,0L);
    i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
    if(i==-1)
    {
	Message(wnd,"Select a GAL type first",NULL,MB_OK|MB_ICONEXCLAMATION);
	return FALSE;
    }
    gal=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
    return TRUE;
}

static void GetPES(char pes[])
{
    int bitmask,byte;

    StrobeRow(galinfo[gal].pesrow);
    if(gal==GAL6001||gal==GAL6002) DiscardBits(20);
    for(byte=0;byte<galinfo[gal].pesbytes;byte++)
    {
	pes[byte]=0;
	for(bitmask=0x1;bitmask<=0x80;bitmask<<=1)
	{
	    if(ReceiveBit()) pes[byte]|=bitmask;
	}
    }
}

static void ReadPES(HWND wnd,char pes[])
{
    TurnOn(wnd,READPES);
    GetPES(pes);
    TurnOff();
}

static void WritePES(HWND wnd,char pes[])
{
    int byte,bitmask;

    if(TurnOn(wnd,WRITEPES))
    {
        switch(gal)
        {
        case GAL16V8:
        case GAL20V8:
            SetRow(galinfo[gal].pesrow);
            for(byte=0;byte<8;byte++) for(bitmask=0x01;bitmask<=0x80;bitmask<<=1) SendBit((pes[byte]&bitmask)!=0);
            break;
        case GAL6001:
        case GAL6002:
            SetRow(0);
            SendBits(20,0);
            for(byte=0;byte<galinfo[gal].pesbytes;byte++) for(bitmask=0x01;bitmask<=0x80;bitmask<<=1) SendBit((pes[byte]&bitmask)!=0);
            if(galinfo[gal].pesbytes*8<galinfo[gal].bits) SendBits(galinfo[gal].bits-galinfo[gal].pesbytes*8,0);
            SendBit(1);
            SendAddress(7,galinfo[gal].pesrow);
            SendBits(16,0);
            SetSDIN(0);
            break;
        default:
            SetRow(0);
            for(byte=0;byte<galinfo[gal].pesbytes;byte++) for(bitmask=0x01;bitmask<=0x80;bitmask<<=1) SendBit((pes[byte]&bitmask)!=0);
            if(galinfo[gal].pesbytes*8<galinfo[gal].bits) SendBits(galinfo[gal].bits-galinfo[gal].pesbytes*8,0);
            SendAddress(6,galinfo[gal].pesrow);
            SetSDIN(0);
        }
        Strobe(pulse);
	TurnOff();
    }
}

static void WriteGAL(HWND wnd,char *fuses)
{
    int row,bit;

    if(TurnOn(wnd,WRITEGAL))
    {
        switch(gal)
        {
        case GAL16V8:
        case GAL20V8:
            for(row=0;row<galinfo[gal].rows;row++)
            {
                SetRow(row);
                for(bit=0;bit<galinfo[gal].bits;bit++)
                {
                    SendBit(fuses[galinfo[gal].rows*bit+row]);
                }
                Strobe(pulse);
            }
            // UES
            SetRow(galinfo[gal].uesrow);
            for(bit=0;bit<64;bit++)
            {
                SendBit(fuses[galinfo[gal].uesfuse+bit]);
            }
            Strobe(pulse);
            // CFG
            SetRow(galinfo[gal].cfgrow);
            for(bit=0;bit<82;bit++)
            {
                switch(pes[2])
                {
                case 0x00:
                    SendBit(fuses[cfg16V8[bit]]);
                    break;
                case 0x1A:
                    SendBit(fuses[cfg16V8AB[bit]]);
                    break;
                case 0x20:
                    SendBit(fuses[cfg20V8[bit]]);
                    break;
                case 0x3A:
                    SendBit(fuses[cfg20V8AB[bit]]);
                    break;
                }
            }
            Strobe(pulse);
            break;
        case GAL6001:
        case GAL6002:
            SetRow(0);
            for(row=0;row<78;row++)
            {
                SendBits(20,0);
                for(bit=0;bit<11;bit++) SendBit(fuses[7296+78*bit+row]);
                for(bit=0;bit<64;bit++) SendBit(fuses[114*bit+row]);
                SendBit(1);
                SendAddress(7,row);
                SendBits(16,0);
                SetSDIN(0);
                Strobe(pulse);
            }
            for(row=0;row<64;row++)
            {
                for(bit=0;bit<20;bit++) SendBit(fuses[78+114*row+bit]);
                SendBits(11,0);
                for(bit=0;bit<64;bit++) SendBit(bit!=row);
                SendBits(8,0);
                for(bit=0;bit<16;bit++) SendBit(fuses[98+114*row+bit]);
                SetSDIN(0);
                Strobe(pulse);
            }
            // UES
            SendBits(20,0);
            for(bit=0;bit<72;bit++) SendBit(fuses[galinfo[gal].uesfuse+bit]);
            SendBits(3,0);
            SendBit(1);
            SendAddress(7,galinfo[gal].uesrow);
            SendBits(16,0);
            SetSDIN(0);
            Strobe(pulse);
            // CFG
            SetRow(galinfo[gal].cfgrow);
            for(bit=0;bit<galinfo[gal].cfgbits;bit++)
            {
                SendBit(fuses[galinfo[gal].cfg[bit]]);
            }
            SetSDIN(0);
            Strobe(pulse);
            break;
        default:
            SetRow(0);
            for(row=0;row<galinfo[gal].rows;row++)
            {
                for(bit=0;bit<galinfo[gal].bits;bit++)
                {
                    SendBit(fuses[galinfo[gal].rows*bit+row]);
                }
                SendAddress(6,row);
                SetSDIN(0);
                Strobe(pulse);
            }
            // UES
            for(bit=0;bit<galinfo[gal].uesbytes*8;bit++)
            {
                SendBit(fuses[galinfo[gal].uesfuse+bit]);
            }
            if(galinfo[gal].uesbytes*8<galinfo[gal].bits)
            {
                SendBits(galinfo[gal].bits-galinfo[gal].uesbytes*8,0);
            }
            SendAddress(6,galinfo[gal].uesrow);
            SetSDIN(0);
            Strobe(pulse);
            // CFG
            SetRow(galinfo[gal].cfgrow);
            for(bit=0;bit<galinfo[gal].cfgbits;bit++)
            {
                SendBit(fuses[galinfo[gal].cfg[bit]]);
            }
            SetSDIN(0);
            Strobe(pulse);
        }
	TurnOff();
    }
}

static void ReadGAL(HWND wnd,char *fuses)
{
    int row,bit;

    TurnOn(wnd,READGAL);
    switch(gal)
    {
    case GAL16V8:
    case GAL20V8:
	for(row=0;row<galinfo[gal].rows;row++)
	{
	    StrobeRow(row);
	    for(bit=0;bit<galinfo[gal].bits;bit++)
	    {
		fuses[galinfo[gal].rows*bit+row]=ReceiveBit();
	    }
	}
	// UES
	StrobeRow(galinfo[gal].uesrow);
	for(bit=0;bit<64;bit++)
	{
	    fuses[galinfo[gal].uesfuse+bit]=ReceiveBit();
	}
	// CFG
	StrobeRow(galinfo[gal].cfgrow);
	for(bit=0;bit<82;bit++)
	{
	    switch(pes[2])
	    {
	    case 0x00:
		fuses[cfg16V8[bit]]=ReceiveBit();
		break;
	    case 0x1A:
		fuses[cfg16V8AB[bit]]=ReceiveBit();
		break;
	    case 0x20:
		fuses[cfg20V8[bit]]=ReceiveBit();
		break;
	    case 0x3A:
		fuses[cfg20V8AB[bit]]=ReceiveBit();
		break;
	    }
	}
	break;
    case GAL6001:
    case GAL6002:
	for(row=0;row<78;row++)
	{
	    StrobeRow(row);
	    DiscardBits(20);
	    for(bit=0;bit<11;bit++) fuses[7296+78*bit+row]=ReceiveBit();
	    for(bit=0;bit<64;bit++) fuses[114*bit+row]=ReceiveBit();
	    DiscardBits(24);
	}
	for(row=0;row<64;row++)
	{
	    SendBits(31,0);
	    for(bit=0;bit<64;bit++) SendBit(bit!=row);
	    SendBits(24,0);
            SetSDIN(0);
	    Strobe(2);
	    for(bit=0;bit<20;bit++) fuses[78+114*row+bit]=ReceiveBit();
	    DiscardBits(83);
	    for(bit=0;bit<16;bit++) fuses[98+114*row+bit]=ReceiveBit();
	}
	// UES
	StrobeRow(galinfo[gal].uesrow);
	DiscardBits(20);
	for(bit=0;bit<72;bit++) fuses[galinfo[gal].uesfuse+bit]=ReceiveBit();
	// CFG
	SetRow(galinfo[gal].cfgrow);
	Strobe(2);
	for(bit=0;bit<galinfo[gal].cfgbits;bit++) fuses[galinfo[gal].cfg[bit]]=ReceiveBit();
	break;
    default:
	for(row=0;row<galinfo[gal].rows;row++)
	{
	    StrobeRow(row);
	    for(bit=0;bit<galinfo[gal].bits;bit++)
	    {
		fuses[galinfo[gal].rows*bit+row]=ReceiveBit();
	    }
	}
	// UES
	StrobeRow(galinfo[gal].uesrow);
	for(bit=0;bit<galinfo[gal].uesbytes*8;bit++)
	{
	    fuses[galinfo[gal].uesfuse+bit]=ReceiveBit();
	}
	// CFG
	SetRow(galinfo[gal].cfgrow);
	Strobe(2);
	for(bit=0;bit<galinfo[gal].cfgbits;bit++)
	{
	    fuses[galinfo[gal].cfg[bit]]=ReceiveBit();
	}
    }
    TurnOff();
}

static void EraseGAL(HWND wnd,int gal)
{
    if(TurnOn(wnd,ERASEGAL))
    {
        SetRow(galinfo[gal].eraserow);
	if(gal==GAL16V8||gal==GAL20V8) SendBit(1);
        Strobe(erase);
	TurnOff();
    }
}

static void EraseWholeGAL(HWND wnd,int gal)
{
    int i;

    if(TurnOn(wnd,ERASEALL))
    {
        SetRow(galinfo[gal].eraseallrow);
	if(gal==GAL16V8||gal==GAL20V8) SendBit(1);
        Strobe(erase);
	TurnOff();
    }
}

static void BurnSecurity(HWND wnd,int gal)
{
    if(TurnOn(wnd,BURNSECURITY))
    {
        switch(gal)
        {
        case GAL16V8:
        case GAL20V8:
            SetRow(61);
            SendBit(1);
            break;
        case GAL6001:
        case GAL6002:
            SetRow(0);
            SendBits(20+11+64,0);
            SendBit(1);
            SendAddress(7,98);
            SendBits(16,0);
            SetSDIN(0);
            break;
        default:
            SetRow(0);
            switch(gal)
            {
            case GAL18V10:
                SendBits(96,0);
                break;
            case GAL20RA10:
                SendBits(80,0);
                break;
            case GAL20XV10:
                SendBits(40,0);
                break;
            case GAL22V10:
                SendBits(132,0);
                break;
            case GAL26CV12:
                SendBits(122,0);
                break;
            }
            SendAddress(6,61);
            SetSDIN(0);
        }
        Strobe(pulse);
	TurnOff();
    }
}

static void FormatJEDEC(int gal,char *fuses,char *buffer)
{
    int i,j,k,n;
    int unused,start;
    time_t now;
    BOOL unquote;
    unsigned char ch;

    time(&now);
    n=wsprintf(buffer,"JEDEC file for %s created on %s",(LPSTR)galinfo[gal].name,(LPSTR)asctime(localtime(&now)))-1;
    n+=wsprintf(buffer+n,"\r\n*QP%d*QF%d*QV0*F0*G0*X0*\r\n",galinfo[gal].pins,galinfo[gal].fuses);
    if(gal==GAL6001||gal==GAL6002)
    {
	for(i=k=0;i<64;i++)
	{
	    start=n;
	    unused=TRUE;
	    n+=wsprintf(buffer+n,"L%04d ",k);
	    for(j=0;j<114;j++)
	    {
		if(fuses[k]) unused=FALSE;
		buffer[n++]='0'+fuses[k++];
	    }
	    n+=wsprintf(buffer+n,"*\r\n");
	    if(unused) n=start;
	}
	for(i=0;i<11;i++)
	{
	    start=n;
	    unused=TRUE;
	    n+=wsprintf(buffer+n,"L%04d ",k);
	    for(j=0;j<78;j++)
	    {
		if(fuses[k]) unused=FALSE;
		buffer[n++]='0'+fuses[k++];
	    }
	    n+=wsprintf(buffer+n,"*\r\n");
	    if(unused) n=start;
	}
    }
    else for(i=k=0;i<galinfo[gal].bits;i++)
    {
	start=n;
	unused=TRUE;
	n+=wsprintf(buffer+n,"L%04d ",k);
	for(j=0;j<galinfo[gal].rows;j++)
	{
	    if(fuses[k]) unused=FALSE;
	    buffer[n++]='0'+fuses[k++];
	}
	n+=wsprintf(buffer+n,"*\r\n");
	if(unused) n=start;
    }
    if(k<galinfo[gal].uesfuse)
    {
	start=n;
	unused=TRUE;
	n+=wsprintf(buffer+n,"L%04d ",k);
	while(k<galinfo[gal].uesfuse)
	{
	    if(fuses[k]) unused=FALSE;
	    buffer[n++]='0'+fuses[k++];
	}
	n+=wsprintf(buffer+n,"*\r\n");
	if(unused) n=start;
    }
    start=n;
    unused=TRUE;
    n+=wsprintf(buffer+n,"N UES");
    unquote=TRUE;
    for(j=0;j<galinfo[gal].uesbytes;j++)
    {
	ch=0;
	for(i=0;i<8;i++) if(fuses[k+8*j+i]) ch|=1<<i;
	if(isprint(ch))
	{
	    if(unquote)
	    {
		buffer[n++]=' ';
		buffer[n++]='"';
		unquote=FALSE;
	    }
	    buffer[n++]=ch;
	}
	else
	{
	    if(!unquote)
	    {
		buffer[n++]='"';
		unquote=TRUE;
	    }
	    n+=wsprintf(buffer+n," %02X",ch);
	}
    }
    n+=wsprintf(buffer+n,"\"*\r\nL%04d "+unquote,k);
    for(j=0;j<8*galinfo[gal].uesbytes;j++)
    {
	if(fuses[k]) unused=FALSE;
	buffer[n++]='0'+fuses[k++];
    }
    n+=wsprintf(buffer+n,"*\r\n");
    if(unused) n=start;
    if(k<galinfo[gal].fuses)
    {
	start=n;
	unused=TRUE;
	n+=wsprintf(buffer+n,"L%04d ",k);
	while(k<galinfo[gal].fuses)
	{
	    if(fuses[k]) unused=FALSE;
	    buffer[n++]='0'+fuses[k++];
	}
	n+=wsprintf(buffer+n,"*\r\n");
	if(unused) n=start;
    }
    n+=wsprintf(buffer+n,"N PES");
    for(i=0;i<galinfo[gal].pesbytes;i++)
    {
	n+=wsprintf(buffer+n," %02X",pes[i]);
    }
    n+=wsprintf(buffer+n,"*\r\nC%04X\r\n*",CheckSum(galinfo[gal].fuses));
    buffer[n]='\0';
}

static BOOL programmable(char *buffer,char *fuses,int bytes)
{
    while(bytes-->0)
    {
	if(~*buffer&*fuses) return FALSE;
	fuses++;
	buffer++;
    }
    return TRUE;
}

static BOOL erased(char *buffer,int bytes)
{
    while(bytes-->0) if(*buffer++!=1) return FALSE;
    return TRUE;
}

static char *compare(char *buffer,char *ptr,int bytes)
{
    while(bytes-->0)
    {
        if(*buffer!=*ptr) return buffer;
        buffer++;
	ptr++;
    }
    return NULL;
}

static int hex(char *buffer)
{
    char *end;
    unsigned long value;

    value=strtoul(buffer,&end,16);
    if(value>255||*end!='\0') return -1;
    return (int)value;
}

BOOL WINAPI AboutDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    DWORD size,handle;
    UINT len;
    char buffer[_MAX_PATH];
    void FAR *ptr;
    char *mem;

    switch(msg)
    {
    case WM_INITDIALOG:
	GetModuleFileName(hInstance,buffer,sizeof(buffer));
	size=GetFileVersionInfoSize(buffer,&handle);
	if(size)
	{
	    mem=malloc(size);
	    if(mem)
	    {
		if(GetFileVersionInfo(buffer,0L,size,mem))
		{
		    if(VerQueryValue(mem,"\\StringFileInfo\\040904E4\\FileDescription",&ptr,&len)&&len&&ptr)
		    {
			SetDlgItemText(wnd,100,ptr);
		    }
		    if(VerQueryValue(mem,"\\StringFileInfo\\040904E4\\FileVersion",&ptr,&len)&&len&&ptr)
		    {
			SetDlgItemText(wnd,101,ptr);
		    }
		    if(VerQueryValue(mem,"\\StringFileInfo\\040904E4\\LegalCopyright",&ptr,&len)&&len&&ptr)
		    {
			SetDlgItemText(wnd,102,ptr);
		    }
		    if(VerQueryValue(mem,"\\StringFileInfo\\040904E4\\CompanyName",&ptr,&len)&&len&&ptr)
		    {
			SetDlgItemText(wnd,103,ptr);
		    }
		}
		free(mem);
	    }
	}
	break;
    }
    return GrayDlgProc(wnd,msg,wParam,lParam);
}

static void ParsePES(void)
{
    if((pes[1]&0x7F)==5)
    {
	erase=(25<<((pes[4]>>2)&7))/2;
	pulse=duration[((((unsigned)pes[5]<<8)|pes[4])>>5)&15];
	vpp=2*((pes[5]>>1)&31)+20;
    }
    else switch(gal)
    {
    case GAL16V8:
    case GAL20V8:
	erase=100;
	goto more;
    case GAL6001:
    case GAL6002:
	erase=50;
    more:
	switch(pes[1]&0x7F)
	{
	case 0:
	    vpp=63; // 15.75V
	    pulse=100;
	    break;
	case 1:
	    vpp=63; // 15.75V
	    pulse=80;
	    break;
	case 2:
	    vpp=66; // 16.5V
	    pulse=10;
	    break;
	case 3:
	    vpp=pes[3]==NATIONAL?60:58; // 15/14.5V
	    pulse=40;
	    break;
	case 4:
	    vpp=56; // 14V
	    pulse=100;
	    break;
	}
	break;
    default:
	erase=pes[3]==NATIONAL?50:100;
	switch(pes[1]&0x7F)
	{
	case 0:
	    vpp=66; // 16.5V
	    pulse=10;
	    break;
	case 1:
	    vpp=63; // 15.75V
	    pulse=100;
	    break;
	case 2:
	    vpp=pes[3]==NATIONAL?60:58; // 15/14.5V
	    pulse=40;
	    break;
	case 3:
	    vpp=56; // 14V
	    pulse=100;
	    break;
	}
	break;
    }
}

BOOL WINAPI ProgDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    int i;
    char buffer[128];

    switch(msg)
    {
    case WM_INITDIALOG:
	for(i=0;i<galinfo[gal].pesbytes;i++)
	{
	    wsprintf(buffer,"%02X",(unsigned char)pes[i]);
	    SetDlgItemText(wnd,110+i,buffer);
	}
	while(i<12)
	{
	    ShowWindow(GetDlgItem(wnd,200+i),SW_HIDE);
	    ShowWindow(GetDlgItem(wnd,110+i++),SW_HIDE);
	}
	SendMessage(wnd,WM_COMMAND,101,0L);
	return TRUE;
    case WM_COMMAND:
	switch(LOWORD(wParam))
	{
	case IDOK:
	    EndDialog(wnd,TRUE);
	    return TRUE;
	case IDCANCEL:
	    EndDialog(wnd,FALSE);
	    return TRUE;
	default:
	    for(i=0;i<galinfo[gal].pesbytes;i++)
	    {
		if(GetDlgItemText(wnd,110+i,buffer,sizeof(buffer))%3==0||hex(buffer)==-1)
		{
		    EnableWindow(GetDlgItem(wnd,IDOK),FALSE);
		    return TRUE;
		}
	    }
	    EnableWindow(GetDlgItem(wnd,IDOK),TRUE);
	    for(i=0;i<galinfo[gal].pesbytes;i++)
	    {
		GetDlgItemText(wnd,110+i,buffer,sizeof(buffer));
		pes[i]=hex(buffer);
	    }
	    switch(pes[3])
	    {
	    case LATTICE:
		strcpy(buffer,"Lattice ");
		break;
	    case NATIONAL:
		strcpy(buffer,"National ");
		break;
	    case SGSTHOMSON:
		strcpy(buffer,"ST Microsystems ");
		break;
	    default:
		strcpy(buffer,"Unknown ");
	    }
	    strcat(buffer,galinfo[gal].name);
	    ParsePES();
	    i=strlen(buffer);
	    wsprintf(buffer+i," VPP=%d.%02dV Pulse=%dms Erase=%dms",vpp/4,(vpp%4)*25,pulse,erase);
	    SetDlgItemText(wnd,100,buffer);
	    return TRUE;
	}
    }
    return GrayDlgProc(wnd,msg,wParam,lParam);
}

static BOOL TestProperGAL(HWND wnd)
{
    int type;

    if(!GetSetup(wnd)) return FALSE;
    ReadPES(wnd,pes);
    for(type=sizeof(galinfo)/sizeof(galinfo[0]);type;type--)
    {
	if(pes[2]==galinfo[type].id0||pes[2]==galinfo[type].id1) break;
    }
    if(type==0)
    {
	pesread=0;
	if(Message(wnd,"Unknown or illegal PES read, continue ?",NULL,MB_YESNO|MB_ICONQUESTION)==IDNO) return FALSE;
    }
    else
    {
	pesread=type;
	if(type!=gal)
	{
	    switch(Message(wnd,"PES indicates a different GAL type than selected. Change GAL type ?",NULL,MB_YESNOCANCEL|MB_ICONQUESTION)==IDCANCEL)
	    {
	    case IDCANCEL:
		return FALSE;
	    case IDYES:
		gal=type;
	    }
	}
    }
    ParsePES();
    return TRUE;
}

BOOL WINAPI _export VoltDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    char temp[6];
    double v;
    char *ptr;

    switch(msg)
    {
    case WM_INITDIALOG:
        vppmul=128;
        SendDlgItemMessage(wnd,101,EM_LIMITTEXT,0,sizeof(temp));
        TurnOn(wnd,FALSE);
        return TRUE;
    case WM_COMMAND:
        switch(wParam)
        {
        case IDOK:
            if(GetDlgItemText(wnd,101,temp,sizeof(temp))>0)
            {
                ptr=strchr(temp,',');
                if(ptr) *ptr='.';
                v=strtod(temp,&ptr);
                if(*ptr!='\0'||v<1.0||v>30.0)
                {
                    Message(wnd,"Invalid value",progname,MB_ICONEXCLAMATION|MB_OK);
                    SendDlgItemMessage(wnd,101,EM_SETSEL,0,MAKELONG(0,32767));
                    SetFocus(GetDlgItem(wnd,101));
                    return TRUE;
                }
                vppmul=(int)((12.0*128)/v+0.5);
                wsprintf(temp,"%d",vppmul);
                WriteProfileString(progname,"MulDiv",temp);
            }
            /* fall thru */
        case IDCANCEL:
            TurnOff();
            EndDialog(wnd,0);
            return TRUE;
        }
        break;
    }
    return GrayDlgProc(wnd,msg,wParam,lParam);
}

BOOL WINAPI GALDlgProc(HWND wnd,unsigned msg,WPARAM wParam,LPARAM lParam)
{
    int i,j,l;
    OPENFILENAME ofn;
    WINDOWPLACEMENT wp;
    char *ptr;
    RECT r;
    static RECT editrect;
    int handle;
    char temp[40];

    switch(msg)
    {
    case WM_INITDIALOG:
	GetClientRect(wnd,&r);
	GetWindowRect(GetDlgItem(wnd,102),&editrect);
	ScreenToClient(wnd,(LPPOINT)&editrect.left);
	ScreenToClient(wnd,(LPPOINT)&editrect.right);
	SetRect(&editrect,editrect.left-r.left,editrect.top-r.top,r.right-editrect.right,r.bottom-editrect.bottom);
	if(GetProfileString(progname,"Window","",temp,sizeof(temp))>0)
	{
	    sscanf(temp,"%u%u%u%u",&r.left,&r.top,&r.right,&r.bottom);
	    MoveWindow(wnd,r.left,r.top,r.right-r.left,r.bottom-r.top,FALSE);
	}
	lpt=GetProfileInt(progname,"Port",1);
	for(i=0;i<3;i++)
	{
	    if(lptbase[i])
	    {
		wsprintf(buffer,"LPT%d: %04x",i+1,lptbase[i]);
		j=(int)SendDlgItemMessage(wnd,100,CB_ADDSTRING,0,(LPARAM)(LPSTR)buffer);
		SendDlgItemMessage(wnd,100,CB_SETITEMDATA,j,i);
		if(i==lpt) SendDlgItemMessage(wnd,100,CB_SETCURSEL,j,0L);
	    }
	}
	SendMessage(wnd,WM_COMMAND,100,MAKELONG(0,CBN_SELCHANGE));
	TurnOff();
	for(i=1;i<sizeof(galinfo)/sizeof(galinfo[0]);i++)
	{
	    j=(int)SendDlgItemMessage(wnd,101,CB_ADDSTRING,0,(LPARAM)(LPSTR)galinfo[i].name),
	    SendDlgItemMessage(wnd,101,CB_SETITEMDATA,j,galinfo[i].type);
	}
	return TRUE;
    case WM_DESTROY:
	wp.length=sizeof(wp);
	GetWindowPlacement(wnd,&wp);
	sprintf(temp,"%u %u %u %u",wp.rcNormalPosition.left,wp.rcNormalPosition.top,wp.rcNormalPosition.right,wp.rcNormalPosition.bottom);
	WriteProfileString(progname,"Window",temp);
        break;
    case WM_SIZE:
	GetClientRect(wnd,&r);
	MoveWindow(GetDlgItem(wnd,102),r.left+editrect.left,r.top+editrect.top,r.right-editrect.right-(r.left+editrect.left),r.bottom-editrect.bottom-(r.top+editrect.top),TRUE);
	break;
    case WM_COMMAND:
	switch(LOWORD(wParam))
	{
	case IDCANCEL:
            if(SendDlgItemMessage(wnd,102,EM_GETMODIFY,0,0L))
            {
                switch(Message(wnd,"JEDEC fuse map changed, save ?",NULL,
                MB_ICONQUESTION|MB_YESNOCANCEL))
                {
                case IDCANCEL:
                    return 0L;
                case IDYES:
                    if(SendMessage(wnd,WM_COMMAND,4,0L)==0) return 0L;
                }
            }
	    EndDialog(wnd,0);
	    return TRUE;
	case 3: // Load
	    memset(&ofn,0,sizeof(OPENFILENAME));
	    buffer[0]='\0';
	    ofn.lStructSize=sizeof(OPENFILENAME);
	    ofn.hwndOwner=wnd;
	    ofn.hInstance=hInstance;
	    ofn.lpstrFilter="Fusemaps(.JED)\0*.jed\0All Files\0*.*\0";
	    ofn.nFilterIndex=1L;
	    ofn.lpstrFile=buffer;
	    ofn.lpstrDefExt="JED";
            ofn.lpstrTitle="Load JEDEC";
	    ofn.nMaxFile=sizeof(buffer);
	    ofn.Flags=OFN_FILEMUSTEXIST|OFN_PATHMUSTEXIST|OFN_HIDEREADONLY;
	    if(GetOpenFileName(&ofn))
	    {
		handle=_lopen(buffer,OF_READ);
		if(handle==-1)
		{
		    Message(wnd,"Can not read from file",NULL,MB_OK|MB_ICONEXCLAMATION);
		}
		else
		{
		    i=_lread(handle,buffer,sizeof(buffer));
		    buffer[i]='\0';
		    ptr=strchr(buffer,'\2');
		    if(ptr)
		    {
			memmove(buffer,ptr+1,i-(ptr-buffer));
		        ptr=strchr(buffer,'\3');
		        if(ptr) *ptr='\0';
		    }
		    SetDlgItemText(wnd,102,buffer);
		    _lclose(handle);
                    SendMessage(wnd,WM_COMMAND,102,MAKELONG(0,EN_CHANGE));
		    i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
		    if(i==-1) gal=0; else gal=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
		    CheckJEDEC(wnd);
		}
	    }
	    break;
	case 4: //  Save
            if(SendDlgItemMessage(wnd,102,WM_GETTEXTLENGTH,0,0L)<1)
            {
                Message(wnd,"Nothing to save",progname,MB_ICONEXCLAMATION|MB_OK);
                return 0L;
            }
	    memset(&ofn,0,sizeof(OPENFILENAME));
	    buffer[0]='\0';
	    ofn.lStructSize=sizeof(OPENFILENAME);
            ofn.lpstrTitle="Save JEDEC";
	    ofn.hwndOwner=wnd;
	    ofn.hInstance=hInstance;
	    ofn.lpstrFilter="Fusemaps(.JED)\0*.jed\0All Files\0*.*\0";
	    ofn.nFilterIndex=1L;
	    ofn.lpstrFile=buffer;
	    ofn.lpstrDefExt="JED";
	    ofn.nMaxFile=sizeof(buffer);
	    ofn.Flags=OFN_PATHMUSTEXIST|OFN_PATHMUSTEXIST|OFN_HIDEREADONLY;
	    if(GetSaveFileName(&ofn))
	    {
		handle=_lcreat(buffer,0);
		if(handle==-1)
		{
		    Message(wnd,"Can not create from file",NULL,MB_OK|MB_ICONEXCLAMATION);
		}
		else
		{
		    buffer[0]='\2';
		    i=GetDlgItemText(wnd,102,buffer+1,sizeof(buffer)-5)+1;
		    for(l=0,j=1;j<i;j++) l+=buffer[j];
		    i+=wsprintf(buffer+i,"\3%04X",l+5);
		    _lwrite(handle,buffer,i);
		    _lclose(handle);
                    SendDlgItemMessage(wnd,102,EM_SETMODIFY,0,0L);
                    return TRUE;
		}
	    }
	    break;
	case 5: // Read GAL
	    if(!TestProperGAL(wnd)) return TRUE;
	    ReadGAL(wnd,fusemap);
	    FormatJEDEC(gal,fusemap,buffer);
	    SetDlgItemText(wnd,102,buffer);
	    SendMessage(wnd,WM_COMMAND,102,MAKELONG(0,EN_CHANGE));
	    return TRUE;
	case 6: // Write GAL
	    i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
	    if(i==-1) gal=0; else gal=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
	    if(GetDlgItemText(wnd,102,buffer,sizeof(buffer))<1)
            {
                MessageBox(wnd,"Load a JEDEC fuse map first",progname,MB_ICONEXCLAMATION|MB_OK);
                return 0L;
            }
	    if(!CheckJEDEC(wnd)) return TRUE;
	    if(!TestProperGAL(wnd)) return TRUE;
	    WriteGAL(wnd,fusemap);
	    if(security)
	    {
		if(Message(wnd,"Programming the security fuse will prohibit the readout and verification of the GAL. Do you want to continue ?",progname,MB_ICONEXCLAMATION|MB_YESNO)!=IDYES) return TRUE;
		BurnSecurity(wnd,gal);
	    }
	    return TRUE;
	case 12: // Verify GAL
	    i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
	    if(i==-1) gal=0; else gal=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
	    if(GetDlgItemText(wnd,102,buffer,sizeof(buffer))<1)
            {
                MessageBox(wnd,"Load a JEDEC fuse map first",progname,MB_ICONEXCLAMATION|MB_OK);
                return 0L;
            }
	    if(!CheckJEDEC(wnd)) return TRUE;
	    if(!TestProperGAL(wnd)) return TRUE;
	    ReadGAL(wnd,backup);
	    if(memcmp(backup,fusemap,galinfo[gal].fuses)==0)
	    {
		Message(wnd,"Verify ok.",NULL,MB_OK);
	    }
	    else
	    {
		Message(wnd,"Verify failed !",NULL,MB_OK);
	    }
	    return TRUE;
	case 7: // Erase GAL
	    if(!TestProperGAL(wnd)) return TRUE;
	    if(pes[1]&0x80) if(Message(wnd,"GAL is a MASTER, still erase ?",NULL,MB_ICONSTOP|MB_YESNO)!=IDYES) return TRUE;
	    EraseGAL(wnd,gal);
	    return TRUE;
	case 8: // Write PES
	    if(!GetSetup(wnd)) return FALSE;
	    if(DialogBox(hInstance,MAKEINTRESOURCE(2),wnd,ProgDlgProc))
	    {
		WritePES(wnd,pes);
	    }
	    return TRUE;
	case 10: // SECURITY
	    if(!TestProperGAL(wnd)) return TRUE;
	    if(Message(wnd,"Programming the security fuse will prohibit the readout and verification of the GAL. Do you want to continue ?",progname,MB_ICONEXCLAMATION|MB_YESNO)!=IDYES) return TRUE;
	    if(pes[1]&0x80) if(Message(wnd,"GAL is a MASTER, still burn security fuse ?",NULL,MB_ICONSTOP|MB_YESNO)!=IDYES) return TRUE;
	    BurnSecurity(wnd,gal);
	    return TRUE;
	case 11: // ERASE ALL
	    if(!TestProperGAL(wnd)) return TRUE;
	    if(Message(wnd,"Erase the whole GAL will also erase the PES (Programmer Electronic Signature) rendering the GAL unusable. Do you want to continue ?",progname,MB_ICONEXCLAMATION|MB_YESNO)!=IDYES) return TRUE;
	    if(pes[1]&0x80) if(Message(wnd,"GAL is a MASTER, still erase whole GAL ?",NULL,MB_ICONSTOP|MB_YESNO)!=IDYES) return TRUE;
	    EraseWholeGAL(wnd,gal);
	    return TRUE;
	case 100:
	    if(HIWORD(lParam)==CBN_SELCHANGE)
	    {
		i=(int)SendDlgItemMessage(wnd,100,CB_GETCURSEL,0,0L);
		if(i!=-1)
		{
		    lpt=(int)SendDlgItemMessage(wnd,100,CB_GETITEMDATA,i,0L);
		    itoa(lpt,buffer,10);
		    WriteProfileString(progname,"Port",buffer);
		    TurnOff();
		}
	    }
	    return TRUE;
	case 101:
	    if(HIWORD(lParam)==CBN_SELCHANGE)
	    {
		SendDlgItemMessage(wnd,103,CB_SETCURSEL,-1,0L);
		SendDlgItemMessage(wnd,104,CB_SETCURSEL,-1,0L);
		SendDlgItemMessage(wnd,105,CB_SETCURSEL,-1,0L);
	    }
	    return TRUE;
	case 900:
	    i=GetModuleFileName(hInstance,buffer,sizeof(buffer));
	    strcpy(buffer+i-3,"HLP");
	    WinHelp(wnd,buffer,HELP_INDEX,0L);
	    return TRUE;
	case 999:
	    return DialogBox(hInstance,MAKEINTRESOURCE(3),wnd,AboutDlgProc);
	case 200:
	case 201:
	case 202:
	case 203:
	case 204:
	case 205:
	case 206:
	case 207:
	case 208:
	case 209:
	    SendDlgItemMessage(wnd,101,CB_SETCURSEL,LOWORD(wParam)-200,0L);
	    return TRUE;
	case 300:
	case 301:
	case 302:
	    SendDlgItemMessage(wnd,100,CB_SETCURSEL,LOWORD(wParam)-300,0L);
	    return TRUE;
        case 310:
            DialogBox(hInstance,MAKEINTRESOURCE(5),wnd,VoltDlgProc);
            return TRUE;
        }
	break;
    case WM_INITMENU:
	i=(int)SendDlgItemMessage(wnd,100,CB_GETCURSEL,0,0L);
	if(i==-1) lpt=-1; else lpt=(int)SendDlgItemMessage(wnd,100,CB_GETITEMDATA,i,0L);
	CheckMenuItem((HMENU)wParam,300,lpt==0?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,301,lpt==1?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,302,lpt==2?MF_CHECKED:MF_UNCHECKED);
	EnableMenuItem((HMENU)wParam,300,lptbase[0]?MF_ENABLED:MF_GRAYED);
	EnableMenuItem((HMENU)wParam,301,lptbase[1]?MF_ENABLED:MF_GRAYED);
	EnableMenuItem((HMENU)wParam,302,lptbase[2]?MF_ENABLED:MF_GRAYED);
	i=(int)SendDlgItemMessage(wnd,101,CB_GETCURSEL,0,0L);
	if(i==-1) gal=0; else gal=(int)SendDlgItemMessage(wnd,101,CB_GETITEMDATA,i,0L);
	CheckMenuItem((HMENU)wParam,200,gal==1?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,201,gal==2?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,202,gal==3?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,203,gal==4?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,204,gal==5?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,205,gal==6?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,206,gal==7?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,207,gal==8?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem((HMENU)wParam,208,gal==9?MF_CHECKED:MF_UNCHECKED);
	return 0L;
    }
    return GrayDlgProc(wnd,msg,wParam,lParam);
}

int PASCAL WinMain(HANDLE hInst,HANDLE hPrev,LPSTR lpCmdLine,int nCmdShow)
{
    hInstance=hInst;
    vppmul=GetProfileInt(progname,"MulDiv",128);
    appicon=LoadIcon(hInst,MAKEINTRESOURCE(1));
    return DialogBox(hInst,MAKEINTRESOURCE(1),0,GALDlgProc);
}

