char wrsh_c[]="$Header: d:/wilhelm/winrsh/RCS/wrsh.c%v 1.6 1994/11/07 04:40:07 wcheung Exp wcheung $";
/*************************************************************************
 *
 * wrsh - User Interface to RSH Module
 *
 * Copyright(C) 1994 William K. W. Cheung
 * All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * WinRSH/WinRSH32 - Remote Shell for Windows
 * Author: William K. W. Cheung (wcheung@ee.ubc.ca)
 * Date:   June 30, 1994
 *
 ************************************************************************/
#define STRICT                  // Microsoft STRICT type-safe compiling
#include <windows.h>
#include <windowsx.h>
#pragma hdrstop
#include <ctl3d.h>
#include <commdlg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "wrsh.h"
#include "rsh.h"
#include "wprintf.h"
#include "msg.h"
#include "err.h"

#define RSHLOGFILE      "winrsh.log"
#define RSHCMD          0x1
#define RSHUSER         0x2
#define RSHHOST         0x4

//
// Local Structure Definition
//
typedef struct
{
	WORD    wVirtualKey;
	int     iMessage;
	WORD    wRequest;
} SCROLLKEYS;

//
// Local Variables
//
static char szAppName[]="WinRSH";
static char FAR *szIniName="winrsh.ini";
static char szRshLogPath[80];
static char szRshViewLog[80];
static HINSTANCE	ghInst;
static HACCEL     ghAccel;
static int        nProtocol;
static BOOL       fLogFile;
static BOOL       fSaveOnExit;
static BOOL       fStartNow=FALSE;
static BOOL       fAutoQuit=FALSE;
static BOOL       fIconic=FALSE;
static int        nTimeOut=0;
static int        nRemTime=0;

//
// Local Function Prototypes
//
static void InitRsh(HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpszCmdLine, int nCmdShow);
static void ParseCmdLine(LPSTR lpszCmdLine);
static void ReadHostList(HWND hWnd);
static void ReadCommandList(HWND hWnd);
static void WriteCommandList(HWND hWnd);
static void ReadInit(void);
static void WriteInit(void);
static BOOL SelectLogFile(HWND hWnd);
static int ReadResponseFile(char *szFileName);
static void CenterDialog(HWND hWnd);

//
// Export Functions
//
extern void LogOutput(char *szBuf);

LRESULT CALLBACK _export RshWndProc(HWND hWnd, UINT iMessage,
									 WPARAM wParam, LPARAM lParam);

LRESULT CALLBACK _export RshDlgProc(HWND hDlg, UINT iMessage,
									 WPARAM wParam, LPARAM lParam);

LRESULT CALLBACK _export RshAboutProc(HWND hDlg, UINT iMessage,
									  WPARAM wParam, LPARAM lParam);

//
// WinMain - Main Program
//

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpszCmdLine, int nCmdShow)
{
	MSG     msg;

	InitRsh(hInstance, hPrevInstance, lpszCmdLine, nCmdShow);
	while (GetMessage(&msg, NULL, NULL, NULL))
	{
		if (!TranslateAccelerator(msg.hwnd, ghAccel, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	Ctl3dUnregister(hInstance);
	WriteInit();

	return msg.wParam;
}

#pragma argsused
static void InitRsh(HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpszCmdLine, int nCmdShow)
{
	WNDCLASS        wcRshClass;
	HWND            hWnd;

	if (Ctl3dRegister(hInstance))
		Ctl3dAutoSubclass(hInstance);

	//
	// Register Rsh Window Class
	//
	if (!hPrevInstance)
	{
		wcRshClass.lpszClassName = szAppName;
		wcRshClass.hInstance     = hInstance;
		wcRshClass.lpfnWndProc   = RshWndProc;
		wcRshClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
		wcRshClass.hIcon         = LoadIcon(hInstance, szAppName);
		wcRshClass.lpszMenuName  = (LPSTR)szAppName;
		wcRshClass.hbrBackground = GetStockObject(LTGRAY_BRUSH);
		wcRshClass.style         = 0;
		wcRshClass.cbClsExtra    = 0;
		wcRshClass.cbWndExtra    = 0;

		RegisterClass(&wcRshClass);
	}
#ifndef __WIN32__
	else
		GetInstanceData(hPrevInstance, (BYTE *)szAppName, sizeof(szAppName));
#endif

	//
	// Create Rsh Window
	//

	hWnd = CreateWindow(
				szAppName,              // Window class name
				szAppName,              // Window title
				WS_OVERLAPPEDWINDOW,
				CW_USEDEFAULT,  // x window location
				CW_USEDEFAULT,  // y window location
				80*9,           // horizontal window size
				24*16,          // vertical window size
				NULL,           // No parent for this window
				NULL,           // Use the class menu
				hInstance,      // owner of this window
				NULL            // No params to pass on
			);

	ParseCmdLine(lpszCmdLine);

	ghAccel = LoadAccelerators(ghInst, (LPSTR)szAppName);

	if (fIconic)
		ShowWindow(hWnd, SW_SHOWMINIMIZED);
	else
		ShowWindow(hWnd, nCmdShow);

	UpdateWindow(hWnd);

	PostMessage(hWnd, WM_COMMAND, IDM_FILECONNECT, 0L);
}

LRESULT CALLBACK _export RshWndProc(HWND hWnd, UINT iMessage,
									 WPARAM wParam, LPARAM lParam)
{
	static  BOOL    bConnected=FALSE;
	DLGPROC lpProcInput;
	HMENU   hMenu;
	int     nRc, nDuration;

	switch(iMessage)
	{
		case WM_CREATE:
			SetWindowText(hWnd, "WinRSH - Remote Shell for Windows");
			ErrInit(hWnd);
			ReadInit();
			hMenu = GetMenu(hWnd);
			EnableMenuItem(hMenu, IDM_FILEDISCONNECT, MF_DISABLED|MF_GRAYED);
			CheckMenuItem(hMenu, IDM_EDITSAVE,
							  MF_BYCOMMAND | (fSaveOnExit ? MF_CHECKED : MF_UNCHECKED));
			CheckMenuItem(hMenu, IDM_EDITLOG,
							  MF_BYCOMMAND | (fLogFile ? MF_CHECKED : MF_UNCHECKED));
			//
			// Save an instance copy for use by other window procs.
			//
			ghInst = ((LPCREATESTRUCT)lParam)->hInstance;
			SetFont("System", 12, 0);
			DoInitMsg(hWnd, ghInst);
			DoInitWprintf(hWnd, ghInst);
			return DefWindowProc(hWnd, iMessage, wParam, lParam);

		case WM_COMMAND:
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDM_FILECONNECT:
					lpProcInput = (DLGPROC)MakeProcInstance((FARPROC)RshDlgProc, ghInst);
					nRc = DialogBox(ghInst, MAKEINTRESOURCE(RSHCONNECT), hWnd, lpProcInput);
#ifndef __WIN32__
					FreeProcInstance((FARPROC)lpProcInput);
#endif
					if (nRc != FALSE)
					{
						if (!RshOpen(nRc == RB_RSH))
							break;

						if (nTimeOut > 0)
						{
							nDuration = nTimeOut % 30;
							if (nDuration == 0)
								nDuration = 30;
							SetTimer(hWnd, RSHTIMEOUT, nDuration * 1000, 0L);
							nRemTime = nTimeOut - nDuration;
						}
						hMenu = GetMenu(hWnd);
						EnableMenuItem(hMenu, IDM_FILECONNECT, MF_DISABLED|MF_GRAYED);
						EnableMenuItem(hMenu, IDM_FILEDISCONNECT, MF_ENABLED);
						if (RshGetHost(hWnd, bConnected) == TRUE)
							bConnected = TRUE;
						else
						{
							EnableMenuItem(hMenu, IDM_FILECONNECT, MF_ENABLED);
							EnableMenuItem(hMenu, IDM_FILEDISCONNECT, MF_DISABLED|MF_GRAYED);
						}
					}
					else if (fAutoQuit)
						PostMessage(hWnd, WM_CLOSE, 0, 0L);
					break;

				case IDM_FILEDISCONNECT:
					if (MessageBox(hWnd, "Disconnect now?", "WinRSH - Disconnect",
						MB_ICONQUESTION|MB_YESNO) == IDYES)
						PostMessage(hWnd, WM_RSHENDSESSION, 0, 0L);
					break;

				case IDM_FILEOPENLOG:
					sprintf(szRshViewLog, "notepad %s", szRshLogPath);
					nRc = WinExec(szRshViewLog, SW_SHOWNORMAL);
					if (nRc == 0)
						ErrPrintf(ERR_WARN, "Out of System Memory.");
					else if (nRc < 5)
						ErrPrintf(ERR_WARN, "Make sure notepad.exe is in your search path.");
					break;

				case IDM_FILELOG:
					SelectLogFile(hWnd);
					break;

				case IDM_QUIT:
					if (bConnected == TRUE)
						RshDisconnect();
					PostMessage(hWnd, WM_CLOSE, 0, 0L);
					break;

				case IDM_EDITSAVE:
					fSaveOnExit = ((GetMenuState(GetMenu(hWnd), IDM_EDITSAVE,
												 MF_BYCOMMAND)& MF_CHECKED) > 0);
					fSaveOnExit ^= TRUE;
					CheckMenuItem(GetMenu(hWnd), IDM_EDITSAVE,
						MF_BYCOMMAND | (fSaveOnExit ? MF_CHECKED : MF_UNCHECKED));
					return(TRUE);

				case IDM_EDITLOG:
					fLogFile = ((GetMenuState(GetMenu(hWnd), IDM_EDITLOG,
													 MF_BYCOMMAND)& MF_CHECKED) > 0);
					fLogFile ^= TRUE;
					CheckMenuItem(GetMenu(hWnd), IDM_EDITLOG,
							MF_BYCOMMAND | (fLogFile ? MF_CHECKED : MF_UNCHECKED));
					return(TRUE);

				case IDM_EDITCLEAR:
					wcls();
					break;

				case IDM_HELPABOUT:
					lpProcInput = (DLGPROC)MakeProcInstance((FARPROC)RshAboutProc, ghInst);
					DialogBox(ghInst, MAKEINTRESOURCE(RSHABOUT), hWnd, lpProcInput);
#ifndef __WIN32__
					FreeProcInstance((FARPROC)lpProcInput);
#endif                  
					break;

				default:
					break;
			}
			break;

		case WM_SIZE:
			// Retrieve the dimensions of the client area.
			DoWprintfSize(lParam, MB_HEIGHT);
			DoMsgSize(lParam, MB_HEIGHT);
			return DefWindowProc(hWnd, iMessage, wParam, lParam);

		case WM_KEYDOWN:
			DoWprintfKeys(wParam, lParam);
			break;

		case WM_SYSCOMMAND:
			if (wParam != SC_CLOSE)
				return DefWindowProc(hWnd, iMessage, wParam, lParam);
		case WM_CLOSE:
			if (bConnected)
			{
				RshDisconnect();
				RshClose();
			}
			DestroyWindow(hWnd);
			break;

		case WM_DESTROY:
			PostQuitMessage(0);
			break;

		case WM_TIMER:
			KillTimer(hWnd, wParam);
			switch(wParam)
			{
				case RSHRECV:
					RshRecvAsync(hWnd);
					break;

				case RSHTIMEOUT:
					if (nRemTime > 0)
					{
						nDuration = nRemTime % 30;
						if (nDuration == 0)
							nDuration = 30;
						SetTimer(hWnd, RSHTIMEOUT, nDuration * 1000, 0L);
						nRemTime -= nDuration;
					}
					else        // Timer expired
					{
						wputs("winrsh: timeout\n");
						LogOutput("winrsh: timeout\n");
						PostMessage(hWnd, WM_RSHENDSESSION, 0, 0L);
					}
					break;

				default:
					DefWindowProc(hWnd, iMessage, wParam, lParam);
					break;
			}
			break;

		case WM_RSH:
			DoRsh(hWnd, wParam, lParam);
			break;

		case WM_RSHGETHOST:
			if (!bConnected || (RshConnect(hWnd, wParam, lParam) == FALSE))
			{
				EnableMenuItem(hMenu, IDM_FILECONNECT, MF_ENABLED);
				EnableMenuItem(hMenu, IDM_FILEDISCONNECT, MF_DISABLED|MF_GRAYED);
			}
			break;

		case WM_RSHENDSESSION:
			if (bConnected)
			{
				bConnected = FALSE;
				KillTimer(hWnd, RSHRECV);
				KillTimer(hWnd, RSHTIMEOUT);
				MsgPrintf("Disconnecting from %s...", rshStruct.hostname);
				RshDisconnect();
				MsgPrintf("Done");
			}
			hMenu = GetMenu(hWnd);
			EnableMenuItem(hMenu, IDM_FILECONNECT, MF_ENABLED);
			EnableMenuItem(hMenu, IDM_FILEDISCONNECT, MF_DISABLED|MF_GRAYED);
			RshClose();
			if (fAutoQuit)
				PostMessage(hWnd, WM_CLOSE, 0, 0L);
			break;

		default:
			return DefWindowProc(hWnd, iMessage, wParam, lParam);
	}

	return(FALSE);
}

#pragma argsused
LRESULT CALLBACK _export RshDlgProc(HWND hDlg, UINT iMessage,
									 WPARAM wParam, LPARAM lParam)
{
	static int iCnt=0;
	BOOL    bVisible;
	int     nIdx;
	char    szBuf[80];

	switch(iMessage)
	{
		case WM_INITDIALOG:
			CenterDialog(hDlg);
			SetFocus(GetDlgItem(hDlg, IDD_COMMAND));
			SendDlgItemMessage(hDlg, IDD_USER, EM_LIMITTEXT, 20, 0L);
			SendDlgItemMessage(hDlg, IDD_PASSWD, EM_LIMITTEXT, 20, 0L);
			SendDlgItemMessage(hDlg, IDD_COMMAND, EM_LIMITTEXT, 256, 0L);
			SendDlgItemMessage(hDlg, IDC_HOST, CB_LIMITTEXT, 40, 0L);
			SetDlgItemText(hDlg, IDD_USER, rshStruct.user);
			SetDlgItemText(hDlg, IDD_PASSWD, rshStruct.passwd);
			SetDlgItemText(hDlg, IDD_COMMAND, rshStruct.cmd);
			SetDlgItemText(hDlg, IDC_HOST, rshStruct.hostname);
			CheckRadioButton(hDlg, RB_REXEC, RB_RSH, nProtocol);
			if (nProtocol == RB_RSH)
			{
				EnableWindow(GetDlgItem(hDlg, ID_PASSWD), FALSE);
				EnableWindow(GetDlgItem(hDlg, IDD_PASSWD), FALSE);
			}
			if (fStartNow)
				SendMessage(hDlg, WM_COMMAND, ID_OK, 0L);
			else
			{
				ReadHostList(hDlg);
				ReadCommandList(hDlg);
			}
			break;

		case WM_COMMAND:
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case ID_OK:
					GetDlgItemText(hDlg, IDD_USER, rshStruct.user,
									sizeof(rshStruct.user));
					GetDlgItemText(hDlg, IDD_PASSWD, rshStruct.passwd,
									sizeof(rshStruct.passwd));
					GetDlgItemText(hDlg, IDD_COMMAND, rshStruct.cmd,
									sizeof(rshStruct.cmd));
					GetDlgItemText(hDlg, IDC_HOST, rshStruct.hostname,
									sizeof(rshStruct.hostname));
					if (IsDlgButtonChecked(hDlg, RB_REXEC))
						nProtocol = RB_REXEC;
					else
						nProtocol = RB_RSH;

					if (!fStartNow)
						WriteCommandList(hDlg);

					fStartNow = FALSE;

					EndDialog(hDlg, nProtocol);
					return(TRUE);

				case ID_CANCEL:
					EndDialog(hDlg, FALSE);
					return(TRUE);

				case RB_REXEC:
					CheckRadioButton(hDlg, RB_REXEC, RB_RSH, nProtocol = RB_REXEC);
					EnableWindow(GetDlgItem(hDlg, ID_PASSWD), TRUE);
					EnableWindow(GetDlgItem(hDlg, IDD_PASSWD), TRUE);
					break;

				case RB_RSH:
					CheckRadioButton(hDlg, RB_REXEC, RB_RSH, nProtocol = RB_RSH);
					EnableWindow(GetDlgItem(hDlg, ID_PASSWD), FALSE);
					EnableWindow(GetDlgItem(hDlg, IDD_PASSWD), FALSE);
					break;

				case IDC_HOST:
					switch(GET_WM_COMMAND_CMD(wParam, lParam))
					{
						case CBN_DROPDOWN:
							if (iCnt == 0)
							{
								iCnt++;
								bVisible = (BOOL)SendMessage((HWND)lParam,
													CB_GETDROPPEDSTATE, 0, 0L);
								if (bVisible)
									SendMessage((HWND)lParam,
											CB_SHOWDROPDOWN, (WPARAM)FALSE, 0L);
								else
									SendMessage((HWND)lParam,
											CB_SHOWDROPDOWN, (WPARAM)TRUE, 0L);
							}
							else
								iCnt = 0;
							break;

						case CBN_KILLFOCUS:
							GetDlgItemText(hDlg, IDC_HOST, szBuf, sizeof(szBuf));
							nIdx = (int)SendDlgItemMessage(hDlg, IDC_HOST, CB_FINDSTRING,
																	 0, (LPARAM)szBuf);
							if (nIdx != CB_ERR)
								SendDlgItemMessage(hDlg, IDC_HOST, CB_SETCURSEL, nIdx, 0L);
							break;

						default:
							break;
					}
					break;

				default:
					break;
			}
			break;

		default:
			break;
	}

	return(FALSE);
}

#pragma argsused
LRESULT CALLBACK _export RshAboutProc(HWND hDlg, UINT iMessage,
									  WPARAM wParam, LPARAM lParam)
{
	switch(iMessage)
	{
		case WM_INITDIALOG:
			SetFocus(GetDlgItem(hDlg, ID_OK));
			break;

		case WM_CLOSE:
			PostMessage(hDlg, WM_COMMAND, ID_OK, 0L);
			break;

		case WM_COMMAND:
			switch(GET_WM_COMMAND_ID(wParam, lParam))
			{
				case ID_OK:
					EndDialog(hDlg, TRUE);
					return(TRUE);

				default:
					break;
			}
			break;

		default:
			break;
	}

	return(FALSE);
}

static void ReadHostList(HWND hWnd)
{
	FILE    *fileHost;
	char    szBuf[256], szHostName[40];
	char    *pszEnv;

	pszEnv = getenv("RSHHOST_DIR");
	if (pszEnv)
		wsprintf(szBuf, "%s\\hosts", pszEnv);
	else
		strcpy(szBuf, "hosts");

	fileHost = fopen(szBuf, "r");
	if (fileHost == NULL)
		return;

	while(fgets(szBuf, 256, fileHost))
	{
		if (szBuf[0] != '#')
		{
			if (sscanf(szBuf, "%*s %s %*s", szHostName) >= 1)
				SendDlgItemMessage(hWnd, IDC_HOST, CB_ADDSTRING,
									0, (LPARAM)szHostName);
		}
	}
	fclose(fileHost);
}

void ReadInit(void)
{
	char    szBuf[256];
	int nLen;

	fSaveOnExit = (BOOL)GetPrivateProfileInt("UserOpt", "SaveOnExit", 1, szIniName);

	fLogFile = (BOOL)GetPrivateProfileInt("UserOpt", "LogToFile", 1, szIniName);

	nLen = GetWindowsDirectory((LPSTR)szBuf, sizeof(szBuf));
	if ((nLen > 0) && (nLen < sizeof(szBuf)-sizeof(RSHLOGFILE)))
	{
		strcat(szBuf, "\\");
		strcat(szBuf, RSHLOGFILE);
	}
	else
		sprintf(szBuf, "c:\\%s", RSHLOGFILE);

	GetPrivateProfileString("UserOpt", "LogFileName", szBuf, szRshLogPath,
									sizeof(szRshLogPath), szIniName);

	if (!rshStruct.user[0])
		GetPrivateProfileString("Login", "Username", "anonymous", rshStruct.user,
										sizeof(rshStruct.user), szIniName);

	if (!rshStruct.hostname[0])
		GetPrivateProfileString("Login", "Hostname", "localhost", rshStruct.hostname,
										sizeof(rshStruct.hostname), szIniName);

	rshStruct.cmd[0] = '\0';

	if (nProtocol == 0)
		nProtocol = GetPrivateProfileInt("Login", "Protocol", RB_REXEC, szIniName);
}

void WriteInit(void)
{
	char    szTmp[20];

	// Do not save preferences if the user runs from command line.
	if (fAutoQuit)
		return;

	WritePrivateProfileString("UserOpt", "SaveOnExit", itoa((int)fSaveOnExit, szTmp, 10), szIniName);
	if (!fSaveOnExit)
		return;

	WritePrivateProfileString("UserOpt", "LogToFile", itoa((int)fLogFile, szTmp, 10), szIniName);
	WritePrivateProfileString("UserOpt", "LogFileName", szRshLogPath, szIniName);
	WritePrivateProfileString("Login", "Username", rshStruct.user, szIniName);
	WritePrivateProfileString("Login", "Hostname", rshStruct.hostname, szIniName);
	WritePrivateProfileString("Login", "Protocol", itoa(nProtocol, szTmp, 10), szIniName);
}

int ReadResponseFile(char *szFileName)
{
	int nParam = 0;
	int	  nLen;
	FILE    *pFile;
	char    szBuf[256];

	pFile = fopen(szFileName, "rt");
	if (pFile == (FILE *)NULL)
	{
		ErrPrintf(ERR_WARN, "Response File %s is not found.", szFileName);
		return 0;
	}

	while(fgets(szBuf, 255, pFile))
	{
		if (szBuf[0] == '-')
		{
			switch(szBuf[1])
			{
				case 'L':
				case 'l':
					if (sscanf(&szBuf[2], "%s", rshStruct.user) > 0)
						nParam |= RSHUSER;
					break;

				case 'P':
				case 'p':
					nProtocol = RB_REXEC;
					rshStruct.passwd[0] = '\0';
					sscanf(&szBuf[2], "%s", rshStruct.passwd);
					break;

				case 'H':
				case 'h':
					if (sscanf(&szBuf[2], "%s", rshStruct.hostname) > 0)
						nParam |= RSHHOST;
					break;

				case 'o':
				case 'O':
					if (sscanf(&szBuf[2], "%s", szRshLogPath) <= 0)
						goto ResponseError;
					fLogFile = TRUE;
					break;

				case 't':
				case 'T':
					if (sscanf(&szBuf[2], "%d", &nTimeOut) <= 0)
						goto ResponseError;
					break;

				case 'I':
				case 'i':
					fIconic = TRUE;
					break;

				case 'Q':
				case 'q':
					fAutoQuit = TRUE;
					break;

				ResponseError:
				case '?':
				default:
					ErrPrintf(ERR_WARN, "Usage: winrsh [-h hostname] [-l username]\n"
								 "      [-p password] [-o output_filename] [-t seconds] [-i] [-q] "
								 "[command]");
					return 0;
			}
		}
		else if (szBuf[0] != '\n')
		{
			strncpy(rshStruct.cmd, szBuf, 255);
			nLen = strlen(rshStruct.cmd);
			if (nLen > 0 && rshStruct.cmd[nLen-1] == '\n')
				rshStruct.cmd[nLen-1] = '\0';
			rshStruct.cmd[255] = '\0';
			nParam |= RSHCMD;
		}
	}

	fclose(pFile);

	return(nParam);
}

void ParseCmdLine(LPSTR lpszCmdLine)
{
	int nParam=0;
	char    szBuf[256];
	int nPos;

#ifdef __WIN32__
	for(; *lpszCmdLine != ' '; lpszCmdLine++);
	lpszCmdLine++;
#endif
	if (!*lpszCmdLine)
		return;

	nProtocol = RB_RSH;
	fStartNow = FALSE;

	for(; *lpszCmdLine; lpszCmdLine++)
	{
		if (sscanf(lpszCmdLine, "%s%n", szBuf, &nPos) > 0)
		{
			lpszCmdLine += nPos;
			if (szBuf[0] == '@')
			{
				nParam = ReadResponseFile(&szBuf[1]);
				break;
			}
			if (szBuf[0] == '-')
			{
				switch(szBuf[1])
				{
					case 'L':
					case 'l':
						if (sscanf(lpszCmdLine, "%s%n", rshStruct.user, &nPos) > 0)
							lpszCmdLine += nPos;
						else
							goto InputError;
						nParam |= RSHUSER;
						break;

					case 'P':
					case 'p':
						nProtocol = RB_REXEC;
						if (sscanf(lpszCmdLine, "%s%n", rshStruct.passwd, &nPos) > 0)
							lpszCmdLine += nPos;
						else
							goto InputError;
						break;

					case 'H':
					case 'h':
						if (sscanf(lpszCmdLine, "%s%n", rshStruct.hostname, &nPos) > 0)
							lpszCmdLine += nPos;
						else
							goto InputError;
						nParam |= RSHHOST;
						break;

					case 'o':
					case 'O':
						if (sscanf(lpszCmdLine, "%s%n", szRshLogPath, &nPos) > 0)
							lpszCmdLine += nPos;
						else
							goto InputError;
						fLogFile = TRUE;
						break;

					case 't':
					case 'T':
						if (sscanf(lpszCmdLine, "%d%n", &nTimeOut, &nPos) > 0)
							lpszCmdLine += nPos;
						else
							goto InputError;
						break;

					case 'I':
					case 'i':
						fIconic = TRUE;
						break;

					case 'Q':
					case 'q':
						fAutoQuit = TRUE;
						break;

					InputError:
					case '?':
					default:
						ErrPrintf(ERR_WARN, "Usage: winrsh [-h hostname] [-l username]\n"
									 "      [-p password] [-o output_filename] [-t seconds] [-i] [-q] "
									 "[command]");
						return;
				}
			}
			else if (szBuf[0] == '"')
			{
				sprintf(rshStruct.cmd, "%s%.225s", &szBuf[1], lpszCmdLine);
				nPos = strlen(rshStruct.cmd);
				// Strip trailing white space
				while(nPos > 0)
				{
					if (isspace(rshStruct.cmd[nPos-1]))
					{
						rshStruct.cmd[nPos-1] = '\0';
						nPos--;
					}
					else
						break;
				}
				if (rshStruct.cmd[nPos-1] == '"')
					rshStruct.cmd[nPos-1] = '\0';
				nParam |= RSHCMD;
				break;
			}
			else
			{
				strcpy(rshStruct.cmd, szBuf);
				nParam |= RSHCMD;
				break;
			}
		}
	}
	if (nProtocol == RB_REXEC && rshStruct.passwd[0] == '\0')
		nParam = 0;

	if ((nParam & RSHCMD) && (nParam & RSHUSER) && (nParam & RSHHOST))
		fStartNow = TRUE;
}

BOOL SelectLogFile(HWND hWnd)
{
	OPENFILENAME ofn;

	/* Set all structure members to zero. */

	memset(&ofn, 0, sizeof(OPENFILENAME));

	/* Initialize the OPENFILENAME members. */

	ofn.lStructSize     = sizeof(OPENFILENAME);
	ofn.hwndOwner       = hWnd;
	ofn.lpstrFilter     = "Log Files (*.log)\0*.log\0\0";
	ofn.lpstrFile       = szRshLogPath;
	ofn.nMaxFile        = sizeof(szRshLogPath);
	ofn.lpstrTitle      = "WinRSH - Select Log File";
	ofn.lpstrInitialDir = szRshLogPath;
	ofn.Flags           = OFN_HIDEREADONLY|OFN_PATHMUSTEXIST;

	return (GetSaveFileName(&ofn));
}

void LogOutput(char *szBuf)
{
	FILE *pFile;

	if (fLogFile)
	{
		pFile = fopen(szRshLogPath, "a");
		if (pFile)
		{
			fprintf(pFile, "%s", szBuf);
			fclose(pFile);
		}
	}
}

static void ReadCommandList(HWND hWnd)
{
	char    szCmdTag[82];   // This corresponds to 20 entries 4 * 20 + 2
	char      szCmd[256];
	char      *pszKey;

	GetPrivateProfileString("Commands", NULL, "", szCmdTag,
									sizeof(szCmdTag), szIniName);
									
	for (pszKey = szCmdTag; *pszKey != '\0'; pszKey += strlen(pszKey) + 1)
	{
		 // Retrieve the value for each entry in the buffer.
		 if (GetPrivateProfileString("Commands", pszKey, "", szCmd, sizeof(szCmd), szIniName))
			 SendDlgItemMessage(hWnd, IDD_COMMAND, CB_ADDSTRING, 0, (LPARAM)szCmd);
	}
	if (rshStruct.cmd[0] == '\0')
	{
		SendDlgItemMessage(hWnd, IDD_COMMAND, CB_SETCURSEL, 0, 0L);
		GetDlgItemText(hWnd, IDD_COMMAND, rshStruct.cmd, sizeof(rshStruct.cmd));
	}
	else
		SetDlgItemText(hWnd, IDD_COMMAND, rshStruct.cmd);
}

static void WriteCommandList(HWND hWnd)
{
	char   szCmdTag[13];
	char     szCmd[256];
	int    nEntry, nIdx, nRc;

	// Erase all entries in the Commands section
	WritePrivateProfileString("Commands", NULL, NULL, szIniName);

	// Recreate the list
	// Step 1: Check if current item is in the list, if not, add it
	GetDlgItemText(hWnd, IDD_COMMAND, szCmd, sizeof(szCmd));
	if (strlen(szCmd) > 0)
	{
		nRc = (int)SendDlgItemMessage(hWnd, IDD_COMMAND, CB_FINDSTRINGEXACT, -1, (LPARAM)szCmd);
		WritePrivateProfileString("Commands", "C00", szCmd, szIniName);
		if (nRc >= 0)   // Item exists, remove from list
			SendDlgItemMessage(hWnd, IDD_COMMAND, CB_DELETESTRING, nRc, 0L);
	}
	// Step 2: Add the rest of the items in the list
	nEntry = (int)SendDlgItemMessage(hWnd, IDD_COMMAND, CB_GETCOUNT, 0, 0L);
	for(nIdx=0; nIdx < nEntry; nIdx++)
	{
		SendDlgItemMessage(hWnd, IDD_COMMAND, CB_GETLBTEXT, nIdx, (LPARAM)szCmd);
		sprintf(szCmdTag, "C%02d", nIdx+1);
		WritePrivateProfileString("Commands", szCmdTag, szCmd, szIniName);
	}
}

static void CenterDialog(HWND hWnd)
{
	RECT    rDesk, rWnd;
	int     nWidth, nHeight;
	int     nOffsetX, nOffsetY;

	GetWindowRect(GetDesktopWindow(), &rDesk);
	GetWindowRect(hWnd, &rWnd);

	nWidth = (rWnd.right - rWnd.left);
	nHeight = (rWnd.bottom - rWnd.top);
	nOffsetX = (rDesk.right - rDesk.left) / 2 + rDesk.left - nWidth/2;
	nOffsetY = (rDesk.bottom - rDesk.top) / 2 + rDesk.top - nHeight/2;
	SetWindowPos(hWnd, NULL, nOffsetX, nOffsetY, nWidth, nHeight, 0);
}

