unit MultiCh;

{  ******
   *
   * Module:    MultiCh
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module implements the TMultiChannelStateAware class, which serves
   *    as a base class for the game controller object.  It defines a
   *    state-aware object that can keep track of multiple state queues.
   *    This is like most game objects, except they only have one state queue.
   *
   ****** }

interface
Uses State, StateCh, Classes, Message;

{ Maximum number of state channels. }
const MAX_CHANNELS = 16;

type TMultiChannelStateAware = class(TMessageAware)
    public
        { Class constructor and destructor. }
        constructor Create;
        destructor Destroy;

        { Method to open up a new state channel.  The function returns a handle
          to the newly opened state queue. }
        function iOpenNewChannel: Integer;

        { Method to return reference to a specific channel. }
        function saGetChannel(iChannel: Integer): TStateChannel;
        function bIsValidChannel(iChannel: Integer): Boolean;

        { Methods to control state channels. }
        procedure AddToStateSequence(iChannel: Integer; enumNewState: enumState; fValue1, fValue2: Real);
        procedure ClearStateSequence(iChannel: Integer);
        procedure ExitCurrentState(iChannel: Integer);

        { Virtual method called to process channel states. }
        procedure Move; Virtual;

    protected

        { Virtual methods for customizing state behavior. }
        procedure EnterState(iChannel: Integer; pstCurrent: PState); Virtual;
        procedure ExitState(iChannel: Integer; pstCurrent: PState); Virtual;
        procedure ProcessState(iChannel: Integer; pstCurrent: PState); Virtual;

    private

        { Array of state channels. }
        m_asaChannel: Array[0 .. MAX_CHANNELS - 1] of TStateChannel;
        m_iChannelsUsed: Integer;
end;

implementation

constructor TMultiChannelStateAware.Create;
begin
    { Perform default processing. }
    inherited Create;

    { Initially, no channels are in use. }
    m_iChannelsUsed := 0;
end;

destructor TMultiChannelStateAware.Destroy;
var
    iIndex: Integer;
begin
    { Free all state channels. }
    for iIndex := 0 to (m_iChannelsUsed - 1) do
        m_asaChannel[iIndex].Free;

    { Perform default cleanup. }
    inherited Destroy;
end;

function TMultiChannelStateAware.iOpenNewChannel: Integer;
var
    iNewChannel: Integer;
begin
    { If no channels can be opened then return an invalid channel ID. }
    if (m_iChannelsUsed >= MAX_CHANNELS) then
    begin
        Result := -1;
        Exit;
    end;

    { Create a new state aware object to handle the channel. }
    iNewChannel := m_iChannelsUsed;
    m_asaChannel[iNewChannel] := TStateChannel.Create;

    { Increment the total number of channels in use, and return the ID of the
      newly opened channel. }
    Inc(m_iChannelsUsed);
    Result := iNewChannel;
end;

function TMultiChannelStateAware.bIsValidChannel(iChannel: Integer): Boolean;
begin
    { Valid the given channel ID. }
    Result := ((iChannel >= 0) and (iChannel < m_iChannelsUsed));
end;

function TMultiChannelStateAware.saGetChannel(iChannel: Integer): TStateChannel;
begin
    { Return a reference to the requested channel, or NULL if the given
      channel ID is invalid. }
    if bIsValidChannel(iChannel) = True then
        Result := m_asaChannel[iChannel]
    else
        Result := nil;
end;

procedure TMultiChannelStateAware.Move;
var
    iIndex: Integer;
begin
    { Process state movements for all currently open channels. }
    for iIndex := 0 to (m_iChannelsUsed - 1) do
    begin
        { Perform any pending state activity. }
        m_asaChannel[iIndex].Move;

        { Check if there was any activity on the channel, and reflect it on
          the common channel. }
        case m_asaChannel[iIndex].iGetLastAction of
            ACTION_STATE_ENTERED:   EnterState(iIndex, m_asaChannel[iIndex].pstGetLastSubject);
            ACTION_STATE_EXITED:    ExitState(iIndex, m_asaChannel[iIndex].pstGetLastSubject);
            ACTION_STATE_PROCESSED: ProcessState(iIndex, m_asaChannel[iIndex].pstGetLastSubject);
        else
        end;
    end;
end;

procedure TMultiChannelStateAware.AddToStateSequence(iChannel: Integer; enumNewState: enumState; fValue1, fValue2: Real);
begin
    { Add a command to the given state channel. }
    if bIsValidChannel(iChannel) = True then
        m_asaChannel[iChannel].AddToStateSequence(enumNewState, fValue1, fValue2);
end;

procedure TMultiChannelStateAware.ClearStateSequence(iChannel: Integer);
begin
    { Clear the given state channel. }
    if bIsValidChannel(iChannel) = True then
        m_asaChannel[iChannel].ClearStateSequence;
end;

procedure TMultiChannelStateAware.ExitCurrentState(iChannel: Integer);
begin
    { End the current command on the given state channel. }
    if bIsValidChannel(iChannel) = True then
        m_asaChannel[iChannel].ExitCurrentState;
end;

procedure TMultiChannelStateAware.EnterState(iChannel: Integer; pstCurrent: PState);
var
    iIndex: Integer;
begin
    { Process the command. }
    case pstCurrent^.enumCommandState of
    EndChannelState:  { End the current state on the given channel. }
        begin
        { End the current command on the requested channel. }
        iIndex := Trunc(pstCurrent^.fValue1);
        if bIsValidChannel(iIndex) then
            saGetChannel(iIndex).ExitCurrentState;

        { The EndChannelState command has no other processing. }
        saGetChannel(iChannel).ExitCurrentState;
        end;
    else
    end;
end;

procedure TMultiChannelStateAware.ExitState(iChannel: Integer; pstCurrent: PState);
begin
end;

procedure TMultiChannelStateAware.ProcessState(iChannel: Integer; pstCurrent: PState);
begin
end;

end.
