{------------------------------------------------------------------------------
  Label Item - item class for RichView.
  Non-text item that looks like a text (but cannot be wrapped and edited)
  Does not support Unicode.

  v1.5:
  impr: hypertext support
  v1.4:
  fix: correct copying by AppendFrom
  v1.3:
  fix: printing
  v1.2:
  impr: correct working with DeleteUnusedStyles
  impr: ApplyText and ApplyStyleConversion affect this item
    (if ProtectTextStyleNo=False)
  impr: can be saved in text, RTF, HTML (in RTF and HTML, it is saved as a
    plain text, i.e. it has style of the preceding text)
-------------------------------------------------------------------------------}

unit LabelItem;

interface
uses SysUtils, Classes, Windows, Graphics, RVFuncs,
     RVScroll, CRVData, RVStyle, RVItem, RVFMisc, DLines, CRVFData, RichView,
     RVClasses, RVERVData, RVEdit;

const
  rvsLabel = -200;

type
  TRVLabelItemInfo = class(TRVRectItemInfo)
    private
      Width, Height, Descend: Integer;
    protected
      function GetDescent: Integer; override;
      function GetHeight: Integer; override;
      function GetWidth: Integer;  override;
    public
      Text: String;
      RVStyle: TRVStyle;
      TextStyleNo: Integer;
      ProtectTextStyleNo: Boolean;
      constructor CreateEx(RVData: TPersistent; TextStyleNo: Integer; const Text: String);
      function GetMinWidth(sad: PRVScreenAndDevice; Canvas: TCanvas; RVData: TPersistent): Integer; override;
      function GetBoolValue(Prop: TRVItemBoolProperty): Boolean; override;
      function GetBoolValueEx(Prop: TRVItemBoolPropertyEx; RVStyle: TRVStyle): Boolean; override;
      procedure Paint(x,y: Integer; Canvas: TCanvas; State: TRVItemDrawStates;
                      Style: TRVStyle; dli: TRVDrawLineInfo); override;
      procedure Print(Canvas: TCanvas; x,y,x2: Integer; Preview, Correction: Boolean;
                      const sad: TRVScreenAndDevice; RichView: TRVScroller;
                      dli: TRVDrawLineInfo;
                      Part: Integer); override;
      procedure AfterLoading(FileFormat: TRVLoadFormat); override;
      procedure SaveRVF(Stream: TStream; RVData: TPersistent; ItemNo, ParaNo: Integer;
                        const Name: String; Part: TRVMultiDrawItemPart;
                        ForceSameAsPrev: Boolean); override;
      function ReadRVFLine(const s: String; RVData: TPersistent;
                           ReadType, LineNo, LineCount: Integer;
                           var Name: String;
                           var ReadMode: TRVFReadMode;
                           var ReadState: TRVFReadState): Boolean; override;
      procedure Assign(Source: TCustomRVItemInfo); override;
      procedure MarkStylesInUse(UsedTextStyles, UsedParaStyles, UsedListStyles: TRVIntegerList); override;
      procedure UpdateStyles(TextStylesShift, ParaStylesShift, ListStylesShift: TRVIntegerList); override;
      procedure ApplyStyleConversion(RVData: TPersistent; UserData: Integer); override;
      procedure UpdateMe;
      procedure Execute(RVData:TPersistent);override;
      procedure SaveRTF(Stream: TStream; RVData: TPersistent; ItemNo: Integer;
        const Name: String; TwipsPerPixel: Double; Level: Integer;
        ColorList: TRVColorList;
        StyleToFont, ListOverrideOffsetsList1, ListOverrideOffsetsList2: TRVIntegerList;
        FontTable: TRVList); override;
      procedure SaveToHTML(Stream: TStream; RVData: TPersistent;
        ItemNo: Integer; const Text, Path: String;
        const imgSavePrefix: String; var imgSaveNo: Integer;
        CurrentFileColor: TColor; SaveOptions: TRVSaveOptions;
        UseCSS: Boolean; Bullets: TRVList); override;
      function AsText(LineWidth: Integer;
        RVData: TPersistent; const Text, Path: String;
        TextOnly,Unicode: Boolean): String; override;
      procedure Inserted(RVData: TObject; ItemNo: Integer); override;        
  end;

implementation

{==============================================================================}
{ TRVLabelItemInfo }
constructor TRVLabelItemInfo.CreateEx(RVData: TPersistent;
  TextStyleNo: Integer; const Text: String);
begin
   inherited Create(RVData);
   StyleNo := rvsLabel;
   VAlign := rvvaBaseLine;
   Self.TextStyleNo := TextStyleNo;
   Self.Text    := Text;
   RVStyle := TCustomRVData(RVData).GetRVStyle;
   UpdateMe;
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.AfterLoading(FileFormat: TRVLoadFormat);
begin
  inherited;
  UpdateMe;
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.UpdateMe;
var DC: HDC;
    Canvas: TCanvas;
    TextMetric: TTextMetric;
begin
   DC := GetDC(0);
   Canvas := TCanvas.Create;
   Canvas.Handle := DC;
   RVStyle.ApplyStyle(Canvas, TextStyleNo, rvbdUnspecified);
   FillChar(TextMetric, sizeof(TextMetric), 0);
   GetTextMetrics(Canvas.Handle, TextMetric);
   Descend := TextMetric.tmDescent;
   Height  := TextMetric.tmHeight;
   Width := Canvas.TextWidth(Text);
   Canvas.Handle := 0;
   Canvas.Free;
   ReleaseDC(0,DC);
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.Assign(Source: TCustomRVItemInfo);
begin
  if Source is TRVLabelItemInfo then begin
    StyleNo := TRVLabelItemInfo(Source).StyleNo;
    Text    := TRVLabelItemInfo(Source).Text;
    ProtectTextStyleNo := TRVLabelItemInfo(Source).ProtectTextStyleNo;
  end;
  inherited;
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.Paint(x, y: Integer; Canvas: TCanvas;
  State: TRVItemDrawStates; Style: TRVStyle; dli: TRVDrawLineInfo);
var TextDrawState: TRVTextDrawStates;
begin
  TextDrawState := [];
  if rvidsSelected in State then
    include(TextDrawState, rvtsSelected);
  if rvidsControlFocused in State then
    include(TextDrawState, rvtsControlFocused);
  if rvidsHover in State then
    include(TextDrawState, rvtsHover);  
  RVStyle.ApplyStyle(Canvas,TextStyleNo,rvbdUnspecified);
  RVStyle.ApplyStyleColor(Canvas,TextStyleNo,TextDrawState);
  Canvas.TextOut(x,y,Text);
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.Print(Canvas: TCanvas; x, y, x2: Integer;
  Preview, Correction: Boolean; const sad: TRVScreenAndDevice;
  RichView: TRVScroller; dli: TRVDrawLineInfo; Part: Integer);
begin
  Paint(x, y, Canvas, [], TCustomRichView(RichView).Style, dli);
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetBoolValueEx(Prop: TRVItemBoolPropertyEx;
  RVStyle: TRVStyle): Boolean;
begin
  case Prop of
    rvbpJump, rvbpAllowsFocus,rvbpXORFocus:
      Result := RVStyle.TextStyles[TextStyleNo].Jump;
    rvbpHotColdJump:
      Result := RVStyle.TextStyles[TextStyleNo].Jump and
                RVStyle.StyleHoverSensitive(StyleNo);
   rvbpPrintToBMP:
     Result := False;
   else
     Result := inherited GetBoolValueEx(Prop, RVStyle);
  end;
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetBoolValue(Prop: TRVItemBoolProperty): Boolean;
begin
  case Prop of
    rvbpAlwaysInText:
      Result := True;
    rvbpDrawingChangesFont:
      Result := True;
    else
      Result := inherited GetBoolValue(Prop);
  end;
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetDescent: Integer;
begin
  Result := Descend;
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetHeight: Integer;
begin
  Result := Height;
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetWidth: Integer;
begin
  Result := Width;
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.GetMinWidth(sad: PRVScreenAndDevice; Canvas: TCanvas; RVData: TPersistent): Integer;
begin
  Result := Width;
  if Sad<>nil then
    Result := MulDiv(Result, sad.ppixDevice, sad.ppixScreen);
end;
{------------------------------------------------------------------------------}
procedure TRVLabelItemInfo.SaveRVF(Stream: TStream; RVData: TPersistent;
  ItemNo, ParaNo: Integer; const Name: String; Part: TRVMultiDrawItemPart;
  ForceSameAsPrev: Boolean);
begin
   // if you want to modify saving/loading, modify
   // 1) second parameter in header - number of additional lines
   // 2) lines after header
   // Do not change other parameters in header
   RVFWriteLine(Stream,
     Format('%d %d %s %d %d %s %s',
            [StyleNo, 2 {Line count after header},
             RVFItemSavePara(ParaNo, TCustomRVData(RVData), ForceSameAsPrev),
             Byte(RVFGetItemOptions(ItemOptions, ForceSameAsPrev)) and RVItemOptionsMask,
             2 {text mode saving},
             RVFSaveTag(rvoTagsArePChars in TCustomRVData(RVData).Options,Tag),
             SaveRVFHeaderTail(RVData)]));
   // lines after header
   RVFWriteLine(Stream, Text);
   RVFWriteLine(Stream, IntToStr(TextStyleNo));
end;
{------------------------------------------------------------------------------}
function TRVLabelItemInfo.ReadRVFLine(const s: String; RVData: TPersistent;
                           ReadType, LineNo, LineCount: Integer;
                           var Name: String;
                           var ReadMode: TRVFReadMode;
                           var ReadState: TRVFReadState): Boolean;
begin
  case LineNo of
    0:
      Text := s;
    1:
      begin
        TextStyleNo := StrToInt(s);
        RVStyle := TCustomRVData(RVData).GetRVStyle;
        ReadState := rstSkip;
      end;
  end;
  Result := True;
end;

procedure TRVLabelItemInfo.MarkStylesInUse(UsedTextStyles, UsedParaStyles,
  UsedListStyles: TRVIntegerList);
begin
  inherited MarkStylesInUse(UsedTextStyles, UsedParaStyles, UsedListStyles);
  UsedTextStyles[TextStyleNo] := 1;
end;

procedure TRVLabelItemInfo.UpdateStyles(TextStylesShift, ParaStylesShift,
  ListStylesShift: TRVIntegerList);
begin
  inherited UpdateStyles(TextStylesShift, ParaStylesShift, ListStylesShift);
  dec(TextStyleNo,TextStylesShift[TextStyleNo]-1);
end;

procedure TRVLabelItemInfo.ApplyStyleConversion(RVData: TPersistent;
  UserData: Integer);
var rve: TCustomRichViewEdit;
begin
  if ProtectTextStyleNo then
    exit;
  rve := TCustomRichViewEdit(TRVEditRVData(RVData).RichView);
  if not Assigned(rve.FCurStyleConversion) then
    exit;
  rve.FCurStyleConversion(rve, TextStyleNo, UserData, True, TextStyleNo);
  UpdateMe;
end;

procedure TRVLabelItemInfo.SaveRTF(Stream: TStream; RVData: TPersistent;
  ItemNo: Integer; const Name: String; TwipsPerPixel: Double;
  Level: Integer; ColorList: TRVColorList; StyleToFont,
  ListOverrideOffsetsList1, ListOverrideOffsetsList2: TRVIntegerList;
  FontTable: TRVList);
begin
  RVFWrite(Stream, RVMakeRTFStr(Text, False));
end;

procedure TRVLabelItemInfo.SaveToHTML(Stream: TStream; RVData: TPersistent;
  ItemNo: Integer; const Text, Path, imgSavePrefix: String;
  var imgSaveNo: Integer; CurrentFileColor: TColor;
  SaveOptions: TRVSaveOptions; UseCSS: Boolean; Bullets: TRVList);
begin
  RVFWrite(Stream, RV_MakeHTMLStr(Self.Text, False, False));
end;

function TRVLabelItemInfo.AsText(LineWidth: Integer; RVData: TPersistent;
  const Text, Path: String; TextOnly, Unicode: Boolean): String;
begin
  Result := Self.Text;
end;

procedure TRVLabelItemInfo.Inserted(RVData: TObject; ItemNo: Integer);
begin
  if RVData<>nil then
    RVStyle := TCustomRVData(RVData).GetRVStyle;
end;

procedure TRVLabelItemInfo.Execute(RVData: TPersistent);
begin
  if RVData is TCustomRVFormattedData then begin
    if GetBoolValueEx(rvbpJump, TCustomRVData(RVData).GetRVStyle) then
      TCustomRVFormattedData(RVData).DoJump(JumpID+
          TCustomRVFormattedData(RVData).FirstJumpNo)
  end;
end;

initialization

  RegisterRichViewItemClass(rvsLabel, TRVLabelItemInfo);

end.
