/* ****************************************************************
  Should be able to sort any size line based character text file.

  Similar to DOS SORT utility.

  Call line has input-file & output-file names. Output file is
  optional.

  If no output file is specified, one is created from the
  input file name by adding or overwriting the extension with
  ".SRT"

  Parameters -R  {reverse order}
              reverse order can apply to -# parameters where
              the sort order has not been specified.

             -D  {remove duplicates}
              duplicates means whole lines equal- not just
              sort fields. If partial line sort fields are
              used (-#) then some duplicate lines may be
              missed.

             -Q quiet mode. No output.

             -#nnn {where nnn = start character to sort on}
              First character of each line is 1.
              This can optionally be extended to number of
              characters to sort on, and "A" or "D" for
              Ascending or Descending sort order.
              e.g. -#12/3/D or -#55/10 or -#200 or all together.
              More than one (maximum 5 sort criteria allowed)
              A check is made for overlapping sort fields.

              The sort is made in the sequence in which the sort
              parameters are input (if multiple -# input).

   Upper or lower case letters can be used for parameters.

   Examples:-

   BSORT -r  ADDRESS.TXT

   BSORT ADDRESS.TXT   ADDRESS.NEW -D

   BSORT MYTEXT -#1/5/A

   BSORT MYTEXT -#12/3/A  -#15 -r -D

  Temporary merge files may be created and deleted by this program
  if the internal sort area is not large enougth. The names of these
  merge files will be of the form "SRTEMPnn.TMP" where nn = 01, 02 etc.

  Author: Paul Bennett   Date: August 12, 2001

  Copyright Paul Bennett & Barton Systems Limited.

  August 31, 2001

 *************************************************************** */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
//      work with 32 & 16 bit compilers.
#if UINT_MAX == 0xFFFFU
//    16 bit case.
typedef long int i32;
//     maximum memory allowed for sort.
#define MAX_CORE 30000
#else
//    32 bit case
typedef int i32;
//     maximum memory allowed for sort.
#define MAX_CORE 10000000
#endif
char tmp_name[13] = "SRTEMP00.TMP";
//     maximum length of an input line
#define MAXLIN 530
char in_line [MAXLIN + 1];
char dup_chk [MAXLIN + 1];
i32 line_count = 0;
char c_store [MAX_CORE];
int point;
//     maximum number of entries in sort.
#define MAX_ENTRY MAX_CORE/20
int index_0 [MAX_ENTRY];
int posit = 0;
//    maximum number of sort rules allowed.
#define MAX_RULE 5
int start_char   [MAX_RULE];
int number_chars [MAX_RULE];
char a_or_d      [MAX_RULE];
//   maximum character offset value
#define MAX_OFF 250
//   maximum sort field size
#define MAX_FIELD 30
int  num_rules;
//    number of primary merge files.
#define P_MRG 4
//    number of secondary merge files.
#define S_MRG 4
#define MAX_FILE (P_MRG + S_MRG + 1)
FILE *fi, *fo [MAX_FILE];
i32 num_rec   [MAX_FILE];
char num_merge, mrg_lst [P_MRG+1];
char max_active, mrg_order [P_MRG+1];
char mrg_rec [P_MRG+1][MAXLIN];
i32  shad_count [P_MRG+1];
//    flags to indicate which files have been used.
char f_flag [MAX_FILE];
char num_free,r_value,d_value,q_value;
i32  num_delete;
int getline (void);
void sort_it (char flush);
void open_file (char num,char  a_type);
//   comparison functions.
int strcmp_n (const int * abc,const int * def);
int  cmp2 (char * s1, char * s2);
//   other functions.
void send_out(char ttt);
void merge_it (void);
void get_record (const char pp);
void delete_file(char zz);
void conv_decimal (i32 in_num, char out_s[]);
void help_out (void);
int main (int argc, char *argv[]){
   //    loop counters.
   int i,j;
   int len, offset, s_point;
   int min_len, beg_pnt, end_pnt, s2, e2;
   int num_ch, levl;
   char ad;
   char chs, in_file, out_file, d_sort;
   //   size of maximum input/output file name
#define MAX_NAME 160
   //   size of maximum start position
#define MAX_START 300
   char file_name      [MAX_NAME+1];
   char file_name_save [MAX_NAME+1];
   char line_string [15];
   unsigned int it;
   //    initialise
   r_value  = 0;
   d_value  = 0;
   q_value  = 0;
   in_file  = 0;
   out_file = 0;
   offset   = 0;
   num_rules   = 0;
   num_delete = 0;
   //    set output record counts
   for (i=0; i < MAX_FILE; i++){
      num_rec[i] = 0;
      f_flag [i] = 0;}
   num_free = MAX_FILE;
   if (argc < 2){
      help_out();
      return 0;}
   for (i=1; i < argc; i++){
      //    get the parameters.
      if (strlen(argv[i]) > MAX_NAME){
         fprintf(stderr,"File name too long\n");
         return 1;}
      strcpy(file_name, argv[i]);
      //     check type of input
      if (file_name[0] == '-') {
         //         parameter input
         if (file_name[1] == '#'){
            if (num_rules == MAX_RULE){
               fprintf(stderr,"Too many sort rules\n");
               return 1;}
            len = strlen(file_name);
            offset = 0;
            num_ch = 0;
            levl   = 0;
            ad     = ' ';
            for (j=2; j < len; j++){
               chs = file_name[j];
               if (isdigit(chs)) {
                  switch (levl){
                  case 0:
                     {
                        offset = offset*10 + chs - '0';
                        break;
                     }
                  case  1:
                     {
                        num_ch = num_ch * 10 + chs - '0';
                        break;
                     }
                  default:
                     {
                        fprintf(stderr,"Invalid format\n");
                        return 1;}
                  }
                  if (offset > MAX_OFF ||
                     offset == 0      ||
                     num_ch > MAX_FIELD){
                     fprintf(stderr,"Parameter limits\n");
                     return 1;}
                  continue;
               }
               //   non-digit processing.
               if (chs == '/'){
                  levl += 1;
                  continue;}
               if (levl > 0){
                  chs = toupper(chs);
                  if (chs == 'A' || chs == 'D'){
                     ad = chs;
                     if (j +1 != len){
                        //  should be end.
                        fprintf(stderr,"Excess characters\n");
                        return 1;}
                     continue;
                  }
               }
               fprintf (stderr, "Invalid offset input\n");
               return 1;
            }
            //         store values
            start_char   [num_rules] = offset;
            number_chars [num_rules] = num_ch;
            a_or_d       [num_rules] = ad;
            num_rules++;
         }
         else {
            if(strlen(file_name) > 2){
bad_param:
               fprintf(stderr,"Invalid parameter entered\n");
               return 1;}
            chs = file_name[1];
            chs = toupper(chs);
            if (!(chs == 'D'  || chs == 'R'|| chs == 'Q'))goto bad_param;
            if (chs == 'D'){
               if (d_value != 0) goto bad_param;
               d_value = 'D';
               continue;
            }
            if (chs == 'R'){
               if (r_value != 0) goto bad_param;
               r_value = 'R';
               continue;
            }
            if (chs == 'Q'){
               if (q_value != 0) goto bad_param;
               q_value = 'Q';
               continue;
            }
         }
         continue;
      }
      //         assume entry is file name.
      if (in_file != 1) {
         in_file = 1;
         if ((fi = fopen (file_name,"rt")) == NULL){
            fprintf(stderr,"Unable to open file %s\n",file_name);
            return 1;}
         //      keep name as template for output name.
         strcpy (file_name_save, file_name);
         continue;
      }
      if (out_file == 1){
         fprintf (stderr, "Too many files specified\n");
         return 1; }
open_output:
      if ((fo[0] = fopen (file_name,"wt")) == NULL){
         fprintf (stderr, "Unable to open file %s\n", file_name);
         return 1;}
      out_file = 1;
      continue;
   }
   //    a few cross-checks
   if (in_file == 0) {
      fprintf (stderr,"No input file specified\n");
      return 1;}
   //    if no output name, use same name as input with extension .SRT
   if (out_file == 0) {
      //     test for extension.
      len = strlen (file_name_save);
      s_point = len;
      for (j = len-1,i = 0; j >= 0 && i < 4; j--, i++){
         chs =  file_name_save[j];
         if (chs == '.'){
            s_point = j;
            break;}
      }
      if (j+3 > MAX_NAME){
         fprintf (stderr,"File name too long\n");
         return 1;}
      //    add on "SRT" extension.
      strcpy (&file_name_save[s_point], ".SRT");
      if ((fo[0] = fopen (file_name_save,"wt")) == NULL){
         fprintf(stderr,"Unable to open file %s\n",file_name_save);
         return 1;}
   }
   //    if any sort rules input check for overlap.
   min_len = 0;
   for (i=0; i < num_rules; i++){
      beg_pnt = start_char [i] ;
      end_pnt = start_char [i] + number_chars [i];
      if (number_chars [i] != 0) end_pnt -= 1;
      if (end_pnt > min_len) min_len = end_pnt;
      if (number_chars [i] == 0) end_pnt = MAXLIN;
      for (j=i+1; j < num_rules; j++){
         s2 = start_char[j];
         e2 = start_char[j] + number_chars [j];
         if (number_chars [j] != 0)
            e2 -= 1;
         else
            e2 = MAXLIN;
         if((beg_pnt >= s2 && beg_pnt <= e2) ||
            (end_pnt >= s2 && end_pnt <= e2) ||
            (beg_pnt <  s2 && end_pnt >  e2)) {
            fprintf(stderr,"Overlapping sort parameters\n");
            return 1;
         }
      }
   }
   //  put in default order.
   d_sort = 'A';
   if (r_value == 'R') d_sort = 'D';
   for (i = 0; i < num_rules; i++){
      if (a_or_d [i] == ' ') a_or_d[i] = d_sort;}
   //        now read all string lines in.
   point = 0;
   posit = 0;
   while ((len = getline())!=0){
      line_count++;
      //   too small for sort parameters?
      if (len-1 < min_len){
         fprintf(stderr,
            "Line %d too small for sorting\n",
            line_count);
         return 1;}
      //        put it in the heap if room
      if ((len+1+point > MAX_CORE) ||(posit >= MAX_ENTRY))
         sort_it(0);
      index_0 [posit++] = point;
      strcpy (&c_store[point], in_line);
      point = point + len + 1;
   }
   //    close input file.
   if (fclose (fi) == EOF){
      fprintf(stderr,"Error in close of input file\n");
      return 1;}
   sort_it(1);
   if (q_value) goto quiet;
   //   16 bit compiler cannot output long integer.
   //   convert to a string.
   conv_decimal (line_count, line_string);
   printf ("Number of input lines = %s\n",line_string);
   if (d_value != 0){
      conv_decimal (num_delete, line_string);
      printf ("Number of lines deleted = %s\n", line_string);
   }
quiet:
   return 0;
}
// =================================================
//  Getline - returns line and number of characters.
//  if EOF returns zero as number of characters.
// =================================================
int getline(void){
   if (fgets (in_line, MAXLIN, fi) == NULL)
      return 0;
   else
      return strlen(in_line);
}
// =====================================================
//  sort_it. If the last sort, then the parameter
//  should be 1, otherwise 0.
// =====================================================
void sort_it (char flush){
   int i, j;
   i32  min_val;
   qsort (index_0, posit, sizeof(int),
      (int (*)(const void *,const void *)) strcmp_n);
   if (flush == 0){
      //   if any primary merge files free use them to output to.
      for (i=1; i < P_MRG+1; i++){
         if (num_rec [i] == 0){
            send_out(i);
            return;
         }
      }
      //         no free primary merge files.
      //         merge existing ones, into secondary file.

      for (j = P_MRG+1; j < MAX_FILE; j++){
         if (num_rec[j] == 0) goto merge_op;
      }
      fprintf (stderr, "System fault- no free 2nd. merge files\n");
      exit (1);
merge_op:
      //       set up merge list.
      num_merge = P_MRG + 1;
      mrg_lst[0] = j;
      for (i=1; i <= P_MRG; i++){mrg_lst[i] = i;}
      merge_it();
      //  check if only one secondary merge slot free.
      //  if so merge two smallest to give more room.
      mrg_lst[0] = 0;
      mrg_lst[1] = 0;
      mrg_lst[2] = 0;
      num_merge  = 3;
second_pass:
      min_val = 0;
      j = 0;
      for (i = P_MRG + 1; i < MAX_FILE; i++){
         if (i == mrg_lst[2]) continue;
         if (num_rec[i] == 0){
            j++;
            if (mrg_lst[0] == 0) mrg_lst[0] = i;
            continue;}
         if (min_val == 0) {
            min_val = num_rec [i];
            mrg_lst[1] = i;
            continue; }
         if (num_rec[i] <= min_val){
            mrg_lst[2] = mrg_lst[1];
            mrg_lst[1] = i;
            min_val = num_rec [i];
            continue;}
      }
      if (j == 1){
         //  only one free slot. Do a merge.
         if (mrg_lst[1] == 0){
            fprintf (stderr, "System error - no merge files\n");
            exit (8);}
         if (mrg_lst[2] == 0){
            // must have picked up smallest file first time through.
            mrg_lst[2] = mrg_lst[1];
            mrg_lst[1] = 0;
            goto second_pass;}
         merge_it();
         return ;
      }
   }
   //
   if (flush == 1){
      //    end- if no output files active, just send sort to main
      //    output file.
dim_merge:
      j = 0;
      for (i=0; i < MAX_FILE; i++){
         if (num_rec[i] != 0){
            j += 1;
         }
      }
      if (j == 0){
         //       output sort to main file.
         send_out(0);
         return;
      }
      //      case when some files still need merging.
      if (j <= P_MRG){
         //      set up list.
         mrg_lst[0] = 0;
         num_merge  = 1;
         for (i=0; i< MAX_FILE; i++){
            if (num_rec[i] != 0){
               mrg_lst[num_merge++] = i;}
         }
         merge_it();
         return;
      }
      else{
         //  more than maximum temporary files outstanding.
         //  merge two smallest one. Repeat until P_MRG left.
         mrg_lst[0] = 0;
         mrg_lst[1] = 0;
         mrg_lst[2] = 0;
         num_merge   = 3;
another_try:
         min_val = 0;
         // look for empty entry & two smallest entries.
         for (i=1; i < MAX_FILE; i++){
            if (mrg_lst[2] == i) continue;
            if (num_rec[i] == 0){
               mrg_lst[0] = i;
               continue;}
            //  first file with any entries in it.
            if (mrg_lst[1] == 0){
               mrg_lst[1] = i;
               min_val = num_rec[i];
               continue;}
            //  test for a smaller one.
            if (num_rec[i] <= min_val){
               // new minimum
               mrg_lst[2] = mrg_lst[1];
               mrg_lst[1] = i;
               min_val = num_rec[i];
               continue;}
         }
         if (mrg_lst[2] == 0){
            // smallest value picked if on first pass.
            mrg_lst[2] = mrg_lst[1];
            mrg_lst[1] = 0;
            goto another_try;}
         //  test all three values are non-zero.
         for (i=0;i<3;i++){
            if (mrg_lst[i] == 0){
               fprintf(stderr,"System error, merging files\n");
               exit (12);}
         }
         merge_it();
         //   repeat process until maximum merge files left.
         goto dim_merge;
      }
   }
   return;
}
void send_out(char f_num){
   int k;
   int local_dels;
   local_dels = 0;
   if (f_num != 0){
      open_file (f_num,'w');}
   for (k=0; k< posit; k++){
      if (d_value != 0 && k > 0)
         if(strcmp(&c_store[index_0[k]],dup_chk)==0){
            num_delete += 1;
            local_dels++;
            continue;}
      if (fputs(&c_store[index_0[k]], fo[f_num])==EOF){
         fprintf (stderr,"Error outputing from sort\n");
         exit (3);}
      if (d_value != 0) strcpy (dup_chk, &c_store[index_0[k]]);
   }
   if(fclose (fo[f_num]) == EOF){
      fprintf(stderr, "Error in close of final file\n");
      exit (9);}
   num_rec [f_num] = posit - local_dels;
   posit = 0;
   point = 0;
   return;
}
void open_file (char num, char a_type){
   char qq[3] = "wt" ;
   if (num == 0) return;
   if (num >= MAX_FILE){
      fprintf(stderr,"System error-number too big\n");
      exit (4);}
   //          create the name
   tmp_name[6] = num/10 + '0';
   tmp_name[7] = num%10 + '0';
   qq[0] = a_type;
   if ((fo[num] = fopen (tmp_name,qq))==0){
      fprintf(stderr,"Unable to open file\n");
      exit (5);}
   //   in use flag.
   f_flag [num] = 1;
   return;
}
// ==========================================================
//  strcmp_n  (normal string compare function.
// ==========================================================
int strcmp_n (const int * aa,const int * bb){
   int p,q,val,i,r,s;
   int offs, numc;
   p = *aa;
   q = *bb;
   if (num_rules == 0){
      if (r_value == 0){
         return  strcmp (&c_store[p], &c_store[q]);}
      else{
         return  strcmp (&c_store[q],&c_store[p]);}
   }
   //    more complicated case of sorting on partial fields.
   for (i=0;i < num_rules; i++){
      r = p;
      s = q;
      offs = start_char[i] - 1;
      numc = number_chars [i];
      if (a_or_d[i] == 'D'){
         //           reverse searching.
         r = q;
         s = p;
      }
      if (numc != 0)
         val = strncmp (&c_store[r+offs], &c_store[s+offs], numc);
      else
         val = strcmp (&c_store[r+offs], &c_store[s+offs]);
      if (val != 0) break;
   }
   return val;
}
// ****************************************************************
//   merge- list of files to merge is in "mrg_lst"
//   number of entries is in "num_merge".
//   entry zero, is output file.
//   the in core sorted entries are also merged in.
// ****************************************************************
void merge_it (void){
   i32 count_out;
   int i, j;
   char fn;
   //   set up shadow counts
   //   entry zero is core stack.
   shad_count [0] = posit;
   for (i=1; i < num_merge; i++){
      j = mrg_lst[i];
      shad_count [i] = num_rec [j];
   }
   count_out = 0;
   //       first open the files.
   open_file (mrg_lst[0], 'w');
   for (i=1; i< num_merge; i++){
      open_file (mrg_lst[i], 'r');
   }
   max_active = 0;
   //    n.b. position zero is for sorted list in core.
   //    read in a "record" from each file & sort in order.
   for (i=0; i < num_merge; i++){
      get_record (i);}
next_pass:
   if (max_active == 0) goto finished;
   //      select top record.
   j = mrg_order[0];
   //      check for duplicates.
   if (d_value != 0 && count_out != 0){
      if (strcmp(mrg_rec[j], dup_chk) == 0){
         num_delete++;
         goto skip_output;
      }
   }
   if (fputs (mrg_rec[j], fo[mrg_lst[0]]) == EOF){
      fprintf (stderr,"Error in outputting a string\n");
      exit (6);}
   count_out++;
   if (d_value != 0) strcpy (dup_chk, mrg_rec[j]);
skip_output:
   //     close list down by one.
   for (i=1; i < max_active; i++){
      mrg_order[i-1] = mrg_order[i];}
   max_active--;
   //    get a replacement record (if any) for one output,
   //    and place it in correct position in list.
   get_record (j);
   goto  next_pass;
finished:
   //     close output & input files.
   for (i=0; i < num_merge; i++){
      if (i == 0)
         num_rec[mrg_lst[0]] = count_out;
      else
         num_rec[mrg_lst[i]] = 0;
      if (fclose (fo[mrg_lst[i]]) == EOF){
         fprintf(stderr,"Error in close of merge file\n");
         exit (7);}
      if (i != 0){
         fn = mrg_lst [i];
         //     delete file.
         delete_file (fn);}
   }
   //         clear memory stack
   point = 0;
   posit = 0;
   return;
}
// ***************************************************************
//  get_record - gets next "record" from input stream
//  and finds its correct position with respect to other
//  records. n.b. position zero is core memory.
// ***************************************************************
void get_record (const char pp){
   int i, j, pos, nxt;
   char lowv, highv, posn, midt;
   if (shad_count[pp] == 0){
      //      all entries used up.
      return; }
   if (pp == 0){
      //     get "record" from memory stack.
      nxt = posit - shad_count[0];
      strcpy (mrg_rec[0], &c_store [index_0 [nxt] ] );
   }
   else{
      //  normal case- record from a file.
      j  = mrg_lst [pp];
      if (fgets (mrg_rec[pp], MAXLIN, fo[j]) == NULL){
         fprintf (stderr, "Error in record fetch\n");
         exit (8);
      }
   }
   shad_count[pp]--;
   //   find correct position to place "record".
   if (max_active == 0){
      //        first time- nothing in list.
      max_active = 1;
      mrg_order[0] = pp;
      return;
   }
   //   find correct order.
   lowv  = 0;
   highv = max_active - 1;
split_again:
   if ((highv - lowv) > 2){
      midt = ((highv - lowv) / 2) + lowv;
      //      compare new value to middle value.
      if ((pos = cmp2 (
         mrg_rec [pp],
         mrg_rec [mrg_order [midt] ] ) ) == 0){
         //  equal case.
         posn = midt;
         goto insert;}
      if (pos < 0)
         //   new string is lower in sort than middle value.
         highv = midt;
      else
         //   higher position.
         lowv = midt+1;
      goto split_again;
   }
   //   now do a serial search for correct position.
   for (j = lowv; j < highv + 1; j++){
      if ((pos =
         cmp2(mrg_rec[pp], mrg_rec[mrg_order[j]])) <= 0){
         posn = j;
         goto insert;
      }
   }
   posn = highv + 1;
insert:
   //      move earlier items down (if any)
   for (i = max_active - 1; i >= posn; i--){
      mrg_order [i+1] = mrg_order [i];
   }
   mrg_order[posn] = pp;
   max_active += 1;
   return;
}
// *******************************************************
//  cmp2 - comparison routine.
//  returns -1 if s1 < s2
//           0    s1 = s2
//          +1    s1 > s2
// *******************************************************
int  cmp2 (char * s1, char * s2){
   char * st1, * st2;
   int val, i, numc, offs;
   if (num_rules == 0){
      if (r_value == 0){
         return  strcmp (s1, s2);}
      else{
         //  reverse order
         return  strcmp (s2, s1);}
   }
   //    more complicated case of sorting on partial fields.
   for (i=0; i < num_rules; i++){
      st1 = s1;
      st2 = s2;
      offs = start_char[i] - 1;
      numc = number_chars [i];
      if (a_or_d [i] == 'D'){
         //           reverse searching.
         st1 = s2;
         st2 = s1;
      }
      if (numc != 0)
         val = strncmp (st1 + offs, st2 + offs, numc);
      else
         val = strcmp (st1 + offs, st2 + offs);
      if (val != 0) break;
   }
   return val;
}
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//  delete_file to delete a temp. file.
//  File number enetered as a parameter.
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void delete_file (char zz){
   char ab;
   ab = zz/10 + '0';
   tmp_name [6] = ab;
   ab = zz%10 + '0';
   tmp_name [7] = ab;
   if (remove (tmp_name) != 0){
      fprintf(stderr,"Unable to delete file %s\n",tmp_name);
      exit (10);}
   f_flag [zz] = 0;
   return;
}
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//   convert a 32 bit integer to a character string.
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
void conv_decimal (i32 in_num, char out_s[]){
   char temp_string[14];
   char c_count, dig, o_cnt;
   i32  t_num;
   int i;
   t_num = in_num;
   c_count = 0;
   do {
      dig = t_num%10 + '0';
      t_num = t_num/10;
      temp_string [c_count++] = dig;
   } while (t_num != 0);
   //  now output digits in reverse sequence.
   o_cnt = 0;
   for (i = c_count -1; i >= 0; i--){
      dig = temp_string [i];
      out_s [o_cnt++] = dig;
      if (i != 0 && i%3 == 0) out_s [o_cnt++] = ',';
   }
   //  terminate string.
   out_s [o_cnt] = 0;
   return;
}
void help_out (void){
   printf ("BSORT -Sort test lines\n\n");
   printf ("Input format:-\n");
   printf ("BSORT file_name <sorted_file name> <options>\n\n");
   printf
   ("If <sorted_file name> is missing, use file_name with ext .SRT\n");
   printf ("<options> are:-\n\n");
   printf ("-r     Reverse sort\n");
   printf ("-d     Delete duplicate lines\n");
   printf ("-q     Quiet mode. No output unless an error\n");
   printf ("-#nnn/<mm>/<A/D>   Sort on a field within line\n");
   printf ("nnn    is character to start sort on. 1 = first position\n");
   printf ("mm     is optional number of characters\n");
   printf ("       {if omitted sort field is until end of line}\n");
   printf ("A       Ascending sequence {default}\n");
   printf ("D       Descending sequence {same as -r}\n\n");
   printf ("        Multiple sort fields can be used\n");
   printf ("Examples:-\n\n");
   printf ("BSORT MYFILE MYFILE.NEW -#1/5/A -#7/2 -#10\n");
   printf ("BSORT MYFILE -r -d\n\n");
   printf
   ("   Copyright  Paul Bennett & Barton Systems Limited.\n");
   printf
   ("   August 31, 2001\n");
   return;
}
