// HexFileList: override of MFC CRecentFileList class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"

#include "HexEdit.h"
#include "HexFileList.h"
#include "HexEditView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// The following are not in a public header
extern BOOL AFXAPI AfxFullPath(LPTSTR lpszPathOut, LPCTSTR lpszFileIn);
extern BOOL AFXAPI AfxComparePath(LPCTSTR lpszPath1, LPCTSTR lpszPath2);

CHexFileList::CHexFileList(UINT nStart, LPCTSTR lpszSection,
                LPCTSTR lpszEntryFormat, int nSize,
                int nMaxDispLen /*= AFX_ABBREV_FILENAME_LEN*/)
        : CRecentFileList(nStart, lpszSection, lpszEntryFormat, nSize, nMaxDispLen)
{
    autofit_ = new BOOL[nSize];
    dec_addr_ = new BOOL[nSize];
    display_char_ = new BOOL[nSize];
    ebcdic_ = new BOOL[nSize];
    oem_ = new BOOL[nSize];
    control_ = new BOOL[nSize];
    graphic_ = new BOOL[nSize];

    ppart_ = new (std::vector<partn> *[nSize]);
    for (int ii = 0; ii < nSize; ++ii)
        ppart_[ii] = NULL;

    allow_mods_ = new BOOL[nSize];
    insert_ = new BOOL[nSize];

    rowsize_ = new int[nSize];
    group_by_ = new int[nSize];
    offset_ = new int[nSize];

    start_addr_ = new long[nSize];
    end_addr_= new long[nSize];
    mark_ = new long[nSize];
    scroll_ = new long[nSize];
    edit_char_ = new BOOL[nSize];

    face_ = new CString[nSize];         // Font face and height
    height_ = new long[nSize];
    oem_face_ = new CString[nSize];     // OEM/IBM font face and height
    oem_height_ = new long[nSize];

    cmd_ = new int[nSize];
    top_ = new int[nSize];
    left_ = new int[nSize];
    bottom_ = new int[nSize];
    right_ = new int[nSize];
}

CHexFileList::~CHexFileList()
{
    delete[] autofit_;
    delete[] dec_addr_;
    delete[] display_char_;
    delete[] ebcdic_;
    delete[] oem_;
    delete[] control_;
    delete[] graphic_;

    for (int ii = 0; ii < m_nSize; ++ii)
        if (ppart_[ii] != NULL)
            delete ppart_[ii];
    delete[] ppart_;

    delete[] allow_mods_;
    delete[] insert_;

    delete[] rowsize_;
    delete[] group_by_;
    delete[] offset_;

    delete[] start_addr_;
    delete[] end_addr_;
    delete[] mark_;
    delete[] scroll_;
    delete[] edit_char_;

    delete[] face_;
    delete[] height_;
    delete[] oem_face_;
    delete[] oem_height_;

    delete[] cmd_;
    delete[] top_;
    delete[] left_;
    delete[] bottom_;
    delete[] right_;
}

void CHexFileList::Add(LPCTSTR lpszPathName)
{
//    CRecentFileList::Add(lpszPathName);

    // Saved option values for the new top file name
    BOOL autofit;
    BOOL dec_addr;
    BOOL display_char;
    BOOL ebcdic, oem;
    BOOL control;
    BOOL graphic;

    std::vector<partn> *ppart;

    BOOL allow_mods;
    BOOL insert;

    int rowsize;
    int group_by;
    int offset;

    long start_addr, end_addr, mark, scroll;
    BOOL edit_char;

    CString face;                       // font face and height
    long height;
    CString oem_face;                   // oem/ibm font face and height
    long oem_height;

    int cmd;
    int top, left, bottom, right;

    ASSERT(m_arrNames != NULL);
    ASSERT(lpszPathName != NULL);
    ASSERT(AfxIsValidString(lpszPathName));

    // Get fully qualified path name
    TCHAR full[_MAX_PATH];
    AfxFullPath(full, lpszPathName);

    // Search for an existing entry for this file name
    for (int ii = 0; ii < m_nSize; ++ii)
    {
        if (AfxComparePath(m_arrNames[ii], full))
        {
            // Found matching entry so save options for it
            autofit = autofit_[ii];
            dec_addr = dec_addr_[ii];
            display_char = display_char_[ii];
            ebcdic = ebcdic_[ii];
            oem = oem_[ii];
            graphic = graphic_[ii];
            control = control_[ii];

            ppart = ppart_[ii];

            allow_mods = allow_mods_[ii];
            insert = insert_[ii];

            rowsize = rowsize_[ii];
            group_by = group_by_[ii];
            offset = offset_[ii];

            start_addr = start_addr_[ii];
            end_addr = end_addr_[ii];
            mark = mark_[ii];
            scroll = scroll_[ii];
            edit_char = edit_char_[ii];

            face = face_[ii];
            height = height_[ii];
            oem_face = oem_face_[ii];
            oem_height = oem_height_[ii];

            cmd = cmd_[ii];
            top = top_[ii];
            left = left_[ii];
            bottom = bottom_[ii];
            right = right_[ii];
            break;
        }
    }

    // If entry was NOT found use default options
    if (ii == m_nSize)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // Entry not found so init options to defaults
        autofit = aa->open_autofit_;
        dec_addr = aa->open_dec_addr_;
        display_char = aa->open_display_char_;
        ebcdic = aa->open_ebcdic_;
        control = aa->open_control_;
        graphic = aa->open_graphic_;
        oem = aa->open_oem_;

        face = "Courier";
        height = 16;
        oem_face = "Terminal";
        oem_height = 18;

        ppart = NULL;                   // Indicates default colours

        allow_mods = aa->open_allow_mods_;
        insert = aa->open_insert_;
        edit_char = 0;                  // default to hex area

        rowsize = aa->open_rowsize_;
        group_by = aa->open_group_by_;
        offset = aa->open_offset_;

        start_addr = end_addr = mark = scroll = 0;
        top = left = bottom = right = -30000;

        if (aa->open_max_)
            cmd = SW_SHOWMAXIMIZED;
        else
            cmd = SW_SHOW;

        // Start move up from 2nd last byte
        ii--;
    }

    // Move entries up to make room for the new top one
    ASSERT(ii < m_nSize);
    for (; ii > 0; ii--)
    {
        m_arrNames[ii] = m_arrNames[ii-1];

        autofit_[ii] = autofit_[ii-1];
        dec_addr_[ii] = dec_addr_[ii-1];
        display_char_[ii] = display_char_[ii-1];
        ebcdic_[ii] = ebcdic_[ii-1];
        oem_[ii] = oem_[ii-1];
        graphic_[ii] = graphic_[ii-1];
        control_[ii] = control_[ii-1];

        ppart_[ii] = ppart_[ii-1];

        allow_mods_[ii] = allow_mods_[ii-1];
        insert_[ii] = insert_[ii-1];

        rowsize_[ii] = rowsize_[ii-1];
        group_by_[ii] = group_by_[ii-1];
        offset_[ii] = offset_[ii-1];

        start_addr_[ii] = start_addr_[ii-1];
        end_addr_[ii] = end_addr_[ii-1];
        mark_[ii] = mark_[ii-1];
        scroll_[ii] = scroll_[ii-1];
        edit_char_[ii] = edit_char_[ii-1];

        face_[ii] = face_[ii-1];
        height_[ii] = height_[ii-1];
        oem_face_[ii] = oem_face_[ii-1];
        oem_height_[ii] = oem_height_[ii-1];

        cmd_[ii] = cmd_[ii-1];
        top_[ii] = top_[ii-1];
        left_[ii] = left_[ii-1];
        bottom_[ii] = bottom_[ii-1];
        right_[ii] = right_[ii-1];
    }
    ASSERT(ii == 0);

    // Put the newly opened file at the top
    m_arrNames[0] = lpszPathName;

    autofit_[0] = autofit;
    dec_addr_[0] = dec_addr;
    display_char_[0] = display_char;
    ebcdic_[0] = ebcdic;
    oem_[0] = oem;
    graphic_[0] = graphic;
    control_[0] = control;

    ppart_[0] = ppart;

    allow_mods_[0] = allow_mods;
    insert_[0] = insert;

    rowsize_[0] = rowsize;
    group_by_[0] = group_by;
    offset_[0] = offset;

    start_addr_[0] = start_addr;
    end_addr_[0] = end_addr;
    mark_[0] = mark;
    scroll_[0] = scroll;
    edit_char_[0] = edit_char;

    face_[0] = face;
    height_[0] = height;
    oem_face_[0] = oem_face;
    oem_height_[0] = oem_height;

    cmd_[0] = cmd;
    top_[0] = top;
    left_[0] = left;
    bottom_[0] = bottom;
    right_[0] = right;
}

void CHexFileList::Remove(int nIndex)
{
//    CRecentFileList::Remove(nIndex);
    ASSERT(nIndex >= 0 && nIndex < m_nSize);

    for (int ii = nIndex; ii < m_nSize-1; ++ii)
    {
        m_arrNames[ii] = m_arrNames[ii+1];
        autofit_[ii] = autofit_[ii+1];
        dec_addr_[ii] = dec_addr_[ii+1];
        display_char_[ii] = display_char_[ii+1];
        control_[ii] = control_[ii+1];
        oem_[ii] = oem_[ii+1];
        graphic_[ii] = graphic_[ii+1];
        ebcdic_[ii] = ebcdic_[ii+1];

        ppart_[ii] = ppart_[ii+1];

        allow_mods_[ii] = allow_mods_[ii+1];
        insert_[ii] = insert_[ii+1];

        rowsize_[ii] = rowsize_[ii+1];
        group_by_[ii] = group_by_[ii+1];
        offset_[ii] = offset_[ii+1];

        start_addr_[ii] = start_addr_[ii+1];
        end_addr_[ii] = end_addr_[ii+1];
        mark_[ii] = mark_[ii+1];
        scroll_[ii] = scroll_[ii+1];
        edit_char_[ii] = edit_char_[ii+1];

        face_[ii] = face_[ii+1];
        height_[ii] = height_[ii+1];
        oem_face_[ii] = oem_face_[ii+1];
        oem_height_[ii] = oem_height_[ii+1];

        cmd_[ii] = cmd_[ii+1];
        top_[ii] = top_[ii+1];
        left_[ii] = left_[ii+1];
        bottom_[ii] = bottom_[ii+1];
        right_[ii] = right_[ii+1];
    }

    ASSERT(ii == m_nSize-1);
    m_arrNames[ii].Empty();
}

void CHexFileList::ReadList()
{
    CRecentFileList::ReadList();

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Write a set of options for each file of the MRU list
    for (int ii = 0; ii < m_nSize; ++ii)
    {
        CString fnum;
        fnum.Format("File%d", ii+1);

        // Read the option values, defaulting to the global (aa->open_*) values
        autofit_[ii] = aa->GetProfileInt(fnum, "AutoFit", aa->open_autofit_) ? TRUE : FALSE;
        dec_addr_[ii] = aa->GetProfileInt(fnum, "DecimalAddresses", aa->open_dec_addr_) ? TRUE : FALSE;
        display_char_[ii] = aa->GetProfileInt(fnum, "DisplayChar", aa->open_display_char_) ? TRUE : FALSE;
        ebcdic_[ii] = aa->GetProfileInt(fnum, "EBCDIC", aa->open_ebcdic_) ? TRUE : FALSE;
        control_[ii] = aa->GetProfileInt(fnum, "ControlChars", aa->open_control_) ? TRUE : FALSE;
        graphic_[ii] = aa->GetProfileInt(fnum, "GraphicChars", aa->open_graphic_) ? TRUE : FALSE;
        oem_[ii] = aa->GetProfileInt(fnum, "OemChars", aa->open_oem_) ? TRUE : FALSE;

        // Get the colour partitions
        ppart_[ii] = new vector<partn>;         // Allocate the vector
        ppart_[ii]->clear();
        for (int pi = 1;  ; ++pi)
        {
            // Fill the vector
            partn pp;
            CString f1, f2, f3;
            f1.Format("Name%d", pi);
            f2.Format("Colour%d", pi);
            f3.Format("Range%d", pi);

            if (!aa->GetColours(fnum, f1, f2, f3, pp))
                break;

            // Add the record (name, colour, range) to end of vector
            ppart_[ii]->push_back(pp);
        }
        if (pi == 1)
        {
            // None found so delete the vector
            delete ppart_[ii];
            ppart_[ii] = NULL;                  // Flag unused so its not deleted again
        }

        allow_mods_[ii] = aa->GetProfileInt(fnum, "AllowMods", aa->open_allow_mods_) ? TRUE : FALSE;
        insert_[ii] = aa->GetProfileInt(fnum, "Insert", aa->open_insert_) ? TRUE : FALSE;

        // Get and validate column info
        rowsize_[ii] = aa->GetProfileInt(fnum, "Columns", aa->open_rowsize_);
        if (rowsize_[ii] < 4 || rowsize_[ii] > CHexEditView::max_buf) rowsize_[ii] = 4;
        group_by_[ii] = aa->GetProfileInt(fnum, "Grouping", aa->open_group_by_);
        if (group_by_[ii] < 2) group_by_[ii] = 2;
        offset_[ii] = aa->GetProfileInt(fnum, "Offset", aa->open_offset_);
        if (offset_[ii] < 0 || offset_[ii] >= rowsize_[ii]) offset_[ii] = 0;

        // The following can't really be validated here.
        start_addr_[ii] = aa->GetProfileInt(fnum, "SelectionStart", 0);
        end_addr_[ii] = aa->GetProfileInt(fnum, "SelectionEnd", 0);
        edit_char_[ii] = aa->GetProfileInt(fnum, "EditChar", 0) ? TRUE : FALSE;
        scroll_[ii] = aa->GetProfileInt(fnum, "Position", 0);
        mark_[ii] = aa->GetProfileInt(fnum, "Mark", 0);

        CString strFont, strTemp;
        strFont = aa->GetProfileString(fnum, "Font");     // Font info
        AfxExtractSubString(face_[ii], strFont, 0, ',');
        if (face_[ii].IsEmpty()) face_[ii] = "Courier";
        AfxExtractSubString(strTemp, strFont, 1, ',');
        height_[ii] = atol(strTemp);
        if (height_[ii] < 2 || height_[ii] > 100) height_[ii] = 16;

        strFont = aa->GetProfileString(fnum, "OemFont");     // OEM/IBM font info
        AfxExtractSubString(oem_face_[ii], strFont, 0, ',');
        if (oem_face_[ii].IsEmpty()) oem_face_[ii] = "Terminal";
        AfxExtractSubString(strTemp, strFont, 1, ',');
        oem_height_[ii] = atol(strTemp);
        if (oem_height_[ii] < 2 || oem_height_[ii] > 100) oem_height_[ii] = 18;

        cmd_[ii] = aa->GetProfileInt(fnum, "WindowState", SW_SHOWNORMAL);
        top_[ii] = aa->GetProfileInt(fnum, "WindowTop", -30000);
        left_[ii] = aa->GetProfileInt(fnum, "WindowLeft", -30000);
        bottom_[ii] = aa->GetProfileInt(fnum, "WindowBottom", -30000);
        right_[ii] = aa->GetProfileInt(fnum, "WindowRight", -30000);
    }
}

void CHexFileList::WriteList()
{
    CRecentFileList::WriteList();

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->save_exit_)
        return;

    for (int ii = 0; ii < m_nSize; ++ii)
    {
        if (!m_arrNames[ii].IsEmpty())
        {
            CString fnum;
            fnum.Format("File%d", ii+1);

            aa->WriteProfileInt(fnum, "AutoFit", autofit_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "DecimalAddresses", dec_addr_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "DisplayChar", display_char_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "EBCDIC", ebcdic_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "OemChars", oem_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "ControlChars", control_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "GraphicChars", graphic_[ii] ? 1 : 0);

            if (ppart_[ii] != NULL)
            {
                int pi;
                std::vector<partn>::iterator pp;
                for (pi = 1, pp = (*ppart_[ii]).begin();
                     pp != (*ppart_[ii]).end(); ++pp, ++pi)
                {
                    CString f1, f2, f3;
                    f1.Format("Name%d", pi);
                    f2.Format("Colour%d", pi);
                    f3.Format("Range%d", pi);
                    aa->SetColours(fnum, f1, f2, f3, (*pp));
                }
            }
            else
                aa->WriteProfileString(fnum, "Name1", "");  // Signal to use defaults

            aa->WriteProfileInt(fnum, "AllowMods", allow_mods_[ii] ? 1 : 0);
            aa->WriteProfileInt(fnum, "Insert", insert_[ii] ? 1 : 0);

            aa->WriteProfileInt(fnum, "Columns", rowsize_[ii]);
            aa->WriteProfileInt(fnum, "Grouping", group_by_[ii]);
            aa->WriteProfileInt(fnum, "Offset", offset_[ii]);

            aa->WriteProfileInt(fnum, "Position", scroll_[ii]);
            aa->WriteProfileInt(fnum, "SelectionStart", start_addr_[ii]);
            aa->WriteProfileInt(fnum, "SelectionEnd", end_addr_[ii]);
            aa->WriteProfileInt(fnum, "EditChar", edit_char_[ii]);
            aa->WriteProfileInt(fnum, "Mark", mark_[ii]);

            CString strFont;
            strFont.Format("%s,%ld", face_[ii], height_[ii]);
            aa->WriteProfileString(fnum, "Font", strFont);
            strFont.Format("%s,%ld", oem_face_[ii], oem_height_[ii]);
            aa->WriteProfileString(fnum, "OemFont", strFont);

            aa->WriteProfileInt(fnum, "WindowState", cmd_[ii]);
            aa->WriteProfileInt(fnum, "WindowTop", top_[ii]);
            aa->WriteProfileInt(fnum, "WindowLeft", left_[ii]);
            aa->WriteProfileInt(fnum, "WindowBottom", bottom_[ii]);
            aa->WriteProfileInt(fnum, "WindowRight", right_[ii]);
        }
    }
}

int CHexFileList::GetIndex(LPCTSTR lpszPathName) const
{
    ASSERT(lpszPathName != NULL);
    ASSERT(AfxIsValidString(lpszPathName));

    for (int ii = 0; ii < m_nSize; ++ii)
    {
        if (AfxComparePath(m_arrNames[ii], lpszPathName))
            return ii;
    }
    return -1;
}
