// Control.cpp : implements control bars and controls thereon
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <locale.h>
#include <assert.h>

#include "HexEdit.h"
#include "MainFrm.h"
#include "HexEditDoc.h"
#include "HexEditView.h"
#include "resource.hm"          // For control help IDs
#include <afxpriv.h>            // for WM_COMMANDHELP and WM_HELPHITTEST

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

static long address_saved;      // Address when an address combo got focus
static BOOL address_edit = FALSE; // TRUE if in edit (Enter not yet pressed)

// Return focus to the currently active view
static void give_view_focus()
{
    CHexEditView *pview = GetView();
    if (pview != NULL && pview != pview->GetFocus())
        pview->SetFocus();      // return focus to view window
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CHexEditControl

CHexEditControl::CHexEditControl()
{
}

CHexEditControl::~CHexEditControl()
{
}


BEGIN_MESSAGE_MAP(CHexEditControl, CEdit)
        //{{AFX_MSG_MAP(CHexEditControl)
        ON_WM_CHAR()
        ON_WM_SETFOCUS()
        ON_WM_GETDLGCODE()
        ON_WM_KILLFOCUS()
        ON_WM_KEYDOWN()
        //}}AFX_MSG_MAP
        ON_MESSAGE(WM_COMMANDHELP, OnCommandHelp)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CHexEditControl message handlers

void CHexEditControl::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    if (nChar == '\033')                // Escape key
    {
        // Restore the text and signal return from edit control
        give_view_focus();
        return;
    }
    else if (nChar == '\t')
    {
        // Move to adjoining edit control
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        if (::GetKeyState(VK_SHIFT) < 0)
            mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)->SetFocus();
        else
            mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC)->SetFocus();
        return;
    }
    else if (nChar == '\r' || nChar == '\n')
    {
        // Signal the main app window to return to the view
        address_edit = FALSE;
        AfxGetMainWnd()->SendMessage(WM_USER, 1, (long)this);
        give_view_focus();
        return;
    }
    else if (nChar == '\b' && start > 1 && start == end && ss[start-1] == ' ')
    {
        // If deleting 1st nybble (hex mode) then also delete preceding space
        SetSel(start-2, end);
        // no return
    }
    else if (nChar == ' ')
    {
        // Just ignore spaces
        return;
    }
    else if (::isprint(nChar) && strchr("0123456789ABCDEFabcdef\b", nChar) == NULL)
    {
        ::Beep(5000,200);
        return;
    }
    CEdit::OnChar(nChar, nRepCnt, nFlags);
    add_spaces();
}

void CHexEditControl::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    if (nChar == VK_DELETE && start == end && ss.GetLength() > start+1 &&
             ss[start+1] == ' ')
        SetSel(start, start+2);
    else if (nChar == VK_LEFT && start == end && start > 0 && ss[start-1] == ' ')
        SetSel(start-1, start-1);

    CEdit::OnKeyDown(nChar, nRepCnt, nFlags);

    // Tidy display if hex and char(s) deleted or caret moved
    GetWindowText(ss);
    if ((nChar == VK_DELETE || nChar == VK_RIGHT || nChar == VK_LEFT) &&
        ::GetKeyState(VK_SHIFT) >= 0 && ss.GetLength() > 0)
    {
        add_spaces();
    }
}

void CHexEditControl::add_spaces()
{
    CString ss;                         // Current string
    GetWindowText(ss);
    int start, end;                     // Current selection (start==end if just caret)
    GetSel(start, end);

    const char *pp;                     // Ptr into orig. (hex digit) string
    int newstart, newend;               // New caret position/selection
    newstart = newend = 0;              // In case of empty string

    // Allocate enough space (allowing for space padding)
    char *out = new char[(ss.GetLength()*3)/2+1]; // Where chars are stored
    size_t ii, jj;                      // Number of hex nybbles written/read
    char *dd = out;                     // Ptr to current output char
    int ndigits;                        // Numbers of chars that are part of number

    for (pp = ss.GetBuffer(0), ndigits = 0; *pp != '\0'; ++pp)
        if (isxdigit(*pp))
            ++ndigits;

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    for (pp = ss.GetBuffer(0), ii = jj = 0; /*forever*/; ++pp, ++ii)
    {
        if (ii == start)
            newstart = dd - out;        // save new caret position
        if (ii == end)
            newend = dd - out;

        if (*pp == '\0')
            break;

        // Ignore spaces (and anything else)
        if (!isxdigit(*pp))
            continue;                   // ignore all but digits

        if (aa->hex_ucase_)
            *dd++ = toupper(*pp);       // convert all to upper case
        else
            *dd++ = tolower(*pp);       // convert all to lower case

        ++jj;
        if ((ndigits - jj) % 4 == 0)
            *dd++ = ' ';                // pad with space after every 4th hex digit
    }
    if (dd > out)
        dd--;                           // Forget last comma if added
    *dd = '\0';                         // Terminate string

    SetWindowText(out);
    SetSel(newstart, newend);
    delete[] out;
}

void CHexEditControl::OnSetFocus(CWnd* pOldWnd) 
{
    CEdit::OnSetFocus(pOldWnd);
    SetSel(0,-1);

    // Save original contents in case Escape pressed or focus leaves
    // address edit control(s) in some other way
    if (!address_edit)
    {
        CString ss;
        GetWindowText(ss);
        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isxdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        address_saved = strtoul(buf, NULL, 16);
        address_edit = TRUE;
    }
}

void CHexEditControl::OnKillFocus(CWnd* pNewWnd) 
{
    CEdit::OnKillFocus(pNewWnd);

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // If not because Enter pressed and not simply flipping between address controls
    if (address_edit && pNewWnd != NULL &&
        pNewWnd->m_hWnd != mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC)->m_hWnd &&
        pNewWnd->m_hWnd != mm->dec_dec_addr_.m_hWnd)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // User changed focus without hitting Enter so revert to address_saved
        CString ss;
        if (aa->hex_ucase_)
            ss.Format("%lX", address_saved);
        else
            ss.Format("%lx", address_saved);
        SetWindowText(ss);
        add_spaces();

        // Keep decimal address combo in line with hex
        ss.Format("%lu", address_saved);
        mm->dec_dec_addr_.SetWindowText(ss);
        mm->dec_dec_addr_.add_commas();

#if 0
        // Move the view caret position back where it was
        CHexEditView *pview = GetView();
        ASSERT(pview != NULL);
        if (pview != NULL)
        {
            // Try to go to the address requested
            (void)pview->GoAddress(address_saved);
            pview->DisplayCaret();
        }
#endif
        address_edit = FALSE;
    }
}

UINT CHexEditControl::OnGetDlgCode() 
{
    // Get all keys so that we see CR and Escape
    return CEdit::OnGetDlgCode() | DLGC_WANTALLKEYS;
}

LRESULT CHexEditControl::OnCommandHelp(WPARAM, LPARAM lParam)
{
    // Since there is only one control of this type just call help with its help ID
    if (!::WinHelp(AfxGetMainWnd()->m_hWnd, AfxGetApp()->m_pszHelpFilePath,
        HELP_CONTEXT, HIDC_JUMP_HEX))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return 1;                           // Indicate help launched
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CDecEditControl

CDecEditControl::CDecEditControl()
{
    struct lconv *plconv = localeconv();
    if (strlen(plconv->thousands_sep) == 1)
    {
        sep_char_ = *plconv->thousands_sep;
        group_ = *plconv->grouping;
    }
    else
    {
        sep_char_ = ',';
        group_ = 3;
    }
}

CDecEditControl::~CDecEditControl()
{
}


BEGIN_MESSAGE_MAP(CDecEditControl, CEdit)
        //{{AFX_MSG_MAP(CDecEditControl)
        ON_WM_CHAR()
        ON_WM_SETFOCUS()
        ON_WM_GETDLGCODE()
        ON_WM_KILLFOCUS()
        ON_WM_KEYDOWN()
        //}}AFX_MSG_MAP
        ON_MESSAGE(WM_COMMANDHELP, OnCommandHelp)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDecEditControl message handlers

void CDecEditControl::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    if (nChar == '\033')                // Escape key
    {
        // Restore the text and signal return from edit control
        give_view_focus();
        return;
    }
    else if (nChar == '\t')
    {
        // Move to adjoining edit control
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        if (::GetKeyState(VK_SHIFT) < 0)
            mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX)->SetFocus();
        else
            mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)->SetFocus();
        return;
    }
    else if (nChar == '\r' || nChar == '\n')
    {
        // Signal the main app window that Return has been pressed
        address_edit = FALSE;
        AfxGetMainWnd()->SendMessage(WM_USER, 1, (long)this);
        give_view_focus();
        return;
    }
    else if (nChar == '\b' && start > 1 && start == end && ss[start-1] == sep_char_)
    {
        // If deleting 1st nybble (hex mode) then also delete preceding space
        SetSel(start-2, end);
        // no return
    }
    else if (nChar == ' ' || nChar == sep_char_)
    {
        // Just ignore these characters
        return;
    }
    else if (::isprint(nChar) && strchr("0123456789\b", nChar) == NULL)
    {
        ::Beep(5000,200);
        return;
    }
    CEdit::OnChar(nChar, nRepCnt, nFlags);
    add_commas();
}

void CDecEditControl::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    if (nChar == VK_DELETE && start == end && ss.GetLength() > start+1 &&
             ss[start+1] == sep_char_)
        SetSel(start, start+2);
    else if (nChar == VK_LEFT && start == end && start > 0 && ss[start-1] == sep_char_)
        SetSel(start-1, start-1);

    CEdit::OnKeyDown(nChar, nRepCnt, nFlags);

    // Tidy display if hex and char(s) deleted or caret moved
    GetWindowText(ss);
    if ((nChar == VK_DELETE || nChar == VK_RIGHT || nChar == VK_LEFT) &&
        ::GetKeyState(VK_SHIFT) >= 0 && ss.GetLength() > 0)
    {
        add_commas();
    }
}

void CDecEditControl::add_commas()
{
    CString ss;                         // Current string
    GetWindowText(ss);
    int start, end;                     // Current selection (start==end if just caret)
    GetSel(start, end);

    const char *pp;                     // Ptr into orig. (hex digit) string
    int newstart, newend;               // New caret position/selection
    newstart = newend = 0;              // In case of empty string

    // Allocate enough space (allowing for comma padding)
    char *out = new char[(ss.GetLength()*(group_+1))/group_ + 1]; // Where chars are stored
    size_t ii, jj;                      // Number of hex nybbles written/read
    char *dd = out;                     // Ptr to current output char
    int ndigits;                        // Numbers of chars that are part of number
    BOOL none_yet = TRUE;               // Have we seen any non-zero digits?

    for (pp = ss.GetBuffer(0), ndigits = 0; *pp != '\0'; ++pp)
        if (isdigit(*pp))
            ++ndigits;

    for (pp = ss.GetBuffer(0), ii = jj = 0; /*forever*/; ++pp, ++ii)
    {
        if (ii == start)
            newstart = dd - out;        // save new caret position
        if (ii == end)
            newend = dd - out;

        if (*pp == '\0')
            break;

        // Ignore spaces (and anything else)
        if (!isdigit(*pp))
            continue;                   // ignore all but digits

        if (++jj < ndigits && none_yet && *pp == '0')
            continue;                   // ignore leading zeroes

        none_yet = FALSE;
        *dd++ = *pp;

        if ((ndigits - jj) % group_ == 0)
            *dd++ = sep_char_;          // put in thousands separator
    }
    if (dd > out)
        dd--;                           // Forget last comma if added
    *dd = '\0';                         // Terminate string

    SetWindowText(out);
    SetSel(newstart, newend);
    delete[] out;
}

void CDecEditControl::OnSetFocus(CWnd* pOldWnd) 
{
    CEdit::OnSetFocus(pOldWnd);
    SetSel(0,-1);
    
    // Save original contents in case Escape pressed or focus leaves
    // address edit control(s) in some other way
    if (!address_edit)
    {
        CString ss;
        GetWindowText(ss);
        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        address_saved = strtoul(buf, NULL, 10);
        address_edit = TRUE;
    }
}

void CDecEditControl::OnKillFocus(CWnd* pNewWnd) 
{
    CEdit::OnKillFocus(pNewWnd);
        
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // If Enter not pressed and not simply flipping between address controls
    if (address_edit && pNewWnd != NULL &&
        pNewWnd->m_hWnd != mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX)->m_hWnd &&
        pNewWnd->m_hWnd != mm->hec_hex_addr_.m_hWnd)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // User changed focus without hitting Enter so revert to address_saved
        CString ss;
        ss.Format("%lu", address_saved);
        SetWindowText(ss);
        add_commas();

        // Keep hex address combo in line with decimal
        if (aa->hex_ucase_)
            ss.Format("%lX", address_saved);
        else
            ss.Format("%lx", address_saved);
        mm->hec_hex_addr_.SetWindowText(ss);
        mm->hec_hex_addr_.add_spaces();

#if 0
        // Move the view caret position back where it was
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            // Try to go to the address requested
            (void)pview->GoAddress(address_saved);
            pview->DisplayCaret();
        }
#endif
        address_edit = FALSE;
    }
}

UINT CDecEditControl::OnGetDlgCode() 
{
    // Get all keys so that we see CR and Escape
    return CEdit::OnGetDlgCode() | DLGC_WANTALLKEYS;
}

LRESULT CDecEditControl::OnCommandHelp(WPARAM, LPARAM lParam)
{
    // Since there is only one control of this type just call help with its help ID
    if (!::WinHelp(AfxGetMainWnd()->m_hWnd, AfxGetApp()->m_pszHelpFilePath,
        HELP_CONTEXT, HIDC_JUMP_DEC))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return 1;                           // Indicate help launched
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CSearchEditControl

CSearchEditControl::CSearchEditControl()
{
    in_edit_ = FALSE;
    mode_ = mode_none;
}

CSearchEditControl::~CSearchEditControl()
{
}

BEGIN_MESSAGE_MAP(CSearchEditControl, CEdit)
        //{{AFX_MSG_MAP(CSearchEditControl)
        ON_WM_CHAR()
        ON_WM_GETDLGCODE()
        ON_WM_SETFOCUS()
        ON_WM_KEYDOWN()
        ON_WM_KILLFOCUS()
        //}}AFX_MSG_MAP
        ON_MESSAGE(WM_COMMANDHELP, OnCommandHelp)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSearchEditControl message handlers

void CSearchEditControl::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;             // Range of current selection
    GetSel(start, end);         // (start == end means no selection just caret)

    if (nChar == '\033')        // Escape key
    {
        // Give focus back to view - OnKillFocus restores address etc
        give_view_focus();
        return;
    }
    else if (nChar == '\t')
    {
        // Move to next tabbable control
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        if (::GetKeyState(VK_SHIFT) < 0)
            mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC)->SetFocus();
        else
            mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX)->SetFocus();
        return;
    }
    else if (nChar == '\r' || nChar == '\n')
    {
        // Signal the main app window that Return has been pressed
        in_edit_ = FALSE;
        AfxGetMainWnd()->SendMessage(WM_USER, 1, (long)this);
        give_view_focus();
        return;
    }
    else if (nChar == '\b' &&                                       // Backspace ...
             (start == 1 && end == 1 || start == 0 && end > 0) &&   // When after 1st char OR they are included in selection
             (ss[0] == sflag_char || ss[0] == iflag_char))          // And 1st char is '~' or '='
    {
        // Convert ASCII chars to equiv. hex digits
        if (start == 0)
        {
            SetSel(1, end);
            Clear();
        }
        convert2hex();
        return;
    }
    else if (nChar == '\b' && start > 1 && start == end && ss[start-1] == ' ' &&
             ss[0] != sflag_char && ss[0] != iflag_char)
    {
        // If deleting 1st nybble (hex mode) then also delete preceding space
        SetSel(start-2, end);
    }
    else if (nChar == sflag_char && start == 0 && ss.GetLength() > 0 &&   // '=' typed when at start
                (ss[0] == iflag_char || ss[0] == sflag_char))             // and doing char search
    {
        // Make search case-sensitive
        char new_text[2] = "?";
        new_text[0] = sflag_char;
        if (end == 0) SetSel(0,1);
        ReplaceSel(new_text);
        return;
    }
    else if (nChar == iflag_char && start == 0 && ss.GetLength() > 0 &&   // '~' typed when at start
                (ss[0] == iflag_char || ss[0] == sflag_char))             // and doing char search
    {
        // Make search case-insensitive
        char new_text[2] = "?";
        new_text[0] = iflag_char;
        if (end == 0) SetSel(0,1);
        ReplaceSel(new_text);
        return;
    }
    else if ((nChar == sflag_char || nChar == iflag_char) &&
             start == 0 && ss.GetLength() > 0 &&
             ss[0] != sflag_char && ss[0] != iflag_char )
    {
        // =/~ at start and not one there already
        Clear();
        convert2chars(nChar);
        SetSel(1,1);
        return;
    }
    else if (nChar == ' ' && ss.GetLength() > 0 &&
             ss[0] != sflag_char && ss[0] != iflag_char )
    {
        // Ignore spaces if in hex mode (tend to type them because of display)
        return;
    }
    else if (::isprint(nChar))
    {
        char valid_empty[] = "??0123456789ABCDEFabcdef";
        valid_empty[0] = sflag_char;
        valid_empty[1] = iflag_char;

        // Disallow the following (where ASCII mode is where ss[0] == iflag or sflag)
        // - any char at start (except sflag/iflag/DEL, handled above) if ASCII mode
        // - any char except sflag/iflag/hex-digit if field empty
        // - non-hexdigit (except BS) if not ASCII mode
        if ((start == 0 && ss.GetLength() > 0 && (ss[0]==sflag_char||ss[0]==iflag_char)) ||
            (ss.GetLength() == 0 && strchr(valid_empty, nChar) == NULL) ||
            (ss.GetLength() > 0 && ss[0] != sflag_char && ss[0] != iflag_char &&
                        strchr("\b0123456789ABCDEFabcdef", nChar) == NULL) )
        {
            ::Beep(5000,200);
            return;
        }
    }

    CEdit::OnChar(nChar, nRepCnt, nFlags);

    // If in hex mode make sure it looks neat
    GetWindowText(ss);
    if (ss.GetLength() > 0 && ss[0] != sflag_char && ss[0] != iflag_char)
        normalize_hex();
}

void CSearchEditControl::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    if (nChar == VK_DELETE && start == 0 && ss.GetLength() > 0 &&
        (ss[0] == sflag_char || ss[0] == iflag_char))
    {
        // Convert ASCII chars to equiv. hex digits
        if (end > 0)
        {
            SetSel(1, end);
            Clear();
        }
        convert2hex();
        return;
    }
    else if (nChar == VK_DELETE && start == end && ss.GetLength() > start+1 &&
             ss[0] != sflag_char && ss[0] != iflag_char && ss[start+1] == ' ')
        SetSel(start, start+2);
    else if (nChar == VK_LEFT && start == end && start > 0 &&
                ss[0] != sflag_char && ss[0] != iflag_char && ss[start-1] == ' ')
        SetSel(start-1, start-1);

    CEdit::OnKeyDown(nChar, nRepCnt, nFlags);

    // Tidy display if hex and char(s) deleted or caret moved
    GetWindowText(ss);
    if ((nChar == VK_DELETE || nChar == VK_RIGHT || nChar == VK_LEFT) &&
        ::GetKeyState(VK_SHIFT) >= 0 && ss.GetLength() > 0 &&
        ss[0] != sflag_char && ss[0] != iflag_char)
    {
        normalize_hex();
    }
}

void CSearchEditControl::OnSetFocus(CWnd* pOldWnd) 
{
    CEdit::OnSetFocus(pOldWnd);
    char new_text[2] = "?";
        
    // Save original contents in case user wants to exit edit without save (Esc)
    GetWindowText(saved_);
    in_edit_ = TRUE;

    if (saved_.GetLength() > 0)
    {
        if (mode_ == mode_hex && (saved_[0] == sflag_char || saved_[0] == iflag_char))
        {
            convert2hex();              // Convert existing chars to hex digits
            SetSel(0,-1);               // Make sure all selected
        }
        else if (mode_ == mode_char && saved_[0] != sflag_char && saved_[0] != iflag_char)
            convert2chars();            // Convert hex digits to chars (as many as possible)
        else if (mode_ == mode_icase && saved_[0] != sflag_char && saved_[0] != iflag_char)
            convert2chars(iflag_char);  // Convert hex digits to chars (icase search)
        else if (mode_ == mode_char && saved_[0] == iflag_char)
        {
            SetSel(0,1);
            new_text[0] = sflag_char;
            ReplaceSel(new_text);
        }
        else if (mode_ == mode_icase && saved_[0] == sflag_char)
        {
            SetSel(0,1);
            new_text[0] = iflag_char;
            ReplaceSel(new_text);
        }
        else if (mode_ == mode_none && saved_[0] == sflag_char)
            mode_ = mode_char;
        else if (mode_ == mode_none && saved_[0] == iflag_char)
            mode_ = mode_icase;
    }
    else if (mode_ == mode_char)
    {
        // No chars but char search mode so add "="
        new_text[0] = sflag_char;
        SetWindowText(new_text);
    }
    else if (mode_ == mode_icase)
    {
        // No chars but case-insensitive search mode so add "~"
        new_text[0] = iflag_char;
        SetWindowText(new_text);
    }

    if (mode_ == mode_char || mode_ == mode_icase)
        SetSel(1,-1);                   // select all but ' so string can easily be overtyped
}

void CSearchEditControl::OnKillFocus(CWnd* pNewWnd) 
{
    CEdit::OnKillFocus(pNewWnd);
        
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    if (in_edit_ && (pNewWnd == NULL ||
        pNewWnd->m_hWnd != mm->m_wndEditBar.GetDlgItem(ID_SEARCH_BACK)->m_hWnd &&
        pNewWnd->m_hWnd != mm->m_wndEditBar.GetDlgItem(ID_SEARCH_FORW)->m_hWnd) )
    {
        SetWindowText(saved_);
        in_edit_ = FALSE;
    }
    mode_ = mode_none;
}

void CSearchEditControl::normalize_hex()
{
    CString ss;
    GetWindowText(ss);
    int start, end;
    GetSel(start, end);

    const char *pp;                     // Ptr into orig. (hex digit) string
    int newstart, newend;               // New caret position/selection
    newstart = newend = 0;              // In case of empty text string

    // Allocate enough space (allowing for space padding)
    char *out = new char[(ss.GetLength()*3)/2+2]; // Where chars are stored
    size_t ii, jj;                      // Number of hex nybbles written/read
    char *dd = out;                     // Ptr to current output char

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    for (pp = ss.GetBuffer(0), ii = jj = 0; /*forever*/; ++pp, ++ii)
    {
        if (ii == start)
            newstart = dd - out;        // save new caret position
        if (ii == end)
            newend = dd - out;

        if (*pp == '\0')
            break;

        // Ignore spaces (and anything else)
        if (!isxdigit(*pp))
            continue;                   // ignore all but hex digits

        if (aa->hex_ucase_)
            *dd++ = toupper(*pp);       // convert all to upper case
        else
            *dd++ = tolower(*pp);       // convert all to lower case

        if ((++jj % 2) == 0)
            *dd++ = ' ';                // pad with space after 2 nybbles
    }
    if ((jj % 2) == 0 && jj > 0)
        dd--;                   // Remove trailing space
    *dd = '\0';                 // Terminate string

    SetWindowText(out);
    SetSel(newstart, newend);
    delete[] out;
}

void CSearchEditControl::convert2hex()
{
    BOOL bad_chars = FALSE;
    BOOL ebcdic = FALSE;
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
        return;

    if (ss[0] != sflag_char && ss[0] != iflag_char)
    {
        convert2chars();                // -> chars then -> hex to normalize
        GetWindowText(ss);
    }

    const char *pp;                     // Ptr into input (ASCII char) string
    char *out = new char[ss.GetLength()*3+1]; // Where output chars are stored
    char *dd = out;                     // Ptr to current output char

    // Display hex in upper or lower case?
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    const char *hex_fmt;
    if (aa->hex_ucase_)
        hex_fmt = "%02X ";
    else
        hex_fmt = "%02x ";

    // If view is displaying EBCDIC chars then convert to hex assuming EBCDIC chars
    ASSERT(GetView() != NULL);
    if (GetView()->EbcdicMode())
        ebcdic = TRUE;

    for (pp = ss.GetBuffer(0) + 1; *pp != '\0'; ++pp)
    {
        if (ebcdic && *pp < 128 && *pp >= 0 && a2e_tab[*pp] != '\0')
        {
            sprintf(dd, hex_fmt, a2e_tab[(unsigned char)*pp]);
            dd += 3;
        }
        else if (ebcdic)
            bad_chars = TRUE;
        else
        {
            sprintf(dd, hex_fmt, (unsigned char)*pp);
            dd += 3;
        }
    }
    if (dd > out) dd--;                 // Forget trailing space
    *dd = '\0';                         // Terminate string

    SetWindowText(out);
    delete[] out;

    if (bad_chars)
    {
        ((CMainFrame *)AfxGetMainWnd())->
            StatusBarText("Hex search: invalid EBCDIC chars ignored");
    }
}

void CSearchEditControl::convert2chars(char c1)
{
    BOOL bad_chars = FALSE;
    BOOL ebcdic = FALSE;
    CString ss;

    GetWindowText(ss);
    if (ss.GetLength() == 0)
    {
        char new_text[2] = "?";
        new_text[0] = c1;
        SetWindowText(new_text);
        return;
    }

    if (ss[0] == sflag_char || ss[0] == iflag_char)
    {
        convert2hex();                  // -> hex then -> chars to normalize
        GetWindowText(ss);
    }

    const char *pp;                     // Ptr to input (hex digit) string

    char *out = new char[ss.GetLength()/2+3]; // Where output chars are stored
    size_t length;                      // Number of hex output nybbles generated
    char *dd = out;                     // Ptr to current output char

    // If view is displaying EBCDIC chars then convert hex to EBCDIC
    ASSERT(GetView() != NULL);
    if (GetView()->EbcdicMode())
        ebcdic = TRUE;

    *(dd++) = c1;                       // Indicate char string (1st char == ' OR *)
    for (pp = ss.GetBuffer(0), length = 0, *dd = '\0'; *pp != '\0'; ++pp)
    {
        // Ignore spaces (and anything else)
        if (!isxdigit(*pp))
            continue;

        *dd = (*dd<<4) + (isdigit(*pp) ? *pp - '0' : toupper(*pp) - 'A' + 10);

        if ((++length % 2) == 0)
        {
            if (ebcdic && (*dd = e2a_tab[(unsigned char)*dd]) != '\0')
                ++dd;                   // Valid EBCDIC so move to next byte
            else if (!ebcdic && (unsigned char)*dd > '\r')
                ++dd;                   // Printable ASCII so move to next byte
            else
            {
                length -= 2;            // Forget byte (2 hex digits) just seen
                bad_chars = TRUE;
            }
            *(dd) = '\0';               // Zero next byte
        }
    }
    if ((length % 2) != 0  && *dd == '\0')  // WAS ... && !isprint(*dd))
        length--;                       // Solo last hex digit -> not printable char
    if (length > 0)
        length = (length-1)/2 + 1;      // Convert nybble count to byte count
    out[length+1] = '\0';

    SetWindowText(out);
    delete[] out;

    if (ebcdic && bad_chars)
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("EBCDIC search: invalid characters ignored");
    else if (bad_chars)
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("ASCII search: control characters ignored");
}

UINT CSearchEditControl::OnGetDlgCode() 
{
    // Get all keys so that we see CR and Escape
    return CEdit::OnGetDlgCode() | DLGC_WANTALLKEYS;
}

LRESULT CSearchEditControl::OnCommandHelp(WPARAM, LPARAM lParam)
{
    // Since there is only one control of this type just call help with its help ID
    if (!::WinHelp(AfxGetMainWnd()->m_hWnd, AfxGetApp()->m_pszHelpFilePath,
        HELP_CONTEXT, HIDC_SEARCH))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return 1;                           // Indicate help launched
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CStatBar

static UINT indicators[] =
{
        ID_SEPARATOR,           // status line indicator
        ID_INDICATOR_OCCURRENCES,
        ID_INDICATOR_VALUES,
        ID_INDICATOR_HEX_ADDR,
        ID_INDICATOR_DEC_ADDR,
        ID_INDICATOR_READONLY,
        ID_INDICATOR_OVR,
        ID_INDICATOR_REC,
        ID_INDICATOR_CAPS,
        ID_INDICATOR_NUM,
//      ID_INDICATOR_SCRL,
};

CStatBar::CStatBar()
{
}

CStatBar::~CStatBar()
{
}

BOOL CStatBar::PreTranslateMessage(MSG* pMsg)
{
    if (ttip_.m_hWnd != NULL)
    {
        switch(pMsg->message)
        {
        case WM_MOUSEMOVE:
        case WM_LBUTTONDOWN:
        case WM_LBUTTONUP:
        case WM_MBUTTONDOWN:
        case WM_MBUTTONUP:
        case WM_RBUTTONDOWN:
        case WM_RBUTTONUP:
            // this will reactivate the tooltip
            ttip_.Activate(TRUE);
            ttip_.RelayEvent(pMsg);
            break;
        }
    }

    return CStatusBar::PreTranslateMessage(pMsg);
}

BEGIN_MESSAGE_MAP(CStatBar, CStatusBar)
        //{{AFX_MSG_MAP(CStatBar)
        ON_WM_LBUTTONDBLCLK()
        ON_WM_RBUTTONDOWN()
        ON_WM_CONTEXTMENU()
        ON_WM_CREATE()
        ON_WM_SIZE()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CStatBar message handlers

int CStatBar::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
    if (CStatusBar::OnCreate(lpCreateStruct) == -1)
        return -1;

    if (!SetIndicators(indicators, sizeof(indicators)/sizeof(*indicators)))
        return -1;

    return 0;
}

void CStatBar::OnSize(UINT nType, int cx, int cy) 
{
    CStatusBar::OnSize(nType, cx, cy);

    if (cx > 0 && cy > 0 && ttip_.m_hWnd == NULL)
    {
        VERIFY(ttip_.Create(this));

        add_tooltip(ID_INDICATOR_OCCURRENCES, "Occurrences of search text");
        add_tooltip(ID_INDICATOR_VALUES, "Hex,Dec,Octal,Binary & char values");
        add_tooltip(ID_INDICATOR_HEX_ADDR, "Distance from mark (hex)");
        add_tooltip(ID_INDICATOR_DEC_ADDR, "Distance from mark (decimal)");
        add_tooltip(ID_INDICATOR_READONLY, "Read-Only/Read-Write");
        add_tooltip(ID_INDICATOR_OVR, "Overtype/Insert");
        add_tooltip(ID_INDICATOR_REC, "Macro Recording");

        ttip_.Activate(TRUE);
        ttip_.SetDelayTime(1000);
    }
    else if (cx > 0 && cy > 0)
    {
        move_tooltip(ID_INDICATOR_OCCURRENCES);
        move_tooltip(ID_INDICATOR_VALUES);
        move_tooltip(ID_INDICATOR_HEX_ADDR);
        move_tooltip(ID_INDICATOR_DEC_ADDR);
        move_tooltip(ID_INDICATOR_READONLY);
        move_tooltip(ID_INDICATOR_OVR);
        move_tooltip(ID_INDICATOR_REC);
    }
}

void CStatBar::add_tooltip(UINT id, const char *ss)
{
    int index;                  // Index of status bar pane
    CRect rect;                 // Rectangle (within status bar) of pane

    index = CommandToIndex(id);
    ASSERT(index > 0);
    GetItemRect(index, &rect);

    // rect may be empty if the window is too small to show this pane but
    // AddTool seems to handle this OK (ie. never shows the tip).
    ttip_.AddTool(this, ss, rect, id);
}

void CStatBar::move_tooltip(UINT id)
{
    int index;                  // Index of status bar pane
    CRect rect;                 // Rectangle (within status bar) of pane

    index = CommandToIndex(id);
    ASSERT(index > 0);
    GetItemRect(index, &rect);
    ttip_.SetToolRect(this, id, rect);
}

void CStatBar::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
    int index;                   // Index (0 based) of current pane we're checking
    CRect pane_rect, pane_rect2; // Bounds (device coords) of pane

    // Check if mouse clicked on hex/dec/octal/binary/char pane
    index = CommandToIndex(ID_INDICATOR_VALUES);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

        // If there is currently a view and no properties dlg exists create it
        if (GetView() != NULL && mm->pprop_ == NULL)
        {
            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
            // If prev prop page was 0 (file info) change to 1 (character info)
            mm->pprop_ = new CPropSheet(aa->prop_page_ == 0 ? 1 : aa->prop_page_);
        }
        else if (mm->pprop_ != NULL)
            mm->pprop_->SetFocus();
        return;
    }

    // Check if mouse clicked in a "distance to" pane
    index = CommandToIndex(ID_INDICATOR_HEX_ADDR);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    index = CommandToIndex(ID_INDICATOR_DEC_ADDR);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect2);
    if (pane_rect.PtInRect(point) ||
        pane_rect2.PtInRect(point))
    {
        // Display the calculator
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        mm->OnEditGoto();
        return;
    }

    // Check if mouse clicked in a "search occurrences" pane
    index = CommandToIndex(ID_INDICATOR_OCCURRENCES);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        // Display the calculator
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
        mm->OnEditFind();
        return;
    }

    // Check if mouse clicked on READONLY pane
    index = CommandToIndex(ID_INDICATOR_READONLY);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        CHexEditView *pview = GetView();
        if (pview != NULL)
            GetView()->AllowMods();
        return;
    }

    // Check if mouse clicked on OVR pane
    index = CommandToIndex(ID_INDICATOR_OVR);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        CHexEditView *pview = GetView();
        if (pview != NULL && !pview->ReadOnly())
            pview->ToggleInsert();
        return;
    }
        
    // Check if mouse clicked on OVR pane
    index = CommandToIndex(ID_INDICATOR_REC);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->OnMacroRecord();
        return;
    }

    index = CommandToIndex(ID_INDICATOR_CAPS);
    ASSERT(index > 0);
    GetItemRect(index, &pane_rect);
    if (pane_rect.PtInRect(point))
    {
        // Simulate Caps Lock key being depressed then released
        keybd_event(VK_CAPITAL, 0x45, KEYEVENTF_EXTENDEDKEY | 0, 0);
        keybd_event(VK_CAPITAL, 0x45, KEYEVENTF_EXTENDEDKEY | KEYEVENTF_KEYUP, 0);
        return;
    }

// This doesn't work under Windows 95 (indicator changes but not light/kb behaviour)
//    index = CommandToIndex(ID_INDICATOR_NUM);
//    ASSERT(index > 0);
//    GetItemRect(index, &pane_rect);
//    if (pane_rect.PtInRect(point))
//    {
//      // Simulate Num Lock key being depressed then released
//      keybd_event(VK_NUMLOCK, 0x45, KEYEVENTF_EXTENDEDKEY | 0, 0);
//      keybd_event(VK_NUMLOCK, 0x45, KEYEVENTF_EXTENDEDKEY | KEYEVENTF_KEYUP, 0);
//      return;
//    }

    CStatusBar::OnLButtonDblClk(nFlags, point);
}

void CStatBar::OnRButtonDown(UINT nFlags, CPoint point) 
{
    CPoint scr_point(point);
    ClientToScreen(&scr_point);
    ((CMainFrame *)AfxGetMainWnd())->bar_context(scr_point);
//    CStatusBar::OnRButtonDown(nFlags, point);
}

void CStatBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
    CPoint scr_point(point);
    ClientToScreen(&scr_point);
    ((CMainFrame *)AfxGetMainWnd())->bar_context(scr_point);
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CDlgBar

BEGIN_MESSAGE_MAP(CDlgBar, CDialogBar)
        //{{AFX_MSG_MAP(CDlgBar)
        ON_WM_RBUTTONDOWN()
        ON_WM_CONTEXTMENU()
        //}}AFX_MSG_MAP
        ON_MESSAGE(WM_HELPHITTEST, OnHelpHitTest)
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CDlgBar message handlers

// The following called for Shift-F1 help when CDlgBar clicked
LRESULT CDlgBar::OnHelpHitTest(WPARAM wParam, LPARAM lParam)
{
    LRESULT retval = CDialogBar::OnHelpHitTest(wParam, lParam);

    // Translate help ID for non-button controls on Edit Bar
    if (retval == 0x10000 + IDC_JUMP_HEX)
        retval = HIDC_JUMP_HEX;
    else if (retval == 0x10000 + IDC_JUMP_DEC)
        retval = HIDC_JUMP_DEC;
    else if (retval == 0x10000 + IDC_SEARCH)
        retval = HIDC_SEARCH;

    return retval;
}

void CDlgBar::OnRButtonDown(UINT nFlags, CPoint point) 
{
    CPoint scr_point(point);
    ClientToScreen(&scr_point);
    ((CMainFrame *)AfxGetMainWnd())->bar_context(scr_point);
        
//    CDialogBar::OnRButtonDown(nFlags, point);
}

void CDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
    CPoint scr_point(point);
    ClientToScreen(&scr_point);
    ((CMainFrame *)AfxGetMainWnd())->bar_context(scr_point);
}
 
//===========================================================================
