package com.tolstoy.imagemeister;

import com.jconfig.*;
import java.io.*;
import java.awt.*;
import java.util.*;

/**
Your IMPluginFactory class should call this singleton's addPlugin() method to register its
plugin(s).

@author Copyright (c) 1997,1998,1999,2000 Samizdat Productions. All Rights Reserved.
*/

public class PluginManager {
	private static final String copyright="ImageMeister Copyright (c) 1998,1999,2000 Samizdat Productions. All Rights Reserved. ImageMeister is a Trademark of Samizdat Productions.";

/**
Holds the PluginFactoryI objects, from which plugins are spawned.
*/

	private static Vector		theFactories;

/**
Holds the PluginWatcherI objects, which are called as each plugin is created.
*/

	private static Vector		theWatchers;

/**
Holds the PluginI objects, which are the plugins.
*/

	private static Vector		thePlugins;

	private static final int	kCurrentInterfaceVersion = 0x100;

	static int initialize( String fileName ) {
		String				strs[];

		theFactories = new Vector( 20, 20 );
		theWatchers = new Vector( 20, 20 );
		thePlugins = new Vector( 20, 20 );

		strs = IMUtils.fileToStringArray( fileName );
		if ( strs != null )
			addExternalFactories( strs );
		
		initializeWatchers();
		registerExternalPlugins();		
		registerStockPlugins();
		
		return 0;
	}

/**
Your IMPluginFactory class should call this method with each plugin to be registered. If the
plugin was added successfully, returns 0. Otherwise, returns a non-zero error code.
@param flags reserved; set to 0
*/

	public static int addPlugin( PluginI plug, int flags ) {
		PluginWatcherI		watcher;
		int					ignored, len, i;

		if ( plug == null )
			return -1;

		len = theWatchers.size();
		for ( i = 0; i < len; i++ ) {
			watcher = (PluginWatcherI) theWatchers.elementAt( i );
			watcher.loadingPlugin( plug, 0 );
		}

		Trace.println( "addPlugin: name=" + plug.getName( PluginI.kDisplayName ) );

		thePlugins.addElement( plug );
		
		return 0;
	}

/**
Search for plugins which can create an ImageViewerI which can display the file
specified by spec.
*/

	static ImageViewerI createImageViewer( OwnedFrame frame, ImageViewerOwner onr,
											FileSpecifier spec, Rectangle rect ) {
		ImageViewerI	imageViewer;
		PluginI			plug;
		int				len, i;

		imageViewer = null;
		len = thePlugins.size();

		for ( i = 0; i < len; i++ ) {
			plug = (PluginI) thePlugins.elementAt( i );
			if ( !plug.canCreateImageViewer( spec ) )
				continue;

			imageViewer = plug.createImageViewer( frame, onr, spec, rect );
			if ( imageViewer != null )
				break;
		}
		
		return imageViewer;
	}
	
/**
Search for plugins which can create an InfoViewerI which can display the file
specified by spec.
*/

	static InfoViewerI createInfoViewer( OwnedFrame frame, InfoViewerOwner onr,
											FileSpecifier spec, Rectangle rect ) {
		InfoViewerI	infoViewer;
		PluginI			plug;
		int				len, i;

		infoViewer = null;
		len = thePlugins.size();

		for ( i = 0; i < len; i++ ) {
			plug = (PluginI) thePlugins.elementAt( i );
			if ( !plug.canCreateInfoViewer( spec ) )
				continue;

			infoViewer = plug.createInfoViewer( frame, onr, spec, rect );
			if ( infoViewer != null )
				break;
		}

		return infoViewer;
	}

/**
Returns a list of the DiskObjects contained by 'cont'.
@param bShowAll if this is true, a GenDiskFilter object is used to get a list of the files,
otherwise, a PIMDiskFilter object is used
*/
 	
	static DiskObject[] runDiskFilter( DiskObject cont, int flags, int maxToReturn, boolean bShowAll ) {
		PIMDiskFilter			pimDiskFilter;
		GenDiskFilter			genDiskFilter;

		if ( bShowAll ) {
			genDiskFilter = new GenDiskFilter( maxToReturn );
			cont.iterate( genDiskFilter, flags, maxToReturn );
			return genDiskFilter.getArray();
		}
		else {
			pimDiskFilter = new PIMDiskFilter( maxToReturn, thePlugins );
			cont.iterate( pimDiskFilter, flags, maxToReturn );
			return pimDiskFilter.getArray();
		}
	}

/**
'strs' contains a list of full class names which implement the PluginFactoryI interface.
For each element in this array, create an object of the class, and add it to the 'theFactories' vector.
*/

	private static void addExternalFactories( String strs[] ) {
		PluginFactoryI		factory;
		PluginWatcherI		watcher;
		Class				factoryClass;
		String				className;
		int					len, i;

		len = strs.length;
		for ( i = 0; i < len; i++ ) {
			try {
				className = IMUtils.zipFileNameToJavaClassName( strs[ i ] );
				Trace.println( "PIM.init, trying to create " + className );
				factoryClass = Class.forName( className );
				factory = (PluginFactoryI) factoryClass.newInstance();
				if ( factory != null ) {
					theFactories.addElement( factory );
					Trace.println( "    succeeded" );
				}
				else
					Trace.println( "    failed" );
			}
			catch ( Exception e ) {
				Trace.println( "    failed" );
				continue;
			}
		}
	}

/**
Call the registerPlugins method of each PluginFactoryI in theFactories
When called, this method should call the addPlugin method to register any of its plugins.
*/

	private static void registerExternalPlugins() {
		PluginFactoryI		factory;
		int					len, i;

		len = theFactories.size();
		for ( i = 0; i < len; i++ ) {
			factory = (PluginFactoryI) theFactories.elementAt( i );
			factory.registerPlugins( kCurrentInterfaceVersion );
		}
	}

/**
Call the getPluginWatcher method of each PluginFactoryI in theFactories
If these methods return PluginWatcherI objects, they are added to the theWatchers vector.
*/

	private static void initializeWatchers() {
		PluginFactoryI		factory;
		PluginWatcherI		watcher;
		int					len, i;

		len = theFactories.size();
		for ( i = 0; i < len; i++ ) {
			factory = (PluginFactoryI) theFactories.elementAt( i );
			watcher = factory.getPluginWatcher();
			if ( watcher != null )
				theWatchers.addElement( watcher );
		}
	}
	
	private static void registerStockPlugins() {
		addPlugin( new SGIFViewerPlugin(), 0 );
		addPlugin( new SJPEGViewerPlugin(), 0 );
		addPlugin( new SInfoViewerPlugin(), 0 );
	}

	private PluginManager() {
	}
}

