/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.ui;

import com.jniwrapper.*;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.LastErrorException;
import com.jniwrapper.win32.Rect;
import com.jniwrapper.win32.gdi.DC;
import com.jniwrapper.win32.gdi.Region;

/**
 * @author Serge Piletsky
 * @author Alexander Evsukov
 */
public class Wnd extends Handle
{
    private static final FunctionName FUNCTION_FIND_WINDOW = new FunctionName("FindWindow");
    private static final FunctionName FUNCTION_CREATE_WINDOW_EX = new FunctionName("CreateWindowEx");
    private static final String FUNCTION_GET_DESKTOP_WINDOW = "GetDesktopWindow";
    private static final String FUNCTION_UPDATE_WINDOW = "UpdateWindow";
    private static final String FUNCTION_DESTROY_WINDOW = "DestroyWindow";
    private static final String FUNCTION_GET_WINDOW_RECT = "GetWindowRect";
    private static final String FUNCTION_GET_CLIENT_RECT = "GetClientRect";
    private static final String FUNCTION_SET_WINDOW_RGN = "SetWindowRgn";
    private static final String FUNCTION_GET_WINDOW_RGN = "GetWindowRgn";
    private static final String FUNCTION_SET_WINDOW_POS = "SetWindowPos";
    private static final String FUNCTION_MOVE_WINDOW = "MoveWindow";
    private static final String FUNCTION_BEGIN_PAINT = "BeginPaint";
    private static final String FUNCTION_END_PAINT = "EndPaint";
    private static final String FUNCTION_SHOW_WINDOW = "ShowWindow";
    private static final FunctionName FUNCTION_SET_WINDOW_LONG = new FunctionName("SetWindowLong");
    private static final FunctionName FUNCTION_GET_WINDOW_LONG = new FunctionName("GetWindowLong");
    private static final String FUNCTION_SET_LAYERED_WINDOW_ATTRIBUTES = "SetLayeredWindowAttributes";
    private static final String FUNCTION_REDRAW_WINDOW = "RedrawWindow";
    private static final FunctionName FUNCTION_SEND_MESSAGE = new FunctionName("SendMessage");
    private static final String FUNCTION_SET_PARENT = "SetParent";
    private static final String FUNCTION_GET_PARENT = "GetParent";
    private static final String FUNCTION_BRING_WINDOW_TO_TOP = "BringWindowToTop";
    private static final FunctionName FUNCTION_CALL_WINDOW_PROC = new FunctionName("CallWindowProc");
    private static final String FUNCTION_GET_UPDATE_RECT = "GetUpdateRect";

    /*
     * Window Styles
     */
    public static final int WS_OVERLAPPED = 0x00000000;
    public static final int WS_POPUP = 0x80000000;
    public static final int WS_CHILD = 0x40000000;
    public static final int WS_MINIMIZE = 0x20000000;
    public static final int WS_VISIBLE = 0x10000000;
    public static final int WS_DISABLED = 0x08000000;
    public static final int WS_CLIPSIBLINGS = 0x04000000;
    public static final int WS_CLIPCHILDREN = 0x02000000;
    public static final int WS_MAXIMIZE = 0x01000000;
    public static final int WS_CAPTION = 0x00C00000;     /* WS_BORDER | WS_DLGFRAME  */
    public static final int WS_BORDER = 0x00800000;
    public static final int WS_DLGFRAME = 0x00400000;
    public static final int WS_VSCROLL = 0x00200000;
    public static final int WS_HSCROLL = 0x00100000;
    public static final int WS_SYSMENU = 0x00080000;
    public static final int WS_THICKFRAME = 0x00040000;
    public static final int WS_GROUP = 0x00020000;
    public static final int WS_TABSTOP = 0x00010000;

    public static final int WS_MINIMIZEBOX = 0x00020000;
    public static final int WS_MAXIMIZEBOX = 0x00010000;

    /*
     * Common Window Styles
     */
    public static final int WS_OVERLAPPEDWINDOW = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_THICKFRAME | WS_MINIMIZEBOX | WS_MAXIMIZEBOX;
    public static final int WS_POPUPWINDOW = WS_POPUP | WS_BORDER | WS_SYSMENU;
    public static final int WS_CHILDWINDOW = WS_CHILD;

    public static final int WS_TILED = WS_OVERLAPPED;
    public static final int WS_ICONIC = WS_MINIMIZE;
    public static final int WS_SIZEBOX = WS_THICKFRAME;
    public static final int WS_TILEDWINDOW = WS_OVERLAPPEDWINDOW;

    /*
     * Extended Window Styles
     */
    public static final int WS_EX_DLGMODALFRAME = 0x00000001;
    public static final int WS_EX_NOPARENTNOTIFY = 0x00000004;
    public static final int WS_EX_TOPMOST = 0x00000008;
    public static final int WS_EX_ACCEPTFILES = 0x00000010;
    public static final int WS_EX_TRANSPARENT = 0x00000020;
    public static final int WS_EX_MDICHILD = 0x00000040;
    public static final int WS_EX_TOOLWINDOW = 0x00000080;
    public static final int WS_EX_WINDOWEDGE = 0x00000100;
    public static final int WS_EX_CLIENTEDGE = 0x00000200;
    public static final int WS_EX_CONTEXTHELP = 0x00000400;
    public static final int WS_EX_RIGHT = 0x00001000;
    public static final int WS_EX_LEFT = 0x00000000;
    public static final int WS_EX_RTLREADING = 0x00002000;
    public static final int WS_EX_LTRREADING = 0x00000000;
    public static final int WS_EX_LEFTSCROLLBAR = 0x00004000;
    public static final int WS_EX_RIGHTSCROLLBAR = 0x00000000;
    public static final int WS_EX_CONTROLPARENT = 0x00010000;
    public static final int WS_EX_STATICEDGE = 0x00020000;
    public static final int WS_EX_APPWINDOW = 0x00040000;
    public static final int WS_EX_OVERLAPPEDWINDOW = (WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE);
    public static final int WS_EX_PALETTEWINDOW = (WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST);
    public static final int WS_EX_LAYERED = 0x00080000;
    public static final int LWA_ALPHA = 0x00000002;

    /*
     * Flags for SetWindowPos() function
     */
    public static final int SWP_NOSIZE = 0x0001;
    public static final int SWP_NOMOVE = 0x0002;
    public static final int SWP_NOZORDER = 0x0004;
    public static final int SWP_NOREDRAW = 0x0008;
    public static final int SWP_NOACTIVATE = 0x0010;
    public static final int SWP_FRAMECHANGED = 0x0020;  /* The frame changed: send WM_NCCALCSIZE */
    public static final int SWP_SHOWWINDOW = 0x0040;
    public static final int SWP_HIDEWINDOW = 0x0080;
    public static final int SWP_NOCOPYBITS = 0x0100;
    public static final int SWP_NOOWNERZORDER = 0x0200;  /* Don't do owner Z ordering */
    public static final int SWP_NOSENDCHANGING = 0x0400;  /* Don't send WM_WINDOWPOSCHANGING */
    public static final int SWP_DRAWFRAME = SWP_FRAMECHANGED;
    public static final int SWP_NOREPOSITION = SWP_NOOWNERZORDER;
    public static final int SWP_DEFERERASE = 0x2000;
    public static final int SWP_ASYNCWINDOWPOS = 0x4000;

    public static final int HWND_TOP = 0;
    public static final int HWND_BOTTOM = 1;
    public static final int HWND_TOPMOST = -1;
    public static final int HWND_NOTOPMOST = -2;

    /*
     * Window field offsets for GetWindowLong() and GetWindowLong() funcnctions
     */
    public static final int GWL_WNDPROC = -4;
    public static final int GWL_HINSTANCE = -6;
    public static final int GWL_HWNDPARENT = -8;
    public static final int GWL_STYLE = -16;
    public static final int GWL_EXSTYLE = -20;
    public static final int GWL_USERDATA = -21;
    public static final int GWL_ID = -12;

    /*
     * Redraw Window Flags for RedrawWindow() API function
     */
    public static final int RDW_INVALIDATE = 0x0001;
    public static final int RDW_INTERNALPAINT = 0x0002;
    public static final int RDW_ERASE = 0x0004;
    public static final int RDW_VALIDATE = 0x0008;
    public static final int RDW_NOINTERNALPAINT = 0x0010;
    public static final int RDW_NOERASE = 0x0020;
    public static final int RDW_NOCHILDREN = 0x0040;
    public static final int RDW_ALLCHILDREN = 0x0080;
    public static final int RDW_UPDATENOW = 0x0100;
    public static final int RDW_ERASENOW = 0x0200;
    public static final int RDW_FRAME = 0x0400;
    public static final int RDW_NOFRAME = 0x0800;

    /*
     * Commands to use in ShowWindow function.
     */
    public static final int SW_HIDE             = 0;
    public static final int SW_SHOWNORMAL       = 1;
    public static final int SW_NORMAL           = 1;
    public static final int SW_SHOWMINIMIZED    = 2;
    public static final int SW_SHOWMAXIMIZED    = 3;
    public static final int SW_MAXIMIZE         = 3;
    public static final int SW_SHOWNOACTIVATE   = 4;
    public static final int SW_SHOW             = 5;
    public static final int SW_MINIMIZE         = 6;
    public static final int SW_SHOWMINNOACTIVE  = 7;
    public static final int SW_SHOWNA           = 8;
    public static final int SW_RESTORE          = 9;
    public static final int SW_SHOWDEFAULT      = 10;
    public static final int SW_FORCEMINIMIZE    = 11;
    public static final int SW_MAX              = 11;

    public Wnd()
    {
        super();
    }

    public Wnd(long value)
    {
        super(value);
    }

    static Function getFunction(Object functionName)
    {
        return User32.getInstance().getFunction(functionName.toString());
    }

    /**
     * Executes event loop for itself.
     */
    public void eventLoop()
    {
        eventLoop(getValue());
    }

    /**
     * Executes infinite event loop handling cycle for the window specified
     * by the passed handle value.
     *
     * @param hwnd window handle to run event loop for
     */
    public static void eventLoop(final long hwnd)
    {
        if (hwnd == 0)
        {
            throw new IllegalArgumentException("Event processing window is not available");
        }

        final Function getMessage = getFunction(User32.FUNCTION_GET_MESSAGE);
        final Function translateMessage = getFunction(User32.FUNCTION_TRANSLATE_MESSAGE);
        final Function dispatchMessage = getFunction(User32.FUNCTION_DISPATCH_MESSAGE);
        final Wnd hWnd = new Wnd(hwnd);

        final ShortInt result = new ShortInt(1);
        final Msg msg = new Msg();
        final UInt32 nullValue = new UInt32(0);

        while (result.getValue() != 0)
        {
            final Pointer msgPointer = new Pointer(msg);

            getMessage.invoke(result, msgPointer, hWnd, nullValue, nullValue);
            if (result.getValue() == -1)
            {
                break;
            }
            else
            {
                translateMessage.invoke(null, msgPointer);
                dispatchMessage.invoke(null, msgPointer);
            }
        }
    }

    public static Wnd findWindow(String className, String windowName)
    {
        final Function function = getFunction(FUNCTION_FIND_WINDOW);
        Wnd result = new Wnd();
        Parameter wndName = windowName == null ?
                (Parameter) new Handle() : (Parameter) User32.getInstance().stringParam(windowName);
        final ZeroTerminatedString cName = User32.getInstance().stringParam(className);
        function.invoke(result, new Pointer(cName), new Pointer(wndName));
        return result;
    }

    public static Wnd findWindow(String className)
    {
        return findWindow(className, null);
    }

    public static Wnd getDesktopWindow()
    {
        final Function function = getFunction(FUNCTION_GET_DESKTOP_WINDOW);
        Wnd result = new Wnd();
        function.invoke(result);
        return result;
    }

    private static Wnd createWindow(Object functionName, Parameter[] parameters)
    {
        Wnd result = new Wnd();
        final Function function = getFunction(functionName);
        function.invoke(result, parameters);
        if (result.getValue() == 0)
        {
            throw new LastErrorException("Cannot create window.");
        }
        return result;
    }

    /**
     * Creates new window.
     *
     * @param className registered window class name
     * @param windowName window name
     * @param windowStyle
     * @param x horizontal position of window
     * @param y vertical position of window
     * @param nWidth window width
     * @param nHeight window height
     * @param hWndParent handle to parent or owner window
     * @param hMenu menu handle or child identifier
     * @param hInstance handle to application instance
     * @param lpParam window-creation data
     * @return instance of newly created window
     */
    public static Wnd createWindow(String className,
                                   String windowName,
                                   int windowStyle,
                                   long x,
                                   long y,
                                   long nWidth,
                                   long nHeight,
                                   Wnd hWndParent,
                                   Handle hMenu,
                                   Handle hInstance,
                                   Handle lpParam)
    {
        return createWindow(FUNCTION_CREATE_WINDOW_EX,
                new Parameter[]{
                    new UInt(0),
                    new Pointer(User32.getInstance().stringParam(className)), // registered class name
                    new Pointer(User32.getInstance().stringParam(windowName)),
                    new Int(windowStyle), // window style
                    new Int(x), // horizontal position of window
                    new Int(y), // vertical position of window
                    new Int(nWidth), // window nWidth
                    new Int(nHeight), // window nHeight
                    hWndParent, // handle to parent or owner window
                    hMenu, // menu handle or child identifier
                    hInstance, // handle to application instance
                    lpParam  // window-creation data
                });
    }

    /**
     * Creates new window with extended style.
     *
     * @param windowStyleEx extended window style
     * @param className registered window class name
     * @param windowName window name
     * @param windowStyle
     * @param x horizontal position of window
     * @param y vertical position of window
     * @param nWidth window width
     * @param nHeight window height
     * @param hWndParent handle to parent or owner window
     * @param hMenu menu handle or child identifier
     * @param hInstance handle to application instance
     * @param lpParam window-creation data
     * @return instance of newly created window
     */
    public static Wnd createWindow(int windowStyleEx,
                                   String className,
                                   String windowName,
                                   int windowStyle,
                                   long x,
                                   long y,
                                   long nWidth,
                                   long nHeight,
                                   Wnd hWndParent,
                                   Handle hMenu,
                                   Handle hInstance,
                                   Handle lpParam)
    {
        return createWindow(FUNCTION_CREATE_WINDOW_EX,
                new Parameter[]{
                    new Int32(windowStyleEx),
                    new Pointer(User32.getInstance().stringParam(className)),
                    new Pointer(User32.getInstance().stringParam(windowName)),
                    new Int(windowStyle),
                    new Int(x),
                    new Int(y),
                    new Int(nWidth),
                    new Int(nHeight),
                    hWndParent,
                    hMenu,
                    hInstance,
                    lpParam
                });
    }

    public static Wnd createWindow(String className)
    {
        Handle NULL = new Handle();
        Wnd parent = new Wnd();
        return createWindow(0, className, className, 0, 0, 0, 0, 0, parent, NULL, NULL, NULL);
    }

    public void update()
    {
        final Function function = getFunction(FUNCTION_UPDATE_WINDOW);
        Int result = new Int();
        function.invoke(result, this);
        if (result.getValue() == 0)
        {
            throw new LastErrorException("Error updating window.");
        }
    }

    public void destroy()
    {
        final Function function = getFunction(FUNCTION_DESTROY_WINDOW);
        Int result = new Int();
        function.invoke(result, this);
        if (result.getValue() == 0)
        {
            throw new LastErrorException("Error destroying window.");
        }
    }

    public static void getWindowRect(Wnd handle, Int width, Int height)
    {
        getWindowRect(handle, new Int(), new Int(), width, height);
    }

    public static void getWindowRect(Wnd handle, Int top, Int left, Int width, Int height)
    {
        final Function function = getFunction(FUNCTION_GET_WINDOW_RECT);
        function.invoke(null, handle,
                new Pointer.OutOnly(new Structure(new Parameter[]{top, left, width, height})));
    }

    public static void getWindowRect(Wnd handle, Rect rect)
    {
        final Function function = getFunction(FUNCTION_GET_WINDOW_RECT);
        function.invoke(null, handle, new Pointer.OutOnly(rect));
    }

    public static void centerWindow(Wnd handle)
    {
        Rect desktop = new Rect();
        getWindowRect(Wnd.getDesktopWindow(), desktop);
        Rect windowRect = new Rect();
        getWindowRect(handle, windowRect);
        final long deltaX = (desktop.getRight() - windowRect.getRight() - windowRect.getLeft()) / 2;
        final long deltaY = (desktop.getBottom() - windowRect.getBottom() - windowRect.getTop()) / 2;
        windowRect.moveBy((int)deltaX, (int)deltaY);
        setWindowPos(handle, new Wnd(0),
                windowRect.getLeft(),
                windowRect.getTop(),
                windowRect.getRight(),
                windowRect.getBottom(),
                SWP_NOACTIVATE | SWP_NOSIZE | SWP_NOZORDER);
    }

    public static boolean getClientRect(Wnd handle, Rect rect)
    {
        Int result = new Int();
        final Function function = getFunction(FUNCTION_GET_CLIENT_RECT);
        function.invoke(result, handle, new Pointer.OutOnly(rect));
        if (result.getValue() > 0)
        {
            return true;
        }
        return false;
    }

    public static void setWindowRgn(Wnd hWnd, Region hRgn, boolean bRedraw)
    {
        final Function function = getFunction(FUNCTION_SET_WINDOW_RGN);
        function.invoke(null, hWnd, hRgn == null ? new Handle() : hRgn, new Bool(bRedraw));
    }

    public void setWindowRegion(Region region)
    {
        setWindowRgn(this, region, true);
    }

    public static Region getWindowRgn(Wnd hWnd)
    {
        final Region result = new Region();
        final Function function = getFunction(FUNCTION_GET_WINDOW_RGN);
        function.invoke(null, hWnd, result);
        return result;
    }

    public static boolean setWindowPos(Wnd hWnd,
                                       Wnd hWndInsertAfter,
                                       long x,
                                       long y,
                                       long cx,
                                       long cy,
                                       long flags)
    {
        final Function function = getFunction(FUNCTION_SET_WINDOW_POS);
        Bool result = new Bool();
        function.invoke(result,
                new Parameter[]
                {
                    hWnd,
                    hWndInsertAfter,
                    new Int(x),
                    new Int(y),
                    new Int(cx),
                    new Int(cy),
                    new UInt(flags)
                });
        return result.getValue();
    }

    public DC beginPaint(PaintStruct paintStruct)
    {
        final Function beginPaint = getFunction(FUNCTION_BEGIN_PAINT);
        DC result = new DC();
        beginPaint.invoke(result, this, new Pointer.OutOnly(paintStruct));
        if (result.isNull())
        {
            throw new LastErrorException("Failed to get DC.", true);
        }
        return result;
    }

    public void endPaint(PaintStruct paintStruct)
    {
        final Function endPaint = User32.getInstance().getFunction(FUNCTION_END_PAINT);
        endPaint.invoke(null, this, new Pointer.Const(paintStruct));
    }

    public boolean sendMessage(int msg, int wParam, int lParam)
    {
        final Function sendMessage = getFunction(FUNCTION_SEND_MESSAGE);
        Bool result = new Bool();
        sendMessage.invoke(result, this, new UInt(msg), new UInt16(wParam), new UInt16(lParam));
        return result.getValue();
    }

    public void show(int nCmdShow)
    {
        final Function show = getFunction(FUNCTION_SHOW_WINDOW);
        show.invoke(null, this, new Int(nCmdShow));
    }

    public void setWindowLong(int nIndex, long dwNewLong)
    {
        setWindowLong(nIndex, new UInt(dwNewLong));
    }

    public long setWindowLong(int nIndex, Parameter newValue)
    {
        UInt result = new UInt();
        final Function setWindowLong = getFunction(FUNCTION_SET_WINDOW_LONG);
        setWindowLong.invoke(result, this, new Int(nIndex), newValue);
        return result.getValue();
    }

    public long getWindowLong(int nIndex)
    {
        final Function getWindowLong = getFunction(FUNCTION_GET_WINDOW_LONG);
        UInt result = new UInt();
        getWindowLong.invoke(result, this, new Int(nIndex));
        return result.getValue();
    }

    public void setLayeredWindowAttributes(int crKey, byte bAlpha, int dwFlags)
    {
        final Function setLayeredWindowAttr = getFunction(FUNCTION_SET_LAYERED_WINDOW_ATTRIBUTES);
        setLayeredWindowAttr.invoke(null, this, new Int(crKey), new Int8(bAlpha), new UInt(dwFlags));
    }

    public void redraw(Handle lprcUpdate, Handle hrgnUpdate, int flags)
    {
        final Function redrawWindow = getFunction(FUNCTION_REDRAW_WINDOW);
        redrawWindow.invoke(null, this, lprcUpdate, hrgnUpdate, new UInt(flags));
    }

    public void setParent(Wnd parent)
    {
        final Function setParent = getFunction(FUNCTION_SET_PARENT);
        setParent.invoke(null, this, parent);
    }

    public Wnd getParent()
    {
        final Function getParent = getFunction(FUNCTION_GET_PARENT);
        Wnd result = new Wnd();
        getParent.invoke(result, this);
        return result;
    }

    public void bringToTop()
    {
        final Function function = getFunction(FUNCTION_BRING_WINDOW_TO_TOP);
        function.invoke(null, this);
    }

    public static long callWindowProc(Handle wndProc, Wnd wnd, UInt msg, UInt32 wParam, UInt32 lParam)
    {
        final Function function = getFunction(FUNCTION_CALL_WINDOW_PROC);
        UInt32 result = new UInt32();
        function.invoke(result, new Parameter[]
        {
            wndProc,
            wnd,
            msg,
            wParam,
            lParam
        });
        return result.getValue();
    }

    public static boolean moveWindow(Wnd wnd, int x, int y, int width, int height, boolean repaint)
    {
        final Function function = getFunction(FUNCTION_MOVE_WINDOW);
        Bool result = new Bool();
        function.invoke(result, new Parameter[]
        {
            wnd,
            new Int(x),
            new Int(y),
            new Int(width),
            new Int(height),
            new Bool(repaint),
        });
        return result.getValue();
    }

    public boolean moveWindow(int x, int y, int width, int height, boolean repaint)
    {
        return moveWindow(this, x, y, width, height, repaint);
    }

    public boolean moveWindow(int x, int y, int width, int height)
    {
        return moveWindow(this, x, y, width, height, false);
    }

    public static boolean getUpdateRect(Wnd wnd, Rect rect, boolean erase)
    {
        final Function function = getFunction(FUNCTION_GET_UPDATE_RECT);
        Bool result = new Bool();
        function.invoke(result, wnd, new Pointer.OutOnly(rect), new Bool(erase));
        return result.getValue();
    }

    public boolean getUpdateRect(Rect rect, boolean erase)
    {
        return getUpdateRect(this, rect, erase);
    }

    public boolean getUpdateRect(Rect rect)
    {
        return getUpdateRect(this, rect, false);
    }
}
