/*
 *    The term 'program' used in the copyright and the license below means
 *    PReP Version 1.0 
 *
 *    Copyright (C) 1998, 1999 DELECis
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *    This copyright notice must appear in every copy and modification.
 *
 *    Contact: prep@delec.com
 *
 *    Read the manual: http://www.delec.com/is/products/prep/manual/jprep/
 */
 
/**
 * (PReP Version 1.0)
 * 
 * PReP: reads arguments from the commandline which are either
 *       tags and values or input files and an output file and
 *       stores them into a TagList, compiles it and outputs the
 *       result in the PrintWriter
 *
 *       The result is a text like the prep file defines
 *
 * History: - Creation nh
 *
 * @author nh
 * @version 1.0
 * @see PRePMain, TagList, PRePTag
 */
 
 
package com.delec.prep;

import com.delec.arguments.*;
import java.io.*;

public class PReP implements ArgumentHandler
{
	protected PRePHandler prepHandler;
	protected final String URL = "jdbc:odbc:";
	protected boolean booOutputFile = false;
	protected PRePMain prepMain = null;
	protected boolean bDBSet = false;

	public static final int SECURITY = 0;
	private static final boolean LOG = false;

	public PReP(String[] strArgs)
	{

		if ( strArgs.length == 0 )
		{
			printHelp();
		}
		else
		{
			this.prepHandler = provideHandler();
			this.prepHandler.createSecurityHashtable();
			this.prepMain = prepHandler.getPRePMain();
			getSpecialTags(prepMain);
			Arguments arguments = new Arguments(strArgs);
			fillSpecialArgs(arguments);
			arguments.process(this);

			if ( bDBSet == false )
			{
				setDBConnection();
			}

			boolean bLoginOk = this.prepHandler.checkCookie(this.prepMain);
			this.prepHandler.getNewCookies(this.prepMain);

			if ( bLoginOk )
			{
				this.prepHandler.loginOk(this.prepMain);
			}
			else
			{
				this.prepHandler.loginNotOk(this.prepMain);
			}

			this.prepHandler.finishHandle(this.prepMain);
			this.prepHandler.destroy();
			this.prepHandler = null;
		}
	}

	protected void getSpecialTags(PRePMain prepMain)
	{
	}

	protected PRePHandler provideHandler()
	{
		return new PRePHandler(SECURITY);
	}

	/**
	* this class provides all keys from 1 to 999 - each subclass will have
	* its own keys always a 1000 bigger than its superclass
	*/
	protected void fillSpecialArgs(Arguments arguments)
	{
		arguments.put("DB", new Integer(1));
	}

	protected void setDBConnection()
	{
		//this.prepHandler.setDatabaseConnection();
		//does nothing, has to be over written in a subclass	   
	}


	/**
	* this handles the normal arguments which consists of a name and
	* a value (-tagName tagValue)
	*/
	public void handleNormalArg(Argument argument, Arguments arguments)
	{
		if ( LOG )
		{
			System.out.println("handleNormalArg "+argument.getName()+" "+argument.getValue());
		}

		PRePTagText prepTagText = new PRePTagText(argument.getName());
		arguments.consume();
		prepTagText.setValue(arguments.getArgument().getName());
		prepHandler.putField(prepMain, prepTagText);
	}


	/**
	* this handles the special args which consists of a name - that means
	* this argument must be a filename
	*/
	public void handleSpecialArg(Argument argument, Arguments arguments)
	{
		handleFileArg(arguments, argument);     
	}

	/**
	* this handles the arguments which are in a list with special arguments.
	* (e.g. DB tag)
	*/
	public boolean handleArgInList(int iKey, Arguments args)
	{
		boolean booWasInList = false;
		switch ( iKey )
		{
			// DB tag
			case 1:

				handleDBTag(args);
				booWasInList = true;
				break;
		}
		// no more special tags available yet...
		return booWasInList;
	}

	/**
	* this handles the filenames - check first if the output file is already
	* defined (must be the last argument). Then call a putFile() with this
	* file
	*/
	protected void handleFileArg(Arguments arguments, Argument argument)
	{
		if ( LOG )
		{
			System.out.println("handleFileArg: "+argument.getName()+" "+argument.getValue());
		}

		if ( !booOutputFile ) // output file not yet defined (must be the last argument)
		{
			handleOutputFileArg(arguments);
			booOutputFile = true;
		}

		if ( !(argument.getName().equals(arguments.getLastArgument().getName())) )
		{

			if ( LOG )
			{
				System.out.println("normal file");
			}

			File file = new File(argument.getName());
			prepHandler.putFile(prepMain, file);
		}
	}


	/**
	* this handles the output file which must be the last argument
	*/
	protected void handleOutputFileArg(Arguments arguments)
	{
		File file = new File(arguments.getLastArgument().getName());
		FileWriter fWriter;
		PrintWriter pWriter;

		if ( LOG )
		{
			System.out.println("handleOutputFileArg: "+file.getPath());
		}

		try
		{
			fWriter = new FileWriter(file);
			pWriter = new PrintWriter(fWriter);
			prepHandler.setPrintWriter(pWriter, prepMain);
		}
		catch ( IOException e )
		{
			// prepHandler.handleException("PReP/handleOutputFileArg()", e);
			System.out.println("IOException while creating printwriter: "+e);
		}

	}


	/**
	* handles the DB tag and initialises the connectionPool in the PRePHandler class
	* with the specified values. This tag is in the list for the special tags
	*/
	protected void handleDBTag(Arguments args)
	{
		int iPoolConnections = 1;
		String strJDBCDRIVER = null;
		String strURL = null;
		String strDBUSER = null;
		String strDBPWD = null;

		args.consume();
		Argument arg = args.getArgument();

		while ( arg != null && arg.isNameValuePair() )
		{

			if ( arg.getName().toLowerCase().equals("drv") )
			{
				strJDBCDRIVER = arg.getValue();
			}
			else if ( arg.getName().toLowerCase().equals("path") )
			{
				strURL = arg.getValue();
			}
			else if ( arg.getName().toLowerCase().equals("uid") )
			{
				strDBUSER = arg.getValue();
			}
			else if ( arg.getName().toLowerCase().equals("pwd") )
			{
				strDBPWD = arg.getValue();
			}
			else if ( arg.getName().toLowerCase().equals("odbc") )
			{
				strURL = URL+arg.getValue();
			}
			else
			{
			}

			args.consume();
			arg = args.getArgument();
		}

		/*if (strJDBCDRIVER.equals("")) 
			{
			strJDBCDRIVER = prepHandler.getJDBCDRIVER();
		}
		if (strURL.equals("")) 
			{
			strURL = prepHandler.getURL();
		}
		if (strDBUSER.equals("")) 
			{
			strDBUSER = prepHandler.getUID();
		}
		if (strDBPWD.equals("")) 
			{
			strDBPWD = prepHandler.getPWD();
		}*/
		//bDBSet = prepHandler.setDatabaseConnection(strJDBCDRIVER, strURL, strDBUSER, strDBPWD);

		//set connectionPool to one connection only
		bDBSet = prepHandler.setDatabaseConnection(strJDBCDRIVER, strURL, strDBUSER, strDBPWD, iPoolConnections);
	}


	/**
	* prints a help if no arguments exist
	*/
	protected void printHelp()
	{
		System.out.println("");
		System.out.println("");
		System.out.println("**********************************************************************************************");
		System.out.println("*                                  PReP Usage                                                *");
		System.out.println("**********************************************************************************************");
		System.out.println("");
		System.out.println("java com.delec.prep.PReP {-tagName tagValue} PRePFile {PRePFile| -tagName tagValue} outputFile");
		System.out.println("");
		System.out.println("-Tag name:    you may specify as many tags as you want to but after ");
		System.out.println("              a tag name must follow a tag value!");
		System.out.println("");
		System.out.println("Tag value:    the tag value to the specified tag name");
		System.out.println("");
		System.out.println("PRePFile:     you may specify as many input files as you want to but");
		System.out.println("              at least one");
		System.out.println("");
		System.out.println("HTML File:    you have to specify exactly one output file, which ");
		System.out.println("              normally is a HTML file and must be at the end");
		System.out.println("");
		System.out.println("-DB tag:      if you want to establish a database connection, you ");
		System.out.println("              will use this DB tag which needs 4 values: ");
		System.out.println("              \"JDBCDriver\" \"URL\" \"Username\" \"Password\" ");
		System.out.println("              you can also use the nameValue pair: odbc='name' where");
		System.out.println("              'name' is the name of the database (e.g. doc)"); 
		System.out.println("");
		System.out.println("*********************************************************************************************");
	}

	/**
	 * main method
	 */

	public static void main(String argv[])
	{
		PReP prep = new PReP(argv);
		prep = null;
		//Systems.exit(0);
	}
}
