
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;


/**
  * An if directive is used to include a block only when a condition 
  * is true. Structurally it has a condition, followed by a block, 
  * followed by an optional "#else" statmeent and another block.
  * 
  * Here is an example:<pre>
  *
  *     #if ($userID) {
  *        Your user id is: $userID
  *     } else {
  *        &lt;a href="register"&gt;Click here to register&lt;/a&gt;
  *     }
  *
  * </pre> The condition is true if it evaluates to a non-null value. The 
  * #if block will be included if (and only if) the condition evaluates
  * to a non-null value. The #else block will be included if (and
  * only if) the condition evaluates to null.
  * <p>
  * When evaluating whether a condition is null or not, the IfDirective
  * will check whether it is a Macro. If it is a Macro, it calls 
  * Macro.evaluate() and checks whether the return value is null. If 
  * it is not a macro, then it is true or false simply based on whether
  * the value passed in was actually null itself.
  * <p>
  * @see Macro
  */
final class IfDirective implements Macro
{

   /**
     * We are true if this object is defined and not null
     */
   final private Condition myCondition;

   /**
     * This is the block we include if myCondition is not null
     */
   final private Macro myIfBlock;

   /**
     * This is the block we include if myCondition is null
     */
   final private Macro myElseBlock;

   /**
     * debug messages
     */
   final private static boolean _debug = Engine.debug;

   /**
     * Create a new IfCondition
     * <p>
     * @param condition True when this evaluates to a non-Null value
     * @param ifBlock included when condition is non-null
     * @param elseBlock (may be null) included when condition is null
     */
   IfDirective(Condition cond, Object ifBlock, Object elseBlock) 
   { 
       // FIXME: not guaranteed to be a macro, may be string
       myCondition = cond; 
       myIfBlock = (Macro) ifBlock;
       myElseBlock = (Macro) elseBlock;
   }

   /**
     * Parse the IfDirective as described in the class comment
     * <p>
     * @exception ParseException if the sytax was invalid and we could not recover
     * @exception IOException if we could not successfullly read the parseTool
     */
   static public final Object parse(ParseTool in) 
      throws IOException, ParseException
   {
      Condition cond;
      Object ifBlock;
      Object elseBlock = null;
      
      // detect if
      if ((in.ttype != in.TT_WORD) && (!in.sval.equals("if"))) {
         if (_debug) {
            Engine.log.debug("If: TT_WORD parsed and sval is 'if'");
         }
         in.pushBack();
         return null;
      }

      in.nextToken();

      // if at space remove spaces 
      in.parseWhitespace();

      // detect conditional bracket
      if (in.ttype != '(') {
         throw new ParseException(in,
            "expected a /(/ in IfDirective, got:" + (char) in.ttype);
      }
      
      // call Expression. parse
      cond = Condition.parseCond(in);

      // look for Whitespaces
      in.parseWhitespace();
      
      // detect and parse block 
      if (in.ttype != '{') {
         throw new ParseException(in,
            "expected an if Block, got:" + (char) in.ttype);
      }
      ifBlock = (Object) Block.parse(in);

      // eat spaces after the brace
      in.parseSpaces();
      in.parseEOL();

      if (in.parseChar('#')) {
         // #else (later on include #else if) may only appear after an #if
         if (in.sval.equals("else")) { 
            in.nextToken();
            
            // look for Whitespaces
            in.parseWhitespace();

            // detect and parse block 
            if (in.ttype != '{') {
               throw new ParseException(in,
                  "expected a else Block, got:" + (char) in.ttype);
            }
            elseBlock = (Object) Block.parse(in);
            in.parseSpaces();
            in.parseEOL();
         } else {
            in.pushBack();
         }
      }
      return new IfDirective(cond, ifBlock, elseBlock);
   }


   /**
     * Evaluate the current macro and return it as a string. Same
     * basic operation as calling write.
     * @exception InvalidContextException is required data is missing
     */ 
   public Object evaluate(Object context)
      throws InvalidContextException
   {
      try {
         StringWriter sw = new SizedStringWriter(512);
         write(sw,context);
         return sw.toString();
      } catch (IOException e) {
         Engine.log.exception(e);
         Engine.log.error(
               "If: evaluate got IO exception on write to StringWriter");
         return "";
      }
   }  

   /**
     * Interpret the directive and write it out to the supplied stream.
     * <p>
     * @exception InvalidContextException if required data was missing from context
     * @exception IOException if we could not successfully write to out
     */
   public void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {

      if (_debug) {
         Engine.log.debug("If: evaluating #if condition:" + myCondition);
      }

      if (myCondition.test(context)) 
      {
         if (_debug) {
            Engine.log.debug("If: writing myIfBlock");
         }
	 if (myIfBlock != null) {
            myIfBlock.write(out, context);
         } else {
	    Engine.log.warning("If: Block for an #if directive was null");
	 }
      } else { 
	 if (myElseBlock != null) {
            if (_debug) Engine.log.debug("If: writing myElseBlock: " + myCondition);
            myElseBlock.write(out, context);
	 }
      } 
   }

   /**
     * Test harness
     */
   public static void main(String arg[]) {
      
      
      System.out.println("Testing IncludeDirective:");

      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
   
   
      Map m = new HashMap();
   
      m.put("name","some content");
      m.put("false", Boolean.FALSE);
      m.put("true", Boolean.TRUE);


      String tests[] = {
            "justIf"
         };
   
      
      System.out.println("- - - - - - - - - - - - - - - - -");
      for (int i = 0; i < tests.length; i++) {
         File inFile = new File("examples", tests[i] + ".wm");
         File outFile = new File("examples", tests[i] + ".html");
         System.out.println("TESTING input file: " + inFile);
         try {
            ParseTool iin = new ParseTool(inFile);
            iin.nextToken();
            iin.nextToken();
            Block id = (Block) Block.parse(iin);
            if (id == null) {
               System.out.println("FAILED--null returned from parse().");
            } else {
               Writer out = new FileWriter(outFile);
               id.write(out, m); 
	       out.close();
            }
         } catch (Exception e) {
            System.out.println("FAILED--threw exception:");
            e.printStackTrace();
            System.out.println();
         }
         System.out.println("- - - - - - - - - - - - - - - - -");
      }
      System.out.println();
      System.out.println("Done.");

   } 
  
}


