/*
 * SHADOW.C
 *
 * Shadow masks and simple sprite animation.
 *
 * This example shows:
 * o  Creating a mask with multiple transparent colours.
 * o  Preparing the board for luma masks and creating a luma levels.
 * o  Setting the opacity level of the sprite, so that the shadow
 *    colours are considered transparent.
 *
 * Compiling:
 *   Borland C++, 16-bit
 *      bcc -W shadow.c as16.lib
 *
 *   Borland C++, 32-bit
 *      bcc32 -tW shadow.c as32b.lib
 *
 *   Microsoft C/C++, 16 bit
 *      cl -Gw shadow.c as16.lib slibcew.lib libw.lib example.def
 *      rc example.rc shadow.exe
 *
 *   Microsoft C/C++, 32 bit
 *      cl -GA shadow.c as32m.lib user32.lib gdi32.lib
 *
 *   Watcom C/C++ 11.0, 32-bit
 *      wcl386 /l=nt_win shadow.c as32w.lib
 *
 * Copyright (c) 1997-1999, ITB CompuPhase. You may use/modify/distribute
 * this file or portions of it. It is provided as an example for the use of
 * the AniSprite API. There are no waranties on the correct behavior of this
 * program.
 */
#include <stdlib.h>
#include <windows.h>
#include "..\include\anispri.h"

#define BOARD_WIDTH     640
#define BOARD_HEIGHT    480
#define TIMER_INTERVAL  50
#define AIRPLANES       3

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASS wc;
  RECT rect;
  DWORD dwStyle;

  wc.style = 0;
  wc.lpfnWndProc = (WNDPROC)AnimWinFunc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, (LPSTR)"anim_icon");
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(LTGRAY_BRUSH);
  wc.lpszMenuName = (LPSTR)NULL;
  wc.lpszClassName = "Animation";
  if (!RegisterClass(&wc))
    return FALSE;

  /* creat a window with the right size for the board */
  SetRect(&rect, 0, 0, BOARD_WIDTH, BOARD_HEIGHT);
  dwStyle = WS_POPUPWINDOW | WS_CAPTION | WS_VISIBLE;
  AdjustWindowRect(&rect, dwStyle, FALSE);
  CreateWindow("Animation", "AniSprite: Shadow mask", dwStyle,
               50, 50, rect.right - rect.left, rect.bottom - rect.top,
               0, 0, hInstance, NULL);

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  } /* while */

  return msg.wParam;
}

static int irand(int top)
{
  return rand() % top;
}

LRESULT CALLBACK AnimWinFunc(HWND hwnd, unsigned message,
                             WPARAM wParam, LPARAM lParam)
{
static COLORREF crTransColor[] = {
    PALETTEINDEX(251), PALETTEINDEX(252), PALETTEINDEX(253)
};
static ASBOARD Board;
static ASPRITE Sprites[AIRPLANES];
static int stepx[AIRPLANES], stepy[AIRPLANES];
  LPBITMAPINFO lpBitsInfo;
  LPVOID lpBits, lpMask;
  PAINTSTRUCT ps;
  RECT rect;
  HDC hdc;
  int x, y, i;
  LPVOID lpLuma;
  char filename[64];

  switch (message) {
  case WM_CREATE:
    /* lpBitsInfo is allocated temporarily */
    lpBitsInfo = as_AllocResource(sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));

    /* create the board (with an identity palette) */
    lpBits = as_LoadDIB("map.bmp", lpBitsInfo, NULL, NULL);
    Board = as_CreateBoard(lpBitsInfo, lpBits, NULL,
                           AS_MODE_PAL_COLORS | AS_MODE_LUMA);
    as_SetPalette(Board, lpBitsInfo->bmiColors, TRUE);

    /* add 2 luma levels to simulate shadows; hard shadows at 50% "darkness"
     * and soft shadows at 25% "darkness" (75% luminance) */
    lpLuma = as_CreateLumaLevel(lpBitsInfo->bmiColors, RGB(0,0,0), 127); /* 50% luma */
    as_SetBoardData(Board, AS_DATA_LUMA_LEVEL, 1, lpLuma);
    lpLuma = as_CreateLumaLevel(lpBitsInfo->bmiColors, RGB(0,0,0), 63);  /* 25% luma */
    as_SetBoardData(Board, AS_DATA_LUMA_LEVEL, 2, lpLuma);

    /* create the sprite */
    for (i=0; i<AIRPLANES; i++) {
      wsprintf(filename, "plane%d.bmp", i+1);
      lpBits = as_LoadDIB(filename, lpBitsInfo, NULL, NULL);
      lpMask = as_CreateMask(lpBitsInfo, lpBits, TRUE, crTransColor, 3);
      Sprites[i] = as_Create((int)lpBitsInfo->bmiHeader.biWidth,
                             (int)lpBitsInfo->bmiHeader.biHeight,
                             lpMask, AS_MASK_LUMA, lpBits, TRUE);

      /* Set the opacity level to 0. Without this line, the sprite
       * would bounce when its shadow touches the window edge. We want
       * it to bounce only when the "opaque" part of the sprite touches
       * the window edge.
       */
      as_SetValue(Sprites[i], AS_VALUE_OPAQUE_LEVEL, 0);

      /* assign and show the sprite */
      as_Assign(Sprites[i], Board, 0);
      as_Show(Sprites[i], TRUE);

      as_SetPos(Sprites[i], -BOARD_WIDTH/2 + irand(3*BOARD_WIDTH/2), BOARD_HEIGHT);
      stepx[i] = 2 + irand(3);
      stepy[i] = -2 - irand(3);
    } /* for */

    /* clean up */
    as_FreeResource(lpBitsInfo);

    /* Create a timer to move the sprite, set initial direction */
    as_SetTimer(Board, hwnd, TIMER_INTERVAL);
    break;

  case WM_DESTROY:
    /* the sprite */
    for (i=0; i<AIRPLANES; i++) {
      as_Assign(Sprites[i], Board, -1);
      as_Delete(Sprites[i], TRUE);
    } /* for */
    /* the luma levels */
    for (i = 1; i <= 2; i++) {
      lpLuma = as_GetBoardData(Board, AS_DATA_LUMA_LEVEL, i);
      as_FreeResource(lpLuma);
    } /* for */
    /* the board */
    as_DeleteBoard(Board, TRUE);
    /* the rest */
    PostQuitMessage(0);
    break;

  case AS_TIMER:
    GetClientRect(hwnd, &rect);
    for (i=0; i<AIRPLANES; i++) {
      /* update position */
      x = as_GetValue(Sprites[i], AS_VALUE_XPOS) + stepx[i];
      y = as_GetValue(Sprites[i], AS_VALUE_YPOS) + stepy[i];
      as_SetPos(Sprites[i], x, y);

      if (!as_CollideBox(Sprites[i], &rect) && y < 0) {
        as_SetPos(Sprites[i], -BOARD_WIDTH/2 + irand(3*BOARD_WIDTH/2), BOARD_HEIGHT);
        stepx[i] = 2 + irand(3);
        stepy[i] = -2 - irand(3);
      } /* if */
    } /* for */

    /* repaint the board */
    hdc = GetDC(hwnd);
    as_PaintBoard(hdc, Board, 0, 0, FALSE);
    ReleaseDC(hwnd, hdc);
    break;

  case WM_PAINT:
    hdc=BeginPaint(hwnd, &ps);
    as_PaintBoard(hdc, Board, 0, 0, TRUE);
    EndPaint(hwnd, &ps);
    break;

  case WM_ERASEBKGND:
  case WM_PALETTECHANGED:
  case WM_QUERYNEWPALETTE:
    return as_ForwardMessage(Board, hwnd, message, wParam, lParam);

  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  } /* switch */

  return 0L;
}

