// WindowLog.cpp: implementation of the WindowLog class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "WindowLog.h"

//////////////////////////////////////////////////////////////////////
// Static Utilities
//////////////////////////////////////////////////////////////////////

#define LOG_FONT_HEIGHT 14

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

WindowLog::WindowLog(HWND hWnd, int length)
{
	this->window = hWnd;
	this->font = 
		CreateFont(LOG_FONT_HEIGHT, 0, 0, 0, 
					FW_NORMAL, 
					false, false, false, 
					ANSI_CHARSET, 
					OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, 
					DEFAULT_QUALITY,
					FIXED_PITCH | FF_MODERN, 
					NULL);

	this->text = (char*)malloc(length + 1);
	this->index = 0;
	this->first = 0;
	this->next = 0;
	this->length = length;
	text[0] = '\0';
	SetScrollPos(hWnd, SB_VERT, 0, TRUE);
	RedrawWindow(hWnd, NULL, NULL, 
				 RDW_INTERNALPAINT | RDW_INVALIDATE | RDW_ERASE | RDW_UPDATENOW);
}

WindowLog::~WindowLog()
{
	free(text);
	DeleteObject(font);
}

//////////////////////////////////////////////////////////////////////
// Log Output Functions
//////////////////////////////////////////////////////////////////////

void WindowLog::print(const char *msg)
{
	int count = 0;

	count = _snprintf(&text[next], length - next, "%s\n", msg);
	while (count < 0) {
		// Ran out of buffer space
		// Allocate a new buffer, three times the previous size
		char *new_text;
		
		length = 3 * length;
		new_text = (char*)malloc(length + 1);
		memcpy(new_text, text, next);
		free(text);
		text = new_text;
		// Now try again
		count = _snprintf(&text[next], length - next, "%s\n", msg);
	}
	text[next + count] = '\0';
	next += count;

	make_new_line_visible();
	SetScrollPos(window, SB_VERT, log_scroll_position(), TRUE);
	RedrawWindow(window, NULL, NULL, 
				 RDW_INTERNALPAINT | RDW_INVALIDATE | RDW_ERASE | RDW_UPDATENOW);
}

const char* WindowLog::get_text()
{
	return text;
}

/*
bool WindowLog::save_to_file(const char *path)
{
	int fd = _creat(path, _S_IREAD | _S_IWRITE);
	
	if (fd == -1) {
		return false;
	}
	else {
		int ix = 0;
		int retry = 3;
		
		while (ix < next) {
			int count = _write(fd, &text[ix], next - ix);
			
			if (count == -1) {
				// Error, try again
				retry--;
				if (retry < 0) {
					break;
				}
			}
			else {
				ix += count;
				retry = 3;
			}
		}
		_close(fd);
		return (ix == next);
	}
}
*/

//////////////////////////////////////////////////////////////////////
// Window Control Functions
//////////////////////////////////////////////////////////////////////

bool WindowLog::do_paint(HWND hWnd)
{
	if (hWnd == window) {
		HGDIOBJ default_font = NULL;
		RECT rt;
		PAINTSTRUCT ps;
		HDC hdc = BeginPaint(hWnd, &ps);
		
		if (font != NULL)
			default_font = SelectObject(hdc, font);
		
		GetClientRect(hWnd, &rt);
		DrawText(hdc, &text[index], next - index, &rt, DT_LEFT);
		
		if (default_font != NULL)
			SelectObject(hdc, default_font);
		
		EndPaint(hWnd, &ps);
		return true;
	}
	else {
		return false;
	}
}

bool WindowLog::do_vscroll(HWND hWnd, 
						   int code,
						   short int pos)
{
	if (hWnd == window) {
		switch (code) {
		case SB_LINEDOWN:
			move_log_index(1);
			break;
		case SB_LINEUP:
			move_log_index(-1);
			break;
		case SB_PAGEDOWN:
			move_log_index(log_page_size() - 1);
			break;
		case SB_PAGEUP:
			move_log_index(-(log_page_size() - 1));
			break;
		case SB_BOTTOM:
			index = next;
			break;
		case SB_TOP:
			index = 0;
			break;
		case SB_THUMBPOSITION:
			index = (next * pos + 99) / 100;
			move_log_index(0);
			break;
		default:
			return false;
		}
		SetScrollPos
			(hWnd, SB_VERT, log_scroll_position(), 
			 (code != SB_THUMBPOSITION));
		RedrawWindow(hWnd, NULL, NULL, 
			RDW_INTERNALPAINT | RDW_INVALIDATE | RDW_ERASE);
		
		return true;
	}
	else {
		return false;
	}
}

//////////////////////////////////////////////////////////////////////
// Local Member Functions
//////////////////////////////////////////////////////////////////////

void WindowLog::move_log_index(int lines)
{
	if (lines > 0) {
		// move forward
		while (lines > 0) {
			switch (text[index]) {
			case '\0':
				// reached end of text
				return;
			case '\n':
				lines--;
				break;
			}
			index++;
		}
	}
	else if (lines <= 0) {
		// move backwards
		while (lines < 1) {
			if (index == 0) {
				// reached beginning of text
				return;
			}
			index--;
			if (text[index] == '\n') {
				lines++;
			}
		}
		index++;
	}
}

int WindowLog::log_scroll_position()
{
	// Scale between 0 and 100
	if (next > 0) {
		return (index * 100) / next;
	}
	else {
		return 0;
	}
}

int WindowLog::log_page_size()
{
	RECT rt;
	int result;

	GetClientRect(window, &rt);
	result = (rt.bottom - rt.top) / LOG_FONT_HEIGHT;
	return result;
}


void WindowLog::make_new_line_visible()
{
	int ix = next;
	int lines = log_page_size();

	while (ix > index) {
		ix--;
		if (text[ix] == '\n') {
			lines--;
			if (lines < 0) {
				index = ix + 1;
				return;
			}
		}
	}
}
