// TaskSynchronizationBase.h: basic task synchronization declarations
//
//////////////////////////////////////////////////////////////////////
#if !defined(TASKSYNCHRONIZATIONBASE_H__INCLUDED_)
#define TASKSYNCHRONIZATIONBASE_H__INCLUDED_


//////////////////////////////////////////////////////////////////////
// Every task (thread) has an event object used for all task-related
// synchronizations

HANDLE get_task_synch_event();
void close_task_synch_event();


//////////////////////////////////////////////////////////////////////
// A reservation flag is a simple means to ensure that only the first thread
// can access shared data

#define NO_RESERVATION_NEEDED -1
#define NOT_RESERVED 0
#define RESERVED 1

class TaskReservationFlag 
{
public:
	TaskReservationFlag(int init_value = NOT_RESERVED) {data = (LONG)init_value;};
	bool try_reserve();
	bool is_reserved() {return data == RESERVED;}
	bool no_reservation_needed() {return data == NO_RESERVATION_NEEDED;}
private:
	LONG volatile data;
};


//////////////////////////////////////////////////////////////////////
// Task termination is implemented by two mutually dependent data classes,
// the TaskTermination class and the TaskTerminationMaster class
// Each task has a private member of the TaskTermination class, named the
// "termination object"
// The term object represents the termination related attributes of a task
// A task may be terminated by any other task - or itself
// In fact, a terminator task may terminate any number of task simultaneously
// A set of tasks that are terminated simultaneously are called a termination
// group
// A terminator task creates a local object of the TaskTerminationMaster class,
// named the "termination master"
// The termination master carries data that is needed to coordinate the 
// simultaneous termination of all tasks in the termination group

class TaskTermination;
class TaskTerminationMaster;

class TaskTermination
{
public:
	static TaskTermination* get_own_object();

	HANDLE start_owner_wait(HANDLE owner_event);
	bool end_owner_wait();	// true <==> was terminated
	bool owner_activate();	// true <==> was terminated
	void owner_terminate();

	TaskTermination();
private:
	TaskTermination *volatile next;
	TaskTerminationMaster *volatile master;
	HANDLE volatile waiting_owner_event;
	bool volatile about_to_terminate;

	friend class TaskTerminationMaster;
};

class TaskTerminationMaster
{
public:
	typedef void** TaskPtrTable;
	// NULL terminated array of Task*

	TaskTerminationMaster(TaskPtrTable pt_table);

	void terminator_wait(TaskPtrTable pt_table);
	void increment(TaskTermination* term_object_ptr, bool &was_terminated);
	void increment_if_uncounted
		(TaskTermination* term_object_ptr, bool &was_terminated);
	HANDLE decrement(TaskTermination* term_object_ptr);

private:
	CRITICAL_SECTION cs;
	TaskTermination *volatile first_uncounted;
	HANDLE volatile terminator_event;
	int volatile counter;
	bool kill_self;

	void initiate_termination();
	void remove_uncounted(TaskTermination* term_object_ptr);
};

#endif