/*
 * Translated from Andrew Tuman's Pascal sources.
 */

#include <stdio.h>
#include <string.h>
#include <mem.h>
#include <time.h>

#include "datety.hpp"

DateTy global_date_value(false);

static const char formatStrRead[]  = "%d%c%d%c%d";
static const char formatDbfRead[]  = "%04d%02d%02d";
static const char formatSwift23ARead[]  = "%02d%02d%02d";
static const char formatStrWrite[] = "%02d%c%02d%c%d";
static const char formatDbfWrite[] = "%04d%02d%02d";

DateTy::Formats DateTy::PrintOption = DateTy::BRITISH;

static const char* MonthNames[12] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov",
	"Dec" };

static const char* WeekDayNames[7] = {
	"Su", "Mn", "Tu", "We", "Th", "Fr", "St" };

DateTy::DateTy(bool use_global) {
	if(!use_global) {
		long clk = ::time(0);
		struct tm *now = ::localtime(&clk);
		Encode(century + now->tm_year, now->tm_mon + 1, now->tm_mday, JulNum);
	}
	else JulNum = global_date_value.JulNum;
}

DateTy::DateTy(YearTy year, MonthTy month, DayTy day) {
	Encode(::get_full_year(year), month, day, JulNum);
}

DateTy::DateTy(JulTy JulDate) {
	JulNum = JulDate;
}

DateTy::DateTy(const char* date, DateTy::Formats _mode) {
	DayTy day; MonthTy month; YearTy year;
	char c1 = '\0', c2 = '\0';
	int scaned, must_be;
	switch(_mode == DEFAULT ? PrintOption : _mode) {
	case AMERICAN:
	case USA:
		must_be = 5;
		scaned = ::sscanf(date, formatStrRead, &month, &c1, &day, &c2, &year);
		break;
	case ANSI:
	case JAPANESE:
		must_be = 5;
		scaned = ::sscanf(date, formatStrRead, &year, &c1, &month, &c2, &day);
		break;
	case BRITISH:
	case GERMAN:
	case ITALIAN:
		must_be = 5;
		scaned = ::sscanf(date, formatStrRead, &day, &c1, &month, &c2, &year);
		break;
	case PACKED:
		h2d(date);
		must_be = 3;
		scaned = ::sscanf(buffer, formatDbfRead, &year, &month, &day);
		break;
	case XBASE:
		must_be = 3;
		scaned = ::sscanf(date, formatDbfRead, &year, &month, &day);
		break;
	case SWIFT32A:
		must_be = 3;
		scaned = ::sscanf(date, formatSwift23ARead, &year, &month, &day);
		break;
	}
	if(scaned != must_be) {
		JulNum = (JulTy)0l;
		return;
	}
	Encode(::get_full_year(year), month, day, JulNum);
}

int DateTy::IsLeap() const {
	DayTy day; MonthTy month; YearTy year;
	Decode(year, month, day, JulNum);
	return LeapYear(year);
}

int DateTy::WeekDay() const {
	DayTy day; MonthTy month; YearTy year;
	Decode(year, month, day, JulNum);
	return DayOfWeek(year, month, day);
}

DayTy DateTy::Day() const {
	DayTy day; MonthTy month; YearTy year;
	Decode(year, month, day, JulNum);
	return day;
}

MonthTy DateTy::Month() const {
	DayTy day; MonthTy month; YearTy year;
	Decode(year, month, day, JulNum);
	return month;
}

YearTy DateTy::Year(bool century) const {
	DayTy day; MonthTy month; YearTy year;
	Decode(year, month, day, JulNum);
	return century ? year : year % 100;
}

void DateTy::PrintMode(Formats newMode) {
	if(newMode != DEFAULT && newMode != XBASE && newMode != PACKED)
		PrintOption = newMode;
}

DayTy DateTy::DaysInMonth() const {
	DayTy day; MonthTy month; YearTy year;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		return ((month == 2) && LeapYear(year)) ?
			(daysInMonth[month] + 1) : (daysInMonth[month]);
	}
	return 0;
}

DayTy DateTy::LastWorkDay() const {
	DayTy day; MonthTy month; YearTy year; JulTy tmp;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		day = DaysInMonth();
		Encode(year, month, day, tmp);
		int wd = DayOfWeek(year, month, day);
		switch(wd) {
		case 1: // Sunday.
			tmp -= 2;
			break;
		case 7: // Saturday.
			tmp -= 1;
			break;
		}
		Decode(year, month, day, tmp);
		return day;
	}
	return 0;
}

DateTy& DateTy::FirstMonthDay() {
	DayTy day; MonthTy month; YearTy year;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		day = 1;
		Encode(year, month, day, JulNum);
	}
	return *this;
}

DateTy& DateTy::LastMonthDay() {
	DayTy day; MonthTy month; YearTy year;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		day = ((month == 2) && LeapYear(year)) ?
			(daysInMonth[month]+1) : (daysInMonth[month]);
		Encode(year, month, day, JulNum);
	}
	return *this;
}

void DateTy::IncOneMonth() {
	DayTy day, limit; MonthTy month; YearTy year;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		month = (month % 12) + 1;
		if(month == 1) year++;
		limit = ((month == 2) && LeapYear(year)) ?
			(daysInMonth[month] + 1) : (daysInMonth[month]);
		Encode(year, month, ((day <= limit ) ? day : limit ), JulNum);
	}
}

void DateTy::DecOneMonth() {
	DayTy day, limit; MonthTy month; YearTy year;
	if(JulNum) {
		Decode(year, month, day, JulNum);
		month = ((month + 10) % 12) + 1;
		if(month == 12 ) year--;
		limit = ((month == 2) && LeapYear(year)) ?
			(daysInMonth[month] + 1) : (daysInMonth[month]);
		Encode(year, month, ((day <= limit ) ? day : limit), JulNum);
	}
}

DateTy& DateTy::DecMonth(unsigned count) {
	if(JulNum && count) {
		for(register unsigned i = 1; i <= count; i++)
			DecOneMonth();
	}
	return *this;
}

DateTy& DateTy::IncMonth(unsigned count) {
	if(JulNum && count) {
		for(register unsigned i = 1; i <= count; i++)
			IncOneMonth();
	}
	return *this;
}

char DateTy::GetDelimiter() {
	switch(PrintOption) {
	case USA:
	case ITALIAN:
		return '-';
	case GERMAN:
	case ANSI:
		return '.';
	case AMERICAN:
	case JAPANESE:
	case BRITISH:
		return '/';
	}
	return '/';
}

const char* DateTy::c_str(DateTy::Formats _mode, bool century) const {
	DayTy day; MonthTy month; YearTy year;
	char delimiter = GetDelimiter();
	::memset(buffer, 0, sizeof(buffer));
	Decode(year, month, day, JulNum);
	if(!century) year = year % 100;
	switch(_mode == DEFAULT ? PrintOption : _mode) {
	case AMERICAN:
	case USA:
		::sprintf(buffer, formatStrWrite, month, delimiter, day, delimiter, year);
		break;
	case ANSI:
	case JAPANESE:
		::sprintf(buffer, formatStrWrite, year, delimiter, month, delimiter, day);
		break;
	case BRITISH:
	case GERMAN:
	case ITALIAN:
		::sprintf(buffer, formatStrWrite, day, delimiter, month, delimiter, year);
		break;
	case XBASE:
		::sprintf(buffer, formatDbfWrite, year, month, day);
		break;
	case PACKED:
		char tmp[9];
		::sprintf(tmp, formatDbfWrite, year, month, day);
		d2h(tmp);
		break;
	}
	return buffer;
}

const char* DateTy::ShortDayName(DayTy weekDayNumber) {
	if(weekDayNumber >=1 && weekDayNumber <= 7)
		::strcpy(buffer, WeekDayNames[weekDayNumber - 1]);
	else ::memset(buffer, 0, sizeof(buffer));
	return buffer;
}

const char* DateTy::MonthName(MonthTy monthNumber) {
	if(monthNumber >=1 && monthNumber <= 12)
		::strcpy(buffer, MonthNames[monthNumber - 1]);
	else ::memset(buffer, 0, sizeof(buffer));
	return buffer;
}

void DateTy::d2h(const char* value) const {
	::d2h(value, buffer);
}

void DateTy::h2d(const char* value) const {
	::h2d(value, buffer);
}

