/*
**  TELNET.C
**
**  Win32 console program.
**
**  This program implements a bare bones "Network Virtual Terminal".
**
**  RFC 854 -- TELNET PROTOCOL SPECIFICATION
**  RFC 855 -- TELNET OPTION SPECIFICATIONS
**
**  The various TELNET options are defined in a series of RFCs, including
**    652, 653, 654, 655, 656, 657, 698,
**    726, 727, 734, 736, 749, 779,
**    856, 857, 858, 859, 860, 885,
**    927, 933, 946, 1041, 1043, 1053, 1409
*/

#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <string.h>
#include "wil.h"
#include "cons.h"

#define MAX_STR  50
#define MAX_BUF 256

#define IAC  255
#define WILL 251
#define WONT 252
#define DO   253
#define DONT 254

#define TELNET_PORT 23

static char Temp[MAX_BUF+8];
static int  InBufLen = 0;
static char InBuffer[MAX_BUF+1];
static int  KbdBufLen = 0;
static char KbdBuffer[MAX_BUF+1];

static char HostName[MAX_STR] = "";
static ULONG HostAddr = 0;
static short Port = TELNET_PORT;
static SOCKET Socket = 0;


static HANDLE ConIn;
static HANDLE ConOut;
static DWORD OldMode;
static DWORD NewMode;

/* display error */

static void DisplayError(HANDLE Handle, int Code, LPSTR Msg)
{PutString(Handle ,"ERROR: ");
 if(Msg) PutString(Handle,Msg);
 if(Code)
   {wilErrorText(Code,(LPSTR)Temp,MAX_BUF);
    PutLine(Handle,(LPSTR)Temp);
   }
}

/* display error & exit (before calling wilAttach) */

void SayError(LPSTR Msg)
{printf("ERROR: %s: %d\n", Msg, GetLastError() );
 exit(1);
}

void ShutDown(void)
{PutLine(ConOut,"Shutting down...");
 wilRelease();
 SetConsoleMode(ConIn, OldMode);
 exit(0);
}

/* read incoming to InBuffer[], parsing IAC commands out */

int ParseIAC(void)
{int i, Code;
 unsigned char c;
 static int nTemp = 0; /* # chars in TempBuf[] */
 static unsigned char TempBuf[MAX_BUF];
 static unsigned char Response[4] = {IAC, 0, 0, 0};
 /* any incoming ? */
 if(!wilDataIsReady(Socket,0)) return 0;
 /* read socket into TempBuf[] */
 Code = wilReadSocket(Socket,(LPSTR)TempBuf,MAX_BUF);
 if(Code==WIL_EOF) return WIL_EOF;
 if(Code<0)
   {DisplayError(ConOut,Code,"wilReadSocket:");
    return WIL_EOF;
   }
 nTemp = Code;
 InBufLen = 0;
 /* scan TempBuf[] for IAC commands */
 for(i=0;i<nTemp;)
   {c = TempBuf[i++];
    /* look for IAC if TELNET port */
    if((c==IAC)&&(Port==TELNET_PORT))
       {/* look at character following IAC */
        c = TempBuf[i++];
        switch(c)
           {case IAC:
              /* IAC is really data */
              InBuffer[InBufLen++] = (char)IAC;
              break;
            case DO:
              /* got "IAC DO" command */
              Response[1] = WONT;
              Response[2] = TempBuf[i++];
              wilWriteSocket(Socket,(LPSTR)Response, 3);
              wsprintf((LPSTR)Temp,"<IAC DO %d>",Response[2]);
              PutString(ConOut,(LPSTR)Temp);
              break;
            case WILL:
              /* got "IAC WILL" command */
              Response[1] = DONT;
              Response[2] = TempBuf[i++];
              wilWriteSocket(Socket,(LPSTR)Response, 3);
              wsprintf((LPSTR)Temp,"<IAC WILL %d>",Response[2]);
              PutString(ConOut,(LPSTR)Temp);
              break;
            default:
              /* ignore IAC command */
              break;
           }
       } /*end-if*/
    else InBuffer[InBufLen++] = c;
   }/*end-for(i) */
 InBuffer[InBufLen] = '\0';
 return InBufLen;
}


/*** main ***/

void main(int argc, char *argv[])
{unsigned char c;
 int  Code;
 int  i, n;
 BOOL Success;
 /* check usage */
 if((argc!=2)&&(argc!=3))
   {printf("Usage: telnet <host>\n");
    printf("   or: telnet <host> <port>\n");
    exit(1);
   }
 /* get handles for stdin & stdout */
 ConIn  = GetStdHandle(STD_INPUT_HANDLE);
 ConOut = GetStdHandle(STD_OUTPUT_HANDLE);
 /* modify console mode: disable line input & line echo */
 Success = GetConsoleMode(ConIn, &OldMode);
 if(!Success) SayError("GetConsoleMode");
 NewMode = OldMode & ~ENABLE_LINE_INPUT & ~ENABLE_ECHO_INPUT;
 Success = SetConsoleMode(ConIn, NewMode);
 if(!Success) SayError("SetConsoleMode");
 /* attach WINSOCK */
 PutString(ConOut,"Attaching WINSOCK...");
 Code = wilAttach();
 PutLine(ConOut,"OK");
 if(Code<0) DisplayError(ConOut,Code,"wilAttach fails:");
 else
   {PutString(ConOut," Description: ");
    wilGetDescription((LPSTR)Temp, MAX_BUF);
    PutLine(ConOut,(LPSTR)Temp);
    PutString(ConOut,"My Host Name: ");
    wilGetMyHostName((LPSTR)Temp, MAX_BUF);
    PutLine(ConOut,(LPSTR)Temp);
    PutString(ConOut,"My Host Addr: ");
    wilGetMyHostDotted(0,(LPSTR)Temp, MAX_BUF);
    PutLine(ConOut,(LPSTR)Temp);
   }

 lstrcpy((LPSTR)HostName, (LPSTR)argv[1]);
 wsprintf((LPSTR)Temp,"Host = '%s'", (LPSTR)HostName);
 PutLine(ConOut,Temp);

 if(argc==3)
   {Port = (short)atoi(argv[2]);
    wsprintf((LPSTR)Temp,"Port = %d", Port);
    PutLine(ConOut,Temp);
   }

 /* get host IP address */
 if(wilIsDotted((LPSTR)HostName))
   {/* already have IP address */
    HostAddr = wilMakeAddr((LPSTR)HostName);
   }
 else
   {/* ask DNS for IP address */
    HostAddr = wilGetHostAddr((LPSTR)HostName,0);
    if(HostAddr==0)
      {DisplayError(ConOut, 0, "Cannot get IP addess");
       ShutDown();
      }
   }
 /* display remote host address */
 PutString(ConOut,"HostAddr = ");
 wilMakeDotted(HostAddr,(LPSTR)Temp,MAX_BUF);
 PutLine(ConOut,(LPSTR)Temp);
 /* create socket */
 Socket = wilTcpSocket();
 if((int)Socket<0)
   {DisplayError(ConOut, (int)Code, NULL);
    ShutDown();
   }
 /* attempt to connect to remote host */
 Code = wilConnect(Socket,HostAddr,Port);
 if(Code<0)
   {DisplayError(ConOut, Code, NULL);
    ShutDown();
   }

 /* wait up to 30 seconds for connection */
 if(wilIsConnected(Socket,30000)) PutLine(ConOut,"Connected to server.");
 else
   {DisplayError(ConOut,0, "Cannot CONNECT");
    ShutDown();
   }

 PutLine(ConOut,"Type QUIT to quit, <ESC> to abort...\n");
 //for(i=0;i<MAX_BUF;i++) InBuffer[i] = '\0';
 InBufLen = 0;

 while(1)
   {/* any incoming ? */
    n = ParseIAC();
    if(n==WIL_EOF) break;
    if(n>0)
      {for(i=0;i<InBufLen;i++)
         {c = InBuffer[i];
          if(c>128)
            {wsprintf((LPSTR)Temp,"(%x)", c);
             PutString(ConOut,(LPSTR)Temp);
            }
          else switch(c)
            {case '\r':
             case '\n':
               PutChar(ConOut,c);
               break;
             default:
               PutChar(ConOut,c);
            }
         }
      }
    /* has user typed key on keyboard ? */
    if(KeyPress(ConIn))
      {/* get user char */
       c = GetChar(ConIn);
       /* does user want to quit ? (by typing <ESC>)*/
       if(c==0x1b) break;
       /* local echo */
       PutChar(ConOut,c);

#if 1
       if(c=='$')
         {wilWriteSocket(Socket,(LPSTR)"PORT 216,180,4,104,4,18\r\n\0" ,25);
          ///KbdBufLen = 0;
          continue;
         }
#endif

       /* put character in buffer */
       KbdBuffer[KbdBufLen++] = c;
       /* carriage return ? */
       if(c=='\r')
         {/* add NULL */
          KbdBuffer[KbdBufLen++] = '\n';
          KbdBuffer[KbdBufLen] = '\0';
          /* write to socket */
          wilWriteSocket(Socket,(LPSTR)KbdBuffer,KbdBufLen);
          KbdBufLen = 0;
          KbdBuffer[0] = '\0';
          PutString(ConOut,(LPSTR)Temp);
         }
      }
   }

 PutLine(ConOut,"\r\nTELNET completed.");
 wilCloseSocket(Socket,500);
 ShutDown();
} /* end main */


