// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : t-rulin.cpp
//
//  DESCRIPTION :
//         Class TuringMachine : program transition methods (Implementation)
//
// ##############################################################


// =================
#include "turing-s.h"
// =================


// =========
void TuringMachine::build_transition ()
{
  show_transitions (true, false, "User-Defined");

  show_daemon_transitions (false, false, "Daemon-Defined");

  build_check_point_transitions ();
  show_transitions (false, true, "Check-Point", "User-Extended");

  build_basement_transitions ();
  show_transitions (false, true, "Basement", "Basement");

  // build_loft_transitions ();
  // show_transitions (false, true, "Loft", "Loft");

  build_floor1_transitions ();
  show_transitions (false, true, "Floor1", "Floor1");

  build_floor2_transitions ();
  show_transitions (false, true, "Floor2", "Floor2");

  build_floor3_transitions ();
  show_transitions (false, true, "Floor3", "Floor3");

  build_floor4_transitions ();
  show_transitions (false, true, "Floor4", "Floor4");

  build_floor5_transitions ();
  show_transitions (false, true, "Floor5", "Floor5");

  build_floor6_transitions ();
  show_transitions (false, true, "Floor6", "Floor6");

  build_floor7_transitions ();
  show_transitions (false, true, "Floor7", "Floor7");

  build_roof1_transitions ();
  show_transitions (false, true, "Roof1", "Roof1");

  build_roof2_transitions ();
  show_transitions (false, true, "Roof2", "Roof2");

  build_outside_tact1_transitions ();
  show_outside_tact1_transitions ();

  build_outside_tact2_transitions ();
  show_outside_tact2_transitions ();

  // -----------------------------------------
  process_statistics_.resize (built_transitions_.size());

}


// =========
void TuringMachine::elongate_user_defined_transitions ()
{

  assert (built_transitions_.empty());
  info_user_transitions_.clear();

Transitions_t	tmp_transitions (user_defined_transitions_);
Transitions_t::const_iterator	  		pos_iter;
typedef Transitions_t::value_type		value_type;
pair<Transitions_t::iterator, bool>		couple;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();
  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    assert (tapes_[i].size() == number_of_tapes);
  }

  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    vector<symbol_mt>		new_cur_vect (pos_iter->first.get_symbols());
    vector<SymbolAndShift>	new_next_vect (pos_iter->second.get_symbols_and_shifts ());
 
    for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
    {
      for (size_t j = 0; j < number_of_tapes; j++)
      {
        switch (i)
        {
          case MASTER_COMPUTATION_TAPE :
            // Do nothing
            break;

          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (!pseudo_never_mind_symbols_alphabet_.empty());
            new_cur_vect.push_back (pseudo_never_mind_symbols_alphabet_.front());

            assert (!pseudo_write_nothing_symbols_alphabet_.empty());
            new_next_vect.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;

          default :    
            assert (0);
            break;
        }
      }
    }


    assert (neutral_program_states_.empty());

    assert (new_cur_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (pos_iter->first.get_substate1() == state_t ());
    assert (pos_iter->first.get_substate2() == state_t ());

    const CurSituation	new_cur_situation (
				pos_iter->first.get_state(), 
				state_t(),
				state_t(),
				new_cur_vect, 
				get_symbol_mtypes (new_cur_vect), 
				pos_iter->first.get_controlling_flag()
				);
    assert (built_transitions_.count(new_cur_situation) == 0);

    assert (new_next_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (pos_iter->second.get_substate1() == state_t ());
    assert (pos_iter->second.get_substate2() == state_t ());
    const NextSituation	new_next_situation (
				pos_iter->second.get_state(), 
				state_t(),
				state_t(),
				new_next_vect
				);

    couple = built_transitions_.insert (value_type(new_cur_situation, new_next_situation));
    assert (couple.second);
    
    assert (info_user_transitions_.count (new_cur_situation) == 0);
    info_user_transitions_[new_cur_situation] = RULE_NEW;
  }


  assert (built_transitions_.size() == user_defined_transitions_.size());
  assert (!built_transitions_.empty());

} // elongate_user_defined_transitions




// =========
void TuringMachine::elongate_daemon_defined_fault_transitions ()
{

  if (daemon_defined_fault_transitions_.empty()) return;

  assert (!daemon_defined_fault_transitions_.empty());

Transitions_t	tmp_transitions (daemon_defined_fault_transitions_);
Transitions_t::const_iterator	  		pos_iter;
typedef Transitions_t::value_type		value_type;
pair<Transitions_t::iterator, bool>		couple;

  daemon_defined_fault_transitions_.clear();
  assert (daemon_defined_fault_transitions_.empty());
  assert (!tmp_transitions.empty());

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();
  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    assert (tapes_[i].size() == number_of_tapes);
  }

  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    vector<symbol_mt>		new_cur_vect (pos_iter->first.get_symbols());
    vector<SymbolAndShift>	new_next_vect (pos_iter->second.get_symbols_and_shifts ());
 
    for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
    {
      for (size_t j = 0; j < number_of_tapes; j++)
      {
        switch (i)
        {
          case MASTER_COMPUTATION_TAPE :
            // Do nothing
            break;

          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (!pseudo_never_mind_symbols_alphabet_.empty());
            new_cur_vect.push_back (pseudo_never_mind_symbols_alphabet_.front());

            assert (!pseudo_write_nothing_symbols_alphabet_.empty());
            new_next_vect.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;

          default :    
            assert (0);
            break;
        }
      }
    }


    assert (!neutral_program_states_.empty());

    assert (new_cur_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (pos_iter->first.get_substate1() == state_t ());
    assert (pos_iter->first.get_substate2() == state_t ());
    const CurSituation	new_cur_situation (
				pos_iter->first.get_state(), 
				get_neutral_cur_substate_alphabet_symbol(),
				get_neutral_cur_substate_alphabet_symbol(),
				new_cur_vect, 
				get_symbol_mtypes (new_cur_vect), 
				pos_iter->first.get_controlling_flag()
				);
    assert (daemon_defined_fault_transitions_.count(new_cur_situation) == 0);

    assert (new_next_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (pos_iter->second.get_substate1() == state_t ());
    assert (pos_iter->second.get_substate2() == state_t ());
    const NextSituation	new_next_situation (
				pos_iter->second.get_state(), 
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				new_next_vect
				);

    couple = daemon_defined_fault_transitions_.insert (value_type(new_cur_situation, new_next_situation));
    assert (couple.second);
    
  }


  assert (daemon_defined_fault_transitions_.size() == daemon_defined_fault_transitions_.size());
  assert (!daemon_defined_fault_transitions_.empty());

  // ----------------------------
  assert (count_daemon_transitions_.empty());

  for (pos_iter = daemon_defined_fault_transitions_.begin(); 
       pos_iter != daemon_defined_fault_transitions_.end();
       pos_iter++
      )
  {
    assert (!belongs_to(pos_iter->first, count_daemon_transitions_));
    count_daemon_transitions_[pos_iter->first] = 0;
  }

  assert (!count_daemon_transitions_.empty());
  assert (count_daemon_transitions_.size() == daemon_defined_fault_transitions_.size());

} // elongate_daemon_defined_fault_transitions




// =========
void TuringMachine::build_check_point_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();
Transitions_t	tmp_transitions (built_transitions_);
  assert (!tmp_transitions.empty());

Transitions_t	tmp2_transitions;
  assert (tmp2_transitions.empty());

  built_transitions_.clear();
  assert (built_transitions_.empty());  
  info_user_transitions_.clear();

Transitions_t::const_iterator	  	pos_iter;
typedef Transitions_t::value_type	value_type;
pair<Transitions_t::iterator, bool>	couple;

  // -------- First loop --------
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    CurSituation  cur_situation (pos_iter->first);
    NextSituation next_situation (pos_iter->second);
   
    assert (
	    (cur_situation.get_substate1() == state_t())
            ||
	    (cur_situation.get_substate1() == get_neutral_cur_substate_alphabet_symbol())
           );
    assert (
	    (cur_situation.get_substate2() == state_t())
            ||
	    (cur_situation.get_substate2() == get_neutral_cur_substate_alphabet_symbol())
           );

    assert (
	    (next_situation.get_substate1() == state_t())
            ||
	    (next_situation.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol())
           );
    assert (
	    (next_situation.get_substate2() == state_t())
            ||
	    (next_situation.get_substate2() == get_neutral_cur_substate_alphabet_symbol())
           );


    if (cur_situation.get_substate1() == state_t())
    {
      cur_situation.set_substate1(get_neutral_cur_substate_alphabet_symbol());
    }
    if (cur_situation.get_substate2() == state_t())
    {
      cur_situation.set_substate2(get_neutral_cur_substate_alphabet_symbol());
    }

    if (next_situation.get_substate1() == state_t())
    {
      next_situation.set_substate1(get_neutral_next_the_same_substate_alphabet_symbol());
    }
    if (next_situation.get_substate2() == state_t())
    {
      next_situation.set_substate2(get_neutral_next_the_same_substate_alphabet_symbol());
    }


    assert (tmp2_transitions.count(cur_situation) == 0);
    couple = tmp2_transitions.insert (value_type(cur_situation, next_situation));
    assert (couple.second);

  }


  assert (!tmp2_transitions.empty());

  // -------- Second loop --------
  for (pos_iter = tmp2_transitions.begin();
       pos_iter != tmp2_transitions.end();
       pos_iter++)
  {
    assert (pos_iter->first.get_substate1() == get_neutral_cur_substate_alphabet_symbol());
    assert (pos_iter->first.get_substate2() == get_neutral_cur_substate_alphabet_symbol());
    assert (pos_iter->second.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol());
    assert (pos_iter->second.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol());

    assert (!(pos_iter->first.get_substate1() == state_t()));
    assert (!(pos_iter->first.get_substate2() == state_t()));
    assert (!(pos_iter->second.get_substate1() == state_t()));
    assert (!(pos_iter->second.get_substate2() == state_t()));

    // ---------------------------
    if (pos_iter->first.get_controlling_flag())
    {
      sub_build_check_point_transitions (pos_iter);
    } 
    else
    {
      assert (built_transitions_.count(pos_iter->first) == 0);
      couple = built_transitions_.insert (value_type(pos_iter->first, pos_iter->second));
      assert (couple.second);

      assert (info_user_transitions_.count (pos_iter->first) == 0);
      info_user_transitions_[pos_iter->first] = RULE_NEW;

    }       
  }



} // build_check_point_transitions



// =========
void TuringMachine::build_basement_transitions ()
{

  remote_master_and_masterbackup_tape (pre_initial_program_states_.front(), EMBSTATE_K1);
  backup_master_and_masterbackup_tape (MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE, EMBSTATE_K1, EMBSTATE_K2);
  remote_master_and_masterbackup_tape (EMBSTATE_K2, user_defined_initial_program_states_.front());


} // build_basement_transitions


// =========
void TuringMachine::build_loft_transitions ()
{

  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();
Transitions_t	tmp_transitions (built_transitions_);
  assert (!tmp_transitions.empty());

  built_transitions_.clear();
  assert (built_transitions_.empty());  
  info_user_transitions_.clear();

Transitions_t::const_iterator	  	pos_iter;
typedef Transitions_t::value_type	value_type;
pair<Transitions_t::iterator, bool>	couple;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();


  // ----------------
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    const CurSituation  cur_situation (pos_iter->first);
    const NextSituation next_situation (pos_iter->second);

    if (is_halting_program_state (next_situation.get_state()))
    {
      assert (!post_halting_program_states_.empty());

      // ------------------
      vector<symbol_mt>		new_cur_vect;
      vector<SymbolAndShift>	new_next_vect;

      assert (cur_situation.get_symbols().size() == next_situation.get_symbols_and_shifts().size());
      const size_t vect_size = cur_situation.get_symbols().size();
      for (size_t id1 = 0; id1 < vect_size; id1++)
      {
        new_cur_vect.push_back (pseudo_never_mind_symbols_alphabet_.front());
        new_next_vect.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
      }

      assert (new_cur_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
      assert (new_next_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


      const CurSituation new_cur_situation (
  				next_situation.get_state(),
				cur_situation.get_substate1(),
				cur_situation.get_substate2(),
				new_cur_vect, 
				get_symbol_mtypes (new_cur_vect), 
				false // controlling_flag 
				);

      const NextSituation new_next_situation (
				post_halting_program_states_.front(),
				next_situation.get_substate1(),
				next_situation.get_substate2(),
				new_next_vect
				);

      couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
      assert (couple.second);

      assert (info_user_transitions_.count (new_cur_situation) == 0);
      info_user_transitions_[new_cur_situation] = RULE_NEW;

      // ------------------
      assert (user_required_check_point_program_states_.count(new_cur_situation) == 0);
      user_required_check_point_program_states_[new_cur_situation] = new_next_situation.get_state();
      assert (user_required_check_point_program_states_.count(new_cur_situation) == 1);
 
    }

    couple = built_transitions_.insert (
			value_type (cur_situation, next_situation)
			);
    assert (couple.second);

    assert (info_user_transitions_.count (cur_situation) == 0);
    info_user_transitions_[cur_situation] = RULE_OLD;

  }

    
    // ---------------------------------------

} // build_loft_transitions


// =========
void TuringMachine::old_build_basement_transitions ()
{

  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

const state_t		cur_pre_initial_state (pre_initial_program_states_.front());
const state_t		next_pre_initial_state (pre_initial_program_states_.front());
const state_t		next_user_initial_state (user_defined_initial_program_states_.front());

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_master_tape_alphabet_symbols ()
			));

  assert (user_defined_blank_alphabet_.size() == 1);
const symbol_mt user_defined_blank (user_defined_blank_alphabet_.front());
const symbol_mt	user_defined_marker (get_user_defined_marker ());

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;
    state_t			next_to_be_state;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];

    const bool homogeneous_blank_flag  = is_homogeneous_permut (all_permuts[i], user_defined_blank);
    const bool homogeneous_marker_flag = is_homogeneous_permut (all_permuts[i], user_defined_marker);
    const bool homogeneous_flag	= homogeneous_blank_flag || homogeneous_marker_flag;
  

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            if (!homogeneous_flag)
            {
              shift_t this_shift;
              next_to_be_state = next_pre_initial_state;
              if (tape_data[s][t] == user_defined_marker)
              {
                this_shift = NO_SHIFT;
              }
              else
              {
                this_shift = LEFT_SHIFT;
              }
              cur_vect_rule.push_back (tape_data[s][t]);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), this_shift));     
            }
            else
            {
              assert (homogeneous_flag);
              if (homogeneous_blank_flag)
              {
                assert (tape_data[s][t] == user_defined_blank);
                next_to_be_state = next_pre_initial_state;
                cur_vect_rule.push_back (tape_data[s][t]);
                next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), LEFT_SHIFT));     
              }
              if (homogeneous_marker_flag)
              {
                assert (tape_data[s][t] == user_defined_marker);
                next_to_be_state = next_user_initial_state;
                cur_vect_rule.push_back (tape_data[s][t]);
                next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), RIGHT_SHIFT));     
              }
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation   cur_situation_rule (
  				cur_pre_initial_state,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    assert ((next_to_be_state == next_pre_initial_state) || (next_to_be_state == next_user_initial_state));

    const NextSituation  next_situation_rule (
				next_to_be_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i


} // old_build_basement_transitions



// =========
void TuringMachine::sub_build_check_point_transitions (const Transitions_t::const_iterator& iter_i)
{
typedef Transitions_t::value_type		value_type;
pair<Transitions_t::iterator, bool>		couple;
map<CurSituation, state_t>::const_iterator	pos2_iter;

  assert (user_required_check_point_program_states_.count(iter_i->first) == 1);
  pos2_iter = user_required_check_point_program_states_.find(iter_i->first);
  assert (pos2_iter != user_required_check_point_program_states_.end());

const state_t	check_point_state (pos2_iter->second);

vector<SymbolAndShift>  prev_next_vect (iter_i->second.get_symbols_and_shifts ());
vector<SymbolAndShift>  new_next1_vect;
vector<symbol_mt>       new_cur2_vect;

  assert (prev_next_vect.size() == (NUMBER_OF_TAPE_KINDS * tapes_[MASTER_COMPUTATION_TAPE].size()));

  for (size_t i = 0; i < prev_next_vect.size(); i++)
  {
    symbol_mt new_cur2_symbol (prev_next_vect[i].get_symbol());
    symbol_mt new_next1_symbol (new_cur2_symbol);

    // ---------------------
    if (is_pseudo_write_nothing_symbol (new_cur2_symbol))
    {
      new_cur2_symbol = pseudo_never_mind_symbols_alphabet_.front();        
    }

    new_cur2_vect.push_back (new_cur2_symbol);
    new_next1_vect.push_back (SymbolAndShift (new_next1_symbol, NO_SHIFT));

  }
  assert (new_next1_vect.size() == (NUMBER_OF_TAPE_KINDS * tapes_[MASTER_COMPUTATION_TAPE].size()));

  // ------
  assert (!neutral_program_states_.empty());

  CurSituation	cur1_situation (iter_i->first);
  cur1_situation.set_substate2(check_point_state);

  const NextSituation	next1_situation (
		check_point_state, 
		get_neutral_next_the_same_substate_alphabet_symbol(), 
		get_neutral_next_the_same_substate_alphabet_symbol(), 
		new_next1_vect
		);


  const CurSituation	cur2_situation (
		check_point_state, 
		get_neutral_cur_substate_alphabet_symbol(), 
		get_neutral_cur_substate_alphabet_symbol(), 
		new_cur2_vect, 
		get_symbol_mtypes (new_cur2_vect), 
		false	// Maybe other flag ??
		);
  const NextSituation	next2_situation (iter_i->second);


  // --------
  // --------
  assert (built_transitions_.count(cur1_situation) == 0);
  couple = built_transitions_.insert (value_type(cur1_situation, next1_situation ));
  assert (couple.second);

  assert (info_user_transitions_.count (cur1_situation) == 0);
  info_user_transitions_[cur1_situation] = RULE_NEW;


  assert (built_transitions_.count(cur2_situation) == 0);
  couple = built_transitions_.insert (value_type(cur2_situation, next2_situation ));
  assert (couple.second);

  assert (info_user_transitions_.count (cur2_situation) == 0);
  info_user_transitions_[cur2_situation] = RULE_NEW;

} // sub_build_check_point_transitions
			  



// =========
void TuringMachine::build_floor1_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

Transitions_t	tmp_transitions (built_transitions_);
  assert (!tmp_transitions.empty());

  built_transitions_.clear();
  assert (built_transitions_.empty());  
  info_user_transitions_.clear();

Transitions_t::const_iterator	  	pos_iter;
typedef Transitions_t::value_type	value_type;
pair<Transitions_t::iterator, bool>	couple;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

const size_t master_tape_start_pos     = number_of_tapes * MASTER_COMPUTATION_TAPE;
const size_t syncronous_tape_start_pos = number_of_tapes * SYNCHRONOUS_COMPUTATION_TAPE;


  // --------- 1.0 - 1.3 ---------
  // ------------- Initial & Internal state ---------------
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    const CurSituation  tmp_cur_situation  (pos_iter->first);
    NextSituation tmp_next_situation (pos_iter->second);
    if (belongs_to(tmp_cur_situation, user_required_check_point_program_states_))
  {
      assert (tmp_cur_situation.get_substate2() == user_required_check_point_program_states_[tmp_cur_situation]);
      tmp_next_situation.set_substate1 (tmp_cur_situation.get_state());
      tmp_next_situation.set_substate2 (user_required_check_point_program_states_[tmp_cur_situation]);
    }

    const vector<symbol_mt>	cur_vect (tmp_cur_situation.get_symbols());
    vector<SymbolAndShift>	next_vect (tmp_next_situation.get_symbols_and_shifts ());

    assert (cur_vect.size() == number_of_tapes * NUMBER_OF_TAPE_KINDS);
    assert (next_vect.size() == number_of_tapes * NUMBER_OF_TAPE_KINDS);


    assert (built_transitions_.count(tmp_cur_situation) == 0);

    // --------- 1.0 ---------
    if (is_pre_initial_program_state (tmp_cur_situation.get_state()))
    {
      assert (built_transitions_.count(tmp_cur_situation) == 0);
      couple = built_transitions_.insert (value_type(tmp_cur_situation, tmp_next_situation));
      assert (couple.second);

      assert (info_user_transitions_.count (tmp_cur_situation) == 0);
      info_user_transitions_[tmp_cur_situation] = RULE_OLD;

      continue;
    }

    // -----------------------------------
    // -----------------------------------
    assert (!is_pre_initial_program_state (tmp_cur_situation.get_state()));

    // --------------------------
    if (tmp_cur_situation.get_controlling_flag())
    {
      // --------- 1.1 ---------
      for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
      {
        for (size_t t = 0; t < number_of_tapes; t++)
        {
          const size_t pos = t + s * number_of_tapes;
          assert (pos < next_vect.size());
          switch (s)
          {
            case MASTER_COMPUTATION_TAPE :
              assert (next_vect[pos].get_shift() == NO_SHIFT);
              break;
      
            case SYNCHRONOUS_COMPUTATION_TAPE :
            case BACKUP_COMPUTATION_TAPE :
            case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            case TESTER_TAPE :
              assert (next_vect[pos].get_shift() == NO_SHIFT);
              break;
      
            default :    
              assert (0);
              break;
          }
        }
      }	// for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)

      for (size_t t = 0; t < number_of_tapes; t++)
      {
        if (cur_vect [master_tape_start_pos + t] == get_user_defined_marker())
        {
          next_vect [syncronous_tape_start_pos + t].set_symbol(get_embedded_synchronous_tape_marker_s());
        }
        else
        {
          next_vect [syncronous_tape_start_pos + t].set_symbol(embedded_blank_synchronous_tape_alphabet_s.front());
        }
      }

      // -------------------


      // -------------------
      assert (built_transitions_.count(tmp_cur_situation) == 0);
      couple = built_transitions_.insert (
			value_type(
				tmp_cur_situation, 
				NextSituation (
  					EMBSTATE_R1,
					tmp_next_situation.get_substate1(),
					tmp_next_situation.get_substate2(),
					next_vect
					)
				)
				);
      assert (couple.second);

      assert (info_user_transitions_.count (tmp_cur_situation) == 0);
      info_user_transitions_[tmp_cur_situation] = RULE_UPDATED;

    }
    else
    {
      // --------- 1.2 ---------
      assert (built_transitions_.count(tmp_cur_situation) == 0);

      for (size_t t = 0; t < number_of_tapes; t++)
      {
        shift_t this_shift (next_vect[master_tape_start_pos + t].get_shift());
        next_vect [syncronous_tape_start_pos + t].set_shift(this_shift);

        bool flag = false;
        if ((this_shift == LEFT_SHIFT) || (this_shift == NO_SHIFT))
        {
          assert (!flag);
          flag = true;
          next_vect [syncronous_tape_start_pos + t].set_symbol(embedded_blank_synchronous_tape_alphabet_s.front());
        }
        if (this_shift == RIGHT_SHIFT)
        {
          assert (!flag);
          flag = true;
          if (cur_vect [master_tape_start_pos + t] == get_user_defined_marker())
          {
            next_vect [syncronous_tape_start_pos + t].set_symbol(get_embedded_synchronous_tape_marker_s());
          }
          else
          {
            next_vect [syncronous_tape_start_pos + t].set_symbol(get_embedded_synchronous_tape_right_sign_s());
          }
        }
        assert (flag);

      }
       

      // -------------------
      assert (built_transitions_.count(tmp_cur_situation) == 0);
      couple = built_transitions_.insert (
			value_type(
				tmp_cur_situation, 
				NextSituation (
					tmp_next_situation.get_state(),
					tmp_next_situation.get_substate1(),
					tmp_next_situation.get_substate2(),
					next_vect
					)
				)
				);
      assert (couple.second);

      assert (info_user_transitions_.count (tmp_cur_situation) == 0);
      info_user_transitions_[tmp_cur_situation] = RULE_UPDATED;

    }

  } // for (pos_iter = tmp_transitions.begin();


  // --------- 1.3 ---------
  // ------------- Halting state ---------------

  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());
  assert (!embedded_blank_synchronous_tape_alphabet_s.empty());

  size_t local_count = 0;
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    const CurSituation  halt_cur_situation  (pos_iter->first);
    const NextSituation halt_next_situation (pos_iter->second);

    if (!is_halting_program_state(halt_next_situation.get_state())) continue; 
    local_count++;
    if (local_count > 1) continue;   

    // -------
    assert (is_halting_program_state(halt_next_situation.get_state()));

    vector<symbol_mt>		new_cur_vect;
    vector<SymbolAndShift>	new_next_vect;

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {

        switch (s)
        {
          case SYNCHRONOUS_COMPUTATION_TAPE :
            new_cur_vect.push_back (pseudo_never_mind_symbols_alphabet_.front());
            new_next_vect.push_back (SymbolAndShift (embedded_blank_synchronous_tape_alphabet_s.front(), NO_SHIFT));     
            break;

          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            new_cur_vect.push_back (pseudo_never_mind_symbols_alphabet_.front());
            new_next_vect.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;

          default :    
            assert (0);
            break;
        }
      }
    } // for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)

    assert (new_cur_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (new_next_vect.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));

const CurSituation	new_cur_situation (
			halt_next_situation.get_state(), 
			get_neutral_cur_substate_alphabet_symbol (),
			get_neutral_cur_substate_alphabet_symbol (),
			new_cur_vect, 
			get_symbol_mtypes (new_cur_vect), 
			false
			);

const NextSituation	new_next_situation (
  			EMBSTATE_U1,
			get_neutral_next_the_same_substate_alphabet_symbol(),
			get_neutral_next_the_same_substate_alphabet_symbol(),
			new_next_vect
			);

pair<Transitions_t::iterator, bool> couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation));
    assert (couple.second);

    assert (info_user_transitions_.count (new_cur_situation) == 0);
    info_user_transitions_[new_cur_situation] = RULE_NEW;


  }

} // build_floor1_transitions




// =========
void TuringMachine::build_floor2_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor2_1_transitions ();
  build_floor2_2_transitions ();
}


// =========
void TuringMachine::build_floor3_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor3_1_transitions ();
  build_floor3_2_transitions ();
  build_floor3_3_transitions ();
  build_floor3_4_transitions ();
}
   

// =========
void TuringMachine::build_floor4_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor4_1_transitions ();
  build_floor4_2_transitions ();
  build_floor4_3_transitions ();
  build_floor4_4_transitions ();
}
   

// =========
void TuringMachine::build_floor5_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor5_1_transitions ();
  build_floor5_2_transitions ();
  build_floor5_3_transitions ();
  build_floor5_4_transitions ();
}
   


// =========
void TuringMachine::build_floor6_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor6_1_transitions ();
  build_floor6_2_transitions ();
  build_floor6_3_transitions ();
  build_floor6_4_transitions ();
}
   

// =========
void TuringMachine::build_floor7_transitions ()
{
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();

  build_floor7_1_transitions ();
  build_floor7_2_transitions ();
}
   


// =========
void TuringMachine::build_roof1_transitions ()
{
  enum TmpEnum
  {
    UNDEF_TmpEnum,
    BOTH_RULE_STATES_ARE_EMBSTATE,
    ONLY_RIGHT_RULE_STATE_IS_EMBSTATE,
    ONLY_LEFT_RULE_STATE_IS_EMBSTATE,
    BOTH_RULE_STATES_ARE_ARE_NOT_EMBSTATE,
    NUMBER_OF_TmpEnum
  };



vector<state_t>	check_point_states_vect;
map<CurSituation, state_t>::const_iterator pos_iter_check_point_vect;

  for (pos_iter_check_point_vect = user_required_check_point_program_states_.begin();
       pos_iter_check_point_vect != user_required_check_point_program_states_.end();
       pos_iter_check_point_vect++
       )
  {
    check_point_states_vect.push_back(pos_iter_check_point_vect->second);
  }


map<CurSituation, state_t>	user_required_initial_check_point_program_states;
  for (pos_iter_check_point_vect = user_required_check_point_program_states_.begin();
       pos_iter_check_point_vect != user_required_check_point_program_states_.end();
       pos_iter_check_point_vect++
       )
  {
    if (is_initial_program_state (pos_iter_check_point_vect->first.get_state()))
    {
      assert (!belongs_to (pos_iter_check_point_vect->first, user_required_initial_check_point_program_states));
      user_required_initial_check_point_program_states [pos_iter_check_point_vect->first] = pos_iter_check_point_vect->second;   
      assert (belongs_to (pos_iter_check_point_vect->first, user_required_initial_check_point_program_states));
    }
  }
map<CurSituation, state_t>::iterator pos_iter_initial_check_point;


  // ---------------------------
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();
Transitions_t	tmp_transitions (built_transitions_);
  assert (!tmp_transitions.empty());

  built_transitions_.clear();
  assert (built_transitions_.empty());  
  info_user_transitions_.clear();

Transitions_t::const_iterator	  	pos_iter;
typedef Transitions_t::value_type	value_type;
pair<Transitions_t::iterator, bool>	couple;

  // ----------------
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    const CurSituation  cur_situation (pos_iter->first);
    const NextSituation next_situation (pos_iter->second);

    bool cur_state_embedded_bool_flag  = false;
    bool next_state_embedded_bool_flag = false;
    TmpEnum	rule_flag = UNDEF_TmpEnum;

    int id_enum;
    for (id_enum = 0; id_enum < NUMBER_OF_EMBEDDED_PROGRAM_STATE_KINDS; id_enum++)
    {
      switch (id_enum)
      {
        case EMBEDDED_PROGRAM_STATE_KIND__COMPUTATION_CONTROL :
        case EMBEDDED_PROGRAM_STATE_KIND__BACKUP :
        case EMBEDDED_PROGRAM_STATE_KIND__BACKUP_CONTROL :
        case EMBEDDED_PROGRAM_STATE_KIND__RECOVERY :
        case EMBEDDED_PROGRAM_STATE_KIND__RECOVERY_CONTROL :
          if (is_embedded_program_state_of_specific_kind (
		static_cast<EmbeddedProgramStateKinds> (id_enum), 
		cur_situation.get_state()
		)
	     )
          {
            cur_state_embedded_bool_flag = true;
          }
          if (is_embedded_program_state_of_specific_kind (
		static_cast<EmbeddedProgramStateKinds> (id_enum), 
		next_situation.get_state()
		)
	     )
          {
            next_state_embedded_bool_flag = true;
          }
          break;
  
        case EMBEDDED_PROGRAM_STATE_KIND__PRE_INITIAL_BACKUP :
        case EMBEDDED_PROGRAM_STATE_KIND__EMERGENCY :
        case EMBEDDED_PROGRAM_STATE_KIND__SHUTTING_DOWN :
        case EMBEDDED_PROGRAM_STATE_KIND__SUMMARY_CONTROL :
          // Do nothing
          break;
  
        default :
          assert (0);
          break;
      }
    }

    if (user_required_check_point_program_states_.empty())
    {
      if (cur_state_embedded_bool_flag)
      {
        assert (built_transitions_.count(cur_situation) == 0);
        couple = built_transitions_.insert (
			value_type (cur_situation, next_situation)
			);
        assert (couple.second);

        assert (info_user_transitions_.count (cur_situation) == 0);
        info_user_transitions_[cur_situation] = RULE_UPDATED;

        continue;
      }
    }    



    if (cur_state_embedded_bool_flag && next_state_embedded_bool_flag)
    {
      assert (rule_flag == UNDEF_TmpEnum);
      rule_flag	= BOTH_RULE_STATES_ARE_EMBSTATE;
    }
    if (!cur_state_embedded_bool_flag && next_state_embedded_bool_flag)
    {
      assert (rule_flag == UNDEF_TmpEnum);
      rule_flag	= ONLY_RIGHT_RULE_STATE_IS_EMBSTATE;
    }
    if (cur_state_embedded_bool_flag && !next_state_embedded_bool_flag)
    {
      assert (rule_flag == UNDEF_TmpEnum);
      rule_flag	= ONLY_LEFT_RULE_STATE_IS_EMBSTATE;
    }
    if (!cur_state_embedded_bool_flag && !next_state_embedded_bool_flag)
    {
      assert (rule_flag == UNDEF_TmpEnum);
      rule_flag	= BOTH_RULE_STATES_ARE_ARE_NOT_EMBSTATE;
    }
    assert (rule_flag != UNDEF_TmpEnum);
    assert (rule_flag < NUMBER_OF_TmpEnum);


    // --------------------------------
    // --------------------------------
    // --------------------------------

    switch (rule_flag)
    {
      case BOTH_RULE_STATES_ARE_EMBSTATE :
        assert (cur_situation.get_substate1() == get_neutral_cur_substate_alphabet_symbol ());
        assert (cur_situation.get_substate2() == get_neutral_cur_substate_alphabet_symbol ());
        assert (next_situation.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol ());
        assert (next_situation.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol ());

        // ------------
        for (pos_iter_initial_check_point = user_required_initial_check_point_program_states.begin();
             pos_iter_initial_check_point != user_required_initial_check_point_program_states.end();
             pos_iter_initial_check_point++
	     )
        {
          const CurSituation new_cur_situation(
				cur_situation.get_state(),
	  			pos_iter_initial_check_point->first.get_state(),			
	  			pos_iter_initial_check_point->second,			
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

          const NextSituation new_next_situation(
				next_situation.get_state(),
	  			pos_iter_initial_check_point->first.get_state(),			
	  			pos_iter_initial_check_point->second,			
				next_situation.get_symbols_and_shifts()
				);


          assert (built_transitions_.count(new_cur_situation) == 0);
          couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
          assert (couple.second);

          assert (info_user_transitions_.count (new_cur_situation) == 0);
          info_user_transitions_[new_cur_situation] = RULE_UPDATED;

        } // for (pos_iter_initial_check_point


        // ------------
        for (size_t id1 = 0; id1 < check_point_states_vect.size(); id1++)
        {
          for (size_t id2 = 0; id2 < check_point_states_vect.size(); id2++)
          {
            const CurSituation new_cur_situation(
				cur_situation.get_state(),
	  			check_point_states_vect[id1],			
	  			check_point_states_vect[id2],			
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

            const NextSituation new_next_situation(
				next_situation.get_state(),
	  			check_point_states_vect[id1],			
	  			check_point_states_vect[id2],			
				next_situation.get_symbols_and_shifts()
				);


            assert (built_transitions_.count(new_cur_situation) == 0);
            couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
            assert (couple.second);

            assert (info_user_transitions_.count (new_cur_situation) == 0);
            info_user_transitions_[new_cur_situation] = RULE_MULTI_UPDATED;

          } // for (size_t id2 = 0
        } // for (size_t id1 = 0

        break;

      case ONLY_RIGHT_RULE_STATE_IS_EMBSTATE :
        if (
	      (is_initial_program_state (cur_situation.get_state()))
              ||
	      (is_embedded_program_state (cur_situation.get_state()))
           )
        {
	  // xxx yyy What about U1, U2 ???
          const CurSituation new_cur_situation(
				cur_situation.get_state(),
				cur_situation.get_state(), // not cur_situation.get_substate1()
				cur_situation.get_substate2(),
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

          const NextSituation new_next_situation(
				next_situation.get_state(),
				cur_situation.get_state(), // not cur_situation.get_substate1()
				cur_situation.get_substate2(),
				next_situation.get_symbols_and_shifts()
				);

          assert (built_transitions_.count(new_cur_situation) == 0);
          couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
          assert (couple.second);

          assert (info_user_transitions_.count (new_cur_situation) == 0);
          info_user_transitions_[new_cur_situation] = RULE_UPDATED;
        }
        else
        {
          for (size_t id1 = 0; id1 < check_point_states_vect.size(); id1++)
          {
            const CurSituation new_cur_situation(
				cur_situation.get_state(),
	  			check_point_states_vect[id1],			
				cur_situation.get_substate2(),
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

            const NextSituation new_next_situation(
				next_situation.get_state(),
	  			check_point_states_vect[id1],			
				cur_situation.get_substate2(),
				next_situation.get_symbols_and_shifts()
				);


            assert (built_transitions_.count(new_cur_situation) == 0);
            couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
            assert (couple.second);

            assert (info_user_transitions_.count (new_cur_situation) == 0);
            info_user_transitions_[new_cur_situation] = RULE_MULTI_UPDATED;

          } // for (size_t id1 = 0
        }

        break;

      case ONLY_LEFT_RULE_STATE_IS_EMBSTATE :
        assert (cur_situation.get_substate1() == get_neutral_cur_substate_alphabet_symbol ());
        assert (cur_situation.get_substate2() != get_neutral_cur_substate_alphabet_symbol ());

        assert (next_situation.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol ());
        assert (next_situation.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol ());

        assert ((cur_situation.get_state() == EMBSTATE_T4) || (cur_situation.get_state() == EMBSTATE_W4));

        if (cur_situation.get_state() == EMBSTATE_T4)
        {

          for (pos_iter_initial_check_point = user_required_initial_check_point_program_states.begin();
               pos_iter_initial_check_point != user_required_initial_check_point_program_states.end();
               pos_iter_initial_check_point++
  	       )
          {
            const CurSituation new_cur_situation(
  				cur_situation.get_state(),
  	  			pos_iter_initial_check_point->first.get_state(),
  				cur_situation.get_substate2(),
  				cur_situation.get_symbols(),
  				get_symbol_mtypes (cur_situation.get_symbols()), 
  				cur_situation.get_controlling_flag()
  				);

            const NextSituation new_next_situation(
				next_situation.get_state(),
	  			cur_situation.get_substate2(),  // not substate1
	  			get_neutral_next_not_the_same_substate_alphabet_symbol(),
				next_situation.get_symbols_and_shifts()
				);

  
            assert (built_transitions_.count(new_cur_situation) == 0);
            couple = built_transitions_.insert (
  			value_type (new_cur_situation, new_next_situation)
  			);
            assert (couple.second);
  
            assert (info_user_transitions_.count (new_cur_situation) == 0);
            info_user_transitions_[new_cur_situation] = RULE_UPDATED;
  
          }

          for (size_t id1 = 0; id1 < check_point_states_vect.size(); id1++)
          {
            const CurSituation new_cur_situation(
  				cur_situation.get_state(),
  	  			check_point_states_vect[id1],
  				cur_situation.get_substate2(),
  				cur_situation.get_symbols(),
  				get_symbol_mtypes (cur_situation.get_symbols()), 
  				cur_situation.get_controlling_flag()
  				);

            const NextSituation new_next_situation(
				next_situation.get_state(),
	  			cur_situation.get_substate2(),  // not substate1
	  			get_neutral_next_not_the_same_substate_alphabet_symbol(),			
				next_situation.get_symbols_and_shifts()
				);

  
            assert (built_transitions_.count(new_cur_situation) == 0);
            couple = built_transitions_.insert (
  			value_type (new_cur_situation, new_next_situation)
  			);
            assert (couple.second);
  
            assert (info_user_transitions_.count (new_cur_situation) == 0);
            info_user_transitions_[new_cur_situation] = RULE_MULTI_UPDATED;
  
          }

        }

        if (cur_situation.get_state() == EMBSTATE_W4)
        {
          for (size_t id1 = 0; id1 < check_point_states_vect.size(); id1++)
	  {
            const CurSituation new_cur_situation(
				cur_situation.get_state(),
	  			cur_situation.get_substate2(),  // not substate1
	  			check_point_states_vect[id1],			
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

            const NextSituation new_next_situation(
				next_situation.get_state(),
				next_situation.get_state(), // not substate1
	  			get_neutral_next_not_the_same_substate_alphabet_symbol(),			
				next_situation.get_symbols_and_shifts()
				);


            assert (built_transitions_.count(new_cur_situation) == 0);
            couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
            assert (couple.second);

            assert (info_user_transitions_.count (new_cur_situation) == 0);
            info_user_transitions_[new_cur_situation] = RULE_MULTI_UPDATED;
          }

        }

        break;

      case BOTH_RULE_STATES_ARE_ARE_NOT_EMBSTATE :
        assert (cur_situation.get_substate1() == get_neutral_cur_substate_alphabet_symbol ());
        assert (cur_situation.get_substate2() == get_neutral_cur_substate_alphabet_symbol ());
        assert (next_situation.get_substate1() == get_neutral_next_the_same_substate_alphabet_symbol ());
        assert (next_situation.get_substate2() == get_neutral_next_the_same_substate_alphabet_symbol ());

        // ----
        assert (built_transitions_.count(cur_situation) == 0);
        couple = built_transitions_.insert (
			value_type (cur_situation, next_situation)
			);
        assert (couple.second);

        assert (info_user_transitions_.count (cur_situation) == 0);
        info_user_transitions_[cur_situation] = RULE_OLD;

        break;

      case UNDEF_TmpEnum :
        assert (0);
        break;

      default :
        assert (0);
        break;

    } // switch (rule_flag)

  } // for (pos_iter = tmp_transitions.begin();


} // build_roof1_transitions



// =========
void TuringMachine::build_roof2_transitions ()
{
vector<state_t>	check_point_and_initial_states_vect;
map<CurSituation, state_t>::const_iterator pos_iter_check_point_vect;

  for (pos_iter_check_point_vect = user_required_check_point_program_states_.begin();
       pos_iter_check_point_vect != user_required_check_point_program_states_.end();
       pos_iter_check_point_vect++
       )
  {
    check_point_and_initial_states_vect.push_back(pos_iter_check_point_vect->second);
  }

  for (size_t id1 = 0; id1 < user_defined_initial_program_states_.size(); id1++)
  {
    assert (!belongs_to (user_defined_initial_program_states_[id1], check_point_and_initial_states_vect));
    check_point_and_initial_states_vect.push_back(user_defined_initial_program_states_[id1]);
    assert (belongs_to (user_defined_initial_program_states_[id1], check_point_and_initial_states_vect));
  }

  // ---------------------------
  assert (!built_transitions_.empty());
  set_old_info_user_transitions ();
Transitions_t	tmp_transitions (built_transitions_);
  assert (!tmp_transitions.empty());

  built_transitions_.clear();
  assert (built_transitions_.empty());  
  info_user_transitions_.clear();

Transitions_t::const_iterator	  	pos_iter;
typedef Transitions_t::value_type	value_type;
pair<Transitions_t::iterator, bool>	couple;

  // ----------------
  for (pos_iter = tmp_transitions.begin();
       pos_iter != tmp_transitions.end();
       pos_iter++)
  {
    const CurSituation  cur_situation (pos_iter->first);
    const NextSituation next_situation (pos_iter->second);

const bool halting_program_state_flag = is_halting_program_state (cur_situation.get_state());
const bool pair_summary_and_not_summary_program_states_flag = (
	belongs_to (cur_situation.get_state(), embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SUMMARY_CONTROL])
	&&
	!belongs_to (next_situation.get_state(), embedded_program_states_[EMBEDDED_PROGRAM_STATE_KIND__SUMMARY_CONTROL])
	);

    if ((!halting_program_state_flag) && (!pair_summary_and_not_summary_program_states_flag))
    {
      assert (built_transitions_.count(cur_situation) == 0);
      couple = built_transitions_.insert (
			value_type (cur_situation, next_situation)
			);
      assert (couple.second);

      assert (info_user_transitions_.count (cur_situation) == 0);
      info_user_transitions_[cur_situation] = RULE_OLD;

      continue;

    }

    // --------------------
    assert (halting_program_state_flag || pair_summary_and_not_summary_program_states_flag);

    for (size_t id1 = 0; id1 < check_point_and_initial_states_vect.size(); id1++)
    {
      const CurSituation new_cur_situation(
				cur_situation.get_state(),
	  			check_point_and_initial_states_vect[id1],
	  			get_neutral_cur_substate_alphabet_symbol(),
				cur_situation.get_symbols(),
				get_symbol_mtypes (cur_situation.get_symbols()), 
				cur_situation.get_controlling_flag()
				);

      const NextSituation new_next_situation(
				next_situation.get_state(),
	  			check_point_and_initial_states_vect[id1],
	  			get_neutral_next_the_same_substate_alphabet_symbol(),
				next_situation.get_symbols_and_shifts()
				);

      assert (built_transitions_.count(new_cur_situation) == 0);
      couple = built_transitions_.insert (
			value_type (new_cur_situation, new_next_situation)
			);
      assert (couple.second);

      assert (info_user_transitions_.count (new_cur_situation) == 0);
      info_user_transitions_[new_cur_situation] = RULE_MULTI_UPDATED;

    } // for (pos_iter_initial_check_point

  }

} // build_roof2_transitions


// =========
void TuringMachine::build_floor2_1_transitions ()
{
  // --- F.2.1. Returning MASTER_COMPUTATION_TAPE and TESTER_TAPE to left side

  remote_master_and_tester_tape (EMBSTATE_R1, EMBSTATE_R2);

} // build_floor2_1_transitions


// =========
void TuringMachine::build_floor2_2_transitions ()
{
  // --- F.2.2. Comparing MASTER_COMPUTATION_TAPE and TESTER_TAPE

  compare_master_and_tester_tape (EMBSTATE_R2, EMBSTATE_S1, EMBSTATE_V1);

} // build_floor2_2_transitions



// =========
void TuringMachine::build_floor3_1_transitions ()
{
  // --- F.3.1. Returning MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_masterbackup_tape (EMBSTATE_S1, EMBSTATE_S2);

} // build_floor3_1_transitions


// =========
void TuringMachine::build_floor3_2_transitions ()
{
  // --- F.3.2. Backuping up MASTER_COMPUTATION_TAPE to BACKUP_COMPUTATION_TAPE

  backup_master_and_masterbackup_tape (MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE, EMBSTATE_S2, EMBSTATE_S3);

} // build_floor3_2_transitions

// =========
void TuringMachine::build_floor3_3_transitions ()
{
  // --- F.3.3. Returning SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE to left side

  remote_synchro_and_synchrobackup_tape (EMBSTATE_S3, EMBSTATE_S4);

} // build_floor3_3_transitions



// =========
void TuringMachine::build_floor3_4_transitions ()
{
  // --- F.3.4. Backuping up SYNCHRONOUS_COMPUTATION_TAPE to SYNCHRONOUS_BACKUP_COMPUTATION_TAPE

  backup_synchro_and_synchrobackup_tape (SYNCHRONOUS_COMPUTATION_TAPE, SYNCHRONOUS_BACKUP_COMPUTATION_TAPE, EMBSTATE_S4, EMBSTATE_T1); 

} // build_floor3_4_transitions



// =========
void TuringMachine::build_floor4_1_transitions ()
{
  // --- F.4.1. Returning MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_masterbackup_tape (EMBSTATE_T1, EMBSTATE_T2);

} // build_floor4_1_transitions



// =========
void TuringMachine::build_floor4_2_transitions ()
{
  // --- F.4.2. Comparing MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE

  compare_master_and_masterbackup_tape (EMBSTATE_T2, EMBSTATE_T3, EMBSTATE_S1);

} // build_floor4_2_transitions



// =========
void TuringMachine::build_floor4_3_transitions ()
{
  // --- F.4.3. Returning MASTER_COMPUTATION_TAPE, SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_synchro_and_synchrobackup_tape (EMBSTATE_T3, EMBSTATE_T4);

} // build_floor4_3_transitions


// =========
void TuringMachine::build_floor4_4_transitions ()
{
  // --- F.4.4. Comparing SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
  // ---------- Setting head of MASTER_COMPUTATION_TAPE to continue computing


vector<state_t>	check_point_states_vect;

map<CurSituation, state_t>::const_iterator pos_iter;
  for (pos_iter = user_required_check_point_program_states_.begin();
       pos_iter != user_required_check_point_program_states_.end();
       pos_iter++
       )
  {
    check_point_states_vect.push_back(pos_iter->second);
  }

  compare_synchro_and_synchrobackup_tape_and_setting_master_tape (EMBSTATE_T4, check_point_states_vect,  EMBSTATE_S1); 

} // build_floor4_4_transitions



// =========
void TuringMachine::build_floor5_1_transitions ()
{
  // --- F.5.1. Returning MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_masterbackup_tape (EMBSTATE_V1, EMBSTATE_V2);

} // build_floor5_1_transitions



// =========
void TuringMachine::build_floor5_2_transitions ()
{
  // --- F.5.2. Backuping up BACKUP_COMPUTATION_TAPE to MASTER_COMPUTATION_TAPE

  backup_master_and_masterbackup_tape (BACKUP_COMPUTATION_TAPE, MASTER_COMPUTATION_TAPE, EMBSTATE_V2, EMBSTATE_V3);

} // build_floor5_2_transitions



// =========
void TuringMachine::build_floor5_3_transitions ()
{
  // --- F.5.3. Returning SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE to left side

  remote_synchro_and_synchrobackup_tape (EMBSTATE_V3, EMBSTATE_V4);

}  // build_floor5_3_transitions

// =========
void TuringMachine::build_floor5_4_transitions ()
{
  // --- F.5.4. Backuping up SYNCHRONOUS_BACKUP_COMPUTATION_TAPE to SYNCHRONOUS_COMPUTATION_TAPE

  backup_synchro_and_synchrobackup_tape (SYNCHRONOUS_BACKUP_COMPUTATION_TAPE, SYNCHRONOUS_COMPUTATION_TAPE, EMBSTATE_V4, EMBSTATE_W1); 

} // build_floor5_4_transitions



// =========
void TuringMachine::build_floor6_1_transitions ()
{
  // --- F.6.1. Returning MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_masterbackup_tape (EMBSTATE_W1, EMBSTATE_W2);

} // build_floor6_1_transitions



// =========
void TuringMachine::build_floor6_2_transitions ()
{
  // --- F.6.2. Comparing MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE

  compare_master_and_masterbackup_tape (EMBSTATE_W2, EMBSTATE_W3, EMBSTATE_V1);

} // build_floor6_2_transitions


// =========
void TuringMachine::build_floor6_3_transitions ()
{
  // --- F.6.3. Returning MASTER_COMPUTATION_TAPE, SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE to left side

  remote_master_and_synchro_and_synchrobackup_tape (EMBSTATE_W3, EMBSTATE_W4);

} // build_floor6_3_transitions


// =========
void TuringMachine::build_floor6_4_transitions ()
{
  // --- F.6.4. Comparing SYNCHRONOUS_COMPUTATION_TAPE and SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
  // ---------- Setting head of MASTER_COMPUTATION_TAPE to continue computing


vector<state_t>	check_point_and_initial_states_vect;

map<CurSituation, state_t>::const_iterator pos_iter;
  for (pos_iter = user_required_check_point_program_states_.begin();
       pos_iter != user_required_check_point_program_states_.end();
       pos_iter++
       )
  {
    check_point_and_initial_states_vect.push_back(pos_iter->second);
  }

  for (size_t p = 0; p < user_defined_initial_program_states_.size(); p++)
  {
    if (!belongs_to (user_defined_initial_program_states_[p], check_point_and_initial_states_vect))
    {
      check_point_and_initial_states_vect.push_back(user_defined_initial_program_states_[p]);
    }
  }

  compare_synchro_and_synchrobackup_tape_and_setting_master_tape (EMBSTATE_W4, check_point_and_initial_states_vect, EMBSTATE_V1); 

} // build_floor6_4_transitions


// =========
void TuringMachine::build_floor7_1_transitions ()
{
  // --- F.7.1. Returning MASTER_COMPUTATION_TAPE and TESTER_TAPE to left side

  remote_master_and_tester_tape (EMBSTATE_U1, EMBSTATE_U2);

} // build_floor7_1_transitions


// =========
void TuringMachine::build_floor7_2_transitions ()
{
  // --- F.7.2. Comparing MASTER_COMPUTATION_TAPE and TESTER_TAPE

  compare_master_and_tester_tape (EMBSTATE_U2, EMBSTATE_N1, EMBSTATE_V1);

} // build_floor7_2_transitions



// =========
void TuringMachine::set_old_info_user_transitions ()
{
map<CurSituation, string>::iterator pos;

  for (pos = info_user_transitions_.begin(); pos != info_user_transitions_.end(); pos++)
  {
    pos->second = RULE_OLD;
  }
  
}



