/*
 * $Workfile: $
 * $Revision: $
 * $Date: $
 *
 * Copyright (c) 1999-2000 National Semiconductor Corporation.
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of National 
 * Semiconductor Corporation. ("Confidential Information").
 * You shall not disclose such Confidential Information and shall use it only
 * in accordance with the terms of the license agreement you entered into
 * with National Semiconductor Corporation.
 * This code is supplied as is.
 *
 * This file contains routines to read the current hardware values.  It is
 * primarily used by diagnostic applications, such as Darwin, to refresh 
 * the hardware values displayed.
 *
 * Frame Buffer:
 *    gfx_get_display_width
 *    gfx_get_display_height
 *
 * Video Overlay:
 *	  gfx_get_video_enable
 *    gfx_get_video_format
 *    gfx_get_video_src_size
 *    gfx_get_video_offset
 *    gfx_get_video_scale
 *    gfx_get_video_dst_size;
 *    gfx_get_video_position
 *	  gfx_get_video_color_key;
 *    gfx_get_video_color_key_mask
 *    gfx_get_video_color_key_src
 *    gfx_set_video_filter
 *
 * Video Port:
 *    gfx_get_vip_enable
 *    gfx_get_vip_base
 *    gfx_get_vip_pitch
 *
 * Video Decoder:
 *    gfx_get_decoder_brightness
 */

/*
 *$History: $
*/

/*----------------*/
/*  FRAME BUFFER  */
/*----------------*/

/*---------------------------------*/
/*  ROUTINES TO READ BACK TIMINGS  */
/*---------------------------------*/

unsigned short gfx_get_hactive(void)
{
	return((unsigned short)((READ_REG32(DC_H_TIMING_1) & 0x07F8) + 8));
}

unsigned short gfx_get_hblank_start(void)
{
	return((unsigned short)((READ_REG32(DC_H_TIMING_2) & 0x07F8) + 8));
}

unsigned short gfx_get_hsync_start(void)
{
	return((unsigned short)((READ_REG32(DC_H_TIMING_3) & 0x07F8) + 8));
}

unsigned short gfx_get_hsync_end(void)
{
	return((unsigned short)(((READ_REG32(DC_H_TIMING_3) >> 16) & 0x07F8) + 8));
}

unsigned short gfx_get_hblank_end(void)
{
	return((unsigned short)(((READ_REG32(DC_H_TIMING_2) >> 16) & 0x07F8) + 8));
}

unsigned short gfx_get_htotal(void)
{
	return((unsigned short)(((READ_REG32(DC_H_TIMING_1) >> 16) & 0x07F8) + 8));
}

unsigned short gfx_get_vactive(void)
{
	return((unsigned short)((READ_REG32(DC_V_TIMING_1) & 0x07FF) + 1));
}

unsigned short gfx_get_vblank_start(void)
{
	return((unsigned short)((READ_REG32(DC_V_TIMING_2) & 0x07FF) + 1));
}

unsigned short gfx_get_vsync_start(void)
{
	return((unsigned short)((READ_REG32(DC_V_TIMING_3) & 0x07FF) + 1));
}

unsigned short gfx_get_vsync_end(void)
{
	return((unsigned short)(((READ_REG32(DC_V_TIMING_3) >> 16) & 0x07FF) + 1));
}

unsigned short gfx_get_vblank_end(void)
{
	return((unsigned short)(((READ_REG32(DC_V_TIMING_2) >> 16) & 0x07FF) + 1));
}

unsigned short gfx_get_vtotal(void)
{
	return((unsigned short)(((READ_REG32(DC_V_TIMING_1) >> 16) & 0x07FF) + 1));
}

/*-----------------------------------------------------------------------------
 * gfx_get_display_width
 *
 * This routine returns the current width, in pixels, of the active display.
 *-----------------------------------------------------------------------------
 */
unsigned short gfx_get_display_width(void)
{
	return((unsigned short)((READ_REG32(DC_H_TIMING_1) & 0x07F8) + 8));
}

/*-----------------------------------------------------------------------------
 * gfx_get_display_height
 *
 * This routine returns the current height, in lines, of the active display.
 *-----------------------------------------------------------------------------
 */
unsigned short gfx_get_display_height(void)
{
	return((unsigned short)((READ_REG32(DC_V_TIMING_1) & 0x07FF) + 1));
}

/*-----------------------------------------------------------------------------
 * gfx_get_display_bpp
 *
 * This routine returns the current color depth of the active display.
 *-----------------------------------------------------------------------------
 */
unsigned short gfx_get_display_bpp(void)
{
	switch(READ_REG32(DC_OUTPUT_CFG) & 3)
	{
		case 0: return(16);
		case 2: return(15);
	}
	return(8);
}

unsigned long gfx_get_display_offset(void)
{
	return(READ_REG32(DC_FB_ST_OFFSET) & 0x003FFFFF);
}

void gfx_get_display_palette(unsigned long *palette)
{
	unsigned long i, data;
	WRITE_REG32(DC_PAL_ADDRESS, 0);
	for (i = 0; i < 256; i++)
	{
		data = READ_REG32(DC_PAL_DATA);
		data = ((data << 2) & 0x000000FC) |
			((data << 4) & 0x0000FC00) |
			((data << 6) & 0x00FC0000);
		palette[i] = data;
	}
}

/*-----------------*/
/* HARDWARE CURSOR */
/*-----------------*/

unsigned long gfx_get_cursor_enable(void)
{
	return(READ_REG32(DC_GENERAL_CFG) & DC_GCFG_CURE);
}

unsigned long gfx_get_cursor_base(void)
{
	return(READ_REG32(DC_CURS_ST_OFFSET) & 0x003FFFFF);
}

unsigned long gfx_get_cursor_position(void)
{
	return((READ_REG32(DC_CURSOR_X) & 0x07FF) |
		((READ_REG32(DC_CURSOR_Y) << 16) & 0x03FF0000));
}

unsigned long gfx_get_cursor_offset(void)
{
	return(((READ_REG32(DC_CURSOR_X) >> 11) & 0x01F) |
		((READ_REG32(DC_CURSOR_Y) << 5) & 0x1F0000));
}

unsigned long gfx_get_cursor_color(int color)
{
	unsigned long data;
	if (color) 
	{
		WRITE_REG32(DC_PAL_ADDRESS, 0x101);
	}
	else
	{
		WRITE_REG32(DC_PAL_ADDRESS, 0x100);
	}
	data = READ_REG32(DC_PAL_DATA);
	data = ((data << 6) & 0x00FC0000) |
		((data << 4) & 0x0000FC00) |
		((data << 2) & 0x000000FC);
	return(data);
}

/*-----------------*/
/*  VIDEO OVERLAY  */
/*-----------------*/

/*-----------------------------------------------------------------------------
 * gfx_get_video_enable
 *
 * This routine returns the value "one" if video overlay is currently enabled,
 * otherwise it returns the value "zero".
 *-----------------------------------------------------------------------------
 */
int gfx_get_video_enable(void)
{
	if (READ_VID32(CS5530_VIDEO_CONFIG) & CS5530_VCFG_VID_EN) return(1);
	return(0);
}

/*-----------------------------------------------------------------------------
 * gfx_get_video_format
 *
 * This routine returns the current video overlay format.
 *-----------------------------------------------------------------------------
 */
int gfx_get_video_format(void)
{
	unsigned long vcfg;
	vcfg = READ_VID32(CS5530_VIDEO_CONFIG);
	if (vcfg & CS5530_VCFG_CSC_BYPASS) return(4);
	else return((vcfg >> 2) & 3);
}

/*-----------------------------------------------------------------------------
 * gfx_get_video_src_size
 *
 * This routine returns the size of the source video overlay buffer.  The 
 * return value is (height << 16) | width.
 *-----------------------------------------------------------------------------
 */
unsigned long gfx_get_video_src_size(void)
{
	unsigned long width = 0, height = 0;

#if GFX_VIDEOHW_CS5530 | GFX_VIDEOHW_SC1400

	/* DETERMINE SOURCE WIDTH FROM THE CS5530 VIDEO LINE SIZE */

	width = (READ_VID32(CS5530_VIDEO_CONFIG) >> 7) & 0x000001FE;
	if (READ_VID32(CS5530_VIDEO_CONFIG) & CS5530_VCFG_LINE_SIZE_UPPER)
		width += 512;

#endif /* GFX_VIDEOHW_CS5530 | GFX_VIDEOHW_SC1400 */

	if (width)
	{
		/* DETERMINE HEIGHT BY DIVIDING TOTAL SIZE BY WIDTH */

		height = ((READ_REG32(DC_BUF_SIZE) >> 10) & 0x000FFFC0) / (width << 1);
	}
	return((height << 16) | width);
}

/*-----------------------------------------------------------------------------
 * gfx_get_video_offset
 *
 * This routine returns the current offset for the video overlay buffer.
 *-----------------------------------------------------------------------------
 */
unsigned long gfx_get_video_offset(void)
{
	return(READ_REG32(DC_VID_ST_OFFSET) & 0x001FFFFF);
}

/*---------------------------------------------------------------------------
 * gfx_get_video_scale
 * 
 * This routine returns the scale factor for the video overlay window.
 *---------------------------------------------------------------------------
 */
unsigned long gfx_get_video_scale(void)
{
	return(READ_VID32(CS5530_VIDEO_SCALE));
}

/*---------------------------------------------------------------------------
 * gfx_get_video_dst_size
 * 
 * This routine returns the size of the displayed video overlay window.
 *---------------------------------------------------------------------------
 */
unsigned long gfx_get_video_dst_size(void)
{
	unsigned long xsize, ysize;

	xsize = READ_VID32(CS5530_VIDEO_X_POS);
	xsize = ((xsize >> 16) & 0x3FF) - (xsize & 0x03FF);
	ysize = READ_VID32(CS5530_VIDEO_Y_POS);
	ysize = ((ysize >> 16) & 0x3FF) - (ysize & 0x03FF);

	/* !!! ADD INITIAL READ ADDRESS */

	return((ysize << 16) | xsize);
}

/*---------------------------------------------------------------------------
 * gfx_get_video_position
 * 
 * This routine returns the position of the video overlay window.  The
 * return value is (ypos << 16) | xpos.
 *---------------------------------------------------------------------------
 */
unsigned long gfx_get_video_position(void)
{
	unsigned long hadjust, vadjust;
	unsigned long xpos, ypos;

	/* READ HARDWARE POSITION */

	xpos = READ_VID32(CS5530_VIDEO_X_POS) & 0x000003FF;
	ypos = READ_VID32(CS5530_VIDEO_Y_POS) & 0x000003FF;
	
	/* GET ADJUSTMENT VALUES */

	hadjust = ((READ_REG32(DC_H_TIMING_1) >> 16) & 0x7FF) - 13;
	hadjust -= ((READ_REG32(DC_H_TIMING_3) >> 16) & 0x7FF);
	vadjust = ((READ_REG32(DC_V_TIMING_1) >> 16) & 0x7FF) + 1;
	vadjust -= ((READ_REG32(DC_V_TIMING_3) >> 16) & 0x7FF);
	xpos -= hadjust;
	ypos -= vadjust;

	/* !!! ADD INITIAL READ ADDRESS */

	return((ypos << 16) | (xpos & 0x0000FFFF));
}

/*---------------------------------------------------------------------------
 * gfx_get_video_color_key
 * 
 * This routine returns the current video color key value.
 *---------------------------------------------------------------------------
 */
unsigned long gfx_get_video_color_key(void)
{
	return(READ_VID32(CS5530_VIDEO_COLOR_KEY));
}

/*---------------------------------------------------------------------------
 * gfx_get_video_color_key_mask
 * 
 * This routine returns the current video color mask value.
 *---------------------------------------------------------------------------
 */
unsigned long gfx_get_video_color_key_mask(void)
{
	return(READ_VID32(CS5530_VIDEO_COLOR_MASK));
}

/*---------------------------------------------------------------------------
 * gfx_get_video_color_key_src
 * 
 * This routine returns 0 for video data compare, 1 for graphics data.
 *---------------------------------------------------------------------------
 */
int gfx_get_video_color_key_src(void)
{
	if (READ_VID32(CS5530_DISPLAY_CONFIG) & CS5530_DCFG_VG_CK) return(0);
	return(1);
}

/*---------------------------------------------------------------------------
 * gfx_get_video_filter
 * 
 * This routine returns if the filters are currently enabled.
 *---------------------------------------------------------------------------
 */
int gfx_get_video_filter(void)
{
	int retval = 0;
	if (READ_VID32(CS5530_VIDEO_CONFIG) & CS5530_VCFG_X_FILTER_EN) 
		retval |= 1;
	if (READ_VID32(CS5530_VIDEO_CONFIG) & CS5530_VCFG_Y_FILTER_EN) 
		retval |= 2;
	return(retval);
}

/*---------------------------------------------------------------------------
 * gfx_get_alpha_enable
 * 
 * This routine returns 1 if the selected alpha window is currently 
 * enabled, or 0 if it is currently disabled.
 *---------------------------------------------------------------------------
 */
void gfx_get_alpha_enable(int *enable)
{
	unsigned long value = 0;
	*enable = 0;

#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1200) {
#endif /* GFX_VIDEOHW_DYNAMIC */

#if GFX_VIDEOHW_SC1200

	if (gfx_alpha_select <= 2)
	{
		value = READ_VID32(SC1200_ALPHA_CONTROL_1 + (gfx_alpha_select << 4));
		if (value & SC1200_ACTRL_WIN_ENABLE) *enable = 1;
	}

#endif /* GFX_VIDEOHW_SC1200 */

#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */

	return;
}

/*---------------------------------------------------------------------------
 * gfx_get_alpha_size
 * 
 * This routine returns the size of the currently selected alpha region.
 *---------------------------------------------------------------------------
 */
void gfx_get_alpha_size(unsigned short *x, unsigned short *y, 
	unsigned short *width, unsigned short *height)
{
	unsigned long value = 0;

	*x = 0;
	*y = 0;
	*width = 0;
	*height = 0;

#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1200) {
#endif /* GFX_VIDEOHW_DYNAMIC */

#if GFX_VIDEOHW_SC1200

	if (gfx_alpha_select <= 2)
	{
		value = READ_VID32(SC1200_ALPHA_XPOS_1 + (gfx_alpha_select << 4));
		*x = (unsigned short) (value & 0x000007FF);
		*width = (unsigned short) ((value >> 16) & 0x000007FF) - *x;
		value = READ_VID32(SC1200_ALPHA_YPOS_1 + (gfx_alpha_select << 4));
		*y = (unsigned short) (value & 0x000007FF);
		*height = (unsigned short) ((value >> 16) & 0x000007FF) - *y;
	}

#endif /* GFX_VIDEOHW_SC1200 */

#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */

	return;
}

/*---------------------------------------------------------------------------
 * gfx_get_alpha_value
 * 
 * This routine returns the alpha value and increment/decrement value of 
 * the currently selected alpha region.
 *---------------------------------------------------------------------------
 */
void gfx_get_alpha_value(unsigned char *alpha, char *delta)
{
	unsigned long value = 0;
	*alpha = 0;
	*delta = 0;

/* CODE FOR SC1200 */

#if GFX_VIDEOHW_SC1200
#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1200) {
#endif /* GFX_VIDEOHW_DYNAMIC */

	if (gfx_alpha_select <= 2)
	{
		value = READ_VID32(SC1200_ALPHA_CONTROL_1 + (gfx_alpha_select << 4));
		*alpha = (unsigned char) (value & 0x00FF);
		*delta = (char) ((value >> 8) & 0x00FF);
	}
#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */
#endif /* GFX_VIDEOHW_SC1200 */
	return;
}

/*---------------------------------------------------------------------------
 * gfx_get_alpha_priority
 * 
 * This routine returns the priority of the currently selected alpha region.
 *---------------------------------------------------------------------------
 */
void gfx_get_alpha_priority(int *priority)
{
	unsigned long pos = 0, value = 0;
	*priority = 0;

#if GFX_VIDEOHW_SC1200
#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1200) 
	{
#endif /* GFX_VIDEOHW_DYNAMIC */

	if (gfx_alpha_select <= 2)
	{
		value = READ_VID32(SC1200_VID_ALPHA_CONTROL);
		pos = 16 + (gfx_alpha_select << 1);
		*priority = (int) ((value >> pos) & 3);
	}

#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */
#endif /* GFX_VIDEOHW_SC1200 */
	return;
}

/*---------------------------------------------------------------------------
 * gfx_get_alpha_color
 * 
 * This routine returns the color register value for the currently selected 
 * alpha region.  Bit 24 is set if the color register is enabled.
 *---------------------------------------------------------------------------
 */
void gfx_get_alpha_color(unsigned long *color)
{
	*color = 0;

#if GFX_VIDEOHW_SC1200

#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1200) 
	{
#endif /* GFX_VIDEOHW_DYNAMIC */

	if (gfx_alpha_select <= 2)
	{
		*color = READ_VID32(SC1200_ALPHA_COLOR_1 + (gfx_alpha_select << 4));
	}

#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */
#endif /* GFX_VIDEOHW_SC1200 */
	return;
}

/*--------------*/
/*  VIDEO PORT  */
/*--------------*/

int gfx_get_vip_enable(void)
{
	if (READ_VIP32(SC1400_VIP_CONTROL) & 0x00000100) return(1);
	return(0);
}

unsigned long gfx_get_vip_base(int odd)
{
	/* MASK BIT 23 AND ABOVE TO MAKE IT A TRUE OFFSET */

	if (odd) return(READ_VIP32(SC1400_VIP_ODD_BASE) & 0x007FFFFF);
	return(READ_VIP32(SC1400_VIP_EVEN_BASE) & 0x007FFFFF);
}

unsigned long gfx_get_vip_pitch(void)
{
	return(READ_VIP32(SC1400_VIP_PITCH) & 0x0000FFFF);
}

int gfx_get_vbi_enable(void)
{
	if (READ_VIP32(SC1400_VIP_CONTROL) & 0x00000200) return(1);
	return(0);
}

unsigned long gfx_get_vbi_base(int odd)
{
	/* MASK BIT 23 AND ABOVE TO MAKE IT A TRUE OFFSET */

	if (odd) return(READ_VIP32(SC1400_VBI_ODD_BASE) & 0x007FFFFF);
	return(READ_VIP32(SC1400_VBI_EVEN_BASE) & 0x007FFFFF);
}

unsigned long gfx_get_vbi_pitch(void)
{
	return(READ_VIP32(SC1400_VBI_PITCH) & 0x0000FFFF);
}

/*-----------------*/
/*  VIDEO DECODER  */
/*-----------------*/

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_brightness
 *
 * This routine returns the current brightness of the video decoder.  
 *-----------------------------------------------------------------------------
 */
unsigned char gfx_get_decoder_brightness(void)
{
	unsigned char brightness = 0;
#if GFX_DECODER_SAA7114
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_BRIGHTNESS, &brightness);
#endif /* GFX_DECODER_SAA7114 */
	return(brightness);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_contrast
 *
 * This routine returns the current contrast of the video decoder.  
 *-----------------------------------------------------------------------------
 */
unsigned char gfx_get_decoder_contrast(void)
{
	unsigned char contrast = 0;
#if GFX_DECODER_SAA7114
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_CONTRAST, &contrast);
	contrast <<= 1;
#endif /* GFX_DECODER_SAA7114 */
	return(contrast);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_hue
 *
 * This routine returns the current hue of the video decoder.  
 *-----------------------------------------------------------------------------
 */
char gfx_get_decoder_hue(void)
{
	unsigned char hue = 0;
#if GFX_DECODER_SAA7114
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HUE, &hue);
#endif /* GFX_DECODER_SAA7114 */
	return((char)hue);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_saturation
 *
 * This routine returns the current saturation of the video decoder.  
 *-----------------------------------------------------------------------------
 */
unsigned char gfx_get_decoder_saturation(void)
{
	unsigned char saturation = 0;
#if GFX_DECODER_SAA7114
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_SATURATION, &saturation);
	saturation <<= 1;
#endif /* GFX_DECODER_SAA7114 */
	return(saturation);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_input_offset
 *
 * This routine returns the offset into the input window.
 *-----------------------------------------------------------------------------
 */
unsigned long gfx_get_decoder_input_offset()
{
	unsigned long value = 0;
	
#if GFX_DECODER_SAA7114

	unsigned char data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_OFFSET_LO, &data);
	value = (unsigned long) data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_OFFSET_HI, &data);
	value |= ((unsigned long) data) << 8;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_OFFSET_LO, &data);
	value |= ((unsigned long) data) << 16;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_OFFSET_HI, &data);
	value |= ((unsigned long) data) << 24;

#endif /* GFX_DECODER_SAA7114 */

	return(value);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_input_size
 *
 * This routine returns the current size of the input window
 *-----------------------------------------------------------------------------
 */
unsigned long gfx_get_decoder_input_size()
{
	unsigned long value = 0;
	
#if GFX_DECODER_SAA7114

	unsigned char data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_INPUT_LO, &data);
	value = (unsigned long) data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_INPUT_HI, &data);
	value |= ((unsigned long) data) << 8;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_INPUT_LO, &data);
	value |= ((unsigned long) data) << 17;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_INPUT_HI, &data);
	value |= ((unsigned long) data) << 25;

#endif /* GFX_DECODER_SAA7114 */

	return(value);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_output_size
 *
 * This routine returns the current size of the output window.
 *-----------------------------------------------------------------------------
 */
unsigned long gfx_get_decoder_output_size()
{
	unsigned long value = 0;
	
#if GFX_DECODER_SAA7114

	unsigned char data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_OUTPUT_LO, &data);
	value = (unsigned long) data;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_HORZ_OUTPUT_HI, &data);
	value |= ((unsigned long) data) << 8;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_OUTPUT_LO, &data);
	value |= ((unsigned long) data) << 17;
	gfx_i2c_read(SAA7114_CHIPADDR, SAA7114_VERT_OUTPUT_HI, &data);
	value |= ((unsigned long) data) << 25;

#endif /* GFX_DECODER_SAA7114 */

	return(value);
}

/*-----------------------------------------------------------------------------
 * gfx_get_decoder_vbi_format
 *
 * This routine returns the current format of the VBI data.
 *-----------------------------------------------------------------------------
 */
int gfx_get_decoder_vbi_format(void)
{
	unsigned char format = 0, data;
#if GFX_DECODER_SAA7114
	gfx_i2c_read(SAA7114_CHIPADDR, 0x54, &data);
	if (data == 0x77) format = 1;
	else if (data == 0x55) format = 2;
#endif /* GFX_DECODER_SAA7114 */
	return(format);
}

unsigned long gfx_read_crc(void)
{
	unsigned long crc = 0xFFFFFFFF;

#if GFX_VIDEOHW_CS5530
#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_CS5530) 
	{
#endif /* GFX_VIDEOHW_DYNAMIC */
	if (gfx_test_timing_active())
	{
		/* WAIT UNTIL ACTIVE DISPLAY */

		while(!gfx_test_vertical_active());

		/* RESET CRC DURING ACTIVE DISPLAY */

		WRITE_VID32(CS5530_CRCSIG_TFT_TV, 0);
		WRITE_VID32(CS5530_CRCSIG_TFT_TV, 1);

		/* WAIT UNTIL NOT ACTIVE, THEN ACTIVE, NOT ACTIVE, THEN ACTIVE */
		
		while(gfx_test_vertical_active());
		while(!gfx_test_vertical_active());
		while(gfx_test_vertical_active());
		while(!gfx_test_vertical_active());
		crc = READ_VID32(CS5530_CRCSIG_TFT_TV) >> 8;
	}
#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */
#endif /* GFX_VIDEOHW_CS5530 */

#if GFX_VIDEOHW_SC1400
#if GFX_VIDEOHW_DYNAMIC
	if (gfx_videohw_type & GFX_VIDEOHW_TYPE_SC1400) 
	{
#endif /* GFX_VIDEOHW_DYNAMIC */
	if (gfx_test_timing_active())
	{
		/* WAIT UNTIL ACTIVE DISPLAY */

		while(!gfx_test_vertical_active());

		// RESET CRC DURING ACTIVE DISPLAY

		WRITE_VID32(SC1400_VID_CRC, 0);
		WRITE_VID32(SC1400_VID_CRC, 1);

		/* WAIT UNTIL NOT ACTIVE, THEN ACTIVE, NOT ACTIVE, THEN ACTIVE */
		
		while(gfx_test_vertical_active());
		while(!gfx_test_vertical_active());
		while(gfx_test_vertical_active());
		while(!gfx_test_vertical_active());
		crc = READ_VID32(SC1400_VID_CRC) >> 8;
	}
#if GFX_VIDEOHW_DYNAMIC
	}
#endif /* GFX_VIDEOHW_DYNAMIC */
#endif /* GFX_VIDEOHW_SC1400 */


	return(crc);
}	

/* END OF FILE */
