/*
 *  kbdtest.c      basic PS/2 keyboard input test
 *
 *  Test operation of a PS/2 keyboard wired to P1 on
 *  the Raspberry Pi.
 *
 *  PS/2 CLK line goes to P1-16 through voltage divider
 *  PS/2 DATA line goes to P1-18 through voltage divider
 *  PS/2 5 VDC line goes to P1-2 or P1-4
 *  PS/2 GND line goes to P1-6
 *
 *  You MUST use a voltage divider between the PS/2 CLK
 *  or DATA lines and the RasPi P1 connector!  The RasPi
 *  cannot handle 5 VDC logic signals.  If you don't
 *  divide the PS/2 signals down to 3.3 VDC, you will
 *  damage the RasPi!
 *
 *  Building this code will yield a kernel.img file.  Copy
 *  this file to the root directory in a RasPi boot SD card,
 *  install the card into your RasPi, power it up, and the
 *  video display should show the test results.
 *
 *  Note that you will have to set the display parameters
 *  in your SD card's config.txt file to match those of
 *  the console display parameters.  For example, if your
 *  console library is built to use 800 x 600 pixel display,
 *  you will need to edit your SD card's config.txt to
 *  show a framebuffer_width of 800 and a framebuffer_height
 *  of 600.
 *
 *  Karl Lunt   7 Aug 2015
 */


#include  <stdio.h>
#include  "target.h"
#include  "bcm2835.h"
#include  "raspi.h"
#include  "chario.h"
#include  "printf.h"
#include  "console.h"
#include  "systimer.h"
#include  "ps2kbd.h"

static void		(*CallbackForPS2Kbd)();
int				c;


/** Main function - we'll never return from here */
void kernel_main( unsigned int r0, unsigned int r1, unsigned int atags )
{
	int				index;
	char			buff[1024];

	console_init();

/*
 *  Hook the character I/O system into the video console for output and the
 *  PS/2 keyboard for input.
 */
	chario_OutRegister((outcharFuncPtr)&console_putc);	// aim printf at the video display
	chario_InRegister((incharFuncPtr)&ps2kbd_Read);		// read chars from PS/2 keyboard
	chario_AvailRegister((availcharFuncPtr)&ps2kbd_Available);	// check PS/2 kbd for chars

	printf("\n\rkbdtest\n\r");
	
/*
 *  Define the GPIO pins used by the PS/2 keyboard.  Also select the keyboard
 *  map and record the PS/2 function for handling keyboard interrupts.
 */
	ps2kbd_Init(PS2_KBD_DATA_PIN, PS2_KBD_CLK_PIN, &PS2Keymap_US, &CallbackForPS2Kbd);


	_enable_interrupts();

	/* Never exit as there is no OS to exit to! */
	while(1)
	{
		index = 0;
		printf("\n\r> ");
		while (get_line_r(buff, sizeof(buff)-1, &index) == 0)
		{
			// I don't have any background stuff, but if I did, it would go here.
		}
		printf("\n\rYou typed: %s", buff);
	}
}



/*
 *  Declare the universal ISR handler for IRQ interrupts.
 *
 *  The only interrupt we expect is from the PS/2 keyboard.  Code
 *  for handling that interrupt is in the callback function recorded
 *  from our earlier call to ps2kbd_Init().  All we have to do is
 *  invoke that callback function and the keyboard interrupt is
 *  handled.
 */
void __attribute__((interrupt("IRQ"))) interrupt_vector(void)
{
	CallbackForPS2Kbd();						// process any keyboard interrupt
}


