#include  <stdio.h>
#include  "LPC17xx.h"
#include  "target_LPC1768.h"
#include  "TinyBasicLike.h"

#include "iap.h"



#define  LPC1768_VERSION_INFO		"v0.01"

const unsigned char					t_TargetIDMsg[] =
		"\nLPC1768 (mbed) TinyBasicLike target (32-bit) " LPC1768_VERSION_INFO "  1 Oct 2023  KEL";



/*
 * Local functions
 */
uint32_t			UART_Init(uint32_t  baudrate);
static void			UART_Putch(int  c);
static int			UART_Getch(void);
static int			UART_Avail(void);

static  void		ADC_Init(void);

static  void  		Flash_EraseSector(void);
static  void  		Flash_WriteSector(void);


/*
 * Uncomment one and only one of the following #defines; this selects the UART
 * to use for the console.
 */
//#define  CONSOLE_UART		UART1
#define  CONSOLE_UART		UART3


struct  Port
{
	DATA_SIZE				*portaddr;					// addr of the port
	unsigned int			size;						// size of data cell (bits)
	unsigned char			writeable;					// TRUE if port can be modified
};

#define  WRITEABLE			TRUE
#define  READONLY			FALSE



uint32_t			SystemCoreClock;		// global variable, core clock frequency
volatile uint32_t	sys_ticks;				// updated by SysTick counter IRQ 



uint32_t			OutputStream;
uint32_t			InputStream;

DATA_SIZE			*timeraddrs[T_NUM_TIMERS] = {0, 0, 0, 0};


unsigned char		t_program[TOTAL_RAM_REQUIRED];  // define RAM block holding entire program, stacks, and vars

/* --------------------------------------------------------------------
 *                 Support for flash program storage
 * --------------------------------------------------------------------
 */
 
/*
 * Define variables used by the IAP (in-application programming) system for modifying
 * on-chip flash.
 */
uint32_t			iap_cmd[5];
uint32_t			iap_result[5];

char		*rfileptr;
char		*rfilestart;
char		*rfiledatastart;
char		*rfiledataend;

char		*wfileptr;
char		*wfilestart;
char		*wfiledatastart;
char		*wfiledataend;

int			filebufferchanged;

extern FILE					*rfp;			// use core's version of read file pointer
extern FILE					*wfp;			// use core's version of write file pointer

 
/*
 * Declare a buffer large enough to hold one entire flash sector (nominally 32K, could be different).
 */
uint8_t				flashbuffer[FLASH_SECTOR_STORAGE_SIZE];
int					filebufferchanged;


/* ----------------- End of flash program storage support ------------------ */



/*
 * Define masks for accessing the GPIO lines that control the four on-board LEDs.
 */
#define		LED1_MASK				(1<<18)
#define		LED2_MASK				(1<<20)
#define		LED3_MASK				(1<<21)
#define		LED4_MASK				(1<<23)


/*
 * Define a range of virtual ports.  These appear as addresses in the port table, but
 * actually identify individual I/O lines or dedicated variables.
 * 
 * For example, a virtual port might be bit 13 of a 16-bit output port.  Writes to this
 * virtual port would manipulate only bit 13 of the corresponding port.
 */
 enum  VirtualPort
 {
	 LED1 = 0,
	 LED2,
	 LED3,
	 LED4,
	 ADC0,
	 ADC1,
	 ADC2,
	 ADC3,
	 NUM_VIRTUAL_PORTS				// this index is used as number of virtual ports
 };


/*
 * Define a table showing the names of all supported ports.  Structure of this table
 * matches those of the similar tables in the core program.
 * 
 * NOTE: The strings for each port name are NOT followed by a comma!  This is a single,
 * large string of space-separated names, it is not an array of strings.
 * 
 * For example:
 *	const unsigned char			ports_tab[] =
 *  {
 *     "PORTA "					// <-- see, no comma!
 *     "PORTB "
 *     "PORTC "
 *     "\0"
 *   }
 */
const unsigned char			ports_tab[] =
{
/* all virtual ports must appear first in this table! */
	"LED1 "
	"LED2 "
	"LED3 "
	"LED4 "
	"ADC0 "
	"ADC1 "
	"ADC2 "
	"ADC3 "
/* all traditional I/O ports must follow the virtual ports! */
	"\0"
};

/*
 * This is a table containing one entry for each supported port.  Each entry
 * contains the address of the corresponding port register and the size (in bits)
 * of the port.
 * 
 * Order is IMPORTANT!  The entries in this table must match exactly the order
 * of the port names in ports_tab[] above.
 */
struct Port					porttable[] =
{
	{(DATA_SIZE *) LED1,	 			1, WRITEABLE},
	{(DATA_SIZE *) LED2,	 			1, WRITEABLE},
	{(DATA_SIZE *) LED3,	 			1, WRITEABLE},
	{(DATA_SIZE *) LED4,	 			1, WRITEABLE},
	{ADC0,							   32, READONLY},
	{ADC1,	   						   32, READONLY},
	{ADC2,	   						   32, READONLY},
	{ADC3,	   						   32, READONLY},
};

#define  NUM_PORTS			(sizeof(porttable)/sizeof(porttable[0]))


/*
 *  init      perform low-level initialization of the mbed board
 *
 *  main() must call this routine before doing any other major tasks on the mbed!
 */
void			init(void)
{
/*
 *  Clock initialization for the Mbed development board.  This board
 *  uses a 12 MHz oscillator as the main clock source.  This code
 *  sets the core clock to 96 MHz.
 */
	LPC_SC->SCS = 0x20;					// enable external oscillator (12 MHz crystal)
	while (!(LPC_SC->SCS & (1<<6))) ;	// wait for main oscillator to stablilize
	
	LPC_SC->CLKSRCSEL = 0x01;			// set main oscillator (12 MHz crystal) as PLL source
	LPC_SC->CCLKCFG = 0x03;				// set CPU clock (CCLK) to PLL0 output / 4
	
	LPC_SC->PCLKSEL0 = 0x0;				// peripherals use CPU clock / 4
	LPC_SC->PCLKSEL1 = 0x0;				// peripherals use CPU clock / 4

/*
 *  The following code sets up the Mbed's core clock to 96 MHz.
 *  This code sets PLL0 output to 384 MHz.  The CPU clock
 *  configuration above (CCLKCFG) divides this value by 4
 *  to get a final CPU clock of 96 MHz.
 */
	LPC_SC->PLL0CFG = (0<<16) | (15<<0);	// PPL0 config, M=16, N=1, Fout=384
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	LPC_SC->PLL0CON = 0x01;				// enable PLL0
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	while (!(LPC_SC->PLL0STAT & (1<<26)))	;		// wait for PLL0 lock

	LPC_SC->PLL0CON = 0x03;				// enable and connect PLL0
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	while (!(LPC_SC->PLL0STAT & ((1<<25) | (1<<24))))  ;	// wait for PLL0 enable and connect

	SystemCoreClock = 96000000;			// need to make this value available globally
	
/*
 *  Set up PLL1 for use as a USB clock.  The output must be set to 48 MHz, based
 *  on an oscillator of 12 MHz.
 */
	LPC_SC->PLL1CFG = 0x00000023;		// PLL1 config, M=4 (bits 4:0=3), P=2 (bits 6:5=1)
	LPC_SC->PLL1FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL1FEED = 0x55;

	LPC_SC->PLL1CON = 0x01;				// enable PLL1
	LPC_SC->PLL1FEED = 0xAA;
	LPC_SC->PLL1FEED = 0x55;

	while (!(LPC_SC->PLL1STAT & (1<<10)))  ;		// wait for PLL1 lock

	LPC_SC->PLL1CON = 0x03;				// enable and connect PLL1
	LPC_SC->PLL1FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL1FEED = 0x55;

	while (!(LPC_SC->PLL1STAT & ((1<<9) | (1<<8))))  ;  // wait for PLL1 enable and connect
}


/*
 *  Define the SysTick IRQ handler.
 *
 *  This function will be used instead of the default handler in startup_LPC17xx.s, which
 *  was declared weak so it could be overwritten.
 */
void  SysTick_Handler(void) 
{
    ++sys_ticks;

	for (int  n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n])
		{
			if (*timeraddrs[n])  *timeraddrs[n] = *timeraddrs[n] - 1;
		}
	}
}



/*
 * Flash_EraseSector      reset all bytes in flash storage sector to 0xFF
 * 
 * This routine supports a single, fixed flash sector for sector erase.
 */
static  void  Flash_EraseSector(void)
{
	iap_cmd[0] = IAP_CMD_PREPARE_SECTORS;			// command, prepare sector for write
	iap_cmd[1] = FLASH_SECTOR_STORAGE_NUM;			// start sector number
	iap_cmd[2] = FLASH_SECTOR_STORAGE_NUM;			// only erase one sector!
	__disable_irq();								// MUST shut off interrupts!
	IAP_EXECUTE_CMD(iap_cmd, iap_result);			// do the op
	__enable_irq();									// as you were
	if (iap_result[0] != IAP_STA_CMD_SUCCESS)		// if that didn't work...
	{
		//printstr("Flash_EraseSector of sector ");	// debug
		//printnum(FLASH_SECTOR_STORAGE_NUM, RADIX_DECIMAL);
		//printstr("failed; result code is ");
		//printnum(iap_result[0], RADIX_DECIMAL);
		//printstr("\n\r");
	}
	else
	{
		iap_cmd[0] = IAP_CMD_ERASE_SECTORS;				// command, erase flash sector
		iap_cmd[1] = FLASH_SECTOR_STORAGE_NUM;			// sector to erase
		iap_cmd[2] = FLASH_SECTOR_STORAGE_NUM;			// only erase one sector!
		iap_cmd[3] = SystemCoreClock/1000UL;			// have to supply the CPU frequency in kHz
		__disable_irq();								// MUST shut off interrupts!
		IAP_EXECUTE_CMD(iap_cmd, iap_result);			// do the op
		__enable_irq();									// as you were
	}
}


/*
 * Flash_WriteSector      copy contents of RAM buffer to flash sector
 * 
 * This routine supports a single, fixed flash sector for program storage.
 */
static  void  Flash_WriteSector(void)
{
	uint32_t		fladdr;
	uint32_t		ramaddr;
	
	fladdr = FLASH_SECTOR_STORAGE_ADDR;
	ramaddr = flashbuffer;

/*
 * The IAP system supports a maximum flash write size of 4096 bytes.  Updating a
 * flash sector must be broken up into several 4K blocks.
 */
	for (int n=0; n<FLASH_SECTOR_STORAGE_SIZE/4096; n++)
	{
		iap_cmd[0] = IAP_CMD_PREPARE_SECTORS;			// command, prepare sector for write
		iap_cmd[1] = FLASH_SECTOR_STORAGE_NUM;			// start sector number
		iap_cmd[2] = FLASH_SECTOR_STORAGE_NUM;			// end sector number
		__disable_irq();								// MUST shut off interrupts!
		IAP_EXECUTE_CMD(iap_cmd, iap_result);			// do the op
		__enable_irq();									// as you were

		iap_cmd[0] = IAP_CMD_COPY_RAM_TO_FLASH;			// command, write RAM to flash
		iap_cmd[1] = fladdr;							// addr in dest flash area to write data
		iap_cmd[2] = (uint32_t)ramaddr;					// addr of source buffer in RAM
		iap_cmd[3] = 4096;								// must be 256|512|1024|4096
		iap_cmd[4] = SystemCoreClock/1000UL;			// have to supply the CPU frequency in kHz
		__disable_irq();								// MUST shut off interrupts!
		IAP_EXECUTE_CMD(iap_cmd, iap_result);			// do the op
		__enable_irq();									// as you were

		fladdr += 4096;									// move to next block in flash area
		ramaddr += 4096;								// move to next block in RAM buffer
	}
}



/*
 * File support routines
 * 
 * These are NOT FAT_Fs routines!  These are simplified versions of file
 * support for using the on-chip flash and a large block of RAM as a file
 * system.
 */

/*
 * FindFileByName      search flash buffer for file name
 * 
 * If this routine finds the requested file, it writes the address of
 * the file block to argument ptr.  Note that this is the address of the
 * first char of the file header!  To access the first byte of file data,
 * you must adjust the address by adding the size of the file header,
 * SIZEOF_FLASH_FILE_HEADER.
 */
static int  FindFileByName(char  *name, char  **ptr)
{
	char			*fptr;
	
	for (int n = 0; n<NUM_FLASH_FILES; n++)
	{
		 fptr = (char *)(flashbuffer + (n*FLASH_FILE_SIZE));
		
		 for (int i=0; i<MAX_LEN_FILENAME+1; i++)
		 {
			 if (fptr[i] == name[i])
			 {
				 if (fptr[i] == 0)		// end of file name is marked with null byte
				 {
					 *ptr = fptr;
					 return  S_OK;
				 }
			 }
			 else  break;
		 }
	 }
	 *ptr = 0;
	 return  E_FILE_NOT_FOUND;
 }
 
 
 static int  FindNextOpenFile(char  **ptr)
 {
	 int			n;
	 
	 *ptr = (char *)flashbuffer;
	 for (n=0; n<NUM_FLASH_FILES; n++)
	 {
		 if (**ptr == 0xff)  return  S_OK;			// empty area is all FFs
		 *ptr = *ptr + FLASH_FILE_SIZE;
	 }
	 return  E_NO_ROOM_IN_FLASH;
 }
 
	 
 static char  *CreateNewFile(char  *name)
 {
	 char						*ptr;
	 int						i;
	 int						s;
	 
	 ptr = (char *)flashbuffer;
	 s = FindNextOpenFile(&ptr);
	 if (s != S_OK)  return  NULL;
	
	 i = 0;
	 while (i<MAX_LEN_FILENAME)
	 {
		 ptr[i] = name[i];
		 if (name[i] == '\0')  break;
		 i++;
	 }
	 if (i == MAX_LEN_FILENAME)
	 {
		 for (i=0; i<MAX_LEN_FILENAME; i++)		// didn't work, need to erase header
		 {
			 ptr[i] = 0xFF;						// empty area is all FFs
		 }
		 return  NULL;
	 }
	 return  ptr;
 }
 


static int			CopyFlashToBuffer(void)
{
	int					n;
	unsigned char		*srcptr;
	unsigned char		*dstptr;
	
	srcptr = (unsigned char *)FLASH_SECTOR_STORAGE_ADDR;
	dstptr = flashbuffer;
	
	for (n=0; n<FLASH_SECTOR_STORAGE_SIZE; n++)
	{
		*dstptr++ = *srcptr++;
	}
	return  0;
}



static void			AskToSaveFileChanges(void)
{
	int				c;
	
	if (filebufferchanged)
	{
		t_SetOutputStream(T_STREAM_SERIAL);
		printstr("\n\nWARNING!  You have made program changes.  Do you want to save");
		printstr("              your changes to flash (y/n)? ");
		while (1)
		{
			c = t_GetChar();
			t_OutChar(c);
			if (c == 'y' || c == 'Y')
			{
				t_ForceWriteToFlash();
				break;
			}
			else if (c == 'n' || c == 'N')
			{
				break;
			}
		}
	}
}

	


/*
 * UART_init      init UART1 as console serial port
 * 
 * This routine is hard-coded for UART1 (Tx on DIP-9, Rx on DIP-10).
 */
uint32_t  UART_Init(uint32_t  baudrate)
{
	uint32_t						fdiv;

	if (baudrate == 0)  return FALSE;		// that would be bad
	fdiv = ((SystemCoreClock/4) / (16 * baudrate));	//baud rate
	
#if CONSOLE_UART==UART1
	LPC_SC->PCONP |= (1<<4);				// power-up UART1
	LPC_SC->PCLKSEL0 &= ~(3UL << 8);		// 00 in bits 9.8 sets UART1's PCLK=CCLK/4
	LPC_PINCON->PINSEL0 |= (1<<30);			// enable TxD1 (P0.15, DIP-13)
	LPC_PINCON->PINSEL1 |= (1<<0);			// enable RxD1 (P0.16, DIP-14)
   	LPC_UART1->LCR = 0x80;          		// enable access to baudrate divisors
	LPC_UART1->DLM = fdiv / 256;                                                    
	LPC_UART1->DLL = fdiv % 256;
	LPC_UART1->LCR = 0x03;          		// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
#endif
#if CONSOLE_UART==UART3
	LPC_SC->PCONP |= (1<<25);				// power-up UART3
	LPC_SC->PCLKSEL1 &= ~(3UL << 18);		// 00 in bits 19.18 sets UART3's PCLK=CCLK/4
	LPC_PINCON->PINSEL0 |= ((2<<0) | (2<<2));  // enable RxD3 P0.1 (DIP-10), TxD3 P0.0 (DIP-9)
   	LPC_UART3->LCR = 0x80;          		// enable access to baudrate divisors
	LPC_UART3->DLM = fdiv / 256;                                                    
	LPC_UART3->DLL = fdiv % 256;
	LPC_UART3->LCR = 0x03;          		// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
#endif

	return  TRUE;
}






/*
 *  UART_Putch     output (with blocking) a char to console UART
 */
void  UART_Putch(int  c)
{
#if CONSOLE_UART==UART1
	while ((LPC_UART1->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
	LPC_UART1->THR = (c & (uint16_t)0xff);
#endif
#if CONSOLE_UART==UART3
	while ((LPC_UART3->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
	LPC_UART3->THR = (c & (uint16_t)0xff);
#endif
}



/*
 *  UART_Getch      get (with blocking) a char from console UART
 */
int  UART_Getch(void)
{
	int					c;

	while (!UART_Avail())  ;					// block until char arrives

#if CONSOLE_UART==UART1
	c = (int)(LPC_UART1->RBR & (uint16_t)0xff);	// get char from holding buffer
#endif
#if CONSOLE_UART==UART3
	c = (int)(LPC_UART3->RBR & (uint16_t)0xff);	// get char from holding buffer
#endif

	return  c;
}


/*
 *  UART_Avail      returns number of chars waiting in console UART's holding area
 */
int  UART_Avail(void)
{
	int						val;

#if CONSOLE_UART==UART1
	if (LPC_UART1->LSR & LSR_RDR)  val = 1;
	else						  val = 0;
#endif
#if CONSOLE_UART==UART3
	if (LPC_UART3->LSR & LSR_RDR)  val = 1;
	else						  val = 0;
#endif
	return  val;
}



/*
 * ADC_Init      initialize the ADC subsystem
 * 
 * This code sets up the first four ADC channels for 12 MHz clock and powers up the ADC.
 * 
 * AD0.0 is on DIP pin 15.
 * AD0.1 is on DIP pin 16.
 * AD0.2 is on DIP pin 17.
 * AD0.3 is on DIP pin 18.
 */
static void			ADC_Init(void)
{
	LPC_SC->PCONP |= (1<<12);				// power-up ADC (set PCADC bit) 
	LPC_ADC->ADCR |= (1<<21);				// enable ADC (set PDN bit)
											// On power-up, LPC_SC->PCLK_ADC is CCLK/4; don't change PCLK_ADC.
	LPC_ADC->ADCR |= (7 << 8);				// set CLKDIV to (7+1); ADC clock is 96 MHz / 8 = 12 MHz
	LPC_PINCON->PINSEL1 |= (1 << 14);		// select AD0.0 on P0.23 (DIP 15)
	LPC_PINCON->PINSEL1 |= (1 << 16);		// select AD0.1 on P0.24 (DIP 16)
	LPC_PINCON->PINSEL1 |= (1 << 18);		// select AD0.2 on P0.25 (DIP 17)
	LPC_PINCON->PINSEL1 |= (1 << 20);		// select AD0.3 on P0.26 (DIP 18)
	LPC_ADC->ADCR |= (0x0f << 0);			// write '1111' to SEL field to select ADC chnls 0-3
	LPC_ADC->ADCR |= (1 << 16);				// write 1 to BURST mode to turn on continuous conversions
}
	


/*
 * t_ColdBoot      perform coldboot operations; usually called once, following power-on or reset
 */ 
void			t_ColdBoot(void)
{
	init();				// system init, including PLA and system timer

	SysTick_Config(SystemCoreClock / 1000);		// setup SysTick Timer for 1 msec interrupts
	__enable_irq();
	

	CopyFlashToBuffer();						// load up the working buffer from flash
	filebufferchanged = FALSE;					// show no changes yet
	
	UART_Init(9600);							// init console UART, set baud rate
	ADC_Init();									// init ADC system, set up ADC channels 0-3
	
	LPC_GPIO1->FIODIR = (LED1_MASK | LED2_MASK | LED3_MASK | LED4_MASK);	// all LED GPIOs are outputs
}


/*
 * t_WarmBoot      perform warmboot operations; can be called after program error or crash
 */
void			t_WarmBoot(void)
{
	OutputStream = T_STREAM_SERIAL;			// set output stream to serial port
	InputStream = T_STREAM_SERIAL;			// set input stream to serial port
}


/*
 * t_Shutdown      performs target-specific actions to shut down TBL
 * 
 * Note that shutting down is not the same as rebooting!  This function only cleans
 * up any hardware registers that might need updating.  The t_ForceHWReset function
 * below handles saving of any user changes to the RAM buffer to flash before HW
 * reset.
 */
void			t_Shutdown(void)
{
	printstr("\n\nShutting down... ");
}


/*
 * t_OutChar      writes (blocks) char to active stream, based on OutputStream.
 */
void			t_OutChar(int  c)
{
	switch  (OutputStream)
	{
		case  T_STREAM_SERIAL:
		UART_Putch(c);
		break;
		
		case  T_STREAM_FILE:
		if (wfileptr)
		{
			if (wfileptr < wfiledataend)
			{
				*wfileptr++ = c;
				filebufferchanged = TRUE;
			}
		}
	}
}


/*
 * t_GetChar      read (blocks) a char from active stream, based on InputStream.
 */
int				t_GetChar(void)
{
	int			c;
	
	switch  (InputStream)
	{
		case  T_STREAM_SERIAL:
		return  UART_Getch();

		case  T_STREAM_FILE:
		if (rfileptr)
		{
			if (rfileptr < rfiledataend)
			{
				c = *rfileptr++;
				if (c == 0xff)	c = EOF;		// empty byte from flash (0xff) means EOF
				return  c;
			}
		}
		return  0;
		
		default:
		return  0;
	}
}



/*
 * t_ConsoleCharAvailable      returns TRUE if char is available from console stream,
 * ignoring InputStream.
 */
int				t_ConsoleCharAvailable(void)
{
	return  UART_Avail();
}


/*
 * t_GetCharFromConole      read (blocks) a char from console stream, ignoring InputStream.
 */
int				t_GetCharFromConsole(void)
{
	return  UART_Getch();
}


/*
 * t_SetInputStream      change character input stream
 */
int				t_SetInputStream(int  s)
{
	if (s != T_STREAM_SERIAL && s != T_STREAM_FILE)  return  -1;
	
	InputStream = s;
	return  0;
}


/*
 * t_SetOutputStream      change character output stream
 */
int				t_SetOutputStream(int  s)
{
	if (s != T_STREAM_SERIAL && s != T_STREAM_FILE)  return  -1;
	
	OutputStream = s;
	return  0;
}



/*
 * t_FileExistsQ      check if named file exists in file buffer
 * 
 * This routine checks the file buffer (NOT flash!) to see if the named
 * file already exists.  Returns TRUE if file exists, else FALSE.
 */
int				t_FileExistsQ(char  *name)
{
	int					r;
	char				*ptr;			// unused, but needed for function call
	
	r = FindFileByName(name, &ptr);
	if (r == S_OK)  return  TRUE;
	else 			return  FALSE;
}



FILE			*t_OpenFile(char  *name, char  *mode)
{
	int				r;
	char			*fptr;
	
	r = FindFileByName(name, &fptr);

	if (*mode == 'r' || *mode == 'R')
	{
		if (r != S_OK)  return  NULL;
		rfilestart = fptr;
		rfiledatastart = rfilestart + SIZEOF_FLASH_FILE_HEADER;
		rfiledataend = rfilestart + FLASH_FILE_SIZE;
		rfileptr = rfiledatastart;
		rfp = (FILE *)1;
		return  rfp;
	}
	else if (*mode == 'w' || *mode == 'W')
	{
		if (r == S_OK)						// if file already exists
		{
			t_DeleteFile(name);				// delete the file
		}
		fptr = CreateNewFile(name);					// try to create the file
		if (fptr == NULL)  return  NULL;		// oops, could not write header
		wfilestart = fptr;
		wfiledatastart = wfilestart + SIZEOF_FLASH_FILE_HEADER;
		wfiledataend = wfilestart + FLASH_FILE_SIZE;
		wfileptr = wfiledatastart;
		wfp = (FILE *)1;
		return  wfp;
	}
	return  NULL;							// nope, didn't work
}


int				t_CloseFile(FILE  *fp)
{
	return  0;
}


int				t_DeleteFile(char  *name)
{
	int				r;
	char			*fptr;
	int				i;
	
	r = FindFileByName(name, &fptr);
	if (r == S_OK)					// if found the file...
	{
//		printstr("  t_DeleteFile:deleting  ");
		for (i=0; i<FLASH_FILE_SIZE; i++)
		{
			*(fptr + i) = 0xff;		// erase a file by writing 0xff to all RAM buffer bytes
		}
		filebufferchanged = TRUE;	// let Shutdown know something happened
	}
	return  r;
}


	

char      *t_GetFirstFileName(void)
{
	int				i;
	char			*ptr;
	
	for (i=0; i<NUM_FLASH_FILES; i++)
	{
		ptr = &flashbuffer[i*FLASH_FILE_SIZE];
		if (*ptr != 0xff)
		{
			return  ptr;
		}
	}
	return  NULL;
}
		

char		*t_GetNextFileName(char  *ptr)
{
	if (!ptr)  return NULL;					// stupid check

	while (1)
	{
		ptr = ptr + FLASH_FILE_SIZE;
		if (ptr >= (char *)(flashbuffer + FLASH_SECTOR_STORAGE_SIZE))
			break;
		if (*ptr != 0xff)
		{
			return  ptr;
		}
	}
	return  NULL;
}



int				t_ForceWriteToFlash(void)
{
	printstr("\nErasing flash...");
	Flash_EraseSector();
	printstr("  saving changes...");
	Flash_WriteSector();
	printstr(" done.\n");
	filebufferchanged = FALSE;
	return  0;
}



/*
 * t_SetTimerRate      select timer tic rate in usecs
 */
void			t_SetTimerRate(unsigned int  usecs)
{
	SysTick_Config((SystemCoreClock / 1000000) * usecs);
}



/*
 * t_AddTimer      add address to timer list
 * 
 * If an open address is available in the timer table, record the address
 * passed as argument t for later use as a down-counting timer.
 * 
 * If the address in argument t already exists in the timer table, pretend
 * the address was added but don't change the table.
 */
int				t_AddTimer(DATA_SIZE  *t)
{
	int				n;
	int				open;
	
	open = T_NUM_TIMERS+1;
	for (n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n] == 0)
		{
			open = n;
		}
		else
		{
			if (t == timeraddrs[n])		// if same address in already in table...
				return  S_OK;			// pretend we added it
		}
	}
	if (open < T_NUM_TIMERS+1)
	{
		timeraddrs[open] = t;			// record addr
		return  S_OK;					// show it worked
	}
	return  E_COULD_NOT_ADD_TIMER;	// could not add timer
}



/*
 * t_DeleteTimer      remove address from timer list
 */
int				t_DeleteTimer(DATA_SIZE  *t)
{
	int				n;
	
	for (n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n] == t)
		{
			timeraddrs[n] = 0;
			return  0;
		}
	}
	return  -1;							// could not remove timer, not in list
}




/*
 * t_ReadPort
 */
DATA_SIZE		t_ReadPort(unsigned int  index)
{
	DATA_SIZE		v;
	
	if (index < NUM_VIRTUAL_PORTS)			// if this is a virtual port...
	{
		switch  (index)
		{
			case  LED1:
			v = LPC_GPIO1->FIOPIN & LED1_MASK;
			return  (v != 0);
			
			case  LED2:
			v = LPC_GPIO1->FIOPIN & LED2_MASK;
			return  (v != 0);
			
			case  LED3:
			v = LPC_GPIO1->FIOPIN & LED3_MASK;
			return  (v != 0);
			
			case  LED4:
			v = LPC_GPIO1->FIOPIN & LED4_MASK;
			return  (v != 0);
			
			case  ADC0:
			v = LPC_ADC->ADDR0;
			v = (v >> 4) & 0xfff;
			return  v;
			
			case  ADC1:
			v = LPC_ADC->ADDR1;
			v = (v >> 4) & 0xfff;
			return  v;

			case  ADC2:
			v = LPC_ADC->ADDR2;
			v = (v >> 4) & 0xfff;
			return  v;

			case  ADC3:
			v = LPC_ADC->ADDR3;
			v = (v >> 4) & 0xfff;
			return  v;
			
			default:
			return  0;
		}
	}
	else 					// if this is a real port...
	{
	}
	
	return  0;				// just a stub, replace with real code later
}


/*
 * t_WritePort
 */
void			t_WritePort(unsigned int  index, DATA_SIZE  value)
{
	if (index >= NUM_PORTS) 	return;
	if (porttable[index].writeable == READONLY)  return;		// cannot change this port, ignore

	switch  (index)
	{
		case  LED1:
		if (value)  LPC_GPIO1->FIOSET = LED1_MASK;
		else 		LPC_GPIO1->FIOCLR = LED1_MASK;
		break;
		
		case  LED2:
		if (value)  LPC_GPIO1->FIOSET = LED2_MASK;
		else 		LPC_GPIO1->FIOCLR = LED2_MASK;
		break;

		case  LED3:
		if (value)  LPC_GPIO1->FIOSET = LED3_MASK;
		else 		LPC_GPIO1->FIOCLR = LED3_MASK;
		break;

		case  LED4:
		if (value)  LPC_GPIO1->FIOSET = LED4_MASK;
		else 		LPC_GPIO1->FIOCLR = LED4_MASK;
		break;
	}
}



/*
 * t_CheckForHWBreak      check USER pushbutton; if pressed, return TRUE
 */
int				t_CheckForHWBreak(void)
{
	return  FALSE;			// just a stub, replace with real code later
}



/*
 * t_CheckToneStatus
 */
int				t_CheckToneStatus(void)
{
	return  FALSE;			// just a stub, replace with real code later
}


/*
 * t_SetTone
 */
void			t_SetTone(DATA_SIZE  freq, DATA_SIZE  duration)
{
							// just a stub, replace with real code later
}



/*
 * t_ForceHWReset
 */
void			t_ForceHWReset(void)
{
	AskToSaveFileChanges();

	printstr("\n\nForcing hardware reset...  ");
	
	LPC_WDT->WDCLKSEL = 0x01;		// WDT clock is APB perpheral clock (watchdog pclk)
	LPC_WDT->WDMOD = 0x03;			// WDEN = 1, WDRESET = 1; force reset on watchdog timeout
	LPC_WDT->WDFEED = 0xaa;			// first part of feed sequence
	LPC_WDT->WDFEED = 0x55;			// second (final) part of feed sequence; this starts the WDT
	while  (1)  ;					// time to die...
}



/*
 * t_Test				generic test function available as TBL keyword TEST
 */
void			t_Test(void)
{
	uint8_t				*ptr;
	
	printstr("\nContents of flash--\n");
	ptr = FLASH_SECTOR_STORAGE_ADDR;
	
	for (int n=0; n<FLASH_SECTOR_STORAGE_SIZE/4096; n++)
	{
		for (int j=0; j<16; j++)
		{
			printnum(ptr[j], RADIX_HEX);
			printstr(" ");
		}
		printstr("\n");
		ptr = ptr + 4096;
	}
}


