/*
 * target_LPC1768.h				header file specific to the LPC1768 (mbed) target compiler
 */

#ifndef  TARGET_LPC1768_H
#define  TARGET_LPC1768_H


/*
 *              			In case your MBED device becomes read-only!!!
 * 
 * Found this after an exhaustive web search.  These instructions are for Linux only; if you
 * are using Windows, you will need to find/use the equivalent tools (and no, I don't know
 * what those are).
 * 
 * Source: https://os.mbed.com/questions/278/mbed-mounts-on-linux-as-a-read-only-file/
 * 
 * Linux User:
 *
 * First, we need to delete the old partition(s) that remain on the USB key.
 *
 * Open a terminal and type the following commands:
 *
 *    sudo blkid -L MBED         # note your MBED device (such as /dev/sdd).
 *    sudo fdisk /dev/sdx        # (replacing x with your drive letter)
 *
 * This starts the Linux fdisk utility; you will see the fdisk command prompt.
 * 
 *   Type p to print the current partition table
 *   Type d to proceed to delete a partition
 *   Type 1 to select the 1st partition and press enter (not a known command in 2.31.1 fdisk; ignore)
 *   Repeat to delete all partitions as required.
 *   Type w to write changes to the MBED and quit fdisk.
 * 
 * Then format by entering: 
 *
 *   mkfs.vfat -F 12 -I -nMBED /dev/sdx   #(replacing x with your USB key drive letter)
 *
 * Power-cycle your MBED.  It should reconnect to the system with an empty MBED hard drive that
 * is read/write enabled; the MBED drive will contain only MBED.html.
 */ 

#include  <stdio.h>
#include  <stdint.h>
#include  <limits.h>


/*
 * Define masks used for accessing fields within the UART registers.
 */
#define IER_RBR         0x01
#define IER_THRE        0x02
#define IER_RLS         0x04

#define IIR_PEND        0x01
#define IIR_RLS         0x03
#define IIR_RDA         0x02
#define IIR_CTI         0x06
#define IIR_THRE        0x01

#define LSR_RDR         0x01
#define LSR_OE          0x02
#define LSR_PE          0x04
#define LSR_FE          0x08
#define LSR_BI          0x10
#define LSR_THRE        0x20
#define LSR_TEMT        0x40
#define LSR_RXFE        0x80


/*
 * Define configuration of flash storage area.  This affects program size and
 * number of files that can be stored.
 */
//#define  FLASH_SECTOR_STORAGE_NUM	24				/* flash sector used for storage */
//#define  FLASH_SECTOR_STORAGE_ADDR	0x00500000	/* flash storage area starts at sector 24 */

#define  FLASH_SECTOR_STORAGE_NUM	16				/* flash sector used for storage */
#define  FLASH_SECTOR_STORAGE_ADDR	0x00010000		/* flash storage area starts at sector 16 */

#define  FLASH_SECTOR_STORAGE_SIZE	0x4000			/* only using 16KB of the 32KB flash sector! */

#define  FLASH_FILE_SIZE			4096			/* size of a program file to store in flash */
#define  NUM_FLASH_FILES			(FLASH_SECTOR_STORAGE_SIZE/FLASH_FILE_SIZE)
#define  SIZEOF_FLASH_FILE_HEADER	16				/* header is null-terminated file name */

#define  MAX_LEN_FILENAME			(SIZEOF_FLASH_FILE_HEADER - 1)




/*
 * Define the amount of RAM available for the user's program.  This is normally the
 * size of a single flash file less any header space.
 */
#define  T_PROGSPACE_SIZE			(FLASH_FILE_SIZE-MAX_LEN_FILENAME-1)


/*
 * Define the size (in bits) of various data elements for the STM32F4 target.
 * These definitions apply to the size of stack elements and variables, among
 * other things.
 */
#define		DATA_SIZE			int32_t
#define		DATA_SIZE_MAX		INT32_MAX
#define		UDATA_SIZE			uint32_t
#define		UDATA_SIZE_MAX		UINT32_MAX

/*
 * Define target characteristics and resources of interest to the core program.
 */
 
/*
 * Define available timers
 * 
 * T_NUM_TIMERS     n					number of down-counting timers target can support
 */
#define		T_NUM_TIMERS		4


/*
 * Define available tone channels
 * 
 * T_NUM_TONE_CHS	n					number of TONE channels
 */
#define		T_NUM_TONE_CHS      0



/*
 * Define available character I/O streams and file support
 */
#define		T_STREAM_SERIAL		0
#define		T_STREAM_FILE		1
#define		T_SUPPORTS_FILES	TRUE		/* includes file support (yay!) */





extern const unsigned char				ports_tab[];



void			t_ColdBoot(void);				// system prep following reset
void			t_WarmBoot(void);				// system prep following crash
void			t_Shutdown(void);				// clean system shutdown at end of program

void			t_OutChar(int  c);				// output a char to active stream
int				t_GetChar(void);				// waits (blocks) for char from active stream, returns char
int				t_ConsoleCharAvailable(void);	// returns TRUE if user has entered a char at console; does not lock
int				t_GetCharFromConsole(void);		// always waits (blocks) for char from console, ignoring InputStream

void			t_SetTimerRate(unsigned int  usecs);	// sets down-counting timer tic rate in usecs; 0 disables timer
int				t_AddTimer(DATA_SIZE  *t);		// adds a variable at address t to the timer list
int				t_DeleteTimer(DATA_SIZE  *t);	// removes a variable at address to from the timer list

void			t_SetTone(DATA_SIZE  freq, DATA_SIZE  duration);	// starts tone at freq Hz for duration msecs
int				t_CheckToneStatus(void);		// return TRUE if tone still active, else FALSE

int				t_SetOutputStream(int  s);		// assigns the output device for outchar() data from core
int				t_SetInputStream(int  s);		// assigns the input device for inchar() data to core

int				t_FileExistsQ(char  *name);		// returns TRUE if named file exists in buffer
FILE			*t_OpenFile(char  *name, char  *mode);	// open a file for text I/O; closes any open files first
int				t_CloseFile(FILE  *fp);			// close file if open
char			*t_GetFirstFileName(void);		// starts list of file names
char			*t_GetNextFileName(char *ptr);		// gets next file name from list
int				t_DeleteFile(char  *name);

int				t_ForceWriteToFlash(void);		// forces writing of RAM disc to flash

DATA_SIZE		t_ReadPort(unsigned int  index);			// reads a port
void			t_WritePort(unsigned int  index, DATA_SIZE  value);		// writes value to indexed port

int				t_CheckForHWBreak(void);		// returns TRUE if USER pushbutton is pressed

void			t_ForceHWReset(void);			// forces a hardware reset using watchdog timeout

void			t_Test(void);					// triggers a target-specific test routine from TBL								




#endif				// TARGET_LPC1768_H
