
/*
 *  gd_test.c      test platform for playing with the Gameduino
 *
 *  This program allows the mbed development board to control a Gameduino
 *  via SPI.
 *
 *  The code here initializes the Gameduino, then displays ASCII text on
 *  the GD's VGA screen.  I developed this primarily as a way to play with
 *  text on the GD, since the rest of the world is using the GD for graphics.  :-)
 */

#define  OWNER

#include  <stdio.h>
#include  <string.h>
#include  <errno.h>

#include  "LPC17xx.h"
#include  "gdsupport.h"
#include  "GD.h"
#include  "ball.h"


/*
 *  I'm going to be the owner of errno (the global error number variable),
 *  so I need to undefine it here, then redefine it as an int.  All other
 *  modules that link with me must do the same, declaring errno as extern.
 */
#undef  errno
int							errno;


/*
 *  Rather than put target-specific SPI code inside this module, I use
 *  a set of three routines that are target-specific, then rely on the
 *  gdsupport module to do all of the non-specific SPI operations.
 *
 *  This works through the following three callback functions.  test_select()
 *  is called by the gdsupport code to select the Gameduino.  test_xchg()
 *  is called to exchange a byte of data with the GD.  test_deslect()
 *  is called to deselect the GD.
 */
void						test_select(void);
char						test_xchg(char);
void						test_deselect(void);

/*
 *  outstr() is a wrapper function that lets me display a null-terminated
 *  string on the GD's VGA screen using the Gameduino outch() function.
 */
void						outstr(char  *s);


/*
 *  Since I'm using sprintf() in this test, I need (as a minimum) to provide
 *  a version of _sbrk().
 */
caddr_t						_sbrk(int incr);



unsigned int				SystemCoreClock;				// global variable, core clock frequency

/*
 *  Use P0.16 as chip-select for the GameDuino.
 */
#define  GD_CS_BIT			16
#define  GD_CS_MASK			(1<<GD_CS_BIT)

#define  GD_ENABLE			(LPC_GPIO0->FIOCLR = GD_CS_MASK)
#define  GD_DISABLE			(LPC_GPIO0->FIOSET = GD_CS_MASK)


/*
 *  The following strings are stored and initialized two different ways.  HelloConst
 *  is read direcly from read-only data (.rodata) while HelloRam is copied from
 *  flash to RAM and read from RAM (.data).
 *
 *  These strings are here to confirm proper operation of the linker script that
 *  builds this program.
 */
const char					HelloConst[] = " Const Hello, world!                    ";
char						HelloRam[] =   " RAM Hello, world!                      ";



void						setup(void);
void						loop(void);




/*
 *  init      perform low-level initialization of the mbed board
 *
 *  main() must call this routine before doing any other major tasks on the mbed!
 */
void init(void)
{
/*
 *  Clock initialization for the Mbed development board.  This board
 *  uses a 12 MHz oscillator as the main clock source.  This code
 *  sets the core clock to 96 MHz.
 */
	LPC_SC->SCS = 0x20;					// enable external oscillator (12 MHz crystal)
	while (!(LPC_SC->SCS & (1<<6))) ;	// wait for main oscillator to stablilize
	
	LPC_SC->CLKSRCSEL = 0x01;			// set main oscillator (12 MHz crystal) as PLL source
	LPC_SC->CCLKCFG = 0x03;				// set CPU clock (CCLK) to PLL0 output / 4
	
	LPC_SC->PCLKSEL0 = 0x0;				// peripherals use CPU clock / 4
	LPC_SC->PCLKSEL1 = 0x0;				// peripherals use CPU clock / 4

/*
 *  The following code sets up the Mbed's core clock to 96 MHz.
 *  This code sets PLL0 output to 384 MHz.  The CPU clock
 *  configuration above (CCLKCFG) divides this value by 4
 *  to get a final CPU clock of 96 MHz.
 */
	LPC_SC->PLL0CFG = (0<<16) | (15<<0);	// PPL0 config, M=16, N=1, Fout=384
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	LPC_SC->PLL0CON = 0x01;				// enable PLL0
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	while (!(LPC_SC->PLL0STAT & (1<<26)))	;		// wait for PLL0 lock

	LPC_SC->PLL0CON = 0x03;				// enable and connect PLL0
	LPC_SC->PLL0FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL0FEED = 0x55;

	while (!(LPC_SC->PLL0STAT & ((1<<25) | (1<<24))))  ;	// wait for PLL0 enable and connect

	SystemCoreClock = 96000000;			// need to make this value available globally
	
/*
 *  Set up PLL1 for use as a USB clock.  The output must be set to 48 MHz, based
 *  on an oscillator of 12 MHz.
 */
	LPC_SC->PLL1CFG = 0x00000023;		// PLL1 config, M=4 (bits 4:0=3), P=2 (bits 6:5=1)
	LPC_SC->PLL1FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL1FEED = 0x55;

	LPC_SC->PLL1CON = 0x01;				// enable PLL1
	LPC_SC->PLL1FEED = 0xAA;
	LPC_SC->PLL1FEED = 0x55;

	while (!(LPC_SC->PLL1STAT & (1<<10)))  ;		// wait for PLL1 lock

	LPC_SC->PLL1CON = 0x03;				// enable and connect PLL1
	LPC_SC->PLL1FEED = 0xAA;			// feed the PLL
	LPC_SC->PLL1FEED = 0x55;

	while (!(LPC_SC->PLL1STAT & ((1<<9) | (1<<8))))  ;  // wait for PLL1 enable and connect
}



int  main(void)
{
	int						n;
	char					buff[80];

	init();										// do low-level setup first!
	
/*
 *  Configure the SPI port for connection to the Gameduino.
 */
	LPC_SC->PCONP |= (1<<8);					// apply power to the SPI interface
	LPC_SC->PCLKSEL0 |= (1<<16);				// use CCLK as SPI clock
	LPC_SPI->SPCCR = 24;						// now divide SPI clock by this much (must be >= 8, must be even!)

	LPC_PINCON->PINSEL0 |= (3<<30);				// set P0.15 as SCK (collides with UART1 TXD1)
	LPC_PINCON->PINSEL1 |= (3<<2);				// set P0.17 as MISO (collides with UART1 CTS1)
	LPC_PINCON->PINSEL1 |= (3<<4);				// set P0.18 as MOSI (collides with UART1 DCD1)

	LPC_PINCON->PINMODE1 |= (2<<2);				// set MISO with no pull-up or pull-down

	LPC_SPI->SPCR = (1<<5);						// set SPI to master mode

	LPC_GPIO0->FIODIR |= GD_CS_MASK;			// use SSEL as GPIO slave select line; make it an output
	GD_DISABLE;									// pull the chip-select line high

/*
 *  The SPI is set up.  Now we need to register our callback functions
 *  with the Gameduino support module.
 */
	GD_register(&test_select, &test_xchg, &test_deselect);

	LPC_GPIO1->FIODIR = 0xffffffff;

	for (n=0; n<5000000; n++)  ;				// startup delay for Gameduino

/*
 *  All done, strut our stuff!
 */
 	GD_begin();									// set up the GD hardware
	GD_ascii();									// set up for using text
	GD_cls();

	for (n=0; n<10; n++)						// use the GD_outch function...
	{
//		outstr("0123 456 789 0123 456 789 0123 456 789\n\r");
		sprintf(buff, "%3d: Hello, world!\n\r", n);
		outstr(buff);
	}

/*
 *  Use the direct-write functions in the GD library.
 */
	GD_putstr(0, 19, HelloConst);
	GD_putstr(0, 20, HelloRam);
	GD_putstr(0, 21, "21 String constant within GD_putstr() Hello, world!");
	GD_putstr(0, 22, "012345678901234567890123456789012345678901234567890");
	GD_putstr(0, 23, "abcdefghijklmnopqrstuvwxyz       ");


/*
 *  Use some local functions that in turn use the GD_outch library function.
 */
	outstr("Row 0\n\r");
	outstr("Row 1\n\r");
	outstr("Row 2\n\r");
	outstr("Row 3\n\r");
	outstr("Row 4\n\r");
	outstr("Row 5: abcdefghijklmnopqrstuvwxyz\n\r");

	outstr("\n\r");
	outstr("\n\rHere comes the bouncing ball demo...  ");

	for (n=0; n<30000000; n++)  ;

	setup();
	loop();

	return  0;					// should never get here!
}


void  setup(void)
{
  GD_begin();

  // Background image
  GD_copy(RAM_PIC, bg_pic, sizeof(bg_pic));
  GD_copy(RAM_CHR, bg_chr, sizeof(bg_chr));
  GD_copy(RAM_PAL, bg_pal, sizeof(bg_pal));
  LPC_GPIO1->FIOSET |= (1<<18);					// turn on LED1

  // Sprite graphics
  GD_uncompress(RAM_SPRIMG, ball);
  LPC_GPIO1->FIOSET |= (1<<20);					// turn on LED2

  // Palettes 0 and 1 are for the ball itself,
  // and palette 2 is the shadow.  Set it to
  // all gray.
  int i;
  for (i = 0; i < 256; i++)
    GD_wr16(RAM_SPRPAL + (2 * (512 + i)), RGB(64, 64, 64));

  // Set color 255 to transparent in all three palettes
  GD_wr16(RAM_SPRPAL + 2 * 0xff,  TRANSPARENT);
  GD_wr16(RAM_SPRPAL + 2 * 0x1ff, TRANSPARENT);
  GD_wr16(RAM_SPRPAL + 2 * 0x2ff, TRANSPARENT);
}

#define RADIUS (112 / 2) // radius of the ball, in pixels

#define YBASE (300 - RADIUS)

void loop()
{
  int x = 200, y = RADIUS;              // ball position
  int xv = 2, yv = 0;                   // ball velocity

  int r;                                // frame counter
  for (r = 0; ; r++) {
    GD__wstartspr((r & 1) ? 256 : 0);  // write sprites to other frame
    draw_ball(x + 15, y + 15, 2);       // draw shadow using palette 2
    draw_ball(x, y, r & 1);             // draw ball using palette 0 or 1
    GD__end();

    // paint the new palette
    uint16_t palette = RAM_SPRPAL + 512 * (r & 1);
    unsigned char li;
    for (li = 0; li < 7; li++) {
      unsigned char liv = 0x90 + 0x10 * li;      // brightness goes 0x90, 0xa0, etc
      int red = RGB(liv, 0, 0);
      int white = RGB(liv, liv, liv);
      unsigned char i;
      for (i = 0; i < 32; i++) {        // palette cycling using 'r'
        GD_wr16(palette, ((i + r) & 16) ? red : white);
        palette += 2;
      }
    }

    // bounce the ball around
    x += xv;
    if ((x < RADIUS) || (x > (400 - RADIUS)))
      xv = -xv;
    y += yv;
    if ((yv > 0) && (y > YBASE)) {
      y = YBASE - (y - YBASE);          // reflect in YBASE
      yv = -yv;                         // reverse Y velocity
    }
    if (0 == (r & 3))
      yv++;                             // gravity

    // swap frames
    GD_waitvblank();
    GD_wr(SPR_PAGE, (r & 1));
	if (r&1)  LPC_GPIO1->FIOSET |= (1<<21);		// use LED3 to show frame-rate
	else      LPC_GPIO1->FIOCLR |= (1<<21);
  }
}




/*
 *  test_select      callback function; selects GD adapter
 */
void  test_select(void)
{
	GD_ENABLE;
}


/*
 *  test_xchg      callback function; exchanges a byte with GD adapter
 */
char  test_xchg(char  val)
{
	char					c;

	LPC_SPI->SPDR = val;								// send a byte
	while ((LPC_SPI->SPSR & (1<<7)) == 0)  ;			// loop until data reg is empty
	c = LPC_SPI->SPSR;									// NXP says read this one more time!
	return  LPC_SPI->SPDR;								// return the result
}


/*
 *  test_deselect      callback function; deselects GD adapter
 */
void  test_deselect(void)
{
	GD_DISABLE;
}



/*
 *  outstr      quick-and-dirty string print function
 */
void  outstr(char  *s)
{
	while (*s)
	{
		GD_outch(*s);
		s++;
	}
}


/*
 sbrk
 Increase program data space.
 Malloc and related functions depend on this
 */
caddr_t _sbrk(int incr)
{

    extern char 					_end_bss; 		// Defined by the linker
    static char						*heap_end;
    char							*prev_heap_end;
	char 							*stack;

    if (heap_end == 0)
	{
        heap_end = &_end_bss;
    }
    prev_heap_end = heap_end;

	stack = (char*) __get_MSP();
	if (heap_end + incr >  stack)
    {
         errno = ENOMEM;
         return  (caddr_t) -1;
         //abort ();
	}

    heap_end += incr;
    return (caddr_t) prev_heap_end;
}


