/*
 *  uart.c      routines for low-level UART initialization
 *
 *  This file is intended to be compiled as a library; refer to the
 *  associated makefile.
 */

#include  "LPC17xx.h"
#include  <stdio.h>
#include  <string.h>
#include  <stdint.h>
#include  "uart.h"


#ifndef  FALSE
#define  FALSE  0
#define  TRUE  !FALSE
#endif


#define  MAX_RCV_Q_CHARS	64


extern  uint32_t			SystemCoreClock;



char						UART0RcvQ[MAX_RCV_Q_CHARS];
uint8_t						UART0RcvOutIndex;
uint8_t						UART0RcvInIndex;

char						UART1RcvQ[MAX_RCV_Q_CHARS];
uint8_t						UART1RcvOutIndex;
uint8_t						UART1RcvInIndex;

char						UART2RcvQ[MAX_RCV_Q_CHARS];
uint8_t						UART2RcvOutIndex;
uint8_t						UART2RcvInIndex;

char						UART3RcvQ[MAX_RCV_Q_CHARS];
uint8_t						UART3RcvOutIndex;
uint8_t						UART3RcvInIndex;


static char					UART0CallbackList[MAX_LEN_CALLBACK_CHAR_LIST];
static char					UART1CallbackList[MAX_LEN_CALLBACK_CHAR_LIST];
static char					UART2CallbackList[MAX_LEN_CALLBACK_CHAR_LIST];
static char					UART3CallbackList[MAX_LEN_CALLBACK_CHAR_LIST];

static void					(*alert0)(char  val) = (void *)0;
static void					(*alert1)(char  val) = (void *)0;
static void					(*alert2)(char  val) = (void *)0;
static void					(*alert3)(char  val) = (void *)0;


/*
 *  The UARTInit routine calcs the divisor needed for the requested baud rate.
 *
 *  Note that the initialization routine always sets the UARTx peripheral
 *  clock to CCLK / 4.
 *
 *  Div = PCLK / (16 * baudrate), assuming no fractional divide is necessary.
 *      = CCLK / 4 / 16 / baudrate
 *
 */



uint32_t  UARTInit(uint32_t  uartnum, uint32_t  baudrate)
{
	uint32_t						fdiv;

	switch  (uartnum)
	{
		case  0:
		LPC_SC->PCONP |= (1<<3);				// power-up UART0
        LPC_PINCON->PINSEL0 = 0x00000050;       // RxD0 is P0.3 and TxD0 is P0.2
    	LPC_UART0->LCR = 0x80;          		// enable access to baudrate divisors
   		fdiv = (SystemCoreClock/4/16) / baudrate ;	// calc baudrate divisor based on core clock
    	LPC_UART0->DLM = fdiv / 256;                                                    
    	LPC_UART0->DLL = fdiv % 256;
        LPC_UART0->LCR = 0x03;              	// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
    	LPC_UART0->FCR = 0x07;          		// enable and reset TX and RX FIFO
		LPC_UART0->IER = (IER_RBR|IER_RLS);		// enable interrupts on RBR and RLS, no THRE for now (someone better enable interrupts!)
		UART0RcvOutIndex = 0;					// need to rewind the receive queue indices
		UART0RcvInIndex = 0;
		NVIC_EnableIRQ(UART0_IRQn);				// now turn on the IRQ handler
    	return (TRUE);

		case  1:
		LPC_SC->PCONP |= (1<<4);				// power-up UART1
		LPC_SC->PCLKSEL0 &= ~(3UL << 8);		// 00 in bits 9.8 sets UART1's PCLK=CCLK/4
		LPC_PINCON->PINSEL0 |= (1<<30);			// enable TxD1 P0.15
		LPC_PINCON->PINSEL1 |= (1<<0);			// enable RxD1 P0.16
    	LPC_UART1->LCR = 0x80;          		// enable access to baudrate divisors
	    fdiv = ((SystemCoreClock/4) / (16 * baudrate));	//baud rate
		LPC_UART1->DLM = fdiv / 256;                                                    
		LPC_UART1->DLL = fdiv % 256;
        LPC_UART1->LCR = 0x03;          		// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
    	LPC_UART1->FCR = 0x07;          		// enable and reset TX and RX FIFO
		LPC_UART1->IER = (IER_RBR|IER_RLS);		// enable interrupts on RBR and RLS, no THRE for now (someone better enable interrupts!)
		UART1RcvOutIndex = 0;					// need to rewind the receive queue indices
		UART1RcvInIndex = 0;
		NVIC_EnableIRQ(UART1_IRQn);				// now turn on the IRQ handler
    	return (TRUE);

		case  2:
		LPC_SC->PCONP |= (1<<24);				// power-up UART2
		LPC_SC->PCLKSEL1 &= ~(3UL << 16);		// 00 in bits 17.16 sets UART2's PCLK=CCLK/4
		LPC_PINCON->PINSEL0 |= ((1<<22) | (1<<20));  // enable RxD2 P0.11, TxD2 P0.10
    	LPC_UART2->LCR = 0x80;          		// enable access to baudrate divisors
	    fdiv = ((SystemCoreClock/4) / (16 * baudrate));	//baud rate
		LPC_UART2->DLM = fdiv / 256;                                                    
		LPC_UART2->DLL = fdiv % 256;
        LPC_UART2->LCR = 0x03;          		// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
    	LPC_UART2->FCR = 0x07;          		// enable and reset TX and RX FIFO
		LPC_UART2->IER = (IER_RBR|IER_RLS);		// enable interrupts on RBR and RLS, no THRE for now (someone better enable interrupts!)
		UART2RcvOutIndex = 0;					// need to rewind the receive queue indices
		UART2RcvInIndex = 0;
		NVIC_EnableIRQ(UART2_IRQn);				// now turn on the IRQ handler
    	return (TRUE);

		case  3:
		LPC_SC->PCONP |= (1<<25);				// power-up UART3
		LPC_SC->PCLKSEL1 &= ~(3UL << 18);		// 00 in bits 19.18 sets UART3's PCLK=CCLK/4
		LPC_PINCON->PINSEL0 |= ((2<<0) | (2<<2));  // enable RxD3 P0.1, TxD3 P0.0
    	LPC_UART3->LCR = 0x80;          		// enable access to baudrate divisors
	    fdiv = ((SystemCoreClock/4UL) / (16UL * baudrate));	//baud rate
		LPC_UART3->DLM = fdiv / 256;                                                    
		LPC_UART3->DLL = fdiv % 256;
        LPC_UART3->LCR = 0x03;          		// disable baudrate divisor access, 8 bits, no parity, 1 stop bit
    	LPC_UART3->FCR = 0x07;          		// enable and reset TX and RX FIFO, trigger one char
		LPC_UART3->IER = (IER_RBR|IER_RLS);		// enable interrupts on RBR and RLS, no THRE for now (someone better enable interrupts!)
		UART3RcvOutIndex = 0;					// need to rewind the receive queue indices
		UART3RcvInIndex = 0;
		NVIC_EnableIRQ(UART3_IRQn);				// now turn on the IRQ handler
    	return (TRUE);

		default:								// unknown uart, nothing we can do
		return (FALSE); 
	}
}



int32_t  UARTRegisterCharCallback(uint8_t portNum, void(* alert)(char  flag), char  *chrs)
{
	if (strlen(chrs) >= MAX_LEN_CALLBACK_CHAR_LIST)  return  -1;

	if  (portNum == 0)
	{
		strcpy(UART0CallbackList, chrs);
		alert0 = alert;
	}
	else if (portNum == 1)
	{
		strcpy(UART1CallbackList, chrs);
		alert1 = alert;
	}
	else if (portNum == 2)
	{
		strcpy(UART2CallbackList, chrs);
		alert2 = alert;
	}
	else if (portNum == 3)
	{
		strcpy(UART3CallbackList, chrs);
		alert3 = alert;
	}
	else  return  -1;

	return  strlen(chrs);
}


	


/*
 *  UART0Open      open file stream to UART0
 *
 *  There is nothing to do to open a stream to a UART.  The calling program must have
 *  already invoked UARTInit() to set up interrupts, baud rate, and other parameters.
 */
int  UART0Open(const char  *path, int  flags, int  mode)
{
	return  0;									// this always works
}





/*
 *  UART0Write      writes chars to UART0
 */
long  UART0Write(int fd, const char *ptr, int len)
{
	int					n;

    for (n = 0; n < len; n++)
	{
		while ((LPC_UART0->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
		LPC_UART0->THR = (*ptr++ & (uint16_t)0xff);
	}

	return  n;
}




/*
 *  UART0Avail      returns number of chars waiting in UART0's holding area
 *
 *  If UART0 was set up with receive polling, the return value is at most 1.
 *  If UART0 was set up with interrupt, the return value is the number of chars
 *  waiting in the receive queue.
 */
int  UART0Avail(int  fd)
{
	int						val;

	if (LPC_UART0->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
	{
		val = (UART0RcvInIndex - UART0RcvOutIndex);	// race condition!  need to turn off interrupts?
		if (val < 0)  val = -val;
	}
	else
	{
		if (LPC_UART0->LSR & LSR_RDR)  val = 1;
		else						   val = 0;
	}
	return  val;
}



/*
 *  UART0Read      reads (blocks) len chars from stream fd, writes to buffer at ptr
 */
long  UART0Read(int fd, char *ptr, int len)
{
	long					c;
	int						chars;

	for (chars=0; chars<len; chars++)
	{
		while (!UART0Avail(fd))  ;					// block until char arrives

		if (LPC_UART0->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
		{
			c = UART0RcvQ[UART0RcvOutIndex];		// get next available char
			UART0RcvOutIndex++;						// bump the index
			UART0RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index legal
		}
		else
		{
			c = (int)(LPC_UART0->RBR & (uint16_t)0xff);	// get char from holding buffer
		}
		*ptr = c;									// save the char we got
		ptr++;										// move to next cell
	}
	return  chars;
}



/*
 *  UART0Close      close stream connection to UART0
 */
int  UART0Close(int  fd)
{
	return  0;									// this always works
}






/*
 *  UART1Open      open file stream to UART1
 *
 *  There is nothing to do to open a stream to a UART.  The calling program must have
 *  already invoked UARTInit() to set up interrupts, baud rate, and other parameters.
 */
int  UART1Open(const char  *path, int  flags, int  mode)
{
	return  0;									// this always works
}


/*
 *  UART1Write      writes chars to UART1
 */
long  UART1Write(int fd, const char *ptr, int len)
{
	int					n;

    for (n = 0; n < len; n++)
	{
		while ((LPC_UART1->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
		LPC_UART1->THR = (*ptr++ & (uint16_t)0xff);
	}

	return  n;
}



/*
 *  UART1Avail      returns number of chars waiting in UART1's holding area
 *
 *  If UART1 was set up with receive polling, the return value is at most 1.
 *  If UART1 was set up with interrupt, the return value is the number of chars
 *  waiting in the receive queue.
 */
int  UART1Avail(int  fd)
{
	int						val;

	if (LPC_UART1->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
	{
		val = (UART1RcvInIndex - UART1RcvOutIndex);	// race condition!  need to turn off interrupts?
		if (val < 0)  val = -val;
	}
	else
	{
		if (LPC_UART1->LSR & LSR_RDR)  val = 1;
		else						   val = 0;
	}
	return  val;
}



/*
 *  UART1Read      reads (blocks) len chars from stream fd, writes to buffer at ptr
 */
long  UART1Read(int fd, char *ptr, int len)
{
	long					c;
	int						chars;

	for (chars=0; chars<len; chars++)
	{
		while (!UART1Avail(fd))  ;					// block until char arrives

		if (LPC_UART1->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
		{
			c = UART1RcvQ[UART1RcvOutIndex];		// get next available char
			UART1RcvOutIndex++;						// bump the index
			UART1RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index legal
		}
		else
		{
			c = (int)(LPC_UART1->RBR & (uint16_t)0xff);	// get char from holding buffer
		}
		*ptr = c;									// save the char we got
		ptr++;										// move to next cell
	}
	return  chars;
}



/*
 *  UART1Close      close stream connection to UART1
 */
int  UART1Close(int  fd)
{
	return  0;									// this always works
}





/*
 *  UART2Open      open file stream to UART2
 *
 *  There is nothing to do to open a stream to a UART.  The calling program must have
 *  already invoked UARTInit() to set up interrupts, baud rate, and other parameters.
 */
int  UART2Open(const char  *path, int  flags, int  mode)
{
	return  0;									// this always works
}



/*
 *  UART2Write      writes chars to UART2
 */
long  UART2Write(int fd, const char *ptr, int len)
{
	int					n;

    for (n = 0; n < len; n++)
	{
		while ((LPC_UART2->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
		LPC_UART2->THR = (*ptr++ & (uint16_t)0xff);
	}

	return  n;
}



/*
 *  UART2Avail      returns number of chars waiting in UART2's holding area
 *
 *  If UART2 was set up with receive polling, the return value is at most 1.
 *  If UART2 was set up with interrupt, the return value is the number of chars
 *  waiting in the receive queue.
 */
int  UART2Avail(int  fd)
{
	int						val;

	if (LPC_UART2->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
	{
		val = (UART2RcvInIndex - UART2RcvOutIndex);	// race condition!  need to turn off interrupts?
		if (val < 0)  val = -val;
	}
	else
	{
		if (LPC_UART2->LSR & LSR_RDR)  val = 1;
		else						   val = 0;
	}
	return  val;
}



/*
 *  UART2Read      reads (blocks) len chars from stream fd, writes to buffer at ptr
 */
long  UART2Read(int fd, char *ptr, int len)
{
	long					c;
	int						chars;

	for (chars=0; chars<len; chars++)
	{
		while (!UART2Avail(fd))  ;					// block until char arrives

		if (LPC_UART2->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
		{
			c = UART2RcvQ[UART2RcvOutIndex];		// get next available char
			UART2RcvOutIndex++;						// bump the index
			UART2RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index legal
		}
		else
		{
			c = (int)(LPC_UART2->RBR & (uint16_t)0xff);	// get char from holding buffer
		}
		*ptr = c;									// save the char we got
		ptr++;										// move to next cell
	}
	return  chars;
}


/*
 *  UART2Close      close stream connection to UART2
 */
int  UART2Close(int  fd)
{
	return  0;									// this always works
}






/*
 *  UART3Open      open file stream to UART3
 *
 *  There is nothing to do to open a stream to a UART.  The calling program must have
 *  already invoked UARTInit() to set up interrupts, baud rate, and other parameters.
 */
int  UART3Open(const char  *path, int  flags, int  mode)
{
	return  0;									// this always works
}

/*
 *  UART3Write      writes chars to UART3
 */
long  UART3Write(int fd, const char *ptr, int len)
{
	int					n;

    for (n = 0; n < len; n++)
	{
		while ((LPC_UART3->LSR & LSR_THRE) == 0)  ;		// blocks until OK to send
		LPC_UART3->THR = (*ptr++ & (uint16_t)0xff);
	}

	return  n;
}



/*
 *  UART3Avail      returns number of chars waiting in UART3's holding area
 *
 *  If UART3 was set up with receive polling, the return value is at most 1.
 *  If UART3 was set up with interrupt, the return value is the number of chars
 *  waiting in the receive queue.
 */
int  UART3Avail(int  fd)
{
	int						val;

	if (LPC_UART3->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
	{
		val = (UART3RcvInIndex - UART3RcvOutIndex);	// race condition!  need to turn off interrupts?
		if (val < 0)  val = -val;
	}
	else
	{
		if (LPC_UART3->LSR & LSR_RDR)  val = 1;
		else						   val = 0;
	}
	return  val;
}



/*
 *  UART3Read      reads (blocks) len chars from stream fd, writes to buffer at ptr
 */
long  UART3Read(int fd, char *ptr, int len)
{
	long					c;
	int						chars;

	for (chars=0; chars<len; chars++)
	{
		while (!UART3Avail(fd))  ;					// block until char arrives

		if (LPC_UART3->IER & (IER_RBR|IER_RLS))		// if receive interrupts enabled...
		{
			c = UART3RcvQ[UART3RcvOutIndex];		// get next available char
			UART3RcvOutIndex++;						// bump the index
			UART3RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index legal
		}
		else
		{
			c = (int)(LPC_UART3->RBR & (uint16_t)0xff);	// get char from holding buffer
		}
		*ptr = c;									// save the char we got
		ptr++;										// move to next cell
	}
	return  chars;
}


/*
 *  UART3Close      close stream connection to UART3
 */
int  UART3Close(int  fd)
{
	return  0;									// this always works
}






/*
 *  UART0_IRQHandler      ISR handler for UART0 interrupts
 *
 *  This handler overwrites the dummy handler in the startup file and
 *  provides support for received data interrupts.
 *
 *  (To do: Add transmit interrupts if needed.)
 */
void UART0_IRQHandler (void) 
{
	uint8_t					IIRValue;
	uint8_t					LSRValue;
	uint8_t					Dummy = Dummy;
	uint8_t					c;
        
	c = 0;										// always clear this!
	IIRValue = LPC_UART0->IIR;					// get the interrupt source bits
 	IIRValue >>= 1;								// skip pending bit in IIR (bit 0)
	IIRValue &= 0x07;							// check original bits 1-3 for interrupt identification
	if (IIRValue == IIR_RLS)					// if Receive Line Status interrupt...
	{
		LSRValue = LPC_UART0->LSR;				// get receive line status
		if (LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI))	// if receive errors...
        {
          Dummy = LPC_UART0->RBR;				// dummy read on RX to clear interrupt
          return;
        }
        if (LSRValue & LSR_RDR)					// if Receive Data Ready interrupt...                        
		{
			c = LPC_UART0->RBR;					// get data, read clears interrupt
			UART0RcvQ[UART0RcvInIndex] = c;		// save in queue
			UART0RcvInIndex++;					// move to next cell
			UART0RcvInIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			if (UART0RcvInIndex == UART0RcvOutIndex)	// if we filled the buffer...
			{
				UART0RcvOutIndex++;				// wipe out oldest char in buffer (good as any other solution)
				UART0RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			}
		}
	}
	else if (IIRValue == IIR_RDA)				// if Receive Data Available...
	{
		c = LPC_UART0->RBR;						// get data, read clears interrupt
		UART0RcvQ[UART0RcvInIndex] = c;			// save to queue
		UART0RcvInIndex++;						// move to next cell
		UART0RcvInIndex %= MAX_RCV_Q_CHARS;		// keep index in legal range
		if (UART0RcvInIndex == UART0RcvOutIndex)	// if we filled the buffer...
		{
			UART0RcvOutIndex++;					// wipe out oldest char in buffer (good as any other solution)
			UART0RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
		}     
	}
	else if (IIRValue == IIR_CTI)			// character timeout indicator
	{
	}
/*
 *  Add code in this IF clause to provide interrupt-driven transmit support.
 *  The following code is left over from the original ISR code and needs to be
 *  replaced.
 */
	else if (IIRValue == IIR_THRE)			// if transmit holding register is empty...
	{
        LSRValue = LPC_UART0->LSR;			// check status in the LSR valid data in U0THR
        if (LSRValue & LSR_THRE)
        {
//			UART3TxEmpty = 1;
        }
        else
        {
//			UART3TxEmpty = 0;
        }
	}
	if (strchr(UART0CallbackList, c))		// if we got a char and it is in the callback list...
	{
		if (*alert0)						// if a callback is active...
		{
			alert0(c);
		}
	}
}



/*
 *  UART1_IRQHandler      ISR handler for UART1 interrupts
 *
 *  This handler overwrites the dummy handler in the startup file and
 *  provides support for received data interrupts.
 *
 *  (To do: Add transmit interrupts if needed.)
 */
void UART1_IRQHandler (void) 
{
	uint8_t					IIRValue;
	uint8_t					LSRValue;
	uint8_t					Dummy = Dummy;
	uint8_t					c;
        
	c = 0;										// always clear this!
	IIRValue = LPC_UART1->IIR;					// get the interrupt source bits
 	IIRValue >>= 1;								// skip pending bit in IIR (bit 0)
	IIRValue &= 0x07;							// check original bits 1-3 for interrupt identification
	if (IIRValue == IIR_RLS)					// if Receive Line Status interrupt...
	{
		LSRValue = LPC_UART1->LSR;				// get receive line status
		if (LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI))	// if receive errors...
        {
          Dummy = LPC_UART1->RBR;				// dummy read on RX to clear interrupt
          return;
        }
        if (LSRValue & LSR_RDR)					// if Receive Data Ready interrupt...                        
		{
			c = LPC_UART1->RBR;					// get data, read clears interrupt
			UART1RcvQ[UART1RcvInIndex] = c;		// save in queue
			UART1RcvInIndex++;					// move to next cell
			UART1RcvInIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			if (UART1RcvInIndex == UART1RcvOutIndex)	// if we filled the buffer...
			{
				UART1RcvOutIndex++;				// wipe out oldest char in buffer (good as any other solution)
				UART1RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			}     
		}
	}
	else if (IIRValue == IIR_RDA)				// if Receive Data Available...
	{
		c = LPC_UART1->RBR;						// get data, read clears interrupt
		UART1RcvQ[UART1RcvInIndex] = c;			// save to queue
		UART1RcvInIndex++;						// move to next cell
		UART1RcvInIndex %= MAX_RCV_Q_CHARS;		// keep index in legal range
		if (UART1RcvInIndex == UART1RcvOutIndex)	// if we filled the buffer...
		{
			UART1RcvOutIndex++;					// wipe out oldest char in buffer (good as any other solution)
			UART1RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
		}     
	}
	else if (IIRValue == IIR_CTI)			// character timeout indicator
	{
	}
/*
 *  Add code in this IF clause to provide interrupt-driven transmit support.
 *  The following code is left over from the original ISR code and needs to be
 *  replaced.
 */
	else if (IIRValue == IIR_THRE)			// if transmit holding register is empty...
	{
        LSRValue = LPC_UART1->LSR;			// check status in the LSR valid data in U0THR
        if (LSRValue & LSR_THRE)
        {
//			UART3TxEmpty = 1;
        }
        else
        {
//			UART3TxEmpty = 0;
        }
	}
	if (strchr(UART1CallbackList, c))		// if we got a char and it is in the callback list...
	{
		if (*alert1)						// if a callback is active...
		{
			alert1(c);
		}
	}
}



/*
 *  UART2_IRQHandler      ISR handler for UART2 interrupts
 *
 *  This handler overwrites the dummy handler in the startup file and
 *  provides support for received data interrupts.
 *
 *  (To do: Add transmit interrupts if needed.)
 */
void UART2_IRQHandler (void) 
{
	uint8_t					IIRValue;
	uint8_t					LSRValue;
	uint8_t					Dummy = Dummy;
	uint8_t					c;
        
	c = 0;										// always clear this!
	IIRValue = LPC_UART2->IIR;					// get the interrupt source bits
 	IIRValue >>= 1;								// skip pending bit in IIR (bit 0)
	IIRValue &= 0x07;							// check original bits 1-3 for interrupt identification
	if (IIRValue == IIR_RLS)					// if Receive Line Status interrupt...
	{
		LSRValue = LPC_UART2->LSR;				// get receive line status
		if (LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI))	// if receive errors...
        {
          Dummy = LPC_UART2->RBR;				// dummy read on RX to clear interrupt
          return;
        }
        if (LSRValue & LSR_RDR)					// if Receive Data Ready interrupt...                        
		{
			c = LPC_UART2->RBR;					// get data, read clears interrupt
			UART2RcvQ[UART2RcvInIndex] = c;		// save in queue
			UART2RcvInIndex++;					// move to next cell
			UART2RcvInIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			if (UART2RcvInIndex == UART2RcvOutIndex)	// if we filled the buffer...
			{
				UART2RcvOutIndex++;				// wipe out oldest char in buffer (good as any other solution)
				UART2RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			}
		}
	}
	else if (IIRValue == IIR_RDA)				// if Receive Data Available...
	{
		c = LPC_UART2->RBR;						// get data, read clears interrupt
		UART2RcvQ[UART2RcvInIndex] = c;			// save to queue
		UART2RcvInIndex++;						// move to next cell
		UART2RcvInIndex %= MAX_RCV_Q_CHARS;		// keep index in legal range
		if (UART2RcvInIndex == UART2RcvOutIndex)	// if we filled the buffer...
		{
			UART2RcvOutIndex++;					// wipe out oldest char in buffer (good as any other solution)
			UART2RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
		}
	}
	else if (IIRValue == IIR_CTI)			// character timeout indicator
	{
	}
/*
 *  Add code in this IF clause to provide interrupt-driven transmit support.
 *  The following code is left over from the original ISR code and needs to be
 *  replaced.
 */
	else if (IIRValue == IIR_THRE)			// if transmit holding register is empty...
	{
        LSRValue = LPC_UART2->LSR;			// check status in the LSR valid data in U0THR
        if (LSRValue & LSR_THRE)
        {
//			UART3TxEmpty = 1;
        }
        else
        {
//			UART3TxEmpty = 0;
        }
	}
	if (strchr(UART2CallbackList, c))		// if we got a char and it is in the callback list...
	{
		if (*alert2)						// if a callback is active...
		{
			alert2(c);
		}
	}
}



/*
 *  UART3_IRQHandler      ISR handler for UART3 interrupts
 *
 *  This handler overwrites the dummy handler in the startup file and
 *  provides support for received data interrupts.
 *
 *  (To do: Add transmit interrupts if needed.)
 */
void UART3_IRQHandler (void) 
{
	uint8_t					IIRValue;
	uint8_t					LSRValue;
	uint8_t					Dummy = Dummy;
	uint8_t					c;
        
	c = 0;										// always clear this!
	IIRValue = LPC_UART3->IIR;					// get the interrupt source bits
 	IIRValue >>= 1;								// skip pending bit in IIR (bit 0)
	IIRValue &= 0x07;							// check original bits 1-3 for interrupt identification
	if (IIRValue == IIR_RLS)					// if Receive Line Status interrupt...
	{
		LSRValue = LPC_UART3->LSR;				// get receive line status
		if (LSRValue & (LSR_OE|LSR_PE|LSR_FE|LSR_RXFE|LSR_BI))	// if receive errors...
        {
          Dummy = LPC_UART3->RBR;				// dummy read on RX to clear interrupt
          return;
        }
        if (LSRValue & LSR_RDR)					// if Receive Data Ready interrupt...                        
		{
			c = LPC_UART3->RBR;					// get data, read clears interrupt
			UART3RcvQ[UART3RcvInIndex] = c;		// save in queue
			UART3RcvInIndex++;					// move to next cell
			UART3RcvInIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			if (UART3RcvInIndex == UART3RcvOutIndex)	// if we filled the buffer...
			{
				UART3RcvOutIndex++;				// wipe out oldest char in buffer (good as any other solution)
				UART3RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
			}     
		}
	}
	else if (IIRValue == IIR_RDA)				// if Receive Data Available...
	{
		c = LPC_UART3->RBR;						// get data, read clears interrupt
		UART3RcvQ[UART3RcvInIndex] = c;			// save to queue
		UART3RcvInIndex++;						// move to next cell
		UART3RcvInIndex %= MAX_RCV_Q_CHARS;		// keep index in legal range
		if (UART3RcvInIndex == UART3RcvOutIndex)	// if we filled the buffer...
		{
			UART3RcvOutIndex++;					// wipe out oldest char in buffer (good as any other solution)
			UART3RcvOutIndex %= MAX_RCV_Q_CHARS;	// keep index in legal range
		}     
	}
	else if (IIRValue == IIR_CTI)			// character timeout indicator
	{
	}
/*
 *  Add code in this IF clause to provide interrupt-driven transmit support.
 *  The following code is left over from the original ISR code and needs to be
 *  replaced.
 */
	else if (IIRValue == IIR_THRE)			// if transmit holding register is empty...
	{
        LSRValue = LPC_UART3->LSR;			// check status in the LSR valid data in U0THR
        if (LSRValue & LSR_THRE)
        {
//			UART3TxEmpty = 1;
        }
        else
        {
//			UART3TxEmpty = 0;
        }
	}
	if (strchr(UART3CallbackList, c))		// if we got a char and it is in the callback list...
	{
		if (*alert3)						// if a callback is active...
		{
			alert3(c);
		}
	}
 }



