###############################################################################
# Makefile for the project sdlocker1k
###############################################################################

## General Flags
PROJECT = sdlocker1k
F_CPU = 8000000
TARGET = $(PROJECT).elf

# The MCU_ARCH macro is used by the GCC compiler to define the AVR architecture;
# see the -mcu compiler option for details.
# For example, ATmega328P uses avr5 and ATtiny1634 uses avr35.
MCU_ARCH = avr5

# The MCU macro is used by the AVR include files to define the specific
# micro you are using; see avr/io.h for MCU definitions.
MCU = __AVR_ATmega48P__

#  Select the toolchain by providing a path to the top level
#  directory; this will be the folder that holds the
#  AVR subfolders.
TOOLPATH = C:/AVR8-gcc/avr8-gnu-toolchain
#TOOLPATH = C:/WinAVR-20100110

#  List of object files to build.  All object files will be built in
#  the working directory, not the source directories.
#
#  Note that this list does NOT include any prebuilt object files you
#  want to add.  Those are listed below in the LINKONLYOBJECTS macro.
#
#  You will need as a minimum your $(PROJECT).o file.
OBJECTS	 = $(PROJECT).o
# OBJECTS += C:/AVR8-gcc/avr8-gnu-toolchain/avr/lib/avr5/crtatmega328p.o
OBJECTS += start_m328p.o

#  Describe the various include and source directories needed.
#  These usually point to files from whatever distribution
#  you are using.  This can also include paths to any needed
#  GCC includes or libraries.
GCC_INC          = $(TOOLPATH)/include
GCC_INC			+= $(TOOLPATH)/avr/include
GCC_INC			+= $(TOOLPATH)/avr/include/avr
GCC_INC			+= $(TOOLPATH)/avr/include/sys
GCC_INC			+= $(TOOLPATH)/avr/include/util

#  Options common to compile, link and assembly rules
COMMON = -mmcu=$(MCU_ARCH)

#  Define the linker script used to assign section of memory
LDSCRIPT = ./ATmega328p_ld.x

#  Compile options common for all C compilation units.
#  Note that including -nostartfiles means the linker will NOT
#  include the standard startup object file, crtatmega328p.o.
#  If you include -nostartfiles, you MUST provide the vector table
#  and startup code, if any, for your project!
CFLAGS = $(COMMON)
CFLAGS += -Wall -gdwarf-2 -std=gnu99 -DAVR -DF_CPU=$(F_CPU)UL -Os -funsigned-char -funsigned-bitfields -fpack-struct -fshort-enums
CFLAGS += -D$(MCU)
# CFLAGS += -D__DEBUG__
CFLAGS += -MD -MP -MT $(*F).o -MF dep/$(@F).d 

#  Assembly specific flags
ASMFLAGS = $(COMMON)
ASMFLAGS += -a

#  All possible source directories other than '.' must be defined in
#  the VPATH variable.  This lets make tell the compiler where to find
#  source files outside of the working directory.  If you need more
#  than one directory, separate them with ':'.
VPATH =

#  List of directories to be searched for include files during compilation
INCDIRS  = -I$(GCC_INC)

#  List of directories to be searched for libraries.  These can be
#  either GCC-supplied libraries (such as libc.a) or your own, pre-
#  build libraries (such as libuart.a).
#
#  Use -L in front of each library directory path (-L../library)
LIBDIRS  = -L../library
LIBDIRS += -L../library/atmega328p

#  List of libraries (archives) used by this program.  These will be
#  included from the libraries called out in LIBDIRS above.
#
#  Use -l in front of each archive name (-lm).
LIBS  = -luart

#  Linker flags
LDFLAGS = $(COMMON)
LDFLAGS +=  -Wl,-Map=$(PROJECT).map -T$(LDSCRIPT)
LDFLAGS += -nostartfiles
LDFLAGS += $(LIBDIRS) $(LIBS)


#  Intel Hex file production flags
HEX_FLASH_FLAGS  = -j .text -j .data
HEX_FLASH_FLAGS += -R .eeprom -R .fuse -R .lock -R .signature

HEX_EEPROM_FLAGS = -j .eeprom


#  Prebuilt objects explicitly added by the user; these will NOT be built!
#
#  If you want to use the stock avr8-gcc startup code, add the path
#  to crtatmega328p.o (or whatever startup object your MCU needs) here.
#  If you will be using a custom startup routine, you can either prebuild
#  it and add a path to it here, or you can include it in the OBJECTS
#  macro defined above.
#LINKONLYOBJECTS = $(TOOLPATH)/avr/lib/avr5/crtatmega328p.o
LINKONLYOBJECTS = 

#  Tools paths
#
#  Define an explicit path to the GNU tools used by make.
#  If you are ABSOLUTELY sure that your PATH variable is
#  set properly, you can remove the BINDIR variable.
#
BINDIR = $(TOOLPATH)/bin

CC = $(BINDIR)/avr-gcc
AS = $(BINDIR)/avr-as
AR = $(BINDIR)/avr-ar
LD = $(BINDIR)/avr-ld
OBJCOPY = $(BINDIR)/avr-objcopy
SIZE = $(BINDIR)/avr-size
OBJDUMP = $(BINDIR)/avr-objdump

#  Define a command for removing folders and files during clean.  The
#  simplest such command is Linux' rm with the -f option.  You can find
#  suitable versions of rm on the web.
REMOVE = rm -f


#  Build
all: $(TARGET) $(PROJECT).eep $(PROJECT).lss $(PROJECT).hex stats


#  Link
$(TARGET): $(OBJECTS) $(LINKONLYOBJECTS) $(PROJECT).mak

%.hex: $(TARGET)
	$(OBJCOPY) $(HEX_FLASH_FLAGS) -O ihex   $< $@

%.eep: $(TARGET)
	$(OBJCOPY) $(HEX_EEPROM_FLAGS) -O ihex $< $@ || exit 0

%.lss: $(TARGET)
	$(OBJDUMP) -h -S $< > $@


## Other dependencies
-include $(shell mkdir dep 2>NUL) $(wildcard dep/*)

#  Linker invocation
#  Uncomment one of the two lines that begin $(LD).
#  If you use the longer line, any errors will be reformatted for use by the Visual Studio
#  Intellisense application, so you can double-click on the error and go directly to the source
#  line.  If you use the shorter line, you get the regular GCC error format.
$(PROJECT).elf: $(OBJECTS)
	$(CC) $(OBJECTS) $(LINKONLYOBJECTS) $(LDFLAGS) -o $(PROJECT).elf 2>&1 | sed -e 's/\(\w\+\):\([ 0-9]\+\):/\1(\2):/'
#	$(CC) $(OBJECTS) $(LINKONLYOBJECTS) $(LDFLAGS) -o $(PROJECT).elf

stats: $(PROJECT).elf
	$(SIZE) $(PROJECT).elf

dump: $(PROJECT).elf
	$(OBJDUMP) -h $(PROJECT).elf	

clean:
	$(REMOVE) *.o
	$(REMOVE) $(PROJECT).hex
	$(REMOVE) $(PROJECT).elf
	$(REMOVE) $(PROJECT).map
	$(REMOVE) $(PROJECT).bin
	$(REMOVE) *.lst

#  The toolvers target provides a sanity check, so you can determine
#  exactly which version of each tool will be used when you build.
#  If you use this target, make will display the first line of each
#  tool invocation.
#  To use this feature, enter from the command-line:
#    make -f $(PROJECT).mak toolvers
toolvers:
	$(CC) --version | sed q
	$(AS) --version | sed q
	$(LD) --version | sed q
	$(AR) --version | sed q
	$(OBJCOPY) --version | sed q
	$(SIZE) --version | sed q
	$(OBJDUMP) --version | sed q

#########################################################################
#  Default rules to compile .c and .cpp file to .o
#  and assemble .s files to .o

#  There are two options for compiling .c files to .o; uncomment only one.
#  The shorter option is suitable for making from the command-line.
#  The option with the sed script on the end is used if you want to
#  compile from Visual Studio; the sed script reformats error messages
#  so Visual Studio's IntelliSense feature can track back to the source
#  file with the error.
.c.o :
	@echo Compiling $<, writing to $@...
#	$(CC) $(CFLAGS) $(INCDIRS) -c $< -o $@
	$(CC) $(CFLAGS) $(INCDIRS) -c $< 2>&1 | sed -e 's/\(\w\+\):\([ 0-9]\+\):/\1(\2):/'

.cpp.o :
	@echo Compiling $<, writing to $@...
	$(CC) $(CFLAGS) -c $<

.s.o :
	@echo Assembling $<, writing to $@...
	$(AS) $(ASMFLAGS) -o $@ $<  > $(basename $@).lst

#########################################################################
