/*======================================================================

    Aironet driver for 4500 and 4800 340/350 series cards

    The contents of this file are subject to the Mozilla Public License
    Version 1.1 (the "License"); you may not use this file except in
    compliance with the License. You may obtain a copy of the License at
    http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS IS" basis,
    WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
    for the specificlanguage governing rights and limitations under the License.

    The Original Code is a driver for Aironet 4500, 4800, 340 and 350 cards.

    The Initial Developer of the Original Code is Benjamin Reed
    <breed@almaden.ibm.com>. Portions created by Benjamin Reed
    are Copyright (C) 1999 Benjamin Reed. All Rights Reserved.

    Contributor(s): Cisco Systems, Inc.

    Portions created by Cisco Systems, Inc. are
    Copyright (C) 2003, 2004 Cisco Systems, Inc. All Rights Reserved.
 
    Dec 2003 - Cisco    - fixed compatibility with newer firmware
                          added driver version request
                          added RF monitor support
			  fixed wireless-tools support
    
======================================================================*/

#ifndef __KERNEL__
#define __KERNEL__
#endif

#ifndef MODULE
#define MODULE
#endif

#include <linux/config.h>
#include <linux/version.h>
#include <linux/init.h>
#include <asm/segment.h>
#ifdef CONFIG_MODVERSIONS
#define MODVERSIONS
#include <linux/modversions.h>
#endif                 

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>

#include <linux/sched.h>
#include <linux/ptrace.h>
#if (LINUX_VERSION_CODE < 0x020363)
#include <linux/malloc.h>
#else
#include <linux/slab.h>
#endif
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/bitops.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>
#include <linux/config.h>
#ifdef CONFIG_PCI
#  include <linux/pci.h>


#include "aes.h"

static struct {
  unsigned short vendor;
  unsigned short id;
} card_ids[] = { { 0x14b9, 1 },
		 { 0x14b9, 0x4500 },
		 { 0x14b9, 0x4800 },
		 { 0x14b9, 0x0340 },
		 { 0x14b9, 0x0350 },
		 { 0, 0 } 
};
#endif

#ifndef _COMPAT_NETDEVICE_H
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,14))
#ifndef net_device
#define net_device              device
#endif
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,43))
#define DEV_KFREE_SKB(skb)      dev_kfree_skb(skb)
#define netif_stop_queue(dev)   set_bit(0, (void *)&(dev)->tbusy)
#define netif_start_queue(dev)  clear_bit(0, (void *)&(dev)->tbusy)
#define netif_wake_queue(dev) \
    do { netif_start_queue(dev); mark_bh(NET_BH); } while (0)
#define netif_device_attach(dev) \
    do { (dev)->start = 1; netif_start_queue(dev); } while (0)
#define netif_device_detach(dev) \
    do { (dev)->start = 0; netif_stop_queue(dev); } while (0)
#define netif_device_present(dev) ((dev)->start)
#define netif_running(dev)      ((dev)->start)
#define netif_mark_up(dev)      do { (dev)->start = 1; } while (0)
#define netif_mark_down(dev)    do { (dev)->start = 0; } while (0)
#define netif_carrier_on(dev)   do { dev->flags |= IFF_RUNNING; } while (0)
#define netif_carrier_off(dev)  do { dev->flags &= ~IFF_RUNNING; } while (0)
#define netif_queue_stopped(dev) ((dev)->tbusy)
#define dev_kfree_skb_irq(skb)  DEV_KFREE_SKB(skb)
#endif

#ifndef netif_mark_up
#define netif_mark_up(dev)	do { } while (0)
#define netif_mark_down(dev)	do { } while (0)
#endif
#endif

/* Don't check perms.if defined 
 */

#define OPENACCESS 


/*
 * rwilcher@cisco.com
 * I made this thing static again for the following reason.
 * If the kmalloc for this memory should fail  FOR ANY REASON
 * during flashing , the card will become an 
 * expensive ornament. If flashing fails midstream the 
 * card will be dead. I have done this and it aint pretty.
 * Allocating 32 K staticly beats the posibility of a user killing 
 * a radio IMHO .
 */

char flashbuffer[1024 * 38 ];   // Be chicken**** and staticly allocate 

/*
 INCLUDE_RFMONITOR - "#define" this will include RF monitor mode code for use
                     with sniffers
                     must have WIRELESS_EXT (include linux/wireless.h)
#define INCLUDE_RFMONITOR
 */

/* Include Wireless Extension definition and check version - Jean II */
/*
#include <linux/wireless.h>
#define WIRELESS_SPY		// enable iwspy support
#if WIRELESS_EXT < 9
#warning "Wireless extension v9 or newer required - please upgrade your kernel"
#undef WIRELESS_EXT
#endif
*/
#define CISCO_EXT		// enable Cisco extensions

#ifdef CISCO_EXT
#include <linux/delay.h>
#endif

/* As you can see this list is HUGH!
   I really don't know what a lot of these counts are about, but they
   are all here for completeness.  If the IGNLABEL macro is put in
   infront of the label, that statistic will not be included in the list
   of statistics in the /proc filesystem */

#define IGNLABEL 0&(int)
static char *statsLabels[] = {
  "RxOverrun",
  IGNLABEL "RxPlcpCrcErr",
  IGNLABEL "RxPlcpFormatErr",
  IGNLABEL "RxPlcpLengthErr",
  "RxMacCrcErr",
  "RxMacCrcOk",
  "RxWepErr",
  "RxWepOk",
  "RetryLong",
  "RetryShort",
  "MaxRetries",
  "NoAck",
  "NoCts",
  "RxAck",
  "RxCts",
  "TxAck",
  "TxRts",
  "TxCts",
  "TxMc",
  "TxBc",
  "TxUcFrags",
  "TxUcPackets",
  "TxBeacon",
  "RxBeacon",
  "TxSinColl",
  "TxMulColl",
  "DefersNo",
  "DefersProt",
  "DefersEngy",
  "DupFram",
  "RxFragDisc",
  "TxAged",
  "RxAged",
  "LostSync-MaxRetry",
  "LostSync-MissedBeacons",
  "LostSync-ArlExceeded",
  "LostSync-Deauth",
  "LostSync-Disassoced",
  "LostSync-TsfTiming",
  "HostTxMc",
  "HostTxBc",
  "HostTxUc",
  "HostTxFail",
  "HostRxMc",
  "HostRxBc",
  "HostRxUc",
  "HostRxDiscard",
  IGNLABEL "HmacTxMc",
  IGNLABEL "HmacTxBc",
  IGNLABEL "HmacTxUc",
  IGNLABEL "HmacTxFail",
  IGNLABEL "HmacRxMc",
  IGNLABEL "HmacRxBc",
  IGNLABEL "HmacRxUc",
  IGNLABEL "HmacRxDiscard",
  IGNLABEL "HmacRxAccepted",
  "SsidMismatch",
  "ApMismatch",
  "RatesMismatch",
  "AuthReject",
  "AuthTimeout",
  "AssocReject",
  "AssocTimeout",
  IGNLABEL "ReasonOutsideTable",
  IGNLABEL "ReasonStatus1",
  IGNLABEL "ReasonStatus2",
  IGNLABEL "ReasonStatus3",
  IGNLABEL "ReasonStatus4",
  IGNLABEL "ReasonStatus5",
  IGNLABEL "ReasonStatus6",
  IGNLABEL "ReasonStatus7",
  IGNLABEL "ReasonStatus8",
  IGNLABEL "ReasonStatus9",
  IGNLABEL "ReasonStatus10",
  IGNLABEL "ReasonStatus11",
  IGNLABEL "ReasonStatus12",
  IGNLABEL "ReasonStatus13",
  IGNLABEL "ReasonStatus14",
  IGNLABEL "ReasonStatus15",
  IGNLABEL "ReasonStatus16",
  IGNLABEL "ReasonStatus17",
  IGNLABEL "ReasonStatus18",
  IGNLABEL "ReasonStatus19",
  "RxMan",
  "TxMan",
  "RxRefresh",
  "TxRefresh",
  "RxPoll",
  "TxPoll",
  "HostRetries",
  "LostSync-HostReq",
  "HostTxBytes",
  "HostRxBytes",
  "ElapsedUsec",
  "ElapsedSec",
  "LostSyncBetterAP",
  "PrivacyMismatch",
  "Jammed",
  "DiscRxNotWepped",
  "PhyEleMismatch",
  (char*)-1 };
#ifndef RUN_AT
#define RUN_AT(x) (jiffies+(x))
#endif


/* These variables are for insmod, since it seems that the rates
   can only be set in setup_card.  Rates should be a comma separated
   (no spaces) list of rates (up to 8). */

static int rates[8] = {0,0,0,0,0,0,0,0};
static int basic_rate = 0;
static char *ssids[3] = {0,0,0};

static int io[4]={ 0,};
static int irq[4]={ 0,};

static
int maxencrypt = 0; /* The highest rate that the card can encrypt at.
		       0 means no limit.  For old cards this was 4 */
static
int aux_bap = 0; /* Checks to see if the aux ports are needed to read
		    the bap, needed on some older cards and buses. */   
static
int adhoc = 0;

#if (LINUX_VERSION_CODE >= 0x20420)
MODULE_LICENSE("MPL");
#endif
#if (LINUX_VERSION_CODE > 0x20155)
/* new kernel */
MODULE_AUTHOR("Benjamin Reed");
MODULE_DESCRIPTION("Support for Cisco/Aironet 802.11 wireless ethernet \
                   cards.  Direct support for ISA/PCI cards and support \
		   for PCMCIA when used with airo_cs.");
MODULE_SUPPORTED_DEVICE("Aironet 4500, 4800 and Cisco 340/350");
MODULE_PARM(io,"1-4i");
MODULE_PARM(irq,"1-4i");
MODULE_PARM(basic_rate,"i");
MODULE_PARM(rates,"1-8i");
MODULE_PARM(ssids,"1-3s");

MODULE_PARM(aux_bap,"i");
MODULE_PARM_DESC(aux_bap, "If non-zero, the driver will switch into a mode \
than seems to work better for older cards with some older buses.  Before \
switching it checks that the switch is needed.");
MODULE_PARM(maxencrypt, "i");
MODULE_PARM_DESC(maxencrypt, "The maximum speed that the card can do \
encryption.  Units are in 512kbs.  Zero (default) means there is no limit. \
Older cards used to be limited to 2mbs (4).");
MODULE_PARM(adhoc, "i");
MODULE_PARM_DESC(adhoc, "If non-zero, the card will start in adhoc mode.");

#include <asm/uaccess.h>

#define KFREE_SKB(a,b)  dev_kfree_skb(a)
#define PROC_REGISTER(a,b) proc_register(a,b)
#else 
/* old kernel */
#define SPIN_LOCK_UNLOCKED 0
#define spinlock_t int
#define spin_lock_irqsave(x, y) save_flags(y); cli()
#define spin_unlock_irqrestore(x, y) restore_flags(y)
#define timer_pending(a) (((a)->prev) != NULL)
#define KFREE_SKB(a,b)  dev_kfree_skb(a,b)
#define PROC_REGISTER(a,b) proc_register_dynamic(a,b)
#endif
#if (LINUX_VERSION_CODE < 0x020311)
#define PROC_UNREGISTER(root, entry) proc_unregister(root, (entry)->low_ino)
#else
#undef PROC_REGISTER
#define PROC_REGISTER(root, entry) error
#define PROC_UNREGISTER(root, entry) error
#endif

#define MIN(x,y) ((x<y)?x:y)

#define isalnum(x) ((x>='a'&&x<='z')||(x>='A'&&x<='Z')||(x>='0'&&x<='9'))

/* This is a kind of sloppy hack to get this information to OUT4500 and
   IN4500.  I would be extremely interested in the situation where this
   doesnt work though!!! */
static int do8bitIO = 0;

/* Return codes */
#define SUCCESS 0
#define ERROR -1
#define NO_PACKET -2

/* Commands */
#define NOP 0x0010
#define MAC_ENABLE 0x0001
#define MAC_DISABLE 0x0002
#define CMD_ACCESS 0x0021
#define CMD_ALLOCATETX 0x000a
#define CMD_TRANSMIT 0x000b
#define HOSTSLEEP 0x85
#define CMD_SETMODE 0x0009
#define CMD_ENABLEAUX 0x0111
#define CMD_SOFTRESET 0x0004

/* Registers */
#define COMMAND 0x00
#define PARAM0 0x02
#define PARAM1 0x04
#define PARAM2 0x06
#define STATUS 0x08
#define RESP0 0x0a
#define RESP1 0x0c
#define RESP2 0x0e
#define LINKSTAT 0x10
#define SELECT0 0x18
#define OFFSET0 0x1c
#define RXFID 0x20
#define TXALLOCFID 0x22
#define TXCOMPLFID 0x24
#define DATA0 0x36
#define EVSTAT 0x30
#define EVINTEN 0x32
#define EVACK 0x34
#define SWS0 0x28
#define SWS1 0x2a
#define SWS2 0x2c
#define SWS3 0x2e
#define AUXPAGE 0x3A
#define AUXOFF 0x3C
#define AUXDATA 0x3E

/* BAP selectors */
#define BAP0 0 // Used for receiving packets
#define BAP1 2 // Used for xmiting packets and working with RIDS

/* Flags */
#define COMMAND_BUSY 0x8000

#define BAP_BUSY 0x8000
#define BAP_ERR 0x4000
#define BAP_DONE 0x2000

#define PROMISC 0xffff

#define EV_CMD 0x10
#define EV_CLEARCOMMANDBUSY 0x4000
#define EV_RX 0x01
#define EV_TX 0x02
#define EV_TXEXC 0x04
#define EV_ALLOC 0x08
#define EV_LINK 0x80
#define EV_AWAKE 0x100
#define EV_UNKNOWN 0x800
#define EV_MIC        0x1000
#define STATUS_INTS ( EV_AWAKE | EV_LINK | EV_TXEXC | EV_TX | EV_RX | EV_MIC |EV_UNKNOWN)

/* The RIDs */
#define RID_CAPABILITIES 0xFF00
#define RID_CONFIG     0xFF10
#define RID_SSID       0xFF11
#define RID_APLIST     0xFF12
#define RID_DRVNAME    0xFF13
#define RID_ETHERENCAP 0xFF14
#define RID_WEP_TEMP   0xFF15
#define RID_WEP_PERM   0xFF16
#define RID_MODULATION 0xFF17
#define RID_ACTUALCONFIG 0xFF20 /*readonly*/
#define RID_LEAPUSERNAME 0xFF23
#define RID_LEAPPASSWORD 0xFF24
#define RID_STATUS     0xFF50
#define RID_STATS      0xFF68
#define RID_STATSDELTA 0xFF69
#define RID_STATSDELTACLEAR 0xFF6A
#define RID_MIC        0xFF57
/*
 * Rids and endian-ness:  The Rids will always be little endian, since
 * this is what the card wants.  So any assignments to are from the
 * rids need to be converted to the correct endian.
 */

/* This structure came from an email sent to me from an engineer at
   aironet for inclusion into this driver */
typedef struct {
  u16 len;
  u16 kindex;
  u8 mac[6];
  u16 klen;
  u8 key[16];  /* 40-bit keys */
} WepKeyRid;

/* These structures are from the Aironet's PC4500 Developers Manual */
typedef struct {
  u16 len;
  u8 ssid[32];
} Ssid;

typedef struct {
  u16 len;
  Ssid ssids[3];
} SsidRid;

typedef struct {
  u16 len;
  u16 modulation;
#define MOD_DEFAULT 0
#define MOD_CCK 1
#define MOD_MOK 2
} ModulationRid;

typedef struct {
  u16 cmd;
  u16 parm0;
  u16 parm1;
  u16 parm2;
} Cmd;

typedef struct {
  u16 status;
  u16 rsp0;
  u16 rsp1;
  u16 rsp2;
} Resp;

typedef struct {
  u16 len; /* sizeof(ConfigRid) */
  u16 opmode; /* operating mode */
#define MODE_STA_IBSS 0
#define MODE_STA_ESS 1
#define MODE_AP 2
#define MODE_AP_RPTR 3
#define MODE_ETHERNET_HOST (0<<8) /* rx payloads converted */
#define MODE_LLC_HOST (1<<8) /* rx payloads left as is */
#define MODE_AIRONET_EXTEND (1<<9) /* enable Aironet extenstions */
#define MODE_AP_INTERFACE (1<<10) /* enable ap interface extensions */
#define MODE_ANTENNA_ALIGN (1<<11) /* enable antenna alignment */
#define MODE_ETHER_LLC (1<<12) /* enable ethernet LLC */
#define MODE_LEAF_NODE (1<<13) /* enable leaf node bridge */
#define MODE_CF_POLLABLE (1<<14) /* enable CF pollable */
  u16 rmode; /* receive mode */
#define RXMODE_BC_MC_ADDR 0
#define RXMODE_BC_ADDR 1 /* ignore multicasts */
#define RXMODE_ADDR 2 /* ignore multicast and broadcast */
#define RXMODE_RFMON 3 /* wireless monitor mode */
#define RXMODE_RFMON_ANYBSS 4
#define RXMODE_LANMON 5 /* lan style monitor -- data packets only */
#define RXMODE_DISABLE_802_3_HEADER (1<<8) /* disables 802.3 header on rx */
#define RXMODE_NORMALIZED_RSSI (1<<9) /* return normalized RSSI */
  u16 fragThresh;
  u16 rtsThres;
  u8 macAddr[6];
  u8 rates[8];
  u16 shortRetryLimit;
  u16 longRetryLimit;
  u16 txLifetime; /* in kusec */
  u16 rxLifetime; /* in kusec */
  u16 stationary;
  u16 ordering;
  u16 u16deviceType; /* for overriding device type */
  u16 cfpRate;
  u16 cfpDuration;
  u16 _reserved1[3];
  /*---------- Scanning/Associating ----------*/
  u16 scanMode;
#define SCANMODE_ACTIVE 0
#define SCANMODE_PASSIVE 1
#define SCANMODE_AIROSCAN 2
  u16 probeDelay; /* in kusec */
  u16 probeEnergyTimeout; /* in kusec */
  u16 probeResponseTimeout;
  u16 beaconListenTimeout;
  u16 joinNetTimeout;
  u16 authTimeout;
  u16 authType;
#define AUTH_OPEN 0x1
#define AUTH_ENCRYPT 0x101
#define AUTH_SHAREDKEY 0x102
#define AUTH_ALLOW_UNENCRYPTED 0x200
  u16 associationTimeout;
  u16 specifiedApTimeout;
  u16 offlineScanInterval;
  u16 offlineScanDuration;
  u16 linkLossDelay;
  u16 maxBeaconLostTime;
  u16 refreshInterval;
#define DISABLE_REFRESH 0xFFFF
  u16 _reserved1a[1];
  /*---------- Power save operation ----------*/
  u16 powerSaveMode;
#define POWERSAVE_CAM 0
#define POWERSAVE_PSP 1
#define POWERSAVE_PSPCAM 2
  u16 sleepForDtims;
  u16 listenInterval;
  u16 fastListenInterval;
  u16 listenDecay;
  u16 fastListenDelay;
  u16 _reserved2[2];
  /*---------- Ap/Ibss config items ----------*/
  u16 beaconPeriod;
  u16 atimDuration;
  u16 hopPeriod;
  u16 channelSet;
  u16 channel;
  u16 dtimPeriod;
  u16 bridgeDistance;
  u16 radioID;
  /*---------- Radio configuration ----------*/
  u16 radioType;
#define RADIOTYPE_DEFAULT 0
#define RADIOTYPE_802_11 1
#define RADIOTYPE_LEGACY 2
  u8 rxDiversity;
  u8 txDiversity;
  u16 txPower;
#define TXPOWER_DEFAULT 0
  u16 rssiThreshold;
#define RSSI_DEFAULT 0
  u16 modulation;
  u16 shortPreamble;
  u16 homeProduct;
  u16 radioSpecific;
  /*---------- Aironet Extensions ----------*/
  u8 nodeName[16];
  u16 arlThreshold;
  u16 arlDecay;
  u16 arlDelay;
  u16 _reserved4[1];
  /*---------- Aironet Extensions ----------*/
  u16 magicAction;
#define MAGIC_ACTION_STSCHG 1
#define MACIC_ACTION_RESUME 2
#define MAGIC_IGNORE_MCAST (1<<8)
#define MAGIC_IGNORE_BCAST (1<<9)
#define MAGIC_SWITCH_TO_PSP (0<<10)
#define MAGIC_STAY_IN_CAM (1<<10)
  u16 magicControl;
  u16 autoWake;
} ConfigRid;

typedef struct {
  u16 len;
  u8 mac[6];
  u16 mode;
  u16 errorCode;
  u16 sigQuality;
  u16 SSIDlen;
  char SSID[32];
  char apName[16];
  char bssid[4][6];
  u16 beaconPeriod;
  u16 dimPeriod;
  u16 atimDuration;
  u16 hopPeriod;
  u16 channelSet;
  u16 channel;
  u16 hopsToBackbone;
  u16 apTotalLoad;
  u16 generatedLoad;
  u16 accumulatedArl;
  u16 signalQuality;
  u16 currentXmitRate;
  u16 apDevExtensions;
  u16 normalizedSignalStrength;
  u16 _reserved[10];
} StatusRid;

typedef struct {
  u16 len;
  u8 ap[4][6];
} APListRid;

typedef struct {
  u16 len;
  char oui[3];
  u16 prodNum;
  char manName[32];
  char prodName[16];
  char prodVer[8];
  char factoryAddr[6];
  char aironetAddr[6];
  u16 radioType;
  u16 country;
  char callid[6];
  char supportedRates[8];
  char rxDiversity;
  char txDiversity;
  u16 txPowerLevels[8];
  u16 hardVer;
  u16 hardCap;
  u16 tempRange;
  u16 softVer;
  u16 softSubVer;
  u16 interfaceVer;
  u16 softCap;
  u16 bootBlockVer;
  u16 requiredHard;
} CapabilityRid;


#define TRUE          1
#define FALSE         0
#define NUM_MODULES   2
#define MIC_MSGLEN_MAX		2000
#define EMMH32_MSGLEN_MAX	MIC_MSGLEN_MAX
#define ETHERNET_ADDRESS_LENGTH 6
#define ETHERNET_HEADER_SIZE    14    /* Size of the ethernet header */
#define CFG_OPMODE_MIC          0x8000
#define EXT_SOFT_CAPS_MIC       0x0001

/* 
 * MIC specific flag bits in airo_info->flags
 */
#define MIC_CAPABLE 8       /* Can do mic    */
#define MIC_ENABLED 16      /* Mic turned on */
#define ADHOC	    32      /* In addhoc     */
#define FLASHING    64      /* Flashing      */   

typedef enum {
    NONE,
    NOMIC,
    NOMICPLUMMED,
    SEQUENCE,
    INCORRECTMIC,
} MIC_ERROR;


/*
 * MIC rid 
 */

typedef struct _STMIC {
    unsigned short ridLen __attribute__ ((packed));
    unsigned short micState __attribute__ ((packed));
    unsigned short micMulticastValid __attribute__ ((packed));
    unsigned char  micMulticast[16] __attribute__ ((packed));
    unsigned short micUnicastValid __attribute__ ((packed));
    unsigned char  micUnicast[16] __attribute__ ((packed));
} STMIC __attribute__ ((packed));

typedef struct _STMICSTATISTICS {
  u32   Size;             // size
  u8    Enabled;          // MIC enabled or not
  u32   RxSuccess;        // successful packets received
  u32   RxIncorrectMIC;   // packets dropped due to incorrect MIC comparison
  u32   RxNotMICed;       // packets dropped due to not being MIC'd
  u32   RxMICPlummed;     // packets dropped due to not having a MIC plummed
  u32   RxWrongSequence;  // packets dropped due to sequence number violation
  u32   Reserve[32];
}STMICSTATISTICS32;

typedef struct {
  unsigned long	coeff[((EMMH32_MSGLEN_MAX)+3)>>2];
  unsigned long long	accum;	/* accumulated mic, reduced to u32 in final() */
  int	 position;		/* current position (byte offset) in message */
  union	{
    unsigned char	d8[4];
    unsigned long	d32;
  } part;		/* saves partial message word across update() calls */
} emmh32_context;

typedef struct _MIC_CNTX {
  emmh32_context      seed __attribute__ ((packed));     //Context - "the seed"
  u32                 rx __attribute__ ((packed)) ;      //Received sequence numbers
  u32                 window __attribute__ ((packed));   //start of window
  u32                 tx __attribute__ ((packed)) ;       //Tx sequence number
  u8                  multicast __attribute__ ((packed));// Flag to say if it is mulitcast or not
  u8                  valid __attribute__ ((packed));    // Flag to say if context is valid or not
  u8                  key[16] __attribute__ ((packed)) ;    
}MICCNTX __attribute__ ((packed));

typedef struct _MIC_MODULE {
  MICCNTX             mCtx;      //Multicast context
  MICCNTX             uCtx;      //Unicast context
} MICMODULE __attribute__ ((packed));

typedef struct _MIC_BUFFER_STRUCT {
  u8  DA[ETHERNET_ADDRESS_LENGTH];
  u8  SA[ETHERNET_ADDRESS_LENGTH];
  u16 TypeLength __attribute__ ((packed));
  
  union {
    u8      snap[8];
    struct {
      u8          dsap;
      u8          ssap;
      u8          control;
      u8          orgCode[3];
      u8          fieldType[2];
    } llc;
  }u  __attribute__ ((packed));

  u32         MIC __attribute__ ((packed));
  u32         SEQ __attribute__ ((packed));
  u8          payload __attribute__ ((packed)); 
}MIC_BUFFER_STRUCT __attribute__ ((packed));

typedef struct _ETH_HEADER_STRUC {
  u8       Destination[ETHERNET_ADDRESS_LENGTH]__attribute__ ((packed));
  u8       Source[ETHERNET_ADDRESS_LENGTH] __attribute__ ((packed));
  u16       TypeLength __attribute__ ((packed));
} ETH_HEADER_STRUC __attribute__ ((packed)) ;


typedef struct _EXSTCAPS{
	u16		u16RidLen;			//0x0000
	u8		au8OUI[3];			//0x0002
	u8		nothing;			//0x0005
	u16		ProuctNum;			//0x0006
	u8		au8ManufacturerName[32];	//0x0008
	u8		au8ProductName[16];		//0x0028
	u8		au8ProductVersion[8];		//0x0038
	u8		au8FactoryAddress[6];		//0x0040
	u8		au8AironetAddress[6];		//0x0046
	u16		u16RadioType;			//0x004C
	u16		u16RegDomain;			//0x004E
	u8		au8Callid[6];			//0x0050
	u8		au8SupportedRates[8];		//0x0056
	u8		u8RxDiversity;			//0x005E
	u8		u8TxDiversity;			//0x005F
	u16		au16TxPowerLevels[8];		//0x0060
	u16		u16HardwareVersion;		//0x0070
	u16		u16HardwareCapabilities;	//0x0072
	u16		u16TemperatureRange;		//0x0074
	u16		u16SoftwareVersion;		//0x0076
	u16		u16SoftwareSubVersion;		//0x0078
	u16		u16InterfaceVersion	;	//0x007A
	u16		u16SoftwareCapabilities;	//0x007C
	u16		u16BootBlockVersion;		//0x007E
	u16		u16SupportedHardwareRev;	//0x0080
        u16             u16ExtSoftwareCapabilities;     //0x0082
}EXSTCAPS;

#define TXCTL_TXOK (1<<1) /* report if tx is ok */
#define TXCTL_TXEX (1<<2) /* report if tx fails */
#define TXCTL_802_3 (0<<3) /* 802.3 packet */
#define TXCTL_802_11 (1<<3) /* 802.11 mac packet */
#define TXCTL_ETHERNET (0<<4) /* payload has ethertype */
#define TXCTL_LLC (1<<4) /* payload is llc */
#define TXCTL_RELEASE (0<<5) /* release after completion */
#define TXCTL_NORELEASE (1<<5) /* on completion returns to host */

#define BUSY_FID 0x10000

#ifdef CISCO_EXT
#define AIROMAGIC	0xa55a
#define AIROIOCTL	SIOCDEVPRIVATE
#define AIROIDIFC 	AIROIOCTL + 1

/* Ioctl constants to be used in airo_ioctl.command */

#define	AIROGCAP  		0	// Capability rid
#define AIROGCFG		1       // USED A LOT 
#define AIROGSLIST		2	// System ID list 
#define AIROGVLIST		3       // List of specified AP's
#define AIROGDRVNAM		4	//  NOTUSED
#define AIROGEHTENC		5	// NOTUSED
#define AIROGWEPKTMP		6
#define AIROGWEPKNV		7
#define AIROGSTAT		8
#define AIROGSTATSC32		9
#define AIROGSTATSD32		10
#define AIROGMICRID             11      /* Get STMIC rid      */
#define AIROGMICSTATS           12      /* Get MIC statistics */
#define AIROGFLAGS              13      /* Get driver flags   */
#define AIRORRID                15      /* Read RID */
#define AIRORSWVERSION          17      /* Get driver version   */


/* Leave gap of 40 commands after AIROGSTATSD32 for future */

#define AIROPCAP               	AIROGSTATSD32 + 40
#define AIROPVLIST              AIROPCAP      + 1
#define AIROPSLIST		AIROPVLIST    + 1
#define AIROPCFG		AIROPSLIST    + 1
#define AIROPSIDS		AIROPCFG      + 1
#define AIROPAPLIST		AIROPSIDS     + 1
#define AIROPMACON		AIROPAPLIST   + 1	/* Enable mac  */
#define AIROPMACOFF		AIROPMACON    + 1 	/* Disable mac */
#define AIROPSTCLR		AIROPMACOFF   + 1
#define AIROPWEPKEY		AIROPSTCLR    + 1
#define AIROPWEPKEYNV		AIROPWEPKEY   + 1
#define AIROPLEAPPWD            AIROPWEPKEYNV + 1
#define AIROPLEAPUSR            AIROPLEAPPWD  + 1

/* Flash codes */

#define AIROFLSHRST	       AIROPWEPKEYNV  + 40
#define AIROFLSHGCHR           AIROFLSHRST    + 1
#define AIROFLSHSTFL           AIROFLSHGCHR   + 1
#define AIROFLSHPCHR           AIROFLSHSTFL   + 1
#define AIROFLPUTBUF           AIROFLSHPCHR   + 1
#define AIRORESTART            AIROFLPUTBUF   + 1

#define FLASHSIZE	32768

typedef struct aironet_ioctl {
  unsigned short command;	// What to do
  unsigned short length;	// Len of data
  unsigned short ridnum;	// rid number
  unsigned char *data;		// d-data
} aironet_ioctl;
#endif 

#ifdef WIRELESS_EXT
// Frequency list (map channels to frequencies)
const long frequency_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				2447, 2452, 2457, 2462, 2467, 2472, 2484 };

// A few details needed for WEP (Wireless Equivalent Privacy)
#define MAX_KEY_SIZE 13			// 128 (?) bits
#define MIN_KEY_SIZE  5			// 40 bits RC4 - WEP
#define MAX_KEYS      4			// 4 different keys
typedef struct wep_key_t {
	u16	len;
	u8	key[16];	/* 40-bit and 104-bit keys */
} wep_key_t;
#endif 

static char *version =
"airo.c 2.1 2001/01/22 (Benjamin Reed ) + MIC ";
static char *swversion = "2.1";

struct airo_info;
/* MIC routines */

void emmh32_setseed(emmh32_context *, unsigned char *, int );
void emmh32_init(emmh32_context *context);
void emmh32_update(emmh32_context *context, unsigned char *pOctets, int len);
void emmh32_final(emmh32_context *context, unsigned char digest[4]);
static void micinit(struct airo_info *, STMIC *);
static void micsetup(struct airo_info *);
static int RxSeqValid (struct airo_info *,MICCNTX *context, u32 micSeq);
static void MoveWindow(MICCNTX *, u32 );
static int Encapsulate(struct airo_info *,ETH_HEADER_STRUC *, u32 *,int );
static int Decapsulate(struct airo_info *,ETH_HEADER_STRUC *, u32 *);
static void UshortByteSwap(u16 *);
static void UlongByteSwap(u32 *);
static void airo_linkstat(struct airo_info *);


static int get_dec_u16( char *buffer, int *start, int limit );
static void OUT4500( struct airo_info *, u16 register, u16 value );
static unsigned short IN4500( struct airo_info *, u16 register );
static u16 setup_card(struct airo_info*, u8 *mac, ConfigRid *);
static void enable_interrupts(struct airo_info*);
static void disable_interrupts(struct airo_info*);
static u16 issuecommand(struct airo_info*, Cmd *pCmd, Resp *pRsp);
static int bap_setup(struct airo_info*, u16 rid, u16 offset, int whichbap);
static int aux_bap_read(struct airo_info*, u16 *pu16Dst, int bytelen, 
			int whichbap);
static int fast_bap_read(struct airo_info*, u16 *pu16Dst, int bytelen, 
			 int whichbap);
static int bap_write(struct airo_info*, const u16 *pu16Src, int bytelen,
		     int whichbap);
static int PC4500_accessrid(struct airo_info*, u16 rid, u16 accmd);

static int PC4500_readrid(struct airo_info*, u16 rid, void *pBuf, int len);

static int PC4500_writerid(struct airo_info*, u16 rid, const void
			   *pBuf, int len);
static int do_writerid( struct airo_info*, u16 rid, const void *rid_data, 
			int len );
static u16 transmit_allocate(struct airo_info*, int lenPayload);
static int transmit_802_3_packet(struct airo_info*, u16 TxFid, char
				 *pPacket, int len);

static void airo_interrupt( int irq, void* dev_id, struct pt_regs
			    *regs);
static int airo_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
#ifdef WIRELESS_EXT
struct iw_statistics *airo_get_wireless_stats (struct net_device *dev);
#endif 
#ifdef CISCO_EXT
static int readrids(struct net_device *dev, aironet_ioctl *comp);
static int writerids(struct net_device *dev, aironet_ioctl *comp);
int flashcard(struct net_device *dev, aironet_ioctl *comp);
#endif 
#ifdef INCLUDE_RFMONITOR
static void set_rfmonitor(struct airo_info *ai);
#endif

#ifdef WIRELESS_EXT
static int getQuality(StatusRid *statusRid, CapabilityRid *capsRid);
#endif

struct airo_info {
  struct net_device_stats	stats;
  int open;
#if (LINUX_VERSION_CODE < 0x020363)
  char name[8];
#endif
  struct net_device             *dev;
  /* Note, we can have MAX_FIDS outstanding.  FIDs are 16-bits, so we
   *     use the high bit to mark wether it is in use. 
   */
#define MAX_FIDS 6
  int                           fids[MAX_FIDS];
  int registered;
  ConfigRid config;
  /* MIC stuff         */
  MICMODULE       mod[2];           
  u8              snap[8];
  u32             updateMultiSeq;
  u32             updateUniSeq;
  STMICSTATISTICS32 micstats;
  u8 *micrxbuf;
  u8 *mictxbuf;
  char defindex; // Needed by proc wep interface. 
#if (LINUX_VERSION_CODE < 0x20311)
  struct proc_dir_entry proc_entry;
  struct proc_dir_entry proc_statsdelta_entry;
  struct proc_dir_entry proc_stats_entry;
  struct proc_dir_entry proc_status_entry;
  struct proc_dir_entry proc_config_entry;
  struct proc_dir_entry proc_SSID_entry;
  struct proc_dir_entry proc_APList_entry;
  struct proc_dir_entry proc_wepkey_entry;
#else
  struct proc_dir_entry *proc_entry;
#endif
  struct airo_info *next;
  spinlock_t bap0_lock;
  spinlock_t bap1_lock;
  spinlock_t aux_lock;
  spinlock_t cmd_lock;
  u32 flags;
#define FLAG_PROMISC   0x01
#define FLAG_RADIO_OFF 0x02
#define FLAG_RFMONITOR 0x04

  int (*bap_read)(struct airo_info*, u16 *pu16Dst, int bytelen, 
		  int whichbap);
#ifdef WIRELESS_EXT
  int			need_commit;	// Need to set config
  struct iw_statistics	wstats;		// wireless stats
#ifdef WIRELESS_SPY
  int			spy_number;
  u_char			spy_address[IW_MAX_SPY][6];
  struct iw_quality	spy_stat[IW_MAX_SPY];
#endif 
#endif 
};

static inline int bap_read(struct airo_info *ai, u16 *pu16Dst, int bytelen, 
			   int whichbap) {
	return ai->bap_read(ai, pu16Dst, bytelen, whichbap);
}

static int setup_proc_entry( struct net_device *dev,
			     struct airo_info *apriv );
static int takedown_proc_entry( struct net_device *dev,
				struct airo_info *apriv );


static int airo_open(struct net_device *dev) {
  struct airo_info *info = dev->priv;

  if(info->flags & FLASHING ) /* Cant touch this */
    return -EIO;

  MOD_INC_USE_COUNT;
  if ( info->open == 0 ) {
    enable_interrupts(info);
  }
  info->open++;
  
  netif_start_queue(dev);
  netif_mark_up(dev);
  return 0;
}

static int airo_start_xmit(struct sk_buff *skb, struct net_device *dev) {
  s16 len;
  s16 retval = 0;
  u16 status;
  u32 flags;
  s8 *buffer;
  int i;
  struct airo_info *priv = (struct airo_info*)dev->priv;
  u32 *fids = priv->fids;
  
  if ( skb == NULL ) {
    printk( KERN_ERR "airo:  skb == NULL!!!\n" );
    return 0;
  }
	
  /* Find a vacant FID */
  spin_lock_irqsave(&priv->bap1_lock, flags);
  for( i = 0; i < MAX_FIDS; i++ ) {
    if ( !( fids[i] & 0xffff0000 ) ) break;
  }
  if ( i == MAX_FIDS ) {
    printk( KERN_ERR "airo:  can't find FID!\n" );
    netif_stop_queue(dev);
    retval = -EBUSY;
    goto tx_done;
  }
  
  len = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN; /* check min length*/
  buffer = skb->data;
  
  status = transmit_802_3_packet( priv, 
				  fids[i],
				  skb->data, len );
  
  if ( status == SUCCESS ) {
    /* Mark fid as used & save length for later */
    fids[i] |= (len << 16); 
		dev->trans_start = jiffies;
  } else {
	  ((struct airo_info*)dev->priv)->stats.tx_errors++;
	}
 tx_done:
  spin_unlock_irqrestore(&priv->bap1_lock, flags);
  KFREE_SKB( skb, FREE_WRITE );
  return 0;
}

static struct net_device_stats *airo_get_stats(struct net_device *dev) {
  return &(((struct airo_info*)dev->priv)->stats);
}

static int enable_MAC( struct airo_info *ai, Resp *rsp );
static void disable_MAC(struct airo_info *ai);

static void airo_set_multicast_list(struct net_device *dev) {
  struct airo_info *ai = (struct airo_info*)dev->priv;
  Resp rsp;
  
  ai->flags &= ~FLAG_PROMISC;
  if (dev->flags&IFF_PROMISC) {
    ai->flags |= FLAG_PROMISC;
  } else if ((dev->flags&IFF_ALLMULTI)||dev->mc_count>0) {
    /* Turn on multicast.  (Should be already setup...) */
  }
  disable_MAC(ai);
  enable_MAC(ai, &rsp);
}

static int airo_change_mtu(struct net_device *dev, int new_mtu)
{
  if ((new_mtu < 68) || (new_mtu > 2400))
    return -EINVAL;
  dev->mtu = new_mtu;
  return 0;
}


static int airo_close(struct net_device *dev) { 
  struct airo_info *ai = (struct airo_info*)dev->priv;
  ai->open--;
  netif_stop_queue(dev);
  netif_mark_down(dev);
  if ( !ai->open ) {
    disable_interrupts( ai );
  }
  MOD_DEC_USE_COUNT;
  return 0;
}

static void del_airo_dev( struct net_device *dev );

/* 
 * Dont free things twice 
 */
void stop_airo_card( struct net_device *dev, int freeres ) 
{
  struct airo_info *ai = (struct airo_info*)dev->priv;
  takedown_proc_entry( dev, ai );
  if (ai->registered) {
    unregister_netdev( dev );
    ai->registered = 0;
  }
  disable_interrupts(ai);
  free_irq( dev->irq, dev );
  if (freeres) {
    /* PCMCIA frees this stuff, so only for PCI and ISA */
    release_region( dev->base_addr, 64 );
  }
  kfree(ai->mictxbuf);
  kfree(ai->micrxbuf);
  del_airo_dev( dev );
  kfree( dev );
}

static void add_airo_dev( struct net_device *dev ); 

struct net_device *init_airo_card( unsigned short irq, int port )
{
  struct net_device *dev;
  struct airo_info *ai;
  int i;
	
  /* Create the network device object. */
  dev = init_etherdev(0, sizeof(*ai));
  if (!dev) {
    printk(KERN_ERR "airo:  Couldn't init_etherdev\n");
    return NULL;
  }
  ai = (struct airo_info *)dev->priv;
  ai->registered = 1;
  ai->dev = dev;
  ai->bap0_lock = SPIN_LOCK_UNLOCKED;
  ai->bap1_lock = SPIN_LOCK_UNLOCKED;
  ai->aux_lock = SPIN_LOCK_UNLOCKED;
  ai->cmd_lock = SPIN_LOCK_UNLOCKED;
  ai->micrxbuf = kmalloc(2048, GFP_KERNEL);
  ai->mictxbuf = kmalloc(2048, GFP_KERNEL);

  add_airo_dev( dev ); 
  
  /* The Airo-specific entries in the device structure. */
  dev->hard_start_xmit = &airo_start_xmit;
  dev->get_stats = &airo_get_stats;
  dev->set_multicast_list = &airo_set_multicast_list;
  dev->do_ioctl = &airo_ioctl;
#ifdef WIRELESS_EXT
  dev->get_wireless_stats = airo_get_wireless_stats;
#endif 
  dev->change_mtu = &airo_change_mtu;
  dev->open = &airo_open;
  dev->stop = &airo_close;
  netif_stop_queue(dev);
  dev->irq = irq;
  dev->base_addr = port;
	
  if ( request_irq( dev->irq, airo_interrupt, 
		    SA_SHIRQ | SA_INTERRUPT, dev->name, dev ) ) {
    printk(KERN_ERR "airo: register interrupt %d failed\n", irq );
  }
  request_region( dev->base_addr, 64, dev->name );
  
  memset( &((struct airo_info*)dev->priv)->config, 0,
	  sizeof(((struct airo_info*)dev->priv)->config));
  
  if ( setup_card( ai, dev->dev_addr, 
		   &((struct airo_info*)dev->priv)->config) 
       != SUCCESS ) {
    printk( KERN_ERR "airo: MAC could not be enabled\n" );
  } else {
    printk( KERN_INFO "airo: MAC enabled %s %x:%x:%x:%x:%x:%x\n",
	    dev->name,
	    dev->dev_addr[0],
	    dev->dev_addr[1],
	    dev->dev_addr[2],
	    dev->dev_addr[3],
	    dev->dev_addr[4],
	    dev->dev_addr[5]
	    );
  }
  /* Allocate the transmit buffers */
  for( i = 0; i < MAX_FIDS; i++ ) {
    ai->fids[i] = transmit_allocate( ai, 2000 );
  }
  
  setup_proc_entry( dev, (struct airo_info*)dev->priv );
  netif_start_queue(dev);

  return dev;
}

int reset_airo_card( struct net_device *dev ) {
	int i;
	struct airo_info *ai = (struct airo_info*)dev->priv;

	if ( setup_card(ai, dev->dev_addr,
			&(ai)->config) != SUCCESS ) {
		printk( KERN_ERR "airo: MAC could not be enabled\n" );
		return -1;
	} else {
		printk( KERN_INFO "airo: MAC enabled %s %x:%x:%x:%x:%x:%x\n",
			dev->name,
			dev->dev_addr[0],
			dev->dev_addr[1],
			dev->dev_addr[2],
			dev->dev_addr[3],
			dev->dev_addr[4],
			dev->dev_addr[5]
			);
		/* Allocate the transmit buffers */
		for( i = 0; i < MAX_FIDS; i++ ) {
			((struct airo_info*)dev->priv)->fids[i] =
				transmit_allocate( ai, 2000 );
		}
	}
	enable_interrupts( ai );
	netif_start_queue(dev);
	return 0;
}

static void airo_linkstat(struct airo_info *ai){
  /* Update sequence numbers 
   * upon assoc 
   */
  ai->updateMultiSeq = TRUE;     
  ai->updateUniSeq = TRUE;
}


#define DEBUG_MIC

static void airo_interrupt ( int irq, void* dev_id, struct pt_regs *regs) {
  struct net_device *dev = (struct net_device *)dev_id;
  u16 len;
  u16 status;
  u16 fid;
  u32 miclen;
  struct airo_info *apriv = (struct airo_info *)dev->priv;
  u16 savedInterrupts;
  STMIC  mic; 
#ifdef INCLUDE_RFMONITOR
  char *buffer;
#endif

  if (!netif_device_present(dev))
    return;

  status = IN4500( apriv, EVSTAT );
  if ( !status ) return;
  
  if ( status & EV_AWAKE ) {
    OUT4500( apriv, EVACK, EV_AWAKE );
    OUT4500( apriv, EVACK, EV_AWAKE );
  }
  savedInterrupts = IN4500( apriv, EVINTEN );
  OUT4500( apriv, EVINTEN, 0 );
 
  if( status & EV_MIC){
#ifdef DEBUG_MIC
    printk(KERN_INFO "MIC interrupt\n");
#endif
    OUT4500( apriv, EVACK, EV_MIC ); /* Must ack MIC first */
    PC4500_readrid((struct airo_info *)dev->priv,RID_MIC,(char *)&mic,sizeof(mic));
    micinit(apriv,&mic);
    OUT4500( apriv, EVINTEN, savedInterrupts );
    return;
  }

  if ( status & EV_LINK ) {
    /* The link status has changed, if you want to put a
       monitor hook in, do it here.  (Remember that
       interrupts are still disabled!)
    */
    u16 newStatus = IN4500(apriv, LINKSTAT);
    /* Here is what newStatus means: */
#define NOBEACON 0x8000 /* Loss of sync - missed beacons */
#define MAXRETRIES 0x8001 /* Loss of sync - max retries */
#define MAXARL 0x8002 /* Loss of sync - average retry level exceeded*/
#define FORCELOSS 0x8003 /* Loss of sync - host request */
#define TSFSYNC 0x8004 /* Loss of sync - TSF synchronization */
#define DEAUTH 0x8100 /* Deauthentication (low byte is reason code) */
#define DISASS 0x8200 /* Disassociation (low byte is reason code) */
#define ASSFAIL 0x8400 /* Association failure (low byte is reason
			  code) */
#define AUTHFAIL 0x0300 /* Authentication failure (low byte is reason
			   code) */
#define ASSOCIATED 0x0400 /* Assocatied */
#define RC_RESERVED 0 /* Reserved return code */
#define RC_NOREASON 1 /* Unspecified reason */
#define RC_AUTHINV 2 /* Previous authentication invalid */
#define RC_DEAUTH 3 /* Deauthenticated because sending station is
		       leaving */
#define RC_NOACT 4 /* Disassociated due to inactivity */
#define RC_MAXLOAD 5 /* Disassociated because AP is unable to handle
			all currently associated stations */
#define RC_BADCLASS2 6 /* Class 2 frame received from
			  non-Authenticated station */
#define RC_BADCLASS3 7 /* Class 3 frame received from
			  non-Associated station */
#define RC_STATLEAVE 8 /* Disassociated because sending station is
			  leaving BSS */
#define RC_NOAUTH 9 /* Station requesting (Re)Association is not
		       Authenticated with the responding station */
    if((newStatus & 0x8400) == 0x400)
      airo_linkstat(apriv );
  }
	
  /* Check to see if there is something to recieve */
  if ( status & EV_RX  ) {
    struct sk_buff *skb;
    int flags;
    
    fid = IN4500( apriv, RXFID );
    
    /* Get the packet length */
    spin_lock_irqsave(&apriv->bap0_lock, flags);
#ifdef INCLUDE_RFMONITOR
    if (apriv->flags&FLAG_RFMONITOR) {
      bap_setup( apriv, fid, 0, BAP0 );
      bap_read( apriv, (u16 *)apriv->micrxbuf, 50, BAP0 );
      bap_read( apriv, &len, sizeof(len), BAP0 );
      len = le16_to_cpu(len);
      if (len) {
        bap_read( apriv, (u16 *)(apriv->micrxbuf+50), len, BAP0 );
      }
      len = *(u16*)((u16*)(apriv->micrxbuf)+3);
      len = le16_to_cpu(len);
      if (apriv->micrxbuf[4] & 2) { /* CRC error */
        len = 0;
      }
      if (len && len < 2312) {
        bap_read( apriv, (u16*)(apriv->micrxbuf+50), len, BAP0 );
        len += 30;
        if (len && len < 2132) {
          skb = dev_alloc_skb( len );
          buffer = skb_put( skb, len );
          memcpy(buffer,apriv->micrxbuf+20,len);
          skb->mac.raw = skb->data;
          skb->pkt_type = PACKET_OTHERHOST;
          skb->protocol = htons(ETH_P_802_2);
          skb->dev = dev;
          skb->ip_summed = CHECKSUM_NONE;
          netif_rx( skb );
        }
      }
    } else
#endif
    {
    bap_setup( apriv, fid, 0x36, BAP0 );
    bap_read( apriv, &len, sizeof(len), BAP0 );    
    len = le16_to_cpu(len);

    /* The length only counts the payload
       not the hw addresses */
    len += 12; 
    if ( len < 12 || len > 2048 ) {
#if LINUX_VERSION_CODE > 0x20127
      apriv->stats.rx_length_errors++;
#endif
      apriv->stats.rx_errors++;
      printk( KERN_ERR "airo: Bad size %d\n", len );
    } 
    else 
      {
	skb = dev_alloc_skb( len + 2 );
	if ( !skb ) {
	  apriv->stats.rx_dropped++;
	} 
	else 
	  {
	    char *buffer;
	    buffer = skb_put( skb, len );
	    bap_setup( apriv, fid, 0x36+sizeof(len), BAP0 );
	    bap_read( apriv, (u16 *)apriv->micrxbuf, len, BAP0 );
	    miclen = len;
	    /* 
	     * Demic
	     */
	    if(Decapsulate(apriv ,(ETH_HEADER_STRUC *)apriv->micrxbuf,&miclen )){
#ifdef DEBUG_MIC
	      printk(KERN_ERR "Dumping packet from decap\n");
#endif
	      dev_kfree_skb_irq(skb);
	      goto unlock_bap;
	    }   
	    else
	      memcpy(buffer,apriv->micrxbuf,miclen); 

#ifdef WIRELESS_SPY
	if (apriv->spy_number > 0) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(1,3,0))
	  char *srcaddr = skb->mac.raw + 6;
#else
	  char *srcaddr = skb->data + MAX_ADDR_SIZE;
#endif
	  u16 rssi;
	  int i;
	  

	  for (i=0; i<apriv->spy_number; i++)
	    if (!memcmp(srcaddr, apriv->spy_address[i], 6))
	      {
		/* Get the rssi signal strength */
		bap_setup( apriv, fid, 0x08, BAP0 );
		bap_read( apriv, &rssi, sizeof(rssi), BAP0 );
		apriv->spy_stat[i].qual = 0;
		apriv->spy_stat[i].level = le16_to_cpu(rssi);
		apriv->spy_stat[i].noise = 0;
		apriv->spy_stat[i].updated = 2;
		break;
	      }
	}
#endif 
	apriv->stats.rx_packets++;
#if LINUX_VERSION_CODE > 0x20127
	apriv->stats.rx_bytes += len;
#endif
	skb->dev = dev;
	skb->ip_summed = CHECKSUM_NONE;
	skb->protocol = eth_type_trans( skb, dev );
	
	netif_rx( skb );
      }
    }
    }
  unlock_bap:
    spin_unlock_irqrestore(&apriv->bap0_lock, flags);
  }

  /* Check to see if a packet has been transmitted */
  if (  status & ( EV_TX|EV_TXEXC ) ) {
    int i;
    int len = 0;
    int full = 1;
    int index = -1;
    
    fid = IN4500(apriv, TXCOMPLFID);
    
    for( i = 0; i < MAX_FIDS; i++ ) {
      if (!(apriv->fids[i] & 0xffff0000)) full = 0;
      if ( ( apriv->fids[i] & 0xffff ) == fid ) {
	len = apriv->fids[i] >> 16;
	index = i;
	/* Set up to be used again */
	apriv->fids[i] &= 0xffff; 
      }
    }
    if (full) netif_wake_queue(dev);
    if (index==-1) {
      printk( KERN_ERR "airo: Unallocated FID was used to xmit\n" );
    }
    if ( status & EV_TX ) {
      apriv->stats.tx_packets++;
#if LINUX_VERSION_CODE > 0x20127
      if(index!=-1)
	apriv->stats.tx_bytes += len;
#endif
    } else {
      apriv->stats.tx_errors++;
    }
  }
  /* MIC was setting this off */
  if ( status & ~STATUS_INTS ) 
    printk( KERN_WARNING 
	    "airo: Got weird status %x\n", 
	    status & ~STATUS_INTS );
  OUT4500( apriv, EVACK, status & STATUS_INTS );
  OUT4500( apriv, EVINTEN, savedInterrupts );
  
  /* done.. */
  return;     
}

/*
 *  Routines to talk to the card
 */

/*
 *  This was originally written for the 4500, hence the name
 *  NOTE:  If use with 8bit mode and SMP bad things will happen!
 *         Why would some one do 8 bit IO in an SMP machine?!?
 */
static void OUT4500( struct airo_info *ai, u16 reg, u16 val ) {
  val = cpu_to_le16(val);
  if ( !do8bitIO )
    outw( val, ai->dev->base_addr + reg );
  else {
    outb( val & 0xff, ai->dev->base_addr + reg );
    outb( val >> 8, ai->dev->base_addr + reg + 1 );
  }
}

static u16 IN4500( struct airo_info *ai, u16 reg ) {
  unsigned short rc;
  
  if ( !do8bitIO )
    rc = inw( ai->dev->base_addr + reg );
  else {
    rc = inb( ai->dev->base_addr + reg );
    rc += ((int)inb( ai->dev->base_addr + reg + 1 )) << 8;
  }
  return le16_to_cpu(rc);
}

#ifdef INCLUDE_RFMONITOR
static void set_rfmonitor(struct airo_info *ai)
{
  ConfigRid configRid;
  SsidRid ssidRid;
  Cmd cmd;
  Resp rsp;
  int status;

  memset(&cmd, 0, sizeof(cmd));
  cmd.cmd = MAC_DISABLE;
  status = issuecommand(ai, &cmd, &rsp);
  if (ai->flags&FLAG_RFMONITOR) {
    memset(&ssidRid, 0, sizeof(SsidRid));
    ssidRid.ssids[0].len = 3;
    memcpy(ssidRid.ssids[0].ssid, "ANY", 3);
    ssidRid.len = sizeof(SsidRid);
    PC4500_writerid(ai, 0xff11, &ssidRid, sizeof(SsidRid));
    PC4500_readrid(ai, 0xff21, &configRid, sizeof(ConfigRid));
    configRid.rmode = 0x300 | RXMODE_RFMON_ANYBSS;
    configRid.opmode = 0x102;
    configRid.atimDuration = 5;
    configRid.refreshInterval = 0xffff;
    configRid.authType = AUTH_OPEN;
    PC4500_writerid(ai, 0xff10, &configRid, sizeof(ConfigRid));
  } else {
    PC4500_readrid(ai, 0xff21, &configRid, sizeof(ConfigRid));
    PC4500_writerid(ai, 0xff10, &configRid, sizeof(ConfigRid));
  }
  memset(&cmd, 0, sizeof(cmd));
  cmd.cmd = MAC_ENABLE;
  status = issuecommand(ai, &cmd, &rsp);
}
#endif


static int enable_MAC( struct airo_info *ai, Resp *rsp ) {
  Cmd cmd;
  int status;
  
  if (ai->flags&FLAG_RADIO_OFF) return SUCCESS;
  memset(&cmd, 0, sizeof(cmd));
  cmd.cmd = MAC_ENABLE; // disable in case already enabled
  status = issuecommand(ai, &cmd, rsp);
  if (status == SUCCESS && ai->flags&FLAG_PROMISC) {
    memset(&cmd, 0, sizeof(cmd));
    cmd.cmd=CMD_SETMODE;
    cmd.parm1=PROMISC;
    issuecommand(ai, &cmd, rsp);
  }
  return status;
}

static void disable_MAC( struct airo_info *ai ) {
  Cmd cmd;
  Resp rsp;

  memset(&cmd, 0, sizeof(cmd));
  cmd.cmd = MAC_DISABLE; // disable in case already enabled
  issuecommand(ai, &cmd, &rsp);
}

static void enable_interrupts( struct airo_info *ai ) {
  /* Reset the status register */
  u16 status = IN4500( ai, EVSTAT );
  OUT4500( ai, EVACK, status );
  /* Enable the interrupts */
  OUT4500( ai, EVINTEN, STATUS_INTS );
  /* Note there is a race condition between the last two lines that
   *   I dont know how to get rid of right now... 
   */
}

static void disable_interrupts( struct airo_info *ai ) {
  OUT4500( ai, EVINTEN, 0 );
}

static u16 setup_card(struct airo_info *ai, u8 *mac,ConfigRid *config)
{
  EXSTCAPS  xcaps;
  Cmd cmd; 
  Resp rsp;
  ConfigRid cfg;
  int status;
  int i;
  SsidRid mySsid;

  memset( &mySsid, 0, sizeof( mySsid ) );
  
  /* The NOP is the first step in getting the card going */
  cmd.cmd = NOP;
  cmd.parm0 = cmd.parm1 = cmd.parm2 = 0;
  if ( issuecommand( ai, &cmd, &rsp ) != SUCCESS ) {
    return ERROR;
  }
  memset(&cmd, 0, sizeof(cmd));
  cmd.cmd = MAC_DISABLE; // disable in case already enabled
  if ( issuecommand( ai, &cmd, &rsp ) != SUCCESS ) {
    return ERROR;
  }
  
  // Let's figure out if we need to use the AUX port
  cmd.cmd = CMD_ENABLEAUX;
  if (issuecommand(ai, &cmd, &rsp) != SUCCESS) {
    printk(KERN_ERR "airo: Error checking for AUX port\n");
    return ERROR;
  }
  if (!aux_bap || rsp.status & 0xff00) {
    ai->bap_read = fast_bap_read;
    printk(KERN_DEBUG "airo: Doing fast bap_reads\n");
  } else {
    ai->bap_read = aux_bap_read;
    printk(KERN_DEBUG "airo: Doing AUX bap_reads\n");
  }
  if ( config->len ) {
    cfg = *config;
  } else {
    // general configuration (read/modify/write)
    status = PC4500_readrid(ai, RID_CONFIG, &cfg, sizeof(cfg));
    if ( status != SUCCESS ) return ERROR;
    cfg.opmode = adhoc ? MODE_STA_IBSS : MODE_STA_ESS;
    
    /* Save off the MAC */
    for( i = 0; i < 6; i++ ) {
      mac[i] = cfg.macAddr[i];
    }
    
    /* Check to see if there are any insmod configured 
       rates to add */
    if ( rates ) {
      int i = 0;
      if ( rates[0] ) memset(cfg.rates,0,sizeof(cfg.rates));
      for( i = 0; i < 8 && rates[i]; i++ ) {
	cfg.rates[i] = rates[i];
      }    
    }
    if ( basic_rate > 0 ) {
      int i;
      for( i = 0; i < 8; i++ ) {
	if ( cfg.rates[i] == basic_rate ||
	     !cfg.rates ) {
	  cfg.rates[i] = basic_rate | 0x80;
	  break;
	}
      }
    }
    *config = cfg;
  }  
  /* Check for MIC capability raw 1/28/02 and do 
   * apropriate setup
   */
  
  PC4500_readrid(ai,RID_CAPABILITIES,(char *)&xcaps,sizeof(xcaps));
  
  /*
   * Find out if an extended capability rid was returned
   */
  if(xcaps.u16RidLen >= sizeof(xcaps)){
#ifdef DEBUG_MIC
    printk(KERN_INFO "Extended cap's\n");
#endif
    /* Check for MIC capability */
    if(xcaps.u16ExtSoftwareCapabilities & EXT_SOFT_CAPS_MIC ){
#ifdef DEBUG_MIC
      printk(KERN_INFO "Mic capable\n");
#endif
      cfg.opmode   |= CFG_OPMODE_MIC;
      ai->flags |= MIC_CAPABLE;
      micsetup(ai);
      *config = cfg;
    }
  }

  /* Setup the SSIDs if present */
  if ( ssids[0] ) {
    int i = 0;
    for( i = 0; i < 3 && ssids[i]; i++ ) {
      mySsid.ssids[i].len = strlen(ssids[i]);
      if ( mySsid.ssids[i].len > 32 ) 
	mySsid.ssids[i].len = 32;
      memcpy(mySsid.ssids[i].ssid, ssids[i],mySsid.ssids[i].len);
      mySsid.ssids[i].len = cpu_to_le16(mySsid.ssids[i].len);
    }
  }
	
  status = PC4500_writerid( ai, RID_CONFIG, &cfg, sizeof(cfg));
  if ( status != SUCCESS ) return ERROR;
  
  /* Set up the SSID list */
  status = PC4500_writerid(ai, RID_SSID, &mySsid, sizeof(mySsid));
  if ( status != SUCCESS ) return ERROR;
  
  status = enable_MAC(ai, &rsp);
  
  if (status != SUCCESS || (rsp.status & 0xFF00) != 0) {
    int reason = rsp.rsp0;
    int badRidNumber = rsp.rsp1;
    int badRidOffset = rsp.rsp2;
    printk( KERN_ERR "airo: Bad MAC enable reason = %x, rid = %x, offset = %d\n",
	    reason,
	    badRidNumber,
	    badRidOffset );
    return ERROR;
  }
  return SUCCESS;
}

static u16 issuecommand(struct airo_info *ai, Cmd *pCmd, Resp *pRsp) {
        // Im really paranoid about letting it run forever!
	int max_tries = 600000;  
        int rc = SUCCESS;
	int flags;

	spin_lock_irqsave(&ai->cmd_lock, flags);
	OUT4500(ai, PARAM0, pCmd->parm0);
	OUT4500(ai, PARAM1, pCmd->parm1);
	OUT4500(ai, PARAM2, pCmd->parm2);
	OUT4500(ai, COMMAND, pCmd->cmd);
	while ( max_tries-- &&
		(IN4500(ai, EVSTAT) & EV_CMD) == 0) {
		if ( IN4500(ai, COMMAND) == pCmd->cmd) { 
			// PC4500 didn't notice command, try again
			OUT4500(ai, COMMAND, pCmd->cmd);
		}
	}
	if ( max_tries == -1 ) {
		printk( KERN_ERR 
			"airo: Max tries exceeded when issueing command\n" );
                rc = ERROR;
                goto done;
	}
	// command completed
	pRsp->status = IN4500(ai, STATUS);
	pRsp->rsp0 = IN4500(ai, RESP0);
	pRsp->rsp1 = IN4500(ai, RESP1);
	pRsp->rsp2 = IN4500(ai, RESP2);
	
	// clear stuck command busy if necessary
	if (IN4500(ai, COMMAND) & COMMAND_BUSY) {
		OUT4500(ai, EVACK, EV_CLEARCOMMANDBUSY);
	}
	// acknowledge processing the status/response
	OUT4500(ai, EVACK, EV_CMD);
done:
	spin_unlock_irqrestore(&ai->cmd_lock, flags);
	return rc;
}

/* Sets up the bap to start exchange data.  whichbap should
 * be one of the BAP0 or BAP1 defines.  Locks should be held before
 * calling! */
static int bap_setup(struct airo_info *ai, u16 rid, u16 offset, int whichbap )
{
	int timeout = 50;
	int max_tries = 3;
	
	OUT4500(ai, SELECT0+whichbap, rid);
	OUT4500(ai, OFFSET0+whichbap, offset);
	while (1) {
		int status = IN4500(ai, OFFSET0+whichbap);
		if (status & BAP_BUSY) {
                        /* This isn't really a timeout, but its kinda
			   close */
			if (timeout--) { 
				continue;
			}
		} else if ( status & BAP_ERR ) {
			/* invalid rid or offset */
			printk( KERN_ERR "airo: BAP error %x %d\n", 
				status, whichbap );
			return ERROR;
		} else if (status & BAP_DONE) { // success
			return SUCCESS;
		}
		if ( !(max_tries--) ) {
			printk( KERN_ERR 
				"airo: BAP setup error too many retries\n" );
			return ERROR;
		}
		// -- PC4500 missed it, try again
		OUT4500(ai, SELECT0+whichbap, rid);
		OUT4500(ai, OFFSET0+whichbap, offset);
		timeout = 50;
	}
}

/* should only be called by aux_bap_read.  This aux function and the
   following use concepts not documented in the developers guide.  I
   got them from a patch given to my by Aironet */
static u16 aux_setup(struct airo_info *ai, u16 page,
		     u16 offset, u16 *len)
{
	u16 next;

	OUT4500(ai, AUXPAGE, page);
	OUT4500(ai, AUXOFF, 0);
	next = IN4500(ai, AUXDATA);
	*len = IN4500(ai, AUXDATA)&0xff;
	if (offset != 4) OUT4500(ai, AUXOFF, offset);
	return next;
}

/* requires call to bap_setup() first */
static int aux_bap_read(struct airo_info *ai, u16 *pu16Dst,
		     int bytelen, int whichbap) 
{
	u16 len;
	u16 page;
	u16 offset;
	u16 next;
	int words;
	int i;
	int flags;

	spin_lock_irqsave(&ai->aux_lock, flags);
	page = IN4500(ai, SWS0+whichbap);
	offset = IN4500(ai, SWS2+whichbap);
	next = aux_setup(ai, page, offset, &len);
	words = (bytelen+1)>>1;

	for (i=0; i<words;) {
		int count;
		count = (len>>1) < (words-i) ? (len>>1) : (words-i);
		if ( !do8bitIO ) 
			insw( ai->dev->base_addr+DATA0+whichbap, 
			      pu16Dst+i,count );
		else
			insb( ai->dev->base_addr+DATA0+whichbap, 
			      pu16Dst+i, count << 1 );
		i += count;
		if (i<words) {
			next = aux_setup(ai, next, 4, &len);
		}
	}
	spin_unlock_irqrestore(&ai->aux_lock, flags);
	return SUCCESS;
}


/* requires call to bap_setup() first */
static int fast_bap_read(struct airo_info *ai, u16 *pu16Dst, 
			 int bytelen, int whichbap)
{
	bytelen = (bytelen + 1) & (~1); // round up to even value
	if ( !do8bitIO ) 
		insw( ai->dev->base_addr+DATA0+whichbap, pu16Dst, bytelen>>1 );
	else
		insb( ai->dev->base_addr+DATA0+whichbap, pu16Dst, bytelen );
	return SUCCESS;
}

/* requires call to bap_setup() first */
static int bap_write(struct airo_info *ai, const u16 *pu16Src, 
		     int bytelen, int whichbap)
{
	bytelen = (bytelen + 1) & (~1); // round up to even value
	if ( !do8bitIO ) 
		outsw( ai->dev->base_addr+DATA0+whichbap, 
		       pu16Src, bytelen>>1 );
	else
		outsb( ai->dev->base_addr+DATA0+whichbap, pu16Src, bytelen );
	return SUCCESS;
}

static int PC4500_accessrid(struct airo_info *ai, u16 rid, u16 accmd)
{
	Cmd cmd; /* for issuing commands */
	Resp rsp; /* response from commands */
	u16 status;

	memset(&cmd, 0, sizeof(cmd));
	cmd.cmd = accmd;
	cmd.parm0 = rid;
	status = issuecommand(ai, &cmd, &rsp);
	if (status != 0) return status;
	if ( (rsp.status & 0x7F00) != 0) {
		return (accmd << 8) + (rsp.rsp0 & 0xFF);
	}
	return 0;
}

/*  Note, that we are using BAP1 which is also used by transmit, so
 *  we must get a lock. */
static int PC4500_readrid(struct airo_info *ai, u16 rid, void *pBuf, int len)
{
	u16 status;
        int flags;
        int rc = SUCCESS;

	spin_lock_irqsave(&ai->bap1_lock, flags);
	if ( (status = PC4500_accessrid(ai, rid, CMD_ACCESS)) != SUCCESS) {
                rc = status;
                goto done;
        }
	if (bap_setup(ai, rid, 0, BAP1) != SUCCESS) {
		rc = ERROR;
                goto done;
        }
	// read the rid length field
	bap_read(ai, pBuf, 2, BAP1);
	// length for remaining part of rid
	len = MIN(len, le16_to_cpu(*(u16*)pBuf)) - 2;
	
	if ( len <= 2 ) {
		printk( KERN_ERR 
			"airo: Rid %x has a length of %d which is too short\n",
			(int)rid,
			(int)len );
		rc = ERROR;
                goto done;
	}
	// read remainder of the rid
	if (bap_setup(ai, rid, 2, BAP1) != SUCCESS) {
                rc = ERROR;
                goto done;
        }
	rc = bap_read(ai, ((u16*)pBuf)+1, len, BAP1);
done:
	spin_unlock_irqrestore(&ai->bap1_lock, flags);
	return rc;
}




/*  Note, that we are using BAP1 which is also used by transmit, so
 *  make sure this isnt called when a transmit is happening */
static int PC4500_writerid(struct airo_info *ai, u16 rid, 
			   const void *pBuf, int len)
{
	u16 status;
        int flags;
	int rc = SUCCESS;

	spin_lock_irqsave(&ai->bap1_lock, flags);
        *(unsigned short *)pBuf = len;
	// --- first access so that we can write the rid data
	if ( (status = PC4500_accessrid(ai, rid, CMD_ACCESS)) != 0) {
                rc = status;
                goto done;
        }
	// --- now write the rid data
	if (bap_setup(ai, rid, 0, BAP1) != SUCCESS) {
                rc = ERROR;
                goto done;
        }
	bap_write(ai, pBuf, len, BAP1);
	// ---now commit the rid data
	rc = PC4500_accessrid(ai, rid, 0x100|CMD_ACCESS);
done:
	spin_unlock_irqrestore(&ai->bap1_lock, flags);
        return rc;
}

/* Allocates a FID to be used for transmitting packets.  We only use
   one for now. */
static u16 transmit_allocate(struct airo_info *ai, int lenPayload)
{
	Cmd cmd;
	Resp rsp;
	u16 txFid;
	u16 txControl;
        int flags;

	cmd.cmd = CMD_ALLOCATETX;
	cmd.parm0 = lenPayload;
	if (issuecommand(ai, &cmd, &rsp) != SUCCESS) return 0;
	if ( (rsp.status & 0xFF00) != 0) return 0;
	/* wait for the allocate event/indication
	 * It makes me kind of nervous that this can just sit here and spin,
	 * but in practice it only loops like four times. */
	while ( (IN4500(ai, EVSTAT) & EV_ALLOC) == 0) ;
	// get the allocated fid and acknowledge
	txFid = IN4500(ai, TXALLOCFID);
	OUT4500(ai, EVACK, EV_ALLOC);
  
	/*  The CARD is pretty cool since it converts the ethernet packet
	 *  into 802.11.  Also note that we don't release the FID since we
	 *  will be using the same one over and over again. */
	/*  We only have to setup the control once since we are not
	 *  releasing the fid. */
	txControl = TXCTL_TXOK | TXCTL_TXEX | TXCTL_802_3
		| TXCTL_ETHERNET | TXCTL_NORELEASE;
	spin_lock_irqsave(&ai->bap1_lock, flags);
	if (bap_setup(ai, txFid, 0x0008, BAP1) != SUCCESS) return ERROR;
	bap_write(ai, &txControl, sizeof(txControl), BAP1);
	spin_unlock_irqrestore(&ai->bap1_lock, flags);

	return txFid;
}

/* In general BAP1 is dedicated to transmiting packets.  However,
   since we need a BAP when accessing RIDs, we also use BAP1 for that.
   Make sure the BAP1 spinlock is held when this is called. 
*/

static int transmit_802_3_packet(struct airo_info *ai, u16 txFid, 
				 char *pPacket, int len)
{
  u16 payloadLen,adhoc;
  u32 miclen=0;
  Cmd cmd;
  Resp rsp;
  //  u8 mictxbuf[2048];
	
  if (len < 12) {
    printk( KERN_WARNING "Short packet %d\n", len );
    return ERROR;
  }

  if(len > 2048 ){
    printk( KERN_WARNING "Long packet %d\n", len );
    return ERROR;
  }
  
  memcpy(ai->mictxbuf,pPacket,len);
  miclen = len;

  /*
   *  packet is destination[6], source[6], payload[len-12]
   *  write the payload length and dst/src/payload
   */
  
  if(ai->flags & MIC_CAPABLE){

    adhoc = (ai->flags & ADHOC) ? 1:0;

    if(Encapsulate(ai,(ETH_HEADER_STRUC *)ai->mictxbuf,&miclen , adhoc))  {
      len = (s16 ) miclen;
    }
    else
      {
//#ifdef DEBUG_MIC
	printk(KERN_INFO "Dumped tx packet\n");
//#endif
	return 1;
      }      
  }

  if (bap_setup(ai, txFid, 0x0036, BAP1) != SUCCESS) return ERROR;

  /* The hardware addresses aren't counted as part of the payload, so
   * we have to subtract the 12 bytes for the addresses off 
   */

  payloadLen = cpu_to_le16(len - 12);
  bap_write(ai, &payloadLen, sizeof(payloadLen),BAP1);
  bap_write(ai, (const u16*)ai->mictxbuf, len, BAP1);

  // issue the transmit command
  memset( &cmd, 0, sizeof( cmd ) );
  cmd.cmd = CMD_TRANSMIT;
  cmd.parm0 = txFid;
  if (issuecommand(ai, &cmd, &rsp) != SUCCESS) return ERROR;
  if ( (rsp.status & 0xFF00) != 0) return ERROR;
  return SUCCESS;
}

/*
 *  This is the proc_fs routines.  It is a bit messier than I would
 *  like!  Feel free to clean it up!
 */

/*
 *  Unfortunately sometime between 2.0 and 2.2 the proc interface changed...
 *  Unfortunately I dont know when it was...
 *  Im guessing it is sometime around 0x20155...  Anybody know?
 */

#if (LINUX_VERSION_CODE > 0x20155)
static ssize_t proc_read( struct file *file,
			  char *buffer,
			  size_t len,
			  loff_t *offset);

static ssize_t proc_write( struct file *file,
			   const char *buffer,
			   size_t len,
			   loff_t *offset );
static int proc_close( struct inode *inode, struct file *file );
#else
static int proc_read( struct inode *inode,
		      struct file *file,
		      char *buffer,
		      int len );

static int proc_write( struct inode *inode,
		       struct file *file,
		       const char *buffer,
		       int len );
static void proc_close( struct inode *inode, struct file *file );
#endif

static int proc_stats_open( struct inode *inode, struct file *file );
static int proc_statsdelta_open( struct inode *inode, struct file *file );
static int proc_status_open( struct inode *inode, struct file *file );
static int proc_SSID_open( struct inode *inode, struct file *file );
static int proc_APList_open( struct inode *inode, struct file *file );
static int proc_config_open( struct inode *inode, struct file *file );
static int proc_wepkey_open( struct inode *inode, struct file *file );

static struct file_operations proc_statsdelta_ops = {
	read:           proc_read,
	open:           proc_statsdelta_open,
	release:        proc_close
};

static struct file_operations proc_stats_ops = {
	read:           proc_read,
	open:           proc_stats_open,
	release:        proc_close
};

static struct file_operations proc_status_ops = {
	read:            proc_read,
	open:            proc_status_open,
	release:         proc_close
};

static struct file_operations proc_SSID_ops = {
	read:          proc_read,
	write:         proc_write,
	open:          proc_SSID_open,
	release:       proc_close
};

static struct file_operations proc_APList_ops = {
	read:          proc_read,
	write:         proc_write,
	open:          proc_APList_open,
	release:       proc_close
};

static struct file_operations proc_config_ops = {
	read:          proc_read,
	write:         proc_write,
	open:          proc_config_open,
	release:       proc_close
};

static struct file_operations proc_wepkey_ops = {
	read:          proc_read,
	write:         proc_write,
	open:          proc_wepkey_open,
	release:       proc_close
};

#if (LINUX_VERSION_CODE < 0x20355)
static struct inode_operations proc_inode_statsdelta_ops = {
	&proc_statsdelta_ops};

static struct inode_operations proc_inode_stats_ops = {
	&proc_stats_ops};

static struct inode_operations proc_inode_status_ops = {
	&proc_status_ops};

static struct inode_operations proc_inode_SSID_ops = {
	&proc_SSID_ops};

static struct inode_operations proc_inode_APList_ops = {
	&proc_APList_ops};

static struct inode_operations proc_inode_config_ops = {
	&proc_config_ops};

static struct inode_operations proc_inode_wepkey_ops = {
	&proc_wepkey_ops};
#endif

#if ((LINUX_VERSION_CODE > 0x20155) && (LINUX_VERSION_CODE < 0x20311))
/*
 * We need to do reference counting here.  When the inode is first used,
 * this will be called with fill non-zero.  When it is released this
 * will be called with fill set to zero.
 */
static void airo_fill_inode( struct inode *i, int fill ) {
	if ( fill ) {
		MOD_INC_USE_COUNT;
	} else {
		MOD_DEC_USE_COUNT;
	}
}
#endif

#if (LINUX_VERSION_CODE < 0x20311)
static struct file_operations airo_file_ops = {
	NULL, // lseek
	NULL, // read
	NULL, // write
	NULL, // readdir
	NULL, // select
	NULL, // ioctl
	NULL, // mmap
	NULL, // open
	NULL, // release
};

static struct inode_operations airo_inode_ops = {
	&airo_file_ops,
	NULL, // create
	NULL, // lookup
};

static struct proc_dir_entry airo_entry = {
	0,
	7,
	"aironet",
	S_IFDIR | S_IRUGO | S_IXUGO,
	1,
	0, 0,
	44,
	&airo_inode_ops,
	0, // get_info
#if ((LINUX_VERSION_CODE > 0x20155) && (LINUX_VERSION_CODE < 0x20311))
	airo_fill_inode
#endif
};
#else
static struct proc_dir_entry *airo_entry = 0;
#endif

#if (LINUX_VERSION_CODE < 0x20311)
static struct proc_dir_entry wepkey_entry = {
	0, 6, "WepKey",
	S_IFREG | S_IWUSR, 2, 0, 0,
	13,
	&proc_inode_wepkey_ops, NULL
};

static struct proc_dir_entry statsdelta_entry = {
	0, 10, "StatsDelta",
	S_IFREG | S_IRUGO | S_IWUSR , 2, 0, 0,
	13,
	&proc_inode_statsdelta_ops, NULL
};

static struct proc_dir_entry stats_entry = {
	0, 5, "Stats",
	S_IFREG | S_IRUGO , 2, 0, 0,
	13,
	&proc_inode_stats_ops, NULL
};

static struct proc_dir_entry status_entry = {
	0, 6, "Status",
	S_IFREG | S_IRUGO , 2, 0, 0,
	13,
	&proc_inode_status_ops, NULL
};

static struct proc_dir_entry SSID_entry = {
	0, 4, "SSID",
	S_IFREG | S_IRUGO | S_IWUSR, 2, 0, 0,
	13,
	&proc_inode_SSID_ops, NULL
};

static struct proc_dir_entry APList_entry = {
	0, 6, "APList",
	S_IFREG | S_IRUGO | S_IWUSR, 2, 0, 0,
	13,
	&proc_inode_APList_ops, NULL
};

static struct proc_dir_entry config_entry = {
	0, 6, "Config",
	S_IFREG | S_IRUGO | S_IWUSR, 2, 0, 0,
	13,
	&proc_inode_config_ops, NULL
};
#endif

struct proc_data {
	int release_buffer;
	int readlen;
	char *rbuffer;
	int writelen;
	int maxwritelen;
	char *wbuffer;
	void (*on_close) (struct inode *, struct file *);
};

#if (LINUX_VERSION_CODE < 0x20311)
static int setup_proc_entry( struct net_device *dev,
			     struct airo_info *apriv ) {
	/* First setup the device directory */
	memset( &apriv->proc_entry, 0, sizeof( apriv->proc_entry ) );
	apriv->proc_entry.namelen = strlen( dev->name );
	apriv->proc_entry.name = dev->name;
	apriv->proc_entry.mode = S_IFDIR | S_IRUGO | S_IXUGO;
	apriv->proc_entry.nlink = 2;
	apriv->proc_entry.ops = airo_entry.ops;
	PROC_REGISTER( &airo_entry, &apriv->proc_entry );
	
	/* Setup the StatsDelta */
	memcpy( &apriv->proc_statsdelta_entry, &statsdelta_entry, 
		sizeof( statsdelta_entry ) );
	apriv->proc_statsdelta_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_statsdelta_entry );
	
	/* Setup the Stats */
	memcpy( &apriv->proc_stats_entry, &stats_entry, 
		sizeof( stats_entry ) );
	apriv->proc_stats_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_stats_entry );
	
	/* Setup the Status */
	memcpy( &apriv->proc_status_entry, &status_entry, 
		sizeof( status_entry ) );
	apriv->proc_status_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_status_entry );
	
	/* Setup the Config */
	memcpy( &apriv->proc_config_entry, &config_entry, 
		sizeof( config_entry ) );
	apriv->proc_config_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_config_entry );

	/* Setup the SSID */
	memcpy( &apriv->proc_SSID_entry, &SSID_entry, sizeof( SSID_entry ) );
	apriv->proc_SSID_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_SSID_entry );
	
	/* Setup the APList */
	memcpy( &apriv->proc_APList_entry, &APList_entry, 
		sizeof( APList_entry ) );
	apriv->proc_APList_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_APList_entry );
	
	/* Setup the WepKey */
	memcpy( &apriv->proc_wepkey_entry, &wepkey_entry, 
		sizeof( wepkey_entry ) );
	apriv->proc_wepkey_entry.data = dev;
	PROC_REGISTER( &apriv->proc_entry, 
		       &apriv->proc_wepkey_entry );
	
	return 0;
}

static int takedown_proc_entry( struct net_device *dev,
				struct airo_info *apriv ) {
	if ( !apriv->proc_entry.namelen ) return 0;
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_statsdelta_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_stats_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_status_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_config_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_SSID_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_APList_entry );
	PROC_UNREGISTER( &apriv->proc_entry, &apriv->proc_wepkey_entry );
	PROC_UNREGISTER( &airo_entry, &apriv->proc_entry );
	return 0;
}
#else
static int setup_proc_entry( struct net_device *dev,
			     struct airo_info *apriv ) {
	struct proc_dir_entry *entry;
	/* First setup the device directory */
	apriv->proc_entry = create_proc_entry(dev->name,
					      S_IFDIR|S_IRUGO|S_IXUGO,
					      airo_entry);
	/* Setup the StatsDelta */
	entry = create_proc_entry("StatsDelta",
				  S_IFREG | S_IRUGO | S_IWUSR,
				  apriv->proc_entry);
	entry->data = dev;
/*	This is what was needed right up to the last few versions
        of 2.3:
	entry->ops = &proc_inode_statsdelta_ops;
*/
	entry->proc_fops = &proc_statsdelta_ops;
	
	/* Setup the Stats */
	entry = create_proc_entry("Stats",
				  S_IFREG | S_IRUGO,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_stats_ops;
	
	/* Setup the Status */
	entry = create_proc_entry("Status",
				  S_IFREG | S_IRUGO,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_status_ops;
	
	/* Setup the Config */
	entry = create_proc_entry("Config",
				  S_IFREG | S_IRUGO | S_IWUGO,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_config_ops;

	/* Setup the SSID */
	entry = create_proc_entry("SSID",
				  S_IFREG | S_IRUGO | S_IWUGO,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_SSID_ops;

	/* Setup the APList */
	entry = create_proc_entry("APList",
				  S_IFREG | S_IRUGO | S_IWUGO,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_APList_ops;

	/* Setup the WepKey */
	entry = create_proc_entry("WepKey",
				  S_IFREG | S_IWUSR,
				  apriv->proc_entry);
	entry->data = dev;
	entry->proc_fops = &proc_wepkey_ops;

	return 0;
}

static int takedown_proc_entry( struct net_device *dev,
				struct airo_info *apriv ) {
	if ( !apriv->proc_entry->namelen ) return 0;
	remove_proc_entry("Stats",apriv->proc_entry);
	remove_proc_entry("StatsDelta",apriv->proc_entry);
	remove_proc_entry("Status",apriv->proc_entry);
	remove_proc_entry("Config",apriv->proc_entry);
	remove_proc_entry("SSID",apriv->proc_entry);
	remove_proc_entry("APList",apriv->proc_entry);
	remove_proc_entry("WepKey",apriv->proc_entry);
	remove_proc_entry(dev->name,airo_entry);
	return 0;
}
#endif

/*
 *  What we want from the proc_fs is to be able to efficiently read
 *  and write the configuration.  To do this, we want to read the
 *  configuration when the file is opened and write it when the file is
 *  closed.  So basically we allocate a read buffer at open and fill it
 *  with data, and allocate a write buffer and read it at close.
 */

/*
 *  The read routine is generic, it relies on the preallocated rbuffer
 *  to supply the data.
 */
#if (LINUX_VERSION_CODE > 0x20155)
static ssize_t proc_read( struct file *file,
			  char *buffer,
			  size_t len,
			  loff_t *offset )
#else
static int proc_read( struct inode *inode,
		      struct file *file,
		      char *buffer,
		      int len ) 
#endif
{
	int i;
	int pos;
	struct proc_data *priv = (struct proc_data*)file->private_data;
	
	if( !priv->rbuffer ) return -EINVAL;
	
#if (LINUX_VERSION_CODE > 0x20155)
	pos = *offset;
#else
	pos = file->f_pos;
#endif
	for( i = 0; i+pos < priv->readlen && i < len; i++ ) {
		put_user( priv->rbuffer[i+pos], buffer+i );
	}
#if (LINUX_VERSION_CODE > 0x20155)
	*offset += i;
#else
	file->f_pos += i;
#endif
	return i;
}

/*
 *  The write routine is generic, it fills in a preallocated rbuffer
 *  to supply the data.
 */
#if (LINUX_VERSION_CODE > 0x20155)
static ssize_t proc_write( struct file *file,
			   const char *buffer,
			   size_t len,
			   loff_t *offset ) 
#else
static int proc_write( struct inode *inode,
		       struct file *file,
		       const char *buffer,
		       int len ) 
#endif
{
	int i;
	int pos;
	struct proc_data *priv = (struct proc_data*)file->private_data;
	
	if ( !priv->wbuffer ) {
		return -EINVAL;
	}
	
#if (LINUX_VERSION_CODE > 0x20155)
	pos = *offset;
#else
	pos = file->f_pos;
#endif
	
	for( i = 0; i + pos <  priv->maxwritelen &&
		     i < len; i++ ) {
#if (LINUX_VERSION_CODE > 0x20155)
		get_user( priv->wbuffer[i+pos], buffer + i );
#else
		priv->wbuffer[i+pos] = get_user( buffer + i );
#endif
	}
	if ( i+pos > priv->writelen ) priv->writelen = i+file->f_pos;
#if (LINUX_VERSION_CODE > 0x20155)
	*offset += i;
#else
	file->f_pos += i;
#endif
	return i;
}

static int proc_status_open( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *apriv = (struct airo_info *)dev->priv;
	CapabilityRid cap_rid;
	StatusRid status_rid;
	
	MOD_INC_USE_COUNT;
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 2048, GFP_KERNEL );
	
	PC4500_readrid(apriv, RID_STATUS, &status_rid,
		       sizeof(status_rid));
	PC4500_readrid(apriv, RID_CAPABILITIES, &cap_rid,
		       sizeof(cap_rid));
	
	sprintf( data->rbuffer, "Mode: %x\n"
		 "Signal Strength: %d\n"
		 "Signal Quality: %d\n"
		 "SSID: %-.*s\n"
		 "AP: %-.16s\n"
		 "Freq: %d\n"
		 "BitRate: %dmbs\n"
		 "Driver Version: %s\n"
		 "Device: %s\nManufacturer: %s\nFirmware Version: %s\n"
		 "Radio type: %x\nCountry: %x\nHardware Version: %x\n"
		 "Software Version: %x\nSoftware Subversion: %x\n"
		 "Boot block version: %x\n",
		 (int)le16_to_cpu(status_rid.mode),
		 (int)le16_to_cpu(status_rid.normalizedSignalStrength),
		 (int)le16_to_cpu(status_rid.signalQuality),
		 (int)status_rid.SSIDlen,
		 status_rid.SSID,
		 status_rid.apName,
		 (int)le16_to_cpu(status_rid.channel),
		 (int)le16_to_cpu(status_rid.currentXmitRate)/2,
		 version,
		 cap_rid.prodName,
		 cap_rid.manName,
		 cap_rid.prodVer,
		 le16_to_cpu(cap_rid.radioType),
		 le16_to_cpu(cap_rid.country),
		 le16_to_cpu(cap_rid.hardVer),
		 (int)le16_to_cpu(cap_rid.softVer),
		 (int)le16_to_cpu(cap_rid.softSubVer),
		 (int)le16_to_cpu(cap_rid.bootBlockVer) );
	data->readlen = strlen( data->rbuffer );
	return 0;
}

static int proc_stats_rid_open(struct inode*, struct file*, u16);
static int proc_statsdelta_open( struct inode *inode, 
				 struct file *file ) {
	if (file->f_mode&FMODE_WRITE) {
	return proc_stats_rid_open(inode, file, RID_STATSDELTACLEAR);
	}
	return proc_stats_rid_open(inode, file, RID_STATSDELTA);
}

static int proc_stats_open( struct inode *inode, struct file *file ) {
	return proc_stats_rid_open(inode, file, RID_STATS);
}

static int proc_stats_rid_open( struct inode *inode, 
				struct file *file,
				u16 rid ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *apriv = (struct airo_info *)dev->priv;
	char buffer[1024];
	int i, j;
	int *vals = (int*)&buffer[4];
	MOD_INC_USE_COUNT;
	
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 4096, GFP_KERNEL );
	
	PC4500_readrid(apriv, rid, buffer,
		       sizeof(buffer));

        j = 0;
	for(i=0; (int)statsLabels[i]!=-1 && 
		    i*4<le16_to_cpu(*(u16*)buffer); i++){
                if (!statsLabels[i]) continue;
		if (j+strlen(statsLabels[i])+16>4096) {
			printk(KERN_WARNING
			       "airo: Potentially disasterous buffer overflow averted!\n");
			break;
		}
		j+=sprintf( data->rbuffer+j, "%s: %d\n",statsLabels[i],
			    le32_to_cpu(vals[i]));
        }
	if (i*4>=le16_to_cpu(*(u16*)buffer)){
		printk(KERN_WARNING
		       "airo: Got a short rid\n");
	}
	data->readlen = j;
	return 0;
}

static int get_dec_u16( char *buffer, int *start, int limit ) {
	u16 value;
	int valid = 0;
	for( value = 0; buffer[*start] >= '0' &&
		     buffer[*start] <= '9' &&
		     *start < limit; (*start)++ ) {
		valid = 1;
		value *= 10;
		value += buffer[*start] - '0';
	}
	if ( !valid ) return -1;
	return value;
}

static void checkThrottle(ConfigRid *config) {
	int i;
	/* Old hardware had a limit on encryption speed */
	if (config->authType != AUTH_OPEN && maxencrypt) {
		for(i=0; i<8; i++) {
			if (config->rates[i] > maxencrypt) {
				config->rates[i] = 0;
			}
		}
	}
}

static void proc_config_on_close( struct inode *inode, struct file *file ) {
	struct proc_data *data = file->private_data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	ConfigRid config;
	Cmd cmd;
	Resp rsp;
	char *line;
	
	if ( !data->writelen ) return;
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	memset(&cmd, 0, sizeof(Cmd));
	cmd.cmd = MAC_DISABLE; // disable in case already enabled
	issuecommand(ai, &cmd, &rsp);
	PC4500_readrid(ai, RID_ACTUALCONFIG, &config,
		       sizeof(config));

	line = data->wbuffer;
	while( line[0] ) {
		/*** Mode processing */
		if ( !strncmp( line, "Mode: ", 6 ) ) {
			line += 6;
			if ( line[0] == 'a' ) config.opmode = 0;
			else config.opmode = cpu_to_le16(1);
		}
		
		/*** Radio status */
		else if (!strncmp(line,"Radio: ", 7)) {
			line += 7;
			if (!strncmp(line,"off",3)) {
				ai->flags |= FLAG_RADIO_OFF;
			} else {
				ai->flags &= ~FLAG_RADIO_OFF;
			}
		}
		/*** NodeName processing */
		else if ( !strncmp( line, "NodeName: ", 10 ) ) {
			int j;
			
			line += 10;
			memset( config.nodeName, 0, 16 );
			/* Do the name, assume a space between the mode and node name */
			for( j = 0; j < 16 && line[j] != '\n'; j++ ) {
				config.nodeName[j] = line[j];
			}
		} 
		
		/*** PowerMode processing */
		else if ( !strncmp( line, "PowerMode: ", 11 ) ) {
			line += 11;
			if ( !strncmp( line, "PSPCAM", 6 ) ) {
				config.powerSaveMode = 
					cpu_to_le16(POWERSAVE_PSPCAM);
			} else if ( !strncmp( line, "PSP", 3 ) ) {
				config.powerSaveMode = 
					cpu_to_le16(POWERSAVE_PSP);
			} else {
				config.powerSaveMode = 
					cpu_to_le16(POWERSAVE_CAM);
			}	
		} else if ( !strncmp( line, "DataRates: ", 11 ) ) {
			int v, i = 0, k = 0; /* i is index into line, 
						k is index to rates */
			
			line += 11;
			while((v = get_dec_u16(line, &i, 3))!=-1) {
				config.rates[k++] = (u8)v;
				line += i + 1;
				i = 0;
			}
		} else if ( !strncmp( line, "Channel: ", 9 ) ) {
			int v, i = 0;
			line += 9;
			v = get_dec_u16(line, &i, i+3);
			if ( v != -1 ) 
				config.channelSet = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "XmitPower: ", 11 ) ) {
			int v, i = 0;
			line += 11;
			v = get_dec_u16(line, &i, i+3);
			if ( v != -1 ) config.txPower = 
					       (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "WEP: ", 5 ) ) {
			line += 5;
			switch( line[0] ) {
			case 's':
				config.authType = cpu_to_le16(AUTH_SHAREDKEY);
				break;
			case 'e':
				config.authType = cpu_to_le16(AUTH_ENCRYPT);
				break;
			default:
				config.authType = cpu_to_le16(AUTH_OPEN);
				break;
			}
		} else if ( !strncmp( line, "LongRetryLimit: ", 16 ) ) {
			int v, i = 0;
			
			line += 16;
			v = get_dec_u16(line, &i, 3);
			v = (v<0) ? 0 : ((v>255) ? 255 : v);
			config.longRetryLimit = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "ShortRetryLimit: ", 17 ) ) {
			int v, i = 0;
			
			line += 17;
			v = get_dec_u16(line, &i, 3);
			v = (v<0) ? 0 : ((v>255) ? 255 : v);
			config.shortRetryLimit = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "RTSThreshold: ", 14 ) ) {
			int v, i = 0;
			
			line += 14;
			v = get_dec_u16(line, &i, 4);
			v = (v<0) ? 0 : ((v>2312) ? 2312 : v);
			config.rtsThres = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "TXMSDULifetime: ", 16 ) ) {
			int v, i = 0;
			
			line += 16;
			v = get_dec_u16(line, &i, 5);
			v = (v<0) ? 0 : v;
			config.txLifetime = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "RXMSDULifetime: ", 16 ) ) {
			int v, i = 0;
			
			line += 16;
			v = get_dec_u16(line, &i, 5);
			v = (v<0) ? 0 : v;
			config.rxLifetime = (u16)cpu_to_le16(v);
		} else if ( !strncmp( line, "TXDiversity: ", 13 ) ) {
			config.txDiversity = 
				(line[13]=='l') ? cpu_to_le16(1) :
				((line[13]=='r')? cpu_to_le16(2):
				 cpu_to_le16(3));
		} else if ( !strncmp( line, "RXDiversity: ", 13 ) ) {
			config.rxDiversity = 
				(line[13]=='l') ? cpu_to_le16(1) :
				((line[13]=='r')? cpu_to_le16(2):
				 cpu_to_le16(3));
		} else if ( !strncmp( line, "FragThreshold: ", 15 ) ) {
			int v, i = 0;
			
			line += 15;
			v = get_dec_u16(line, &i, 4);
			v = (v<256) ? 256 : ((v>2312) ? 2312 : v);
			v = v & 0xfffe; /* Make sure its even */
			config.fragThresh = (u16)cpu_to_le16(v);
		} else if (!strncmp(line, "Modulation: ", 12)) {
		        line += 12;
			switch(*line) {
			case 'd':  config.modulation=MOD_DEFAULT; break;
			case 'c':  config.modulation=MOD_CCK; break;
			case 'm':  config.modulation=MOD_MOK; break;
			default:
				printk( KERN_WARNING "airo: Unknown modulation\n" );
			}
		} else {
			printk( KERN_WARNING "Couldn't figure out %s\n", line );
		}
		while( line[0] && line[0] != '\n' ) line++;
		if ( line[0] ) line++;
	}
	ai->config = config;
	checkThrottle(&config);
	PC4500_writerid(ai, RID_CONFIG, &config,
			sizeof(config));
	enable_MAC(ai, &rsp);
}

static int proc_config_open( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	ConfigRid config;
	int i;
	
	MOD_INC_USE_COUNT;
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 2048, GFP_KERNEL );
	data->wbuffer = kmalloc( 2048, GFP_KERNEL );
	memset( data->wbuffer, 0, 2048 );
	data->maxwritelen = 2048;
	data->on_close = proc_config_on_close;
	
	PC4500_readrid(ai, RID_ACTUALCONFIG, &config,
		       sizeof(config));
	
	i = sprintf( data->rbuffer, 
		     "Mode: %s\n"
		     "Radio: %s\n"
		     "NodeName: %-16s\n"
		     "PowerMode: %s\n"
		     "DataRates: %d %d %d %d %d %d %d %d\n"
		     "Channel: %d\n"
		     "XmitPower: %d\n",
		     config.opmode == 0 ? "adhoc" : 
		     config.opmode == 1 ? "ESS" :
		     config.opmode == 2 ? "AP" : 
		     config.opmode == 3 ? "AP RPTR" : "Error",
		     ai->flags&FLAG_RADIO_OFF ? "off" : "on",
		     config.nodeName,
		     config.powerSaveMode == 0 ? "CAM" :
		     config.powerSaveMode == 1 ? "PSP" :
		     config.powerSaveMode == 2 ? "PSPCAM" : "Error",
		     (int)config.rates[0],
		     (int)config.rates[1],
		     (int)config.rates[2],
		     (int)config.rates[3],
		     (int)config.rates[4],
		     (int)config.rates[5],
		     (int)config.rates[6],
		     (int)config.rates[7],
		     (int)le16_to_cpu(config.channelSet),
		     (int)le16_to_cpu(config.txPower)
		);
	sprintf( data->rbuffer + i,
		 "LongRetryLimit: %d\n"
		 "ShortRetryLimit: %d\n"
		 "RTSThreshold: %d\n"
		 "TXMSDULifetime: %d\n"
		 "RXMSDULifetime: %d\n"
		 "TXDiversity: %s\n"
		 "RXDiversity: %s\n"
		 "FragThreshold: %d\n"
		 "WEP: %s\n"
		 "Modulation: %s\n",
		 (int)le16_to_cpu(config.longRetryLimit),
		 (int)le16_to_cpu(config.shortRetryLimit),
		 (int)le16_to_cpu(config.rtsThres),
		 (int)le16_to_cpu(config.txLifetime),
		 (int)le16_to_cpu(config.rxLifetime),
		 config.txDiversity == 1 ? "left" :
		 config.txDiversity == 2 ? "right" : "both",
		 config.rxDiversity == 1 ? "left" :
		 config.rxDiversity == 2 ? "right" : "both",
		 (int)le16_to_cpu(config.fragThresh),
		 config.authType == AUTH_ENCRYPT ? "encrypt" :
		 config.authType == AUTH_SHAREDKEY ? "shared" : "open",
		 config.modulation == 0 ? "default" :
		 config.modulation == MOD_CCK ? "cck" :
		 config.modulation == MOD_MOK ? "mok" : "error"
		);
	data->readlen = strlen( data->rbuffer );
	return 0;
}

static void proc_SSID_on_close( struct inode *inode, struct file *file ) {
	struct proc_data *data = (struct proc_data *)file->private_data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	SsidRid SSID_rid;
	int i;
	int offset = 0;
	
	if ( !data->writelen ) return;
	
	memset( &SSID_rid, 0, sizeof( SSID_rid ) );
	
	for( i = 0; i < 3; i++ ) {
		int j;
		for( j = 0; j+offset < data->writelen && j < 32 &&
			     data->wbuffer[offset+j] != '\n'; j++ ) {
			SSID_rid.ssids[i].ssid[j] = data->wbuffer[offset+j];
		}
		if ( j == 0 ) break;
		SSID_rid.ssids[i].len = cpu_to_le16(j);
		offset += j;
		while( data->wbuffer[offset] != '\n' && 
		       offset < data->writelen ) offset++;
		offset++;
	}
	do_writerid(ai, RID_SSID, &SSID_rid, sizeof( SSID_rid ));
}

inline static u8 hexVal(char c) {
	if (c>='0' && c<='9') return c -= '0';
	if (c>='a' && c<='f') return c -= 'a'-10;
	if (c>='A' && c<='F') return c -= 'A'-10;
	return 0;
}

static void proc_APList_on_close( struct inode *inode, struct file *file ) {
	struct proc_data *data = (struct proc_data *)file->private_data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	APListRid APList_rid;
	int i;
	
	if ( !data->writelen ) return;
	
	memset( &APList_rid, 0, sizeof(APList_rid) );
	APList_rid.len = sizeof(APList_rid);
	
	for( i = 0; i < 4 && data->writelen >= (i+1)*6*3; i++ ) {
		int j;
		for( j = 0; j < 6*3 && data->wbuffer[j+i*6*3]; j++ ) {
			switch(j%3) {
			case 0:
				APList_rid.ap[i][j/3]=
					hexVal(data->wbuffer[j+i*6*3])<<4;
				break;
			case 1:
				APList_rid.ap[i][j/3]|=
					hexVal(data->wbuffer[j+i*6*3]);
				break;
			}
		}
	}
	do_writerid(ai, RID_APLIST, &APList_rid, sizeof( APList_rid ));
}

/* This function wraps PC4500_writerid with a MAC disable */
static int do_writerid( struct airo_info *ai, u16 rid, const void *rid_data,
			int len ) {
	int rc;
	Cmd cmd;
	Resp rsp;
	
	memset(&cmd, 0, sizeof(cmd));
	cmd.cmd = MAC_DISABLE; // disable in case already enabled
	issuecommand(ai, &cmd, &rsp);
	rc = PC4500_writerid(ai, 
			     rid, rid_data, len);
	enable_MAC(ai, &rsp);
	return rc;
}

/* Returns the length of the key at the index.  If index == 0xffff
 * the index of the transmit key is returned.  If the key doesn't exist,
 * -1 will be returned.
 */
#ifdef WIRELESS_EXT
static int get_wep_key(struct airo_info *ai, u16 index) {
	WepKeyRid wkr;
	int rc;
	u16 lastindex;

	rc = PC4500_readrid(ai, RID_WEP_TEMP, &wkr, sizeof(wkr));
	if (rc == SUCCESS) do {
		lastindex = wkr.kindex;
		if (wkr.kindex == index) {
			if (index == 0xffff) {
				return wkr.mac[0];
			}
			return wkr.klen;
		}
		PC4500_readrid(ai, RID_WEP_PERM, &wkr, sizeof(wkr));
	} while(lastindex != wkr.kindex);
	return -1;
}
#endif

static int set_wep_key(struct airo_info *ai, u16 index,
		       const char *key, u16 keylen, int perm ) {
	static const unsigned char macaddr[6] = { 0x01, 0, 0, 0, 0, 0 };
	WepKeyRid wkr;
        int rc;

	memset(&wkr, 0, sizeof(wkr));
	if (keylen == 0) {
		// We are selecting which key to use
		wkr.len = cpu_to_le16(sizeof(wkr));
		wkr.kindex = cpu_to_le16(0xffff);
		wkr.mac[0] = (char)index;
		if (perm) printk(KERN_INFO "Setting transmit key to %d\n", index);
		if (perm) ai->defindex = (char)index;
	} else {
		// We are actually setting the key
		wkr.len = cpu_to_le16(sizeof(wkr));
		wkr.kindex = cpu_to_le16(index);
		wkr.klen = cpu_to_le16(keylen);
		memcpy( wkr.key, key, keylen );
	        memcpy( wkr.mac, macaddr, 6 );
		printk(KERN_INFO "Setting key %d\n", index);
	}
	rc = do_writerid(ai, RID_WEP_TEMP, &wkr, sizeof(wkr));
        if (rc!=SUCCESS) printk(KERN_ERR "airo:  WEP_TEMP set %x\n", rc); 
	if (perm) {
		rc = do_writerid(ai, RID_WEP_PERM, &wkr, sizeof(wkr));
		if (rc!=SUCCESS) {
			printk(KERN_ERR "airo:  WEP_PERM set %x\n", rc);
		}
	}
	return 0;
}

static void proc_wepkey_on_close( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	int i;
	char key[16];
	u16 index = 0;
	int j = 0;

	memset(key, 0, sizeof(key));
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	data = (struct proc_data *)file->private_data;
	if ( !data->writelen ) return;
	
	if (data->wbuffer[0] >= '0' && data->wbuffer[0] <= '3' &&
	    (data->wbuffer[1] == ' ' || data->wbuffer[1] == '\n')) {
		index = data->wbuffer[0] - '0';
		if (data->wbuffer[1] == '\n') {
			set_wep_key(ai, index, 0, 0, 1);
			return;
		}
		j = 2;
	} else {
		printk(KERN_ERR "airo:  WepKey passed invalid key index\n");
		return;
	}

	for( i = 0; i < 16*3 && data->wbuffer[i+j]; i++ ) {
		switch(i%3) {
		case 0:
			key[i/3] = hexVal(data->wbuffer[i+j])<<4;
			break;
		case 1:
			key[i/3] |= hexVal(data->wbuffer[i+j]);
			break;
		}
	}
	set_wep_key(ai, index, key, i/3, 1);
}

static int proc_wepkey_open( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	char *ptr;
	WepKeyRid wkr;
	u16 lastindex;
	int j=0;
	int rc;
	
	MOD_INC_USE_COUNT;
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	memset(&wkr, 0, sizeof(wkr));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 180, GFP_KERNEL );
	memset(data->rbuffer, 0, 180);
	data->writelen = 0;
	data->maxwritelen = 80;
	data->wbuffer = kmalloc( 80, GFP_KERNEL );
	memset( data->wbuffer, 0, 80 );
	data->on_close = proc_wepkey_on_close;
	
	ptr = data->rbuffer;
	strcpy(ptr, "No wep keys\n");
	rc = PC4500_readrid(ai, RID_WEP_TEMP, &wkr, sizeof(wkr));
	if (rc == SUCCESS) do {
		lastindex = wkr.kindex;
		if (wkr.kindex == 0xffff) {
			j += sprintf(ptr+j, "Tx key = %d\n",
			             (int)wkr.mac[0]);
		} else {
		        j += sprintf(ptr+j, "Key %d set with length = %d\n",
                                     (int)wkr.kindex, (int)wkr.klen);
                }
	        PC4500_readrid(ai, RID_WEP_PERM, &wkr, sizeof(wkr));
	} while((lastindex != wkr.kindex) && (j < 180-30));

	data->readlen = strlen( data->rbuffer );
	return 0;
}

static int proc_SSID_open( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	int i;
	char *ptr;
	SsidRid SSID_rid;

	MOD_INC_USE_COUNT;
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 104, GFP_KERNEL );
	data->writelen = 0;
	data->maxwritelen = 33*3;
	data->wbuffer = kmalloc( 33*3, GFP_KERNEL );
	memset( data->wbuffer, 0, 33*3 );
	data->on_close = proc_SSID_on_close;
	
	PC4500_readrid(ai, RID_SSID, 
		       &SSID_rid, sizeof( SSID_rid ));
	ptr = data->rbuffer;
	for( i = 0; i < 3; i++ ) {
		int j;
		if ( !SSID_rid.ssids[i].len ) break;
		for( j = 0; j < 32 && 
			     j < le16_to_cpu(SSID_rid.ssids[i].len) && 
			     SSID_rid.ssids[i].ssid[j]; j++ ) {
			*ptr++ = SSID_rid.ssids[i].ssid[j]; 
		}
		*ptr++ = '\n';
	}
	*ptr = '\0';
	data->readlen = strlen( data->rbuffer );
	return 0;
}

static int proc_APList_open( struct inode *inode, struct file *file ) {
	struct proc_data *data;
	struct proc_dir_entry *dp = inode->u.generic_ip;
	struct net_device *dev = dp->data;
	struct airo_info *ai = (struct airo_info*)dev->priv;
	int i;
	char *ptr;
	APListRid APList_rid;

	MOD_INC_USE_COUNT;
	
	dp = (struct proc_dir_entry *) inode->u.generic_ip;
	
	file->private_data = kmalloc(sizeof(struct proc_data ), GFP_KERNEL);
	memset(file->private_data, 0, sizeof(struct proc_data));
	data = (struct proc_data *)file->private_data;
	data->rbuffer = kmalloc( 104, GFP_KERNEL );
	data->writelen = 0;
	data->maxwritelen = 4*6*3;
	data->wbuffer = kmalloc( data->maxwritelen, GFP_KERNEL );
	memset( data->wbuffer, 0, data->maxwritelen );
	data->on_close = proc_APList_on_close;
	
	PC4500_readrid(ai, RID_APLIST, 
		       &APList_rid, sizeof(APList_rid));
	ptr = data->rbuffer;
	for( i = 0; i < 4; i++ ) {
		// We end when we find a zero MAC
		if ( !*(int*)APList_rid.ap[i] &&
		     !*(int*)&APList_rid.ap[i][2]) break;
		ptr += sprintf(ptr, "%02x:%02x:%02x:%02x:%02x:%02x\n",
			       (int)APList_rid.ap[i][0],
			       (int)APList_rid.ap[i][1],
			       (int)APList_rid.ap[i][2],
			       (int)APList_rid.ap[i][3],
			       (int)APList_rid.ap[i][4],
			       (int)APList_rid.ap[i][5]);
	}
        if (i==0) ptr += sprintf(ptr, "Not using specific APs\n");

	*ptr = '\0';
	data->readlen = strlen( data->rbuffer );
	return 0;
}

#if (LINUX_VERSION_CODE > 0x20155)
static int proc_close( struct inode *inode, struct file *file ) 
#else
static void proc_close( struct inode *inode, struct file *file ) 
#endif
{
	struct proc_data *data = (struct proc_data *)file->private_data;
	if ( data->on_close != NULL ) data->on_close( inode, file );
	MOD_DEC_USE_COUNT;
	if ( data->rbuffer ) kfree( data->rbuffer );
	if ( data->wbuffer ) kfree( data->wbuffer );
	kfree( data );
#if (LINUX_VERSION_CODE > 0x20155)
	return 0;
#endif
}

static struct net_device_list {
	struct net_device *dev;
	struct net_device_list *next;
} *airo_devices = 0;


static void add_airo_dev( struct net_device *dev ) {
	struct net_device_list *node =
		(struct net_device_list*)kmalloc( sizeof( *node ), GFP_KERNEL );
	if ( !node ) {
		printk( KERN_ERR "airo_pci:  Out of memory\n" );
	} else {
	
		node->dev = dev;
		node->next = airo_devices;
		airo_devices = node;
	}
}

static void del_airo_dev( struct net_device *dev ) {
	struct net_device_list **p = &airo_devices;
	while( *p && ( (*p)->dev != dev ) )
		p = &(*p)->next;
	if ( *p && (*p)->dev == dev )
		*p = (*p)->next; 
}

int init_module( void )
{
  int i;
	
#if (LINUX_VERSION_CODE > 0x20155)
#if (LINUX_VERSION_CODE < 0x20311)
  airo_entry.ops->lookup = 
    proc_net->ops->lookup;
  airo_entry.ops->default_file_ops->readdir = 
    proc_net->ops->default_file_ops->readdir;
#endif
#else
  airo_entry.ops = proc_net.ops;
#endif
#if (LINUX_VERSION_CODE > 0x20311)
  airo_entry = create_proc_entry("aironet",
				 S_IFDIR | S_IRUGO,
				 proc_root_driver);
#else
  PROC_REGISTER( &proc_root, &airo_entry );
#endif
	
  for( i = 0; i < 4 && io[i] && irq[i]; i++ ) {
    printk( KERN_INFO "airo:  Trying to configure ISA adapter at irq=%d io=0x%x\n",
	    irq[i], io[i] );
    init_airo_card( irq[i], io[i] );
  }
  
#ifdef CONFIG_PCI
  if ( pcibios_present() ) {
    int i;
    printk( KERN_INFO "airo:  Probing for PCI adapters\n" );
    for( i = 0; card_ids[i].vendor; i++ ) {
#if (LINUX_VERSION_CODE > 0x20155)
      struct pci_dev *dev = 0;
      while((dev = pci_find_device(card_ids[i].vendor, card_ids[i].id,dev))) {
	init_airo_card( dev->irq, 
#if (LINUX_VERSION_CODE < 0x2030d)
			dev->base_address[2] & 
			PCI_BASE_ADDRESS_IO_MASK
#else
			dev->resource[2].start
#endif
			);
      }
#else
      int j;
      unsigned char bus, fun;
      
      /* We are running fast and loose here, it would be nice to fix it... */
      for( j = 0; j < 4; j++ ) {
	unsigned char irq;
	unsigned int io;
	if ( pcibios_find_device( card_ids[i].vendor, card_ids[i].id,j, &bus, &fun ) 
	     != PCIBIOS_SUCCESSFUL ) 
	  break;
	pcibios_read_config_byte( bus, fun, PCI_INTERRUPT_LINE, &irq );
	pcibios_read_config_dword( bus, fun, PCI_BASE_ADDRESS_2, &io );
	io &= PCI_BASE_ADDRESS_IO_MASK;
	init_airo_card( irq, io );
      }
#endif
    }
    printk( KERN_INFO "airo:  Finished probing for PCI adapters\n" );
  }
#endif
  return 0;
}

void cleanup_module( void )
{
  while( airo_devices ) {
    printk( KERN_INFO "airo: Unregistering %s\n", airo_devices->dev->name );
    stop_airo_card( airo_devices->dev,1 );
  }
#if (LINUX_VERSION_CODE < 0x20311)
  PROC_UNREGISTER( &proc_root, &airo_entry );
#else
  remove_proc_entry("aironet", proc_root_driver);
#endif
}

#ifdef WIRELESS_EXT
/*
 * Initial Wireless Extension code for Aironet driver by :
 *	Jean Tourrilhes <jt@hpl.hp.com> - HPL - 17 November 00
 */
#ifndef IW_ENCODE_NOKEY
#define IW_ENCODE_NOKEY         0x0800  /* Key is write only, so not present */
#define IW_ENCODE_MODE  (IW_ENCODE_DISABLED | IW_ENCODE_RESTRICTED | IW_ENCODE_OPEN)
#endif 
#endif 

#ifdef WIRELESS_EXT
int getQuality(StatusRid *statusRid, CapabilityRid *capsRid)
{
  int quality = 0;

  if ((statusRid->mode&0x3f) == 0x3f) {
    if (capsRid->hardCap & 0x0008) {
      if (!memcmp(capsRid->prodName, "350", 3)) {
        if (statusRid->signalQuality > 0xb0) {
          quality = 0;
        } else if (statusRid->signalQuality < 0x10) {
          quality = 100;
        } else {
          quality = ((0xb0-statusRid->signalQuality)*100)/0xa0;
        }
      } else {
        if (statusRid->signalQuality > 0x20) {
          quality = 0;
        } else {
          quality = ((0x20-statusRid->signalQuality)*100)/0x20;
        }
      }
    } else {
      quality = 0;
    }
  }
  return(quality);
}
#endif

/*
 * This defines the configuration part of the Wireless Extensions
 * Note : irq and spinlock protection will occur in the subroutines
 *
 * TODO :
 *	o Check input value more carefully and fill correct values in range
 *	o Implement : POWER, SPY, APLIST
 *	o Optimise when adapter is closed (aggregate changes, commit later)
 *	o Test and shakeout the bugs (if any)
 *
 * Jean II
 *
 * Javier Achirica did a great job of merging code from the unnamed CISCO
 * developer that added support for flashing the card.
 */
static int airo_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	int rc = 0;
	int status;
#ifdef WIRELESS_EXT
	struct airo_info *local = (struct airo_info*) dev->priv;
	struct iwreq *wrq = (struct iwreq *) rq;
	ConfigRid config;		/* Configuration info */
	CapabilityRid cap_rid;		/* Card capability info */
	StatusRid status_rid;		/* Card status info */
#if WIRELESS_EXT > 9
        int     i;
#endif

#ifdef CISCO_EXT
       if (cmd != AIROIDIFC && cmd != AIROIOCTL) {
           if (local->flags & FLASHING) {
               return(-EBUSY);
           }
       }
#endif
#ifdef CISCO_EXT
	if (cmd != SIOCGIWPRIV && cmd != AIROIOCTL && cmd != AIROIDIFC)
#endif 
	{
		/* If the command read some stuff, we better get it out of
		 * the card first... */
		if(IW_IS_GET(cmd) || (cmd == SIOCSIWRATE)) {
			PC4500_readrid(local, RID_STATUS, &status_rid,
				       sizeof(status_rid));
			PC4500_readrid(local, RID_CAPABILITIES, &cap_rid,
				       sizeof(cap_rid));
		}
		/* Get config in all cases, because SET will just modify it */
		PC4500_readrid(local, RID_ACTUALCONFIG, &config,
			       sizeof(config));
	}
#endif 

	switch (cmd) {
#ifdef WIRELESS_EXT
	// Get name
	case SIOCGIWNAME:
		strcpy(wrq->u.name, "IEEE 802.11-DS");
		break;

	// Set frequency/channel
	case SIOCSIWFREQ:
		/* If setting by frequency, convert to a channel */
		if((wrq->u.freq.e == 1) &&
		   (wrq->u.freq.m >= (int) 2.412e8) &&
		   (wrq->u.freq.m <= (int) 2.487e8)) {
			int f = wrq->u.freq.m / 100000;
			int c = 0;
			while((c < 14) && (f != frequency_list[c]))
				c++;
			/* Hack to fall through... */
			wrq->u.freq.e = 0;
			wrq->u.freq.m = c + 1;
		}
		/* Setting by channel number */
		if((wrq->u.freq.m > 1000) || (wrq->u.freq.e > 0))
			rc = -EOPNOTSUPP;
		else {
			int channel = wrq->u.freq.m;
			/* We should do a better check than that,
			 * based on the card capability !!! */
			if((channel < 1) || (channel > 16)) {
				printk(KERN_DEBUG "%s: New channel value of %d is invalid!\n", dev->name, wrq->u.freq.m);
				rc = -EINVAL;
			} else {
				/* Yes ! We can set it !!! */
				config.channelSet = (u16)cpu_to_le16(channel);
				local->need_commit = 1;
			}
		}
		break;

	// Get frequency/channel
	case SIOCGIWFREQ:
#ifdef WEXT_USECHANNELS
		wrq->u.freq.m = ((int) le16_to_cpu(status_rid.channel) + 1);
		wrq->u.freq.e = 0;
#else
		{
			int f = (int) le16_to_cpu(status_rid.channel);
			wrq->u.freq.m = frequency_list[f] * 100000;
			wrq->u.freq.e = 1;
		}
#endif
		break;

	// Set desired network name (ESSID)
	case SIOCSIWESSID:
		if (wrq->u.data.pointer) {
			char	essid[IW_ESSID_MAX_SIZE + 1];
			SsidRid SSID_rid;		/* SSIDs */

			/* Reload the list of current SSID */
			PC4500_readrid(local, RID_SSID, 
				       &SSID_rid, sizeof(SSID_rid));

			/* Check if we asked for `any' */
			if(wrq->u.data.flags == 0) {
				/* Just send an empty SSID list */
				memset(&SSID_rid, 0, sizeof(SSID_rid));
			} else {
				int	index = (wrq->u.data.flags &
						 IW_ENCODE_INDEX) - 1;

				/* Check the size of the string */
				if(wrq->u.data.length > IW_ESSID_MAX_SIZE+1) {
					rc = -E2BIG;
					break;
				}
				/* Check if index is valid */
				if((index < 0) || (index >= 4)) {
					rc = -EINVAL;
					break;
				}

				/* Set the SSID */
				memset(essid, 0, sizeof(essid));
				copy_from_user(essid,
					       wrq->u.data.pointer,
					       wrq->u.data.length);
				memcpy(SSID_rid.ssids[index].ssid, essid,
				       sizeof(essid) - 1);
				SSID_rid.ssids[index].len = cpu_to_le16(wrq->u.data.length - 1);
			}
			/* Write it to the card */
			do_writerid(local, RID_SSID,
				    &SSID_rid, sizeof(SSID_rid));
		}
		break;

	// Get current network name (ESSID)
	case SIOCGIWESSID:
		if (wrq->u.data.pointer) {
			char essid[IW_ESSID_MAX_SIZE + 1];

			/* Note : if wrq->u.data.flags != 0, we should
			 * get the relevant SSID from the SSID list... */

			/* Get the current SSID */
			memcpy(essid, status_rid.SSID, status_rid.SSIDlen);
			essid[status_rid.SSIDlen] = '\0';
			/* If none, we may want to get the one that was set */

			/* Push it out ! */
			wrq->u.data.length = strlen(essid) + 1;
			wrq->u.data.flags = 1; /* active */
			copy_to_user(wrq->u.data.pointer,
				     essid, sizeof(essid));
		}
		break;

	case SIOCSIWAP:
		if (wrq->u.ap_addr.sa_family != ARPHRD_ETHER)
			rc = -EINVAL;
		else {
			APListRid APList_rid;

			memset(&APList_rid, 0, sizeof(APList_rid));
			APList_rid.len = sizeof(APList_rid);
			memcpy(APList_rid.ap[0], wrq->u.ap_addr.sa_data, 6);
			do_writerid(local, RID_APLIST, &APList_rid, sizeof(APList_rid));
			local->need_commit = 1;
		}
		break;

	// Get current Access Point (BSSID)
	case SIOCGIWAP:
		/* Tentative. This seems to work, wow, I'm lucky !!! */
		memcpy(wrq->u.ap_addr.sa_data, status_rid.bssid[0], 6);
		wrq->u.ap_addr.sa_family = ARPHRD_ETHER;
		break;

	// Set desired station name
	case SIOCSIWNICKN:
		if (wrq->u.data.pointer) {
			char	name[16 + 1];

			/* Check the size of the string */
			if(wrq->u.data.length > 16 + 1) {
				rc = -E2BIG;
				break;
			}
			memset(name, 0, sizeof(name));
			copy_from_user(name, wrq->u.data.pointer, wrq->u.data.length);
			memcpy(config.nodeName, name, 16);
			local->need_commit = 1;
		}
		break;

	// Get current station name
	case SIOCGIWNICKN:
		if (wrq->u.data.pointer) {
			char name[IW_ESSID_MAX_SIZE + 1];

			strncpy(name, config.nodeName, 16);
			name[16] = '\0';
			wrq->u.data.length = strlen(name) + 1;
			copy_to_user(wrq->u.data.pointer, name, sizeof(name));
		}
		break;

	// Set the desired bit-rate
	case SIOCSIWRATE:
	{
		/* First : get a valid bit rate value */
		u8	brate = 0;
		int	i;

		/* Which type of value ? */
		if((wrq->u.bitrate.value < 8) &&
		   (wrq->u.bitrate.value >= 0)) {
			/* Setting by rate index */
			/* Find value in the magic rate table */
			brate = cap_rid.supportedRates[wrq->u.bitrate.value];
		} else {
			/* Setting by frequency value */
			u8	normvalue = (u8) (wrq->u.bitrate.value/500000);

			/* Check if rate is valid */
			for(i = 0 ; i < 8 ; i++) {
				if(normvalue == cap_rid.supportedRates[i]) {
					brate = normvalue;
					break;
				}
			}
		}
		/* -1 designed the max rate (mostly auto mode) */
		if(wrq->u.bitrate.value == -1) {
			/* Get the highest available rate */
			for(i = 0 ; i < 8 ; i++) {
				if(cap_rid.supportedRates[i] == 0)
					break;
			}
			if(i != 0)
				brate = cap_rid.supportedRates[i - 1];
		}
		/* Check that it is valid */
		if(brate == 0) {
			rc = -EINVAL;
			break;
		}

		/* Now, check if we want a fixed or auto value */
		if(wrq->u.bitrate.fixed == 0) {
			/* Fill all the rates up to this max rate */
			memset(config.rates, 0, 8);
			for(i = 0 ; i < 8 ; i++) {
				config.rates[i] = cap_rid.supportedRates[i];
				if(config.rates[i] == brate)
					break;
			}
			local->need_commit = 1;
		} else {
			/* Fixed mode */
			/* One rate, fixed */
			memset(config.rates, 0, 8);
			config.rates[0] = brate;
			local->need_commit = 1;
		}
		break;
	}

	// Get the current bit-rate
	case SIOCGIWRATE:
		{
			int brate = le16_to_cpu(status_rid.currentXmitRate);
			wrq->u.bitrate.value = brate * 500000;
			/* If more than one rate, set auto */
			wrq->u.rts.fixed = (config.rates[1] == 0);
		}
		break;

	// Set the desired RTS threshold
	case SIOCSIWRTS:
		{
			int rthr = wrq->u.rts.value;
			if(wrq->u.rts.disabled)
				rthr = 2312;
			if((rthr < 0) || (rthr > 2312)) {
				rc = -EINVAL;
			} else {
				config.rtsThres = (u16)cpu_to_le16(rthr);
				local->need_commit = 1;
			}
		}
		break;

	// Get the current RTS threshold
	case SIOCGIWRTS:
		wrq->u.rts.value = le16_to_cpu(config.rtsThres);
		wrq->u.rts.disabled = (wrq->u.rts.value >= 2312);
		wrq->u.rts.fixed = 1;
		break;

	// Set the desired fragmentation threshold
	case SIOCSIWFRAG:
		{
			int fthr = wrq->u.frag.value;
			if(wrq->u.frag.disabled)
				fthr = 2312;
			if((fthr < 256) || (fthr > 2312)) {
				rc = -EINVAL;
			} else {
				fthr &= ~0x1;	/* Get an even value */
				config.fragThresh = (u16)cpu_to_le16(fthr);
				local->need_commit = 1;
			}
		}
		break;

	// Get the current fragmentation threshold
	case SIOCGIWFRAG:
		wrq->u.frag.value = le16_to_cpu(config.fragThresh);
		wrq->u.frag.disabled = (wrq->u.frag.value >= 2312);
		wrq->u.frag.fixed = 1;
		break;

	// Set mode of operation
	case SIOCSIWMODE:
		switch(wrq->u.mode) {
		case IW_MODE_ADHOC:
			config.opmode = cpu_to_le16(MODE_STA_IBSS);
			local->need_commit = 1;
			break;
		case IW_MODE_INFRA:
			config.opmode = cpu_to_le16(MODE_STA_ESS);
			local->need_commit = 1;
			break;
		case IW_MODE_MASTER:
			config.opmode = cpu_to_le16(MODE_AP);
			local->need_commit = 1;
			break;
		case IW_MODE_REPEAT:
			config.opmode = cpu_to_le16(MODE_AP_RPTR);
			local->need_commit = 1;
			break;
		default:
			rc = -EINVAL;
		}
		break;

	// Get mode of operation
	case SIOCGIWMODE:
		/* If not managed, assume it's ad-hoc */
		switch (le16_to_cpu(config.opmode) & 0xFF) {
		case MODE_STA_ESS:
			wrq->u.mode = IW_MODE_INFRA;
			break;
		case MODE_AP:
			wrq->u.mode = IW_MODE_MASTER;
			break;
		case MODE_AP_RPTR:
			wrq->u.mode = IW_MODE_REPEAT;
			break;
		default:
			wrq->u.mode = IW_MODE_ADHOC;
		}
		break;

	// Set WEP keys and mode
	case SIOCSIWENCODE:
		/* Is WEP supported ? */
		if(!(cap_rid.softCap & 2)) {
			rc = -EOPNOTSUPP;
			break;
		}
		/* Basic checking: do we have a key to set ? */
		if (wrq->u.encoding.pointer != (caddr_t) 0) {
			wep_key_t key;
			int index = (wrq->u.encoding.flags & IW_ENCODE_INDEX) - 1;
			int current_index = get_wep_key(local, 0xffff);
			/* Check the size of the key */
			if (wrq->u.encoding.length > MAX_KEY_SIZE) {
				rc = -EINVAL;
				break;
			}
			/* Check the index (none -> use current) */
			if ((index < 0) || (index >= MAX_KEYS))
				index = current_index;
			/* Set the length */
			if (wrq->u.encoding.length > MIN_KEY_SIZE)
				key.len = MAX_KEY_SIZE;
			else
				if (wrq->u.encoding.length > 0)
					key.len = MIN_KEY_SIZE;
				else
					/* Disable the key */
					key.len = 0;
			/* Check if the key is not marked as invalid */
			if(!(wrq->u.encoding.flags & IW_ENCODE_NOKEY)) {
				/* Cleanup */
				memset(key.key, 0, MAX_KEY_SIZE);
				/* Copy the key in the driver */
				if(copy_from_user(key.key,
						  wrq->u.encoding.pointer,
						  wrq->u.encoding.length)) {
					key.len = 0;
					rc = -EFAULT;
					break;
				}
				/* Send the key to the card */
				set_wep_key(local, index, key.key,
					    key.len, 1);
			}
			/* WE specify that if a valid key is set, encryption
			 * should be enabled (user may turn it off later)
			 * This is also how "iwconfig ethX key on" works */
			if((index == current_index) && (key.len > 0) &&
			   (config.authType == AUTH_OPEN)) {
				config.authType = AUTH_ENCRYPT;
				local->need_commit = 1;
			}
		} else {
			/* Do we want to just set the transmit key index ? */
			int index = (wrq->u.encoding.flags & IW_ENCODE_INDEX) - 1;
			if ((index >= 0) && (index < MAX_KEYS)) {
				set_wep_key(local, index, 0, 0, 1);
			} else
				/* Don't complain if only change the mode */
				if(!wrq->u.encoding.flags & IW_ENCODE_MODE) {
					rc = -EINVAL;
					break;
				}
		}
		/* Read the flags */
		if(wrq->u.encoding.flags & IW_ENCODE_DISABLED)
			config.authType = AUTH_OPEN;	// disable encryption
		if(wrq->u.encoding.flags & IW_ENCODE_RESTRICTED)
			config.authType = AUTH_SHAREDKEY;	// Only Both
		if(wrq->u.encoding.flags & IW_ENCODE_OPEN)
			config.authType = AUTH_ENCRYPT;	// Only Wep
		/* Commit the changes if needed */
		if(wrq->u.encoding.flags & IW_ENCODE_MODE)
			local->need_commit = 1;
		break;

	// Get the WEP keys and mode
	case SIOCGIWENCODE:
		/* Is it supported ? */
		if(!(cap_rid.softCap & 2)) {
			rc = -EOPNOTSUPP;
			break;
		}
#ifndef OPENACCESS 
		// Only super-user can see WEP key

		if (!capable(CAP_NET_ADMIN)) {
			rc = -EPERM;
			break;
		}
#endif
		// Basic checking...
		if (wrq->u.encoding.pointer != (caddr_t) 0) {
			char zeros[16];
			int index = (wrq->u.encoding.flags & IW_ENCODE_INDEX) - 1;

			memset(zeros,0, sizeof(zeros));
			/* Check encryption mode */
			wrq->u.encoding.flags = IW_ENCODE_NOKEY;
			/* Is WEP enabled ??? */
			switch(config.authType)	{
			case AUTH_ENCRYPT:
				wrq->u.encoding.flags |= IW_ENCODE_OPEN;
				break;
			case AUTH_SHAREDKEY:
				wrq->u.encoding.flags |= IW_ENCODE_RESTRICTED;
				break;
			default:
			case AUTH_OPEN:
				wrq->u.encoding.flags |= IW_ENCODE_DISABLED;
				break;
			}

			/* Which key do we want ? -1 -> tx index */
			if((index < 0) || (index >= MAX_KEYS))
				index = get_wep_key(local, 0xffff);
			wrq->u.encoding.flags |= index + 1;
			/* Copy the key to the user buffer */
			wrq->u.encoding.length = get_wep_key(local, index);
			if (wrq->u.encoding.length > 16) {
				wrq->u.encoding.length=0;
			}
			if(copy_to_user(wrq->u.encoding.pointer,
					zeros,
					wrq->u.encoding.length))
				rc = -EFAULT;
		}
		break;

#if WIRELESS_EXT > 9
	// Get the current Tx-Power
	case SIOCGIWTXPOW:
		wrq->u.txpower.value = le16_to_cpu(config.txPower);
		wrq->u.txpower.fixed = 1;	/* No power control */
		wrq->u.txpower.disabled = (local->flags & FLAG_RADIO_OFF);
		wrq->u.txpower.flags = IW_TXPOW_MWATT;
		break;
	case SIOCSIWTXPOW:
		if (wrq->u.txpower.disabled) {
			local->flags |= FLAG_RADIO_OFF;
			local->need_commit = 1;
			break;
		}
		if (wrq->u.txpower.flags != IW_TXPOW_MWATT) {
			rc = -EINVAL;
			break;
		}
		local->flags &= ~FLAG_RADIO_OFF;
		rc = -EINVAL;
		for (i = 0; cap_rid.txPowerLevels[i] && (i < 8); i++)
			if ((wrq->u.txpower.value=cap_rid.txPowerLevels[i])) {
				config.txPower = wrq->u.txpower.value;
				local->need_commit = 1;
				rc = 0;
				break;
			}
		break;
#endif 

	// Get range of parameters
	case SIOCGIWRANGE:
		if (wrq->u.data.pointer) {
			struct iw_range range;
			int		i;
			int		k;

			rc = verify_area(VERIFY_WRITE, wrq->u.data.pointer, sizeof(struct iw_range));
			if (rc)
				break;
			wrq->u.data.length = sizeof(range);
			/* Should adapt depending on max rate */
			range.throughput = 1.6 * 1024 * 1024;
			range.min_nwid = 0x0000;
			range.max_nwid = 0x0000;
			range.num_channels = 14;
			/* Should be based on cap_rid.country to give only
			 * what the current card support */
			k = 0;
			for(i = 0; i < 14; i++) {
				range.freq[k].i = i + 1; /* List index */
				range.freq[k].m = frequency_list[i] * 100000;
				range.freq[k++].e = 1;	/* Values in table in MHz -> * 10^5 * 10 */
			}
			range.num_frequency = k;

			/* Hum... Should put the right values there */
			range.max_qual.qual = 0xFF;
			range.max_qual.level = 100;
			range.max_qual.noise = 0;

			for(i = 0 ; i < 8 ; i++) {
				range.bitrate[i] = cap_rid.supportedRates[i] * 500000;
				if(range.bitrate[i] == 0)
					break;
			}
			range.num_bitrates = i;

			range.min_rts = 0;
			range.max_rts = 2312;
			range.min_frag = 256;
			range.max_frag = 2312;

			if(cap_rid.softCap & 2) {
				// WEP: RC4 40 bits
				range.encoding_size[0] = 5;
				// RC4 ~128 bits
				if (cap_rid.softCap & 0x100) {
					range.encoding_size[1] = 13;
					range.num_encoding_sizes = 2;
				} else
					range.num_encoding_sizes = 1;
				range.max_encoding_tokens = 4;	// 4 keys
			} else {
				range.num_encoding_sizes = 0;
				range.max_encoding_tokens = 0;
			}
#if WIRELESS_EXT > 9
			range.min_pmp = 0;
			range.max_pmp = 5000000;	/* 5 secs */
			range.min_pmt = 0;
			range.max_pmt = 65535 * 1024;	/* ??? */
			range.pmp_flags = IW_POWER_PERIOD;
			range.pmt_flags = IW_POWER_TIMEOUT;
			range.pm_capa = IW_POWER_PERIOD | IW_POWER_TIMEOUT;

			/* Transmit Power - values are in mW */
			for(i = 0 ; i < 8 ; i++) {
				range.txpower[i] = cap_rid.txPowerLevels[i];
				if(range.txpower[i] == 0)
					break;
			}
			range.num_txpower = i;
			range.txpower_capa = IW_TXPOW_MWATT;
#endif 

			copy_to_user(wrq->u.data.pointer, &range, sizeof(struct iw_range));
		}
		break;

	case SIOCGIWPOWER:
		{
			int mode = le16_to_cpu(config.powerSaveMode);
			if ((wrq->u.power.disabled = (mode == POWERSAVE_CAM)))
				break;
			if ((wrq->u.power.flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
				wrq->u.power.value = (int)le16_to_cpu(config.fastListenDelay) * 1024;
				wrq->u.power.flags = IW_POWER_TIMEOUT;
			} else {
				wrq->u.power.value = (int)le16_to_cpu(config.fastListenInterval) * 1024;
				wrq->u.power.flags = IW_POWER_PERIOD;
			}
			if ((le16_to_cpu(config.rmode) & 0xFF) == RXMODE_ADDR)
				wrq->u.power.flags |= IW_POWER_UNICAST_R;
			else
				wrq->u.power.flags |= IW_POWER_ALL_R;
		}
		break;

	case SIOCSIWPOWER: 
		if (wrq->u.power.disabled) {
			config.powerSaveMode = cpu_to_le16(POWERSAVE_CAM);
			config.rmode = cpu_to_le16(RXMODE_BC_MC_ADDR);
			local->need_commit = 1;
			break;
		}
		if ((wrq->u.power.flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
			config.fastListenDelay = cpu_to_le16((wrq->u.power.value + 500) / 1024);
			config.powerSaveMode = cpu_to_le16(POWERSAVE_PSPCAM);
			local->need_commit = 1;
		} else if ((wrq->u.power.flags & IW_POWER_TYPE) == IW_POWER_PERIOD) {
			config.fastListenInterval = config.listenInterval = cpu_to_le16((wrq->u.power.value + 500) / 1024);
			config.powerSaveMode = cpu_to_le16(POWERSAVE_PSPCAM);
			local->need_commit = 1;
		}
		switch (wrq->u.power.flags & IW_POWER_MODE) {
		case IW_POWER_UNICAST_R:
			config.rmode = cpu_to_le16(RXMODE_ADDR);
			local->need_commit = 1;
			break;
		case IW_POWER_ALL_R:
			config.rmode = cpu_to_le16(RXMODE_BC_MC_ADDR);
			local->need_commit = 1;
			break;
		case IW_POWER_ON:
			break;
		default:
			local->need_commit = 0;
			rc = -EINVAL;
		}
		break;

	case SIOCGIWSENS:
		wrq->u.sens.value = le16_to_cpu(config.rssiThreshold);
		wrq->u.sens.disabled = (wrq->u.sens.value == 0);
		wrq->u.sens.fixed = 1;
		break;

	case SIOCSIWSENS:
		config.rssiThreshold = cpu_to_le16(wrq->u.sens.disabled ? RSSI_DEFAULT : wrq->u.sens.value);
		local->need_commit = 1;
		break;

	case SIOCGIWAPLIST:
		if (wrq->u.data.pointer) {
			int i;
			struct sockaddr s[4];
			for (i = 0; i < 4; i++) {
				memcpy(s[i].sa_data, status_rid.bssid[i], 6);
				s[i].sa_family = ARPHRD_ETHER;
			}
			if (copy_to_user(wrq->u.data.pointer, &s, sizeof(s)))
				rc = -EFAULT;
		}
		break;

#ifdef WIRELESS_SPY
	// Set the spy list
	case SIOCSIWSPY:
		if (wrq->u.data.length > IW_MAX_SPY)
		{
			rc = -E2BIG;
			break;
		}
		local->spy_number = wrq->u.data.length;
		if (local->spy_number > 0)
		{
			struct sockaddr address[IW_MAX_SPY];
			int i;
			rc = verify_area(VERIFY_READ, wrq->u.data.pointer, sizeof(struct sockaddr) * local->spy_number);
			if (rc)
				break;
			copy_from_user(address, wrq->u.data.pointer, sizeof(struct sockaddr) * local->spy_number);
			for (i=0; i<local->spy_number; i++)
				memcpy(local->spy_address[i], address[i].sa_data, 6);
			memset(local->spy_stat, 0, sizeof(struct iw_quality) * IW_MAX_SPY);
		}
		break;

	// Get the spy list
	case SIOCGIWSPY:
		wrq->u.data.length = local->spy_number;
		if ((local->spy_number > 0) && (wrq->u.data.pointer))
		{
			struct sockaddr address[IW_MAX_SPY];
			int i;
			rc = verify_area(VERIFY_WRITE, wrq->u.data.pointer, (sizeof(struct iw_quality)+sizeof(struct sockaddr)) * IW_MAX_SPY);
			if (rc)
				break;
			for (i=0; i<local->spy_number; i++)
			{
				memcpy(address[i].sa_data, local->spy_address[i], 6);
				address[i].sa_family = AF_UNIX;
			}
			copy_to_user(wrq->u.data.pointer, address, sizeof(struct sockaddr) * local->spy_number);
			copy_to_user(wrq->u.data.pointer + (sizeof(struct sockaddr)*local->spy_number), local->spy_stat, sizeof(struct iw_quality) * local->spy_number);
			for (i=0; i<local->spy_number; i++)
				local->spy_stat[i].updated = 0;
		}
		break;
#endif 

#ifdef CISCO_EXT
	case SIOCGIWPRIV:
		if(wrq->u.data.pointer)
		{
			struct iw_priv_args   priv[] =
			{ /* cmd, set_args, get_args, name */
			{ AIROIOCTL, IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | sizeof (aironet_ioctl), IW_PRIV_TYPE_BYTE | 2047, "airoioctl" },
			{ AIROIDIFC, IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | sizeof (aironet_ioctl), IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, "airoidifc" },
                        { SIOCIWFIRSTPRIV,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 3,
                          0,
                          "enableLeap" },
                        { SIOCIWFIRSTPRIV+2,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 3,
                          0,
                          "powerSaving" },
                        { SIOCIWFIRSTPRIV+3,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 3,
                          "getPowerSaving" },
                        { SIOCIWFIRSTPRIV+5,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 10,
                          "getAssocStatus" },
                        { SIOCIWFIRSTPRIV+7,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 10,
                          "getFWVersion" },
                        { SIOCIWFIRSTPRIV+9,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 10,
                          "getDrvrVersion" },
                        { SIOCIWFIRSTPRIV+11,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 16,
                          "getProductName" },
#ifdef INCLUDE_RFMONITOR
                        { SIOCIWFIRSTPRIV+12,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 3,
                          0,
                          "setRFMonitor" },
                        { SIOCIWFIRSTPRIV+13,
                          0,
                          IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 3,
                          "getRFMonitor" },
#endif
			};

			/* Set the number of ioctl available */
#ifdef INCLUDE_RFMONITOR
			wrq->u.data.length = 11;
#else
			wrq->u.data.length = 9;
#endif

			/* Copy structure to the user buffer */
			if(copy_to_user(wrq->u.data.pointer, (u_char *) priv,
					sizeof(priv)))
				rc = -EFAULT;
		}
		break;
#endif 
#endif 
#ifdef WIRELESS_EXT
       case SIOCIWFIRSTPRIV:	/* enable leap */
         PC4500_readrid(local, 0xff10, &config, sizeof(ConfigRid));
         rc = 0;
         if (!strcmp(wrq->u.name, "yes")) {
           config.authType = 0x1101;
         } else if (!strcmp(wrq->u.name, "no")) {
           config.authType = 0x0001;
         } else {
           rc = -EINVAL;
         }
         if (rc == 0) {
           PC4500_writerid(local, RID_CONFIG, &config, sizeof(ConfigRid));
         }
         break;
       case SIOCIWFIRSTPRIV+2:	/* set power saving mode */
         PC4500_readrid(local, 0xff10, &config, sizeof(ConfigRid));
         rc = 0;
         if (!strcmp(wrq->u.name, "cam")) {
         config.powerSaveMode = POWERSAVE_CAM;
         } else if (!strcmp(wrq->u.name, "psp")) {
           config.powerSaveMode = POWERSAVE_PSP;
         } else if (!strcmp(wrq->u.name, "max")) {
           config.powerSaveMode = POWERSAVE_PSPCAM;
         } else {
           rc = -EINVAL;
         }
         if (rc == 0) {
           PC4500_writerid(local, RID_CONFIG, &config, sizeof(ConfigRid));
         }
         break;
       case SIOCIWFIRSTPRIV+3:	/* get power saving mode */
         PC4500_readrid(local, 0xff20, &config, sizeof(ConfigRid));
         if (config.powerSaveMode == POWERSAVE_CAM) {
           strcpy(wrq->u.name, "cam");
         } else if (config.powerSaveMode == POWERSAVE_PSP) {
           strcpy(wrq->u.name, "psp");
         } else if (config.powerSaveMode == POWERSAVE_PSPCAM) {
           strcpy(wrq->u.name, "max");
         }
         rc = 0;
         break;
       case SIOCIWFIRSTPRIV+5:	/* association status */
         PC4500_readrid(local, 0xff50, &status_rid, sizeof(StatusRid));
         if (!status_rid.mode) {
           strcpy(wrq->u.name, "disabled");
         } else if ((status_rid.mode & 0x3f) != 0x3f) {
           strcpy(wrq->u.name, "enabled");
         } else {
           strcpy(wrq->u.name, "associated");
         }
         rc = 0;
         break;
       case SIOCIWFIRSTPRIV+7:	/* FW version */
         PC4500_readrid(local, 0xff00, &cap_rid, sizeof(CapabilityRid));
         strcpy(wrq->u.name, cap_rid.prodVer);
         rc = 0;
         break;
       case SIOCIWFIRSTPRIV+9:	/* driver version */
         strcpy(wrq->u.name, swversion);
         rc = 0;
         break;
       case SIOCIWFIRSTPRIV+11:	/* product name */
         PC4500_readrid(local, 0xff00, &cap_rid, sizeof(CapabilityRid));
         strcpy(wrq->u.name, cap_rid.prodName);
         rc = 0;
         break;
#ifdef INCLUDE_RFMONITOR
       case SIOCIWFIRSTPRIV+12:	/* rf monitor */
         rc = 0;
         if (!strcmp(wrq->u.name, "on")) {
           local->flags |= FLAG_RFMONITOR;
         } else if (!strcmp(wrq->u.name, "off")) {
           local->flags &= ~FLAG_RFMONITOR;
         } else {
           rc = -EINVAL;
         }
         if (rc == 0) {
           set_rfmonitor(local);
         }
         break;
       case SIOCIWFIRSTPRIV+13:	/* rf monitor */
         PC4500_readrid(local, 0xff20, &config, sizeof(ConfigRid));
         if (config.rmode == (0x300 | RXMODE_RFMON_ANYBSS)) {
           strcpy(wrq->u.name, "on");
         } else {
           strcpy(wrq->u.name, "off");
         }
         rc = 0;
         break;
#endif
#endif
#ifdef CISCO_EXT
	case AIROIDIFC:
		{
			int val = AIROMAGIC;
			aironet_ioctl com;
			copy_from_user(&com,rq->ifr_data,sizeof(com));
			if (copy_to_user(com.data,(char *)&val,sizeof(val)))
				rc = -EFAULT;
		}
		break;
  
	case AIROIOCTL:
	   /* Get the command struct and hand it off for evaluation by 
	    * the proper subfunction
	    */
		{
			aironet_ioctl com;
			copy_from_user(&com,rq->ifr_data,sizeof(com));
			
	   /* Seperate R/W functions bracket legality here
	    */
			status = com.command;

                        if( status == AIRORSWVERSION) {
                          if (copy_to_user(com.data, swversion, MIN (com.length, strlen(swversion)+1))) {
                            rc = -EFAULT;
                          } else {
                            rc = 0;
                          }
                        } else if( status == AIRORRID) {
                          rc = readrids(dev,&com);

			} else if ( status >= AIROGCAP && com.command <= AIROGMICSTATS )
				rc = readrids(dev,&com);
			else if ( status >= AIROPCAP && status <= (AIROPLEAPUSR+2) )
				rc = writerids(dev,&com);
			else if ( status >= AIROFLSHRST && status <= AIRORESTART )
				rc = flashcard(dev,&com);
			else
				rc = -EINVAL;      /* Bad command in ioctl */
		}
		break;
#endif 

		// All other calls are currently unsupported
		default:
			rc = -EOPNOTSUPP;
	}

#ifdef WIRELESS_EXT
	/* Some of the "SET" function may have modified some of the
	 * parameters. It's now time to commit them in the card */
	if(local->need_commit) {
		/* A classical optimisation here is to not commit any change
		 * if the card is not "opened". This is what we do in
		 * wvlan_cs (see for details).
		 * For that, we would need to have the config RID saved in
		 * the airo_info struct and make sure to not re-read it if
		 * local->need_commit != 0. Then, you need to patch "open"
		 * to do the final commit of all parameters...
		 * Jean II */
		Cmd command;
		Resp rsp;

		command.cmd = MAC_DISABLE; // disable in case already enabled
		issuecommand(local, &command, &rsp);

		local->config = config;	/* ???? config is local !!! */
		checkThrottle(&config);
		PC4500_writerid(local, RID_CONFIG, &config,
				sizeof(config));
		enable_MAC(local, &rsp);

		local->need_commit = 0;
	}
#endif 

	return(rc);
}

#ifdef WIRELESS_EXT
/*
 * Get the Wireless stats out of the driver
 * Note : irq and spinlock protection will occur in the subroutines
 *
 * TODO :
 *	o Check if work in Ad-Hoc mode (otherwise, use SPY, as in wvlan_cs)
 *	o Find the noise level
 *	o Convert values to dBm
 *	o Fill out discard.misc with something interesting
 *
 * Jean
 */
struct iw_statistics *airo_get_wireless_stats(struct net_device *dev)
{
	struct airo_info *local = (struct airo_info*) dev->priv;
	StatusRid status_rid;
	CapabilityRid caps_rid;
	char buffer[1024];
	int *vals = (int*)&buffer[4];
        int value;

        if (local->flags & FLASHING) {
          memset(&local->wstats, 0, sizeof(struct iw_statistics));
          return (&local->wstats);
        }

	/* Get stats out of the card */
	PC4500_readrid(local, 0xff00, &caps_rid,
		       sizeof(caps_rid));
	PC4500_readrid(local, RID_STATUS, &status_rid,
		       sizeof(status_rid));
	PC4500_readrid(local, RID_STATS, buffer,
		       sizeof(buffer));

	/* The status */
	local->wstats.status = le16_to_cpu(status_rid.mode);

	/* Signal quality and co. But where is the noise level ??? */
        value = getQuality(&status_rid, &caps_rid);
	local->wstats.qual.qual = le16_to_cpu(value);
        if ((status_rid.mode&0x3f) == 0x3f) {
	  local->wstats.qual.level = le16_to_cpu(status_rid.normalizedSignalStrength);
        } else {
	  local->wstats.qual.level = 0;
        }
	local->wstats.qual.noise = le16_to_cpu(status_rid._reserved[3]);
	local->wstats.qual.updated = 3;

	/* Packets discarded in the wireless adapter due to wireless
	 * specific problems */
	local->wstats.discard.nwid = le32_to_cpu(vals[56]) + le32_to_cpu(vals[57]) + le32_to_cpu(vals[58]);/* SSID Mismatch */
	local->wstats.discard.code = le32_to_cpu(vals[6]);/* RxWepErr */
	local->wstats.discard.misc = le32_to_cpu(vals[1]) + le32_to_cpu(vals[2])
		+ le32_to_cpu(vals[3]) + le32_to_cpu(vals[4])
		+ le32_to_cpu(vals[30]) + le32_to_cpu(vals[32]);
	return (&local->wstats);
}
#endif 

#ifdef CISCO_EXT
/*
 *                         Aironet ACU API + MIC
 */

/*
 * This just translates from driver IOCTL codes to the command codes to 
 * feed to the radio's host interface. Things can be added/deleted 
 * as needed.  This represents the READ side of control I/O to 
 * the card
 */
static int readrids(struct net_device *dev, aironet_ioctl *comp) {
  unsigned short ridcode;
  unsigned char iobuf[2048]; 
  struct airo_info *ai = (struct airo_info*) dev->priv;


  if(ai->flags & FLASHING ) /* Is busy */
    return -EIO;


  switch(comp->command)
    {
    case AIROGCAP:      ridcode = RID_CAPABILITIES; break;
    case AIROGCFG:      ridcode = RID_CONFIG;       break;
    case AIROGSLIST:    ridcode = RID_SSID;         break;
    case AIROGVLIST:    ridcode = RID_APLIST;       break;
    case AIROGDRVNAM:   ridcode = RID_DRVNAME;      break;
    case AIROGEHTENC:   ridcode = RID_ETHERENCAP;   break;
    case AIROGWEPKTMP:  ridcode = RID_WEP_TEMP;
#ifndef OPENACCESS
      /* Only super-user can read WEP keys */
      if (!capable(CAP_NET_ADMIN))
	return -EPERM;
#endif
      break;

    case AIROGWEPKNV:   ridcode = RID_WEP_PERM;
#ifndef OPENACCESS
      /* Only super-user can read WEP keys */
      if (!capable(CAP_NET_ADMIN))
	return -EPERM;
#endif
      break;
    case AIROGSTAT:     ridcode = RID_STATUS;       break;
    case AIROGSTATSD32: ridcode = RID_STATSDELTA;   break;
    case AIROGSTATSC32: ridcode = RID_STATS;        break;
    case AIROGMICSTATS:
      if(copy_to_user(comp->data,&ai->micstats,MIN(comp->length,sizeof(STMICSTATISTICS32)))){
	return -EFAULT; 
      }
      else
	return 0;
      break;
    case AIRORRID:
      ridcode = comp->ridnum;
      break;
    default:
      return -EINVAL;  

      break;
    }

  PC4500_readrid((struct airo_info *)dev->priv,ridcode,iobuf,sizeof(iobuf));
  /* get the count of bytes in the rid  docs say 1st 2 bytes is it.
   * then return it to the user 
   * 9/22/2000 Honor user given length
   */
  if (copy_to_user(comp->data, iobuf, MIN (comp->length, sizeof(iobuf))))
    return -EFAULT;
  return 0;
}

/*
 * Danger Will Robinson write the rids here
 */

static int writerids(struct net_device *dev, aironet_ioctl *comp) {
  int  ridcode;
  struct airo_info *ai = (struct airo_info*) dev->priv;
  Resp      rsp;
  ConfigRid *cfg;
  static int (* writer)(struct airo_info *, u16 rid, const void *, int);
  unsigned char iobuf[2048]; 
	
#ifndef OPENACCESS
  /* Only super-user can write RIDs */
  if (!capable(CAP_NET_ADMIN))
    return -EPERM;
#endif
  ridcode = 0;
  writer = do_writerid;

  if(ai->flags & FLASHING ) /* Is busy */
    return -EIO;
 
  switch(comp->command)
    {
    case AIROPSIDS:     ridcode = RID_SSID;         break;
    case AIROPCAP:      ridcode = RID_CAPABILITIES; break;
    case AIROPAPLIST:   ridcode = RID_APLIST;       break;
    case AIROPCFG:      ridcode = RID_CONFIG;       break;
    case AIROPWEPKEYNV: ridcode = RID_WEP_PERM;     break;
    case AIROPLEAPUSR:  ridcode = RID_LEAPUSERNAME; break;
    case AIROPLEAPPWD:  ridcode = RID_LEAPPASSWORD; break;
    case AIROPWEPKEY:   ridcode = RID_WEP_TEMP; writer = PC4500_writerid;break;
    case AIROPLEAPUSR+1: ridcode = 0xff2a; break;
    case AIROPLEAPUSR+2: ridcode = 0xff2b ;break;
      
      /* this is not really a rid but a command given to the card 
       * same with MAC off
       */
    case AIROPMACON:
      if (enable_MAC((struct airo_info *)dev->priv, &rsp) != 0)
	return -EIO;
      return 0;
    case AIROPMACOFF:
      disable_MAC((struct airo_info *)dev->priv);
      return 0;
      
      /* This command merely clears the counts does not actually store any data
       * only reads rid. But as it changes the cards state, I put it in the
       * writerid routines.
       */
    case AIROPSTCLR:
      ridcode = RID_STATSDELTACLEAR;
      
      PC4500_readrid((struct airo_info *)dev->priv,ridcode,iobuf,sizeof(iobuf));
      
      memset(&ai->micstats,0,sizeof(STMICSTATISTICS32));
      ai->micstats.Enabled = ai->flags & MIC_ENABLED ? 1:0;

      if (copy_to_user(comp->data,iobuf,MIN(comp->length,sizeof(iobuf))))
	return -EFAULT;
      return 0;
      
    default:
      return -EOPNOTSUPP;	/* Blarg! */
    }
  if(comp->length > sizeof(iobuf))
    return -EINVAL;

  copy_from_user(iobuf,comp->data,comp->length);

  /* 
   * Whenever a cfg write is done, if card is MIC capable 
   * turn it on also check for ADHOC mode as MIC needs to
   * know about it.
   */
  if(comp->command ==  AIROPCFG){
    cfg = (ConfigRid *)iobuf;
    if(((struct airo_info *)dev->priv)->flags & MIC_CAPABLE)
      cfg->opmode |= CFG_OPMODE_MIC;

    if((cfg->opmode & 1)==0)
      ((struct airo_info *)dev->priv)->flags |= ADHOC;
    else
      ((struct airo_info *)dev->priv)->flags &= ~(ADHOC);
  }

  if((*writer)((struct airo_info *)dev->priv, ridcode, iobuf,comp->length))
    return -EIO;
  return 0;
}

/*****************************************************************************
 * Ancillary flash / mod functions much black magic lurkes here              *
 *****************************************************************************
 */

/* 
 * Flash command switch table
 */

int flashcard(struct net_device *dev, aironet_ioctl *comp) {
  int z;
  int cmdreset(struct airo_info *);
  int setflashmode(struct airo_info *);
  int flashgchar(struct airo_info *,int,int);
  int flashpchar(struct airo_info *,int,int);
  int flashputbuf(struct airo_info *);
  int flashrestart(struct airo_info *,struct net_device *);

#ifndef OPENACCESS
  /* Only super-user can modify flash */
  if (!capable(CAP_NET_ADMIN))
    return -EPERM;
#endif
  switch(comp->command)
    {
    case AIROFLSHRST:
      return cmdreset((struct airo_info *)dev->priv);
      
    case AIROFLSHSTFL:
      return setflashmode((struct airo_info *)dev->priv);
      
    case AIROFLSHGCHR: /* Get char from aux */
      if(comp->length != sizeof(int))
	return -EINVAL;
      copy_from_user(&z,comp->data,comp->length);
      return flashgchar((struct airo_info *)dev->priv,z,8000);

    case AIROFLSHPCHR: /* Send char to card. */
      if(comp->length != sizeof(int))
	return -EINVAL;
      copy_from_user(&z,comp->data,comp->length);
      return flashpchar((struct airo_info *)dev->priv,z,8000);

    case AIROFLPUTBUF: /* Send 32k to card */
      if(comp->length > FLASHSIZE)
	return -EINVAL;
      if(copy_from_user(flashbuffer,comp->data,comp->length)) {
	return -EINVAL;
      }

      flashputbuf((struct airo_info *)dev->priv);
      return 0;

    case AIRORESTART:
      if(flashrestart((struct airo_info *)dev->priv,dev))
	return -EIO;
      return 0;
    }
  return -EINVAL;
}

#define FLASH_COMMAND  0x7e7e

int unstickbusy(struct airo_info *ai) {
  unsigned short uword;

  /* clear stuck command busy if necessary */
  if ((uword=IN4500(ai, COMMAND)) & COMMAND_BUSY) {
    OUT4500(ai, EVACK, EV_CLEARCOMMANDBUSY);
    return 1;
  } 
  return 0;
}

/* Wait for busy completion from card
 * wait for delay uSec's Return true 
 * for success meaning command reg is 
 * clear
 */
int WaitBusy(struct airo_info *ai,int uSec){
  int statword =0xffff;
  int delay =0; 
  
  while((statword & COMMAND_BUSY) && delay <= (1000 * 100) ){
    udelay(10);
    delay += 10;
    statword = IN4500(ai,COMMAND);  
    
    if((COMMAND_BUSY & statword) && (delay%200)){
      unstickbusy(ai);
    }
  }  
  return 0 == (COMMAND_BUSY & statword);
}
/* 
 * STEP 1)
 * Disable MAC and do soft reset on 
 * card. 
 */

int cmdreset(struct airo_info *ai) {
  int status;
  
  disable_MAC(ai);
  
  if(!(status = WaitBusy(ai,600))){
    printk(KERN_INFO "Waitbusy hang b4 RESET =%d\n",status);
    return -EBUSY;
  }
  ai->flags |= FLASHING;
  
  OUT4500(ai,COMMAND,CMD_SOFTRESET);
  
  mdelay(1024);          /* WAS 600 12/7/00 */
  
  if(!(status=WaitBusy(ai,100))){
    printk(KERN_INFO "Waitbusy hang AFTER RESET =%d\n",status);
    ai->flags &= ~FLASHING;
    return -EBUSY;
  }
  return 0;
}

/* STEP 2)
 * Put the card in legendary flash 
 * mode
 */

int setflashmode (struct airo_info *ai) {
  int   status;
  ai->flags |= FLASHING;
  OUT4500(ai, SWS0, FLASH_COMMAND);
  OUT4500(ai, SWS1, FLASH_COMMAND);
  OUT4500(ai, SWS0, FLASH_COMMAND);
  OUT4500(ai, COMMAND,0x10);
  mdelay(500); /* 500ms delay */
  
  if(!(status=WaitBusy(ai,600))) {
    ai->flags &=~(FLASHING);
    printk(KERN_INFO "Waitbusy hang after setflash mode\n");
    return -EIO;
  }
  return 0;
}

/* Put character to SWS0 wait for dwelltime 
 * x 50us for  echo . 
 */

int flashpchar(struct airo_info *ai,int byte,int dwelltime) {
  int echo;
  int pollbusy,waittime;
  
  byte |= 0x8000;
  
  if(dwelltime == 0 )
    dwelltime = 200;
  
  waittime=dwelltime;
  
  /* Wait for busy bit d15 to 
   * go false indicating buffer
   * empty
   */
  do {
    pollbusy = IN4500(ai,SWS0);
    
    if(pollbusy & 0x8000){
      udelay(50);
      waittime -= 50;
    } else 
      break;
  } while(waittime >=0);

  /* timeout for busy clear wait */

  if(waittime <= 0 ){
    printk(KERN_INFO "flash putchar busywait timeout! \n");
    return -EBUSY;
  }
  
  /* Port is clear now write byte and 
   *  wait for it to echo back
   */
  do{
    OUT4500(ai,SWS0,byte);
    udelay(50);
    dwelltime -= 50;
    echo = IN4500(ai,SWS1);
  }while (dwelltime >= 0 && echo != byte);

  OUT4500(ai,SWS1,0);
  
  return (echo == byte) ? 0 : -EIO;
}

/*
 * Get a character from the card matching matchbyte
 * Step 3)
 */
int flashgchar(struct airo_info *ai,int matchbyte,int dwelltime){
  int           rchar;
  unsigned char rbyte=0;
  
  do {
    rchar = IN4500(ai,SWS1);
    
    if(dwelltime && !(0x8000 & rchar)){
      dwelltime -= 10;
      mdelay(10);
      continue;
    }
    rbyte = 0xff & rchar;

    if( (rbyte == matchbyte) && (0x8000 & rchar) ){
      OUT4500(ai,SWS1,0);
      return 0;
    }
    if( rbyte == 0x81 || rbyte == 0x82 || rbyte == 0x83 || rbyte == 0x1a || 0xffff == rchar)
      break;
    OUT4500(ai,SWS1,0);
    
  }while(dwelltime > 0);
  return -EIO;
}

/* 
 * Transfer 32k of firmware data from user buffer to our buffer and 
 * send to the card 
 */
int flashputbuf(struct airo_info *ai){
  unsigned short *bufp;
  int            nwords;

  /* Write stuff */

  bufp=(unsigned short *)flashbuffer;

  OUT4500(ai,AUXPAGE,0x100);
  OUT4500(ai,AUXOFF,0);

  for(nwords=0;nwords!=FLASHSIZE/2;nwords++){
    OUT4500(ai,AUXDATA,bufp[nwords] & 0xffff);
  }
  
  OUT4500(ai,SWS0,0x8000);

  return 0;
}



/*
 *
 */
int flashrestart(struct airo_info *ai,struct net_device *dev){
  int    i,status;
  
  mdelay(1024);          /* Added 12/7/00 */
  ai->flags &= ~(FLASHING);
  status = setup_card(ai, dev->dev_addr,&((struct airo_info*)dev->priv)->config);
  
  for( i = 0; i < MAX_FIDS; i++ ) {
    ai->fids[i] = transmit_allocate( ai, 2000 );
  }
  
  mdelay(1024);          /* Added 12/7/00 */
  return status;
}

/***********************************************************************
 *                              MIC ROUTINES                           *
 ***********************************************************************
 */

/*
 * micinit
 * Initialize mic seed 
 */

static void micinit(struct airo_info *ainfo, STMIC *key)
{
  int micEnabled=0;

  if (NULL == key) {
    return;
  }
#ifdef DEBUG_MIC
  printk(KERN_INFO "MiC stat=%d mcastvalid=%d unicastvalid=%d\n",
	 key->micState,
	 key->micMulticastValid,
	 key-> micUnicastValid);
#endif
  micEnabled = key->micState & 0x00FF;

  if(micEnabled){
    ainfo->flags |= MIC_ENABLED;
    ainfo->micstats.Enabled = 1;
  }
  else
    {
      ainfo->flags &= ~(MIC_ENABLED);
      ainfo->micstats.Enabled = 0;
    }

  if (micEnabled) {
    /* Key must be valid */
    if (key->micMulticastValid) {
      /* Key must be different */
      if (!ainfo->mod[0].mCtx.valid || (memcmp(ainfo->mod[0].mCtx.key,
		  key->micMulticast,sizeof(ainfo->mod[0].mCtx.key))!=0)) {
#ifdef DEBUG_MIC     
	printk(KERN_INFO "Updating multicast Key\n");
#endif
	
	/* Age current mic Context */
	memcpy(&ainfo->mod[1].mCtx,&ainfo->mod[0].mCtx,sizeof(MICCNTX));
	  
	/* Initialize new context */
	  
	memcpy(&ainfo->mod[0].mCtx.key,key->micMulticast,sizeof(key->micMulticast));
	ainfo->mod[0].mCtx.window  = 33;       //Window always points to the middle
	ainfo->mod[0].mCtx.rx      = 0;        //Rx Sequence numbers
	ainfo->mod[0].mCtx.tx      = 0;        //Tx sequence numbers
	ainfo->mod[0].mCtx.valid   = 1;
	  
	/* Give key to mic seed */
	emmh32_setseed(&ainfo->mod[0].mCtx.seed,key->micMulticast,sizeof(key->micMulticast));
      }
    }

    //Key must be valid 
    if (key->micUnicastValid) {
      //Key must be different
      if (!ainfo->mod[0].uCtx.valid || 
	  (memcmp(ainfo->mod[0].uCtx.key,
		  key->micUnicast,
		  sizeof(ainfo->mod[0].uCtx.key)) != 0)) {
#ifdef DEBUG_MIC     
	printk(KERN_INFO "Updating unicast Key\n");
#endif
	
	//Age current mic Context
	memcpy(&ainfo->mod[1].uCtx, &ainfo->mod[0].uCtx,sizeof(MICCNTX));
	
	//Initialize new context
	memcpy(&ainfo->mod[0].uCtx.key,key->micUnicast,sizeof(key->micUnicast));
	
	ainfo->mod[0].uCtx.window  = 33;       //Window always points to the middle
	ainfo->mod[0].uCtx.rx      = 0;        //Rx Sequence numbers
	ainfo->mod[0].uCtx.tx      = 0;        //Tx sequence numbers
	ainfo->mod[0].uCtx.valid   = 1;     //Key is now valid
	
	//Give key to mic seed
	emmh32_setseed(&ainfo->mod[0].uCtx.seed, key->micUnicast, sizeof(key->micUnicast));
      }
    }
  }
  else 
    {
      //So next time we have a valid key and mic is enabled, we will update
      //the sequence number if the key is the same as before.
      ainfo->mod[0].uCtx.valid = 0;
      ainfo->mod[0].mCtx.valid = 0;
    }
#ifdef DEBUG_MIC     
	printk(KERN_INFO "Device flags=%x\n",ainfo->flags);
#endif
}

/*
 * micsetup 
 * Get ready for business
 */

static void micsetup(struct airo_info *a_info){
  int x;
  u8 tmp[]= {0xAA,0xAA,0x03,0x00,0x40,0x96,0x00,0x02};
#ifdef DEBUG_MIC
    printk(KERN_INFO "micsetup \n");
#endif

  memcpy(&a_info->snap,tmp,sizeof(a_info->snap));

  for (x=0; x < NUM_MODULES; x++) {
    memset(&a_info->mod[x].mCtx,0,sizeof(MICCNTX));
    memset(&a_info->mod[x].uCtx,0,sizeof(MICCNTX));
    //Mark multicast context as multicast.
    a_info->mod[x].mCtx.multicast=TRUE;
  }
}


 

/*===========================================================================
 * Description: Mic a packet
 *    
 *      Inputs: ETH_HEADER_STRUC * pointer to an 802.3 frame
 *    
 *     Returns: BOOLEAN if successful, otherwise false.
 *             PacketTxLen will be updated with the mic'd packets size.
 *
 *    Caveats: It is assumed that the frame buffer will already
 *             be big enough to hold the largets mic message possible.
 *            (No memory allocationis done here).
 *  
 *    Author: sbraneky (10/15/01)
 *    Merciless hacks by rwilcher (1/14/02)
 */

static int Encapsulate(struct airo_info *a_info
		       ,ETH_HEADER_STRUC *frame, u32 *PacketTxLen,int adhoc)
{
  MICCNTX   *context;
  int       micEnabled;
  u32       tmp;
  u16       len;
  u16       payLen;
  MIC_BUFFER_STRUCT *mic;

  micEnabled = (a_info->flags & MIC_ENABLED) ? 1:0;

  if (micEnabled && (frame->TypeLength != 0x8E88)) {
    mic = (MIC_BUFFER_STRUCT*)frame;
    len = (u16 )*PacketTxLen;             //Original Packet len
    payLen = (u16)((*PacketTxLen) - 12);  //skip DA, SA

    //Determine correct context

    if (adhoc) {
      if (mic->DA[0] & 0x1) {
	//Broadcast message
	context =  &a_info->mod[0].mCtx;
      }
      else {
	context =  &a_info->mod[0].uCtx;
      }
    }
    else {
      //If not adhoc, always use unicast key
      context =  &a_info->mod[0].uCtx;
    }
    
    if (!context->valid) {
#ifdef DEBUG_MIC
      printk( KERN_INFO "[ENCAP] Context is not valid, not encapsulating\n");
#endif
      return FALSE;
    }

    *PacketTxLen = len +18;                             //Add mic bytes to packet length
    
    //Move payload into new position
    memmove(&mic->payload,&frame->TypeLength,payLen);
    
    //Add Snap
    memcpy(&mic->u.snap,a_info->snap,sizeof(a_info->snap));
    //Add Tx sequence
    tmp = context->tx ;
    UlongByteSwap(&tmp);                    //Convert big/little endian
    memcpy(&mic->SEQ,&tmp,4);
    context->tx += 2;

    //Mic the packet
    emmh32_init(&context->seed);
    
    mic->TypeLength = payLen+16;            //Length of Mic'd packet
    UshortByteSwap(&mic->TypeLength);      //Put in network byte order
    
    emmh32_update(&context->seed, mic->DA,22);  //DA,SA Type/Length, and Snap
    emmh32_update(&context->seed, (u8 *)&mic->SEQ, sizeof(mic->SEQ)+ payLen); //SEQ + payload
    emmh32_final(&context->seed,  (u8 *)&mic->MIC);

    /*    New Type/length ?????????? */
     mic->TypeLength = 0x0000;           //Let NIC know it could be an oversized packet
  }
  return TRUE;
}

/*===========================================================================
 *  Description: Decapsulates a MIC'd packet and returns the 802.3 packet
 *               (removes the MIC stuff) if packet is a valid packet.
 *      
 *       Inputs: ETH_HEADER_STRUC  pointer to the 802.3 packet             
 *     
 *      Returns: BOOLEAN - TRUE if packet should be dropped otherwise FALSE
 *     
 *      Author: sbraneky (10/15/01)
 *    Merciless hacks by rwilcher (1/14/02)
 *---------------------------------------------------------------------------
 */

static int Decapsulate(struct airo_info *a_info,ETH_HEADER_STRUC *ptr, u32 *PacketRxLen)
{
  int      micEnabled,micPacket,x;
  int      valid = TRUE;
  u32      micSEQ,seq;
  u16      payLen;
  MICCNTX  *context;
  MIC_BUFFER_STRUCT *mic ;
  u8       digest[4],*miccp;
  MIC_ERROR micError = NONE;

  mic = (MIC_BUFFER_STRUCT*)ptr ;
  micEnabled = (a_info->flags & MIC_ENABLED) ? 1:0;

  if (mic->TypeLength <= 0x5DC || mic->TypeLength != 0x8E88) {
      valid = FALSE;                             //Assume failure
      //Check if the packet is a Mic'd packet

      micPacket = (0 == memcmp(mic->u.snap,a_info->snap,sizeof(a_info->snap)));

      if (!micPacket && !micEnabled ) {
	return FALSE; /* MIC is off here. */
      }
      
      if (micPacket &&  !micEnabled) {
	//No Mic set or Mic OFF but we received a MIC'd packet.
#ifdef DEBUG_MIC
	printk(KERN_INFO "[DEMIC] Mic'd packet received but mic not enabled\n");
#endif
	a_info->micstats.RxMICPlummed++;
	return TRUE;
      }
      
      if (!micPacket && micEnabled) {
	//Mic enabled but packet isn't Mic'd
#ifdef DEBUG_MIC
	printk(KERN_INFO "[DEMIC] Non mic'd packet received when mic enabled\n");
#endif
	a_info->micstats.RxMICPlummed++;
    	return TRUE;
      }

      micSEQ = mic->SEQ;            //store SEQ as little endian
      UlongByteSwap(&micSEQ);       //Convert to little/big endian

      //At this point we a have a mic'd packet and mic is enabled
      //Now do the mic error checking.

      if ( 0 == ( 1 & micSEQ)) {
	//Receive seq must be odd
#ifdef DEBUG_MIC
	printk(KERN_INFO "[DEMIC] Even seqence number received %x\n",micSEQ);
#endif
	a_info->micstats.RxWrongSequence++;
	return 1;
      }

      payLen    = (u16) ((*PacketRxLen) - 30);   //Payload length
      seq       = 0;                               //Sequence # relative to window

      
      for (x=0; !valid && x < NUM_MODULES; x++) {
	//Determine proper context 
	context = (mic->DA[0] & 0x1) ? &a_info->mod[x].mCtx : &a_info->mod[x].uCtx;
	
	//Make sure context is valid
	if (!context->valid) {
#ifdef DEBUG_MIC
	  printk(KERN_INFO "[DEMIC] Context is not valid, not decapsulating\n");
#endif
	  micError = NOMICPLUMMED;
	  continue;                
	}
       	//DeMic it 

	if (0 == mic->TypeLength) {
	  mic->TypeLength = (*PacketRxLen - sizeof(ETH_HEADER_STRUC));
	  UshortByteSwap(&mic->TypeLength);    //Convert to Network byte order
	}
	
	emmh32_init(&context->seed);
	emmh32_update(&context->seed, mic->DA,
		      sizeof(mic->DA)+sizeof(mic->SA)+sizeof(mic->TypeLength)+sizeof(mic->u.snap)); 
	emmh32_update(&context->seed, (u8 *) &mic->SEQ,sizeof(mic->SEQ)+payLen);	
	//Calculate MIC
	emmh32_final(&context->seed, digest);
	
	if (memcmp(digest,&mic->MIC,4 )) { 	//Make sure the mics match
	  //Invalid Mic
#ifdef DEBUG_MIC
	  printk(KERN_INFO "[DEMIC] Invalid Mic'd message received %x%x%x%x\n",
		 digest[0],digest[1],digest[2],digest[3]);
#endif	  
	  if (0 == x) {
	    micError = INCORRECTMIC;
	  }
	  continue;
	}
	else
	  {
	    miccp = (u8 *)&mic->MIC;
	  }

	//Check Sequence number if mics pass
	if (FALSE == RxSeqValid(a_info,context,micSEQ)) {
#ifdef DEBUG_MIC
	  printk(KERN_INFO "[DEMIC] Invalid sequence number\n");
#endif 	  
	  if (0 == x) {
	    micError = SEQUENCE;
	  }
	  continue;
	}

	//Mic and SEQ match
	valid = TRUE;
	
	//Remove  0 eType, snap, MIC, and SEQ
	memmove(&mic->TypeLength,&mic->payload,payLen);

	//Update Packet length
	*PacketRxLen  -= 18;
	
	if( *PacketRxLen < 60 ) {
	  //Promote to minimum length
	  *PacketRxLen = 60;
	}

	//Update statistics
        switch (micError) 
	  {
            case NONE:
                a_info->micstats.RxSuccess++;
            break;
            case NOMIC:
                a_info->micstats.RxNotMICed++;
            break;
            case NOMICPLUMMED:
                a_info->micstats.RxMICPlummed++;
            break;
            case SEQUENCE:
                a_info->micstats.RxWrongSequence++;
            break;
            case INCORRECTMIC:
                a_info->micstats.RxIncorrectMIC++;
            break;
	  }

      }      
  }
  return (valid==1 ? 0 : 1);
}


/*===========================================================================
 * Description:  Checks the Rx Seq number to make sure it is valid
 *               and hasn't already been received
 *   
 *     Inputs: MICCNTX - mic context to check seq against
 *             micSeq  - the Mic seq number
 *   
 *    Returns: TRUE if valid otherwise FALSE. 
 *
 *    Author: sbraneky (10/15/01)
 *    Merciless hacks by rwilcher (1/14/02)
 *---------------------------------------------------------------------------
 */

static int RxSeqValid (struct airo_info *a_info,MICCNTX *context, u32 micSeq)
{
    u32 seq,index ;

    #if DEBUG_SEQ
        printk(KERN_INFO "Mic Seq %d,window %d, rx=%x\n",micSeq,context->window,context->rx);
    #endif

    //Allow for the ap being rebooted - if it is then use the next 
    //sequence number of the current sequence number - might go backwards

    if (a_info->updateMultiSeq && context->multicast) {
        a_info->updateMultiSeq  = FALSE;
        //Move window
        context->window = ((micSeq > 33) ? micSeq : 33);
        context->rx     = 0;            //Reset rx
        #if DEBUG_SEQ
            printk( KERN_INFO "Updating mulicast context window=%x\n",context->window);
        #endif

    }
    else {
        if (a_info->updateUniSeq && !context->multicast) {
            a_info->updateUniSeq    = FALSE;
            context->window = ((micSeq > 33) ? micSeq : 33);
            context->rx     = 0;        //Reset rx
            #if DEBUG_SEQ
                printk( KERN_INFO "Updating unicast context window=%x\n",context->window);
            #endif
        }
    }

    //Make sequence number relative to START of window
    seq = micSeq - (context->window-33);

    if ((u32)seq < 0) {
        //Too old of a SEQ number to check.
        #if DEBUG_SEQ
            printk( KERN_INFO "Seq number is too old to check %x.\n",seq);
        #endif
        return FALSE;
    }
    
    if ( seq > 64 )  {
        #if DEBUG_SEQ
            printk( KERN_INFO "Seq number is greater than window %x\n",seq);
        #endif
        //Window is infinite forward
        MoveWindow(context,micSeq);
        return TRUE;
    }

    //We are in the window. Now check the context rx bit to see if it was already sent
    seq >>= 1;                      //divide by 2 because we only have odd numbers
    index = (0x01 << seq);    //Get an index number

    #if DEBUG_SEQ
    printk( KERN_INFO "Relative seq = %x, index = %x rx =%x\n",seq,index,context->rx);
    #endif

    if (0 == (context->rx & index)) {
        //micSEQ falls inside the window.
        //Add seqence number to the list of received numbers.
        context->rx |= index;

        MoveWindow(context,micSeq);

        return TRUE;
    }

    #if DEBUG_SEQ
        printk( KERN_INFO "Mic Sequnce number already received window=%x rx=%x seq= %x index=%x \n",context->window,context->rx,seq,index);
    #endif

    return FALSE;
}

static void MoveWindow(MICCNTX *context, u32 micSeq)
{
  u32 shift;

  //Move window if seq greater than the middle of the window
  if (micSeq > context->window) {
    shift = micSeq - context->window;    //Move relative to middle of window
    shift >>= 1;                         //divide by 2
    
    //Shift out old
    if (shift < 32) {
      context->rx >>= shift;
    }
    else {
      //Are we going to shift everything out
      context->rx = 0;
    }

    context->window = micSeq;      //Move window
#if DEBUG_SEQ
    printk( KERN_INFO "Moved window window=%x, rx=%x\n",context->window,context->rx);
#endif
  }
}

static void UshortByteSwap(u16 *source)
{
    u16 tmp = *source;    
    u8 *ptr = (u8*)source;
    
    ptr[0] = (u8)(tmp >> 8) & 0x00FF;
    ptr[1] = (u8)(tmp  & 0x00FF);
}

static void UlongByteSwap(u32 *source)
{
    u32 tmp = *source;    
    u8 *ptr = (u8*)source;
    
    ptr[0] = (u8)(tmp >> 24) & 0x00FF;
    ptr[1] = (u8)(tmp >> 16) & 0x00FF;
    ptr[2] = (u8)(tmp >>  8) & 0x00FF;
    ptr[3] = (u8) tmp        & 0x00FF;
}

#endif /* CISCO_EXT */
  

#define GENBC
#undef  GENBC

/*==========================================================================================*/
/*========== ENDIANESS =====================================================================*/
/*==========================================================================================*/
/* macros for dealing with endianess */
#define SWAPU32(d)				( ((d)<<24) | ( ((d)&0xFF00)<<8) | \
						  (((d)>>8)&0xFF00) | ((d)>>24) )


#if defined(BIG_ENDIAN) && defined(LITTLE_ENDIAN)
#error ENDIAN -- both BIG_ENDIAN and LITTLE_ENDIAN are defined
#elif defined(BIG_ENDIAN)
#define BIGEND32(d)				(d)
#elif defined(LITTLE_ENDIAN)
#define BIGEND32(d)				SWAPU32(d)
#else
/* fine, make a runtime decision, not as efficient.... */
static unsigned short endian_ref = { 0x1234 };
#define ISBIGENDIAN					( (*(u8*)(&endian_ref)) == 0x12 )
#define BIGEND32(d)				( ISBIGENDIAN ? (d) : SWAPU32(d))
#endif

#ifndef UNALIGN32
#error UNALIGN32 must be defined.
#elif UNALIGN32
/* unaligned accesses are allowed -- fetch u32 and swap endian */
#define	GETBIG32(p)				BIGEND32(*(u32*)(p))
#else
/* unaligned accesses are disallowed ... slow GET32() */
#define GB(p,i,s)				( ((u32) *((u8*)(p)+i) ) << (s) )
#define GETBIG32(p)				GB(p,0,24)|GB(p,1,16)|GB(p,2,8)|GB(p,3,0)
#endif


/*==========================================================================================*/
/*========== EMMH ROUTINES  ================================================================*/
/*==========================================================================================*/

/* mic accumulate */
#define MIC_ACCUM(v)	\
	context->accum += (u64)val * context->coeff[coeff_position++];

#ifdef GENBC
/* use unix BC for verification since BC supports arbitrary length integers */
FILE *fpbc = stdout;
#undef  MIC_ACCUM
#define MIC_ACCUM(v)	\
	fprintf(fpbc, "bc:(.+%8lX*%8lX)%%10000000000000000\n", v, context->coeff[coeff_position]); \
	context->accum += (u64)val * context->coeff[coeff_position++]; \
	printf("%08lX:%08lX\n", (u32)(context->accum >> 32), (u32)(context->accum));
#endif

static aes cx;
static unsigned char aes_counter[16];
static unsigned char aes_cipher[16];

/* expand the key to fill the MMH coefficient array */
void emmh32_setseed(emmh32_context *context, u8 *pkey, int keylen)
{
  /* take the keying material, expand if necessary, truncate at 16-bytes */
  /* run through AES counter mode to generate context->coeff[] */
  
  int i,j;
  u32 val;
  u32 counter;
#if 0
  if (keylen != 16) { /* @RAW Cant touch this */
    fprintf(stderr, "ERROR: key length not 16\n");
    exit(1);
  }
#endif 

#if 0
dumpbytes("micseed = ", pkey, 16);
#endif

 set_key(pkey, 16, enc, &cx);
 counter = 0;
 for (i=0; i< (sizeof(context->coeff)/sizeof(context->coeff[0])); ) {
   aes_counter[15] = (u8)(counter >> 0);
   aes_counter[14] = (u8)(counter >> 8);
   aes_counter[13] = (u8)(counter >> 16);
   aes_counter[12] = (u8)(counter >> 24);
   counter++;
   encrypt(&aes_counter[0], &aes_cipher[0], &cx);
   for (j=0; (j<sizeof(aes_cipher)) && (i< (sizeof(context->coeff)/sizeof(context->coeff[0]))); ) {
     val = GETBIG32(&aes_cipher[j]);
     context->coeff[i++] = val;
     j += 4;
   }
 }
#if 0
 dumplong("coeff = ", &context->coeff[0], 8);
#endif
}

/* prepare for calculation of a new mic */
void emmh32_init(emmh32_context *context)
{
	/* prepare for new mic calculation */
  context->accum = 0;
  context->position = 0;
#ifdef GENBC
  fprintf(fpbc, "bc:ibase=10\n");
  fprintf(fpbc, "bc:ibase=A\n");
  fprintf(fpbc, "bc:obase=16\n");
  fprintf(fpbc, "bc:ibase=16\n");
  fprintf(fpbc, "bc:0\n");
#endif
}

/* add some bytes to the mic calculation */
void emmh32_update(emmh32_context *context, u8 *pOctets, int len)
{
  int	coeff_position, byte_position;
  u32	val;
  
  if (len == 0) return;
  
  coeff_position = context->position >> 2;
  
  /* deal with partial 32-bit word left over from last update */
  if ( (byte_position = (context->position & 3)) != 0) {
    /* have a partial word in part to deal with */
    do {
      if (len == 0) return;
      context->part.d8[byte_position++] = *pOctets++;
      context->position++;
      len--;
    } while (byte_position < 4);
    val = context->part.d32;
    val = BIGEND32(val);
    MIC_ACCUM(val);
  }

  /* deal with full 32-bit words */
  while (len >= 4) {
    val = GETBIG32(pOctets);
    MIC_ACCUM(val);
    context->position += 4;
    pOctets += 4;
    len -= 4;
  }

  /* deal with partial 32-bit word that will be left over from this update */
  byte_position = 0;
  while (len > 0) {
    if (len == 0) return;
    context->part.d8[byte_position++] = *pOctets++;
    context->position++;
    len--;
  }
}

/* mask used to zero empty bytes for final parial word */
static u32 mask32[4] = { 0x00000000L, 0xFF000000L, 0xFFFF0000L, 0xFFFFFF00L };

/* calculate the mic */
void emmh32_final(emmh32_context *context, u8 digest[4])
{
  int	coeff_position, byte_position;
  u32	val;
  
  u64 sum, utmp;
  s64 stmp;

  coeff_position = context->position >> 2;
  
  /* deal with partial 32-bit word left over from last update */
  if ( (byte_position = (context->position & 3)) != 0) {
    /* have a partial word in part to deal with */
    val = context->part.d32;
    val = BIGEND32(val);		/* convert to big endian if required */
    val &= mask32[byte_position];	/* zero empty bytes */
    MIC_ACCUM(val);
  }

  /* reduce the accumulated u64 to a 32-bit MIC */
  sum = context->accum;
  stmp = (sum  & 0xffffffffLL) - ((sum >> 32)  * 15);
  utmp = (stmp & 0xffffffffLL) - ((stmp >> 32) * 15);
  sum = utmp & 0xffffffffLL;
  if (utmp > 0x10000000fLL)
    sum -= 15;
  
  val = (u32)sum;
  digest[0] = (val>>24) & 0xFF;
  digest[1] = (val>>16) & 0xFF;
  digest[2] = (val>>8) & 0xFF;
  digest[3] = (val>>0) & 0xFF;
#ifdef GENBC
  fprintf(fpbc, "bc:.%%10000000000000000\n");
  fprintf(fpbc, "bc:.%%10000000F\n");
  fprintf(fpbc, "bc:\"result = \"\n");
  fprintf(fpbc, "bc:.\n");
#endif
}


 /*
   -----------------------------------------------------------------------
   Copyright (c) 2001 Dr Brian Gladman <brg@gladman.uk.net>, Worcester, UK
   
   TERMS

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   This software is provided 'as is' with no guarantees of correctness or
   fitness for purpose.
   -----------------------------------------------------------------------

   1. FUNCTION
 
   The AES algorithm Rijndael implemented for block and key sizes of 128,
   192 and 256 bits (16, 24 and 32 bytes) by Brian Gladman.

   This is an implementation of the AES encryption algorithm (Rijndael)
   designed by Joan Daemen and Vincent Rijmen. This version is designed
   to provide both fixed and dynamic block and key lengths and can also 
   run with either big or little endian internal byte order (see aes.h). 
   It inputs block and key lengths in bytes with the legal values being 
   16, 24 and 32.
 
   2. CONFIGURATION OPTIONS (see also aes.h)
 
   a.  Define UNROLL for full loop unrolling in encryption and decryption.
   b.  Define PARTIAL_UNROLL to unroll two loops in encryption and decryption.
   c.  Define FIXED_TABLES for compiled rather than dynamic tables.
   d.  Define FF_TABLES to use tables for field multiplies and inverses.
   e.  Define ARRAYS to use arrays to hold the local state block. If this
       is not defined, individually declared 32-bit words are used.
   f.  Define FAST_VARIABLE if a high speed variable block implementation
       is needed (essentially three separate fixed block size code sequences)
   g.  Define either ONE_TABLE or FOUR_TABLES for a fast table driven 
       version using 1 table (2 kbytes of table space) or 4 tables (8
       kbytes of table space) for higher speed.
   h.  Define either ONE_LR_TABLE or FOUR_LR_TABLES for a further speed 
       increase by using tables for the last rounds but with more table
       space (2 or 8 kbytes extra).
   i.  If neither ONE_TABLE nor FOUR_TABLES is defined, a compact but 
       slower version is provided.
   j.  If fast decryption key scheduling is needed define ONE_IM_TABLE
       or FOUR_IM_TABLES for higher speed (2 or 8 kbytes extra).

   3. USE OF DEFINES
  
   NOTE: some combinations of the following defines are disabled below.

   UNROLL or PARTIAL_UNROLL control the extent to which loops are unrolled
   in the main encryption and decryption routines. UNROLL does a complete
   unroll while PARTIAL_UNROLL uses a loop with two rounds in it.
 
#define UNROLL
#define PARTIAL_UNROLL
 
   If FIXED_TABLES is defined, the tables are comipled statically into the 
   code, otherwise they are computed once when the code is first used.
 
#define FIXED_TABLES
 
   If FF_TABLES is defined faster finite field arithmetic is performed by 
   using tables.
 
#define FF_TABLES

   If ARRAYS is defined the state variables for encryption are defined as
   arrays, otherwise they are defined as individual variables. The latter
   is useful on machines where these variables can be mapped to registers. 
 
#define ARRAYS

   If FAST_VARIABLE is defined with variable block length, faster but larger
   code is used for encryption and decryption.

#define FAST_VARIABLE
 */

#define UNROLL
#define FIXED_TABLES
#define FF_TABLES
#undef ARRAYS
#define FAST_VARIABLE

 /*
   This code uses three sets of tables, each of which can be a single table
   or four sub-tables to gain a further speed advantage.

   The defines ONE_TABLE and FOUR_TABLES control the use of tables in the 
   main encryption rounds and have the greatest impact on speed.  If neither
   is defined, tables are not used and the resulting code is then very slow.
   Defining ONE_TABLE gives a substantial speed increase using 2 kbytes of 
   table space; FOUR_TABLES gives a further speed increase but uses 8 kbytes
   of table space.
   
#define ONE_TABLE
#define FOUR_TABLES

   The defines ONE_LR_TABLE and FOUR_LR_TABLES apply to the last round only
   and their impact on speed is hence less. It is unlikely to be sensible to
   apply these options unless the correspnding option above is also used.    

#define ONE_LR_TABLE
#define FOUR_LR_TABLES

   The ONE_IM_TABLE and FOUR_IM_TABLES options use tables to speed up the 
   generation of the decryption key schedule. This will only be useful in
   limited situations where decryption speed with frequent re-keying is
   needed.

#define ONE_IM_TABLE
#define FOUR_IM_TABLES
 */

#define FOUR_TABLES
#define FOUR_LR_TABLES
#define FOUR_IM_TABLES

 /*
   In this implementation the columns of the state array are each held in
   32-bit words. The state array can be held in various ways: in an array
   of words, in a number of individual word variables or in a number of 
   processor registers. The following define maps a variable name x and
   a column number c to the way the state array variable is to be held.
 */

#if defined(ARRAYS)
#define s(x,c) x[c]
#else
#define s(x,c) x##c
#endif

#if defined(MMHBLOCK_SIZE) && (MMHBLOCK_SIZE == 20 || MMHBLOCK_SIZE == 28)
#error an illegal block size has been specified
#endif  

#if defined(UNROLL) && defined (PARTIAL_UNROLL)
#error both UNROLL and PARTIAL_UNROLL are defined
#endif

#if defined(ONE_TABLE) && defined (FOUR_TABLES)
#error both ONE_TABLE and FOUR_TABLES are defined
#endif

#if defined(ONE_LR_TABLE) && defined (FOUR_LR_TABLES)
#error both ONE_LR_TABLE and FOUR_LR_TABLES are defined
#endif

#if defined(ONE_IM_TABLE) && defined (FOUR_IM_TABLES)
#error both ONE_IM_TABLE and FOUR_IM_TABLES are defined
#endif

/* End of configuration options */

#include "aes.h"

/* Disable at least some poor combinations of options */

#if !defined(ONE_TABLE) && !defined(FOUR_TABLES)
#define FIXED_TABLES
#undef  UNROLL
#undef  ONE_LR_TABLE
#undef  FOUR_LR_TABLES
#undef  ONE_IM_TABLE
#undef  FOUR_IM_TABLES
#elif !defined(FOUR_TABLES)
#ifdef  FOUR_LR_TABLES
#undef  FOUR_LR_TABLES
#define ONE_LR_TABLE
#endif
#ifdef  FOUR_IM_TABLES
#undef  FOUR_IM_TABLES
#define ONE_IM_TABLE
#endif
#elif !defined(MMHBLOCK_SIZE)
#if defined(UNROLL)
#define PARTIAL_UNROLL
#undef UNROLL
#endif
#endif

/* the finite field modular polynomial and elements */

#define ff_poly 0x011b
#define ff_hi   0x80

/* multiply four bytes in GF(2^8) by 'x' {02} in parallel */

#define m1  0x80808080
#define m2  0x7f7f7f7f
#define m3  0x0000001b
#define FFmulX(x)  ((((x) & m2) << 1) ^ ((((x) & m1) >> 7) * m3))

 /* 
   The following defines provide alternative definitions of FFmulX that might
   give improved performance if a fast 32-bit multiply is not available. Note
   that a temporary variable u needs to be defined where FFmulX is used.

#define FFmulX(x) (u = (x) & m1, u |= (u >> 1), ((x) & m2) << 1) ^ ((u >> 3) | (u >> 6)) 
#define m4  0x1b1b1b1b
#define FFmulX(x) (u = (x) & m1, ((x) & m2) << 1) ^ ((u - (u >> 7)) & m4) 

 */

/* perform column mix operation on four bytes in parallel */

#define fwd_mcol(x) (f2 = FFmulX(x), f2 ^ upr(x ^ f2,3) ^ upr(x,2) ^ upr(x,1))

#if defined(FIXED_TABLES)

#include "aestab.h"

#else

static byte  s_box[256];
static byte  inv_s_box[256];
static word  rcon_tab[RC_LENGTH];

#if defined(ONE_TABLE)
static word  ft_tab[256];
static word  it_tab[256];
#elif defined(FOUR_TABLES)
static word  ft_tab[4][256];
static word  it_tab[4][256];
#endif

#if defined(ONE_LR_TABLE)
static word  fl_tab[256];
static word  il_tab[256];
#elif defined(FOUR_LR_TABLES)
static word  fl_tab[4][256];
static word  il_tab[4][256];
#endif

#if defined(ONE_IM_TABLE)
static word  im_tab[256];
#elif defined(FOUR_IM_TABLES)
static word  im_tab[4][256];
#endif

#if !defined(FF_TABLES)

/*
   Generate the tables for the dynamic table option

   It will generally be sensible to use tables to compute finite 
   field multiplies and inverses but where memory is scarse this 
   code might sometimes be better.

   return 2 ^ (n - 1) where n is the bit number of the highest bit
   set in x with x in the range 1 < x < 0x00000200.   This form is
   used so that locals within FFinv can be bytes rather than words
*/

static byte hibit(const word x)
{   byte r = (byte)((x >> 1) | (x >> 2));
    
    r |= (r >> 2);
    r |= (r >> 4);
    return (r + 1) >> 1;
}

/* return the inverse of the finite field element x */

static byte FFinv(const byte x)
{   byte    p1 = x, p2 = 0x1b, n1 = hibit(x), n2 = 0x80, v1 = 1, v2 = 0;

    if(x < 2) return x;

    for(;;)
    {
        if(!n1) return v1;

        while(n2 >= n1)
        {   
            n2 /= n1; p2 ^= p1 * n2; v2 ^= v1 * n2; n2 = hibit(p2);
        }
        
        if(!n2) return v2;

        while(n1 >= n2)
        {   
            n1 /= n2; p1 ^= p2 * n1; v1 ^= v2 * n1; n1 = hibit(p1);
        }
    }
}

/* define the finite field multiplies required for Rijndael */

#define FFmul02(x)  ((((x) & 0x7f) << 1) ^ ((x) & 0x80 ? 0x1b : 0))
#define FFmul03(x)  ((x) ^ FFmul02(x))
#define FFmul09(x)  ((x) ^ FFmul02(FFmul02(FFmul02(x))))
#define FFmul0b(x)  ((x) ^ FFmul02((x) ^ FFmul02(FFmul02(x))))
#define FFmul0d(x)  ((x) ^ FFmul02(FFmul02((x) ^ FFmul02(x))))
#define FFmul0e(x)  FFmul02((x) ^ FFmul02((x) ^ FFmul02(x)))

#else

#define FFinv(x)    ((x) ? pow[255 - log[x]]: 0)

#define FFmul02(x) (x ? pow[log[x] + 0x19] : 0)
#define FFmul03(x) (x ? pow[log[x] + 0x01] : 0)
#define FFmul09(x) (x ? pow[log[x] + 0xc7] : 0)
#define FFmul0b(x) (x ? pow[log[x] + 0x68] : 0)
#define FFmul0d(x) (x ? pow[log[x] + 0xee] : 0)
#define FFmul0e(x) (x ? pow[log[x] + 0xdf] : 0)

#endif

/* The forward and inverse affine transformations used in the S-box */

#define fwd_affine(x) \
    (w = (word)x, w ^= (w<<1)^(w<<2)^(w<<3)^(w<<4), 0x63^(byte)(w^(w>>8)))

#define inv_affine(x) \
    (w = (word)x, w = (w<<1)^(w<<3)^(w<<6), 0x05^(byte)(w^(w>>8)))

static void gen_tabs(void)
{   word  i, w;

#if defined(FF_TABLES)

    byte  pow[512], log[256];

    /*
       log and power tables for GF(2^8) finite field with
       0x011b as modular polynomial - the simplest primitive
       root is 0x03, used here to generate the tables
    */

    i = 0; w = 1; 
    do
    {   
        pow[i] = (byte)w;
        pow[i + 255] = (byte)w;
        log[w] = (byte)i++;
        w ^=  (w << 1) ^ (w & ff_hi ? ff_poly : 0);
    }
    while (w != 1);

#endif

    for(i = 0, w = 1; i < RC_LENGTH; ++i)
    {
        rcon_tab[i] = bytes2word(w, 0, 0, 0);
        w = (w << 1) ^ (w & ff_hi ? ff_poly : 0);
    }

    for(i = 0; i < 256; ++i)
    {   byte    b;

        s_box[i] = b = fwd_affine(FFinv((byte)i));

        w = bytes2word(b, 0, 0, 0);
#if defined(ONE_LR_TABLE)
        fl_tab[i] = w;
#elif defined(FOUR_LR_TABLES)
        fl_tab[0][i] = w;
        fl_tab[1][i] = upr(w,1);
        fl_tab[2][i] = upr(w,2);
        fl_tab[3][i] = upr(w,3);
#endif
        w = bytes2word(FFmul02(b), b, b, FFmul03(b));
#if defined(ONE_TABLE)
        ft_tab[i] = w;
#elif defined(FOUR_TABLES)
        ft_tab[0][i] = w;
        ft_tab[1][i] = upr(w,1);
        ft_tab[2][i] = upr(w,2);
        ft_tab[3][i] = upr(w,3);
#endif
        inv_s_box[i] = b = FFinv(inv_affine((byte)i));

        w = bytes2word(b, 0, 0, 0);
#if defined(ONE_LR_TABLE)
        il_tab[i] = w;
#elif defined(FOUR_LR_TABLES)
        il_tab[0][i] = w;
        il_tab[1][i] = upr(w,1);
        il_tab[2][i] = upr(w,2);
        il_tab[3][i] = upr(w,3);
#endif
        w = bytes2word(FFmul0e(b), FFmul09(b), FFmul0d(b), FFmul0b(b));
#if defined(ONE_TABLE)
        it_tab[i] = w;
#elif defined(FOUR_TABLES)
        it_tab[0][i] = w;
        it_tab[1][i] = upr(w,1);
        it_tab[2][i] = upr(w,2);
        it_tab[3][i] = upr(w,3);
#endif
#if defined(ONE_IM_TABLE)
        im_tab[b] = w;
#elif defined(FOUR_IM_TABLES)
        im_tab[0][b] = w;
        im_tab[1][b] = upr(w,1);
        im_tab[2][b] = upr(w,2);
        im_tab[3][b] = upr(w,3);
#endif

    }
}

#endif

#define no_table(x,box,vf,rf,c) bytes2word( \
    box[bval(vf(x,0,c),rf(0,c))], \
    box[bval(vf(x,1,c),rf(1,c))], \
    box[bval(vf(x,2,c),rf(2,c))], \
    box[bval(vf(x,3,c),rf(3,c))])

#define one_table(x,op,tab,vf,rf,c) \
 (     tab[bval(vf(x,0,c),rf(0,c))] \
  ^ op(tab[bval(vf(x,1,c),rf(1,c))],1) \
  ^ op(tab[bval(vf(x,2,c),rf(2,c))],2) \
  ^ op(tab[bval(vf(x,3,c),rf(3,c))],3))

#define four_tables(x,tab,vf,rf,c) \
 (  tab[0][bval(vf(x,0,c),rf(0,c))] \
  ^ tab[1][bval(vf(x,1,c),rf(1,c))] \
  ^ tab[2][bval(vf(x,2,c),rf(2,c))] \
  ^ tab[3][bval(vf(x,3,c),rf(3,c))])

#define vf1(x,r,c)  (x)
#define rf1(r,c)    (r)
#define rf2(r,c)    ((r-c)&3)

#if defined(FOUR_LR_TABLES)
#define ls_box(x,c)     four_tables(x,fl_tab,vf1,rf2,c)
#elif defined(ONE_LR_TABLE)
#define ls_box(x,c)     one_table(x,upr,fl_tab,vf1,rf2,c)
#else
#define ls_box(x,c)     no_table(x,s_box,vf1,rf2,c)
#endif

#if defined(FOUR_IM_TABLES)
#define inv_mcol(x)     four_tables(x,im_tab,vf1,rf1,0)
#elif defined(ONE_IM_TABLE)
#define inv_mcol(x)     one_table(x,upr,im_tab,vf1,rf1,0)
#else
#define inv_mcol(x) \
    (f9 = (x),f2 = FFmulX(f9), f4 = FFmulX(f2), f8 = FFmulX(f4), f9 ^= f8, \
    f2 ^= f4 ^ f8 ^ upr(f2 ^ f9,3) ^ upr(f4 ^ f9,2) ^ upr(f9,1))
#endif

 /* 
   Subroutine to set the block size (if variable) in bytes, legal
   values being 16, 24 and 32.
 */

#if defined(MMHBLOCK_SIZE)
#define nc   (Ncol)
#else
#define nc   (cx->Ncol)

cf_dec c_name(set_blk)(const word n_bytes, c_name(aes) *cx)
{
#if !defined(FIXED_TABLES)
    if(!(cx->mode & 0x08)) { gen_tabs(); cx->mode = 0x08; }
#endif

    if((n_bytes & 7) || n_bytes < 16 || n_bytes > 32) 
    {     
        return (n_bytes ? cx->mode &= ~0x07, aes_bad : (aes_ret)(nc << 2));
    }

    cx->mode = cx->mode & ~0x07 | 0x04;
    nc = n_bytes >> 2;
    return aes_good;
}

#endif

 /*
   Initialise the key schedule from the user supplied key. The key
   length is now specified in bytes - 16, 24 or 32 as appropriate.
   This corresponds to bit lengths of 128, 192 and 256 bits, and
   to Nk values of 4, 6 and 8 respectively.
 */

#define mx(t,f) (*t++ = inv_mcol(*f),f++)
#define cp(t,f) *t++ = *f++

#if   MMHBLOCK_SIZE == 16
#define cpy(d,s)    cp(d,s); cp(d,s); cp(d,s); cp(d,s)
#define mix(d,s)    mx(d,s); mx(d,s); mx(d,s); mx(d,s)
#elif MMHBLOCK_SIZE == 24
#define cpy(d,s)    cp(d,s); cp(d,s); cp(d,s); cp(d,s); \
                    cp(d,s); cp(d,s)
#define mix(d,s)    mx(d,s); mx(d,s); mx(d,s); mx(d,s); \
                    mx(d,s); mx(d,s)
#elif MMHBLOCK_SIZE == 32
#define cpy(d,s)    cp(d,s); cp(d,s); cp(d,s); cp(d,s); \
                    cp(d,s); cp(d,s); cp(d,s); cp(d,s)
#define mix(d,s)    mx(d,s); mx(d,s); mx(d,s); mx(d,s); \
                    mx(d,s); mx(d,s); mx(d,s); mx(d,s)
#else

#define cpy(d,s) \
switch(nc) \
{   case 8: cp(d,s); cp(d,s); \
    case 6: cp(d,s); cp(d,s); \
    case 4: cp(d,s); cp(d,s); \
            cp(d,s); cp(d,s); \
}

#define mix(d,s) \
switch(nc) \
{   case 8: mx(d,s); mx(d,s); \
    case 6: mx(d,s); mx(d,s); \
    case 4: mx(d,s); mx(d,s); \
            mx(d,s); mx(d,s); \
}

#endif

cf_dec c_name(set_key)(const byte in_key[], const word n_bytes, const enum aes_key f, c_name(aes) *cx)
{   word    *kf, *kt, rci;

#if !defined(FIXED_TABLES)
    if(!(cx->mode & 0x08)) { gen_tabs(); cx->mode = 0x08; }
#endif

#if !defined(MMHBLOCK_SIZE)
    if(!(cx->mode & 0x04)) c_name(set_blk)(16, cx);
#endif
    if((n_bytes & 7) || n_bytes < 16 || n_bytes > 32 || (!(f & 1) && !(f & 2)) )
    {     
        return (n_bytes ? cx->mode &= ~0x03, aes_bad : (aes_ret)(cx->Nkey << 2));
    }
    cx->mode = (cx->mode & ~0x03) | ((byte)f & 0x03);
    cx->Nkey = n_bytes >> 2;
    cx->Nrnd = Nr(cx->Nkey, nc);

    cx->e_key[0] = word_in(in_key     );
    cx->e_key[1] = word_in(in_key +  4);
    cx->e_key[2] = word_in(in_key +  8);
    cx->e_key[3] = word_in(in_key + 12);

    kf = cx->e_key; 
    kt = kf + nc * (cx->Nrnd + 1) - cx->Nkey; 
    rci = 0;

    switch(cx->Nkey)
    {
    case 4: do
            {   kf[4] = kf[0] ^ ls_box(kf[3],3) ^ rcon_tab[rci++];
                kf[5] = kf[1] ^ kf[4];
                kf[6] = kf[2] ^ kf[5];
                kf[7] = kf[3] ^ kf[6];
                kf += 4;
            }
            while(kf < kt);
            break;

    case 6: cx->e_key[4] = word_in(in_key + 16);
            cx->e_key[5] = word_in(in_key + 20);
            do
            {   kf[ 6] = kf[0] ^ ls_box(kf[5],3) ^ rcon_tab[rci++];
                kf[ 7] = kf[1] ^ kf[ 6];
                kf[ 8] = kf[2] ^ kf[ 7];
                kf[ 9] = kf[3] ^ kf[ 8];
                kf[10] = kf[4] ^ kf[ 9];
                kf[11] = kf[5] ^ kf[10];
                kf += 6;
            }
            while(kf < kt);
            break;

    case 8: cx->e_key[4] = word_in(in_key + 16);
            cx->e_key[5] = word_in(in_key + 20);
            cx->e_key[6] = word_in(in_key + 24);
            cx->e_key[7] = word_in(in_key + 28);
            do
            {   kf[ 8] = kf[0] ^ ls_box(kf[7],3) ^ rcon_tab[rci++];
                kf[ 9] = kf[1] ^ kf[ 8];
                kf[10] = kf[2] ^ kf[ 9];
                kf[11] = kf[3] ^ kf[10];
                kf[12] = kf[4] ^ ls_box(kf[11],0);
                kf[13] = kf[5] ^ kf[12];
                kf[14] = kf[6] ^ kf[13];
                kf[15] = kf[7] ^ kf[14];
                kf += 8;
            }
            while (kf < kt);
            break;
    }

    if((cx->mode & 3) != enc)
    {   word    i;
        
        kt = cx->d_key + nc * cx->Nrnd;
        kf = cx->e_key;
        
        cpy(kt, kf); kt -= 2 * nc;

        for(i = 1; i < cx->Nrnd; ++i)
        { 
#if defined(ONE_TABLE) || defined(FOUR_TABLES)
#if !defined(ONE_IM_TABLE) && !defined(FOUR_IM_TABLES)
            word    f2, f4, f8, f9;
#endif
            mix(kt, kf);
#else
            cpy(kt, kf);
#endif
            kt -= 2 * nc;
        }
        
        cpy(kt, kf);
    }

    return aes_good;
}

 /*
   I am grateful to Frank Yellin for the following constructions
   which, given the column (c) of the output state variable, give
   the input state variables which are needed for each row (r) of 
   the state.

   For the fixed block size options, compilers should reduce these 
   two expressions to fixed variable references. But for variable 
   block size code conditional clauses will sometimes be returned.

   y = output word, x = input word, r = row, c = column for r = 0, 
   1, 2 and 3 = column accessed for row r.
 */

#define unused  77  /* Sunset Strip */

#define fwd_var(x,r,c) \
 ( r==0 ?           \
    ( c==0 ? s(x,0) \
    : c==1 ? s(x,1) \
    : c==2 ? s(x,2) \
    : c==3 ? s(x,3) \
    : c==4 ? s(x,4) \
    : c==5 ? s(x,5) \
    : c==6 ? s(x,6) \
    : s(x,7))       \
 : r==1 ?           \
    ( c==0 ? s(x,1) \
    : c==1 ? s(x,2) \
    : c==2 ? s(x,3) \
    : c==3 ? nc==4 ? s(x,0) : s(x,4) \
    : c==4 ? s(x,5) \
    : c==5 ? nc==8 ? s(x,6) : s(x,0) \
    : c==6 ? s(x,7) \
    : s(x,0))       \
 : r==2 ?           \
    ( c==0 ? nc==8 ? s(x,3) : s(x,2) \
    : c==1 ? nc==8 ? s(x,4) : s(x,3) \
    : c==2 ? nc==4 ? s(x,0) : nc==8 ? s(x,5) : s(x,4) \
    : c==3 ? nc==4 ? s(x,1) : nc==8 ? s(x,6) : s(x,5) \
    : c==4 ? nc==8 ? s(x,7) : s(x,0) \
    : c==5 ? nc==8 ? s(x,0) : s(x,1) \
    : c==6 ? s(x,1) \
    : s(x,2))       \
 :                  \
    ( c==0 ? nc==8 ? s(x,4) : s(x,3) \
    : c==1 ? nc==4 ? s(x,0) : nc==8 ? s(x,5) : s(x,4) \
    : c==2 ? nc==4 ? s(x,1) : nc==8 ? s(x,6) : s(x,5) \
    : c==3 ? nc==4 ? s(x,2) : nc==8 ? s(x,7) : s(x,0) \
    : c==4 ? nc==8 ? s(x,0) : s(x,1) \
    : c==5 ? nc==8 ? s(x,1) : s(x,2) \
    : c==6 ? s(x,2) \
    : s(x,3)))

#define inv_var(x,r,c) \
 ( r==0 ?           \
    ( c==0 ? s(x,0) \
    : c==1 ? s(x,1) \
    : c==2 ? s(x,2) \
    : c==3 ? s(x,3) \
    : c==4 ? s(x,4) \
    : c==5 ? s(x,5) \
    : c==6 ? s(x,6) \
    : s(x,7))       \
 : r==1 ?           \
    ( c==0 ? nc==4 ? s(x,3) : nc==8 ? s(x,7) : s(x,5) \
    : c==1 ? s(x,0) \
    : c==2 ? s(x,1) \
    : c==3 ? s(x,2) \
    : c==4 ? s(x,3) \
    : c==5 ? s(x,4) \
    : c==6 ? s(x,5) \
    : s(x,6))       \
 : r==2 ?           \
    ( c==0 ? nc==4 ? s(x,2) : nc==8 ? s(x,5) : s(x,4) \
    : c==1 ? nc==4 ? s(x,3) : nc==8 ? s(x,6) : s(x,5) \
    : c==2 ? nc==8 ? s(x,7) : s(x,0) \
    : c==3 ? nc==8 ? s(x,0) : s(x,1) \
    : c==4 ? nc==8 ? s(x,1) : s(x,2) \
    : c==5 ? nc==8 ? s(x,2) : s(x,3) \
    : c==6 ? s(x,3) \
    : s(x,4))       \
 :                  \
    ( c==0 ? nc==4 ? s(x,1) : nc==8 ? s(x,4) : s(x,3) \
    : c==1 ? nc==4 ? s(x,2) : nc==8 ? s(x,5) : s(x,4) \
    : c==2 ? nc==4 ? s(x,3) : nc==8 ? s(x,6) : s(x,5) \
    : c==3 ? nc==8 ? s(x,7) : s(x,0) \
    : c==4 ? nc==8 ? s(x,0) : s(x,1) \
    : c==5 ? nc==8 ? s(x,1) : s(x,2) \
    : c==6 ? s(x,2) \
    : s(x,3)))

#define si(y,x,k,c) s(y,c) = word_in(x + 4 * c) ^ k[c]
#define so(y,x,c)   word_out(y + 4 * c, s(x,c))

#if defined(FOUR_TABLES)
#define fwd_rnd(y,x,k,c)    s(y,c)= (k)[c] ^ four_tables(x,ft_tab,fwd_var,rf1,c)
#define inv_rnd(y,x,k,c)    s(y,c)= (k)[c] ^ four_tables(x,it_tab,inv_var,rf1,c)
#elif defined(ONE_TABLE)
#define fwd_rnd(y,x,k,c)    s(y,c)= (k)[c] ^ one_table(x,upr,ft_tab,fwd_var,rf1,c)
#define inv_rnd(y,x,k,c)    s(y,c)= (k)[c] ^ one_table(x,upr,it_tab,inv_var,rf1,c)
#else
#define fwd_rnd(y,x,k,c)    s(y,c) = fwd_mcol(no_table(x,s_box,fwd_var,rf1,c)) ^ (k)[c]
#define inv_rnd(y,x,k,c)    s(y,c) = inv_mcol(no_table(x,inv_s_box,inv_var,rf1,c) ^ (k)[c])
#endif

#if defined(FOUR_LR_TABLES)
#define fwd_lrnd(y,x,k,c)   s(y,c)= (k)[c] ^ four_tables(x,fl_tab,fwd_var,rf1,c)
#define inv_lrnd(y,x,k,c)   s(y,c)= (k)[c] ^ four_tables(x,il_tab,inv_var,rf1,c)
#elif defined(ONE_LR_TABLE)
#define fwd_lrnd(y,x,k,c)   s(y,c)= (k)[c] ^ one_table(x,ups,fl_tab,fwd_var,rf1,c)
#define inv_lrnd(y,x,k,c)   s(y,c)= (k)[c] ^ one_table(x,ups,il_tab,inv_var,rf1,c)
#else
#define fwd_lrnd(y,x,k,c)   s(y,c) = no_table(x,s_box,fwd_var,rf1,c) ^ (k)[c]
#define inv_lrnd(y,x,k,c)   s(y,c) = no_table(x,inv_s_box,inv_var,rf1,c) ^ (k)[c]
#endif

#if MMHBLOCK_SIZE == 16

#if defined(ARRAYS)
#define locals(y,x)     x[4],y[4]
#else
#define locals(y,x)     x##0,x##1,x##2,x##3,y##0,y##1,y##2,y##3
 /* 
   the following defines prevent the compiler requiring the declaration
   of generated but unused variables in the fwd_var and inv_var macros
 */
#define b04 unused
#define b05 unused
#define b06 unused
#define b07 unused
#define b14 unused
#define b15 unused
#define b16 unused
#define b17 unused
#endif
#define l_copy(y, x)    s(y,0) = s(x,0); s(y,1) = s(x,1); \
                        s(y,2) = s(x,2); s(y,3) = s(x,3);
#define state_in(y,x,k) si(y,x,k,0); si(y,x,k,1); si(y,x,k,2); si(y,x,k,3)
#define state_out(y,x)  so(y,x,0); so(y,x,1); so(y,x,2); so(y,x,3)
#define round(rm,y,x,k) rm(y,x,k,0); rm(y,x,k,1); rm(y,x,k,2); rm(y,x,k,3)

#elif MMHBLOCK_SIZE == 24

#if defined(ARRAYS)
#define locals(y,x)     x[6],y[6]
#else
#define locals(y,x)     x##0,x##1,x##2,x##3,x##4,x##5, \
                        y##0,y##1,y##2,y##3,y##4,y##5
#define b06 unused
#define b07 unused
#define b16 unused
#define b17 unused
#endif
#define l_copy(y, x)    s(y,0) = s(x,0); s(y,1) = s(x,1); \
                        s(y,2) = s(x,2); s(y,3) = s(x,3); \
                        s(y,4) = s(x,4); s(y,5) = s(x,5);
#define state_in(y,x,k) si(y,x,k,0); si(y,x,k,1); si(y,x,k,2); \
                        si(y,x,k,3); si(y,x,k,4); si(y,x,k,5)
#define state_out(y,x)  so(y,x,0); so(y,x,1); so(y,x,2); \
                        so(y,x,3); so(y,x,4); so(y,x,5)
#define round(rm,y,x,k) rm(y,x,k,0); rm(y,x,k,1); rm(y,x,k,2); \
                        rm(y,x,k,3); rm(y,x,k,4); rm(y,x,k,5)
#else

#if defined(ARRAYS)
#define locals(y,x)     x[8],y[8]
#else
#define locals(y,x)     x##0,x##1,x##2,x##3,x##4,x##5,x##6,x##7, \
                        y##0,y##1,y##2,y##3,y##4,y##5,y##6,y##7
#endif
#define l_copy(y, x)    s(y,0) = s(x,0); s(y,1) = s(x,1); \
                        s(y,2) = s(x,2); s(y,3) = s(x,3); \
                        s(y,4) = s(x,4); s(y,5) = s(x,5); \
                        s(y,6) = s(x,6); s(y,7) = s(x,7);

#if MMHBLOCK_SIZE == 32

#define state_in(y,x,k) si(y,x,k,0); si(y,x,k,1); si(y,x,k,2); si(y,x,k,3); \
                        si(y,x,k,4); si(y,x,k,5); si(y,x,k,6); si(y,x,k,7)
#define state_out(y,x)  so(y,x,0); so(y,x,1); so(y,x,2); so(y,x,3); \
                        so(y,x,4); so(y,x,5); so(y,x,6); so(y,x,7)
#define round(rm,y,x,k) rm(y,x,k,0); rm(y,x,k,1); rm(y,x,k,2); rm(y,x,k,3); \
                        rm(y,x,k,4); rm(y,x,k,5); rm(y,x,k,6); rm(y,x,k,7)
#else

#define state_in(y,x,k) \
switch(nc) \
{   case 8: si(y,x,k,7); si(y,x,k,6); \
    case 6: si(y,x,k,5); si(y,x,k,4); \
    case 4: si(y,x,k,3); si(y,x,k,2); \
            si(y,x,k,1); si(y,x,k,0); \
}

#define state_out(y,x) \
switch(nc) \
{   case 8: so(y,x,7); so(y,x,6); \
    case 6: so(y,x,5); so(y,x,4); \
    case 4: so(y,x,3); so(y,x,2); \
            so(y,x,1); so(y,x,0); \
}

#if defined(FAST_VARIABLE)

#define round(rm,y,x,k) \
switch(nc) \
{   case 8: rm(y,x,k,7); rm(y,x,k,6); \
            rm(y,x,k,5); rm(y,x,k,4); \
            rm(y,x,k,3); rm(y,x,k,2); \
            rm(y,x,k,1); rm(y,x,k,0); \
            break; \
    case 6: rm(y,x,k,5); rm(y,x,k,4); \
            rm(y,x,k,3); rm(y,x,k,2); \
            rm(y,x,k,1); rm(y,x,k,0); \
            break; \
    case 4: rm(y,x,k,3); rm(y,x,k,2); \
            rm(y,x,k,1); rm(y,x,k,0); \
            break; \
}
#else

#define round(rm,y,x,k) \
switch(nc) \
{   case 8: rm(y,x,k,7); rm(y,x,k,6); \
    case 6: rm(y,x,k,5); rm(y,x,k,4); \
    case 4: rm(y,x,k,3); rm(y,x,k,2); \
            rm(y,x,k,1); rm(y,x,k,0); \
}

#endif

#endif
#endif

cf_dec c_name(encrypt)(const byte in_blk[], byte out_blk[], const c_name(aes) *cx)
{   word        locals(b0, b1);
    const word  *kp = cx->e_key;

#if !defined(ONE_TABLE) && !defined(FOUR_TABLES)
    word        f2;
#endif

    if(!(cx->mode & 0x01)) return aes_bad;

    state_in(b0, in_blk, kp); kp += nc;

#if defined(UNROLL)

    switch(cx->Nrnd)
    {
    case 14:    round(fwd_rnd,  b1, b0, kp         ); 
                round(fwd_rnd,  b0, b1, kp + nc    ); kp += 2 * nc;
    case 12:    round(fwd_rnd,  b1, b0, kp         ); 
                round(fwd_rnd,  b0, b1, kp + nc    ); kp += 2 * nc;
    case 10:    round(fwd_rnd,  b1, b0, kp         );             
                round(fwd_rnd,  b0, b1, kp +     nc);
                round(fwd_rnd,  b1, b0, kp + 2 * nc); 
                round(fwd_rnd,  b0, b1, kp + 3 * nc);
                round(fwd_rnd,  b1, b0, kp + 4 * nc); 
                round(fwd_rnd,  b0, b1, kp + 5 * nc);
                round(fwd_rnd,  b1, b0, kp + 6 * nc); 
                round(fwd_rnd,  b0, b1, kp + 7 * nc);
                round(fwd_rnd,  b1, b0, kp + 8 * nc);
                round(fwd_lrnd, b0, b1, kp + 9 * nc);
    }
#elif defined(PARTIAL_UNROLL)
    {   word    rnd;

        for(rnd = 0; rnd < (cx->Nrnd >> 1) - 1; ++rnd)
        {
            round(fwd_rnd, b1, b0, kp); 
            round(fwd_rnd, b0, b1, kp + nc); kp += 2 * nc;
        }

        round(fwd_rnd,  b1, b0, kp);
        round(fwd_lrnd, b0, b1, kp + nc);
    }
#else
    {   word    rnd;

        for(rnd = 0; rnd < cx->Nrnd - 1; ++rnd)
        {
            round(fwd_rnd, b1, b0, kp); 
            l_copy(b0, b1); kp += nc;
        }

        round(fwd_lrnd, b0, b1, kp);
    }
#endif

    state_out(out_blk, b0);
    return aes_good;
}

cf_dec c_name(decrypt)(const byte in_blk[], byte out_blk[], const c_name(aes) *cx)
{   word        locals(b0, b1);
    const word  *kp = cx->d_key;

#if !defined(ONE_TABLE) && !defined(FOUR_TABLES)
    word        f2, f4, f8, f9; 
#endif

    if(!(cx->mode & 0x02)) return aes_bad;

    state_in(b0, in_blk, kp); kp += nc;

#if defined(UNROLL)

    switch(cx->Nrnd)
    {
    case 14:    round(inv_rnd,  b1, b0, kp         );
                round(inv_rnd,  b0, b1, kp + nc    ); kp += 2 * nc;
    case 12:    round(inv_rnd,  b1, b0, kp         );
                round(inv_rnd,  b0, b1, kp + nc    ); kp += 2 * nc;
    case 10:    round(inv_rnd,  b1, b0, kp         );             
                round(inv_rnd,  b0, b1, kp +     nc);
                round(inv_rnd,  b1, b0, kp + 2 * nc); 
                round(inv_rnd,  b0, b1, kp + 3 * nc);
                round(inv_rnd,  b1, b0, kp + 4 * nc); 
                round(inv_rnd,  b0, b1, kp + 5 * nc);
                round(inv_rnd,  b1, b0, kp + 6 * nc); 
                round(inv_rnd,  b0, b1, kp + 7 * nc);
                round(inv_rnd,  b1, b0, kp + 8 * nc);
                round(inv_lrnd, b0, b1, kp + 9 * nc);
    }
#elif defined(PARTIAL_UNROLL)
    {   word    rnd;

        for(rnd = 0; rnd < (cx->Nrnd >> 1) - 1; ++rnd)
        {
            round(inv_rnd, b1, b0, kp); 
            round(inv_rnd, b0, b1, kp + nc); kp += 2 * nc;
        }

        round(inv_rnd,  b1, b0, kp);
        round(inv_lrnd, b0, b1, kp + nc);
    }
#else
    {   word    rnd;

        for(rnd = 0; rnd < cx->Nrnd - 1; ++rnd)
        {
            round(inv_rnd, b1, b0, kp); 
            l_copy(b0, b1); kp += nc;
        }

        round(inv_lrnd, b0, b1, kp);
    }
#endif

    state_out(out_blk, b0);
    return aes_good;
}
