/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Servers.frm  -  Base Class for TCP/IP Socket Servers                    */
/*                                                                          */
/*  History:                                                                */
/*    06/30/97  added client alias option for multiple clients per host     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
                                       
::REQUIRES "sockets.frm"               /* Load the sockets framework        */   

/*--------------------------------------------------------------------------*/
/* tcpServer Class definition                                               */
/*--------------------------------------------------------------------------*/
::CLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Get sockets and binds port        */ 
  expose sSocket port
  use arg port                         /* Server port is parameter          */                          

  say "*** tcpServer starting"
  sSocket = .tcpSocket~new             /* Get a stream socket               */
  
/*--------------------------------------------------------------------------*/
::METHOD StartAccepting                /* Accept and service clients        */ 
  expose sSocket port

  if sSocket~Bind(port) < 0 then do    /* Bind server socket to known port  */
    say "Bind of port" port "to" sSocket "failed!"
    say "Server could not be started!"
    return -999                        /* Terminate server process          */
  end

  say "*** tcpServer at port" port "started on" sSocket
  say "    Enter 'Ctrl-c' to shut down server"
         
  sSocket~listen(10)                   /* Listen to queue of max 10 entries */
  signal on syntax                     /* Intercept ctrl-c                  */

  do forever                           /* Start service until 'shutdown'    */   
    say "*** Server is waiting to accept clients"
                                       /* Accept client request, get socket */             
    cSocket = sSocket~accept           /* this is blocking until a client   */
                                       /* connects / server socket goes down*/ 

    if cSocket = -1 then leave         /* request queue still active?       */ 

    prefix = cSocket~ReceiveData       /* Receive client prefix             */   
    client = clientName(cSocket, prefix~word(1)) /* get full client name    */                       
    prefix = client prefix~subword(2)  /* expand the prefix                 */

    say " ** Accepted client:" client '(' || prefix~word(2) || ')'
    say "    IP address" cSocket~dotAddress "on" cSocket

    self~NewClient(cSocket, prefix)    /* New client, must be non-blocking! */
  end

syntax:                                
  return                               /* return upon 'Ctrl-c' or Close     */

/*--------------------------------------------------------------------------*/
::METHOD Close UNGUARDED               /* Close server request queue        */ 
  expose sSocket

  sSocket~shutdown                     /* Shutdown the server socket        */ 

/*--------------------------------------------------------------------------*/
::METHOD Shutdown UNGUARDED            /* Shutdown the server               */ 
  expose sSocket
          
  say "*** tcpServer shutdown"
  sSocket~Close                        /* Close the server socket           */

/*--------------------------------------------------------------------------*/
::METHOD NewClient UNGUARDED           /* Process client request on socket  */    
  use arg cSocket, prefix                     

  cSocket~SendData(cSocket~ReceiveData)/* Reflect packet back to client     */   
  cSocket~Close                        /* Close client's socket             */  

  /* NOTE: nothing must be returned otherwise accepting new clients would   */
  /*       be blocked!                                                      */

/*--------------------------------------------------------------------------*/
::METHOD RemoveClient UNGUARDED        /* Remove a client connection        */    
  use arg cSocket, Name                                                       
                    
  cSocket~Close                        /* Close client's socket             */  
  say " ** Removed client:" Name                                              
                                                                              
/*--------------------------------------------------------------------------*/ 
::ROUTINE ClientName                   /* Generate compound client name     */    
  use arg cSocket, name                                                      
                                                                               
  if name = '.' then                   /* if not specified, use hostname    */ 
    return cSocket~HostName                                                    
  else                                 /* use client_server compound name   */ 
    return name'_'cSocket~HostName                                   

