/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Sockets.frm  -  Classes for using TCP Sockets                           */
/*                                                                          */
/*    Requirements:                                                         */
/*       RxSock library (Rexx interface to TCP Sockets library)             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
                                       /* Load RxSock interface             */
if RxFuncQuery("SockLoadFuncs") then do
  rc = RxFuncAdd("SockLoadFuncs", "rxsock", "SockLoadFuncs")
  rc = "SockLoadFuncs"()
end

/*--------------------------------------------------------------------------*/
/* tcpSocket Class definition                                               */
/*--------------------------------------------------------------------------*/
::CLASS tcpSocket public

/*--------------------------------------------------------------------------*/
::METHOD init
  expose descriptor notClosed

  notClosed = 1
  if arg() = 1 then                    /* Socket exists; accept descriptor  */
    descriptor = arg(1)                /* Simply reuse this socket          */
  else do
    domain = "AF_INET"                 /* "AF_INET" is default domain       */
    type = "SOCK_STREAM"               /* "SOCK_STREAM" is default type     */
    protocol = 0                       /* use the default protocol          */
                                       /* Create a specific socket          */
    descriptor = SockSocket(domain, type, protocol)
    if descriptor = -1 then do         /* Failed to create socket           */
      SockPSock_Errno("SockSocket")
      return -1
    end 
    else do                            /* Socket successfully created       */  
      if value('tcpDebug',, 'ENVIRONMENT')~translate = 'YES' then
        say self~string "created."
      return 0
    end
  end

/*--------------------------------------------------------------------------*/
::METHOD string
  expose descriptor

  return "tcpSocket" descriptor        /* Return the default string         */

/*--------------------------------------------------------------------------*/
::METHOD accept                        /* Accept client request             */
  expose descriptor

  connection = SockAccept(descriptor)  /* ... this is a blocking call!      */
  if connection = -1 then do
    if errno \= 'ECONNABORTED' then 
      SockPSock_Errno("SockAccept")
    return -1
  end
  return .tcpSocket~new(connection)    /* Return a new client socket        */ 

/*--------------------------------------------------------------------------*/
::METHOD bind                          /* Bind local name (port) to socket  */
  expose descriptor
  use arg port

  server.!family = "AF_INET"           /* Default domain                    */ 
  server.!port = port                  /* Local name (port)                 */ 
  server.!addr = "INADDR_ANY"          /* Accept any client address         */

  rc = SockBind(descriptor, "server.!")
  if rc = -1 then do
    SockPSock_Errno("SockBind")
    return -1
  end
  else 
    return 0

/*--------------------------------------------------------------------------*/
::METHOD connect                       /* Connect client socket to server   */
  expose descriptor
  use arg port, addr

  server.!family = "AF_INET"           /* Use default server domain         */ 
  server.!port = port                  /* at wellknown port                 */                     
  server.!addr = addr                  /* with IP address                   */

  rc = SockConnect(descriptor, "server.!")
  if rc = -1 then do
    SockPSock_Errno("SockConnect")
    return -1
  end
  return 0

/*--------------------------------------------------------------------------*/
::METHOD listen                        /* Listen to client's conn. request  */
  expose descriptor                    /* queue with max. backlog requests  */ 
  use arg backlog                      

  rc = SockListen(descriptor, backlog)
  if rc = -1 then do
    SockPSock_Errno("SockListen")
    return -1
  end
  return 0
                     
/*--------------------------------------------------------------------------*/
::METHOD DotAddress                    /* Get dotaddr. of system on socket  */
  expose descriptor

  rc = SockGetPeerName(descriptor, "Addr.!")
  if rc = -1 then return -1
  return Addr.!addr

/*--------------------------------------------------------------------------*/
::METHOD HostName                      /* Get hostname of system on socket  */ 
  expose descriptor

  DotAddr = self~DotAddress            /* Retrieve dot-address of host      */ 
  if DotAddr = -1 then  
    return -1                          /* No host available, socket invalid */ 
  else do                              /* Retrieve host name from address   */
    rc = SockGetHostByAddr(DotAddr, "Host.!")
    if rc = -1 then                      
      return "*Unknown*"               /* Host name not available           */
    else 
      return Host.!name                /* Return the host name              */
  end

/*--------------------------------------------------------------------------*/
::METHOD SendData UNGUARDED            /* Send data over socket             */ 
  expose descriptor
  use arg data
 
  return SockSend(descriptor, data)

  /* Alternative solution to transmit large amount od data   (with header)  */
/*esc = x2c('00')                      /* use null for separator            */ 
  header = 'size='data~length          /* store length in header            */
  data = header || esc || data         /* prefix data by header             */
  do while data \= ''
    parse var data packet 2000 data    /* 2000 bytes per packet             */
    rc = SockSend(descriptor, packet)  /* send the packet                   */
  end
  return 0                                                                  */


/*--------------------------------------------------------------------------*/
::METHOD ReceiveData UNGUARDED         /* Receive data from socket          */  
  expose descriptor                    /* ... this is a blocking read!      */ 
  
  rc = SockRecv(descriptor, "packet", 2000)
  if rc > 0 then 
    return packet                      /* Return the received packet of data*/ 
  else do
    self~shutdown                      /* Shutdown communication on socket  */ 
    self~close                         /* and close it                      */
  end
  return rc

  /* Alternative solution to transmit large amount od data (with header)    */
/*esc = x2c('00')                      /* use null for separator            */ 
  rc = SockRecv(descriptor, "data", 2000)/* receive first packet            */
  parse var data 'size='size (esc) data/* parse the header                  */
  do while data~length < size
    rc = SockRecv(descriptor, "packet", 2000)/* receive next packet         */
    data = data || packet              /* append packet to data             */
  end
  return data                                                               */


/*--------------------------------------------------------------------------*/
::METHOD sendFile UNGUARDED            /* send the content of a file        */
  expose descriptor 
  use arg file
      
  stream = .stream~new(file)           /* create a stream object for file   */
                                       /* does file exist?                  */ 
  if stream~query("EXISTS") = "" then do
    say "The file '"file"' does not exist" 
    header = 'size='0'00'x             /* generate a dummy header           */
    rc = SockSend(descriptor, header)  /* send a dummy packet               */
  end
  else do
    stream~open('READ')                /* open the file in read mode        */ 
    size = stream~chars                /* get the size of the file          */
    header = 'size='size'00'x          /* store length in header            */
                                       /* determin the size to be read      */ 
    gsize = size~min(2000 - header~length)
                                       /* generate the first packet         */  
    packet = header || stream~charin(, gsize)
    rc = SockSend(descriptor, packet)  /* send the packet                   */
    size = size - gsize                /* calculate the remaining size      */ 

    do while size>0 & rc>0             /* read and send the remaining data  */
      gsize = size~min(2000)           /* determin the size to be read      */
      packet = stream~charin(, gsize)  /* read the next packet from file    */
      rc = SockSend(descriptor, packet)/* send the packet                   */
      size = size - gsize              /* calculate the remaining size      */
    end
    stream~close                       /* close the file                    */
  end 
  return rc

/*--------------------------------------------------------------------------*/
::METHOD receiveFile UNGUARDED         /* receive the content of a file     */
  expose descriptor 
  
  if arg() = 0 then do                 /* receive as transient data         */
    size = fileHeader(descriptor)      /* receive file header               */  
    if size <= 0 then do
      rc = size
      size = 0
    end
    data = ''                          /* start with empty data buffer      */
    if size > 0 then do
      do while data~length < size      /* receive 'size' bytes of data      */
        rc = SockRecv(descriptor, "packet", (size-data~length)~min(2000) )
        if rc <= 0 then signal problem  
        data = data || packet          /* append packet to data             */
      end
    end 
    return data                        /* return the received file content  */                 
  end
  else do                              /* receive as persistent data (file) */
    length = 0                         /* open file                         */  
    stream = .stream~new(arg(1))~~open('WRITE REPLACE')
    if arg() = 1 | arg() > 1 & arg(2) \= 'NOHEADER' then do                
      size = fileHeader(descriptor)    /* receive file header               */
      if size <= 0 then do             
        rc = size
        size = 0
      end
      do while length < size           /* receive 'size' bytes of data      */
        rc = SockRecv(descriptor, "packet", (size-length)~min(2000))
        if rc <= 0 then leave 
        stream~charout(packet)         /* and write received packet to file */ 
        length = length + packet~length
      end 
    end 
    else do forever                    /* stream w/o header, NOHEADER opt.  */
      rc = SockRecv(descriptor, "packet", 2000)
      if rc <= 0 | packet = '' then leave 
      stream~charout(packet)           /* and write received packet to file */ 
      length = length + packet~length
    end 
    state = stream~state               /* check the stream state            */
    stream~close                       /* now close the file                */
    if state \= 'READY' then return -1 /* return with stream problems       */
    if rc >= 0 then return length      /* if ok, return number of bytes rec */ 
  end 

problem:
  self~shutdown                        /* Shutdown communication on socket  */ 
  self~close                           /* and close it                      */
  return rc                            /* return bad (negative) code        */

/*--------------------------------------------------------------------------*/
::METHOD Shutdown UNGUARDED            /* Shutdown communication on socket  */ 
  expose descriptor notClosed 
  use arg mode
    
  if arg() = 0 then mode = 2           /* default: shutdown for read/write  */ 

  if notClosed then do 
    rc = SockShutDown(descriptor, mode)
    if rc \= 0 then do
      SockPSock_Errno("SockShutDown")
      return rc
    end
    if value('tcpDebug',, 'ENVIRONMENT')~translate = 'YES' then
      say self~string "shutdown"
  end
  return 0

/*--------------------------------------------------------------------------*/
::METHOD Close UNGUARDED               /* Recycle the socket resource       */   
  expose descriptor notClosed
  
  if notClosed then do 
    notClosed = 0                      /* Set its status closed             */ 
    rc = SockSoClose(descriptor)
    if rc = -1 then do                 /* Problem with closing socket       */
      SockPSock_Errno("SockSoClose") 
      return -1
    end
    if value('tcpDebug',, 'ENVIRONMENT')~translate = 'YES' then
      say self~string "closed"
  end
  return 0

/*--------------------------------------------------------------------------*/
::METHOD StillOpen UNGUARDED           /* Provide status information        */
  expose NotClosed

  return NotClosed

/*--------------------------------------------------------------------------*/
::METHOD uninit                        /* Just in case we forgot the Close  */  
  expose notClosed
 
  if notClosed then self~Close

/*--------------------------------------------------------------------------*/
::ROUTINE fileHeader                   /* Read the file header              */  
  use arg descriptor
                                       /* peek for the header               */
  rc = SockRecv(descriptor, "data", 20, 'MSG_PEEK')
  if rc <= 0 then signal problem  
  parse var data 'size=' size '00'x .  /* parse the header                  */
                                       /* discard the header                */
  rc = SockRecv(descriptor, "data", data~pos('00'x))
  if rc > 0 then
    return size

problem: 
  return rc


/*--------------------------------------------------------------------------*/
/* tcpHost Class definition                                                 */
/*                                                                          */
/*   tcpHost objects are used to test a supplied hostname. If the host is   */
/*   found, information about it is available via various methods.          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
::CLASS tcpHost PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init
  expose name address addressArray aliasArray
  use arg hoststring
  
  if arg() = 0 | hoststring = '' then
    hoststring = SockGetHostId()

  if hoststring~left(1)~datatype('N') then
    rc = SockGetHostByAddr(hoststring, "Host.!")
  else
    rc = SockGetHostByName(hoststring, "Host.!")

  if rc = 0 then do
    name = ""
    address = ""
    say "Error" h_errno "accessing host. '"hoststring"' unknown."
    return 
  end 

  name = Host.!name
  address = Host.!addr

  addressArray = .array~new
  do i = 1 to Host.!addr.0
    addressArray[i] = Host.!addr.i
  end
    
  aliasArray = .array~new
  do i = 1 to Host.!alias.0
    aliasArray[i] = Host.!alias.i
  end

/*--------------------------------------------------------------------------*/
::METHOD name
  expose name 
  return name

/*--------------------------------------------------------------------------*/
::METHOD address
  expose address 
  return address

/*--------------------------------------------------------------------------*/
::METHOD addressArray
  expose addressArray
  return addressArray~copy

/*--------------------------------------------------------------------------*/
::METHOD aliasArray
  expose aliasArray
  return aliasArray~copy
