/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Yodel_s.cmd  -  A TCP/IP Server with sessions                           */
/*                                                                          */
/*  History:                                                                */
/*    07/20/97  improved shutdown behaviour of server                       */
/*              waiter instance with defined wating time                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/

Parse Arg Port 
if Port = '' then Port = 1923          /* Default server port is 1923       */
aYodelServer = .YodelServer~new(Port)  /* Create a yodel server at port     */
aYodelServer~startAccepting            /* Listen at socket for clients      */    
aYodelServer~shutdown                  /* shut the yodel server down        */    

/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   

/*--------------------------------------------------------------------------*/
/* Repeater Class definition                                                */
/*--------------------------------------------------------------------------*/
::CLASS YodelServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD NewClient                     /* Over-write superclass method      */    
  use arg cSocket, prefix
                                       /* Create a yodeller instance        */
  aYodelClient = .Yodeller~new(cSocket, prefix~word(1))
  aYodelClient~Listen                  /* make him listening to any words   */
  aYodelClient~Yodel                   /* make him yodel the words heard    */


/*--------------------------------------------------------------------------*/
/* Yodeller Class definition                                                */
/*--------------------------------------------------------------------------*/
::CLASS Yodeller                       /* One for each client               */    

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Initialize a yodeller instance    */    
  expose mySocket myName theWords 
  use arg mySocket, fullName

  myName = fullName                    /* This is my name                   */    
  theWords = "Yoda-LADY-oh!"           /* The initial words                 */

/*--------------------------------------------------------------------------*/
::METHOD Listen UNGUARDED 
  expose mySocket myName theWords 
  reply                                /* Frees up for other concurrent work*/ 
                                       /* Process the rest in another thread*/
  say "  * Yodeller started listening (client:" myName ||")"
                                       /* Repeat until the words are 'quit' */    
  do until theWords~word(1)~translate = "QUIT" 
    theWords = mySocket~ReceiveData
    if \mySocket~stillOpen then leave  /* if problems, end client session   */ 
    if theWords = '' then 
      theWords = "Yoda-LADY-Yohoh!"
  end
  say "  * Yodeller ended listening (client:" myName ||")"
  
/*--------------------------------------------------------------------------*/
::METHOD Yodel UNGUARDED
  expose mySocket myName theWords 
  reply                                /* Frees up for other concurrent work*/ 
                                       /* Process the rest in another thread*/
  say "  * Yodeller started yodelling (client:" myName ||")"
  myWaiter = .Waiter~new(2)            /* Create an appropriate waiter      */
                                       /* Yodel loop until 'quit'           */   
  do while theWords~word(1)~translate \= 'QUIT'  
    myWaiter~wait                      /* Wait for the defined time         */
    mySocket~SendData(theWords)        /* then send the words               */
    if \mySocket~stillOpen then leave  /* if problems, end yodelling        */ 
  end

  self~shutdown                        /* Now shutdown the yodel client     */

/*--------------------------------------------------------------------------*/
::METHOD shutdown                      /* Shutdown the client               */
  expose mySocket myName theWords    

  say "  * Yodeller ended yodelling (client:" myName ||")"
  mySocket~SendData("#>>End_of_session<<#")
  say " ** Shutdown yodeller:" myName 
  mySocket~Close                       /* Close this client socket          */ 


/*--------------------------------------------------------------------------*/
/* Waiter Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS Waiter PUBLIC                  

/*--------------------------------------------------------------------------*/
::METHOD init                          /* This method allows to pause       */ 
  expose seconds
  use arg seconds                      

/*--------------------------------------------------------------------------*/
::METHOD wait UNGUARDED                /* This method allows to pause       */ 
  expose seconds                      
                                       /* Create a 'complete' message       */  
  msg = .message~new(self, "COMPLETE", 'A', .array~new)
  .alarm~new(seconds, msg)             /* Wait and then send the message    */ 
  rc = msg~result                      /* Block on message completion       */ 

/*--------------------------------------------------------------------------*/
::METHOD complete UNGUARDED
  return 0                             /* Must have a return with a value!  */ 
