/*---------------------------------------------------------------------------*/
/*                                                                           */
/*  NetCentric Computing with Object Rexx                                    */
/*  Programming Example                                                      */
/*                                                                           */
/*  (c) Copyright IBM Corporation 1999                                       */
/*                                                                           */
/*  html.frm - HTML Framwork for dynamic HTML generation                     */
/*                                                                           */
/*  Requirements:                                                            */
/*    none                                                                   */
/*                                                                           */
/*  Public Classes:                                                          */
/*    html     -  base HTML support class                                    */
/*    htmldoc  -  HTML document class                                        */
/*    parmdir  -  HTTP parameter support class                               */
/*    head     -  HTML head class                                            */
/*    style    -  HTML style class                                           */
/*    object   -  HTML object class                                          */
/*    iframe   -  HTML inline frame class                                    */
/*    frameset -  HTML frameset class                                        */
/*    br       -  HTML br class                                              */
/*    p        -  HTML p class                                               */
/*    ul       -  HTML ul class                                              */
/*    ol       -  HTML ol class                                              */
/*    dl       -  HTML dl class                                              */
/*    table    -  HTML table class                                           */
/*    form     -  HTML form class                                            */
/*                                                                           */
/*  This code is sample code, provided "AS IS", without warranty of any kind */
/*                                                                           */
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
::METHOD empty                         /* floating method --> parmDir class  */
  return ''                            /* return empty string                */

/*****************************************************************************/
::CLASS htmlbase PUBLIC SUBCLASS queue

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize an html object          */
  forward class (super) arguments(.array~new) continue
  self~setmethod('[]')                 /* hide these inherited methods       */
  self~setmethod('[]=')
  self~setmethod('REMOVE')
  self~setmethod('PEEK')
  self~setmethod('HASINDEX')

/*---------------------------------------------------------------------------*/
::METHOD put                           /* override of the put method         */
  use arg obj
  return self~~queue(obj)

/*---------------------------------------------------------------------------*/
::METHOD resolve                       /* resolve html object                */
  if self~class = .string then
    say self
  else if self~hasmethod('makeHtml') then do
    x = self~makeHtml
    if x~class = .string then
      say self
    else x~resolve
  end
  else do elem over self
    if elem~class = .string then
      say elem
    else if elem~hasmethod('makeHtml') then do
      x = elem~makeHtml
      if x~class = .string then
        say x
      else x~resolve
    end
    else elem~resolve
  end
  return self

/*****************************************************************************/
::CLASS htmldoc PUBLIC subclass html
::METHOD workdir ATTRIBUTE             /* work directory, system dependent   */
::METHOD slash ATTRIBUTE               /* slash character                    */
::METHOD type ATTRIBUTE                /* MIME type                          */

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize an html object          */
  forward class (super) arguments (.array~new) continue
  parse source . . pathname            /* get the pathname of this script    */
  if pathname~substr(1, 1) = '/' then
    self~slash= '/'                    /* Unix style path                    */
  else
    self~slash= '\'                    /* Dos style path                     */
  self~workdir= pathname~left(pathname~lastpos(self~slash))
  self~type = 'text/html'              /* default document type              */
  if arg(1, 'e') then do
    if arg(1)~class = .head then
      self~queue(arg(1))               /* include the head object            */
    else
      self~queue('<title>'arg(1)'</title>')
    self~queue('<body>')
  end

/*---------------------------------------------------------------------------*/
::METHOD write                         /* write html to file                 */
  use arg output                       /* output file                        */

  stream = .stream~new(output)         /* get a stream object                */
  do line over self                    /* write data to file line by line    */
    stream~lineout(line)               /* write out the next line            */
  end
  stream~close                         /* close the file                     */

/*---------------------------------------------------------------------------*/
::METHOD include                       /* include html file                  */
  use arg infile                       /* input file to be included          */

  filespec = changestr('/', infile, self~slash)
  fullname = self~workdir||filespec    /* look for it in the work directoy   */
  stream = .stream~new(fullname)       /* get a stream object                */
  retstr = stream~open('READ')         /* open file for reading              */
  if retstr = "READY:" then do
    do line over stream                /* read the data line by line         */
      self~queue(line)                 /* include line                       */
    end
    stream~close                       /* close the file                     */
  end
  else                                 /* include error message instead      */
    self~queue('<font color=red>Error including file' fullname '--' retstr'</font>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD filelist                      /* get list of files                  */
  use arg filespec                     /* file the file size is queried      */

  filespec = changestr('/', filespec, self~slash)
  fullname = self~workdir||filespec  /* look for it in the work directoy   */

  if sysfiletree(fullname, stem.) = 0 then do
    list = .array~new(stem.0)
    do i = 1 to stem.0
      fullname = word(stem.i, 5)
      list[i] = fullname~substr(1+fullname~lastpos(self~slash))
    end
    return list
  end
  else
    return -1

/*---------------------------------------------------------------------------*/
::METHOD filesize                      /* get size of file                   */
  use arg file                         /* file the size of which is queried  */

  filespec = changestr('/', file, self~slash)
  fullname = self~workdir||filespec    /* look for it in the work directoy   */

  if sysfiletree(fullname, stem.) = 0 then do
    if stem.0 > 0 then
      return word(stem.1, 3)
    else
      return "[File" fullname "not found]"
  end
  else
    return -1

/*---------------------------------------------------------------------------*/
::METHOD end                           /* end HTML document                  */
  self~queue('</body></html>')         /* end body and html                  */
  self~push('<!doctype html public "html4.0">')
  self~push('')
  self~push('Content-Type:' self~type) /* add the content type to the top    */
  return self

/*****************************************************************************/
::CLASS htmlcore PUBLIC

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize an html core object     */
  expose id class lang title style events
  forward class (super) arguments (.array~new) continue
  id = ''
  class = ''
  lang = ''
  title = ''
  style = ''
  events = ''

::method id
  expose id
  use arg id
  return self

::method !class
  expose class
  use arg class
  return self

::method lang
  expose lang
  use arg lang
  return self

::method title
  expose title
  use arg title
  return self

::method style
  expose style
  use arg style
  return self

::method events
  expose events
  use arg events
  return self

::method coreattrs
  expose id class lang title style events
  string = ''
  if id \= '' then string = string' id="'id'"'
  if class \= '' then string = string' class="'class'"'
  if title \= '' then string = string' title="'title'"'
  if style \= '' then string = string' style="'style'"'
  if events \= '' then string = string' events="'events'"'
  if lang \= '' then string = string' lang="'lang'"'
  return string

/*****************************************************************************/
::CLASS h1 PUBLIC SUBCLASS htmlcore
::method align attribute
::method text attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  forward class (super) continue
  self~align = ''
  use arg text
  self~text= text

::method makeHtml
  string = '<h1'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h1>'

/*===========================================================================*/
::CLASS h2 PUBLIC SUBCLASS h1

/*---------------------------------------------------------------------------*/
::method makeHtml
  string = '<h2'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h2>'

/*===========================================================================*/
::CLASS h3 PUBLIC SUBCLASS h1

/*---------------------------------------------------------------------------*/
::method makeHtml
  string = '<h3'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h3>'

/*===========================================================================*/
::CLASS h4 PUBLIC SUBCLASS h1

/*---------------------------------------------------------------------------*/
::method makeHtml
  string = '<h4'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h4>'

/*===========================================================================*/
::CLASS h5 PUBLIC SUBCLASS h1

/*---------------------------------------------------------------------------*/
::method makeHtml
  string = '<h5'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h5>'

/*===========================================================================*/
::CLASS h6 PUBLIC SUBCLASS h1

/*---------------------------------------------------------------------------*/
::method makeHtml
  string = '<h6'self~coreattrs
  if self~align \= '' then string = string 'align="'self~align'"'
  return string'>'self~text'</h6>'

/*****************************************************************************/
::CLASS img PUBLIC SUBCLASS htmlcore

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose src align width height border hspace vspace
  forward class (super) continue
  align = ''
  width = ''
  height = ''
  border = ''
  hspace = ''
  vspace = ''
  use arg src
  if src~pos('.') = 0 then src = src".gif"

::method align
  expose align
  use arg align
  return self

::method width
  expose width
  use arg width
  return self

::method height
  expose height
  use arg height
  return self

::method border
  expose border
  use arg border
  return self

::method hspace
  expose hspace
  use arg hspace
  return self

::method vspace
  expose vspace
  use arg vspace
  return self

::method makeHtml
  expose src align width height border hspace vspace
  string = '<img'self~coreattrs 'src="'src'"'
  if align \= '' then string = string 'align="'align'"'
  if width \= '' then string = string 'width="'width'"'
  if height \= '' then string = string 'height="'height'"'
  if border \= '' then string = string 'border="'border'"'
  if hspace \= '' then string = string 'hspace="''hspace"'
  if vspace \= '' then string = string 'vspace="'vspace'"'
  return string'>'

/*****************************************************************************/
::CLASS ul PUBLIC subclass htmlcore
::method body attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose type compact
  forward class (super) continue
  type = ''
  compact = ''
  self~body = .html~new

::method type
  expose type
  use arg type
  return self

::method compact
  expose compact
  use arg compact
  return self

::method li
  if arg(1, 'e') then
    li = .li~new(arg(1))
  else li = .li~new
  self~body~queue(li)
  return li

::method makeHtml
  expose type compact
  string = '<ul'self~coreattrs
  if type \= '' then string = string 'type="'type'"'
  if compact \= '' then string = string 'compact'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</ul>')

/*****************************************************************************/
::CLASS ol PUBLIC subclass htmlcore
::method body attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose type start compact
  forward class (super) continue
  type = ''
  compact = ''
  self~body = .html~new

::method type
  expose type
  use arg type
  return self

::method start
  expose start
  use arg start
  return self

::method compact
  expose compact
  use arg compact
  return self

::method li
  if arg(1, 'e') then
    li = .li~new(arg(1))
  else li = .li~new
  self~body~queue(li)
  return li

::method makeHtml
  expose type start compact
  string = '<ol'self~coreattrs
  if type \= '' then string = string 'type="'type'"'
  if start \= '' then string = string 'start="'start'"'
  if compact \= '' then string = string 'compact'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</ol>')

/*===========================================================================*/
::CLASS li subclass htmlcore                 /* list item class              */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose type value
  forward class (super) continue
  type = ''
  value = ''
  self~body= .html~new
  if arg(1, 'e') then
    self~body~queue(arg(1))

::method type
  expose type
  use arg type
  return self

::method value
  expose value
  use arg value
  return self

::method makeHtml
  expose type value
  string = '<li'self~coreattrs
  if type \= '' then string = string 'type="'type'"'
  if value \= '' then string = string 'value="'value'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</li>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS dl PUBLIC subclass htmlcore
::method body attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose compact
  forward class (super) continue
  compact = ''
  self~body = .html~new

::method compact
  expose compact
  use arg compact
  return self

::method dt
  if arg(1, 'e') then
    dt = .dt~new(arg(1))
  else dt = .dt~new
  self~body~queue(dt)
  return dt

::method dd
  if arg(1, 'e') then
    dd = .dd~new(arg(1))
  else dd = .dd~new
  self~body~queue(dd)
  return dd

::method makeHtml
  expose compact
  string = '<dl'self~coreattrs
  if compact \= '' then string = string 'compact'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</dl>')

/*===========================================================================*/
::CLASS dt PUBLIC subclass htmlcore    /* definition term class              */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  forward class (super) continue
  self~body= .html~new
  self~body~queue(arg(1))

::method makeHtml
  string = '<dt'self~coreattrs
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</dt>')

/*===========================================================================*/
::CLASS dd PUBLIC subclass htmlcore    /* definition data class              */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  forward class (super) continue
  self~body= .html~new
  self~body~queue(arg(1))

::method makeHtml
  string = '<dd'self~coreattrs
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</dd>')

/*****************************************************************************/
::CLASS br PUBLIC subclass htmlcore    /* line break class                   */

/*---------------------------------------------------------------------------*/
::method init
  expose clear body
  forward class (super) continue
  clear = ''
  body= .html~new
  if arg(1, 'e') then
    body~queue(arg(1))

::method clear
  expose clear
  use arg clear
  return self

::method makeHtml
  expose clear body
  string = '<br'self~coreattrs
  if clear \= '' then string = string 'clear="'clear'"'
  return body~~push(string'>')

::method unknown
  expose body
  forward message (arg(1)) arguments(arg(2)) to (body)

/*****************************************************************************/
::CLASS p PUBLIC subclass htmlcore     /* paragraph class                    */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose align
  forward class (super) continue
  align = ''
  self~body= .html~new
  if arg(1, 'e') then
    self~body~queue(arg(1))

::method align
  expose align
  use arg align
  return self

::method makeHtml
  expose align
  string = '<p'self~coreattrs
  if align \= '' then string = string 'align="'align'"'
  return self~body~~push(string'>')~~queue('</p>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS pre PUBLIC subclass htmlcore    /* definition term class              */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  forward class (super) continue
  self~body= .htmlbase~new
  if arg(1, 'e') then
    self~body~queue(arg(1))

::method width
  expose width
  use arg width
  return self

::method makeHtml
  string = '<pre'self~coreattrs
  if width \= '' then string = string 'width="'width'"'
  self~body~push(string'>')
  return self~body~~queue('</pre>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS form PUBLIC SUBCLASS htmlcore
::method body attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose action method enctype onsubmit onreset
  forward class (super) continue
  use arg action
  method = 'GET'
  enctype = ''
  onsubmit = ''
  onreset = ''
  self~body = .html~new

::method method
  expose method
  use arg method
  return self

::method enctype
  expose enctype
  use arg enctype
  return self

::method onsubmit
  expose onsubmit
  use arg onsubmit
  return self

::method onreset
  expose onreset
  use arg onreset
  return self

::method makeHtml
  expose method action onsubmit onreset enctype
  string = '<form'self~coreattrs 'action="'action'"'
  if method \= '' then string = string 'method="'method'"'
  if enctype \= '' then string = string 'enctype="'enctype'"'
  if onsubmit \= '' then string = string 'onsubmit='onsubmit
  if onreset \= '' then string = string 'onreset='onreset
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</form>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS input PUBLIC SUBCLASS htmlcore

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose name value type src align size maxlength checked disabled readonly tabindex
  forward class (super) continue
  name = arg(1)
  value = arg(2)
  type = 'text'
  align = ''
  size = ''
  maxlength = ''
  src = ''
  checked = ''
  disabled = ''
  readonly = ''
  tabindex = ''

::method type
  expose type
  use arg type
  return self

::method align
  expose align
  use arg align
  return self

::method size
  expose size
  use arg size
  return self

::method maxlength
  expose maxlength
  use arg maxlength
  return self

::method src
  expose src
  use arg src
  return self

::method checked
  expose checked
  use arg checked
  return self

::method disabled
  expose disabled
  use arg disabled
  return self

::method readonly
  expose readonly
  use arg readonly
  return self

::method tabindex
  expose tabindex
  use arg tabindex
  return self

::method makeHtml
  expose name value type src align size maxlength checked disabled readonly tabindex
  string = '<input'self~coreattrs 'type="'type'" name="'name'" value="'value'"'
  if size \= '' then string = string 'size="'size'"'
  if maxlength \= '' then string = string 'maxlength="'maxlength'"'
  if src \= '' then string = string 'src="'src'"'
  if align \= '' then string = string 'align="'align'"'
  if checked \= '' then string = string 'checked'
  if disabled \= '' then string = string 'disabled'
  if readonly \= '' then string = string 'readonly'
  if tabindex \= '' then string = string 'tabindex="'tabindex'"'
  return string'>'

/*****************************************************************************/
::CLASS select PUBLIC SUBCLASS htmlcore
::method body attribute

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose name size multiple disabled tabindex
  forward class (super) continue
  use arg name
  size = ''
  multiple = ''
  disabled = ''
  tabindex = ''
  self~body = .html~new

::method name
  expose type
  use arg type
  return self

::method multiple
  expose multiple
  use arg multiple
  return self

::method size
  expose size
  use arg size
  return self

::method disabled
  expose disabled
  use arg disabled
  return self

::method tabindex
  expose tabindex
  use arg tabindex
  return self

::method option
  if arg(1)~class = .option then
    option = arg(1)
  else
    option = .option~new(arg(1))
  self~body~queue(option)
  return option

::method makeHtml
  expose name size multiple disabled tabindex
  string = '<select'self~coreattrs 'name="'name'"'
  if size \= '' then string = string 'size="'size'"'
  if multiple \= '' then string = string 'multiple'
  if disabled \= '' then string = string 'disabled'
  if tabindex \= '' then string = string 'tabindex="'tabindex'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</select>')

/*===========================================================================*/
::CLASS option PUBLIC SUBCLASS htmlcore
/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose content value selected disabled
  forward class (super) continue
  use arg content
  value = ''
  selected = ''
  disabled = ''

::method value
  expose value
  use arg value
  return self

::method selected
  expose selected
  use arg selected
  return self

::method disabled
  expose disabled
  use arg disabled
  return self

::method option
  if arg(1)~class = .option then
    option = arg(1)
  else
    option = .option~new(arg(1))
  self~body~queue(option)
  return option

::method makeHtml
  expose content value selected disabled
  string = '<option'self~coreattrs
  if value \= '' then string = string 'value="'value'"'
  if selected \= '' then string = string 'selected'
  if disabled \= '' then string = string 'disabled'
  return string'>'content'</option>'

/*****************************************************************************/
::CLASS textarea PUBLIC SUBCLASS htmlcore

/*---------------------------------------------------------------------------*/
::method init                          /* initialize an html object          */
  expose content name size disabled readonly tabindex
  forward class (super) continue
  use arg name, size
  content = ''
  disabled = ''
  readonly = ''
  tabindex = ''

::method name
  expose type
  use arg type
  return self

::method size
  expose size
  use arg size
  return self

::method _
  expose content
  use arg content
  return self

::method readonly
  expose readonly
  use arg readonly
  return self

::method disabled
  expose disabled
  use arg disabled
  return self

::method tabindex
  expose tabindex
  use arg tabindex
  return self

::method makeHtml
  expose content name size disabled readonly tabindex
  string = '<textarea'self~coreattrs 'name="'name'" rows="'size~word(1)'" cols="'size~word(2)'"'
  if readonly \= '' then string = string 'readonly'
  if disabled \= '' then string = string 'disabled'
  if tabindex \= '' then string = string 'tabindex="'tabindex'"'
  return .htmlbase~new~~queue(string'>')~~queue(content)~~queue('</textarea>')

/*****************************************************************************/
::CLASS html PUBLIC subclass htmlbase

/*---------------------------------------------------------------------------*/
::METHOD address                       /* address                            */
  return self~~queue('<address>'arg(1)'</eaddress>')

/*===========================================================================*/
::METHOD a                             /* anchor           parms: name, html */
  self~queue('<a name="'arg(1)'">')
  self~queue(arg(2))
  return self~~queue('</a>')

/*---------------------------------------------------------------------------*/
::METHOD aref                          /* anchor reference                   */
  use arg ref, html, pic, addopts

  self~queue('<a href="'ref'">')
  if arg(4, 'o') then addopts = 'border=0'
  if arg() >= 3 then
    self~queue('<img align=middle' addopts 'src="'pic'">')
  if arg(2, e) then
    self~queue(html)
  return self~~queue('</a>')

/*---------------------------------------------------------------------------*/
::METHOD arefmouse                     /* anchor reference with mouse events */
  use arg ref, num, pic, picmouse, text, addopts

  if arg(6, 'o') then addopts = 'border=0'
  self~queue('<a href="'ref'"',
              'onMouseOver="document.images['num'].src='''picmouse''';"',
              'onMouseOut="document.images['num'].src='''pic''';">',
              '<img align=middle' addopts 'src="'pic'">')
  if arg(5, 'e') then
    self~queue('<b>'text'</b></a>')
  else
    self~queue('</a>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD t                             /* add raw text to the last enty      */
  if self~items = 0 then                          /****** TEST ********/
    self~queue(arg(1))
  else do
    item = self~at(self~items)
    say 'item:' item
    self~put:super(item arg(1), self~items)
    say 'item danach:' self~at(self~items)
  end
  return self

/*---------------------------------------------------------------------------*/
::METHOD _                             /* add html element to last construct */
  return self~~queue(arg(1))

/*---------------------------------------------------------------------------*/
::METHOD .                             /* add html element                   */
  return self~~queue(arg(1))

/*---------------------------------------------------------------------------*/
::METHOD div                           /* division               parms: opts */
  return self~~queue('<div arg(1)>')

/*---------------------------------------------------------------------------*/
::METHOD ediv                          /* end division                       */
  return self~~queue('</div>')

/*---------------------------------------------------------------------------*/
::METHOD span                          /* span      parms: opts, sptxt, text */
  self~queue('<span' arg(1)'>')
  if arg(2, 'e') then
    self~queue(arg(2)'</span>'arg(3))
  return self

/*---------------------------------------------------------------------------*/
::METHOD espan                         /* end span                           */
  return self~~queue('</span>')

/*===========================================================================*/
::METHOD strong                        /* strong                             */
  if arg(1, 'e') then
    self~queue('<strong>'arg(1)'</strong>')
  else
    self~queue('<strong>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD estrong                       /* end strong                         */
  return self~~queue('</strong>')

/*---------------------------------------------------------------------------*/
::METHOD big                           /* big                                */
  if arg(1, 'e') then
    self~queue('<big>'arg(1)'</big>')
  else
    self~queue('<big>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD ebig                           /* end big                           */
  return self~~queue('</big>')

/*---------------------------------------------------------------------------*/
::METHOD small                         /* small                              */
  if arg(1, 'e') then
    self~queue('<small>'arg(1)'</small>')
  else
    self~queue('<samll>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD esmall                        /* end small                          */
  return self~~queue('</small>')

/*---------------------------------------------------------------------------*/
::METHOD em                            /* emphasis                           */
  if arg(1, 'e') then
    self~queue('<em>'arg(1)'</em>')
  else
    self~queue('<em>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD eem                           /* end emphasis                       */
  return self~~queue('</em>')

/*---------------------------------------------------------------------------*/
::METHOD b                             /* bold                               */
  if arg(1, 'e') then
    self~queue('<b>'arg(1)'</b>')
  else
    self~queue('<b>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD eb                            /* end bold                           */
  return self~~queue('</b>')

/*---------------------------------------------------------------------------*/
::METHOD i                             /* italic                             */
  if arg(1, 'e') then
    self~queue('<i>'arg(1)'</i>')
  else
    self~queue('<i>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD ei                            /* end italic                         */
  return self~~queue('</i>')

/*---------------------------------------------------------------------------*/
::METHOD u                             /* underscore                         */
  return self~~queue('<u>'arg(1)'</u>')

/*---------------------------------------------------------------------------*/
::METHOD sub                           /* subscript                          */
  return self~~queue('<sub>'arg(1)'</sub>')

/*---------------------------------------------------------------------------*/
::METHOD sup                           /* superscript                        */
  return self~~queue('<sup>'arg(1)'</sup>')

/*---------------------------------------------------------------------------*/
::METHOD tt                            /* teletype                           */
  return self~~queue('<tt>'arg(1)'</tt>')

/*---------------------------------------------------------------------------*/
::METHOD blink                         /* blink                              */
  return self~~queue('<blink>'arg(1)'</blink>')

/*---------------------------------------------------------------------------*/
::METHOD code                          /* code                    parm: text */
  if arg(1, 'e') then
    self~queue('<code>'arg(1)'</code>')
  else
    self~queue('<code>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD ecode                         /* end code                           */
  return self~~queue('</code>')

/*---------------------------------------------------------------------------*/
::METHOD strike                        /* strikeout text                     */
  if arg(1, 'e') then
    self~queue('<strike>'arg(1)'</strike>')
  else
    self~queue('<strike>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD s                             /* strikeout text                     */
  forward message 'strike'

/*---------------------------------------------------------------------------*/
::METHOD estrike                       /* end strikeout text                 */
  return self~~queue('</strike>')

/*---------------------------------------------------------------------------*/
::METHOD es                            /* end strikeout text                 */
  forward message 'estrike'

/*---------------------------------------------------------------------------*/
::METHOD q                             /* quote                              */
  if arg(1, 'e') then
    self~queue('<q>'text'</q>')
  else
    self~queue('<q>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD eq                            /* end quote                          */
  return self~~queue('</q>')

/*---------------------------------------------------------------------------*/
::METHOD font                          /* font         parm: opts, [text]    */
  use arg opts

  if arg(2, 'e') then
    self~queue('<font' opts'>'arg(2)'</font>')
  else
    self~queue('<font' opts'>')
  return self

/*---------------------------------------------------------------------------*/
::METHOD efont                         /* end font                           */
  return self~~queue('</font>')

/*---------------------------------------------------------------------------*/
::METHOD plaintext                     /* plaintext                          */
  return self~~queue('<plaintext>'arg(1))

/*---------------------------------------------------------------------------*/
::METHOD space                         /* required blank space (no break!)   */
  use arg n

  if arg(1, e) then
    self~queue(copies('&nbsp;', n))    /* count of n blank spaces            */
  else
    self~queue('&nbsp;')               /* single blank space (default)       */
  return self

/*===========================================================================*/
::METHOD h1                            /* header 1 tag                       */
  return self~~queue(.h1~new(arg(1)))

/*---------------------------------------------------------------------------*/
::METHOD h2                            /* header 2 tag                       */
  return self~~queue(.h2~new(arg(1)))

/*---------------------------------------------------------------------------*/
::METHOD h3                            /* header 3 tag                       */
  return self~~queue(.h3~new(arg(1)))

/*---------------------------------------------------------------------------*/
::METHOD h4                            /* header 4 tag                       */
  return self~~queue(.h4~new(arg(1)))

/*---------------------------------------------------------------------------*/
::METHOD h5                            /* header 5 tag                       */
  return self~~queue(.h5~new(arg(1)))

/*---------------------------------------------------------------------------*/
::METHOD h6                            /* header 6 tag                       */
  return self~~queue(.h6~new(arg(1)))

/*===========================================================================*/
::METHOD hr                            /* horizontal rule      parms: size   */
  hr = .hr~new(arg(1))
  self~queue(hr)
  return hr

/*===========================================================================*/
::METHOD br                            /* line break                         */
  if arg(1, 'e') then
    br = .br~new(arg(1))
  else br = .br~new
  self~queue(br)
  return br

/*===========================================================================*/
::METHOD p                             /* paragraph            parms: text   */
  if arg(1, 'e') then
    p = .p~new(arg(1))
  else p = .p~new
  self~queue(p)
  return p

/*===========================================================================*/
::METHOD pre                           /* preformatted text    parms: text   */
  pre = .pre~new(arg(1))
  self~~queue(pre)
  return pre

/*===========================================================================*/
::METHOD img                           /* image         parms: imgfile       */
  if arg(1)~class \= .img then
    img = .img~new(arg(1))
  self~~queue(img)
  return img

/*===========================================================================*/
::METHOD ul                            /* unordered list                     */
  if arg(1)~class \= .ul then
    ul = .ul~new
  self~~queue(ul)
  return ul

/*===========================================================================*/
::METHOD ol                            /* ordered list                       */
  if arg(1)~class \= .ol then
    ol = .ol~new
  self~~queue(ol)
  return ol

/*===========================================================================*/
::METHOD dl                            /* definition list                    */
  if arg(1)~class \= .dl then
    dl = .dl~new
  self~~queue(dl)
  return dl

/*===========================================================================*/
::METHOD table                         /* table         parms: width         */
  if arg(1)~class = .table then
    table = arg(1)
  else
    table = .table~new(arg(1))
  self~queue(table)
  return table

/*===========================================================================*/
::METHOD object                        /* object  parms: data, type          */
  if arg(1)~class = .object then do
    self~queue(arg(1))
    return self
  end
  obj = .object~new(arg(1), arg(2))
  self~queue(obj)
  return obj

/*===========================================================================*/
::METHOD iframe                        /* iframe        parms: content       */
  if arg(1)~class = .iframe then
    form = arg(1)
  else
    iframe = .iframe~new(arg(1))
  self~queue(iframe)
  return iframe

/*===========================================================================*/
::METHOD form                          /* form          parms: action        */
  if arg(1)~class = .form then
    form = arg(1)
  else
    form = .form~new(arg(1))
  self~queue(form)
  return form

/*===========================================================================*/
::METHOD input                         /* input         parms: name, value   */
  if arg(1)~class = .input then
    input = arg(1)
  else
    input = .input~new(arg(1), arg(2))
  self~queue(input)
  return input

/*===========================================================================*/
::METHOD select                        /* select        parms: name, value   */
  if arg(1)~class = .select then
    sel = arg(1)
  else
    sel = .select~new(arg(1), arg(2))
  self~queue(sel)
  return sel

/*===========================================================================*/
::METHOD textarea                      /* textarea      parms: name, size    */
  if arg(1)~class = .textarea then
    tarea = arg(1)
  else
    tarea = .textarea~new(arg(1), arg(2))
  self~queue(tarea)
  return tarea

/*---------------------------------------------------------------------------*/
::METHOD play                          /* play soundfile  (Windows only)     */
  use arg soundfile

  if soundfile~pos('.') = 0 then
    soundfile = soundfile".WAV"
  return self~~queue('<bgsound src='soundfile'>')

/*****************************************************************************/
::CLASS head PUBLIC                   /* Html head class                     */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose title
  title = ''
  self~body = .html~new

::method title
  expose title
  use arg title
  return self

::method meta
  if arg(1)~class = .meta then
    meta = arg(1)~head= self
  else
    meta = .meta~new(arg(1), self)
  self~body~queue(meta)
  return meta

::method link
  if arg(1)~class = .link then
    link = arg(1)~head= self
  else
    link = .link~new(arg(1), self)
  self~body~queue(link)
  return link

::method frameset
  if arg(1)~class = .frameset then
    frameset = arg(1)
  else
    frameset = .frameset~new(arg(1), arg(2))
  self~body~queue(frameset)
  return frameset

::method style
  if arg(1)~class = .style then
    style = arg(1)~head= self
  else
    style = .style~new(, self)
  self~body~queue(style)
  return style

::method makeHtml
  expose title
  html = .htmlbase~new~~queue('<head>')
  if title \= '' then html~queue('<title>'title'</title>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</head>')

/*===========================================================================*/
::CLASS link SUBCLASS htmlcore         /* Link class                         */
::method ltype attribute
::method head attribute

/*---------------------------------------------------------------------------*/
::method init
  expose href title
  title = ''
  href = arg(1)
  self~ltype= ''
  self~head= arg(2)

::method link
  if arg(1)~class = .link then
    link = arg(1)~head= self~head
  else
    link = .link~new(arg(1), self~head)
  self~head~body~queue(link)
  return link

::method rev
  self~ltype= 'rev="'arg(1)'"'
  return self

::method rel
  self~ltype= 'rel="'arg(1)'"'
  return self

::method title
  expose title
  title = arg(1)
  return self

::method makeHtml
  expose href title
  string = '<link'self~coreattrs
  if title \= '' then string = string 'title="'title'"'
  if href \= '' then string = string 'href="'href'"'
  if self~ltype \= '' then string = string self~ltype
  return string'>'

/*===========================================================================*/
::CLASS meta                           /* Meta class                         */
::method head attribute

/*---------------------------------------------------------------------------*/
::method init
  expose name content lang httpequiv
  name = arg(1)
  self~head= arg(2)
  content = ''
  lang = ''
  httpequiv = ''

::method name
  expose name
  use arg name
  return self

::method _
  expose content
  use arg content
  return self

::method httpequiv
  expose httpequiv
  use arg httpequiv
  return self

::method lang
  expose lang
  use arg lang
  return self

::method meta
  if arg(1)~class = .meta then
    meta = arg(1)~head= self~head
  else
    meta = .meta~new(arg(1), self~head)
  self~head~body~queue(meta)
  return meta

::method makeHtml
  expose name content lang httpequiv
  string = '<meta'
  if name \= '' then string = string 'name="'name'"'
  if httpequiv \= '' then string = string 'http-equiv="'httpequiv'"'
  if content \= '' then string = string 'content="'content'"'
  if lang \= '' then string = string 'lang="'lang'"'
  return string'>'

/*===========================================================================*/
::CLASS style                           /* CCS Style class                   */
::method head attribute

/*---------------------------------------------------------------------------*/
::method init
  expose type media body
  body = .htmlbase~new
  media = ''
  if arg(1, 'e') then
    type = arg(1)
  else
    type = 'text/css'
  self~head= arg(2)

::method media
  expose media
  use arg media
  return self

::method elem
  expose elem body
  use arg selem
  body~queue(selem)
  return self

::method style
  expose body
  if arg(1)~class = .style then
    style = arg(1)~head= self~head
  else
    style = .style~new(arg(1), self~head)
  body~queue(style)
  return style

::method makeHtml
  expose type media body
  string = '<style type="'type'"'
  if media \= '' then string = string 'media="'media'"'
  html = .htmlbase~new~~queue(string'>')
  do selem over body
    html~queue(selem)
  end
  return html~~queue('</style>')

/*****************************************************************************/
::CLASS table SUBCLASS htmlcore PUBLIC /* Table class                        */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose align bgcolor width cols border frame rules cellspacing cellpadding
  forward class (super) continue
  align = ''
  bgcolor = ''
  width = ''
  cols = ''
  border = ''
  frame = ''
  rules = ''
  cellspacing = ''
  cellpadding = ''
  self~body = .html~new

::method align
  expose align
  use arg align
  return self

::method width
  expose width
  width = arg(1)
  return self

::method cols
  expose cols
  cols = arg(1)
  return self

::method border
  expose border
  border = arg(1)
  return self

::method frame
  expose frame
  frame = arg(1)
  return self

::method rules
  expose rules
  rules = arg(1)
  return self

::method cellspacing
  expose cellspacing
  cellspacing = arg(1)
  return self

::method cellpadding
  expose cellpadding
  cellpadding = arg(1)
  return self

::method bgcolor
  expose bgcolor
  bgcolor = arg(1)
  return self

::method colgroup
  use arg colgroup
  if colgroup~class \= .colgroup then
    colgroup = .colgroup~new
  self~body~queue(colgroup)
  return colgroup

::method caption
  use arg caption
  if caption~class \= .caption then
    caption = .caption~new(caption)
  self~body~queue(caption)
  return caption

::method tr
  if arg(1, 'e') then do
    tr = arg(1)
    if tr~class \= .tr then
      tr = .tr~new(tr)
  end
  else tr = .tr~new
  self~body~queue(tr)
  return tr

::method makeHtml
  expose align bgcolor width cols border frame rules cellspacing cellpadding
  string = '<table'
  if align \= '' then string = string 'align="'align'"'
  if width \= '' then string = string 'width="'width'"'
  if bgcolor \= '' then string = string 'bgcolor="'bgcolor'"'
  if cols \= '' then string = string 'cols="'cols'"'
  if border \= '' then string = string 'border="'border'"'
  if frame \= '' then string = string 'frame="'frame'"'
  if rules \= '' then string = string 'rules="'rules'"'
  if cellspacing \= '' then string = string 'cellspacing="'cellspacing'"'
  if cellpadding \= '' then string = string 'cellpadding="'cellpadding'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)                                 --put(elem~request('html'))
  end
  return html~~queue('</table>')

/*===========================================================================*/
::CLASS colgroup SUBCLASS htmlcore     /* Table column group class           */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose span width
  forward class (super) continue
  span = ''
  width = ''
  self~body = .html~new

::method span
  expose span
  use arg span
  return self

::method width
  expose width
  use arg width
  return self

::method col
  col = .col~new(self)
  self~body~queue(col)
  return col

::method makeHtml
  expose span width
  string = '<colgroup'
  if span \= '' then string = string 'span="'span'"'
  if width \= '' then string = string 'width="'width'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</colgroup>')

/*===========================================================================*/
::CLASS col SUBCLASS htmlcore          /* Table column class                 */
::method colgroup attribute

/*---------------------------------------------------------------------------*/
::method init
  expose span width align valign char charoff
  forward class (super) continue
  span = ''
  width = ''
  align = ''
  valign = ''
  char = ''
  charoff = ''
  self~colgroup= arg(1)

::method span
  expose span
  use arg span
  return self

::method width
  expose width
  use arg width
  return self

::method align
  expose align
  use arg align
  return self

::method valign
  expose valign
  use arg valign
  return self

::method char
  expose char
  use arg char
  return self

::method charoff
  expose charoff
  use arg charoff
  return self

::method col
  colgroup = self~colgroup
  col = .col~new(colgroup)
  colgroup~body~queue(col)
  return col

::method makeHtml
  expose span width align char charoff valign
  string = '<col'
  if span \= '' then string = string 'span="'span'"'
  if width \= '' then string = string 'width="'width'"'
  if align \= '' then string = string 'align="'align'"'
  if valign \= '' then string = string 'valign="'valign'"'
  if char \= '' then string = string 'char="'char'"'
  if charoff \= '' then string = string 'charoff="'charoff'"'
  return string'>'

/*===========================================================================*/
::CLASS caption SUBCLASS htmlcore      /* Table caption class                */

::method init
  expose align text
  forward class (super) continue
  use arg text
  align = ''

::method align
  expose align
  use arg align
  return self

::method makeHtml
  expose align text
  string = '<caption'
  if align \= '' then string = string 'align="'align'"'
  return string'>'text'</caption>'

/*===========================================================================*/
::CLASS tr SUBCLASS htmlcore           /* Table row class                    */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose bgcolor
  forward class (super) continue
  bgcolor = ''
  self~body = .html~new

::method bgcolor
  expose bgcolor
  bgcolor = arg(1)
  return self

::method th
  if arg(1, 'e') then do
    th = arg(1)
    if th~class \= .th then
      th = .th~new(th, self)
    else th~tr= self
  end
  else th = .th~new(, self)
  self~body~queue(th)
  return th

::method td
  if arg(1, 'e') then do
    td = arg(1)
    if td~class \= .td then
      td = .td~new(td, self)
    else td~tr= self
  end
  else td = .td~new(, self)
  self~body~queue(td)
  return td

::method makeHtml
  expose bgcolor
  string = '<tr'
  if bgcolor \= '' then string = string 'bgcolor="'bgcolor'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</tr>')

/*===========================================================================*/
::CLASS td SUBCLASS htmlcore PUBLIC    /* Table data class                   */
::method tr attribute
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose axis axes bgcolor rowspan colspan align valign
  forward class (super) continue
  axis = ''
  axes = ''
  bgcolor = ''
  rowspan = ''
  colspan = ''
  align = ''
  valign = ''
  self~body= .html~new
  if arg(1, 'e') then self~body~queue(arg(1))
  self~tr= arg(2)

::method td
  if arg(1, 'e') then do
    td = arg(1)
    if td~class \= .td then
      td = .td~new(td, self~tr)
    else td~tr= self~tr
  end
  else td = .td~new(, self~tr)
  self~tr~body~queue(td)
  return td

::method th
  if arg(1, 'e') then do
    th = arg(1)
    if th~class \= .th then
      th = .th~new(th, self~tr)
    else th~tr= self~tr
  end
  else th = .th~new(, self~tr)
  self~tr~body~queue(th)
  return th

::method axis
  expose axis
  use arg axis
  return self

::method axes
  expose axes
  use arg axes
  return self

::method bgcolor
  expose bgcolor
  use arg bgcolor
  return self

::method rowspan
  expose rowspan
  use arg rowspan
  return self

::method colspan
  expose colspan
  use arg colspan
  return self

::method align
  expose align
  use arg align
  return self

::method valign
  expose valign
  use arg valign
  return self

::method makeHtml
  expose axis axes bgcolor rowspan colspan align valign
  string = '<td'
  if axis \= '' then string = string 'axis="'axis'"'
  if axes \= '' then string = string 'axes="'axes'"'
  if bgcolor \= '' then string = string 'bgcolor="'bgcolor'"'
  if rowspan \= '' then string = string 'rowspan="'rowspan'"'
  if colspan \= '' then string = string 'colspan="'colspan'"'
  if align \= '' then string = string 'align="'align'"'
  if valign \= '' then string = string 'valign="'valign'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</td>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*===========================================================================*/
::CLASS th SUBCLASS htmlcore PUBLIC    /* Table header class                 */
::method tr attribute
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose axis axes bgcolor rowspan colspan align valign
  forward class (super) continue
  axis = ''
  axes = ''
  bgcolor = ''
  rowspan = ''
  colspan = ''
  align = ''
  valign = ''
  self~body = .html~new
  if arg(1, 'e') then self~body~queue(arg(1))
  self~tr= arg(2)

::method td
  if arg(1, 'e') then do
    td = arg(1)
    if td~class \= .td then
      td = .td~new(td, self~tr)
    else td~tr= self~tr
  end
  else td = .td~new(, self~tr)
  self~tr~body~queue(td)
  return td

::method th
  if arg(1, 'e') then do
    th = arg(1)
    if th~class \= .th then
      th = .th~new(th, self~tr)
    else th~tr= self~tr
  end
  else th = .th~new(, self~tr)
  self~tr~body~queue(th)
  return th

::method axis
  expose axis
  use arg axis
  return self

::method axes
  expose axes
  use arg axes
  return self

::method bgcolor
  expose bgcolor
  use arg bgcolor
  return self

::method rowspan
  expose rowspan
  use arg rowspan
  return self

::method colspan
  expose colspan
  use arg colspan
  return self

::method align
  expose align
  use arg align
  return self

::method valign
  expose valign
  use arg valign
  return self

::method makeHtml
  expose axis axes bgcolor rowspan colspan align valign
  string = '<th'
  if axis \= '' then string = string 'axis="'axis'"'
  if axes \= '' then string = string 'axes="'axes'"'
  if bgcolor \= '' then string = string 'bgcolor="'bgcolor'"'
  if rowspan \= '' then string = string 'rowspan="'rowspan'"'
  if colspan \= '' then string = string 'colspan="'colspan'"'
  if align \= '' then string = string 'align="'align'"'
  if valign \= '' then string = string 'valign="'valign'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</th>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS param SUBCLASS htmlbase PUBLIC /* Param class                        */

::method init
  expose name value valuetype
  name = arg(1)
  value = arg(2)
  valuetype = ''

::method name
  expose name
  use arg name

::method value
  expose value
  value = arg(1)

::method valuetype
  expose valuetype
  if .bag~of("DATA", "REF", "OBJECT")~hasindex(translate(arg(1))) then
    valuetype = arg(1)                 /* ok, use value                      */
  else
    raise syntax 40                    /* html syntax error, be more specific!     */

::method makeHtml
  expose name value valuetype
  string = '<param name="'name'" value="'value'"'
  if valuetype \= '' then string = string 'valuetype="'valuetype'"'
  return string'>'

/*****************************************************************************/
::CLASS object SUBCLASS htmlcore PUBLIC/* Html Object class                  */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose declare classid data type size name standby
  forward class (super) continue
  declare = .false
  classid = ''
  standby = ''
  name = ''
  data = arg(1)
  type = arg(2)
  size = ''
  self~body = .html~new

::method classid
  expose classid
  use arg classid
  return self

::method name
  expose name
  use arg name
  return self

::method declare
  expose declare
  declare = .true
  return self

::method size
  expose size
  size = 'width="'arg(1)~word(1)'" height="'arg(1)~word(2)'"'
  return self

::method standby
  expose standby
  use arg standby
  return self

::method param
  if arg(1)~class = .param then
    param = arg(1)
  else
    param = .param~new(arg(1), arg(2))
  self~body~queue(param)
  return self

::method makeHtml unguarded
  expose data type size name declare classid standby
  string = '<object'
  if classid \= '' then string = string 'classid="'classid'"'
  if data \= '' then string = string 'data="'data'"'
  if type \= '' then string = string 'type="'type'"'
  if size \= '' then string = string size
  if name \= '' then string = string 'name="'name'"'
  if declare = .true then string = string 'declare'
  if standby \= '' then string = string 'standby="'standby'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</object>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS Frameset PUBLIC                   /* Frameset class                  */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose rows cols
  rows = arg(1)
  cols = arg(2)
  self~body= .htmlbase~new

::method frame
  frame = .frame~new(arg(1))
  self~body~queue(frame)
  return frame

::method frameset
  if arg(1)~class \= .frameset then
    frameset = .frameset~new(arg(1), arg(2))
  else
    frameset = arg(1)
  self~body~queue(frameset)
  return frameset

::method makeHtml unguarded
  expose rows cols
  string = '<frameset'
  if rows \= '' then string = string 'rows="'rows'"'
  if cols \= '' then string = string 'cols="'cols'"'
  self~body~push(string'>')
  return self~body~~queue('</frameset>')

/*===========================================================================*/
::CLASS Frame PUBLIC                   /* Frame class                        */

::method init
  expose src name scrolling marginwidth marginheight frameborder
  name = ''
  scrolling = ''
  frameborder = ''
  marginwidth = ''
  marginheight = ''
  src = arg(1)

::method name
  expose name
  name = arg(1)
  return self

::method marginwidth
  expose marginwidth
  marginwidth = arg(1)
  return self

::method marginheight
  expose marginheight
  marginheight = arg(1)
  return self

::method scrolling
  expose scrolling
  scrolling = arg(1)
  return self

::method frameborder
  expose frameborder
  frameborder = arg(1)
  return self

::method makeHtml
  expose src name scrolling marginwidth marginheight frameborder
  string = '<frame'
  if name \= '' then string = string 'name="'name'"'
  if src \= '' then string = string 'src="'src'"'
  if marginheight \= '' then string = string 'marginheight="'marginheight'"'
  if marginwidth \= '' then string = string 'marginwidth="'marginwidth'"'
  if scrolling \= '' then string = string 'scrolling="'scrolling'"'
  if frameborder \= '' then string = string 'frameborder="'frameborder'"'
  return string'>'

/*===========================================================================*/
::CLASS iFrame PUBLIC                   /* Inline Frame class                */
::method body attribute

/*---------------------------------------------------------------------------*/
::method init
  expose src name scrolling marginwidth marginheight frameborder width height
  name = ''
  scrolling = ''
  frameborder = ''
  marginwidth = ''
  marginheight = ''
  self~body= .html~new
  width = ''
  height = ''
  src = arg(1)

::method name
  expose name
  use arg name
  return self

::method marginwidth
  expose marginwidth
  use arg marginwidth
  return self

::method marginheight
  expose marginheight
  use arg marginheight
  return self

::method scrolling
  expose scrolling
  use arg scrolling
  return self

::method frameborder
  expose frameborder
  use arg frameborder
  return self

::method width
  expose width
  use arg width
  return self

::method height
  expose height
  use arg heigh
  return self

::method makeHtml
  expose src name scrolling marginwidth marginheight frameborder content width height
  string = '<iframe'
  if name \= '' then string = string 'name="'name'"'
  if src \= '' then string = string 'src="'src'"'
  if width \= '' then string = string 'width="'width'"'
  if height \= '' then string = string 'height="'height'"'
  if marginheight \= '' then string = string 'marginheight="'marginheight'"'
  if marginwidth \= '' then string = string 'marginwidth="'marginwidth'"'
  if scrolling \= '' then string = string 'scrolling="'scrolling'"'
  if frameborder \= '' then string = string 'frameborder="'frameborder'"'
  html = .htmlbase~new~~queue(string'>')
  do elem over self~body
    html~queue(elem)
  end
  return html~~queue('</iframe>')

::method unknown
  forward message (arg(1)) arguments(arg(2)) to (self~body)

/*****************************************************************************/
::CLASS hr SUBCLASS htmlcore PUBLIC    /* Horizontal rule class              */
                                       /* parms:  size                       */
/*---------------------------------------------------------------------------*/
::method init
  expose align size width noshade
  size = arg(1)
  align = ''
  width = ''
  noshade = ''

::method align
  expose align
  use arg align
  return self

::method size
  expose size
  use arg size
  return self

::method width
  expose width
  use arg width
  return self

::method noshade
  expose noshade
  use arg noshade
  return self

::method makeHtml
  expose align size width noshade
  string = '<hr'
  if align \= '' then string = string 'align="'align'"'
  if size \= '' then string = string 'size="'size'"'
  if width \= '' then string = string 'width="'width'"'
  if noshade \= '' then string = string 'noshade'
  return string'>'

/*****************************************************************************/
::CLASS parmDir SUBCLASS directory PUBLIC /* parameter directory             */

/*---------------------------------------------------------------------------*/
::METHOD init                          /* initialize the parameter directory */
  use arg parmstring                   /* parameter string from evironment   */

  parse var parmstring parm '&' rest   /* initialize parmdir from parmstring */
  do while parm \= ''                  /* process the whole parmlist         */
    parse var parm name '=' value      /* get each parameter from list ...   */
    value = normalize(value)           /* filter out escape sequences        */
    if self~hasindex(name) then do     /* is there an entry already?         */
      item = self[name]                /* get it                             */
      if item~class=.list then         /* is it a List?                      */
        item~insert(value)             /* add the value to the list          */
      else do
        item = .list~of(item, value)   /* make an item list                  */
      end
      self[name]= item                 /* update the parmDir entry           */
    end
    else
      self[name]= value                /* and  put it in directory           */
    parse var rest parm '&' rest       /* get next parameter                 */
  end
                                       /* unknown parameters are handled     */
  self~setmethod('UNKNOWN', .methods['EMPTY'])

/*---------------------------------------------------------------------------*/
::METHOD list                          /* Parameter list          parm: name */

  item = self[arg(1)]                  /* get pameter by name              */
  if item~class=.list then do          /* is it a list?                      */
    list = ''
    do i over item                     /* linearize the list object          */
      if list = '' then
        list = i
      else
        list = list',' i               /* separate items by comma            */
    end
    return list                        /* return list of items               */
  end
  else
    return item                        /* just return the item               */

/*****************************************************************************/
/*                                                                           */
/* Auxiliary Routines                                                        */
/*                                                                           */
/*****************************************************************************/
::ROUTINE isHtmlCollection PUBLIC      /* object instance of kind html class?*/
  use arg obj
  return isClassSubclassOf(obj~class, .htmlbase)

/*---------------------------------------------------------------------------*/
::ROUTINE isHtmlObject PUBLIC          /* object instance of kind html class?*/
  use arg obj
  return isClassSubclassOf(obj~class, .htmlcore)

/*---------------------------------------------------------------------------*/
::ROUTINE isClassSubclassOf            /* is myclass descended from class?   */
  use arg myclass, class
  if myclass = class then return .true
  subclasses = class~subclasses
  do subclass over subclasses
    if isClassSubclassOf(myclass, subclass) then return .true
  end
  return .false

/*---------------------------------------------------------------------------*/
::ROUTINE normalize                    /* translate escape seq's  parm: text */
  use arg text                         /* e.g. '%3F' --> '?'                 */
  pi = pos('%', text)
  do while(pi > 0)
    seq = substr(text, pi, 3)
    text = changestr(seq, text, x2c(substr(seq, 2)))
    pi = pos('%', text, pi)
  end
  return text                          /* return the normalized text string  */

/*===========================================================================*/
/*                                                                           */
/* Inline Routines (used within text line)                                   */
/*                                                                           */
/*===========================================================================*/
::ROUTINE blink PUBLIC                 /* blink                              */
  return  '<blink>'arg(1)'</blink>'

/*---------------------------------------------------------------------------*/
::ROUTINE sub PUBLIC                   /* subscript                          */
  return  '<sub>'arg(1)'</sub>'

/*---------------------------------------------------------------------------*/
::ROUTINE sup PUBLIC                   /* superscript                        */
  return  '<sup>'arg(1)'</sup>'

/*---------------------------------------------------------------------------*/
::ROUTINE u PUBLIC                     /* underscore                         */
  return  '<u>'arg(1)'</u>'

/*---------------------------------------------------------------------------*/
::ROUTINE i PUBLIC                     /* italic                             */
  return  '<i>'arg(1)'</i>'

/*---------------------------------------------------------------------------*/
::ROUTINE b PUBLIC                     /* bold                               */
  return  '<b>'arg(1)'</b>'

/*---------------------------------------------------------------------------*/
::ROUTINE small PUBLIC                 /* small                              */
  return  '<small>'arg(1)'</small>'

/*---------------------------------------------------------------------------*/
::ROUTINE big PUBLIC                   /* big                                */
  return  '<big>'arg(1)'</big>'

/*---------------------------------------------------------------------------*/
::ROUTINE em PUBLIC                    /* emphasis                           */
  return  '<em>'arg(1)'</em>'

/*---------------------------------------------------------------------------*/
::ROUTINE strong PUBLIC                /* strong                             */
  return  '<strong>'arg(1)'</strong>'

/*---------------------------------------------------------------------------*/
::ROUTINE font PUBLIC                  /* font            parm: option, text */
  return  '<font' arg(1)'>'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE size PUBLIC                  /* size              parm: size, text */
  return  '<font size="'arg(1)'">'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE color PUBLIC                 /* color            parm: color, text */
  return  '<font color='arg(1)'>'arg(2)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE red PUBLIC                   /* color red               parm: text */
  return  '<font color=#FF0000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE green PUBLIC                 /* color green             parm: text */
  return  '<font color=#008000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE blue PUBLIC                  /* color blue              parm: text */
  return  '<font color=#0000FF>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE yellow PUBLIC                /* color yellow            parm: text */
  return  '<font color=#FFFF00>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE brown PUBLIC                 /* color brown             parm: text */
  return  '<font color=#A52A2A>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE white PUBLIC                 /* color white             parm: text */
  return  '<font color=#FFFFFF>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE grey PUBLIC                  /* color grey              parm: text */
  return  '<font color=#808080>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE black PUBLIC                 /* color black             parm: text */
  return '<font color=#000000>'arg(1)'</font>'

/*---------------------------------------------------------------------------*/
::ROUTINE breaks PUBLIC                /* insert breaks           parm: text */
  return changestr('0A'x, changestr('0D'x, arg(1), ''), '<br>')

/*---------------------------------------------------------------------------*/
::ROUTINE img PUBLIC                   /* inline image           parm: image */
  return '<img src='arg(1)'>'

/*===========================================================================*/
/* End of Html framework definition                                          */
/*---------------------------------------------------------------------------*/

