/*-- C -----------------------------------------------------------------------*/
/*                                                                            */
/* Module:      rxicapi.c                                                     */
/*                                                                            */
/* Version:     1.02.01                                                       */
/*                                                                            */
/* Author:      W. David Ashley                                               */
/*                                                                            */
/* Description: Domino Go WWW Server ICAPI interface to Object Rexx.          */
/*                                                                            */
/* (C) Copyright IBM Corp. 1998. All rights reserved.                         */
/*                                                                            */
/* US Government Users Restricted Rights Use, duplication or disclosure       */
/* restricted by GSA ADP Schedule Contract with IBM Corp.                     */
/*                                                                            */
/* The program is provided "as is" without any warranty express or implied,   */
/* including the warranty of non-infringement and the implied warranties of   */
/* merchantibility and fitness for a particular purpose. IBM will not be      */
/* liable for any damages suffered by you as a result of using the Program.   */
/* In no event will IBM be liable for any special, indirect or consequential  */
/* damages or lost profits even if IBM has been advised of the possibility of */
/* their occurrence. IBM will not be liable for any third party claims        */
/* against you.                                                               */
/*                                                                            */
/* This code provide a direct interface from the Domino Go WWW Server to      */
/* Object Rexx. This interface allows the execution of an Object Rexx program */
/* as a CGI service from the WWW server. It thus allows the programmer the    */
/* ability to write CGI/ICAPI programs in Object Rexx.                        */
/*                                                                            */
/* Modifications:                                                             */
/*                                                                            */
/* Date     Author      Description                                           */
/* -------- ----------- ----------------------------------------------------- */
/* 12/01/98 WD Ashley   Initial Release                                       */
/* 12/01/98 WD Ashley   v1.01.01 Fixed problem where WWW server fails after   */
/*                      the Rexx procedure reports an error. Also added error */
/*                      messages from a failed Rexx procedure to the          */
/*                      httpd-error log file and also report them to the page */
/*                      returned to the browser.                              */
/* 12/08/98 WD Ashley   v1.02.01 Added the balance of the HTTPD functions.    */
/*                                                                            */
/*----------------------------------------------------------------------------*/
 
 
#define INCL_RXFUNC
#define INCL_RXSHV
#define INCL_RXSYSEXIT
#include <rexx.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <HTAPI.h>
 
 
/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    SetRexxVar                                                    */
/*                                                                            */
/* Description: Set a Rexx variable.                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

INT SetRexxVar (
   PSZ        name,                     /* REXX variable to set               */
   PSZ        value)                    /* value to assign                    */
   {
   SHVBLOCK   block;                    /* variable pool control block        */
 
   block.shvcode = RXSHV_SYSET;         /* do a symbolic set operation        */
   block.shvret = (UCHAR) 0;            /* clear return code field            */
   block.shvnext = (PSHVBLOCK) 0;       /* no next block                      */
                                        /* set variable name string           */
   MAKERXSTRING (block.shvname, name, strlen (name));
                                        /* set value string                   */
   MAKERXSTRING (block.shvvalue, value, strlen (value));
   block.shvvaluelen = strlen (value);  /* set value length                   */
   return (RexxVariablePool (&block));  /* set the variable                   */
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    Rexx_IO_Exit                                                  */
/*                                                                            */
/* Description: I/O exit for the Rexx procedure.                              */
/*                                                                            */
/*----------------------------------------------------------------------------*/

LONG APIENTRY Rexx_IO_Exit (
   LONG  ExitNumber,                    /* exit function code                 */
   LONG  Subfunction,                   /* exit subfunction code              */
   PEXIT ParmBlock)                     /* function control block             */
   {
   PUCHAR          handle;              /* HTTPD handle                       */
   UCHAR           buffer[8];           /* o; returned query data             */ 
   USHORT          query_flag;          /* o; query flag                      */
   unsigned char * value;               /* i; write data buffer               */
   unsigned long   value_length;        /* i/o; size/length of buffer         */
   long            return_code;         /* o; return code                     */
   RXSIOSAY_PARM * sparm;               /* Rexx i/o parameter                 */
   RXSIOTRC_PARM * tparm;               /* Rexx i/o parameter                 */
   ULONG           name_length;         /* name length                        */
   static char *   name = {"Content-type"}; 
   static char *   lf = {"\n"};
   static char *   pre = {"<pre>"};
   static char *   epre = {"</pre>"};

    
   /* get the saved work area and the HTTPD handle */
   RexxQueryExit ("WWW_EXIT", NULL, &query_flag, &buffer[0]);
   handle = &buffer[0];
    
   /* in case the content-type is not set, set it */
   name_length = strlen (name);
   value = "text/html";
   value_length = strlen (value);
   HTTPD_set (handle, name, &name_length, value, &value_length, &return_code);
    
   /* perform the exit function */
   switch (Subfunction)
      {
      case RXSIOSAY:
         sparm = (RXSIOSAY_PARM *) ParmBlock;
         value = sparm -> rxsio_string.strptr;
         value_length = sparm -> rxsio_string.strlength;
         HTTPD_write (handle, value, &value_length, &return_code); 
         value = lf;
         value_length = strlen (lf);
         HTTPD_write (handle, value, &value_length, &return_code); 
         return (RXEXIT_HANDLED);
      case RXSIOTRC:
         value = pre;
         value_length = strlen (pre);
         HTTPD_write (handle, value, &value_length, &return_code); 
         tparm = (RXSIOTRC_PARM *) ParmBlock;
         value = tparm -> rxsio_string.strptr;
         value_length = tparm -> rxsio_string.strlength;
         HTTPD_log_error (handle, value, &value_length, &return_code);
         HTTPD_write (handle, value, &value_length, &return_code); 
         value = epre;
         value_length = strlen (epre);
         HTTPD_write (handle, value, &value_length, &return_code); 
         return (RXEXIT_HANDLED);
      case RXSIOTRD:
      case RXSIODTR:
      default:
         break;
      }
   return (RXEXIT_RAISE_ERROR);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_attributes                                           */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to return the */
/*              attributes of a file.                                         */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              file path/name                                                */
/*              Rexx variable name for the attributes                         */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_attributes (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name of the file                */
   unsigned long   name_length;         /* i; length of the name              */
   unsigned char * value;               /* o; buffer containing attributes    */
   unsigned long   value_length;        /* i/o;size/length of buffer attributes */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 3 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]) ||
    !RXVALIDSTRING (Argv[2]))
      return (40);

   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   name_length = Argv[1].strlength;
   value = malloc (256);
   value_length = 256;

   /* perform function and set any Rexx variables */ 
   HTTPD_attributes (handle, name, &name_length, value,
                     &value_length, &return_code);
   if (return_code == HTTPD_BUFFER_TOO_SMALL)
      {
      free (value);
      value = malloc (value_length);
      HTTPD_attributes (handle, name, &name_length, value,
                        &value_length, &return_code);
      }
   if (return_code == HTTPD_SUCCESS)
      SetRexxVar (Argv[2].strptr, value);
   free (value);
    
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_authenticate                                         */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to authenti-  */
/*              cate a userid/password.                                       */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_authenticate (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 1 || !RXVALIDSTRING (Argv[0]))
      return (40);

   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));

   /* perform function and set any Rexx variables */ 
   HTTPD_authenticate (handle, &return_code);
    
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_extract                                              */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to extract    */
/*              environment variables.                                        */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              variable name                                                 */
/*              Rexx variable name for the value                              */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_extract (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name of value to extract        */
   unsigned long   name_length;         /* i; length of the name              */
   unsigned char * value;               /* o; buffer to put the value         */
   unsigned long   value_length;        /* i/o;size of buffer/length of value */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 3 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]) ||
    !RXVALIDSTRING (Argv[2]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   name_length = Argv[1].strlength;
   value = malloc (256);
   value_length = 256;

   /* perform function and set any Rexx variables */ 
   HTTPD_extract (handle, name, &name_length, value,
                  &value_length, &return_code); 
   if (return_code == HTTPD_BUFFER_TOO_SMALL)
      {
      free (value);
      value = malloc (value_length);
      HTTPD_extract (handle, name, &name_length, value,
                     &value_length, &return_code);
      }
   if (return_code == HTTPD_SUCCESS)
      SetRexxVar (Argv[2].strptr, value);
   free (value);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);           
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_reverse_translate                                    */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to translate  */
/*              a file system object to a URL.                                */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              file path/name                                                */
/*              Rexx variable name for the URL                                */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_reverse_translate (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name of file system object      */
   unsigned long   name_length;         /* i; length of the name              */
   unsigned char * value;               /* o; buffer which contains the URL   */
   unsigned long   value_length;        /* i/o;size/length of buffer URL      */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 3 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]) ||
    !RXVALIDSTRING (Argv[2]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   value = malloc (256);
   value_length = 256;

   /* perform function and set any Rexx variables */ 
   HTTPD_reverse_translate (handle, name, &name_length,
                            value, &value_length, &return_code);
   if (return_code == HTTPD_BUFFER_TOO_SMALL)
      {
      free (value);
      value = malloc (value_length);
      HTTPD_reverse_translate (handle, name, &name_length,
                               value, &value_length, &return_code);
      }
   if (return_code == HTTPD_SUCCESS)
      SetRexxVar (Argv[2].strptr, value);
   free (value);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
                     

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_set                                                  */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to set/create */
/*              an environment variable.                                      */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              variable name                                                 */
/*              new value for the variable                                    */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_set (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name object                     */
   unsigned long   name_length;         /* i; length of the name              */
   unsigned char * value;               /* i; buffer of the data to set e     */
   unsigned long   value_length;        /* i;o; size/length of buffer         */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 3 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]) ||
    !RXVALIDSTRING (Argv[2]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   name_length = Argv[1].strlength;
   value = Argv[2].strptr;
   value_length = Argv[2].strlength;

   /* perform function */ 
   HTTPD_set (handle, name, &name_length, value, &value_length, &return_code);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_file                                                 */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function which sends   */
/*              the contents of a file to fulfill the request.                */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              file path/name                                                */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_file (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name of file to send            */
   unsigned long   name_length;         /* i; size/length of buffer           */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   name_length = Argv[1].strlength;

   /* perform function */ 
   HTTPD_file (handle, name, &name_length, &return_code); 
   
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_exec                                                 */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function which         */
/*              executes a script to satisfy the request.                     */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              script name                                                   */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_exec (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; name of file to send            */
   unsigned long   name_length;         /* i; size/length of buffer           */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   name_length = Argv[1].strlength;

   /* perform function */ 
   HTTPD_exec (handle, name, &name_length, &return_code); 
   
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_read                                                 */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to read the   */
/*              request.                                                      */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              Rexx variable name for the request string                     */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_read (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char   value[256];          /* o; buffer for the data to read     */
   unsigned long   value_length;        /* i/o; size/length of buffer         */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   value_length = sizeof (value);

   /* perform function and set any Rexx variables */ 
   HTTPD_read (handle, value, &value_length, &return_code); 
   if (return_code == HTTPD_SUCCESS)
      SetRexxVar (Argv[1].strptr, value);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_restart                                              */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to restart    */
/*              the WWW server.                                               */
/*                                                                            */
/* Rexx Args:   none                                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_restart (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 0)
      return (40);

   /* perform function and set any Rexx variables */ 
   HTTPD_restart (&return_code); 
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_write                                                */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to write a    */
/*              string to fulfill the request.                                */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              string to be written                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_write (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * value;               /* i; buffer of the data to write     */
   unsigned long   value_length;        /* i; size/length of buffer           */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   value = Argv[1].strptr;
   value_length = Argv[1].strlength;

   /* perform function */ 
   HTTPD_write (handle, value, &value_length, &return_code); 
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_log_error                                            */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to write a    */
/*              string to the httpd-error file.                               */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              string to be written                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_log_error (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * value;               /* o; buffer of the data to write     */
   unsigned long   value_length;        /* i/o; size/length of buffer         */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   value = Argv[1].strptr;
   value_length = Argv[1].strlength;

   /* perform function */ 
   HTTPD_log_error (handle, value, &value_length, &return_code);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_log_trace                                            */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to write a    */
/*              string to the httpd-log file.                                 */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              string to be written                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_log_trace (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * value;               /* o; buffer of the data to write     */
   unsigned long   value_length;        /* i/o; size/length of the buffer     */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   value = Argv[0].strptr;
   value_length = Argv[0].strlength;

   /* perform function */ 
   HTTPD_log_trace (handle, value, &value_length, &return_code);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_supply_label                                         */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to a PICS     */
/*              label generated or received during the PICSDBLookup step.     */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              label data                                                    */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_supply_label (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * value;               /* o; buffer of the data to write     */
   unsigned long   value_length;        /* i/o; size/length of the buffer     */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   value = Argv[0].strptr;
   value_length = Argv[0].strlength;

   /* perform function */ 
   HTTPD_supply_label (handle, value, &value_length, &return_code);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_translate                                            */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to translate  */
/*              a URL to a file system object.                                */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              URL                                                           */
/*              Rexx variable name for the translated URL                     */
/*              Rexx variable name for the PATH_TRANSLATED                    */
/*              Rexx variable name for the QUERY_STRING                       */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_translate (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* i; buffer which contains the URL   */
   unsigned long   name_length;         /* i; length of the URL               */
   unsigned char * url;                 /* o; buffer for the translated URL   */
   unsigned long   url_length;          /* i/o; size/length of the URL        */
   unsigned char * path;                /* o; buffer which contains the PATH  */
   unsigned long   path_length;         /* i/o; size/length of the PATH       */
   unsigned char * query;               /* o; buffer which contains the QUERY */
   unsigned long   query_length;        /* i/o; size/length of the QUERY      */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 5 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]) ||
    !RXVALIDSTRING (Argv[2]) || !RXVALIDSTRING (Argv[3]) ||
    !RXVALIDSTRING (Argv[4]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[1].strptr;
   url = malloc (256);
   url_length = 256;
   path = malloc (256);
   path_length = 256;
   query = malloc (256);
   query_length = 256;

   /* perform function and set any Rexx variables */ 
   HTTPD_translate (handle, name, &name_length, url, &url_length, path,
                    &path_length, query, &query_length, &return_code);
   if (return_code == HTTPD_BUFFER_TOO_SMALL)
      {
      if (url_length > 256)
         {
         free (url);
         url = malloc (url_length);
         }
      if (path_length > 256)
         {
         free (path);
         path = malloc (path_length);
         }
      if (query_length > 256)
         {
         free (query);
         query = malloc (query_length);
         }
      HTTPD_translate (handle, name, &name_length, url, &url_length, path,
                       &path_length, query, &query_length, &return_code);
      }
   if (return_code == HTTPD_SUCCESS)
      {
      SetRexxVar (Argv[2].strptr, url);
      SetRexxVar (Argv[3].strptr, path);
      SetRexxVar (Argv[4].strptr, query);
      }
   free (url);
   free (path);
   free (query);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 

/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    GW_HTTPD_proxy                                                */
/*                                                                            */
/* Description: Rexx callable external function. ICAPI function to make a     */
/*              proxy request.                                                */
/*                                                                            */
/* Rexx Args:   ICAPI handle                                                  */
/*              string to be written                                          */
/*                                                                            */
/*----------------------------------------------------------------------------*/

ULONG APIENTRY GW_HTTPD_proxy (
   PSZ       Name,                      /* name of the function               */
   LONG      Argc,                      /* number of arguments                */
   RXSTRING  Argv[],                    /* list of argument strings           */
   PSZ       Queuename,                 /* current queue name                 */
   PRXSTRING Retstr)                    /* returned result string             */
   {
   unsigned char * handle;              /* i; HTTPD handle                    */
   unsigned char * name;                /* o; buffer which contains the URL   */
   unsigned long   name_length;         /* i/o; size/length of the buffer     */
   unsigned char * value;               /* i; buffer of the request           */
   unsigned long   value_length;        /* i/o; size/length of the body       */
   long            return_code;         /* o; return code                     */
    
   /* check for valid arguments */
   if (Argc != 2 || !RXVALIDSTRING (Argv[0]) || !RXVALIDSTRING (Argv[1]))
      return (40);
    
   /* initialize function parameters */
   memcpy (&handle, Argv[0].strptr, sizeof (PUCHAR));
   name = Argv[0].strptr;
   name_length = Argv[0].strlength;
   value = Argv[1].strptr;
   value_length = Argv[1].strlength;

   /* perform function */ 
   HTTPD_proxy (handle, name, &name_length, value, &value_length, &return_code);
   
   /* set the return code */
   sprintf (Retstr -> strptr, "%d", return_code);
   Retstr -> strlength = strlen (Retstr -> strptr);
                      
   return (RXFUNC_OK);
   }
 
 
/*----------------------------------------------------------------------------*/
/*                                                                            */
/* Function:    CallRexx                                                      */
/*                                                                            */
/* Description: Call the Rexx interpreter from the WWW service.               */
/*                                                                            */
/*----------------------------------------------------------------------------*/

void HTTPD_LINKAGE CallRexx (
   unsigned char * handle,              /* HTTP service handle                */
   long *          return_code)         /* HTTP service return code           */
   {
   unsigned long value_len, name_len;   /* buffer lengths                     */
   static char * name = {"URL"};        /* HTTPD_extract request string       */
   CHAR        * value;                 /* buffer for HTTPD_extract           */
   LONG          rc;                    /* HTTPD_extract return code          */
   char *        rxproc;                /* Rexx procedure name                */
   char *        eos;                   /* end of the Rexx procedure name     */
   LONG          rxrc;                  /* Rexx procedure return code         */
   RXSTRING      argv[1];               /* Rexx procedure argument(s)         */
   RXSTRING      retstr;                /* Rexx procedure return string       */
   short         src;                   /* Rexx procedure return value        */
   CHAR          return_buffer[250];    /* Rexx procedure return string buffer*/
   PUCHAR        userarea[2];           /* registration user area buffer      */
   RXSYSEXIT     sysexit[2];            /* Rexx system exit structs           */
   char          ermsg1[200];           /* error message buffer               */

       
   /* get URL info to find the rexx procedure file name */
   name_len = strlen (name);
   value = malloc (256);
   value_len = 255;
   HTTPD_extract (handle, name, &name_len, value, &value_len, &rc);
   if (rc == HTTPD_BUFFER_TOO_SMALL)
      {
      free (value);
      value = malloc (value_len + 1);
      HTTPD_extract (handle, name, &name_len, value, &value_len, &rc);
      }
   *(value + value_len) = '\0';
    
   /* if we find a question mark in the URL, set it to end of string */
   eos = strrchr (value, '?');
   if (eos != NULL)
      *eos = '\0';
       
   /* find the last forward slash in the URL string */
   rxproc = strrchr (value, '/');
   if (rxproc == NULL)
      {
      *return_code = HTTP_SERVER_ERROR;
      return;
      }
       
   /* bump the ptr to the Rexx procedure name just past the slash character */
   rxproc++;
       
   /* register the Rexx HTTPD external functions */
   RexxRegisterFunctionExe ("HTTPD_ATTRIBUTES", GW_HTTPD_attributes);
   RexxRegisterFunctionExe ("HTTPD_AUTHENTICATE", GW_HTTPD_authenticate);
   RexxRegisterFunctionExe ("HTTPD_EXEC", GW_HTTPD_exec);
   RexxRegisterFunctionExe ("HTTPD_EXTRACT", GW_HTTPD_extract);
   RexxRegisterFunctionExe ("HTTPD_FILE", GW_HTTPD_file);
   RexxRegisterFunctionExe ("HTTPD_LOG_ERROR", GW_HTTPD_log_error);
   RexxRegisterFunctionExe ("HTTPD_LOG_TRACE", GW_HTTPD_log_trace);
   RexxRegisterFunctionExe ("HTTPD_PROXY", GW_HTTPD_proxy);
   RexxRegisterFunctionExe ("HTTPD_READ", GW_HTTPD_read);
   RexxRegisterFunctionExe ("HTTPD_RESTART", GW_HTTPD_restart);
   RexxRegisterFunctionExe ("HTTPD_REVERSE_TRANSLATE", GW_HTTPD_reverse_translate);
   RexxRegisterFunctionExe ("HTTPD_SET", GW_HTTPD_set);
   RexxRegisterFunctionExe ("HTTPD_SUPPLY_LABEL", GW_HTTPD_supply_label);
   RexxRegisterFunctionExe ("HTTPD_TRANSLATE", GW_HTTPD_translate);
   RexxRegisterFunctionExe ("HTTPD_WRITE", GW_HTTPD_write);
       
   /* register the Rexx exits */
   userarea[0] = handle;
   userarea[1] = NULL;
   RexxRegisterExitExe ("WWW_EXIT", Rexx_IO_Exit, (PUCHAR) userarea);
       
   /* call the Rexx interpreter */
   MAKERXSTRING (argv[0], &handle, sizeof (PUCHAR));
   sysexit[0].sysexit_name = "WWW_EXIT";
   sysexit[0].sysexit_code = RXSIO;
   sysexit[1].sysexit_name = NULL;
   sysexit[1].sysexit_code = RXENDLST;
   MAKERXSTRING (retstr, return_buffer, sizeof (return_buffer));
   rxrc = RexxStart (1, argv, rxproc, NULL, "WWW", RXCOMMAND, sysexit, &src,
                     &retstr);
   RexxWaitForTermination (); 
   if (RXSTRPTR (retstr) != return_buffer)
      GlobalFree (RXSTRPTR (retstr));
       
   /* cleanup and set the WWW ICAPI return code */
   free (value);
   if (rxrc != 0)
      {
      sprintf (ermsg1, "Rexx procedure returned rc = %d.", rxrc);
      value_len = strlen (ermsg1); 
      HTTPD_log_error (handle, ermsg1, &value_len, &rc); 
      }
   *return_code = HTTP_OK;
   }

