/***************************************************************************/
/*                                                                         */
/*  Licensed Materials - Property of IBM                                   */
/*                                                                         */
/*  IBM Visual Warehouse                                                   */
/*  Copyright (C) International Business Machines Corp., 1996              */
/*  All rights reserved                                                    */
/*                                                                         */
/***************************************************************************/

/*

        (C) Copyright IBM Corporation 1996, Visual Warehouse

        Description : Determine the name of the country 
                      when the currency type is expressed in the form of
                      DM, FB, German Mark, dollar, ...
                      This is a simple example of data cleansing.

        DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
        sample code created by IBM Corporation. This sample code is
        not part of any standard IBM product and is provided to you
        solely for the purpose of assisting you in the development
        of your applications.  The code is provided "AS IS",
        without warranty of any kind. No IBM shall be liable for
        any damages arising out of your use of the sample code,
        even if they have been advised of the possibility of
        such damages.


        Linkage: The input is a single DB table/edition and the 
        output is to a single DB table.
        Invocation syntax: UDPCLNS.EXE 9 DSName UserID PassWD TtableName Oedt 
                                         DSName UserID PassWD Stablename  
*/

// Developer's note:
// This file consists of several object class definitions which could be separated
// into smaller files.  User may modify this User Defined Program for double byte
// support or use ones own object classes.
// Methods of each object class are grouped together with the class definition
// for easy reading.

// include files --------------------------------------------------------------
#include <windows.h>
#include <iostream.h>
#include <tchar.h>
#include <sqlext.h>
#include <stdio.h>
#include <time.h>

// constants defined for this module ------------------------------------------
#define UDPCLNS_NUM_ARGS       11     // number of input arguments for UDPCLNS
#define UDPCLNS_ARG_TO_FOLLOW   9     // number of args to follow = UDPCLNS_NUM_ARGS - 2
#define VW_ODBC_SUCCESS    "00000"    // successful sqlstate
// error codes used -----------------------------------------------------------
#define VW_OK                 0  
#define VW_ODBC_ERROR      -999       // database processing error
#define VW_ERR_NUM_ARGS    1000       // invalid number of input arguments
#define VW_ERR_INV_ARGS    1010       // invalid argument values
// class declarations ---------------------------------------------------------
class CommandLine;      // parser for command line arguments
class SqlRC;            // returned SQLCODE, SQLSTATE, SLQ Messages 
class SqlStmt;          // ODBC cursor object class
typedef SqlStmt   *PSQLSTMT;
class SqlOdbc;          // ODBC Object class
typedef SqlOdbc   *PSQLODBC;

// handle and cursor states
enum state { VW_NOTALLOCATED, VW_ALLOCATED, VW_INUSE, 
             VW_COMMITTED, VW_ROLLBACKED, VW_CLOSED};


// class definitions ----------------------------------------------------------
class CommandLine {
public:
      CommandLine( void ) {};
      long commandLineParser( const int argc,  TCHAR *argv[] );
      TCHAR * getSrcTableName ( void ) { return srcTableName; }
      TCHAR * getTgtTableName ( void ) { return tgtTableName; }
      TCHAR * getConnectString( void );
      long getOutEdNum( void ) { return ( _ttol( pszOutEdition ));}
protected:
      TCHAR *DSName, *Userid, *Passwd;
      TCHAR *srcTableName, *tgtTableName;
      TCHAR pszOutEdition[12];
      BOOL  numStrValidation( const char * digitStr );
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       CommandLine methods 
// Command Line Parser ********************************************************
long CommandLine::commandLineParser( const int argc,  TCHAR *argv[] )
{     
   // input: UDPCLNS.EXE 9 DSName UserID PassWD TtableName Oedt 
   //                      DSName UserID PassWD Stablename  
   //   - or - 
   // input: UDPCLNS.EXE 10 DSName UserID PassWD TtableName Oedt 
   //                       DSName UserID PassWD Stablename Iedt 
   // input has to be at least UDPCLNS_NUM_ARGS number of arguments; 
   //   including this exe name
   // the second parameter which is number of arguments to follow,
   // must equal to UDPCLNS_ARG_TO_FOLLOW
   // note: if user is going to add user specified paramters, this section 
   // needs to be modified
   long rc = (argc >= UDPCLNS_NUM_ARGS) ? VW_OK : VW_ERR_NUM_ARGS ;
   if ( !rc ) {
      // validate the input is numeric string before calling atol to convert to integer
      if ( numStrValidation( argv[1] )) {
         long lArgNum = _ttol( argv[1] );
         rc = ( lArgNum >= UDPCLNS_ARG_TO_FOLLOW ) ? VW_OK : VW_ERR_NUM_ARGS;
      }
      else rc = VW_ERR_INV_ARGS;    // first argument is not a digit
   }
   if ( !rc ) {
      for ( int i = 1; ( i < argc ); i++ ) {
         switch (i) {
            // In the original OS/2 version, all input arguments were uppercased.
            // In this NT version, DSN, USERID and PASSWORD are left exactly as they were 
            //    defined in VisualWarehouse.
            case 2:  DSName = argv[i];
                     break;
            case 3:  Userid = argv[i];
                     break;
            case 4:  Passwd = argv[i];
                     break;
            case 5:  tgtTableName = _tcsupr( argv[i] );
                     break;
            case 6:  strcpy( pszOutEdition, argv[i] );
                     break;
            case 10: srcTableName = _tcsupr( argv[i] );
                     break;
            default:
                     break;
         }
      }
      rc = numStrValidation( pszOutEdition ) ? VW_OK : VW_ERR_INV_ARGS;
   }
   return rc;
}
// Input numeric string verification ************************************************
BOOL CommandLine::numStrValidation( const char * digitStr )
{
   BOOL rc = TRUE;
   unsigned int i;
   unsigned int len = _tcslen( digitStr );
   // Make sure edition numbers are integers
   for ( i = 0; (( rc ) && ( i < len )); i++ )
      rc = _istdigit( digitStr[i] );
   return rc;
}
// retrieve connect string ****************************************************
TCHAR *CommandLine::getConnectString( void )
{ 
   TCHAR *pszStr = new TCHAR[SQL_MAX_OPTION_STRING_LENGTH]; 
   _tcscpy( pszStr, _TEXT("DSN="));
   _tcscat( pszStr, DSName );
   _tcscat( pszStr, _TEXT(";UID="));
   _tcscat( pszStr, Userid );
   _tcscat( pszStr, _TEXT(";PWD="));
   _tcscat( pszStr, Passwd );
   return ( pszStr );
}


class SqlRC {
public:  
   SqlRC();
   void print( void );
   SqlRC &set( const HENV &hEnv, const HDBC &hDbc, const HSTMT &hStmt );

   SQLINTEGER  getSQLCode ( void ) { return sqlCode; }
   TCHAR *     getSQLState( void ) { return (char *)sqlState; }
   // in dealing with ODBC drivers, sqlstate returned is more important than sqlcode.
   // sqlcode is meaningless until a connection to DB2 is established.
   long        getRC( void ) { return(
               (!_tcscmp( (const char *)sqlState, VW_ODBC_SUCCESS )) ? VW_OK : VW_ODBC_ERROR); }
private:
  	UCHAR       sqlMessage[SQL_MAX_MESSAGE_LENGTH];
	UCHAR       sqlState[SQL_SQLSTATE_SIZE + 1];
	SDWORD      sqlCode;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlRC methods 
// SqlRC constructor **********************************************************
SqlRC::SqlRC( void): sqlCode ( 0 )
{
	sqlMessage[0] = NULL; 
   _tcscpy( (char *)sqlState, VW_ODBC_SUCCESS );
}
// SqlRC Print ****************************************************************
void SqlRC::print( void)
{
   FILE * pfOutFile = NULL;
   pfOutFile = fopen( "UDPCLNS.LOG", "a+" );
   if ( pfOutFile ) {
      unsigned char date[10],time[10];
      fprintf( pfOutFile, "UDPCLNS: %s  %s\n", _strdate((char *)date ), _strtime((char *)time ));
      fprintf( pfOutFile, " ODBC returned SQLCODE = %d, SQLSTATE = %s, System Message: %s\n",
      sqlCode, sqlState, ( _tcslen((const TCHAR *)sqlMessage )) ? sqlMessage : NULL );
      fclose( pfOutFile );
   }
}
// SqlRC set sqlcode, sqlstate, sql message ***********************************
SqlRC &SqlRC::set( const HENV &hEnv, const HDBC &hDbc, const HSTMT &hStmt )
{
   UCHAR    ucSQLState[5];                   // local SQLSTATE
   SDWORD   dwSQLCode;                       // local SQLCODE
   UCHAR    ucErrMsg[SQL_MAX_MESSAGE_LENGTH];// local SQL MESSAGE 
   SWORD    wErrMsgLen;
   RETCODE  retcode = SQLError ( hEnv, hDbc, hStmt, sqlState, &sqlCode,
                                 sqlMessage, SQL_MAX_MESSAGE_LENGTH - 1, &wErrMsgLen );
   if (retcode != SQL_NO_DATA_FOUND) {
      while ( retcode == SQL_SUCCESS )
      {  // pops out all accumulated errors
         retcode = SQLError ( hEnv, hDbc, hStmt, ucSQLState, &dwSQLCode,
                      ucErrMsg, SQL_MAX_MESSAGE_LENGTH - 1, &wErrMsgLen );
      } // end of retriving all the errors accumulated
   }
   return *this;
}
// ODBC Connection object
class SqlOdbc {
public:
   SqlOdbc ( void );
   ~SqlOdbc( void );
   SqlOdbc ( SqlOdbc & Conn ) { hEnv = Conn.hEnv; eEnvState = Conn.eEnvState;
                                hDbc = Conn.hDbc; eDbcState = Conn.eDbcState; }
   virtual RETCODE connect( const TCHAR * connectStr, SqlRC & src );
   virtual SqlRC commit( void );
   virtual SqlRC rollback( void );
   virtual void  disconnect( void );
   virtual HDBC  getHDBC( void ) { return ( hDbc ); }
   virtual state getDBCSTATE( void ) { return ( eDbcState ); }
protected:
   HENV     hEnv;                // odbc environment handle
   HDBC     hDbc;                // odbc connection handle
   state    eEnvState;
   state    eDbcState;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlOdbc Methods 
// SqlOdbc constructor ********************************************************
SqlOdbc::SqlOdbc( void): hDbc(SQL_NULL_HDBC),hEnv(SQL_NULL_HENV),
eEnvState(VW_NOTALLOCATED),eDbcState(VW_NOTALLOCATED)
{
}
// SqlOdbc destructor *********************************************************
SqlOdbc::~SqlOdbc( void )
{
   if ( eDbcState != VW_NOTALLOCATED ) {
      // if process went down unexpectly, first rollback the transaction;   
      //  then drop the open cursor before disconnect from db.
      if ( eDbcState == VW_INUSE )   {
         SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      }
      SQLFreeConnect( hDbc );
   }
   SQLFreeEnv ( hEnv );
}
// SqlOdbc Connect ************************************************************
RETCODE SqlOdbc::connect( const TCHAR * connectStr, SqlRC & sRC )
{
   RETCODE  retcode;
   unsigned char szConnStrOut[ SQL_MAX_OPTION_STRING_LENGTH ];
   SWORD    cbConnStrOut;

   // allocate env handle if not already allocated
   if ( eEnvState == VW_NOTALLOCATED ) {
      retcode = SQLAllocEnv ( &hEnv );
      if ( retcode == SQL_SUCCESS )    {
         eEnvState = VW_ALLOCATED;
         // allocate connection handle
         retcode = SQLAllocConnect( hEnv, &hDbc );
         if ( retcode == SQL_SUCCESS )    {
            eDbcState = VW_ALLOCATED;
            // establish connection
            retcode = SQLDriverConnect ( 
                              hDbc,     // connection handle
                              NULL,     // no window handle
       (unsigned char *)connectStr,     // connect string
               _tcslen(connectStr),     // length of connect string
                      szConnStrOut,     // returned connect buffer
  SQL_MAX_OPTION_STRING_LENGTH - 1,     // len of ret conn buffer
                     &cbConnStrOut,     // actual len of conn buffer
               SQL_DRIVER_NOPROMPT      // Driver Mgr take it as is
                              );
            if ( retcode == SQL_SUCCESS ) {                                 
               eDbcState = VW_INUSE;
               // AUTOCOMMIT must be set off explicitly.
               // Otherwise SQLSTATE 25000 will result at the time of Disconnect, 
               // because SQLTransact calls would be all ignored.
               retcode =  SQLSetConnectOption( hDbc, 
                                     SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF );
            }
         }
      }
   }
   if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
   return retcode;
}
// SqlOdbc Disconnect *********************************************************
void SqlOdbc::disconnect(void)
{
   if ( eDbcState != VW_NOTALLOCATED ) {
      RETCODE retcode;
      // if process went down unexpectly, first rollback the transaction;   
      //  then drop the open cursor before disconnect from db.
      if ( eDbcState == VW_INUSE )   {
         retcode = SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      }
      retcode = SQLDisconnect( hDbc );
      retcode = SQLFreeConnect( hDbc );
   }
}
// SqlOdbc commit *************************************************************
SqlRC SqlOdbc::commit(void)
{
   SqlRC sRC;
   if ( eDbcState == VW_INUSE )   {
      RETCODE retcode = SQLTransact( hEnv, hDbc, SQL_COMMIT );
      if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
      else eDbcState = VW_COMMITTED;
   }
   return sRC;
}
// SqlOdbc rollback ***********************************************************
SqlRC SqlOdbc::rollback(void)
{
   SqlRC sRC;
   if ( eDbcState == VW_INUSE )   {
      RETCODE retcode = SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
      else eDbcState = VW_ROLLBACKED;
   }
   return sRC;
}


// ODBC Cursor object: derived from SqlOdbc
class SqlStmt : public SqlOdbc {
public:
   SqlStmt ( SqlOdbc & Conn );
   ~SqlStmt( void ); 

   state    getState( void ) { return eStmtState; }
   HSTMT    openCursor( void );   // open and return a cursor
   HSTMT    getCursor( void ) { return (eStmtState!=VW_NOTALLOCATED)?hStmt:SQL_NULL_HSTMT; }    
   SqlRC    prepare( UCHAR * instructions );
   SqlRC    execute( void );
   RETCODE  fetch( SqlRC & src );
private:
   HSTMT    hStmt;
   state    eStmtState;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlStmt Methods 
// SqlStmt constructor ********************************************************
SqlStmt::SqlStmt( SqlOdbc & Conn)
:SqlOdbc( Conn ), hStmt(SQL_NULL_HSTMT),eStmtState(VW_NOTALLOCATED)
{
}
// SqlStmt destructor *********************************************************
SqlStmt::~SqlStmt( void )
{
   if ( eStmtState != VW_NOTALLOCATED ) {
     RETCODE  retcode =  SQLFreeStmt( hStmt, SQL_DROP );
     eStmtState = VW_NOTALLOCATED;
   }
}
// SqlStmt openCursor *********************************************************
HSTMT SqlStmt::openCursor( void )
{
   RETCODE  retcode = SQLAllocStmt ( hDbc, &hStmt );
   if ( retcode == SQL_SUCCESS )    {
      eStmtState = VW_ALLOCATED;
   }
   return ( (retcode == SQL_SUCCESS) ? hStmt : SQL_NULL_HSTMT );
}
// SqlStmt prepare ************************************************************
SqlRC SqlStmt::prepare( UCHAR * instructions ) 
{
   SqlRC    rc;
   RETCODE  retcode = SQLPrepare ( hStmt, instructions, _tcslen((const char*)instructions));
   if ( retcode == SQL_SUCCESS )    {
      eStmtState = eDbcState = VW_INUSE;
   }
   return ( (retcode == SQL_SUCCESS) ? rc : rc.set(hEnv, hDbc, hStmt) );
}
// SqlStmt execute  ***********************************************************
SqlRC SqlStmt::execute( void )
{
   SqlRC    rc;
   RETCODE  retcode = SQLExecute ( hStmt );
   return ( (retcode == SQL_SUCCESS) ? rc : rc.set(hEnv, hDbc, hStmt) );
}
// SqlStmt fetch **************************************************************
RETCODE SqlStmt::fetch( SqlRC & src )
{
   RETCODE  retcode = SQLFetch ( hStmt );
   if (( retcode == SQL_ERROR ) || ( retcode == SQL_SUCCESS_WITH_INFO ))
      src.set(hEnv, hDbc, hStmt);
   return ( retcode );
}

// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       MAIN           
// local function declarations ------------------------------------------------
void print( int rc );
#define Ssel  "SELECT $$BVEDITION,MACHINE_TYPE,SERIAL,SALES_PRICE,YEARMONTH,LOCAL_CURRENCY FROM "
#define Tsel  "SELECT $$BVEDITION,MACHINE_TYPE,SERIAL,SALES_PRICE,YEARMONTH,COUNTRY FROM "
#define Tins1 "INSERT INTO " 
#define Tins2 " ($$BVEDITION,MACHINE_TYPE,SERIAL,SALES_PRICE,YEARMONTH,COUNTRY) VALUES(?,?,?,?,?,?) "


// main ***********************************************************************
int main( int argc, TCHAR *argv[] )
{
   int         rc = VW_OK;
   SqlRC      src;
   CommandLine cmdArgs;

   cout << "Enter UDPCleanse" << endl;
   rc = cmdArgs.commandLineParser( argc, argv );
   if ( !rc )  {  // command line input was parsed and validated
      SqlOdbc DBConnection;
      rc = DBConnection.connect( cmdArgs.getConnectString(), src );
      if ( rc == VW_OK ) {  // connection established
         PSQLSTMT  psSrcCursor = new SqlStmt(DBConnection);
         PSQLSTMT  psTgtCursor = new SqlStmt(DBConnection);
         HSTMT     hSrc;
         HSTMT     hTgt;
         hSrc = psSrcCursor->openCursor();
         hTgt = psTgtCursor->openCursor();
         if (( psSrcCursor->getState() == VW_ALLOCATED ) && 
             ( psTgtCursor->getState() == VW_ALLOCATED )) // cursors opened
         {  
            // initialize source SELECT and target INSERT statements
            UCHAR srcStmt[200];
            UCHAR tgtStmt[200];
            _tcscpy( (char *) srcStmt, Ssel );
            _tcscat( (char *) srcStmt, cmdArgs.getSrcTableName() );
            _tcscpy( (char *) tgtStmt, Tins1 );
            _tcscat( (char *) tgtStmt, cmdArgs.getTgtTableName() );
            _tcscat( (char *) tgtStmt, Tins2 );
            src = psSrcCursor->prepare( srcStmt );    
            if ( (rc = src.getRC()) == VW_OK ) {
               src = psTgtCursor->prepare( tgtStmt );
            }
            // execute SELECT statement
            if ( (rc = src.getRC()) == VW_OK ) 
               src = psSrcCursor->execute();
            if ( (rc = src.getRC()) == VW_OK ) { //ready to fetch and insert
               RETCODE  retcode;
               SDWORD   rowCount = 0;
               // Declare storage locations for result set data 
               UCHAR  pszMachineType[11], pszSerial[6], pszYearMonth[33],pszCountry[16];
               SDWORD lBVEdition,lSales;
               // Declare storage locations for bytes available to return */
               SDWORD cbBVEdition,cbMachineType,cbSerial,cbSales,cbYearMonth,cbCountry;
               // Bind columns in SELECT result set to storage locations 
               SQLBindCol(hSrc, 1, SQL_C_SLONG, &lBVEdition   ,  0, &cbBVEdition  );
               SQLBindCol(hSrc, 2, SQL_C_CHAR , pszMachineType, 11, &cbMachineType);
               SQLBindCol(hSrc, 3, SQL_C_CHAR , pszSerial     ,  6, &cbSerial     );
               SQLBindCol(hSrc, 4, SQL_C_SLONG, &lSales       ,  0, &cbSales      );
               // SQLBindCol(hSrc, 5, SQL_C_SLONG, &lTax         ,  0, &cbTax        );
               SQLBindCol(hSrc, 5, SQL_C_CHAR , pszYearMonth  , 33, &cbYearMonth  );
               SQLBindCol(hSrc, 6, SQL_C_CHAR , pszCountry    , 16, &cbCountry    );
               // Bind storage locations to parameter markers in INSERT statement
               SQLBindParameter(hTgt, 1, SQL_PARAM_INPUT, SQL_C_SLONG, SQL_INTEGER, 
                      4, 0, &lBVEdition, 0, &cbBVEdition);
               SQLBindParameter(hTgt, 2, SQL_PARAM_INPUT, SQL_C_CHAR , SQL_CHAR, 
                     11, 0, pszMachineType, 10, &cbMachineType);
               SQLBindParameter(hTgt, 3, SQL_PARAM_INPUT, SQL_C_CHAR , SQL_VARCHAR, 
                      6, 0, pszSerial, 5, &cbSerial);
               SQLBindParameter(hTgt, 4, SQL_PARAM_INPUT, SQL_C_SLONG, SQL_INTEGER, 
                      4, 0, &lSales    , 0, &cbSales    );
               // SQLBindParameter(hTgt, 5, SQL_PARAM_INPUT, SQL_C_SLONG, SQL_INTEGER, 
               //       4, 0, &lTax      , 0, &cbTax      );
               SQLBindParameter(hTgt, 5, SQL_PARAM_INPUT, SQL_C_CHAR , SQL_VARCHAR, 
                     33, 0, pszYearMonth, 32, &cbYearMonth);
               SQLBindParameter(hTgt, 6, SQL_PARAM_INPUT, SQL_C_CHAR , SQL_VARCHAR, 
                     16, 0, pszCountry, 15, &cbCountry);   // calculated column
               // ready to fetch, cleanse and insert
               while  (( retcode = psSrcCursor->fetch( src )) == SQL_SUCCESS ) {
                  lBVEdition = cmdArgs.getOutEdNum();
                  if (( !_tcsncmp( (const char *)pszCountry, _TEXT("FB"), 2) ) || 
                      ( !_tcsncmp( (const char *)pszCountry, _TEXT("BF"), 2) )) 
                     _tcscpy((char *)pszCountry, "BELGIUM");
                  else if  ( !_tcsncmp( (const char *)pszCountry, _TEXT("FF"), 2) ) 
                     _tcscpy((char *)pszCountry, "FRANCE");
                  else if (( !_tcsncmp( (const char *)pszCountry, _TEXT("DM"), 2) ) ||
                           ( !_tcsncmp( (const char *)pszCountry, _TEXT("German Mark"), 11) )) 
                     _tcscpy((char *)pszCountry, "GERMAN");
                  else if  ( !_tcsncmp( (const char *)pszCountry, _TEXT("FS"), 2) )
                     _tcscpy((char *)pszCountry, "SWITZELAND");
                  else if  ( !_tcsncmp( (const char *)pszCountry, _TEXT("Lira"), 4) )
                     _tcscpy((char *)pszCountry, "ITALY");
                  // resize the country string length
                  cbCountry = _tcslen( (const char *)pszCountry );
                  src = psTgtCursor->execute();
                  if ( src.getRC() != VW_OK ) break;
                  else rowCount++;
               }
               // commit after insert completed successfully
               if (( rowCount ) && ( src.getRC() == VW_OK )) {
                  src = psTgtCursor->commit();
                  rc = src.getSQLCode();     // return the commit status
               }
               else if ( rowCount ) {   // erred during fetch and insert
                  rc = src.getSQLCode();     // return execution error
                  psTgtCursor->rollback();
               }
               // end if update was successful
            } // endif both source and target are prepared
         } // end if cursors opened successfully
         // free cursors 
         delete psSrcCursor;
         delete psTgtCursor;
         DBConnection.disconnect();
      } // end if connection was successful
   } // end if command line parsing was successful
   // show processing result
   if ( rc == VW_ODBC_ERROR ) rc = src.getSQLCode();
   src.print();
   print (rc); 
   return rc;
} // end of main


// local function print RC ****************************************************
void print( int rc ) 
{
   TCHAR message[200];
   switch ( rc ) {
   case VW_OK: 
      _tcscpy( message, _TEXT("UDPCleanse completed successfully!!") ); 
      break;
   case VW_ERR_NUM_ARGS:
      _tcscpy( message, _TEXT("ERROR: Invalid number of input arguments.") );
      break;
   case VW_ERR_INV_ARGS:
      _tcscpy( message, _TEXT("ERROR: Invalid argument values.") );
      break;
   default:
      _stprintf( message, _TEXT("SQL ERROR: SQLCODE = %d."), rc );
      break;
   }
   FILE * pfOutFile = NULL;
   pfOutFile = fopen( "UDPCLNS.LOG", "a+" );
   if ( pfOutFile ) {
      unsigned char date[10],time[10];
      _ftprintf( pfOutFile, "\n\nUDPCLNS: %s  %s\n", _strdate((char *)date ), _strtime((char *)time ));
      _ftprintf( pfOutFile, " %s\n", message);
      fclose( pfOutFile );
   }
}


